#!/usr/bin/env python
# -*- coding: utf-8 -*-
from django.contrib.auth.decorators import login_required
from django.core.urlresolvers import reverse
from django.shortcuts import redirect, render

from .models import PhitbitUser
from . import utils


@login_required
def login(request):
    """ Login to fitbit.com
    URL name:
        `phitbit-login`
    """
    next_url = request.GET.get('next', None)
    if next_url:
        request.session['fitbit_next'] = next_url
    else:
        request.session.pop('fitbit_next', None)

    usr, _ = PhitbitUser.objects.get_or_create(user=request.user)
    phb = usr._build_oauth()
    callback_url = request.build_absolute_uri(reverse('phitbit-complete'))
    parameters = {'oauth_callback': callback_url}
    token = phb.client.fetch_request_token(parameters)
    token_url = phb.client.authorize_token_url(token)
    request.session['token'] = token
    return redirect(token_url)


@login_required
def complete(request):
    """ Save the user keys to the Phitbit account
    URL name:
        `phitbit-complete`
    """
    usr, _ = PhitbitUser.objects.get_or_create(user=request.user)
    phb = usr._build_oauth()

    try:
        token = request.session.pop('token')
        verifier = request.GET.get('oauth_verifier')
    except KeyError:
        return redirect(reverse('phitbit-error'))

    try:
        access_token = phb.client.fetch_access_token(token, verifier)
    except:
        return redirect(reverse('phitbit-error'))

    usr.USER_KEY = access_token.key
    usr.USER_SECRET = access_token.secret
    usr.save()
    next_url = request.session.pop(
        'fitbit_next',
        None) or utils.get_setting('PHITBIT_SUCCESS_REDIRECT')
    return redirect(next_url)


@login_required
def error(request):
    """
    URL name:
        `phitbit-error`
    """
    return render(request, utils.get_setting('PHITBIT_ERROR_REDIRECT'), {})


@login_required
def logout(request):
    """clears this user's Phitbit keys.
    URL name:
        `phitbit-logout`
    """
    phbuser = request.user.phitbit
    phbuser.USER_KEY = ''
    phbuser.USER_SECRET = ''
    phbuser.authenticated = False
    phbuser.save()
    next_url = request.GET.get(
        'next',
        None) or '/'
    return redirect(next_url)
