from __future__ import unicode_literals

from django.contrib.staticfiles.finders import BaseStorageFinder, get_finders
from django.core.files import File
from django.core.files.base import ContentFile
from django.utils.module_loading import import_string

from pipedash import settings
from pipedash.exceptions import PipedashResourceNotFound
from pipedash.storage import TemporaryFileSystemStorage


class PipedashFinder(BaseStorageFinder):
    """
    The core functionality of pipedash

    Preprocesses, compresses and concatenates files based upon the PIPEDASH_PACKAGES settings
    """
    def __init__(
        self,
        concat=settings.PIPEDASH_CONCAT,
        compressors=settings.PIPEDASH_COMPRESSORS,
        compilers=settings.PIPEDASH_COMPILERS,
        packages=settings.PIPEDASH_PACKAGES
    ):

        self.concat = concat
        self.packages = packages

        self.processors = []
        self.processors += tuple(compilers) if compilers else ()
        self.processors += tuple(compressors) if compressors else ()

        self.storage = TemporaryFileSystemStorage()
        self.build_storage = TemporaryFileSystemStorage()

        self.find = self.find_package if self.concat else self.find_resource
        self.list = self.list_package if self.concat else self.list_resource

        # Setup the BaseStorageFinder
        super(PipedashFinder, self).__init__()

    def find_resource(self, path, all=False):
        """
        Creates and returns any resources that matches the provided path
        """
        for package in self.packages.values():
            for resource in package['files']:
                if resource[0] == path:
                    self.create_resource(resource)
        return super(PipedashFinder, self).find(path, all=all)

    def find_package(self, path, all=False):
        """
        Creates and returns any package (concatenated resource files) that matches the provided path
        """
        for package in self.packages.values():
            if package['path'] == path:
                self.create_package(package)
        return super(PipedashFinder, self).find(path, all=all)

    def list_resource(self, ignore_patterns):
        """
        Creates and returns all the resources files
        """
        for package in self.packages.values():
            for resource in package['files']:
                self.create_resource(resource)
        return super(PipedashFinder, self).list(ignore_patterns)

    def list_package(self, ignore_patterns):
        """
        Creates and returns all the package files (concatenated resource files)
        """
        for package in self.packages.values():
            self.create_package(package)
        return super(PipedashFinder, self).list(ignore_patterns)

    def create_package(self, package):
        """
        Creates a package in the current PipedashFinder's temporary storage

        Loops through all the resource in this package and saves to a local per package temporary
        storage to stop cross pollution of packages
        """
        content = b''

        for resource in package['files']:
            resource_path, resource_mime = resource
            self.create_resource((resource_path, resource_mime), storage=self.build_storage)
            content += self.build_storage.open(resource_path).read()

        self.storage.save(package['path'], ContentFile(content))

    def create_resource(self, resource, storage=None):
        """
        Create a resource and run it through the processors that match the MIME

        Saves to ether the PipedashFinder storage or a temporary build storage if the resource
        is being built to be bundled in a package
        """
        path, mime = resource

        # Default to PipedashFinder storage
        storage = storage if storage else self.storage

        # Skip if we have already made this file
        if storage.exists(path):
            return

        # Find the location of the file we want to run through the processors
        resolved_path = self.find_from_finders(path)

        # Log an error saying we couldn't find the file to process
        if not resolved_path:
            raise PipedashResourceNotFound("Resource not found: %s" % path)

        # Load current file into memory
        content = File(open(resolved_path, 'rb')).read()

        # Loop though all the processors
        for processor_class in self.processors:
            # This is our Processor class
            Processor = import_string(processor_class)
            # Only process if the Processor has our target input MIME
            if mime == Processor.input_mime:
                # Save the output to the input for use in the next processor
                content = Processor(content, resolved_path).run()
                mime = Processor.output_mime

        # Save the processed content to our storage
        storage.save(path, ContentFile(content))

    def find_from_finders(self, path):
        """
        Finds the file from any other finders excluding this one
        """
        for finder in get_finders():
            if isinstance(finder, PipedashFinder):
                continue
            result = finder.find(path, all=False)
            if result:
                return result
        return None
