from __future__ import unicode_literals

from django import template
from django.template.loader import render_to_string
from django.forms.utils import flatatt
from django.templatetags.static import static
from django.utils.module_loading import import_string

from .. import settings

register = template.Library()


class PipedashNode(template.Node):

    def __init__(
        self,
        name,
        concat=settings.PIPEDASH_CONCAT,
        compressors=settings.PIPEDASH_COMPRESSORS,
        compilers=settings.PIPEDASH_COMPILERS,
        packages=settings.PIPEDASH_PACKAGES
    ):
        self.name = template.Variable(name)

        self.concat = concat
        self.packages = packages
        self.processors = []
        self.processors += tuple(compilers) if compilers else ()
        self.processors += tuple(compressors) if compressors else ()

    def render(self, context):
        package_name = self.name.resolve(context)
        package = self.packages.get(package_name)

        if not package:
            # Log that we received an invalid package name
            return ''

        if self.concat:
            return self.render_package(package)

        attrs = package.get('attrs', {})
        resources = [self.render_resource(resource, attrs=attrs) for resource in package['files']]
        return "\n".join(resources)

    def render_package(self, package):
        return self.render_tag(package['path'], package['mime'], package.get('attrs', {}))

    def render_resource(self, resource, attrs=None):
        resource_path, resource_mime = resource

        for processor_class in self.processors:
            Processor = import_string(processor_class)

            # We don't run the resource through the processor here
            # as that has already happened via the staticfiles finder
            # we just want its final mime type
            if resource_mime == Processor.input_mime:
                resource_mime = Processor.output_mime

        return self.render_tag(resource_path, resource_mime, attrs)

    def render_tag(self, path, mime, attrs=None):
        return render_to_string('pipedash/%s.html' % mime.lower(), {
            'path': static(path),
            'mime': mime,
            'attrs': attrs,
            'flatatt': flatatt(attrs)
        })


@register.tag
def pipedash(parser, token):
    try:
        tag_name, name = token.split_contents()
    except ValueError:
        raise template.TemplateSyntaxError(
            '%r requires exactly one argument: the name of a package'
            ' in the PIPEDASH_PACKAGE setting' % token.split_contents()[0]
        )
    return PipedashNode(name)
