from __future__ import unicode_literals

from django.test import TestCase
from django.conf import settings
from pipedash.finders import PipedashFinder
from pipedash.exceptions import PipedashResourceNotFound


class TestNonConcatFinder(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(concat=False)

    def test_list(self):
        expected = set()
        for resource in settings.PIPEDASH_PACKAGES.values():
            for src, _ in resource['files']:
                expected.add(src)
        results = [x for x, _ in self.finder.list(None)]
        self.assertListEqual(sorted(results), sorted(expected))

    def test_find(self):
        for resource in settings.PIPEDASH_PACKAGES.values():
            for src, _ in resource['files']:
                result = self.finder.find(src)
                self.assertTrue(result.endswith(src))

    def test_multiple_builds(self):
        expected = set()
        for resource in settings.PIPEDASH_PACKAGES.values():
            for src, _ in resource['files']:
                expected.add(src)
        expected = sorted(expected)

        for _ in range(5):
            results = sorted([x for x, _ in self.finder.list(None)])
            self.assertListEqual(results, expected)


class TestConcatFinder(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(concat=True)

    def test_files_are_provided_by_finder(self):
        results = [x for x, _ in self.finder.list(None)]
        expected = set()
        for resource in settings.PIPEDASH_PACKAGES.values():
            expected.add(resource['path'])
        self.assertListEqual(sorted(results), sorted(expected))

    def test_find(self):
        for resource in settings.PIPEDASH_PACKAGES.values():
            result = self.finder.find(resource['path'])
            self.assertTrue(result.endswith(resource['path']))


class TestInvalidResource(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(packages={
            'foo': {
                'path': 'js/main.js',
                'mime': 'text/javascript',
                'files': [
                    ('js/does-not-exist.js', 'text/javascript'),
                ],
            },
        })

    def test_resource_missing_raises_exception(self):
        with self.assertRaises(PipedashResourceNotFound) as exception:
            self.finder.list(None)
        self.assertEqual(str(exception.exception), 'Resource not found: js/does-not-exist.js')
