from __future__ import unicode_literals

from django.contrib.staticfiles.finders import BaseStorageFinder, get_finders
from django.core.files.base import ContentFile
from django.utils.module_loading import import_string
import itertools

from pipedash import settings
from pipedash.exceptions import PipedashResourceNotFound, PipedashFileAlreadyExists
from pipedash.storage import TemporaryFileSystemStorage


class PipedashFinder(BaseStorageFinder):
    """
    The core functionality of pipedash

    Preprocesses, compresses and concatenates files based upon the PIPEDASH_PACKAGES settings
    """
    def __init__(
        self,
        concat=settings.PIPEDASH_CONCAT,
        compressors=settings.PIPEDASH_COMPRESSORS,
        compilers=settings.PIPEDASH_COMPILERS,
        packages=settings.PIPEDASH_PACKAGES
    ):

        self.concat = concat
        self.packages = packages

        self.processors = []
        self.processors += tuple(compilers) if compilers else ()
        self.processors += tuple(compressors) if compressors else ()

        self.storage = TemporaryFileSystemStorage()

        # Setup the BaseStorageFinder
        super(PipedashFinder, self).__init__()

        self.generate = self.generate_packages if self.concat else self.generate_resources
        self.processed = False

    def find(self, path, all=False):
        self.generate()
        return super(PipedashFinder, self).find(path, all=all)

    def list(self, ignore_patterns):
        self.generate()
        return super(PipedashFinder, self).list(ignore_patterns)

    def generate_packages(self):
        if self.processed:
            return

        for package in self.packages.values():
            package_content = b''

            for path in package['files']:
                content, resolved_path = self.read_resource(path)
                content, _ = self.process(content, path, resolved_path)
                package_content += content

            if self.storage.exists(package['path']):
                raise PipedashFileAlreadyExists(
                    "A package already exists with the target path: %s" % package['path']
                )

            self.storage.save(package['path'], ContentFile(package_content))

        self.processed = True

    def generate_resources(self):

        if self.processed:
            return

        resources = set(itertools.chain(*[package['files'] for package in self.packages.values()]))

        for path in resources:
            content, resolved_path = self.read_resource(path)
            content, rewrite_path = self.process(content, path, resolved_path)

            if self.storage.exists(rewrite_path):
                raise PipedashFileAlreadyExists(
                    "A resource already exists with the target path: %s" % rewrite_path
                )

            self.storage.save(rewrite_path, ContentFile(content))

        self.processed = True

    def process(self, content, path, resolved_path):
        """
        Runs a set of processes on the inputted content
        """

        # Store a reference to the normal path because we can append
        # a new extension for example foo.less will become foo.less.css
        rewrite_path = path

        for processor_class in self.processors:

            # Imports the processes class from a dot patch
            processor = import_string(processor_class)

            # Skip processing if we don't deal with these files
            if not rewrite_path.endswith(".%s" % processor.input_extension):
                continue

            # We want the path location not the rewrite path
            content = processor(content, resolved_path).run()

            # Append the new extension if its different from the input file
            # to make sure its served with the right MIME type for runserver
            if not rewrite_path.endswith(".%s" % processor.output_extension):
                rewrite_path += ".%s" % processor.output_extension

        return content, rewrite_path

    def read_resource(self, path):
        """
        Attempts to read the path file into memory.

        Throws a PipedashResourceNotFound exception if the file was not found in
        the other static files finders
        """
        resolved_path = self.find_from_finders(path)
        if not resolved_path:
            raise PipedashResourceNotFound("Resource not found: %s" % path)
        with open(resolved_path, 'rb') as handle:
            content = handle.read()
        return content, resolved_path

    def find_from_finders(self, path):
        """
        Finds the file from any other finders excluding this one
        """
        for finder in get_finders():
            if isinstance(finder, PipedashFinder):
                continue
            result = finder.find(path, all=False)
            if result:
                return result
        return None
