from __future__ import unicode_literals
import os

from django import template
from django.template.loader import render_to_string
from django.forms.utils import flatatt
from django.templatetags.static import static
from django.utils.module_loading import import_string

from .. import settings

register = template.Library()


class PipedashNode(template.Node):

    def __init__(
        self,
        name,
        concat=settings.PIPEDASH_CONCAT,
        compressors=settings.PIPEDASH_COMPRESSORS,
        compilers=settings.PIPEDASH_COMPILERS,
        packages=settings.PIPEDASH_PACKAGES
    ):
        self.name = template.Variable(name)

        self.concat = concat
        self.packages = packages
        self.processors = []
        self.processors += tuple(compilers) if compilers else ()
        self.processors += tuple(compressors) if compressors else ()

    def render(self, context):
        package_name = self.name.resolve(context)
        package = self.packages.get(package_name, {})
        attrs = package.get('attrs', {})

        if not package:
            # Log that we received an invalid package name
            return ''

        if self.concat:
            return self.render_tag(package['path'], attrs=attrs)

        tags = [self.render_tag(self.final_path(path), attrs=attrs) for path in package['files']]
        return "\n".join(tags)

    def final_path(self, path):
        """
        Simulates a file going through all the processors and returns
        the final path where the file will be stored in the finder
        """
        for processor_class in self.processors:

            # Load the processor class
            processor = import_string(processor_class)

            # Skip if we don't use this processor
            if not path.endswith(".%s" % processor.input_extension):
                continue

            # Skip if we have the same extension
            if path.endswith(".%s" % processor.output_extension):
                continue

            # Append the new output extension
            path += ".%s" % processor.output_extension

        return path

    @staticmethod
    def render_tag(path, attrs=None):
        extension = os.path.splitext(path)[1].lower()[1:]
        return render_to_string('pipedash/%s.html' % extension, {
            'path': static(path),
            'attrs': attrs,
            'flatatt': flatatt(attrs)
        })


@register.tag
def pipedash(parser, token):
    try:
        tag_name, name = token.split_contents()
    except ValueError:
        raise template.TemplateSyntaxError(
            '%r requires exactly one argument: the name of a package'
            ' in the PIPEDASH_PACKAGE setting' % token.split_contents()[0]
        )
    return PipedashNode(name)
