from __future__ import unicode_literals

from django.test import TestCase
from pipedash.finders import PipedashFinder
from pipedash.exceptions import PipedashResourceNotFound, PipedashFileAlreadyExists


class TestNonConcatFinder(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(concat=False)
        self.expected = sorted([
            'js/foo.js',
            'js/bar.js',
            'css/red.css',
            'css/blue.css',
            'less/green.less.css',
            'sass/john.sass.css',
        ])

    def test_list(self):
        results = [x for x, _ in self.finder.list(None)]
        self.assertListEqual(sorted(results), self.expected)

    def test_find(self):
        for resource in self.expected:
            result = self.finder.find(resource, False)
            self.assertTrue(result.endswith(resource))


class TestConcatFinder(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(concat=True)
        self.expected = sorted([
            'js/main.js',
            'css/main.css',
            'css/contact.css',
        ])

    def test_files_are_provided_by_finder(self):
        results = [x for x, _ in self.finder.list(None)]
        self.assertListEqual(sorted(results), self.expected)

    def test_find(self):
        for resource in self.expected:
            result = self.finder.find(resource)
            self.assertTrue(result.endswith(resource))


class TestInvalidResource(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(packages={
            'foo': {
                'path': 'js/main.js',
                'files': [
                    'js/does-not-exist.js',
                ],
            },
        })

    def test_resource_missing_raises_exception(self):
        with self.assertRaises(PipedashResourceNotFound) as exception:
            self.finder.list(None)
        self.assertEqual(str(exception.exception), 'Resource not found: js/does-not-exist.js')


class TestResourceAlreadyExistsConcat(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(
            concat=True,
            packages={
                'first': {
                    'path': 'css/output.css',
                    'files': [
                        'misc/exists.less',
                    ]
                },
                'second': {
                    'path': 'css/output.css',
                    'files': [
                        'misc/exists.less',
                    ]
                }
            }
        )

    def test_resource_already_exists_raises_exception(self):
        with self.assertRaises(PipedashFileAlreadyExists) as exception:
            self.finder.list(None)
        self.assertEqual(
            str(exception.exception),
            'A package already exists with the target path: css/output.css'
        )


class TestResourceAlreadyExistsNonConcat(TestCase):

    def setUp(self):
        self.finder = PipedashFinder(
            concat=False,
            packages={
                'first': {
                    'path': 'css/output.css',
                    'files': [
                        'misc/exists.less.css',
                        'misc/exists.less',
                    ]
                }
            }
        )

    def test_resource_already_exists_raises_exception(self):
        with self.assertRaises(PipedashFileAlreadyExists) as exception:
            self.finder.list(None)
        self.assertEqual(
            str(exception.exception),
            'A resource already exists with the target path: misc/exists.less.css'
        )
