# -*- coding: utf-8 -*-

from datetime import datetime
import os

from django.conf import settings
from django.contrib.sites.models import Site
from django.contrib.syndication.views import Feed
from django.core.urlresolvers import reverse
from django.http import HttpResponse
from django.shortcuts import get_object_or_404, Http404
from django.template.defaultfilters import linebreaks, escape, capfirst
from django.utils.translation import ugettext_lazy as _

from planet.models import Post, Author, Feed, Blog

from tagging.models import Tag, TaggedItem


ITEMS_PER_FEED = getattr(settings, 'PLANET_ITEMS_PER_FEED', 50)

class BasePostFeed(Feed):

    def __init__(self, *args, **kwargs):
        super(BasePostFeed, self).__init__(*args, **kwargs)
        self.site = Site.objects.get(pk=settings.SITE_ID)
        
    def item_id(self, post):
        return post.guid
    
    def item_title(self, post):
        return post.title
    
    def item_updated(self, post):
        return post.date_modified
    
    def item_published(self, post):
        return post.date_created
    
    def item_content(self, post):
        return {"type" : "html", }, linebreaks(escape(post.content))
    
    def item_links(self, post):
        return [{"href" : reverse("planet_post_detail", args=( post.pk,))}]
    
    def item_authors(self, post):
        return [{"name" : post.author}]

class PostFeed(Feed):

    def feed_id(self):
        return reverse("planet_posts_list")
    
    def title(self):
        return _("Posts in %s") % self.site.name

    def subtitle(self):
        return _("All posts")

    def updated(self):
        qs = Post.objects.filter(feed__site=self.site)
        # We return an arbitrary date if there are no results, because there
        # must be a feed_updated field as per the Atom specifications, however
        # there is no real data to go by, and an arbitrary date can be static.
        if qs.count() == 0:
            return datetime(year=2008, month=7, day=1)
        return qs.latest('date_created').date_created

    def links(self):
        return ({'href': reverse('planet_posts_list')},)

    def items(self):
        posts_list = Post.objects.filter(feed__site=self.site
            ).order_by("-date_created")[:ITEMS_PER_FEED]
        return posts_list


class AuthorFeed(Feed):

    def get_object(self, params):
        return get_object_or_404(Author, pk=params[0], is_active=True)
    
    def feed_id(self, author):
        return reverse("planet_author_show", args=(author.pk, ))
    
    def title(self, author):
        return _("Posts by %(author_name)s - %(site_name)s") % {'author_name': author.name, 'site_name': self.site.name}

    def updated(self, author):
        qs = Post.objects.filter(feed__author=author).distinct()
        # We return an arbitrary date if there are no results, because there
        # must be a feed_updated field as per the Atom specifications, however
        # there is no real data to go by, and an arbitrary date can be static.
        if qs.count() == 0:
            return datetime(year=2008, month=7, day=1)
        return qs.latest('date_created').date_created

    def links(self, author):
        return ({'href': reverse("planet_author_show", args=(author.pk, ))},)

    def items(self, author):
        return Post.objects.filter(feed__author=author,
            ).distinct().order_by("-date_created")[:ITEMS_PER_FEED]


class BlogFeed(Feed):

    def get_object(self, params):
        return get_object_or_404(Feed, pk=params[0], is_active=True)
    
    def feed_id(self, feed):
        return reverse("planet_feed_detail", args=(feed.pk, ))
    
    def title(self, feed):
        return _("Posts in %(feed_title)s - %(site_name)s") % {'feed_title': feed.title, 'site_name':self.site.name}

    def subtitle(self, feed):
        return "%s - %s" % (feed.tagline, feed.link)

    def updated(self, feed):
        qs = Post.objects.filter(feed=feed,
            feed__site=self.site).distinct()
        # We return an arbitrary date if there are no results, because there
        # must be a feed_updated field as per the Atom specifications, however
        # there is no real data to go by, and an arbitrary date can be static.
        if qs.count() == 0:
            return datetime(year=2008, month=7, day=1)
        return qs.latest('date_created').date_created

    def links(self, feed):
        return ({'href': reverse("planet_feed_detail", args=(feed.pk, ))},)

    def items(self, feed):
        return Post.objects.filter(feed=feed,
            feed__site=self.site).distinct(
            ).order_by("-date_created")[:ITEMS_PER_FEED]


class TagFeed(Feed):

    def get_object(self, params):
        return get_object_or_404(Tag, name=params[0])
    
    def feed_id(self, tag):
        return reverse("planet_tag_detail", args=(tag.pk, ))
    
    def title(self, tag):
        return _("Posts under %(tag)s tag - %(site_name)s") % {'tag': tag, 'site_name': self.site.name}

    def updated(self, tag):
        qs = Post.objects.filter(tags__name=tag,
            feed__site=self.site).distinct()
        # We return an arbitrary date if there are no results, because there
        # must be a feed_updated field as per the Atom specifications, however
        # there is no real data to go by, and an arbitrary date can be static.
        if qs.count() == 0:
            return datetime(year=2008, month=7, day=1)
        return qs.latest('date_created').date_created

    def links(self, tag):
        return ({'href': reverse("planet_tag_detail", args=(tag.pk, ))},)

    def items(self, tag):
        return Post.objects.filter(tags__name=tag, feed__site=self.site
            ).distinct().order_by("-date_created")[:ITEMS_PER_FEED]


class AuthorTagFeed(Feed):

    def get_object(self, author_id, tag):
        self.tag = tag
        return get_object_or_404(Author, pk=author_id, is_active=True)    
    
    def feed_id(self, author):
        return reverse("planet_by_tag_author_show", args=(author.pk, self.tag))
    
    def title(self, author):
        return _("Posts by %(author_name)s under %(tag)s tag - %(site_name)s")\
            % {'author_name': author.name, 'tag': self.tag, 'site_name': self.site.name}

    def updated(self, author):
        qs = Post.objects.filter(feed__author=author,
            tags__name=self.tag).distinct()
        # We return an arbitrary date if there are no results, because there
        # must be a feed_updated field as per the Atom specifications, however
        # there is no real data to go by, and an arbitrary date can be static.
        if qs.count() == 0:
            return datetime(year=2008, month=7, day=1)
        return qs.latest('date_created').date_created

    def links(self, author):
        return ({'href': reverse("planet_by_tag_author_show", args=(author.pk, self.tag))},)

    def items(self, author):
        return Post.objects.filter(
            feed__author=author, tags__name=self.tag
            ).distinct().order_by("-date_created")[:ITEMS_PER_FEED]
