# -*- coding: utf-8 -*-

# Standard library imports
# Third party imports
from django.template import Context
from django.template import TemplateSyntaxError
from django.template.base import Template
from django.test import TestCase

# Local application / specific library imports


class TemplateTagsTestCase(TestCase):
    BBCODE_FILTER_EXPRESSIONS_TESTS = (
        (
            '{{ "[b]hello world![/b]"|bbcode }}',
            '<strong>hello world!</strong>',
        ),
        (
            '{% with "[b]Write some bbcodes![/b] The [i]current[/i] text was generated by using the bbcode filter..." as testval %}'
            '{{ testval|bbcode }}'
            '{% endwith %}',
            '<strong>Write some bbcodes!</strong> The <em>current</em> text was generated by using the bbcode filter...',
        ),
    )

    BBCODE_TAG_EXPRESSIONS_TESTS = (
        (
            '{% bbcode "This [b]one[/b] was generated using the [i][color=green]bbcode[/color][/i] template tag." %}',
            'This <strong>one</strong> was generated using the <em><span style="color:green;">bbcode</span></em> template tag.',
        ),
        (
            '{% bbcode "Hello [u]world![/u]" as renderedvar %}'
            '{{ renderedvar }}',
            'Hello <u>world!</u>',
        ),
        (
            "{% bbcode '[i]a \"small\" test[/i]' %}",
            "<em>a &quot;small&quot; test</em>",
        )
    )

    BBCODE_TAG_ERRONEOUS_EXPRESSIONS_TESTS = (
        '{% bbcode %}',
        '{% bbcode "[b]hello world![/b] noop %}'
        '{% bbcode "[b]hello world![/b] as %}'
    )

    def setUp(self):
        self.loadstatement = '{% load bbcode_tags %}'

    def test_bbcode_filter_rendering(self):
        # Run & check
        for template_content, expected_html_text in self.BBCODE_FILTER_EXPRESSIONS_TESTS:
            t = Template(self.loadstatement + template_content)
            rendered = t.render(Context())
            self.assertEqual(rendered, expected_html_text)

    def test_bbcode_tag_rendering(self):
        # Run & check
        for template_content, expected_html_text in self.BBCODE_TAG_EXPRESSIONS_TESTS:
            t = Template(self.loadstatement + template_content)
            rendered = t.render(Context())
            self.assertEqual(rendered, expected_html_text)

    def test_bbcode_erroneous_tag_should_raise(self):
        # Run & check
        for template_content in self.BBCODE_TAG_ERRONEOUS_EXPRESSIONS_TESTS:
            with self.assertRaises(TemplateSyntaxError):
                t = Template(self.loadstatement + template_content)
