Django React
============

Render and bundle React components from a Django application.

Documentation
-------------

- [Basic usage](#basic-usage)
- [Installation](#installation)
- [ReactComponent](#reactcomponent)
  - [render_to_string()](#reactcomponentrender_to_string)
  - [render_to_static_markup()](#reactcomponentrender_to_static_markup)
  - [render_js()](#reactcomponentrender_js)
  - [render_container()](#reactcomponentrender_container)
  - [render_props()](#reactcomponentrender_props)
  - [render_source()](#reactcomponentrender_source)
  - [render_init()](#reactcomponentrender_init)
- [ReactBundle](#reactbundle)
- [render_component()](#render_component)
- [Running the tests](#running-the-tests)

Basic usage
-----------

Define your component

```python
from django_react import ReactComponent

class MyComponent(ReactComponent):
    source = 'path/to/file.jsx'
```

Create an instance of your component and pass it some props

```python
my_component = MyComponent(
    some_prop='foo',
    some_other_prop=[1, 2, 3]
)
```

Render the component to its initial HTML

```html
{{ my_component.render_to_string }}
```

Render JavaScript to mount your component

```html
<script src="path/to/react.js"></script>

{{ my_component.render_js }}
```

The rendered JavaScript will automatically include:
- Your props, serialized to JSON
- Your source, which will have been JSX transformed and bundled with Webpack
- Initialization code that immediately mounts your component with React

Rendering the component to its initial markup ensures that your user will see
your content immediately and React will mount the component as soon as the 
page has loaded the JavaScript.

Installation
------------

```bash
pip install django-react
```

Add `'django_react'` to your `INSTALLED_APPS` setting

```python
INSTALLED_APPS = (
    # ...
    'django_react',
)
```

ReactComponent
--------------

Integrates server-side rendering of React components with Webpack's bundling.

```python
from django_react import ReactComponent

# Define your component
class MyComponent(ReactComponent):
    source = 'path/to/file.jsx'
    
# Create an instance of your component with some props
component = MyComponent(
  foo=1,
  bar=[1,2,3]
)

# Render the component to its initial HTML
markup = component.render_to_string()

# Render JS to mount the component with React
js = component.render_js()
```

### ReactComponent.render_to_string()

Render a component to its initial HTML. You can use this method to generate HTML
on the server and send the markup down on the initial request for faster page loads
and to allow search engines to crawl your pages for SEO purposes.

`render_to_string` takes an optional argument `wrap` which, if set to False, will
prevent the the rendered output from being wrapped in the container element
generated by the component's `render_container` method.

```html
{{ component.render_to_string }}
```

### ReactComponent.render_to_static_markup()

Similar to `ReactComponent.render_to_string`, except this doesn't create
extra DOM attributes such as `data-react-id`, that React uses internally.
This is useful if you want to use React as a simple static page generator,
as stripping away the extra attributes can save lots of bytes.

```html
{{ component.render_to_static_markup }}
```

### ReactComponent.render_js()

Renders the script elements containing the component's props, source and
initialisation. The script elements enable the browser to download your JS
assets and add interactivity to a rendered component.

`render_js` is effectively shorthand for calling each of the component's
`render_props`, `render_source`, and `render_init` methods.

```html
{{ component.render_js }}
```

### ReactComponent.render_container()

Renders a HTML element which Django React uses on the client-side to 
mount components with React. By default, `render_to_string` and `render_to_static_markup`
are wrapped in this element.

`render_container` takes an optional argument, `content` which should be a string to be
inserted within the element.

```html
{{ component.render_container }}
```

### ReactComponent.render_props()

Render the component's props as a JavaScript object.

The props will be defined within the browser's global scope under a
variable name generated by the component's `get_props_variable` method.

```
{{ component.render_props }}
```

### ReactComponent.render_source()

Render a script element pointing to the bundled source of the component.

The bundled component will be defined within the browser's global scope
under a variable name generated by the component's `get_variable` method.

```
{{ component.render_source }}
```

### ReactComponent.render_init()

Render a script element which will mount the component to the container 
created with the component's `render_container`, `render_to_string`, 
or `render_to_static_markup` methods.

```
{{ component.render_init }}
```

ReactBundle
-----------

A `django_webpack.WebpackBundle` which is configured to support loading of JSX files.

By default, a `ReactBundle` is configured to omit React from the generated bundle
and instead rely on a global variable, `window.React`. By omitting React from the bundle,
multiple components can be included in a single page without duplicating React's code.

You can extend the Webpack configuration by inheriting from ReactBundle and assigning
the subclass as the `bundle` attribute on your ReactComponent subclass.

```python
from django_react import ReactBundle, ReactComponent

class JqueryOptimisedBundle(ReactBundle):
    no_parse = ('jquery',)

class MyComponent(ReactComponent):
    bundle = JqueryOptimisedBundle
```

render_component()
------------------

Render a component to its initial HTML. You can use this method to generate HTML
on the server and send the markup down on the initial request for faster page loads
and to allow search engines to crawl your pages for SEO purposes.

Arguments:

- `path_to_source`: an absolute path to a JS or JSX file which exports the component.
- `serialized_props`: [optional] a string containing the JSON-serialized props which will
  be passed to the component.
- `to_static_markup`: [optional] a boolean indicating that React's `render_to_static_markup`
  method should be used for the rendering. Set this to True if you are rendering the component
  to static HTML and React will not be used on the client-side.

```python
import json
from django.contrib.staticfiles import finders
from django_react import render_component

path_to_source = finders.find('path/to/component.jsx')

serialized_props = json.dumps({
    'foo': 1,
    'bar': [1, 2, 3],
})

# Render the component to its initial HTML
html = render_component(path_to_source, serialized_props)
```

Running the tests
-----------------

```bash
mkvirtualenv django-react
pip install -r requirements.txt
python django_react/tests/runner.py
```
