import os
import hashlib
import json
from django.template.loader import render_to_string
from django.contrib.staticfiles import finders
from django.utils.safestring import mark_safe
from django_react.settings import REACT_EXTERNAL
from .exceptions import (
    ReactComponentCalledDirectly, ReactComponentMissingSource, SourceFileNotFound, PropSerializationError,
    ComponentBundlingError,
)
from .render import render_component
from .bundles import ReactBundle


class ReactComponent(object):
    source = None
    props = None
    variable = None
    props_variable = None
    serialized_props = None
    bundle = ReactBundle
    source_url = None

    def __init__(self, **kwargs):
        # As we use the subclass's name to generate a number of client-side
        # variables, we disallow directly calling the ReactComponent class
        if self.__class__ is ReactComponent:
            raise ReactComponentCalledDirectly('Components must inherit from ReactComponent')
        # Sanity check
        if self.get_source() is None:
            raise ReactComponentMissingSource(self)
        # All kwargs are passed to the component as props, this replicates
        # the API used in React+JSX
        self.props = kwargs

    def render_to_string(self, wrap=None):
        """
        Render a component to its initial HTML. You can use this method to generate HTML
        on the server and send the markup down on the initial request for faster page loads
        and to allow search engines to crawl you pages for SEO purposes.

        `render_to_string` takes an optional argument `wrap` which, if set to False, will
        prevent the the rendered output from being wrapped in the container element
        generated by the component's `render_container` method.

        Note that the rendered HTML will be wrapped in a container element generated by the component's
        `render_container` method.

        ```
        {{ component.render_to_string }}
        ```
        """
        rendered = render_component(
            path_to_source=self.get_path_to_source(),
            serialized_props=self.get_serialized_props(),
        )
        if wrap is False:
            return rendered
        return self.render_container(
            content=mark_safe(rendered)
        )

    def render_to_static_markup(self, wrap=None):
        """
        Similar to `ReactComponent.render_to_string`, except this doesn't create
        extra DOM attributes such as `data-react-id`, that React uses internally.
        This is useful if you want to use React as a simple static page generator,
        as stripping away the extra attributes can save lots of bytes.

        ```
        {{ component.render_to_static_markup }}
        ```
        """
        if wrap is None:
            wrap = True
        rendered = render_component(
            path_to_source=self.get_path_to_source(),
            serialized_props=self.get_serialized_props(),
            to_static_markup=True,
        )
        if wrap is False:
            return rendered
        return self.render_container(
            content=mark_safe(rendered)
        )

    def render_js(self):
        """
        Renders the script elements containing the component's props, source and
        initialisation.

        `render_js` is effectively shorthand for calling each of the component's
        `render_props`, `render_source`, and `render_init` methods.

        ```
        {{ component.render_js }}
        ```
        """
        return render_to_string('django_react/js.html', self.get_render_context(
            rendered_props=self.render_props(),
            rendered_source=self.render_source(),
            rendered_init=self.render_init(),
        ))

    def render_container(self, content=None):
        """
        Renders a HTML element with attributes which Django React uses internally to
        facilitate mounting components with React.

        The rendered element is provided with a id attribute which is generated by the component's
        `get_container_id` method.

        `render_container` takes an optional argument, `content` which should be a string to be
        inserted within the element.

        ```
        {{ component.render_container }}

        <script>
            console.log(document.getElementById('{{ component.get_container_id }}'));
        </script>
        ```
        """
        return render_to_string('django_react/container.html', self.get_render_context(
            content=content,
            container_id=self.get_container_id(),
            container_class_name=self.get_container_class_name(),
        ))

    def render_props(self):
        """
        Render the component's props as a JavaScript object.

        The props will be defined within the browser's global scope under a
        variable name generated by the component's `get_props_variable` method.

        ```
        {{ component.render_props }}

        <script>
          console.log({{ component.get_props_variable }});
        </script>
        ```
        """
        return render_to_string('django_react/props.html', self.get_render_context(
            props_variable=self.get_props_variable(),
            serialized_props=self.get_serialized_props(),
        ))

    def render_source(self):
        """
        Render a script element pointing to the bundled source of the component.

        The bundled component will be defined within the browser's global scope
        under a variable name generated by the component's `get_variable` method.

        ```
        {{ component.render_source }}

        <script>
          console.log({{ component.get_variable }});
        </script>
        ```
        """
        return render_to_string('django_react/source.html', self.get_render_context(
            source_url=self.get_source_url()
        ))

    def render_init(self):
        """
        Render a script element which will create an instance of the component and use
        React to mount it in the container created with the component's `render_container`,
        `render_to_string`, or `render_to_static_markup` methods.

        ```
        {{ component.render_init }}
        ```
        """
        return render_to_string('django_react/init.html', self.get_render_context(
            REACT_EXTERNAL=REACT_EXTERNAL,
            variable=self.get_variable(),
            props_variable=self.get_props_variable(),
            container_id=self.get_container_id(),
        ))

    def get_render_context(self, **kwargs):
        # As a convenience for template overrides, the component instance
        # is passed into the template
        context = {
            'component': self,
        }
        context.update(kwargs)
        return context

    def get_source(self):
        return self.source

    def get_path_to_source(self):
        source = self.get_source()
        path_to_source = finders.find(source)
        if not path_to_source or not os.path.exists(path_to_source):
            raise SourceFileNotFound(path_to_source)
        return path_to_source

    def get_props(self):
        return self.props

    def get_variable(self):
        if self.variable is None:
            self.variable = self.__class__.__name__
        return self.variable

    def get_container_id(self):
        return 'reactComponentContainer-{id}'.format(
            id=unicode(id(self)),
        )

    def get_container_class_name(self):
        return 'reactComponentContainer reactComponentContainer--{variable}'.format(
            variable=self.get_variable(),
        )

    def get_serialized_props(self):
        if self.serialized_props is None:
            # While rendering templates Django will silently ignore some types of exceptions,
            # so we need to intercept them and raise our own class of exception
            try:
                self.serialized_props = json.dumps(self.get_props())
            except (TypeError, AttributeError) as e:
                raise PropSerializationError(e.__class__.__name__, *e.args)
        return self.serialized_props

    def get_props_variable(self):
        if self.props_variable is None:
            serialized_props = self.get_serialized_props()
            md5 = hashlib.md5()
            md5.update(serialized_props)
            self.props_variable = '__propsFor{variable}_{hash}__'.format(
                variable=self.get_variable(),
                hash=md5.hexdigest(),
            )
        return self.props_variable

    def get_source_url(self):
        if self.source_url is None:
            bundle = self.bundle(
                entry=self.get_source(),
                library=self.get_variable(),
            )
            # While rendering templates Django will silently ignore some types of exceptions,
            # so we need to intercept them and raise our own class of exception
            try:
                self.source_url = bundle.get_url()
            except (TypeError, AttributeError) as e:
                raise ComponentBundlingError(e.__class__.__name__, *e.args)
        return self.source_url