# -*- coding: utf-8 -*-
#
#  This file is part of django-recaptcha-works.
#
#  django-recaptcha-works provides a Django form field that integrates the
#  reCaptcha service.
#
#  Development Web Site:
#    - http://www.codetrax.org/projects/django-recaptcha-works
#  Public Source Code Repository:
#    - https://source.codetrax.org/hgroot/django-recaptcha-works
#
#  Copyright 2010 George Notaras <gnot [at] g-loaded.eu>
#
#  Based on the code snippet #1644 as published on:
#    - http://djangosnippets.org/snippets/1644/
#
#  Copyright 2009-2010 Chris Beaven, http://smileychris.com/
#
#  Licensed under the BSD License.
#
#  Redistribution and use in source and binary forms, with or without modification,
#  are permitted provided that the following conditions are met:
#
#      1. Redistributions of source code must retain the above copyright notice, 
#         this list of conditions and the following disclaimer.
#      
#      2. Redistributions in binary form must reproduce the above copyright 
#         notice, this list of conditions and the following disclaimer in the
#         documentation and/or other materials provided with the distribution.
#
#      3. Neither the name of Django nor the names of its contributors may be used
#         to endorse or promote products derived from this software without
#         specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
#  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
#  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
#  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
#  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
#  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

from django import forms
from django.utils.safestring import mark_safe
from django.utils import simplejson

from recaptcha_works import settings


class RecaptchaWidget(forms.Widget):
    
    RECAPTCHA_OPTIONS_SCRIPT = u'''<script type="text/javascript">
var RecaptchaOptions = {
%s
};
</script>
'''
    
    def __init__(self, public_key=None, use_ssl=False, recaptcha_options=None):
        '''RecaptchaWidget constructor
        
        For customization information visit:
        
        http://code.google.com/apis/recaptcha/docs/customization.html
        
        The optional ``recaptcha_options`` argument can be used to override
        the default project wide ``RECAPTCHA_OPTIONS`` setting.
        
        The optional ``public_key`` argument can be used to override the
        default use of the project-wide ``RECAPTCHA_PUBLIC_KEY`` setting.
        
        The optional ``use_ssl`` argument can be used to override the default
        use of the project-wide ``RECAPTCHA_USE_SSL`` setting.
        '''
        self.options = recaptcha_options
        if recaptcha_options is None:
            self.options = settings.RECAPTCHA_OPTIONS
        
        self.proto = 'http'
        if use_ssl or settings.RECAPTCHA_USE_SSL:
            self.proto = 'https'
        
        self.public_key = public_key or settings.RECAPTCHA_PUBLIC_KEY
        
        super(RecaptchaWidget, self).__init__()

    def render(self, name, value, attrs=None):
        args = dict(public_key=self.public_key, proto=self.proto, options='')
        if self.options:
            args['options'] = self.RECAPTCHA_OPTIONS_SCRIPT % simplejson.dumps(
                self.options, indent=4).strip('{}')
        return mark_safe(settings.RECAPTCHA_HTML % args)

    def value_from_datadict(self, data, files, name):
        challenge = data.get('recaptcha_challenge_field')
        response = data.get('recaptcha_response_field')
        # This is added by utils.post_payload_add_recaptcha_remote_ip_field()
        remote_ip = data.get('recaptcha_remote_ip_field')
        return (challenge, response, remote_ip)

    def id_for_label(self, id_):
        return None

