from itertools import chain
from urllib import urlencode

from django import forms
from django.conf import settings
from django.db.models.fields import BLANK_CHOICE_DASH
from django.forms.widgets import flatatt
from django.utils.encoding import force_unicode
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext as _

PREPEND_LOOKUP_FIELD = getattr(settings, 'REFINERY_PREPEND_LOOKUP_FIELD', False)


class LinkWidget(forms.Widget):
    def __init__(self, attrs=None, choices=()):
        super(LinkWidget, self).__init__(attrs)

        self.choices = choices

    def value_from_datadict(self, data, files, name):
        value = super(LinkWidget, self).value_from_datadict(data, files, name)
        self.data = data
        return value

    def render(self, name, value, attrs=None, choices=()):
        if not hasattr(self, 'data'):
            self.data = {}
        if value is None:
            value = ''
        final_attrs = self.build_attrs(attrs)
        output = [u'<ul%s>' % flatatt(final_attrs)]
        options = self.render_options(choices, [value], name)
        if options:
            output.append(options)
        output.append('</ul>')
        return mark_safe(u'\n'.join(output))

    def render_options(self, choices, selected_choices, name):
        selected_choices = set(force_unicode(v) for v in selected_choices)
        output = []
        for option_value, option_label in chain(self.choices, choices):
            if isinstance(option_label, (list, tuple)):
                for option in option_label:
                    output.append(self.render_option(name, selected_choices, *option))
            else:
                output.append(self.render_option(name, selected_choices, option_value, option_label))
        return u'\n'.join(output)

    def render_option(self, name, selected_choices, option_value, option_label):
        # option_value = force_unicode(option_value)
        data = self.data.copy()
        if option_value is not None:
            option_value = force_unicode(option_value)
            data[name] = option_value
        else:
            if data.get(name, None) is not None:
                del data[name]
        if option_label == BLANK_CHOICE_DASH[0][1]:
            option_label = _("All")
        # data = self.data.copy()
        # data[name] = option_value
        selected = data == self.data or option_value in selected_choices
        # Make sure that data is utf-8 encoded before being passed to urlencode.
        # import types
        # data = dict([(k,v.encode('utf-8') if type(v) is types.UnicodeType else v) for (k,v) in data.items()])
        try:
            url = data.urlencode()
        except AttributeError:
            url = urlencode(data)
        return self.option_string() % {
             'attrs': selected and ' class="selected"' or '',
             'query_string': url,
             'label': force_unicode(option_label)
        }

    def option_string(self):
        return '<li><a%(attrs)s href="?%(query_string)s">%(label)s</a></li>'


class RangeWidget(forms.MultiWidget):
    def __init__(self, attrs=None):
        widgets = (forms.TextInput(attrs=attrs), forms.TextInput(attrs=attrs))
        super(RangeWidget, self).__init__(widgets, attrs)

    def decompress(self, value):
        if value:
            return [value.start, value.stop]
        return [None, None]

    def format_output(self, rendered_widgets):
        return u'-'.join(rendered_widgets)


class LookupTypeWidget(forms.MultiWidget):
    def decompress(self, value):
        if value is None:
            return [None, None]
        return value
    
    def format_output(self, rendered_widgets):
        # allow for a global setting that requests the lookup select field be displayed
        # before the primary field
        if PREPEND_LOOKUP_FIELD:
            # rendered_widgets.reverse()
            rendered_widgets = rendered_widgets[-1:] + rendered_widgets[:-1]
        return u''.join(rendered_widgets)


