# Copyright (C) 2010 Linaro Limited
#
# Author: Zygmunt Krynicki <zygmunt.krynicki@linaro.org>
#
# This file is part of django-reports.
#
# django-reports is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License version 3
# as published by the Free Software Foundation
#
# django-reports is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with django-reports.  If not, see <http://www.gnu.org/licenses/>.

from django.core.exceptions import ValidationError
from django.db import models
from django_jsonfield.models import JSONField
from django_restricted_resource.models import RestrictedResource
from linaro_json import (
    Validator as JSONValidator,
    ValidationError as JSONValidationError,
)

from django_reports.utils import resolve_entrypoints, entrypoint_for_name
from django_reports.settings import (
    INSTALLED_REPORTS,
    INSTALLED_DATA_SOURCES,
)


_REPORT_CLASSES = resolve_entrypoints(INSTALLED_REPORTS)
_DATA_SOURCE_CLASSES = resolve_entrypoints(INSTALLED_DATA_SOURCES)


class ReportConfiguration(RestrictedResource):
    """
    ReportConfiguration model contains configuration and reference to
    report implementation of a manually created report.
    """
    REPORT_CLS_CHOICES = [
        (report_cls_name, report_cls.get_report_name())
        for (report_cls_name, report_cls)
        in _REPORT_CLASSES.iteritems()]

    settings = JSONField(max_length=4096, blank=True, default={})

    name = models.CharField(max_length=128)

    report_cls = models.CharField(
        max_length=128, blank=False,
        choices=REPORT_CLS_CHOICES)

    def __unicode__(self):
        return self.name

    @models.permalink
    def get_absolute_url(self):
        return ("django_reports.views.display_report", [self.pk])

    def clean(self):
        report_cls = entrypoint_for_name(self.report_cls)
        schema = report_cls.get_settings_schema()
        try:
            JSONValidator.validate(schema, self.settings)
        except JSONValidationError as ex:
            raise ValidationError(
                u"Unable to validate configuration: %s" % str(ex))
        super(ReportConfiguration, self).clean()

    def save(self, *args, **kwargs):
        self.clean()
        super(ReportConfiguration, self).save(*args, **kwargs)
