# Copyright (C) 2010 Linaro Limited
#
# Author: Zygmunt Krynicki <zygmunt.krynicki@linaro.org>
#
# This file is part of django-reports.
#
# django-reports is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation
#
# django-reports is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with django-reports.  If not, see <http://www.gnu.org/licenses/>.

"""
Demonstration report
"""

from django import forms
from django.http import HttpResponse
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.utils.translation import ugettext as _
from linaro_json import Schema, json
from pkg_resources import resource_string

from django_reports.interfaces import IReport


class DemoSettingsForm(forms.Form):

    toggle = forms.BooleanField(
        label = _(u"Example toggle field"),
        required = False,
    )

    string = forms.CharField(
        label = _(u"Example text field"),
        required = False,
    )


class DemoReport(IReport):
    """
    Demo report that is available if no other reports are configured in
    project-wide settings.py.
    """
    _SETTINGS_SCHEMA = Schema(json.loads(resource_string(
        __name__, 'demo_config_schema.json')))

    def __init__(self, config):
        self.config = config

    @property
    def toggle(self):
        return self.config.settings.get(
            "toggle",
            Schema(self._SETTINGS_SCHEMA.properties["toggle"]).default)

    @property
    def string(self):
        return self.config.settings.get("string")

    def render(self, request):
        return render_to_response(
            "django_reports/demo_report/render.html", {
                "report": self,
                "report_config": self.config,
            }, RequestContext(request)
        )

    @classmethod
    def get_report_name(cls):
        return _(u"Demo report")

    @classmethod
    def get_settings_form(cls):
        return DemoSettingsForm

    @classmethod
    def get_settings_schema(cls):
        return cls._SETTINGS_SCHEMA

