# Copyright (C) 2010 Linaro Limited
#
# Author: Zygmunt Krynicki <zygmunt.krynicki@linaro.org>
#
# This file is part of django-reports.
#
# django-reports is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License version 3
# as published by the Free Software Foundation
#
# django-reports is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with django-reports.  If not, see <http://www.gnu.org/licenses/>.

"""
Entry point utilities
"""

from django.core.exceptions import ImproperlyConfigured


def entrypoint_for_name(name):
    """
    Return entry-point class for the specified name.
    The name syntax is regular python 'package_or_module.class'
    """
    try:
        entrypoint_module_name, entrypoint_class_name = name.rsplit('.', 1)
    except ValueError as ex:
        raise ImproperlyConfigured(
            "Unable to parse entry point name {0!r}. "
            "Expected `module.class' syntax.".format(name))
    try:
        entrypoint_module = __import__(entrypoint_module_name, fromlist=[''])
    except ImportError as ex:
        raise ImproperlyConfigured(
            "Unable to import module {0!r} required to resolve entry "
            "point: {1!r}".format(entrypoint_module_name, str(ex)))
    try:
        entrypoint_class = getattr(entrypoint_module, entrypoint_class_name)
    except AttributeError:
        raise ImproperlyConfigured(
            "Unable to access entry point {0!r} in module "
            "{1!r}".format(entrypoint_class_name, entrypoint_module))
    return entrypoint_class


def resolve_entrypoints(entrypoints):
    """
    Resolve a list of entry points and build a dictionary mapping entry
    point name to class. Each entry point is resolved using
    entrypoint_for_name()
    """
    resolved_entrypoints = {}
    for entrypoint in entrypoints:
        resolved_entrypoints[entrypoint] = entrypoint_for_name(entrypoint)
    return resolved_entrypoints
