# Copyright (C) 2010 Linaro Limited
#
# Author: Zygmunt Krynicki <zygmunt.krynicki@linaro.org>
#
# This file is part of django-reports.
#
# django-reports is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License version 3
# as published by the Free Software Foundation
#
# django-reports is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with django-reports.  If not, see <http://www.gnu.org/licenses/>.

from django import forms
from django.contrib.auth.models import Group
from django.contrib.formtools.wizard import FormWizard
from django.http import HttpResponseRedirect
from django.utils.translation import ugettext as _

from django_reports.models import ReportConfiguration


class _ReportConfigurationMetaDataMixIn(forms.Form):

    name = forms.CharField(
        label = _(u"Report name"),
        help_text = _("Report name must be unique"),
    )

    group = forms.ModelChoiceField(
        label = _(u"Group owner"),
        help_text = _(u"Members of the selected group will be able to "
                      "both view and edit this report"),
        empty_label = _("None, I'll be the owner"),
        required = False,
        queryset = Group.objects.all(),
    )

    is_public = forms.BooleanField(
        label = _(u"Make the report public"),
        help_text = _(u"If you uncheck this box only the owner (or the "
                      "members of the owning group) will be able to "
                      "display this report"),
        required = False,
    )

    def _fix_group_field(self, user):
        assert user is not None
        self.fields['group'].queryset = user.groups.all()


class ReportConfigurationUpdateForm(_ReportConfigurationMetaDataMixIn):
    pass


class ReportConfigurationCreateStepOne(_ReportConfigurationMetaDataMixIn):

    report_cls = forms.ChoiceField(
        label = _(u"Report template"),
        required = True,
        choices = ReportConfiguration.REPORT_CLS_CHOICES,
    )

    class Meta:
        fields = ('report_cls', 'name', 'owner', 'is_public')


class ReportConfigurationCreateDummyStepTwo(forms.Form):
    pass


class ReportFormCreateWizard(FormWizard):

    def __init__(self, user, form_list, initial=None):
        self._user = user
        super(ReportFormCreateWizard, self).__init__(form_list, initial)

    def done(self, request, form_list):
        basic_config = form_list[0].cleaned_data
        specific_config = form_list[1].cleaned_data
        report_config = ReportConfiguration.objects.create(
            settings = specific_config,
            name = basic_config["name"],
            report_cls = basic_config["report_cls"],
            owner = basic_config["group"] or request.user)
        return HttpResponseRedirect(report_config.get_absolute_url())

    def get_template(self, step):
        return "django_reports/reportconfiguration_create_step%d.html" % step

    def get_form(self, step, data=None):
        form = super(ReportFormCreateWizard, self).get_form(step, data)
        if step == 0:
            form._fix_group_field(self._user)
        return form
