# Copyright (C) 2010 Linaro Limited
#
# Author: Zygmunt Krynicki <zygmunt.krynicki@linaro.org>
#
# This file is part of django-reports.
#
# django-reports is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License version 3
# as published by the Free Software Foundation
#
# django-reports is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with django-reports.  If not, see <http://www.gnu.org/licenses/>.

from django.contrib.auth.decorators import login_required
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.views.generic import list_detail, create_update
from django.core.urlresolvers import reverse

from django_reports import models, forms


def report_list(request):
    accessible_reports = models.ReportConfiguration.objects.accessible_by_principal(
        request.user)
    return list_detail.object_list(
        request,
        template_object_name = "report_config",
        queryset = accessible_reports.all().order_by('name'),
    )


def display_report(request, report_id):
    accessible_reports = models.ReportConfiguration.objects.accessible_by_principal(
        request.user)
    report_config = get_object_or_404(accessible_reports, pk=report_id)
    report_cls = models._REPORT_CLASSES[report_config.report_cls]
    report = report_cls(report_config)
    return report.render(request)


@login_required
def new_report(request):
    if request.method == "POST":
        # Do a little trick and get the right report_cls to show the
        # second stage form correctly if it was set in the form already
        report_cls_name = request.POST["0-report_cls"]
        report_cls = models._REPORT_CLASSES[report_cls_name]
        form = forms.ReportFormCreateWizard(request.user, [
            forms.ReportConfigurationCreateStepOne,
            report_cls.get_settings_form(),
        ])
    else:
        form = forms.ReportFormCreateWizard(request.user, [
            forms.ReportConfigurationCreateStepOne,
            forms.ReportConfigurationCreateDummyStepTwo,
        ])
    return form(request)

@login_required
def update_report_metadata(request, report_id):
    report_config = get_object_or_404(
        models.ReportConfiguration.objects.owned_by_principal(request.user),
        pk=report_id)
    data = request.POST or {
        'name': report_config.name,
        'group': report_config.group.pk if report_config.group else '',
        'is_public': report_config.is_public
    }
    form = forms.ReportConfigurationUpdateForm(data)
    form._fix_group_field(request.user)
    if request.method == 'POST':
        if form.is_valid():
            report_config.owner = form.cleaned_data['group'] or request.user
            report_config.name = form.cleaned_data['name']
            report_config.is_public = form.cleaned_data['is_public']
            report_config.save()
            return HttpResponseRedirect(report_config.get_absolute_url())
    return render_to_response(
        "django_reports/reportconfiguration_update_metadata.html",
        {"form": form,
         "report_config": report_config},
        RequestContext(request))


@login_required
def update_report_settings(request, report_id):
    report_config = get_object_or_404(
        models.ReportConfiguration.objects.owned_by_principal(request.user),
        pk=report_id)
    report_cls = models._REPORT_CLASSES[report_config.report_cls]
    data = request.POST or report_config.settings
    form = report_cls.get_settings_form()(data)
    if request.method == 'POST':
        if form.is_valid():
            report_config.settings = form.cleaned_data
            print "New settings are", report_config.settings
            report_config.save()
            return HttpResponseRedirect(report_config.get_absolute_url())
    return render_to_response(
        "django_reports/reportconfiguration_update_settings.html",
        {"form": form,
         "report_config": report_config},
        RequestContext(request))


def delete_report(request, report_id):
    return create_update.delete_object(
        request,
        login_required = True,
        model = models.ReportConfiguration,
        object_id = report_id,
        post_delete_redirect = reverse('django_reports.views.report_list'),
        template_object_name = 'report_config',
    )
