from __future__ import unicode_literals

from selectable.base import LookupBase, ModelLookup
from selectable.compat import force_text
from selectable.exceptions import (LookupAlreadyRegistered, LookupNotRegistered,
                                    LookupInvalid)


class LookupRegistry(object):

    def __init__(self):
        self._registry = {}

    def validate(self, lookup):
        if not issubclass(lookup, LookupBase):
            raise LookupInvalid('Registered lookups must inherit from the LookupBase class')

    def register(self, lookup):
        self.validate(lookup)
        name = force_text(lookup.name())
        if name in self._registry:
            raise LookupAlreadyRegistered('The name %s is already registered' % name)
        self._registry[name] = lookup

    def unregister(self, lookup):
        self.validate(lookup)
        name = force_text(lookup.name())
        if name not in self._registry:
           raise LookupNotRegistered('The name %s is not registered' % name)
        del self._registry[name]

    def get(self, key):
        return self._registry.get(key, None)


registry = LookupRegistry()


def autodiscover():

    import copy
    from django.conf import settings
    
    try:
        from django.utils.module_loading import autodiscover_modules
    except ImportError:
        from django.utils.importlib import import_module
        from django.utils.module_loading import module_has_submodule

        def autodiscover_modules(submod, **kwargs):
            for app_name in settings.INSTALLED_APPS:
                mod = import_module(app_name)
                try:
                    before_import_registry = copy.copy(registry._registry)
                    import_module('%s.lookups' % app_name)
                except:
                    registry._registry = before_import_registry
                    if module_has_submodule(mod, 'lookups'):
                        raise

    # Attempt to import the app's lookups module.
    autodiscover_modules('lookups', register_to=registry)
