from functools import wraps
from selenium import webdriver
from selenium.common.exceptions import WebDriverException
import time
import os

from django.db import transaction
from django.core.urlresolvers import reverse
from django.test import TransactionTestCase
from django.utils.html import strip_tags

from django_selenium import settings


def wait(func):
    @wraps(func)
    def wrapper(self, *args, **kwargs):
        i = settings.SELENIUM_DRIVER_TIMEOUT
        if kwargs.has_key('timeout'):
            i = kwargs.pop('timeout')
        res = func(self, *args, **kwargs)
        while not res and i:
            time.sleep(1)
            res = func(self, *args, **kwargs)
            i-=1
        return res
    return wrapper


class MyDriver(object):
    def __init__(self):
        driver = getattr(webdriver, settings.SELENIUM_DRIVER, None)
        assert driver, "settings.SELENIUM_DRIVER contains non-existing driver"
        if driver is webdriver.Remote:
            if isinstance(settings.SELENIUM_CAPABILITY, dict):
                capability = settings.SELENIUM_CAPABILITY
            else:
                capability = getattr(webdriver.DesiredCapabilities, settings.SELENIUM_CAPABILITY, None)
                assert capability, 'settings.SELENIUM_CAPABILITY contains non-existing capability'
            self.driver = driver('http://%s:%d/wd/hub' % (settings.SELENIUM_HOST, settings.SELENIUM_PORT), capability)
        else:
            self.driver = driver()
        self.live_server_url = 'http://%s:%s' % (settings.SELENIUM_TESTSERVER_HOST , str(settings.SELENIUM_TESTSERVER_PORT))
        self.text = ''

    def __getattribute__(self, name):
        try:
            attr = object.__getattribute__(self, name)
        except AttributeError:
            attr = self.driver.__getattribute__(name)
        return attr

    def _wait_for_page_source(self):
        try:
            page_source = self.page_source
            time.sleep(1)
            while page_source!=self.page_source:
                page_source = self.page_source
                time.sleep(1)
            self.update_text()
        except WebDriverException:
            pass

    def authorize(self, username, password):
        self.open_url(reverse('login'))
        self.type_in("#id_username", username)
        self.type_in("#id_password", password)
        self.click("#login-form input[type='submit']")

    def update_text(self):
        self.text = strip_tags(unicode(self.page_source))

    def open_url(self, url):
        self.get('%s%s' % (self.live_server_url, url))
        self._wait_for_page_source()

    def click(self, selector):
        """This function also refreshes page text"""
        self.find(selector).click()
        self._wait_for_page_source()

    def click_and_wait(self, selector, newselector):
        self.click(selector)
        return self.wait_element_present(newselector)

    def is_element_present(self, selector):
        return len(self.find_elements_by_css_selector(selector)) > 0

    def is_text_present(self, text):
        return text in self.text

    def get_alert_text(self):
        alert = self.switch_to_alert()
        # Selenium can return either dict or text,
        # TODO: Need to investigate why
        try:
            text = alert.text['text']
        except TypeError:
            text = alert.text
        alert.dismiss()
        self.switch_to_default_content()
        self.update_text()
        return text

    def get_text(self, selector):
        return self.find(selector).text

    def drop_image(self, file_path, droparea_selector, append_to):
        """drop image to the element specified by selector"""
        self.execute_script("file_input = window.$('<input/>').attr({id: 'file_input', type:'file'}).appendTo('" + append_to + "');")
        self.find('#file_input').send_keys(os.path.join(os.getcwd(), file_path))
        self.execute_script('fileList = Array();fileList.push(file_input.get(0).files[0]);')
        self.execute_script("e = $.Event('drop'); e.originalEvent = {dataTransfer : { files : fileList } }; $('" + droparea_selector + "').trigger(e);")
        self.execute_script("$('#file_input').remove()")

    @wait
    def wait_for_text(self, selector, text):
        return text in self.find(selector).text

    @wait
    def wait_for_visible(self, selector, visible=True):
        return self.find(selector).is_displayed() == visible

    @wait
    def wait_element_present(self, selector, present=True):
        return self.is_element_present(selector)==present

    def get_title(self):
        return self.title

    def get_value(self, selector):
        return self.find(selector).get_attribute('value')

    def find(self, cssselector):
        return self.find_element_by_css_selector(cssselector)

    def select(self, selector, value):
        self.click(selector + (" option[value='%s']" % value))

    def type_in(self, selector, text):
        elem = self.find(selector)
        elem.clear()
        elem.send_keys(text)

class SeleniumTestCase(TransactionTestCase):

    def __getattribute__(self, name):
        try:
            attr = object.__getattribute__(self, name)
        except AttributeError:
            attr = object.__getattribute__(self,'driver').__getattribute__(name)
        return attr

    def _fixture_setup(self):
        transaction.commit_unless_managed()
        transaction.enter_transaction_management()
        transaction.managed(True)
        super(SeleniumTestCase, self)._fixture_setup()
        transaction.commit()
        transaction.leave_transaction_management()

    def setUp(self):
        import socket
        socket.setdefaulttimeout(settings.SELENIUM_TIMEOUT)
        self.driver = MyDriver()

    def tearDown(self):
        self.driver.quit()

