""" Apply-Views
"""
import StringIO
import socket
import sys
from datetime import datetime

from django.views.generic import TemplateView
from paramiko import DSSKey, SSHClient, AutoAddPolicy, \
    AuthenticationException, SSHException, RSAKey
from django.utils.translation import ugettext as _

from ..models import ApplyLog, Configuration, Key, ActionLog


class ApplyView(TemplateView):
    """
    Applies the current configuration and delivers the authorized_keys-files.

    **Template:**

    :template:`keys/apply.html`

    """

    template_name = "skd/apply.html"

    mode = "index"

    affected_hosts = set()

    ssh_messages = []

    apply_status = None

    def get_context_data(self, **kwargs):
        context = super(ApplyView, self).get_context_data(**kwargs)

        context["mode"] = self.mode

        if self.mode == "scanned":
            context["affected_hosts"] = ApplyLog.objects.all()

        if len(self.ssh_messages) > 0:
            context["log_output"] = "\n".join(self.ssh_messages)

        if self.apply_status:
            context["apply_status"] = self.apply_status

        return context

    def post(self, request, *args, **kwargs):

        if "do_scan" in request.POST:

            self.mode = "scanned"

        elif "do_apply" in request.POST:

            self.mode = "done"

            self.affected_hosts = ApplyLog.objects.all()

            public_key = Configuration.objects.get(
                key="sshkey_public"
            )

            private_key_config = Configuration.objects.get(
                key="sshkey_private"
            )

            keytype = Configuration.objects.get(
                key="sshkey_type"
            )

            if not public_key or not private_key_config or not keytype:
                raise _("Invalid configuration. Did you run setup already?")

            passphrase = request.POST.get("passphrase", "")
            if passphrase == "":
                passphrase = None

            private_key = None
            self.ssh_messages = []
            self.apply_status = "success"

            try:

                private_key_file = StringIO.StringIO(
                    str(private_key_config.value)
                )

                if keytype.value == "dsa":

                    private_key = DSSKey.from_private_key(
                        private_key_file, passphrase
                    )

                else:

                    private_key = RSAKey.from_private_key(
                        private_key_file, passphrase
                    )

                private_key_file.close()

            except SSHException:

                self.ssh_messages.append(_(
                    "Cannot open skd private key. Perhaps you specified the "
                    "wrong passphrase?"
                ))

                self.apply_status = "error"

            hosts_applied = []

            if private_key:

                for apply_line in self.affected_hosts:

                    host = apply_line.host

                    if host in hosts_applied:
                        continue

                    # Find all users, that have access to this host.

                    authorized_keys = []

                    user_keys = Key.objects.filter(
                        user__useringroup__group__usergroupinhostgroup__hostgroup__hostingroup__host__id =
                        host.id
                    )

                    for key in user_keys:
                        authorized_keys.append("# %s (%s)" % (
                            key.user.fullname,
                            key.name
                        ))
                        authorized_keys.append(key.key)

                    # Add our own public key to the keys

                    authorized_keys.append("# Generated by skd")
                    authorized_keys.append(
                        "ssh-dss %s skd" % (public_key.value,)
                    )

                    # Generate the authorized_keys file onto the server.

                    client = SSHClient()
                    client.load_system_host_keys()
                    client.set_missing_host_key_policy(AutoAddPolicy)

                    is_connected = False

                    try:

                        client.connect(
                            hostname=str(host.fqdn),
                            username=str(host.user),
                            pkey=private_key
                        )

                        is_connected = True

                    except AuthenticationException:

                        self.ssh_messages.append(_(
                            "Cannot connect to host %(name)s as user %(user)s. "
                            "Perhaps the skd-key hasn't  been added to it's "
                            "authorized_keys-file" %
                            {
                                "name": host.name,
                                "user": host.user
                            }
                        ))

                        self.apply_status = "error"

                    except SSHException, socket.error:

                        self.ssh_messages.append(_(
                            "System failure connecting to SSH host %(host)s as "
                            "user %(user)s: %(error)s" %
                            {
                                "error": sys.exc_info()[0],
                                "host": host.name,
                                "user": host.user
                            }
                        ))

                        self.apply_status = "error"

                    if is_connected:

                        try:

                            def noAscii(k):
                                return ''.join([x for x in k if ord(x)<128])

                            command = 'echo -e "%s" > ~/' \
                                      '.ssh/authorized_keys' % \
                                      (
                                          noAscii("\n".join(authorized_keys))
                                      )

                            client.exec_command(command = command)

                            self.ssh_messages.append(_(
                                "Host %(host)s with user %(user)s completed." %
                                {
                                    "host": host.name,
                                    "user": host.user
                                }
                            ))

                            # Add note

                            ActionLog(
                                timestamp=datetime.now(),
                                user=request.user,
                                action="APPLY"
                            ).save()

                            # Delete host from apply-Log.

                            hosts_applied.append(host)

                        except SSHException:

                            self.ssh_messages.append(_(
                                "Error deploying the authorized_keys-file of "
                                "host %(host)s / user %(user)s: %(error)s" %\
                                {
                                    "error": sys.exc_info()[0],
                                    "host": host.name,
                                    "user": host.user
                                }
                            ))

                            self.apply_status = "error"

                ApplyLog.objects.filter(host__in=hosts_applied).delete()

        return self.get(request, *args, **kwargs)
