"""
Hostgroup-Views
"""
from datetime import datetime

from django.core.urlresolvers import reverse, reverse_lazy
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, UpdateView, DeleteView, ListView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext as _

from ..models import HostGroup, ActionLog, Host, ApplyLog, HostInGroup, \
    UserGroupInHostGroup


class HostGroupCreateView(CreateView):
    """
    Creates a hostgroup.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/host_form.html`
    """

    model = HostGroup
    success_url = reverse_lazy("hostgroups_list")

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_HOSTGROUP",
            objectid=self.object.id
        ).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostGroupUpdateView(UpdateView):
    """
    Update a hostgroup.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/host_form.html`
    """

    model = HostGroup
    success_url = reverse_lazy("hostgroups_list")

    def form_valid(self, form):

        # Log Update

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_HOSTGROUP",
            objectid=self.object.id,
            comment=_(
                "Updated hostgroup.\n"
                "Original data:\n"
                "Name: %(name)s\n"
                "%(comment)s\n"
                "New data:\n"
                "Name: %(newname)s\n"
                "%(newcomment)s\n"
                % {
                    "name": self.object.name,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newcomment": form.data["comment"]
                }

            )
        ).save()

        # Save object

        self.object = form.save()

        return super(ModelFormMixin, self).form_valid(form)


class HostGroupDeleteView(DeleteView):
    """
    Deletes a hostgroup.

    **Context**

    ``RequestContext`

    **Template:**

    :template:`keys/hostgroup_confirm_delete.html`

    """

    model = HostGroup
    success_url = reverse_lazy("hostgroups_list")

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__id=self.object.id
        )

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_HOSTGROUP",
            comment=_(
                "Deleting hostgroup %(name)s with comment "
                "%(comment)s" %
                {
                    "name": self.object.name,
                    "comment": self.object.comment
                }
            )
        ).save()

        for host in affected_hosts:

            # Log host loosing conection

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="DELETE_HOSTGROUP_AFFECTED",
                comment=_(
                    "Host %(host)s has lost connection to the now missing "
                    "hostgroup %(name)s" %
                    {
                        "host": host.name,
                        "name": self.object.name,
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


class HostGroupListView(ListView):
    """
    Lists hostgroups a host is member of

    **Context**

    ``RequestContext``

    ``group_member``
        The group member.

    **Template:**

    :template:`keys/host_groups_list.html`

    """

    context_object_name = "host_groups"
    template_name = "skd/host_groups_list.html"

    def get_queryset(self):
        self.group_member = get_object_or_404(Host, id=self.kwargs["host"])
        return HostInGroup.objects.filter(host=self.group_member)

    def get_context_data(self, **kwargs):
        # Call the base implementation first to get a context
        context = super(HostGroupListView, self).get_context_data(**kwargs)
        # Add in the Owner
        context["group_member"] = self.group_member
        return context


class HostGroupAssignView(CreateView):
    """
    Assign a host to a hostgroup

    **Context**

    ``RequestContext``

    ``group_member``
        The group member to be.

    **Template:**

    :template:`keys/host_groups_assign.html`

    """

    template_name = "skd/host_groups_assign.html"

    def get_queryset(self):
        self.group_member = get_object_or_404(Host, id=self.kwargs["host"])
        return HostInGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(HostGroupAssignView, self).get_context_data(**kwargs)
        context["group_member"] = self.group_member
        return context

    def get_initial(self):
        return {"host": self.group_member}

    def get_success_url(self):
        return reverse(
            "hosts_groups_list",
            kwargs={
                "host": self.group_member.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_HOSTINGROUP",
            objectid=self.object.host.id,
            comment=_(
                "Assigned host %(user)s@%(host)s to group %(group)s" % {
                    "user": self.object.host.user,
                    "host": self.object.host.name,
                    "group": self.object.group.name
                }
            )
        )
        log_item.save()

        # Log affected host

        ApplyLog(host=self.object.host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostGroupUnassignView(DeleteView):
    """
    Unassign a host from a hostgroup

    **Context**

    ``RequestContext``

    ``group_member``
        The group member.

    **Template:**

    :template:`keys/host_groups_unassign_confirm.html`

    """

    template_name = "skd/host_groups_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.group_member = get_object_or_404(Host, id=self.kwargs["host"])
        return get_object_or_404(
            HostInGroup,
            id=self.kwargs["pk"],
            host=self.group_member
        )

    def get_context_data(self, **kwargs):
        context = super(HostGroupUnassignView, self).get_context_data(**kwargs)
        context["group_member"] = self.group_member
        return context

    def get_success_url(self):
        return reverse(
            "hosts_groups_list",
            kwargs={
                "host": self.group_member.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_HOSTINGROUP",
            objectid=self.object.host.id,
            comment=_(
                "Removed host %(user)s@%(host)s from group %(group)s" % {
                    "user": self.object.host.user,
                    "host": self.object.host.name,
                    "group": self.object.group.name
                }
            )
        )

        log_item.save()

        # Log affected host

        ApplyLog(host=self.object.host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())

# Hostgroup/Host


class HostGroupHostListView(ListView):
    """
    Lists the member hosts of a hostgroups

    **Context**

    ``RequestContext``

    ``hostgroup``
        The hostgroup.

    **Template:**

    :template:`keys/hostgroup_hosts_list.html`

    """

    context_object_name = "hostgroup_hosts"
    template_name = "skd/hostgroup_hosts_list.html"

    def get_queryset(self):
        self.hostgroup = get_object_or_404(
            HostGroup,
            id=self.kwargs["hostgroup"]
        )
        return HostInGroup.objects.filter(group=self.hostgroup)

    def get_context_data(self, **kwargs):
        # Call the base implementation first to get a context
        context = super(HostGroupHostListView, self).get_context_data(**kwargs)
        # Add in the Owner
        context["hostgroup"] = self.hostgroup
        return context


class HostGroupHostAssignView(CreateView):
    """
    Adds a host to a hostgroup

    **Context**

    ``RequestContext``

    ``hostgroup``
        The hostgroup.

    **Template:**

    :template:`keys/hostgroup_hosts_assign.html`

    """

    template_name = "skd/hostgroup_hosts_assign.html"

    def get_queryset(self):
        self.hostgroup = get_object_or_404(
            HostGroup,
            id=self.kwargs["hostgroup"]
        )
        return HostInGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(
            HostGroupHostAssignView,
            self).get_context_data(**kwargs)
        context["hostgroup"] = self.hostgroup
        return context

    def get_initial(self):
        return {"group": self.hostgroup}

    def get_success_url(self):
        return reverse(
            "hostgroups_hosts_list",
            kwargs={
                "hostgroup": self.hostgroup.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_HOSTINGROUP",
            objectid=self.object.host.id,
            objectid2=self.object.group.id
        )

        log_item.save()

        # Log affected host

        ApplyLog(host=self.object.host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostGroupHostUnassignView(DeleteView):
    """
    Removes a host from a hostgroup

    **Context**

    ``RequestContext``

    ``hostgroup``
        The hostgroup.

    **Template:**

    :template:`keys/hostgroup_hosts_unassign_confirm.html`

    """

    template_name = "skd/hostgroup_hosts_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.hostgroup = get_object_or_404(
            HostGroup,
            id=self.kwargs["hostgroup"]
        )
        return get_object_or_404(
            HostInGroup,
            id=self.kwargs["pk"],
            group=self.hostgroup
        )

    def get_context_data(self, **kwargs):
        context = super(
            HostGroupHostUnassignView,
            self).get_context_data(**kwargs)
        context["hostgroup"] = self.hostgroup
        return context

    def get_success_url(self):
        return reverse(
            "hostgroups_hosts_list",
            kwargs={
                "hostgroup": self.hostgroup.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_HOSTINGROUP",
            objectid=self.object.host.id,
            objectid2=self.object.group.id
        )

        log_item.save()

        # Log affected host

        ApplyLog(host=self.object.host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


# Hostgroup/Usergroup


class HostGroupUserGroupListView(ListView):
    """
    Lists the usergroups assigned to a hostgroup

    **Context**

    ``RequestContext``

    ``hostgroup``
        The hostgroup.

    **Template:**

    :template:`keys/hostgroup_usergroups_list.html`

    """

    context_object_name = "hostgroup_usergroups"
    template_name = "skd/hostgroup_usergroups_list.html"

    def get_queryset(self):
        self.hostgroup = get_object_or_404(
            HostGroup,
            id=self.kwargs["hostgroup"]
        )
        return UserGroupInHostGroup.objects.filter(hostgroup=self.hostgroup)

    def get_context_data(self, **kwargs):
        # Call the base implementation first to get a context
        context = super(
            HostGroupUserGroupListView,
            self).get_context_data(**kwargs)
        # Add in the Owner
        context["hostgroup"] = self.hostgroup
        return context


class HostGroupUserGroupAssignView(CreateView):
    """
    Assigns an usergroup to a hostgroup

    **Context**

    ``RequestContext``

    ``hostgroup``
        The hostgroup.

    **Template:**

    :template:`keys/hostgroup_usergroups_assign.html`

    """

    template_name = "skd/hostgroup_usergroups_assign.html"

    def get_queryset(self):
        self.hostgroup = get_object_or_404(
            HostGroup,
            id=self.kwargs["hostgroup"]
        )
        return UserGroupInHostGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(
            HostGroupUserGroupAssignView,
            self).get_context_data(**kwargs)
        context["hostgroup"] = self.hostgroup
        return context

    def get_initial(self):
        return {"hostgroup": self.hostgroup}

    def get_success_url(self):
        return reverse(
            "hostgroups_usergroups_list",
            kwargs={
                "hostgroup": self.hostgroup.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_USERGROUPINHOSTGROUP",
            objectid=self.object.usergroup.id,
            comment=_(
                "Assigned usergroup %(usergroup)s to hostgroup %(hostgroup)s" %
                {
                    "usergroup": self.object.usergroup.name,
                    "hostgroup": self.object.hostgroup.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__id=self.object.hostgroup.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="ASSIGN_USERGROUPINHOSTGROUP_AFFECTED",
                objectid=self.object.usergroup.id,
                comment=_(
                    "Usergroup %(usergroup)s has now access to host "
                    "%(host)s" %
                    {
                        "usergroup": self.object.usergroup.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostGroupUserGroupUnassignView(DeleteView):
    """
    Unassigns a hostgroup from an usergroup

    **Context**

    ``RequestContext``

    ``hostgroup``
        The hostgroup.

    **Template:**

    :template:`keys/hostgroup_usergroups_unassign_confirm.html`

    """

    template_name = "skd/hostgroup_usergroups_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.hostgroup = get_object_or_404(
            HostGroup,
            id=self.kwargs["hostgroup"]
        )
        return get_object_or_404(
            UserGroupInHostGroup,
            id=self.kwargs["pk"],
            hostgroup=self.hostgroup
        )

    def get_context_data(self, **kwargs):
        context = super(
            HostGroupUserGroupUnassignView,
            self).get_context_data(**kwargs)
        context["hostgroup"] = self.hostgroup
        return context

    def get_success_url(self):
        return reverse(
            "hostgroups_usergroups_list",
            kwargs={
                "hostgroup": self.hostgroup.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_USERGROUPINHOSTGROUP",
            objectid=self.object.usergroup.id,
            comment=_(
                "Removed association of usergroup %(usergroup)s with hostgroup"
                " %(hostgroup)s" %
                {
                    "usergroup": self.object.usergroup.name,
                    "hostgroup": self.object.hostgroup.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__id=self.object.hostgroup.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UNASSIGN_USERGROUPINHOSTGROUP_AFFECTED",
                objectid=self.object.usergroup.id,
                comment=_(
                    "Usergroup %(usergroup)s looses access to host %(host)s" %
                    {
                        "usergroup": self.object.usergroup.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())
