"""
User views
"""
from datetime import datetime
from django.core.urlresolvers import reverse_lazy

from django.http import HttpResponseRedirect
from django.views.generic import CreateView, UpdateView, DeleteView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext as _

from ..models import User, ActionLog, Host, ApplyLog


class UserCreateView(CreateView):
    """
    Creates a user.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/user_form.html`
    """

    model = User
    success_url = reverse_lazy("users_list")

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_USER",
            objectid=self.object.id
        ).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserUpdateView(UpdateView):
    """
    Updates a user.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/user_form.html`
    """

    model = User
    success_url = reverse_lazy("users_list")

    def form_valid(self, form):

        # Log Update

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_USER",
            objectid=self.object.id,
            comment = _(
                "Updated user:\n"
                "Original data: \n"
                "%(name)s (%(fullname)s)\n"
                "%(comment)s\n"
                "\n"
                "New data: \n"
                "%(newname)s (%(newfullname)s)\n"
                "%(newcomment)s\n"
                % {
                    "name": self.object.name,
                    "fullname": self.object.fullname,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newfullname": form.data["fullname"],
                    "newcomment": form.data["comment"]
                }
            )
        ).save()

        # Save object

        self.object = form.save()

        return super(ModelFormMixin, self).form_valid(form)



class UserDeleteView(DeleteView):
    """
    Deletes a user.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/user_confirm_delete.html`

    """

    model = User
    success_url = reverse_lazy("users_list")

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__usergroupinhostgroup__usergroup__useringroup__user__id =
            self.object.id
        )

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_USER",
            comment=_(
                "Deleting user %(name)s (%(fullname)s) with comment %"
                "(comment)s" %
                {
                    "name": self.object.name,
                    "fullname": self.object.fullname,
                    "comment": self.object.comment
                }
            )
        ).save()

        for host in affected_hosts:

            # Log Deletion

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="DELETE_USER_AFFECTED",
                comment=_(
                    "User %(name)s looses access to %(host)s" %
                    {
                        "name": self.object.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())