from django.db import models
import django.contrib.auth.models
from django.utils.translation import ugettext_lazy as _


class User(models.Model):
    """
    Stores a SSH-user record
    """

    name = models.CharField(
        _("Username"),
        max_length=200,
        blank=False
    )
    fullname = models.CharField(
        _("Full name of User"),
        max_length=200
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("User")
        verbose_name_plural = _("Users")
        permissions = (
            ("list_users", _("Can list all users")),
        )

    def __unicode__(self):
        return "%s (%s)" % (self.name, self.fullname)


class Key(models.Model):
    """
    Stores keys of a SSH-User. Relates to :model:`User`
    """

    user = models.ForeignKey(
        User
    )
    name = models.CharField(
        _("Key name"),
        max_length=200,
        blank=False
    )
    key = models.TextField(
        _("Public key"),
        help_text=_(
            "Public key in form 'ssh-dss AAA(...)'"
            "or 'ssh-rsa AAA(...)'. DSA-Keys are recommended."
        ),
        blank=False
    )
    comment = models.TextField(blank=True)

    class Meta:
        verbose_name = _("Key")
        verbose_name_plural = _("Keys")
        permissions = (
            ("list_users_keys", _("Can list all keys of every user")),
        )

    def __unicode__(self):
        return "%s - %s" % (self.user.name, self.name)


class UserGroup(models.Model):
    """
    Groups of users.
    """
    name = models.CharField(
        _("Name of usergroup"),
        max_length=200,
        blank=False
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Usergroup")
        verbose_name_plural = _("Usergroups")

        permissions = (
            ("list_usergroups", _("Can list all usergroups")),
        )

    def __unicode__(self):
        return self.name


class UserInGroup(models.Model):
    """
    Binds users to usergroups. Relates to :model:`User` and
    :model:`UserGroup
    """
    group = models.ForeignKey(
        UserGroup
    )
    user = models.ForeignKey(
        User
    )

    class Meta:
        verbose_name = _("User-Group Assignment")
        verbose_name_plural = _("User-Group Assignments")
        unique_together = ("group", "user")
        permissions = (
            (
                "list_users_in_usergroups",
                _("Can see which users are in which usergroups")
            ),
        )

    def __unicode__(self):
        return "%s <=> %s" % (self.group.name, self.user.name)


class Host(models.Model):
    """
    SSH-aware hosts.
    """

    name = models.CharField(
        _("Internal name of host"),
        max_length=200,
        blank=False
    )
    fqdn = models.CharField(
        _("FQDN"),
        max_length=200,
        blank=False,
        help_text=_("Resolvable, fully qualified domain name of the physical "
                    "host (or IP-address)")
    )
    user = models.CharField(
        _("Local username"),
        max_length=200,
        blank=False,
        help_text=_(
            "When updating a username, <strong>skd</strong> will "
            "<strong>not</strong> automatically delete the keys from "
            "the old username. You have to do this manually!"
        )
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Host")
        verbose_name_plural = _("Hosts")
        permissions = (
            ("list_hosts", _("Can list all hosts")),
            ("setup_host", _("Write skd public key to host"))
        )

    def __unicode__(self):
        return "%s (%s)" % (self.name, self.fqdn)


class HostGroup(models.Model):
    """
    Groups of hosts.
    """

    name = models.CharField(
        _("Name of hostgroup"),
        max_length=200,
        blank=False
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Hostgroup")
        verbose_name_plural = _("Hostgroups")
        permissions = (
            ("list_hostgroups", _("Can list all hostgroups")),
        )

    def __unicode__(self):
        return self.name


class HostInGroup(models.Model):
    """
    Binds hosts to hostgroups. Relates to :model:`Host` and
    :model:`HostGroup`
    """

    group = models.ForeignKey(
        HostGroup
    )
    host = models.ForeignKey(
        Host
    )

    class Meta:
        verbose_name = _("Host-Group Assignment")
        verbose_name_plural = _("Host-Group Assignments")
        unique_together = ("group", "host")
        permissions = (
            (
                "list_hosts_in_hostgroups",
                _("Can see which hosts are in which hostgroups")
            ),
        )
    
    def __unicode__(self):
        return "%s <=> %s" % (self.group.name, self.host.name)


class UserGroupInHostGroup(models.Model):
    """
    Assigns usergroups to hostgroups, meaning that all users in the
    usergroup can login into all hosts in the hostgroup.

    Relates to :model:`UserGroup` and :model:`HostGroup`
    """
    usergroup = models.ForeignKey(
        UserGroup
    )
    hostgroup = models.ForeignKey(
        HostGroup
    )

    class Meta:
        verbose_name = _("Usergroup-Hostgroup Assignment")
        verbose_name_plural = _("Usergroup-Hostgroup Assignments")
        unique_together = ("usergroup", "hostgroup")
        permissions = (
            (
                "list_usergroups_in_hostgroups",
                _("Can see which hostgroups are assigned to which usergroups")
            ),
        )

    def __unicode__(self):
        return "%s <=> %s" % (self.usergroup.name, self.hostgroup.name)


class ActionLog(models.Model):
    """
    Logs changes in the UI for an audit trail.

    Has a relation to :model:`auth.User`
    """

    timestamp = models.DateTimeField(
        _("Timestamp"),
        null=False
    )
    user = models.ForeignKey(
        django.contrib.auth.models.User
    )
    action = models.CharField(
        _("Action"),
        max_length=100,
        blank=False
    )
    objectid = models.IntegerField(
        _("Assigned object"),
        null=True
    )
    objectid2 = models.IntegerField(
        _("Assigned object #2"),
        null=True
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Action Log entry")
        verbose_name_plural = _("Action log entries")
        permissions = (
            (
                "list_actionlog",
                _("The user can see the action log.")
            ),
        )


class UserMap(models.Model):
    """
    Map of user username and host username
    """
    user = models.ForeignKey(User)
    host = models.ForeignKey(Host)
    username = models.CharField(
        _("Username on host"),
        max_length=200,
        blank=False
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Username map")
        verbose_name_plural = _("Username maps")

        unique_together = ("user", "host", )

        permissions = (
            (
                "list_usermaps",
                _("Can see usermaps")
            ),
        )

    def __unicode__(self):
        return self.username


class ApplyLog(models.Model):
    """
    Logs hosts, that are affected by changes in the UI and the
    corresponding actionlog-entries.
    These hosts are taken into account, when the user clicks on "Apply".
    """

    host = models.ForeignKey(
        Host
    )
    log = models.ForeignKey(
        ActionLog
    )

    class Meta:
        verbose_name = _("Apply-Log Entry")
        verbose_name_plural = _("Apply-Log Entries")
        permissions = (
            (
                "can_apply",
                _("The user can apply key-deployment")
            ),
        )


class Configuration(models.Model):
    """
    Internal configuration of skd.

    Currently available keys:

    ssh_key_private: Private SSH-Key blob in base64
    ssh_key_public: Public SSH-Key blob in base64
    """

    key = models.CharField(
        _("Key"),
        max_length=100,
        blank=False,
        unique=True
    )
    value = models.TextField(
        _("Value"),
        blank = True
    )