""" Apply-Views
"""
import StringIO
import socket
import sys
from datetime import datetime

from django.views.generic import TemplateView
from paramiko import DSSKey, SSHClient, AutoAddPolicy, \
    AuthenticationException, SSHException, RSAKey
from django.utils.translation import ugettext as _

from ..models import ApplyLog, Configuration, Key, ActionLog, User, UserMap


class ApplyView(TemplateView):
    """
    Applies the current configuration and delivers the authorized_keys-files.

    **Template:**

    :template:`keys/apply.html`

    """

    template_name = "skd/apply.html"

    mode = None

    affected_hosts = set()

    ssh_messages = []

    apply_status = None

    def get_context_data(self, **kwargs):
        context = super(ApplyView, self).get_context_data(**kwargs)

        context["mode"] = self.mode

        if not self.mode:
            context["affected_hosts"] = ApplyLog.objects.all()

        if len(self.ssh_messages) > 0:
            context["log_output"] = "\n".join(self.ssh_messages)

        if self.apply_status:
            context["apply_status"] = self.apply_status

        return context

    def post(self, request, *args, **kwargs):

        if "do_apply" in request.POST:

            # We should apply the changes.

            # Add note about that.

            ActionLog(
                timestamp=datetime.now(),
                user=request.user,
                action="APPLY"
            ).save()

            # Do some initialisation

            self.mode = "applied"

            self.affected_hosts = ApplyLog.objects.all()

            public_key = Configuration.objects.get(
                key="sshkey_public"
            )

            private_key_config = Configuration.objects.get(
                key="sshkey_private"
            )

            keytype = Configuration.objects.get(
                key="sshkey_type"
            )

            if not public_key or not private_key_config or not keytype:
                raise _("Invalid configuration. Did you run setup already?")

            passphrase = request.POST.get("passphrase", "")
            if passphrase == "":
                passphrase = None

            private_key = None
            self.ssh_messages = []
            self.apply_status = "success"

            try:

                # Generate private key to use

                private_key_file = StringIO.StringIO(
                    str(private_key_config.value)
                )

                if keytype.value == "dsa":

                    private_key = DSSKey.from_private_key(
                        private_key_file, passphrase
                    )

                else:

                    private_key = RSAKey.from_private_key(
                        private_key_file, passphrase
                    )

                private_key_file.close()

            except SSHException:

                self.ssh_messages.append(_(
                    "Cannot open skd private key. Perhaps you specified the "
                    "wrong passphrase?"
                ))

                self.apply_status = "error"

            hosts_applied = []

            if private_key:

                client = SSHClient()
                client.load_system_host_keys()
                client.set_missing_host_key_policy(AutoAddPolicy)

                for apply_line in self.affected_hosts:

                    host = apply_line.host

                    if host in hosts_applied:
                        continue

                    workload = []

                    # Find all users, that have access to this host.

                    user_keys = Key.objects.filter(
                        **{
                            "user__useringroup__group__"
                            "usergroupinhostgroup__hostgroup__"
                            "hostingroup__host__id": host.id
                        }
                    )

                    if not user_keys:

                        # This seems to be an old entry. Obviously, no one
                        # has access to the host anymore. Skip it.

                        hosts_applied.append(host)
                        self.ssh_messages.append(
                            _(
                                "Removing seemingly orphaned host %(host)s "
                                "from workload" %
                                {
                                    'host': host.name
                                }
                            )
                        )
                        continue

                    if host.user == '*':
                        # This host is a wildcard host.

                        # Build up workload using the usernames of the
                        # assigned users and include optional username mappings

                        assigned_users = User.objects.filter(
                            **{
                                "useringroup__group__usergroupinhostgroup__"
                                "hostgroup__hostingroup__host__id": host.id
                            }
                        )

                        for assigned_user in assigned_users:

                            namemaps = UserMap.objects.filter(
                                user=assigned_user,
                                host=host
                            )

                            assigned_username = assigned_user.name

                            if namemaps:

                                assigned_username = namemaps[0].username

                            # Find all users, that also share this username,
                            # either inside their user data or via a username
                            # mapping.

                            # Namemaps

                            all_users = []

                            same_namemap = UserMap.objects.filter(
                                host=host,
                                username=assigned_username
                            )

                            for mapping in same_namemap:
                                all_users.append(mapping.user)

                            # user data

                            same_users = User.objects.filter(
                                **{
                                    "name": assigned_username,
                                    "useringroup__group__"
                                    "usergroupinhostgroup__hostgroup__"
                                    "hostingroup__host__id": host.id
                                }
                            )

                            for found_user in same_users:
                                all_users.append(found_user)

                            user_keys = Key.objects.filter(
                                user__in=all_users
                            )

                            workload.append(
                                {
                                    'host': host,
                                    'user': assigned_username,
                                    'user_keys': user_keys
                                }
                            )

                    else:

                        workload = [
                            {
                                'host': host,
                                'user': host.user,
                                'user_keys': user_keys
                            }
                        ]

                    error_in_workload = False

                    for step in workload:

                        # build up authorized_keys-filecontent

                        authorized_keys = []

                        for key in user_keys:
                            authorized_keys.append("# %s (%s)" % (
                                key.user.fullname,
                                key.name
                            ))
                            authorized_keys.append(key.key)

                        # Add our own public key to the keys

                        authorized_keys.append("# Generated by skd")
                        authorized_keys.append(
                            "ssh-dss %s skd" % (public_key.value,)
                        )

                        # Connect to the server

                        is_connected = False

                        try:

                            client.connect(
                                hostname=str(step['host'].fqdn),
                                username=str(step['user']),
                                pkey=private_key
                            )

                            is_connected = True

                        except AuthenticationException:

                            self.ssh_messages.append(_(
                                "Cannot connect to host %(name)s as user "
                                "%(user)s. Perhaps the skd-key hasn't  been "
                                "added to it's authorized_keys-file" %
                                {
                                    "name": step['host'].name,
                                    "user": step['user']
                                }
                            ))

                            self.apply_status = "error"

                        except SSHException, socket.error:

                            self.ssh_messages.append(_(
                                "System failure connecting to SSH host "
                                "%(host)s as user %(user)s: %(error)s" %
                                {
                                    "error": sys.exc_info()[0],
                                    "host": step['host'].name,
                                    "user": step['user']
                                }
                            ))

                            self.apply_status = "error"

                        if is_connected:

                            try:

                                # Deploy the authorized_keys file onto the
                                # server.

                                def noAscii(k):
                                    return ''.join(
                                        [x for x in k if ord(x) < 128]
                                    )

                                command = 'echo -e "%s" > ~/' \
                                          '.ssh/authorized_keys' % \
                                          (
                                              noAscii(
                                                  "\n".join(authorized_keys)
                                              )
                                          )

                                client.exec_command(command=command)

                                self.ssh_messages.append(_(
                                    "Host %(host)s with user %(user)s "
                                    "completed." %
                                    {
                                        "host": step['host'].name,
                                        "user": step['user']
                                    }
                                ))

                            except SSHException:

                                self.ssh_messages.append(_(
                                    "Error deploying the authorized_keys-file "
                                    "of host %(host)s / user %(user)s: "
                                    "%(error)s" %\
                                    {
                                        "error": sys.exc_info()[0],
                                        "host": host.name,
                                        "user": host.user
                                    }
                                ))

                                self.apply_status = "error"

                                error_in_workload = True

                            if not error_in_workload:

                                # All went well, delete host from apply-Log.

                                hosts_applied.append(host)

                            client.close()

                # Remove succesful hosts from applylog

                ApplyLog.objects.filter(host__in=hosts_applied).delete()

        return self.get(request, *args, **kwargs)
