"""
Host-Views
"""
from datetime import datetime
import socket
import sys
from django.core.urlresolvers import reverse_lazy

from django.http import HttpResponseRedirect
from django.views.generic import CreateView, UpdateView, DeleteView, \
    TemplateView
from django.views.generic.edit import ModelFormMixin
from paramiko import SSHClient, AutoAddPolicy, AuthenticationException, \
    SSHException
from django.utils.translation import ugettext as _

from ..models import Host, ActionLog, ApplyLog, Configuration


class HostCreateView(CreateView):
    """
    Creates a host.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/host_form.html`
    """

    model = Host
    success_url = reverse_lazy("hosts_list")

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_HOST",
            objectid=self.object.id
        ).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostUpdateView(UpdateView):
    """
    Updates a host.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/host_form.html`
    """

    model = Host
    success_url = reverse_lazy("hosts_list")

    def form_valid(self, form):

        # Log Update

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_HOST",
            objectid=self.object.id,
            comment=_(
                "Updated host.\n"
                "Original data:\n"
                "Name: %(name)s\n"
                "FQDN: %(fqdn)s\n"
                "User: %(user)s\n"
                "%(comment)s\n"
                "\n"
                "New data:\n"
                "Name: %(newname)s\n"
                "FQDN: %(newfqdn)s\n"
                "User: %(newuser)s\n"
                "%(comment)s"
                % {
                    "name": self.object.name,
                    "fqdn": self.object.fqdn,
                    "user": self.object.user,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newfqdn": form.data["fqdn"],
                    "newuser": form.data["user"],
                    "newcomment": form.data["comment"]
                }
            )
        )

        log_item.save()

        # Save object

        self.object = form.save()

        # Log affected host

        ApplyLog(host=self.object, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostDeleteView(DeleteView):
    """
    Deletes a host.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/host_confirm_delete.html`

    """

    model = Host
    success_url = reverse_lazy("hosts_list")

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_HOST",
            comment=_(
                "Deleting host %(user)s@%(name)s (%(fqdn)s) with comment %"
                "(comment)s" %
                {
                    "name": self.object.name,
                    "fqdn": self.object.fqdn,
                    "user": self.object.user,
                    "comment": self.object.comment
                }
            )
        ).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


# Setup


class HostSetupView(TemplateView):
    """
    Copies the skd public key into the host after a password authentication.

    **Template:**

    :template:`keys/setup.html

    """

    template_name = "skd/host_setup.html"
    ssh_message = ""

    def get_context_data(self, **kwargs):
        context = super(HostSetupView, self).get_context_data(**kwargs)

        context["host"] = Host.objects.get(id=kwargs["host"])
        context["sshkey_public"] = Configuration.objects.get(
            key="sshkey_public"
        )
        context["ssh_message"] = self.ssh_message

        return context

    def post(self, request, *args, **kwargs):

        if request.POST.has_key("do_setup"):

            password = request.POST["password"]
            host = Host.objects.get(id=request.POST["host"])

            sshkey_public = Configuration.objects.get(
                key="sshkey_public"
            )

            client = SSHClient()
            client.load_system_host_keys()
            client.set_missing_host_key_policy(AutoAddPolicy)

            self.ssh_message = ""

            is_connected = False

            try:

                client.connect(
                    hostname=str(host.fqdn),
                    username=str(host.user),
                    password=str(password)
                )

                is_connected = True

            except AuthenticationException:

                self.ssh_message = \
                    _("Cannot connect to host. Perhaps the password is wrong")

            except SSHException, socket.error:

                self.ssh_message = _(
                    "System failure connecting to SSH host: "
                    "%(error)s" % {"error": sys.exc_info()[0]}
                )

            if is_connected:

                try:

                    command = 'echo "ssh-dss %s skd" >> ~/' \
                              '.ssh/authorized_keys' % \
                              (sshkey_public.value, )

                    client.exec_command(command=command)

                    self.ssh_message = _("Host is set up.")

                except SSHException:

                    self.ssh_message = _(
                        "Error adding my public key to the "
                        "authorized_keys-file: %(error)s" %
                        {"error": sys.exc_info()[0]}
                    )

        return self.get(request, *args, **kwargs)