"""
Usergroup-Views
"""
from datetime import datetime

from django.core.urlresolvers import reverse, reverse_lazy
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, UpdateView, DeleteView, ListView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext as _

from ..models import UserGroup, ActionLog, Host, ApplyLog, User, \
    UserInGroup, UserGroupInHostGroup


class UserGroupCreateView(CreateView):
    """
    Creates a usergroup.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/usergroup_form.html`
    """

    model = UserGroup
    success_url = reverse_lazy("usergroups_list")

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_USERGROUP",
            objectid=self.object.id
        ).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserGroupUpdateView(UpdateView):
    """
    Updates a usergroup.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/usergroup_form.html`
    """

    model = UserGroup
    success_url = reverse_lazy("usergroups_list")

    def form_valid(self, form):

        # Log Update

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_USERGROUP",
            objectid=self.object.id,
            comment=_(
                "Updated usergroup.\n"
                "Original data:\n"
                "Name: %(name)s\n"
                "%(comment)s\n"
                "New data:\n"
                "Name: %(newname)s\n"
                "%(newcomment)s\n"
                % {
                    "name": self.object.name,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newcomment": form.data["comment"]
                }
            )
        ).save()

        # Save object

        self.object = form.save()

        return super(ModelFormMixin, self).form_valid(form)


class UserGroupDeleteView(DeleteView):
    """
    Deletes a usergroup.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`keys/usergroup_confirm_delete.html`

    """

    model = UserGroup
    success_url = reverse_lazy("usergroups_list")

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_USERGROUP",
            comment=_(
                "Deleting usergroup %(name)s with comment %(comment)s" %
                {
                    "name": self.object.name,
                    "comment": self.object.comment
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__usergroupinhostgroup__usergroup__id=
            self.object.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="DELETE_USERGROUP_AFFECTED",
                comment=_(
                    "Users in now missing usergroup %(name)s loose access to "
                    "host %(host)s" %
                    {
                        "name": self.object.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


class UserGroupListView(ListView):
    """
    Lists groups a user is member of.

    **Context**

    ``RequestContext``

    ``group_member``
        The member of the groups.

    **Template:**

    :template:`keys/user_groups_list.html`

    """

    context_object_name = "user_groups"
    template_name = "skd/user_groups_list.html"

    def get_queryset(self):
        self.group_member = get_object_or_404(User, id=self.kwargs["user"])
        return UserInGroup.objects.filter(user=self.group_member)

    def get_context_data(self, **kwargs):
        # Call the base implementation first to get a context
        context = super(UserGroupListView, self).get_context_data(**kwargs)
        # Add in the Owner
        context["group_member"] = self.group_member
        return context


class UserGroupAssignView(CreateView):
    """
    Assigns a user to a new group.

    **Context**

    ``RequestContext``

    ``group_member``
        The member of the group to be.

    **Template:**

    :template:`keys/user_groups_assign.html`

    """

    template_name = "skd/user_groups_assign.html"

    def get_queryset(self):
        self.group_member = get_object_or_404(User, id=self.kwargs["user"])
        return UserInGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(UserGroupAssignView, self).get_context_data(**kwargs)
        context["group_member"] = self.group_member
        return context

    def get_initial(self):
        return {"user": self.group_member}

    def get_success_url(self):
        return reverse(
            "users_groups_list",
            kwargs={
                "user": self.group_member.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_USERINGROUP",
            objectid=self.object.user.id,
            comment=_(
                "Assigned user %(user)s to usergroup %(usergroup)s" %
                {
                    "user": self.object.user.name,
                    "usergroup": self.object.group.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__usergroupinhostgroup__usergroup__useringroup__group__id =
            self.object.group.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="ASSIGN_USERINGROUP_AFFECTED",
                objectid=self.object.user.id,
                comment=_(
                    "User %(user)s now has access to host %(host)s" %
                    {
                        "user": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()



        return super(ModelFormMixin, self).form_valid(form)


class UserGroupUnassignView(DeleteView):
    """
    Removes a member from a group.

    **Context**

    ``RequestContext``

    ``group_member``
        The member of the group.

    **Template:**

    :template:`keys/user_groups_unassign_confirm.html`

    """

    template_name = "skd/user_groups_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.group_member = get_object_or_404(User, id=self.kwargs["user"])
        return get_object_or_404(
            UserInGroup,
            id=self.kwargs["pk"],
            user=self.group_member
        )

    def get_context_data(self, **kwargs):
        context = super(UserGroupUnassignView, self).get_context_data(**kwargs)
        context["group_member"] = self.group_member
        return context

    def get_success_url(self):
        return reverse(
            "users_groups_list",
            kwargs={
                "user": self.group_member.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_USERINGROUP",
            objectid=self.object.user.id,
            comment=_(
                "Removed user %(user)s from usergroup %(usergroup)s" %
                {
                    "user": self.object.user.name,
                    "usergroup": self.object.group.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__usergroupinhostgroup__usergroup__useringroup__id =
            self.object.group.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UNASSIGN_USERINGROUP_AFFECTED",
                objectid=self.object.user.id,
                comment=_(
                    "User %(user)s looses access to host %(host)s" %
                    {
                        "user": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


# Usergroup/User


class UserGroupUserListView(ListView):
    """
    Lists members of a usergroup.

    **Context**

    ``RequestContext``

    ``usergroup``
        The usergroup.

    **Template:**

    :template:`keys/usergroup_users_list.html`

    """

    context_object_name = "usergroup_users"
    template_name = "skd/usergroup_users_list.html"

    def get_queryset(self):
        self.usergroup = get_object_or_404(
            UserGroup,
            id=self.kwargs["usergroup"]
        )
        return UserInGroup.objects.filter(group=self.usergroup)

    def get_context_data(self, **kwargs):
        # Call the base implementation first to get a context
        context = super(UserGroupUserListView, self).get_context_data(**kwargs)
        # Add in the Owner
        context["usergroup"] = self.usergroup
        return context


class UserGroupUserAssignView(CreateView):
    """
    Adds a new member to an usergroup.

    **Context**

    ``RequestContext``

    ``usergroup``
        The usergroup.

    **Template:**

    :template:`keys/usergroup_users_assign.html`

    """

    template_name = "skd/usergroup_users_assign.html"

    def get_queryset(self):
        self.usergroup = get_object_or_404(
            UserGroup,
            id=self.kwargs["usergroup"]
        )
        return UserInGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(
            UserGroupUserAssignView,
            self).get_context_data(**kwargs)
        context["usergroup"] = self.usergroup
        return context

    def get_initial(self):
        return {"group": self.usergroup}

    def get_success_url(self):
        return reverse(
            "usergroups_users_list",
            kwargs={
                "usergroup": self.usergroup.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_USERINGROUP",
            objectid=self.object.user.id,
            objectid2=self.object.group.id
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__usergroupinhostgroup__usergroup__useringroup__group__id =
            self.object.group.id
        )

        for host in affected_hosts:
            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="ASSIGN_USERINGROUP_AFFECTED",
                objectid=self.object.user.id,
                comment=_(
                    "User %(user)s now has access to host %(host)s" %
                    {
                        "user": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserGroupUserUnassignView(DeleteView):
    """
    Removes a member from an usergroup.

    **Context**

    ``RequestContext``

    ``usergroup``
        The usergroup.

    **Template:**

    :template:`keys/usergroup_users_unassign_confirm.html`

    """

    template_name = "skd/usergroup_users_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.usergroup = get_object_or_404(
            UserGroup,
            id=self.kwargs["usergroup"]
        )
        return get_object_or_404(
            UserInGroup,
            id=self.kwargs["pk"],
            group=self.usergroup
        )

    def get_context_data(self, **kwargs):
        context = super(
            UserGroupUserUnassignView,
            self).get_context_data(**kwargs)
        context["usergroup"] = self.usergroup
        return context

    def get_success_url(self):
        return reverse(
            "usergroups_users_list",
            kwargs={
                "usergroup": self.usergroup.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_USERINGROUP",
            objectid=self.object.user.id,
            objectid2=self.object.group.id,
            comment=_(
                "Removed user %(user)s from usergroup %(usergroup)s" %
                {
                    "user": self.object.user.name,
                    "usergroup": self.object.group.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__usergroupinhostgroup__usergroup__useringroup__id =
            self.object.group.id
        )

        for host in affected_hosts:
            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UNASSIGN_USERINGROUP_AFFECTED",
                objectid=self.object.user.id,
                comment=_(
                    "User %(user)s looses access to host %(host)s" %
                    {
                        "user": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


# Usergroup/Hostgroup


class UserGroupHostGroupListView(ListView):
    """
    Lists hostgroup assignments of an usergroup

    **Context**

    ``RequestContext``

    ``usergroup``
        The usergroup.

    **Template:**

    :template:`keys/usergroup_hostgroups_list.html`

    """

    context_object_name = "usergroup_hostgroups"
    template_name = "skd/usergroup_hostgroups_list.html"

    def get_queryset(self):
        self.usergroup = get_object_or_404(
            UserGroup,
            id=self.kwargs["usergroup"]
        )
        return UserGroupInHostGroup.objects.filter(usergroup=self.usergroup)

    def get_context_data(self, **kwargs):
        # Call the base implementation first to get a context
        context = super(
            UserGroupHostGroupListView,
            self).get_context_data(**kwargs)
        # Add in the Owner
        context["usergroup"] = self.usergroup
        return context


class UserGroupHostGroupAssignView(CreateView):
    """
    Assigns a new hostgroup to a usergroup

    **Context**

    ``RequestContext``

    ``usergroup``
        The usergroup.

    **Template:**

    :template:`keys/usergroup_hostgroups_assign.html`

    """

    template_name = "skd/usergroup_hostgroups_assign.html"

    def get_queryset(self):
        self.usergroup = get_object_or_404(
            UserGroup,
            id=self.kwargs["usergroup"]
        )
        return UserGroupInHostGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(
            UserGroupHostGroupAssignView,
            self).get_context_data(**kwargs)
        context["usergroup"] = self.usergroup
        return context

    def get_initial(self):
        return {"usergroup": self.usergroup}

    def get_success_url(self):
        return reverse(
            "usergroups_hostgroups_list",
            kwargs={
                "usergroup": self.usergroup.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_USERGROUPINHOSTGROUP",
            objectid=self.object.usergroup.id,
            comment=_(
                "Assigned usergroup %(usergroup)s to hostgroup %(hostgroup)s" %
                {
                    "usergroup": self.object.usergroup.name,
                    "hostgroup": self.object.hostgroup.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__id=self.object.hostgroup.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="ASSIGN_USERGROUPINHOSTGROUP_AFFECTED",
                objectid=self.object.usergroup.id,
                comment=_(
                    "Usergroup %(usergroup)s has now access to host "
                    "%(host)s" %
                    {
                        "usergroup": self.object.usergroup.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserGroupHostGroupUnassignView(DeleteView):
    """
    Unassigns a hostgroup from a usergroup

    **Context**

    ``RequestContext``

    ``usergroup``
        The usergroup.

    **Template:**

    :template:`keys/usergroup_hostgroups_unassign_confirm.html`

    """

    template_name = "skd/usergroup_hostgroups_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.usergroup = get_object_or_404(
            UserGroup,
            id=self.kwargs["usergroup"]
        )
        return get_object_or_404(
            UserGroupInHostGroup,
            id=self.kwargs["pk"],
            usergroup=self.usergroup
        )

    def get_context_data(self, **kwargs):
        context = super(
            UserGroupHostGroupUnassignView,
            self).get_context_data(**kwargs)
        context["usergroup"] = self.usergroup
        return context

    def get_success_url(self):
        return reverse(
            "usergroups_hostgroups_list",
            kwargs={
                "usergroup": self.usergroup.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_USERGROUPINHOSTGROUP",
            objectid=self.object.usergroup.id,
            comment=_(
                "Removed association of usergroup %(usergroup)s with hostgroup"
                " %(hostgroup)s" %
                {
                    "usergroup": self.object.usergroup.name,
                    "hostgroup": self.object.hostgroup.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            hostingroup__group__id=self.object.hostgroup.id
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UNASSIGN_USERGROUPINHOSTGROUP_AFFECTED",
                objectid=self.object.usergroup.id,
                comment=_(
                    "Usergroup %(usergroup)s looses access to host %(host)s" %
                    {
                        "usergroup": self.object.usergroup.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())