""" skd-Models
"""
from datetime import datetime

from django.db import models
import django.contrib.auth.models
from django.utils.translation import ugettext_lazy as _
import socket
from paramiko import SSHClient, AutoAddPolicy, AuthenticationException, \
    SSHException, DSSKey, RSAKey
import sys
import StringIO


class ApplyLogManager(models.Manager):

    """ ApplyLog-Handling
    """

    def apply_config(self, user, timestamp, passphrase):

        """ Apply the configuration

        This will check the Affectedlog and generates proper authorized_keys
         file for each host affected.

        """

        retval = {
            "status": "success",
            "messages": []
        }

        # Add note about that.

        ActionLog(
            timestamp=timestamp,
            user=user,
            action="APPLY"
        ).save()

        # Do some initialisation

        affected_hosts = self.all()

        public_key = Configuration.objects.get(
            key="sshkey_public"
        )

        private_key_config = Configuration.objects.get(
            key="sshkey_private"
        )

        keytype = Configuration.objects.get(
            key="sshkey_type"
        )

        if not public_key or not private_key_config or not keytype:
            raise _("Invalid configuration. Did you run setup already?")

        if passphrase == "":
            passphrase = None

        private_key = None

        try:

            # Generate private key to use

            private_key_file = StringIO.StringIO(
                str(private_key_config.value)
            )

            if keytype.value == "dsa":

                # noinspection PyTypeChecker
                private_key = DSSKey.from_private_key(
                    private_key_file, passphrase
                )

            else:

                # noinspection PyTypeChecker
                private_key = RSAKey.from_private_key(
                    private_key_file, passphrase
                )

            private_key_file.close()

        except SSHException:

            retval['ssh_messages'].append(_(
                "Cannot open skd private key. Perhaps you specified the "
                "wrong passphrase?"
            ))

            retval['status'] = "error"

        hosts_applied = []

        if private_key:

            client = SSHClient()
            client.load_system_host_keys()
            client.set_missing_host_key_policy(AutoAddPolicy)

            for apply_line in affected_hosts:

                host = apply_line.host

                if host in hosts_applied:
                    continue

                workload = []

                # Find all users, that have access to this host.

                user_keys = UserKey.objects.filter(
                    **{
                        "user__useringroup__group__"
                        "usergroupinhostgroup__hostgroup__"
                        "hostingroup__host__id": host.id
                    }
                )

                if not user_keys:

                    # This seems to be an old entry. Obviously, no one
                    # has access to the host anymore. Skip it.

                    hosts_applied.append(host)
                    retval['ssh_messages'].append(
                        _(
                            "Removing seemingly orphaned host %(host)s "
                            "from workload" %
                            {
                                'host': host.name
                            }
                        )
                    )
                    continue

                if host.user == '*':
                    # This host is a wildcard host.

                    # Build up workload using the usernames of the
                    # assigned users and include optional username mappings

                    assigned_users = User.objects.filter(
                        **{
                            "useringroup__group__usergroupinhostgroup__"
                            "hostgroup__hostingroup__host__id": host.id
                        }
                    )

                    for assigned_user in assigned_users:

                        namemaps = UserMap.objects.filter(
                            user=assigned_user,
                            host=host
                        )

                        assigned_username = assigned_user.name

                        if namemaps:

                            assigned_username = namemaps[0].username

                        # Find all users, that also share this username,
                        # either inside their user data or via a username
                        # mapping.

                        # Namemaps

                        all_users = []

                        same_namemap = UserMap.objects.filter(
                            host=host,
                            username=assigned_username
                        )

                        for mapping in same_namemap:
                            all_users.append(mapping.user)

                        # User data

                        same_users = User.objects.filter(
                            **{
                                "name": assigned_username,
                                "useringroup__group__"
                                "usergroupinhostgroup__hostgroup__"
                                "hostingroup__host__id": host.id
                            }
                        )

                        for found_user in same_users:
                            all_users.append(found_user)

                        user_keys = UserKey.objects.filter(
                            user__in=all_users
                        )

                        workload.append(
                            {
                                'host': host,
                                'user': assigned_username,
                                'user_keys': user_keys
                            }
                        )

                else:

                    # No wildcard host, just us.

                    workload = [
                        {
                            'host': host,
                            'user': host.user,
                            'user_keys': user_keys
                        }
                    ]

                error_in_workload = False

                for step in workload:

                    # Build up authorized_keys-filecontent

                    authorized_keys = []

                    for key in user_keys:
                        authorized_keys.append("# %s (%s)" % (
                            key.user.fullname,
                            key.name
                        ))
                        authorized_keys.append(key.key)

                    # Add our own public key to the keys

                    authorized_keys.append("# Generated by skd")
                    authorized_keys.append(
                        public_key.value
                    )

                    # Connect to the server

                    is_connected = False

                    try:

                        client.connect(
                            hostname=str(step['host'].fqdn),
                            username=str(step['user']),
                            pkey=private_key
                        )

                        is_connected = True

                    except AuthenticationException:

                        retval['ssh_messages'].append(_(
                            "Cannot connect to host %(name)s as user "
                            "%(user)s. Perhaps the skd-key hasn't  been "
                            "added to it's authorized_keys-file" %
                            {
                                "name": step['host'].name,
                                "user": step['user']
                            }
                        ))

                        retval['status'] = "error"

                    except (SSHException, socket.error, socket.gaierror):

                        retval['ssh_messages'].append(_(
                            "System failure connecting to SSH host "
                            "%(host)s as user %(user)s: %(error)s" %
                            {
                                "error": sys.exc_info()[1],
                                "host": step['host'].name,
                                "user": step['user']
                            }
                        ))

                        retval['status'] = "error"

                    if is_connected:

                        try:

                            # Deploy the authorized_keys file onto the
                            # server.

                            def noAscii(k):
                                return ''.join(
                                    [x for x in k if ord(x) < 128]
                                )

                            command = 'echo -e "%s" > ~/' \
                                      '.ssh/authorized_keys' % \
                                      (
                                          noAscii(
                                              "\n".join(authorized_keys)
                                          )
                                      )

                            client.exec_command(command=command)

                            retval['ssh_messages'].append(_(
                                "Host %(host)s with user %(user)s "
                                "completed." %
                                {
                                    "host": step['host'].name,
                                    "user": step['user']
                                }
                            ))

                        except SSHException:

                            retval['ssh_messages'].append(_(
                                "Error deploying the authorized_keys-file "
                                "of host %(host)s / user %(user)s: "
                                "%(error)s" %
                                {
                                    "error": sys.exc_info()[1],
                                    "host": host.name,
                                    "user": host.user
                                }
                            ))

                            retval['status'] = "error"

                            error_in_workload = True

                        if not error_in_workload:

                            # All went well, delete host from apply-Log.

                            hosts_applied.append(host)

                        client.close()

            # Remove succesful hosts from applylog

            self.filter(host__in=hosts_applied).delete()

        return retval


class ActionLogManager(models.Manager):

    """ Enhanced handling of actionlog
    """

    def clean_actionlog(self, user):

        """ Clean actionlog and log that.
        """

        self.all().delete()

        # Add note about that

        self.model(
            timestamp=datetime.now(),
            user=user,
            action="DELETE_ACTIONLOG"
        ).save()


class UserKeyManager(models.Manager):

    """ Handling of User keys
    """

    def keygen(self, keyname, keytype, bits, passphrase):

        """ Generates a private/public keypair and returns it.
        """

        # Initialisations and sanity checks

        retval = {"generation_messages": "", "generation_status": "success"}

        if keytype not in ['rsa', 'dsa']:
            retval["generation_messages"] = _("Invalid keytype: %s" % keytype)
            retval["generation_status"] = "error"

        if bits not in [1024, 2048, 3072, 4096]:
            retval["generation_messages"] = \
                _("Invalid number of bits: %s" % bits)
            retval["generation_status"] = "error"

        if keytype == "dsa" and bits != 1024:
            retval["generation_messages"] = _("DSA only supports 1024 bits.")
            retval["generation_status"] = "error"

        if retval["generation_status"] == "success":

            # Generate private key

            if keytype == "rsa":

                key = RSAKey.generate(bits=bits)
                typestring = "ssh-rss "

            else:

                key = DSSKey.generate(bits=bits)
                typestring = "ssh-dss "

            # Format public key

            keystring = "%s %s %s" % (
                typestring,
                key.get_base64(),
                keyname
            )

            retval["public_key"] = keystring

            tmp = StringIO.StringIO()

            if passphrase == "":
                passphrase = None

            key.write_private_key(tmp, passphrase)

            retval["private_key"] = tmp.getvalue()

        return retval


class User(models.Model):
    """
    Stores a SSH-user record
    """

    name = models.CharField(
        _("Username"),
        max_length=200,
        blank=False
    )
    fullname = models.CharField(
        _("Full name of User"),
        max_length=200
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("User")
        verbose_name_plural = _("Users")
        permissions = (
            ("list_users", _("Can list all users")),
        )

    def __unicode__(self):
        return "%s (%s)" % (self.name, self.fullname)


class UserKey(models.Model):
    """
    Stores keys of a SSH-User. Relates to :model:`User`
    """

    objects = UserKeyManager()

    user = models.ForeignKey(
        User
    )
    name = models.CharField(
        _("Key name"),
        max_length=200,
        blank=False
    )
    key = models.TextField(
        _("Public key"),
        help_text=_(
            "Public key in form 'ssh-dss AAA(...)'"
            "or 'ssh-rsa AAA(...)'. DSA-Keys are recommended."
        ),
        blank=False
    )
    comment = models.TextField(blank=True)

    class Meta:
        verbose_name = _("Key")
        verbose_name_plural = _("Keys")
        permissions = (
            ("list_users_keys", _("Can list all keys of every user")),
        )

    def __unicode__(self):
        return "%s - %s" % (self.user.name, self.name)


class UserGroup(models.Model):
    """
    Groups of users.
    """
    name = models.CharField(
        _("Name of usergroup"),
        max_length=200,
        blank=False
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Usergroup")
        verbose_name_plural = _("Usergroups")

        permissions = (
            ("list_usergroups", _("Can list all usergroups")),
        )

    def __unicode__(self):
        return self.name


class UserInGroup(models.Model):
    """
    Binds users to usergroups. Relates to :model:`User` and
    :model:`UserGroup
    """
    group = models.ForeignKey(
        UserGroup
    )
    user = models.ForeignKey(
        User
    )

    class Meta:
        verbose_name = _("User-Group Assignment")
        verbose_name_plural = _("User-Group Assignments")
        unique_together = ("group", "user")
        permissions = (
            (
                "list_users_in_usergroups",
                _("Can see which users are in which usergroups")
            ),
        )

    def __unicode__(self):
        return "%s <=> %s" % (self.group.name, self.user.name)


class Host(models.Model):
    """
    SSH-aware hosts.
    """

    name = models.CharField(
        _("Internal name of host"),
        max_length=200,
        blank=False
    )
    fqdn = models.CharField(
        _("FQDN"),
        max_length=200,
        blank=False,
        help_text=_("Resolvable, fully qualified domain name of the physical "
                    "host (or IP-address)")
    )
    user = models.CharField(
        _("Local username"),
        max_length=200,
        blank=False,
        help_text=_(
            "When updating a username, <strong>skd</strong> will "
            "<strong>not</strong> automatically delete the keys from "
            "the old username. You have to do this manually!"
        )
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Host")
        verbose_name_plural = _("Hosts")
        permissions = (
            ("list_hosts", _("Can list all hosts")),
            ("setup_host", _("Write skd public key to host"))
        )

    def setup_host(self, password):

        """
        Setup a host based on password authentication
        """

        retval = {
            "status": "success",
            "ssh_message": ""
        }

        sshkey_public = Configuration.objects.get(
            key="sshkey_public"
        )

        client = SSHClient()
        client.load_system_host_keys()
        client.set_missing_host_key_policy(AutoAddPolicy)

        is_connected = False

        try:

            # Connect to host using supplied password

            client.connect(
                hostname=str(self.fqdn),
                username=str(self.user),
                password=str(password)
            )

            is_connected = True

        except AuthenticationException:

            retval["status"] = "error"

            retval["ssh_message"] = \
                _("Cannot connect to host. Perhaps the password is wrong")

        except (SSHException, socket.error, socket.gaierror):

            retval["status"] = "error"

            retval["ssh_message"] = _(
                "System failure connecting to SSH host: "
                "%(error)s" % {"error": sys.exc_info()[1]}
            )

        if is_connected:

            try:

                # Add our public key to the authorized_keys-file

                command = 'echo "%s" >> ~/.ssh/authorized_keys' \
                          % sshkey_public.value

                client.exec_command(command=command)

                retval["ssh_message"] = _("Host is set up.")

            except SSHException:

                retval["status"] = "error"

                retval["ssh_message"] = _(
                    "Error adding my public key to the "
                    "authorized_keys-file: %(error)s" %
                    {"error": sys.exc_info()[1]}
                )

        return retval

    def __unicode__(self):
        return "%s (%s)" % (self.name, self.fqdn)


class HostGroup(models.Model):
    """
    Groups of hosts.
    """

    name = models.CharField(
        _("Name of hostgroup"),
        max_length=200,
        blank=False
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Hostgroup")
        verbose_name_plural = _("Hostgroups")
        permissions = (
            ("list_hostgroups", _("Can list all hostgroups")),
        )

    def __unicode__(self):
        return self.name


class HostInGroup(models.Model):
    """
    Binds hosts to hostgroups. Relates to :model:`Host` and
    :model:`HostGroup`
    """

    group = models.ForeignKey(
        HostGroup
    )
    host = models.ForeignKey(
        Host
    )

    class Meta:
        verbose_name = _("Host-Group Assignment")
        verbose_name_plural = _("Host-Group Assignments")
        unique_together = ("group", "host")
        permissions = (
            (
                "list_hosts_in_hostgroups",
                _("Can see which hosts are in which hostgroups")
            ),
        )

    def __unicode__(self):
        return "%s <=> %s" % (self.group.name, self.host.name)


class UserGroupInHostGroup(models.Model):
    """
    Assigns usergroups to hostgroups, meaning that all users in the
    usergroup can login into all hosts in the hostgroup.

    Relates to :model:`UserGroup` and :model:`HostGroup`
    """
    usergroup = models.ForeignKey(
        UserGroup
    )
    hostgroup = models.ForeignKey(
        HostGroup
    )

    class Meta:
        verbose_name = _("Usergroup-Hostgroup Assignment")
        verbose_name_plural = _("Usergroup-Hostgroup Assignments")
        unique_together = ("usergroup", "hostgroup")
        permissions = (
            (
                "list_usergroups_in_hostgroups",
                _("Can see which hostgroups are assigned to which usergroups")
            ),
        )

    def __unicode__(self):
        return "%s <=> %s" % (self.usergroup.name, self.hostgroup.name)


class ActionLog(models.Model):
    """
    Logs changes in the UI for an audit trail.

    Has a relation to :model:`auth.User`
    """

    objects = ActionLogManager()

    timestamp = models.DateTimeField(
        _("Timestamp"),
        null=False
    )
    user = models.ForeignKey(
        django.contrib.auth.models.User
    )
    action = models.CharField(
        _("Action"),
        max_length=100,
        blank=False
    )
    objectid = models.IntegerField(
        _("Assigned object"),
        null=True
    )
    objectid2 = models.IntegerField(
        _("Assigned object #2"),
        null=True
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Action Log entry")
        verbose_name_plural = _("Action log entries")
        permissions = (
            (
                "list_actionlog",
                _("The user can see the action log.")
            ),
        )
        ordering = ["-timestamp"]


class UserMap(models.Model):
    """
    Map of user username and host username
    """
    user = models.ForeignKey(User)
    host = models.ForeignKey(Host)
    username = models.CharField(
        _("Username on host"),
        max_length=200,
        blank=False
    )
    comment = models.TextField(
        _("Comment"),
        blank=True
    )

    class Meta:
        verbose_name = _("Username map")
        verbose_name_plural = _("Username maps")

        unique_together = ("user", "host", )

        permissions = (
            (
                "list_usermaps",
                _("Can see usermaps")
            ),
        )

    def __unicode__(self):
        return self.username


class ApplyLog(models.Model):
    """
    Logs hosts, that are affected by changes in the UI and the
    corresponding actionlog-entries.
    These hosts are taken into account, when the user clicks on "Apply".
    """

    objects = ApplyLogManager()

    host = models.ForeignKey(
        Host
    )
    log = models.ForeignKey(
        ActionLog
    )

    class Meta:
        verbose_name = _("Apply-Log Entry")
        verbose_name_plural = _("Apply-Log Entries")
        permissions = (
            (
                "can_apply",
                _("The user can apply key-deployment")
            ),
        )


class Configuration(models.Model):
    """
    Internal configuration of skd.

    Currently available keys:

    ssh_key_private: Private SSH-Key blob in base64
    ssh_key_public: Public SSH-Key blob in base64
    """

    key = models.CharField(
        _("Key"),
        max_length=100,
        blank=False,
        unique=True
    )
    value = models.TextField(
        _("Value"),
        blank = True
    )