"""
skd Templatetags specific to listview

Every tag method is called with the specific template object and a context
defined as "list_context" class variable in the specific list view.
"""

import re
from django import template
from django.core.urlresolvers import reverse
from django.utils.translation import ugettext as _


register = template.Library()


def actionlogize(value):
    """
    Converts an ActionLog-Actionkey into a translateable, human string
    """

    # Dictionary of Actionlog-Tag => Human readable text

    dictionary = {

        "DELETE_ACTIONLOG": _("Deleted the action-Log."),
        "APPLY": _("Applied pending changes."),
        "CREATE_USER": _("Created a user."),
        "UPDATE_USER": _("Updated a user."),
        "DELETE_USER": _("Deleted a user."),
        "DELETE_USER_AFFECTED": _("Deleted a user. (affected host)"),
        "CREATE_KEY": _("Created a key."),
        "CREATE_KEY_AFFECTED": _("Created a key. (affected host)"),
        "UPDATE_KEY": _("Updated a key."),
        "UPDATE_KEY_AFFECTED": _("Updated a key. (affected host)"),
        "DELETE_KEY_FROM_USER": _("Deleted a key from a user."),
        "DELETE_KEY_FROM_USER_AFFECTED": _("Deleted a key from a user. "
                                           "(affected host)"),
        "CREATE_USERGROUP": _("Created a usergroup."),
        "UPDATE_USERGROUP": _("Updated a usergroup."),
        "DELETE_USERGROUP": _("Deleted a usergroup."),
        "DELETE_USERGROUP_AFFECTED": _("Deleted a usergroup. (affected host)"),
        "ASSIGN_USERINGROUP": _("Assigned a user to a usergroup."),
        "ASSIGN_USERINGROUP_AFFECTED": _("Assigned a user to a usergroup. "
                                         "(affected host)"),
        "UNASSIGN_USERINGROUP": _("Removed a user from a usergroup."),
        "UNASSIGN_USERINGROUP_AFFECTED": _("Removed a user from a usergroup. "
                                           "(affected host)"),
        "ASSIGN_USERGROUPINHOSTGROUP": _("Associated a hostgroup with an "
                                         "usergroup."),
        "ASSIGN_USERGROUPINHOSTGROUP_AFFECTED": _("Associated a hostgroup"
                                                  " with an usergroup. ("
                                                  "affected host)"),
        "UNASSIGN_USERGROUPINHOSTGROUP": _("Removed the association of a "
                                           "hostgroup with an usergroup."),
        "UNASSIGN_USERGROUPINHOSTGROUP_AFFECTED": _("Removed the association of"
                                                    " a hostgroup with an "
                                                    "usergroup. (affected "
                                                    "host)"),
        "CREATE_HOST": _("Created a host."),
        "UPDATE_HOST": _("Updated a host."),
        "DELETE_HOST": _("Deleted a host."),
        "CREATE_HOSTGROUP": _("Created a hostgroup."),
        "UPDATE_HOSTGROUP": _("Updated a hostgroup."),
        "DELETE_HOSTGROUP": _("Deleted a hostgroup."),
        "DELETE_HOSTGROUP_AFFECTED": _("Deleted a hostgroup. (affected host)"),
        "ASSIGN_HOSTINGROUP": _("Assigned a host to a hostgroup."),
        "UNASSIGN_HOSTINGROUP": _("Removed a host from a hostgroup."),
        "CREATE_USERNAMEMAP": _("Created usermap"),
        "CREATE_USERMAP_AFFECTED": _("Created usermap. (affected host)"),
        "UPDATE_USERMAP": _("Updated usermap."),
        "UPDATE_USERMAP_AFFECTED": _("Updated usermap. (affected host)"),
        "DELETE_USERMAP": _("Deleted usermap."),
        "DELETE_USERMAP_AFFECTED": _("Deleted usermap. (affected host)")

    }

    if value in dictionary:
        return dictionary[value]

    else:
        return value


def get_actionobject(action, object_id, object_id2, user_perms):

    """Return the url reference for a specific action
    """

    # Dictionary of Actionlog tags => URL-Parameters

    dictionary = {

        "CREATE_USER": {
            "view": "users_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_user"
        },
        "UPDATE_USER": {
            "view": "users_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_user"
        },
        "CREATE_KEY": {
            "view": "users_keys_edit",
            "params": {
                "pk": object_id,
                "user": object_id2
            },
            "need_right": "skd.change_userkey"
        },
        "UPDATE_KEY": {
            "view": "users_keys_edit",
            "params": {
                "pk": object_id,
                "user": object_id2
            },
            "need_right": "skd.change_userkey"
        },
        "CREATE_USERGROUP": {
            "view": "usergroups_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_usergroup"
        },
        "UPDATE_USERGROUP": {
            "view": "usergroups_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_usergroup"
        },
        "ASSIGN_USERINGROUP": {
            "view": "users_usergroups_list",
            "params": {
                "user": object_id
            },
            "need_right": "skd.list_users_in_usergroups"
        },
        "UNASSIGN_USERINGROUP": {
            "view": "users_usergroups_list",
            "params": {
                "user": object_id
            },
            "need_right": "skd.list_users_in_usergroups"
        },
        "ASSIGN_USERGROUPINHOSTGROUP": {
            "view": "usergroups_hostgroups_list",
            "params": {
                "usergroup": object_id
            },
            "need_right": "skd.list_usergroups_in_hostgroups"
        },
        "UNASSIGN_USERGROUPINHOSTGROUP": {
            "view": "usergroups_hostgroups_list",
            "params": {
                "usergroup": object_id
            },
            "need_right": "skd.list_usergroups_in_hostgroups"
        },
        "CREATE_HOST": {
            "view": "hosts_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_host"
        },
        "UPDATE_HOST": {
            "view": "hosts_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_host"
        },
        "CREATE_HOSTGROUP": {
            "view": "hostgroups_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_hostgroup"
        },
        "UPDATE_HOSTGROUP": {
            "view": "hostgroups_edit",
            "params": {
                "pk": object_id
            },
            "need_right": "skd.change_hostgroup"
        },
        "ASSIGN_HOSTINGROUP": {
            "view": "hosts_hostgroups_list",
            "params": {
                "host": object_id
            },
            "need_right": "skd.list_hosts_in_hostgroups"
        },
        "UNASSIGN_HOSTINGROUP": {
            "view": "hosts_hostgroups_list",
            "params": {
                "host": object_id
            },
            "need_right": "skd.list_hosts_in_hostgroups"
        }

    }

    affected_action = re.match("^(.*)_AFFECTED$", action)

    if affected_action:

        # For affected actions, return the original action

        action = affected_action.group(1)

    if action in dictionary:

        if dictionary[action]["need_right"] in user_perms:

            return reverse(
                dictionary[action]["view"],
                kwargs=dictionary[action]["params"]
            )

    return None


@register.assignment_tag()
def get_row_title(template_object, context):

    """Get row title

    The returned string is used for each row as a tooltip.
    """

    object_type = type(template_object).__name__

    if object_type == "UserInGroup":

        if context == "user":

            return template_object.group.comment

        elif context == "usergroup":

            return template_object.user.comment

    if object_type == "HostInGroup":

        if context == "hostgroup":

            return template_object.host.comment

    elif object_type == "UserGroupInHostGroup":

        if context == "hostgroup":
            return template_object.usergroup.comment

        elif context == "usergroup":

            return template_object.hostgroup.comment

    else:

        return template_object.comment


@register.assignment_tag()
def get_row_columns(template_object, context):

    """Get column data

    Returns the data that should be displayed in the template for each row
    """

    object_type = type(template_object).__name__

    if object_type == "User":

        return [
            template_object.name,
            template_object.fullname
        ]

    elif object_type == "UserKey":

        return [
            template_object.name
        ]

    elif object_type == "UserMap":

        return [
            template_object.host.name,
            template_object.username
        ]

    elif object_type == "UserInGroup":

        if context == "user":

            return [
                template_object.group.name
            ]

        elif context == "usergroup":

            return [
                template_object.user.name
            ]

    elif object_type == "UserGroup":

        return [
            template_object.name
        ]

    elif object_type == "Host":

        return [
            template_object.name,
            template_object.fqdn,
            template_object.user
        ]

    elif object_type == "HostGroup":

        return [
            template_object.name
        ]

    elif object_type == "HostInGroup":

        if context == "host":

            return [template_object.group.name]

        elif context == "hostgroup":

            return [
                template_object.host.name,
                template_object.host.fqdn,
                template_object.host.user
            ]

    elif object_type == "UserGroupInHostGroup":

        if context == "hostgroup":

            return [
                template_object.usergroup.name
            ]

        elif context == "usergroup":

            return [
                template_object.hostgroup.name
            ]

    elif object_type == "ActionLog":

        return [
            template_object.timestamp,
            template_object.user,
            _(actionlogize(template_object.action))
        ]

    return None


@register.assignment_tag(takes_context=True)
def get_row_actions(context, template_object, view_context):

    """Get actions for a row

    get custom actions per each row. Returns a list of dictionaries having
    the keys "title" and "url".
    """

    object_type = type(template_object).__name__

    user_perms = context["user"].get_group_permissions()
    retval = []

    if object_type == "User":

        if "skd.change_user" in user_perms:

            retval.append(
                {
                    "title": _("Edit"),
                    "url": reverse(
                        "users_edit",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.delete_user" in user_perms:

            retval.append(
                {
                    "title": _("Delete"),
                    "url": reverse(
                        "users_delete",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.list_users_keys" in user_perms:

            retval.append(
                {
                    "title": _("Keys"),
                    "url": reverse(
                        "users_keys_list",
                        kwargs={
                            "user": template_object.id
                        }
                    )
                }
            )

        if "skd.list_usermaps" in user_perms:

            retval.append(
                {
                    "title": _("Mapping"),
                    "url": reverse(
                        "users_map_list",
                        kwargs={
                            "user": template_object.id
                        }
                    )
                }
            )

        if "skd.list_usergroups" in user_perms:

            retval.append(
                {
                    "title": _("Groups"),
                    "url": reverse(
                        "users_usergroups_list",
                        kwargs={
                            "user": template_object.id
                        }
                    )
                }
            )

    elif object_type == "UserKey":

        if "skd.change_userkey" in user_perms:

            retval.append(
                {
                    "title": _("Edit"),
                    "url": reverse(
                        "users_keys_edit",
                        kwargs={
                            "user": template_object.user.id,
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.delete_userkey" in user_perms:

            retval.append(
                {
                    "title": _("Delete"),
                    "url": reverse(
                        "users_keys_delete",
                        kwargs={
                            "user": template_object.user.id,
                            "pk": template_object.id
                        }
                    )
                }
            )

    elif object_type == "UserMap":

        if "skd.change_usermap" in user_perms:

            retval.append(
                {
                    "title": _("Edit"),
                    "url": reverse(
                        "users_map_edit",
                        kwargs={
                            "user": template_object.user.id,
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.delete_usermap" in user_perms:

            retval.append(
                {
                    "title": _("Delete"),
                    "url": reverse(
                        "users_map_delete",
                        kwargs={
                            "user": template_object.user.id,
                            "pk": template_object.id
                        }
                    )
                }
            )

    elif object_type == "UserInGroup":

        if "skd.delete_useringroup" in user_perms:

            if view_context == "user":

                retval.append(
                    {
                        "title": _("Unassign"),
                        "url": reverse(
                            "users_usergroups_unassign",
                            kwargs={
                                "user": template_object.user.id,
                                "pk": template_object.id
                            }
                        )
                    }
                )

            elif view_context == "usergroup":

                retval.append(
                    {
                        "title": _("Unassign"),
                        "url": reverse(
                            "usergroups_users_unassign",
                            kwargs={
                                "usergroup": template_object.group.id,
                                "pk": template_object.user.id
                            }
                        )
                    }
                )

    elif object_type == "UserGroup":

        if "skd.change_usergroup" in user_perms:

            retval.append(
                {
                    "title": _("Edit"),
                    "url": reverse(
                        "usergroups_edit",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.delete_usergroup" in user_perms:

            retval.append(
                {
                    "title": _("Delete"),
                    "url": reverse(
                        "usergroups_delete",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.list_users_in_usergroups" in user_perms:

            retval.append(
                {
                    "title": _("Members"),
                    "url": reverse(
                        "usergroups_users_list",
                        kwargs={
                            "usergroup": template_object.id
                        }
                    )
                }
            )

        if "skd.list_usergroups_in_hostgroups" in user_perms:

            retval.append(
                {
                    "title": _("Assigned hostgroups"),
                    "url": reverse(
                        "usergroups_hostgroups_list",
                        kwargs={
                            "usergroup": template_object.id
                        }
                    )
                }
            )

    elif object_type == "Host":

        if "skd.change_host" in user_perms:

            retval.append(
                {
                    "title": _("Edit"),
                    "url": reverse(
                        "hosts_edit",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.delete_host" in user_perms:

            retval.append(
                {
                    "title": _("Delete"),
                    "url": reverse(
                        "hosts_delete",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.list_hosts_in_hostgroups":

            retval.append(
                {
                    "title": _("Groups"),
                    "url": reverse(
                        "hosts_hostgroups_list",
                        kwargs={
                            "host": template_object.id
                        }
                    )
                }
            )

        if "skd.setup_host" in user_perms:

            retval.append(
                {
                    "title": _("Setup"),
                    "url": reverse(
                        "hosts_setup",
                        kwargs={
                            "host": template_object.id
                        }
                    )
                }
            )

    elif object_type == "HostGroup":

        if "skd.change_hostgroup" in user_perms:

            retval.append(
                {
                    "title": _("Edit"),
                    "url": reverse(
                        "hostgroups_edit",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.delete_hostgroup" in user_perms:

            retval.append(
                {
                    "title": _("Delete"),
                    "url": reverse(
                        "hostgroups_delete",
                        kwargs={
                            "pk": template_object.id
                        }
                    )
                }
            )

        if "skd.list_hosts_in_hostgroups" in user_perms:

            retval.append(
                {
                    "title": _("Members"),
                    "url": reverse(
                        "hostgroups_hosts_list",
                        kwargs={
                            "hostgroup": template_object.id
                        }
                    )
                }
            )

        if "skd.list_usergroups_in_hostgroups" in user_perms:

            retval.append(
                {
                    "title": _("Assigned usergroups"),
                    "url": reverse(
                        "hostgroups_usergroups_list",
                        kwargs={
                            "hostgroup": template_object.id
                        }
                    )
                }
            )

    elif object_type == "HostInGroup":

        if "skd.delete_hostingroup" in user_perms:

            retval.append(
                {
                    "title": _("Unassign"),
                    "url": reverse(
                        "hosts_hostgroups_unassign",
                        kwargs={
                            "pk": template_object.id,
                            "host": template_object.host.id
                        }
                    )
                }
            )

    elif object_type == "UserGroupInHostGroup":

        if "skd.delete_usergroupinhostgroup" in user_perms:

            if view_context == "hostgroup":

                return [
                    {
                        "title": _("Unassign"),
                        "url": reverse(
                            "hostgroups_usergroups_unassign",
                            kwargs={
                                "hostgroup": template_object.hostgroup.id,
                                "pk": template_object.usergroup.id
                            }
                        )
                    }
                ]

            elif view_context == "usergroup":

                return [
                    {
                        "title": _("Unassign"),
                        "url": reverse(
                            "usergroups_hostgroups_unassign",
                            kwargs={
                                "usergroup": template_object.usergroup.id,
                                "pk": template_object.id
                            }
                        )
                    }
                ]

    elif object_type == "ActionLog":

        url = get_actionobject(
            template_object.action,
            template_object.objectid,
            template_object.objectid2,
            user_perms
        )

        if url:

            return [
                {
                    "title": _("Reference"),
                    "url": url
                }
            ]

    return retval