"""
Host-Views
"""
from datetime import datetime
from django.contrib import messages
from django.core.urlresolvers import reverse_lazy, reverse

from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, UpdateView, DeleteView, \
    TemplateView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext_lazy as _

from skd.models import Host, ActionLog, ApplyLog, Configuration, HostInGroup
from skd.views import SkdListView


class HostListView(SkdListView):
    """
    List all hosts

    **Template:**

    :template:`skd/listbase.html`

    """

    template_name = "skd/listbase.html"
    model = Host
    list_columns = [
        _("Name"),
        _("FQDN"),
        _("User")
    ]
    current_category = "hosts"
    no_rows = _("No hosts exist.")

    def get_actions(self):

        if self.request.user.has_perm('skd.add_host'):

            return [
                {
                    "title": _("Create host"),
                    "url": reverse("hosts_create")
                }
            ]

        else:

            return []


class HostCreateView(CreateView):
    """
    Creates a host.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/formbase.html`
    """

    model = Host
    success_url = reverse_lazy("hosts_list")
    template_name = 'skd/formbase.html'

    def get_context_data(self, **kwargs):
        context = super(HostCreateView, self).get_context_data(**kwargs)
        context['current_category'] = 'hosts'

        return context

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_HOST",
            objectid=self.object.id
        ).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostUpdateView(UpdateView):
    """
    Updates a host.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/formbase.html`
    """

    model = Host
    success_url = reverse_lazy("hosts_list")
    template_name = 'skd/formbase.html'

    def get_context_data(self, **kwargs):
        context = super(HostUpdateView, self).get_context_data(**kwargs)
        context['current_category'] = 'hosts'

        return context

    def form_valid(self, form):

        # Log Update

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_HOST",
            objectid=self.object.id,
            comment=_(
                "Updated host.\n"
                "Original data:\n"
                "Name: %(name)s\n"
                "FQDN: %(fqdn)s\n"
                "User: %(user)s\n"
                "%(comment)s\n"
                "\n"
                "New data:\n"
                "Name: %(newname)s\n"
                "FQDN: %(newfqdn)s\n"
                "User: %(newuser)s\n"
                "%(comment)s"
                % {
                    "name": self.object.name,
                    "fqdn": self.object.fqdn,
                    "user": self.object.user,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newfqdn": form.data["fqdn"],
                    "newuser": form.data["user"],
                    "newcomment": form.data["comment"]
                }
            )
        )

        log_item.save()

        # Save object

        self.object = form.save()

        # Log affected host

        ApplyLog(host=self.object, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostDeleteView(DeleteView):
    """
    Deletes a host.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/host_confirm_delete.html`

    """

    model = Host
    success_url = reverse_lazy("hosts_list")

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_HOST",
            comment=_(
                "Deleting host %(user)s@%(name)s (%(fqdn)s) with comment %"
                "(comment)s" %
                {
                    "name": self.object.name,
                    "fqdn": self.object.fqdn,
                    "user": self.object.user,
                    "comment": self.object.comment
                }
            )
        ).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


class HostSetupView(TemplateView):
    """
    Copies the skd public key into the host after a password authentication.

    **Template:**

    :template:`skd/host_setup.html`

    """

    template_name = "skd/host_setup.html"

    status = "none"

    def get_context_data(self, **kwargs):
        context = super(HostSetupView, self).get_context_data(**kwargs)

        context["host"] = Host.objects.get(id=kwargs["host"])
        context["sshkey_public"] = Configuration.objects.get(
            key="sshkey_public"
        )
        context["status"] = self.status

        return context

    def post(self, request, *args, **kwargs):

        if "do_setup" in request.POST:

            # Initializations

            host = Host.objects.get(id=request.POST["host"])
            password = request.POST["password"]

            setup_result = host.setup_host(password)

            self.status = setup_result["status"]

            if setup_result["status"] == "success":

                messages.success(request, setup_result["ssh_message"])

            elif setup_result["status"] == "error":

                messages.error(request, setup_result["ssh_message"])

        return self.get(request, *args, **kwargs)


class HostGroupAssignView(CreateView):
    """
    Assign a host to a hostgroup

    **Context**

    ``RequestContext``

    ``group_member``
        The group member to be.

    **Template:**

    :template:`skd/formbase.html`

    """

    template_name = "skd/formbase.html"

    def get_queryset(self):
        self.group_member = get_object_or_404(Host, id=self.kwargs["host"])
        return HostInGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(HostGroupAssignView, self).get_context_data(**kwargs)
        context["group_member"] = self.group_member
        context["current_category"] = "hosts"
        return context

    def get_initial(self):
        return {"host": self.group_member}

    def get_success_url(self):
        return reverse(
            "hosts_hostgroups_list",
            kwargs={
                "host": self.group_member.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_HOSTINGROUP",
            objectid=self.object.host.id,
            comment=_(
                "Assigned host %(user)s@%(host)s to group %(group)s" % {
                    "user": self.object.host.user,
                    "host": self.object.host.name,
                    "group": self.object.group.name
                }
            )
        )
        log_item.save()

        # Log affected host

        ApplyLog(host=self.object.host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class HostGroupUnassignView(DeleteView):
    """
    Unassign a host from a hostgroup

    **Context**

    ``RequestContext``

    ``group_member``
        The group member.

    **Template:**

    :template:`skd/host_hostgroups_unassign_confirm.html`

    """

    template_name = "skd/host_hostgroups_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.group_member = get_object_or_404(Host, id=self.kwargs["host"])
        return get_object_or_404(
            HostInGroup,
            id=self.kwargs["pk"],
            host=self.group_member
        )

    def get_context_data(self, **kwargs):
        context = super(HostGroupUnassignView, self).get_context_data(**kwargs)
        context["group_member"] = self.group_member
        return context

    def get_success_url(self):
        return reverse(
            "hosts_hostgroups_list",
            kwargs={
                "host": self.group_member.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        log_item = ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_HOSTINGROUP",
            objectid=self.object.host.id,
            comment=_(
                "Removed host %(user)s@%(host)s from group %(group)s" % {
                    "user": self.object.host.user,
                    "host": self.object.host.name,
                    "group": self.object.group.name
                }
            )
        )

        log_item.save()

        # Log affected host

        ApplyLog(host=self.object.host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


class HostHostGroupListView(SkdListView):
    """
    Lists hostgroups a host is member of
    """

    current_category = "hosts"
    no_rows = _("The host is not a member of any group.")
    list_columns = [
        _("Groupname")
    ]
    list_context = "host"

    def get_actions(self):

        if self.request.user.has_perm('skd.add_hostingroup'):

            return [
                {
                    "title": _("Assign to a group"),
                    "url": reverse(
                        "hosts_hostgroups_assign",
                        kwargs={
                            "host": self.kwargs["host"]
                        }
                    )
                }
            ]

        else:

            return []

    def get_queryset(self):
        return HostInGroup.objects.filter(host=self.kwargs["host"])