"""
Setup-View
"""
from django.contrib import messages
from django.contrib.auth.models import Group, Permission
from django.contrib.contenttypes.models import ContentType
from django.views.generic import TemplateView
from django.utils.translation import ugettext_lazy as _
from skd.models import Configuration, UserKey


class SetupView(TemplateView):
    """
    Creates a setup view for basic installation tasks such as creating
    a new ssh key pair.

    **Template:**

    :template:`skd/setup.html`

    """

    template_name = "skd/setup.html"

    # SKD administrative groups

    default_groups = [
        {
            "name": "skd_viewer",
            "description": "Users can only view and list objects like users, "
                           "keys, hosts, etc. They cannot edit, delete or "
                           "create them and cannot apply changes. They cannot"
                           " see the Action Log.",
            "permissions": [
                "skd.list_users",
                "skd.list_users_keys",
                "skd.list_usermaps",
                "skd.list_usergroups",
                "skd.list_users_in_usergroups",
                "skd.list_hosts",
                "skd.list_hostgroups",
                "skd.list_hosts_in_hostgroups",
                "skd.list_usergroups_in_hostgroups"
            ]
        },
        {
            "name": "skd_audit",
            "description": "Users can see the action log.",
            "permissions": [
                "skd.list_users",
                "skd.list_users_keys",
                "skd.list_usermaps",
                "skd.list_usergroups",
                "skd.list_users_in_usergroups",
                "skd.list_hosts",
                "skd.list_hostgroups",
                "skd.list_hosts_in_hostgroups",
                "skd.list_usergroups_in_hostgroups",
                "skd.list_actionlog"
            ]
        },
        {
            "name": "skd_usermanager",
            "description": "Users have full control of the user, "
                           "usergroup and key objects",
            "permissions": [
                "skd.list_users",
                "skd.list_users_keys",
                "skd.list_usermaps",
                "skd.list_usergroups",
                "skd.list_users_in_usergroups",
                "skd.list_hosts",
                "skd.list_hostgroups",
                "skd.list_hosts_in_hostgroups",
                "skd.list_usergroups_in_hostgroups",
                "skd.list_actionlog",
                "skd.add_user",
                "skd.change_user",
                "skd.delete_user",
                "skd.add_userkey",
                "skd.change_userkey",
                "skd.delete_userkey",
                "skd.add_usermap",
                "skd.change_usermap",
                "skd.delete_usermap",
                "skd.add_usergroup",
                "skd.change_usergroup",
                "skd.delete_usergroup",
                "skd.add_useringroup",
                "skd.change_useringroup",
                "skd.delete_useringroup",
                "skd.add_actionlog",
                "skd.add_applylog"
            ]
        },
        {
            "name": "skd_hostmanager",
            "description": "Users have the permissions in <em>skd_audit</em> "
                           "plus full control of the host and hostgroup "
                           "objects.",
            "permissions": [
                "skd.list_users",
                "skd.list_users_keys",
                "skd.list_usermaps",
                "skd.list_usergroups",
                "skd.list_users_in_usergroups",
                "skd.list_hosts",
                "skd.list_hostgroups",
                "skd.list_hosts_in_hostgroups",
                "skd.list_usergroups_in_hostgroups",
                "skd.list_actionlog",
                "skd.add_host",
                "skd.change_host",
                "skd.delete_host",
                "skd.add_hostgroup",
                "skd.change_hostgroup",
                "skd.delete_hostgroup",
                "skd.add_hostingroup",
                "skd.change_hostingroup",
                "skd.delete_hostingroup",
                "skd.add_actionlog",
                "skd.add_applylog",
                "skd.setup_host"
            ]
        },
        {
            "name": "skd_manager",
            "description": "<em>skd_usermanager</em> and "
                           "<em>skd_hostmanager</em> combined.",
            "permissions": [
                "skd.list_users",
                "skd.list_users_keys",
                "skd.list_usermaps",
                "skd.list_usergroups",
                "skd.list_users_in_usergroups",
                "skd.list_hosts",
                "skd.list_hostgroups",
                "skd.list_hosts_in_hostgroups",
                "skd.list_usergroups_in_hostgroups",
                "skd.list_actionlog",
                "skd.add_user",
                "skd.change_user",
                "skd.delete_user",
                "skd.add_userkey",
                "skd.change_userkey",
                "skd.delete_userkey",
                "skd.add_usermap",
                "skd.change_usermap",
                "skd.delete_usermap",
                "skd.add_usergroup",
                "skd.change_usergroup",
                "skd.delete_usergroup",
                "skd.add_useringroup",
                "skd.change_useringroup",
                "skd.delete_useringroup",
                "skd.add_host",
                "skd.change_host",
                "skd.delete_host",
                "skd.add_hostgroup",
                "skd.change_hostgroup",
                "skd.delete_hostgroup",
                "skd.add_hostingroup",
                "skd.change_hostingroup",
                "skd.delete_hostingroup",
                "skd.add_usergroupinhostgroup",
                "skd.change_usergroupinhostgroup",
                "skd.delete_usergroupinhostgroup",
                "skd.add_actionlog",
                "skd.add_applylog"
            ]
        },
        {
            "name": "skd_deployer",
            "description": "Can manage the whole database and apply the "
                           "changes (thus deploy new keys). Can delete the "
                           "actionlog.",
            "permissions": [
                "skd.list_users",
                "skd.list_users_keys",
                "skd.list_usermaps",
                "skd.list_usergroups",
                "skd.list_users_in_usergroups",
                "skd.list_hosts",
                "skd.list_hostgroups",
                "skd.list_hosts_in_hostgroups",
                "skd.list_usergroups_in_hostgroups",
                "skd.list_actionlog",
                "skd.add_user",
                "skd.change_user",
                "skd.delete_user",
                "skd.add_userkey",
                "skd.change_userkey",
                "skd.delete_userkey",
                "skd.add_usermap",
                "skd.change_usermap",
                "skd.delete_usermap",
                "skd.add_usergroup",
                "skd.change_usergroup",
                "skd.delete_usergroup",
                "skd.add_useringroup",
                "skd.change_useringroup",
                "skd.delete_useringroup",
                "skd.add_host",
                "skd.change_host",
                "skd.delete_host",
                "skd.add_hostgroup",
                "skd.change_hostgroup",
                "skd.delete_hostgroup",
                "skd.add_hostingroup",
                "skd.change_hostingroup",
                "skd.delete_hostingroup",
                "skd.add_usergroupinhostgroup",
                "skd.change_usergroupinhostgroup",
                "skd.delete_usergroupinhostgroup",
                "skd.add_actionlog",
                "skd.change_actionlog",
                "skd.delete_actionlog",
                "skd.can_apply",
                "skd.add_applylog",
                "skd.change_applylog",
                "skd.delete_applylog"
            ]
        }
    ]

    generation_messages = ""
    generation_status = "none"

    def get_context_data(self, **kwargs):
        context = super(SetupView, self).get_context_data(**kwargs)

        context["generation_messages"] = self.generation_messages
        context["generation_status"] = self.generation_status

        if Configuration.objects.filter(key="sshkey_public").exists():

            context["sshkey_public"] = Configuration.objects.get(
                key="sshkey_public"
            )

        if not Group.objects.filter(name__istartswith="skd_").exists():

            context["no_groups"] = True
            context["default_groups"] = self.default_groups

        return context

    def post(self, request, *args, **kwargs):

        if "sshkey_generate" in request.POST:

            # Generate a new Keypair

            if Configuration.objects.filter(key="sshkey_public"):

                # Get current configuration objects

                public_key = Configuration.objects.get(
                    key="sshkey_public"
                )

                private_key = Configuration.objects.get(
                    key="sshkey_private"
                )

            else:

                # Generate new configuration objects

                public_key = Configuration()
                public_key.key = "sshkey_public"

                private_key = Configuration()
                private_key.key = "sshkey_private"

            passphrase = self.request.POST.get("passphrase", None)
            keytype = self.request.POST.get("keytype", "dsa")
            bits = int(self.request.POST.get("bits", "1024"))

            # Call keygen to generate the keypair

            keyret = UserKey.objects.keygen(
                _("skd master key"),
                keytype,
                bits,
                passphrase
            )

            self.generation_status = keyret["generation_status"]

            if self.generation_status == "success":

                # Generation successful. Store keypair and type.

                if Configuration.objects.filter(key="sshkey_type"):

                    keytype_conf = Configuration.objects.get(key="sshkey_type")

                else:

                    keytype_conf = Configuration()
                    keytype_conf.key = "sshkey_type"

                keytype_conf.value = keytype
                keytype_conf.save()

                public_key.value = keyret["public_key"]

                public_key.save()

                private_key.value = keyret["private_key"]

                private_key.save()

                messages.success(
                    request,
                    _("The skd master key has been successfully generated.")
                )

            else:

                messages.error(request, keyret["generation_messages"])

        if "groups_generate" in request.POST:

            # We should generate the default groups.

            if not Group.objects.filter(name__istartswith="skd_").exists():

                # But only, if they haven't been already.

                for group in self.default_groups:

                    # Generate the default groups based on the permission
                    # dictionary above.

                    current_group = Group(
                        name=group["name"]
                    )

                    current_group.save()

                    for permission in group["permissions"]:
                        app, codename = permission.split(".")

                        app_ids = ContentType.objects.filter(
                            app_label=app
                        )

                        permission_object = Permission.objects.get(
                            codename=codename,
                            content_type_id__in=app_ids
                        )

                        current_group.permissions.add(permission_object)

                    current_group.save()

                messages.success(
                    request,
                    _("Default groups have been successfully generated.")
                )

            else:

                messages.error(
                    request,
                    _("Default groups have already been generated and can't "
                      "automatically been recreated. You have to manually "
                      "remove all groups starting with 'skd_'")
                )

        return self.get(request, *args, **kwargs)