"""
User views
"""
from datetime import datetime
from django.core.urlresolvers import reverse_lazy, reverse

from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, UpdateView, DeleteView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext_lazy as _

from skd.models import User, ActionLog, Host, ApplyLog, UserInGroup
from views import SkdListView


class UserListView(SkdListView):
    """
    Lists all users
    """

    model = User
    current_category = "users"
    no_rows = _("No users exist")
    list_columns = [
        _("Username"),
        _("Full name")
    ]

    def get_actions(self):

        if self.request.user.has_perm('skd.add_user'):

            return [
                {
                    "title": _("Create user"),
                    "url": reverse(
                        "users_create"
                    )
                }
            ]

        else:

            return []


class UserCreateView(CreateView):
    """
    Creates a user.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/formbase.html`
    """

    model = User
    success_url = reverse_lazy("users_list")
    template_name = 'skd/formbase.html'

    def get_context_data(self, **kwargs):
        context = super(UserCreateView, self).get_context_data(**kwargs)
        context['current_category'] = 'users'

        return context

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_USER",
            objectid=self.object.id
        ).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserUpdateView(UpdateView):
    """
    Updates a user.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/formbase.html`
    """

    model = User
    success_url = reverse_lazy("users_list")
    template_name = 'skd/formbase.html'

    def get_context_data(self, **kwargs):
        context = super(UserUpdateView, self).get_context_data(**kwargs)
        context['current_category'] = 'users'

        return context

    def form_valid(self, form):

        # Log Update

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_USER",
            objectid=self.object.id,
            comment=_(
                "Updated user:\n"
                "Original data: \n"
                "%(name)s (%(fullname)s)\n"
                "%(comment)s\n"
                "\n"
                "New data: \n"
                "%(newname)s (%(newfullname)s)\n"
                "%(newcomment)s\n"
                % {
                    "name": self.object.name,
                    "fullname": self.object.fullname,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newfullname": form.data["fullname"],
                    "newcomment": form.data["comment"]
                }
            )
        ).save()

        # Save object

        self.object = form.save()

        return super(ModelFormMixin, self).form_valid(form)


class UserDeleteView(DeleteView):
    """
    Deletes a user.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/user_confirm_delete.html`

    """

    model = User
    success_url = reverse_lazy("users_list")

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_USER",
            comment=_(
                "Deleting user %(name)s (%(fullname)s) with comment %"
                "(comment)s" %
                {
                    "name": self.object.name,
                    "fullname": self.object.fullname,
                    "comment": self.object.comment
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            **{
                "hostingroup__group__usergroupinhostgroup__usergroup"
                "__useringroup__user__id": self.object.id
            }
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="DELETE_USER_AFFECTED",
                comment=_(
                    "User %(name)s looses access to %(host)s" %
                    {
                        "name": self.object.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())


class UserUserGroupListView(SkdListView):
    """
    Lists groups a user is member of.
    """

    current_category = "users"
    no_rows = _("The user is not a member of any group.")
    list_columns = [
        _("Groupname")
    ]
    list_context = "user"

    def get_actions(self):

        if self.request.user.has_perm('skd.add_useringroup'):

            return [
                {
                    "title": _("Assign to a group"),
                    "url": reverse(
                        "users_usergroups_assign",
                        kwargs={
                            "user": self.kwargs["user"]
                        }
                    )
                }
            ]

        else:

            return []

    def get_queryset(self):
        return UserInGroup.objects.filter(user=self.kwargs["user"])


class UserUserGroupAssignView(CreateView):
    """
    Assigns a user to a new group.

    **Context**

    ``RequestContext``

    ``group_member``
        The member of the group to be.

    **Template:**

    :template:`skd/formbase.html`

    """

    template_name = "skd/formbase.html"

    def get_queryset(self):
        self.group_member = get_object_or_404(User, id=self.kwargs["user"])
        return UserInGroup.objects.all()

    def get_context_data(self, **kwargs):
        context = super(UserUserGroupAssignView, self).get_context_data(
            **kwargs
        )
        context["group_member"] = self.group_member
        context["current_category"] = "users"
        return context

    def get_initial(self):
        return {"user": self.group_member}

    def get_success_url(self):
        return reverse(
            "users_usergroups_list",
            kwargs={
                "user": self.group_member.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Assignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="ASSIGN_USERINGROUP",
            objectid=self.object.user.id,
            comment=_(
                "Assigned user %(user)s to usergroup %(usergroup)s" %
                {
                    "user": self.object.user.name,
                    "usergroup": self.object.group.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            **{
                "hostingroup__group__usergroupinhostgroup__usergroup__"
                "useringroup__group__id": self.object.group.id
            }
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="ASSIGN_USERINGROUP_AFFECTED",
                objectid=self.object.user.id,
                comment=_(
                    "User %(user)s now has access to host %(host)s" %
                    {
                        "user": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserUserGroupUnassignView(DeleteView):
    """
    Removes a member from a group.

    **Context**

    ``RequestContext``

    ``group_member``
        The member of the group.

    **Template:**

    :template:`skd/user_usergroups_unassign_confirm.html`

    """

    template_name = "skd/user_usergroups_unassign_confirm.html"

    def get_object(self, queryset=None):
        self.group_member = get_object_or_404(User, id=self.kwargs["user"])
        return get_object_or_404(
            UserInGroup,
            id=self.kwargs["pk"],
            user=self.group_member
        )

    def get_context_data(self, **kwargs):
        context = super(UserUserGroupUnassignView, self).get_context_data(
            **kwargs
        )
        context["group_member"] = self.group_member
        return context

    def get_success_url(self):
        return reverse(
            "users_usergroups_list",
            kwargs={
                "user": self.group_member.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Unassignment

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UNASSIGN_USERINGROUP",
            objectid=self.object.user.id,
            comment=_(
                "Removed user %(user)s from usergroup %(usergroup)s" %
                {
                    "user": self.object.user.name,
                    "usergroup": self.object.group.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            **{
                "hostingroup__group__usergroupinhostgroup__usergroup"
                "__useringroup__id": self.object.group.id
            }
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UNASSIGN_USERINGROUP_AFFECTED",
                objectid=self.object.user.id,
                comment=_(
                    "User %(user)s looses access to host %(host)s" %
                    {
                        "user": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())