"""
User/Key-Views
"""
from datetime import datetime
from django.contrib import messages

from django.core.urlresolvers import reverse
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, UpdateView, DeleteView, \
    TemplateView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext_lazy as _, ugettext
from skd.models import User, UserKey, Host, ApplyLog, ActionLog
from views import SkdListView


class UserKeyListView(SkdListView):
    """
    Lists keys from a given user.
    """

    current_category = "users"
    no_rows = _("The user has no keys.")
    list_columns = [
        _("Keyname")
    ]

    def get_actions(self):

        if self.request.user.has_perm('skd.add_userkey'):

            return [
                {
                    "title": _("Generate key"),
                    "url": reverse(
                        "users_keys_generate",
                        kwargs={
                            "user": self.kwargs["user"]
                        }
                    )
                },
                {
                    "title": _("Add Key"),
                    "url": reverse(
                        "users_keys_create",
                        kwargs={
                            "user": self.kwargs["user"]
                        }
                    )
                }
            ]

        else:

            return []

    def get_queryset(self):
        return UserKey.objects.filter(user=self.kwargs["user"])


class UserKeyGenerateView(TemplateView):
    """
    Generates a keypair for a user

    **Template:**

    :template:`skd/userkey_generate.html`
    """

    template_name = "skd/userkey_generate.html"

    generation_status = None
    generation_messages = ""

    public_key = ""
    private_key = ""

    mode = "view"

    def get_context_data(self, **kwargs):
        context = super(UserKeyGenerateView, self).get_context_data(**kwargs)

        context["mode"] = self.mode

        if self.mode == "generate":

            context["generation_status"] = self.generation_status
            context["public_key"] = self.public_key
            context["private_key"] = self.private_key

        return context

    def post(self, request, *args, **kwargs):

        user = self.kwargs["user"]

        user_object = User.objects.get(id=user)

        if not user_object:

            raise _("Cannot find user object with id %s" % user)

        if "generate" in request.POST:

            # We should generate a keypair

            self.mode = "generate"

            self.generation_status = "success"

            passphrase = self.request.POST.get("passphrase", None)
            keytype = self.request.POST.get("keytype", "dsa")
            bits = int(self.request.POST.get("bits", "1024"))

            keyname = request.POST.get("keyname", _("skd-key"))
            if keyname == "":
                keyname = _("skd-key")

            # Call keygen to generate keypair

            keyret = UserKey.objects.keygen(keyname, keytype, bits, passphrase)

            self.generation_status = keyret["generation_status"]

            if self.generation_status == "success":

                # Success. Save keypair for displaying later

                self.public_key = keyret["public_key"]
                self.private_key = keyret["private_key"]

                skd_key = UserKey()
                skd_key.name = keyname
                skd_key.comment = _(
                    "Key generated by skd on %s" %
                    datetime.now().strftime(ugettext("%Y-%m-%d %H:%M:%S"))
                )
                skd_key.key = keyret["public_key"]
                skd_key.user = user_object

                skd_key.save()

                messages.success(
                    request,
                    _("The key has been generated successfully, added to "
                      "skd and is shown below.")
                )

            else:

                messages.error(
                    request,
                    keyret["generation_messages"]
                )

        return self.get(request, *args, **kwargs)


class UserKeyCreateView(CreateView):
    """
    Adds a new key for a user.

    **Context**

    ``RequestContext``

    ``key_owner``
        An instance of the owner holding the key.

    **Template:**

    :template:`skd/formbase.html`

    """

    template_name = "skd/formbase.html"

    def get_queryset(self):
        self.key_owner = get_object_or_404(User, id=self.kwargs["user"])
        return UserKey.objects.all()

    def get_context_data(self, **kwargs):
        context = super(UserKeyCreateView, self).get_context_data(**kwargs)
        context["key_owner"] = self.key_owner
        context["current_category"] = "users"
        return context

    def get_initial(self):
        return {"user": self.key_owner}

    def get_success_url(self):
        return reverse(
            "users_keys_list",
            kwargs={
                "user": self.key_owner.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_KEY",
            objectid=self.object.id,
            comment=_(
                "User %(user)s now has the key %(key)s" % {
                    "user": self.object.user.name,
                    "key": self.object.name
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            **{
                "hostingroup__group__usergroupinhostgroup__usergroup__"
                "useringroup__user__id": self.object.user.id
            }
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="CREATE_KEY_AFFECTED",
                objectid=self.object.id,
                comment=_(
                    "User %(user)s now has access to host %(host)s with key "
                    "%(key)s" % {
                        "user": self.object.user.name,
                        "host": host.name,
                        "key": self.object.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserKeyUpdateView(UpdateView):
    """
    Edits an existing key of a user.

    **Context**

    ``RequestContext``

    ``key_owner``
        An instance of the owner holding the key.

    **Template:**

    :template:`skd/formbase.html`

    """

    template_name = "skd/formbase.html"

    def get_queryset(self):
        self.key_owner = get_object_or_404(User, id=self.kwargs["user"])
        return UserKey.objects.all()

    def get_context_data(self, **kwargs):
        context = super(UserKeyUpdateView, self).get_context_data(**kwargs)
        context["key_owner"] = self.key_owner
        context["current_category"] = "users"
        return context

    def get_initial(self):
        return {"user": self.key_owner}

    def get_success_url(self):
        return reverse(
            "users_keys_list",
            kwargs={
                "user": self.key_owner.id
            }
        )

    def form_valid(self, form):

        # Log Update

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_KEY",
            objectid=self.object.id,
            objectid2=self.object.user.id,
            comment=_(
                "Updated key of user %(user)s.\n"
                "Original data:\n"
                "Name: %(name)s\n"
                "Key: %(key)s\n"
                "%(comment)s\n"
                "\n"
                "New data:\n"
                "Name: %(newname)s\n"
                "Key: %(newkey)s\n"
                "%(newcomment)s\n"
                % {
                    "user": self.object.user.name,
                    "name": self.object.name,
                    "key": self.object.key,
                    "comment": self.object.comment,
                    "newname": form.data["name"],
                    "newkey": form.data["key"],
                    "newcomment": form.data["comment"]
                }
            )
        ).save()

        # Save object

        self.object = form.save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            **{
                "hostingroup__group__usergroupinhostgroup__usergroup__"
                "useringroup__user__id": self.object.user.id
            }
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UPDATE_KEY_AFFECTED",
                objectid=self.object.id,
                objectid2=self.object.user.id,
                comment=_(
                    "Key %(key)s of user %(user)s has been updated. Notify "
                    "host %(host)s"
                    % {
                        "user": self.object.user.name,
                        "key": self.object.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserKeyDeleteView(DeleteView):
    """
    Removes an existing key for a user.

    **Context**

    ``RequestContext``

    ``key_owner``
        An instance of the owner holding the key.

    **Template:**

    :template:`skd/userkey_confirm_delete.html`

    """

    template_name = "skd/userkey_confirm_delete.html"

    def get_object(self, queryset=None):
        self.key_owner = get_object_or_404(User, id=self.kwargs["user"])
        return get_object_or_404(
            UserKey,
            id=self.kwargs["pk"],
            user=self.key_owner
        )

    def get_context_data(self, **kwargs):
        context = super(UserKeyDeleteView, self).get_context_data(**kwargs)
        context["key_owner"] = self.key_owner
        return context

    def get_success_url(self):
        return reverse(
            "users_keys_list",
            kwargs={
                "user": self.key_owner.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_KEY_FROM_USER",
            comment=_(
                "Deleting key '%(name)s' from user %(username)s with comment "
                "%(comment)s" % {
                    "name": self.object.name,
                    "username": self.object.user.name,
                    "comment": self.object.comment
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = Host.objects.filter(
            **{
                "hostingroup__group__usergroupinhostgroup__usergroup__"
                "useringroup__user__id": self.object.user.id
            }
        )

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="DELETE_KEY_FROM_USER_AFFECTED",
                comment=_(
                    "Now missing Key %(name)s from user %(username)s looses "
                    "access to host %(host)s" % {
                        "name": self.object.name,
                        "username": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())