"""
User views
"""
from datetime import datetime
from django.core.urlresolvers import reverse_lazy, reverse

from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.views.generic import CreateView, UpdateView, DeleteView
from django.views.generic.edit import ModelFormMixin
from django.utils.translation import ugettext_lazy as _

from skd.models import UserMap, ActionLog, Host, ApplyLog, User
from views import SkdListView


class UserMapListView(SkdListView):
    """
    Lists usernampe maps from a given user.
    """

    current_category = "users"
    no_rows = _("No username mappings exist.")
    list_columns = [
        _("Host"),
        _("Username on host")
    ]

    def get_actions(self):

        if self.request.user.has_perm('skd.add_usermap'):

            return [
                {
                    "title": _("Create mapping"),
                    "url": reverse(
                        "users_map_create",
                        kwargs={
                            "user": self.kwargs["user"]
                        }
                    )
                }
            ]

        else:

            return []

    def get_queryset(self):
        return UserMap.objects.filter(user=self.kwargs["user"])


class UserMapCreateView(CreateView):
    """
    Creates a usermap.

    **Context**

    ``RequestContext``

    ``assoc_user``
        An instance of the user associated with this usermap

    **Template:**

    :template:`skd/formbase.html`

    """

    template_name = "skd/formbase.html"

    def get_queryset(self):
        self.assoc_user = get_object_or_404(User, id=self.kwargs["user"])
        return UserMap.objects.all()

    def get_context_data(self, **kwargs):
        context = super(UserMapCreateView, self).get_context_data(**kwargs)
        context["assoc_user"] = self.assoc_user
        context["current_category"] = "users"
        return context

    def get_initial(self):
        return {"user": self.assoc_user}

    def get_success_url(self):
        return reverse(
            "users_map_list",
            kwargs={
                "user": self.assoc_user.id
            }
        )

    def form_valid(self, form):

        # Save object

        self.object = form.save()

        # Log Creation

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="CREATE_USERNAMEMAP",
            objectid=self.object.id
        ).save()

        affected_hosts = [self.object.host]

        for host in affected_hosts:

            # Log Deletion

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="CREATE_USERMAP_AFFECTED",
                comment=_(
                    "User mapping of %(name)s on %(host)s created" %
                    {
                        "name": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        return super(ModelFormMixin, self).form_valid(form)


class UserMapUpdateView(UpdateView):
    """
    Updates a usermap.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/formbase.html`
    """

    model = UserMap
    template_name = "skd/formbase.html"

    def get_queryset(self):
        self.assoc_user = get_object_or_404(User, id=self.kwargs["user"])
        return UserMap.objects.all()

    def get_context_data(self, **kwargs):
        context = super(UserMapUpdateView, self).get_context_data(**kwargs)
        context["assoc_user"] = self.assoc_user
        context["current_category"] = "users"
        return context

    def get_initial(self):
        return {"user": self.assoc_user}

    def get_success_url(self):
        return reverse(
            "users_map_list",
            kwargs={
                "user": self.assoc_user.id
            }
        )

    def form_valid(self, form):

        # Get original host

        original_host = Host.objects.get(id = form.initial['host'])

        # Log Update

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="UPDATE_USERMAP",
            objectid=self.object.id,
            comment=_(
                "Updated usermap:\n"
                "Original data: \n"
                "User: %(user)s\n"
                "Host: %(host)s\n"
                "Username: %(username)s\n"
                "%(comment)s\n"
                "\n"
                "New data: \n"
                "User: %(newuser)s\n"
                "Host: %(newhost)s\n"
                "Username: %(newusername)s\n"
                "%(newcomment)s\n"
                % {
                    "user": self.object.user.name,
                    "host": self.object.host.name,
                    "username": self.object.username,
                    "comment": self.object.comment,
                    "newuser": form.data["user"],
                    "newhost": original_host,
                    "newusername": form.data["username"],
                    "newcomment": form.data["comment"]
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = [self.object.host, original_host]

        for host in affected_hosts:

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="UPDATE_USERMAP_AFFECTED",
                comment=_(
                    "User mapping of %(name)s changed on %(host)s" %
                    {
                        "name": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Save object

        self.object = form.save()

        return super(ModelFormMixin, self).form_valid(form)


class UserMapDeleteView(DeleteView):
    """
    Deletes a usermap.

    **Context**

    ``RequestContext``

    **Template:**

    :template:`skd/usermap_confirm_delete.html`

    """

    success_url = reverse_lazy("users_map_list")

    def get_object(self, queryset=None):
        self.assoc_user = get_object_or_404(User, id=self.kwargs["user"])
        return get_object_or_404(
            UserMap,
            id=self.kwargs["pk"],
            user=self.assoc_user
        )

    def get_context_data(self, **kwargs):
        context = super(UserMapDeleteView, self).get_context_data(**kwargs)
        context["assoc_user"] = self.assoc_user
        return context

    def get_success_url(self):
        return reverse(
            "users_map_list",
            kwargs={
                "user": self.assoc_user.id
            }
        )

    def delete(self, request, *args, **kwargs):

        # Delete object

        self.object = self.get_object()

        # Log Deletion

        ActionLog(
            timestamp=datetime.now(),
            user=self.request.user,
            action="DELETE_USERMAP",
            comment=_(
                "Deleting usermap %(name)s@%(host)s => %(username)s  with "
                "comment %(comment)s" %
                {
                    "name": self.object.user.name,
                    "host": self.object.host.name,
                    "username": self.object.username,
                    "comment": self.object.comment
                }
            )
        ).save()

        # Log affected hosts

        affected_hosts = [self.object.host]

        for host in affected_hosts:

            # Log Deletion

            log_item = ActionLog(
                timestamp=datetime.now(),
                user=self.request.user,
                action="DELETE_USERMAP_AFFECTED",
                comment=_(
                    "User %(name)s looses access to %(host)s" %
                    {
                        "name": self.object.user.name,
                        "host": host.name
                    }
                )
            )

            log_item.save()

            ApplyLog(host=host, log=log_item).save()

        # Delete object

        self.object.delete()

        return HttpResponseRedirect(self.get_success_url())