# -*- coding: utf-8 -*-

import requests
from django.core.exceptions import ValidationError
from django.core.validators import URLValidator
from django.http import HttpResponse


class SmartProxyRequest(object):
    """
    Container object for requests generated by django-smart-proxy.
    Uses `requests` library.
    """

    _methods = {
        'get': requests.get,
        'put': requests.put,
        'post': requests.post,
        'delete': requests.delete,
        'head': requests.head,
        'options': requests.options,
        'patch': requests.patch,
    }
    """
    Maps request method string representations to the appropriate request
    method in `requests`.
    """

    HTTP_METHODS = ('get', 'put', 'post', 'delete',
                    'head', 'options', 'patch',)
    """Comprehensive iterable of all available HTTP request methods."""

    HEADER_BLACKLIST = ('content-encoding',
                        'content-length',)
    """
    The `request` library decompresses the response before we get here,
    so we need to remove the outdated encoding headers.
    """

    validate = URLValidator()
    """Used to validate `url`."""


    def __init__(self,
                 method='get',
                 url=None,
                 headers={},
                 timeout=60.0,
                 settings={},
                 auth=None):
        """
        Initializes the SmartProxyRequest.
        """

        super(SmartProxyRequest, self).__init__()

        self.method = method
        self.url = url
        self.headers = headers
        self.timeout = timeout
        self.settings = settings
        self.auth = auth

        if self.settings:
            self.HEADER_BLACKLIST += self.settings.get('blacklisted_headers',
                                                       ())

    @property
    def method(self):
        """Gets the hidden `_method` instance variable."""
        return self._method

    @method.setter
    def method(self, value):
        """Validates and sets the hidden `_method` instance variable."""

        value_lowercase = value.lower()
        if value_lowercase not in self.HTTP_METHODS:
            raise ValidationError(
                '"{0}" is not an allowed HTTP method.'.format(value_lowercase))
        self._method = value_lowercase

    @property
    def url(self):
        """Gets the hidden `_url` instance variable."""
        return self._url

    @url.setter
    def url(self, value):
        """Validates and sets the hidden `_url` instance variable."""

        # Will raise ValidationError if invalid URL is supplied.
        self.validate(value)
        self._url = value

    @property
    def timeout(self):
        """Gets the hidden `_timeout` instance variable."""
        return self._timeout

    @timeout.setter
    def timeout(self, value):
        """Validates and sets the hidden `_timeout` instance variable."""

        # Will raise ValueError if not convertible to float.
        self._timeout = float(value)

    def _build_http_response(self, response):
        """
        Converts the response from `requests` to a Django-compatible
        HttpResponse object.
        """

        http_response = HttpResponse(
            response.text,
            status=int(response.status_code),
            content_type=response.headers['content-type'])

        # Copy headers to response
        for key in filter(lambda header: (
                            header not in self.HEADER_BLACKLIST),
                          response.headers.keys()):
            http_response[key] = response.headers[key]

        return http_response

    def send(self, method=None, raw=False):
        """
        Sends the request and returns a Django-compatible HttpResponse (unless
        a 'raw' response is requested, in which case the original response
        from `requests` is returned.

        Optionally, an HTTP request method/verb override can be specified.
        """

        response = self._methods[method or self.method](
            self.url,
            headers=self.headers,
            timeout=self.timeout,
            auth=self.auth)

        if raw:
            return response

        return self._build_http_response(response)
