import os, errno, importlib, uuid, threading

from django.core.files import base, temp
from django.utils.six.moves import queue as six_queue

__all__ = [
    'VALUE_NOT_SET', 'ProcessingError', 'NamedTemporaryFile', 'UploadTo', 
    'AsynchronousFileReader'
]

class VALUE_NOT_SET(object):
    pass


class ProcessingError(Exception):
    pass


class NamedTemporaryFile(base.File):
    """This class is required for FileStorage to make an attempt in moving the
    file, instead of copying it by chunks in memory

    """
    def __init__(self, **kwargs):
        file = temp.NamedTemporaryFile(**kwargs)
        super(NamedTemporaryFile, self).__init__(file)

    def temporary_file_path(self):
        """
        Returns the full path of this file.
        """
        return self.file.name

    def close(self):
        # caching the size before closing for proper file moving and saving
        self._size = self._get_size()
        try:
            return self.file.close()
        except OSError as e:
            if e.errno != errno.ENOENT:
                # Means the file was moved or deleted before the tempfile
                # could unlink it.  Still sets self.file.close_called and
                # calls self.file.file.close() before the exception
                raise


class UploadTo(object):
    """This is an upload filename generator to be used to create a function to be
    passed as an ``upload_to`` keyword argument to the ``models.FileField`` and
    it's derivatives. It will generate the path in the form:
    basefolder/app_label/model_name/parent_pk/subfolder/pk/field_name/filename.ext
    
    :keyword str basefolder: path that will be prepended to the filename.  
    
    :keyword str subfolder: path that will be a container of the model instances

    :keyword str filename: will replace the actual file name completely, ex:
    ``filename='file.ext' -> in_filename='foo.bar' -> out_filename='file.ext'``

    :keyword str name: will replace the name portion of the file, ex:
    ``name='file' -> in_filename='foo.bar' -> out_filename='file.bar'``

    :keyword str ext: will replace the extension portion of the file, ex:
    ``ext='pdf' -> in_filename='foo.bar' -> out_filename='foo.pdf'``

    :keyword str app_label: if ``None`` will insert the ``app_label`` retrieved from
    the model instance (Default). Otherwise specify a string to enforce a
    specific app_label or anything else evaluating to ``False`` except ``None`` 
    in order to skip insertion of an app_label.

    :keyword str model_name: if ``None`` will insert the ``model_name`` retrieved
    from the model instance (Default). Otherwise specify a string to enforce a
    specific model_name or anything else evaluating to ``False`` except ``None`` 
    in order to skip insertion of a model_name.

    :keyword str field_name: if ``None`` will skip insertion of a field_name
    (Default), otherwise ``field_name`` string will be inserted before the subfolder.

    :keyword bool generate: if set to ``True`` will replace the ``name`` portion
    of the file with a string generated by ``generator`` function.

    :keyword function generator: function that will generate a new name portion
    of the file in case if ``generate`` is set to ``True``. Default is
    :func:`uuid.uuid1`.

    :keyword function parent_pk_getter: function that takes model instance and
    returns pk of the parent

    """

    def __init__(self, basefolder=None, subfolder=None, filename=None, name=None, ext=None,
                 app_label=None, model_name=None, parent_pk_getter=None, field_name=None, 
                 generate=False, generator=uuid.uuid1):
        assert filename is None or name is None, \
            "Cannot have 'filename' and 'name' specified at the same time." 
        assert not generate or (filename is None and name is None), \
            "Cannot specify a name and enforce name generation"
        assert not generate or callable(generator), \
            "Supply a function for a generator"
        self.basefolder = basefolder
        self.subfolder = subfolder
        self.filename = filename
        self.name = name
        self.ext = ext
        self.app_label = app_label
        self.model_name = model_name
        self.parent_pk_getter = parent_pk_getter if callable(parent_pk_getter) else None
        self.field_name = field_name
        self.generate = generate
        self.generator = generator

    def __call__(self, instance, filename):
        structure = []
        if self.basefolder is not None:
            structure.append(self.basefolder)
        if self.app_label is None:
            structure.append(instance._meta.app_label)
        elif self.app_label:
            structure.append(self.app_label)
        if self.model_name is None:
            structure.append(instance._meta.model_name)
        elif self.model_name:
            structure.append(self.model_name)
        if self.parent_pk_getter is not None:
            structure.append(str(self.parent_pk_getter(instance)))
        if self.subfolder is not None:
            structure.append(self.subfolder)
        if instance.pk is not None:
            structure.append(str(instance.pk))
        if self.field_name is not None:
            structure.append(self.field_name)
        if self.filename is not None:
            filename = self.filename
        else:
            name, ext = os.path.splitext(filename)
            if ext: # remove a dot, but only if there is an extension
                ext = ext[1:]
            if self.generate:
                name = self.generator()
            elif self.name is not None:
                name = self.name
            if self.ext is not None:
                ext = self.ext
            if ext:
                filename = "%s.%s" % (name, ext.lower())
            else: 
                filename = name
        structure.append(filename)
        return os.path.join(*structure)


class AsynchronousFileReader(threading.Thread):
    '''
    Helper class to implement asynchronous reading of a file
    in a separate thread. Pushes read lines on a queue to
    be consumed in another thread.
    source: http://stefaanlippens.net/python-asynchronous-subprocess-pipe-reading
    '''

    def __init__(self, fd, queue):
        assert isinstance(queue, six_queue.Queue)
        assert callable(fd.readline)
        super(AsynchronousFileReader, self).__init__()
        self._fd = fd
        self._queue = queue

    def run(self):
        '''The body of the tread: read lines and put them on the queue.'''
        for line in iter(self._fd.readline, ''):
            self._queue.put(line)

    def eof(self):
        '''Check whether there is no more content to expect.'''
        return not self.is_alive() and self._queue.empty()
