
/**
 * @class draw2d
 * global namespace declarations
 * 
 * @private
 */
var draw2d = 
{
    geo: {
    },

    io:{
        json:{},
        png:{},
        svg:{}  
    },
    
       
    storage:{
    },
    
    util : {
    	spline: {}
    },

    shape : {
    	basic:{},
        arrow:{},
        node: {},
        note: {},
        diagram:{},
        analog:{},
        icon:{},
        layout:{},
        pert:{},
        state:{},
        widget:{}
    },
    
    policy : {
        canvas:{},
        line:{},
        port:{},
        figure:{}
    },
    
    command : {
    },

    decoration:{
    	connection:{}
    }, 
    
    layout: {
        connection :{},
	    anchor :{},
	    mesh :{},
	    locator: {}
    },
    
    
    ui :{
    	
    },
    
    isTouchDevice : (
            //Detect iPhone
            (navigator.platform.indexOf("iPhone") != -1) ||
            //Detect iPod
            (navigator.platform.indexOf("iPod") != -1)||
            //Detect iPad
            (navigator.platform.indexOf("iPad") != -1)
        )
    
};


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/draw2d.util.Base64 = {

    /**
     * Maps bytes to characters.
     * @type {Object}
     * @private
     */
    byteToCharMap_ :null,


    /**
     * Maps characters to bytes.
     * @type {Object}
     * @private
     */
    charToByteMap_: null,


    /**
     * Maps bytes to websafe characters.
     * @type {Object}
     * @private
     */
    byteToCharMapWebSafe_ : null,

    /**
     * Maps websafe characters to bytes.
     * @type {Object}
     * @private
     */
    charToByteMapWebSafe_ : null,


    /**
     * Our default alphabet, shared between
     * ENCODED_VALS and ENCODED_VALS_WEBSAFE
     * @type {string}
     */
    ENCODED_VALS_BASE : 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789',

    /**
     * Our default alphabet. Value 64 (=) is special; it means "nothing."
     * @type {string}
     */
    ENCODED_VALS : 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789' + '+/=',


    /**
     * Our websafe alphabet.
     * @type {string}
     */
    ENCODED_VALS_WEBSAFE :'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789' + '-_.',

    
    encodeByteArray : function(input, opt_webSafe) {
        draw2d.util.Base64.init();

        var byteToCharMap = opt_webSafe ?  draw2d.util.Base64.byteToCharMapWebSafe_ : draw2d.util.Base64.byteToCharMap_;

        var output = [];

        for (var i = 0; i < input.length; i += 3) {
          var byte1 = input[i];
          var haveByte2 = i + 1 < input.length;
          var byte2 = haveByte2 ? input[i + 1] : 0;
          var haveByte3 = i + 2 < input.length;
          var byte3 = haveByte3 ? input[i + 2] : 0;

          var outByte1 = byte1 >> 2;
          var outByte2 = ((byte1 & 0x03) << 4) | (byte2 >> 4);
          var outByte3 = ((byte2 & 0x0F) << 2) | (byte3 >> 6);
          var outByte4 = byte3 & 0x3F;

          if (!haveByte3) {
            outByte4 = 64;

            if (!haveByte2) {
              outByte3 = 64;
            }
          }

          output.push(byteToCharMap[outByte1],
                      byteToCharMap[outByte2],
                      byteToCharMap[outByte3],
                      byteToCharMap[outByte4]);
        }

        return output.join('');
      },
      

      /**
       * @method
       * Base64-encode a string.
       *
       * @param {string} input A string to encode.
       * @param {boolean=} opt_webSafe If true, we should use the alternative alphabet.
       * @return {string} The base64 encoded string.
       */
     encode: function(input, opt_webSafe) {
        return draw2d.util.Base64.encodeByteArray( draw2d.util.Base64.stringToByteArray(input), opt_webSafe);
      },

      
      /**
       * @method
       * Base64-decode a string.
       *
       * @param {String} input to decode (length not required to be a multiple of 4).
       * @param {boolean=} opt_webSafe True if we should use the
       *     alternative alphabet.
       * @return {Array} bytes representing the decoded value.
       */
      decode: function(input, opt_webSafe) {
        draw2d.util.Base64.init();

        var charToByteMap = opt_webSafe ?draw2d.util.Base64.charToByteMapWebSafe_ : draw2d.util.Base64.charToByteMap_;

        var output = [];

        for (var i = 0; i < input.length; ) {
          var byte1 = charToByteMap[input.charAt(i++)];

          var haveByte2 = i < input.length;
          var byte2 = haveByte2 ? charToByteMap[input.charAt(i)] : 0;
          ++i;

          var haveByte3 = i < input.length;
          var byte3 = haveByte3 ? charToByteMap[input.charAt(i)] : 0;
          ++i;

          var haveByte4 = i < input.length;
          var byte4 = haveByte4 ? charToByteMap[input.charAt(i)] : 0;
          ++i;

          if (byte1 == null || byte2 == null ||
              byte3 == null || byte4 == null) {
            throw Error();
          }

          var outByte1 = (byte1 << 2) | (byte2 >> 4);
          output.push(outByte1);

          if (byte3 != 64) {
            var outByte2 = ((byte2 << 4) & 0xF0) | (byte3 >> 2);
            output.push(outByte2);

            if (byte4 != 64) {
              var outByte3 = ((byte3 << 6) & 0xC0) | byte4;
              output.push(outByte3);
            }
          }
        }

        return output;
     },
      
    /**
     * Turns a string into an array of bytes; a "byte" being a JS number in the
     * range 0-255.
     * @param {string} str String value to arrify.
     * @return {!Array.<number>} Array of numbers corresponding to the
     *     UCS character codes of each character in str.
     */
    stringToByteArray : function(str) {
      var output = [], p = 0;
      for (var i = 0; i < str.length; i++) {
        var c = str.charCodeAt(i);
        while (c > 0xff) {
          output[p++] = c & 0xff;
          c >>= 8;
        }
        output[p++] = c;
      }
      return output;
    },
    
    init : function() {
        if (!draw2d.util.Base64.byteToCharMap_) {
            draw2d.util.Base64.byteToCharMap_ = {};
            draw2d.util.Base64.charToByteMap_ = {};
            draw2d.util.Base64.byteToCharMapWebSafe_ = {};
            draw2d.util.Base64.charToByteMapWebSafe_ = {};

          // We want quick mappings back and forth, so we precompute two maps.
          for (var i = 0; i < draw2d.util.Base64.ENCODED_VALS.length; i++) {
              draw2d.util.Base64.byteToCharMap_[i] = draw2d.util.Base64.ENCODED_VALS.charAt(i);
              draw2d.util.Base64.charToByteMap_[draw2d.util.Base64.byteToCharMap_[i]] = i;
              draw2d.util.Base64.byteToCharMapWebSafe_[i] = draw2d.util.Base64.ENCODED_VALS_WEBSAFE.charAt(i);
              draw2d.util.Base64.charToByteMapWebSafe_[draw2d.util.Base64.byteToCharMapWebSafe_[i]] = i;
          }
        }
    }
};
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//*!
 * JavaScript Debug - v0.4 - 6/22/2010
 * http://benalman.com/projects/javascript-debug-console-log/
 * 
 * Copyright (c) 2010 "Cowboy" Ben Alman
 * Dual licensed under the MIT and GPL licenses.
 * http://benalman.com/about/license/
 * 
 * With lots of help from Paul Irish!
 * http://paulirish.com/
 */

// Script: JavaScript Debug: A simple wrapper for console.log
//
// *Version: 0.4, Last Updated: 6/22/2010*
// 
// Tested with Internet Explorer 6-8, Firefox 3-3.6, Safari 3-4, Chrome 3-5, Opera 9.6-10.5
// 
// Home       - http://benalman.com/projects/javascript-debug-console-log/
// GitHub     - http://github.com/cowboy/javascript-debug/
// Source     - http://github.com/cowboy/javascript-debug/raw/master/ba-debug.js
// (Minified) - http://github.com/cowboy/javascript-debug/raw/master/ba-debug.min.js (1.1kb)
// 
// About: License
// 
// Copyright (c) 2010 "Cowboy" Ben Alman,
// Dual licensed under the MIT and GPL licenses.
// http://benalman.com/about/license/
// 
// About: Support and Testing
// 
// Information about what browsers this code has been tested in.
// 
// Browsers Tested - Internet Explorer 6-8, Firefox 3-3.6, Safari 3-4, Chrome
// 3-5, Opera 9.6-10.5
// 
// About: Examples
// 
// These working examples, complete with fully commented code, illustrate a few
// ways in which this plugin can be used.
// 
// Examples - http://benalman.com/code/projects/javascript-debug/examples/debug/
// 
// About: Revision History
// 
// 0.4 - (6/22/2010) Added missing passthrough methods: exception,
//       groupCollapsed, table
// 0.3 - (6/8/2009) Initial release
// 
// Topic: Pass-through console methods
// 
// assert, clear, count, dir, dirxml, exception, group, groupCollapsed,
// groupEnd, profile, profileEnd, table, time, timeEnd, trace
// 
// These console methods are passed through (but only if both the console and
// the method exists), so use them without fear of reprisal. Note that these
// methods will not be passed through if the logging level is set to 0 via
// <debug.setLevel>.

window.debug = (function(){
  var window = this,
    
    // Some convenient shortcuts.
    aps = Array.prototype.slice,
    con = window.console,
    
    // Public object to be returned.
    that = {},
    
    callback_func,
    callback_force,
    
    // Default logging level, show everything.
    log_level = 9,
    
    // Logging methods, in "priority order". Not all console implementations
    // will utilize these, but they will be used in the callback passed to
    // setCallback.
    log_methods = [ 'error', 'warn', 'info', 'debug', 'log' ],
    
    // Pass these methods through to the console if they exist, otherwise just
    // fail gracefully. These methods are provided for convenience.
    pass_methods = 'assert clear count dir dirxml exception group groupCollapsed groupEnd profile profileEnd table time timeEnd trace'.split(' '),
    idx = pass_methods.length,
    
    // Logs are stored here so that they can be recalled as necessary.
    logs = [];
  
  while ( --idx >= 0 ) {
    (function( method ){
      
      // Generate pass-through methods. These methods will be called, if they
      // exist, as long as the logging level is non-zero.
      that[ method ] = function() {
        log_level !== 0 && con && con[ method ]
          && con[ method ].apply( con, arguments );
      };
      
    })( pass_methods[idx] );
  }
  
  idx = log_methods.length;
  while ( --idx >= 0 ) {
    (function( idx, level ){
      
      // Method: debug.log
      // 
      // Call the console.log method if available. Adds an entry into the logs
      // array for a callback specified via <debug.setCallback>.
      // 
      // Usage:
      // 
      //  debug.log( object [, object, ...] );                               - -
      // 
      // Arguments:
      // 
      //  object - (Object) Any valid JavaScript object.
      
      // Method: debug.debug
      // 
      // Call the console.debug method if available, otherwise call console.log.
      // Adds an entry into the logs array for a callback specified via
      // <debug.setCallback>.
      // 
      // Usage:
      // 
      //  debug.debug( object [, object, ...] );                             - -
      // 
      // Arguments:
      // 
      //  object - (Object) Any valid JavaScript object.
      
      // Method: debug.info
      // 
      // Call the console.info method if available, otherwise call console.log.
      // Adds an entry into the logs array for a callback specified via
      // <debug.setCallback>.
      // 
      // Usage:
      // 
      //  debug.info( object [, object, ...] );                              - -
      // 
      // Arguments:
      // 
      //  object - (Object) Any valid JavaScript object.
      
      // Method: debug.warn
      // 
      // Call the console.warn method if available, otherwise call console.log.
      // Adds an entry into the logs array for a callback specified via
      // <debug.setCallback>.
      // 
      // Usage:
      // 
      //  debug.warn( object [, object, ...] );                              - -
      // 
      // Arguments:
      // 
      //  object - (Object) Any valid JavaScript object.
      
      // Method: debug.error
      // 
      // Call the console.error method if available, otherwise call console.log.
      // Adds an entry into the logs array for a callback specified via
      // <debug.setCallback>.
      // 
      // Usage:
      // 
      //  debug.error( object [, object, ...] );                             - -
      // 
      // Arguments:
      // 
      //  object - (Object) Any valid JavaScript object.
      
      that[ level ] = function() {
        var args = aps.call( arguments ),
          log_arr = [ level ].concat( args );
        
        logs.push( log_arr );
        exec_callback( log_arr );
        
        if ( !con || !is_level( idx ) ) { return; }
        
        con.firebug ? con[ level ].apply( window, args )
          : con[ level ] ? con[ level ]( args )
          : con.log( args );
      };
      
    })( idx, log_methods[idx] );
  }
  
  // Execute the callback function if set.
  function exec_callback( args ) {
    if ( callback_func && (callback_force || !con || !con.log) ) {
      callback_func.apply( window, args );
    }
  };
  
  // Method: debug.setLevel
  // 
  // Set a minimum or maximum logging level for the console. Doesn't affect
  // the <debug.setCallback> callback function, but if set to 0 to disable
  // logging, <Pass-through console methods> will be disabled as well.
  // 
  // Usage:
  // 
  //  debug.setLevel( [ level ] )                                            - -
  // 
  // Arguments:
  // 
  //  level - (Number) If 0, disables logging. If negative, shows N lowest
  //    priority levels of log messages. If positive, shows N highest priority
  //    levels of log messages.
  //
  // Priority levels:
  // 
  //   log (1) < debug (2) < info (3) < warn (4) < error (5)
  
  that.setLevel = function( level ) {
    log_level = typeof level === 'number' ? level : 9;
  };
  
  // Determine if the level is visible given the current log_level.
  function is_level( level ) {
    return log_level > 0
      ? log_level > level
      : log_methods.length + log_level <= level;
  };
  
  // Method: debug.setCallback
  // 
  // Set a callback to be used if logging isn't possible due to console.log
  // not existing. If unlogged logs exist when callback is set, they will all
  // be logged immediately unless a limit is specified.
  // 
  // Usage:
  // 
  //  debug.setCallback( callback [, force ] [, limit ] )
  // 
  // Arguments:
  // 
  //  callback - (Function) The aforementioned callback function. The first
  //    argument is the logging level, and all subsequent arguments are those
  //    passed to the initial debug logging method.
  //  force - (Boolean) If false, log to console.log if available, otherwise
  //    callback. If true, log to both console.log and callback.
  //  limit - (Number) If specified, number of lines to limit initial scrollback
  //    to.
  
  that.setCallback = function() {
    var args = aps.call( arguments ),
      max = logs.length,
      i = max;
    
    callback_func = args.shift() || null;
    callback_force = typeof args[0] === 'boolean' ? args.shift() : false;
    
    i -= typeof args[0] === 'number' ? args.shift() : max;
    
    while ( i < max ) {
      exec_callback( logs[i++] );
    }
  };
  
  return that;
})();


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class
 * Util class to handle colors in the draw2d enviroment.
 * 
 *      // Create a new Color with RGB values
 *      var color = new draw2d.util.Color(127,0,0);
 * 
 *      // of from a hex string
 *      var color2 = new draw2d.util.Color("#f00000");
 *     
 *      // Create a little bit darker color 
 *      var darkerColor = color.darker(0.2); // 20% darker
 *     
 *      // create a optimal text color if 'color' the background color
 *      // (best in meaning of contrast and readability)
 *      var fontColor = color.getIdealTextColor();
 *     
 */
draw2d.util.Color = Class.extend({

    /**
     * @constructor
     * Create a new Color object
     * 
     * @param {Number|String|draw2d.util.Color|Array} red 
     * @param {Number} green 
     * @param {Number} blue 
     */
    init: function( red, green, blue) {
    
      this.hashString = null;
      
      if(typeof red === "undefined" || red===null){
          this.hashString = "none";
      }
      else if(red instanceof draw2d.util.Color){
          if(red.hashString==="none"){
              this.hashString = "none";
          }
          else{
              this.red = red.red;
              this.green = red.green;
              this.blue = red.blue;
          }
      }
      else if(typeof red === "string")
      {
           if (red === "none") {
              this.hashString = "none";
           }
           else {
              var rgb = this.hex2rgb(red);
              this.red = rgb[0];
              this.green = rgb[1];
              this.blue = rgb[2];
          }
      }
      // JSON struct of {red:###, green:###, blue:### }
      else if(typeof red === "object" && typeof red.red==="number")
      {
        this.red= red.red;
        this.green = red.green;
        this.blue = red.blue;
      }
      // array detection 1
      else if(red instanceof Array && red.length===3)
      {
        this.red= red[0];
        this.green = red[1];
        this.blue = red[2];
      }
      // array detection 2
      else if(typeof red === "object" && typeof red.length ==="number" && red.length===3)
      {
        this.red= red[0];
        this.green = red[1];
        this.blue = red[2];
      }
      else
      {
        this.red= parseInt(red);
        this.green = parseInt(green);
        this.blue = parseInt(blue);
      }
    },
    

    /**
     * @method
     * Convert the color object into a HTML CSS representation
     * @return {String} the color in rgb(##,##,##) representation
     **/
    getHTMLStyle:function()
    {
      return "rgb("+this.red+","+this.green+","+this.blue+")";
    },
    
    
    /**
     * @method
     * The red part of the color.
     * 
     * @return {Number} the [red] part of the color.
     **/
    getRed:function()
    {
      return this.red;
    },
    
    
    /**
     * @method
     * The green part of the color.
     * 
     * @return {Number} the [green] part of the color.
     **/
    getGreen:function()
    {
      return this.green;
    },
    
    
    /**
     * @method
     * The blue part of the color
     * 
     * @return {Number} the [blue] part of the color.
     **/
    getBlue:function()
    {
      return this.blue;
    },
    
    /**
     * @method
     * Returns the ideal Text Color. Useful for font color selection by a given background color.
     *
     * @return {draw2d.util.Color} The <i>ideal</i> inverse color.
     **/
    getIdealTextColor:function()
    {
       var nThreshold = 105;
       var bgDelta = (this.red * 0.299) + (this.green * 0.587) + (this.blue * 0.114);
       return (255 - bgDelta < nThreshold) ? new  draw2d.util.Color(0,0,0) : new  draw2d.util.Color(255,255,255);
    },
    
    
    /**
     * @private
     */
    hex2rgb:function(/*:String */hexcolor)
    {
      hexcolor = hexcolor.replace("#","");
      return(
             {0:parseInt(hexcolor.substr(0,2),16),
              1:parseInt(hexcolor.substr(2,2),16),
              2:parseInt(hexcolor.substr(4,2),16)}
             );
    },
    
    /**
     * @private
     **/
    hex:function()
    { 
      return(this.int2hex(this.red)+this.int2hex(this.green)+this.int2hex(this.blue)); 
    },
    
    
    /**
     * @method
     * Convert the color object into a HTML CSS representation
     * @return {String} the color in #RRGGBB representation
     **/
    hash:function()
    {
        if(this.hashString===null){
            this.hashString= "#"+this.hex();
        }
        return this.hashString;
    },
    
    /**
     * @private
     */
    int2hex:function(v) 
    {
      v=Math.round(Math.min(Math.max(0,v),255));
      return("0123456789ABCDEF".charAt((v-v%16)/16)+"0123456789ABCDEF".charAt(v%16));
    },
    
    /**
     * @method
     * Returns a darker color of the given one. The original color is unchanged.
     * 
     * @param {Number} fraction  Darkness fraction between [0..1].
     * @return{draw2d.util.Color}        Darker color.
     */
    darker:function(fraction)
    {
       // we can "darker" a undefined color. In this case we return the undefnied color itself
       //
       if(this.hashString==="none")
           return this;
        
       var red   = parseInt(Math.round (this.getRed()   * (1.0 - fraction)));
       var green = parseInt(Math.round (this.getGreen() * (1.0 - fraction)));
       var blue  = parseInt(Math.round (this.getBlue()  * (1.0 - fraction)));
    
       if (red   < 0) red   = 0; else if (red   > 255) red   = 255;
       if (green < 0) green = 0; else if (green > 255) green = 255;
       if (blue  < 0) blue  = 0; else if (blue  > 255) blue  = 255;
    
       return new draw2d.util.Color(red, green, blue);
    },
    
    
    /**
     * @method
     * Make a color lighter. The original color is unchanged.
     * 
     * @param {Number} fraction  lighter fraction between [0..1].
     * @return {draw2d.util.Color} Lighter color.
     */
    lighter:function( fraction)
    {
        // we can "lighter" a undefined color. In this case we return the undefined color itself
        //
        if(this.hashString==="none")
            return this;
        
        var red   = parseInt(Math.round (this.getRed()   * (1.0 + fraction)));
        var green = parseInt(Math.round (this.getGreen() * (1.0 + fraction)));
        var blue  = parseInt(Math.round (this.getBlue()  * (1.0 + fraction)));
    
        if (red   < 0) red   = 0; else if (red   > 255) red   = 255;
        if (green < 0) green = 0; else if (green > 255) green = 255;
        if (blue  < 0) blue  = 0; else if (blue  > 255) blue  = 255;
    
        return new draw2d.util.Color(red, green, blue);
    },
    
    /**
     * @method
     * Return a new color wich is faded to the given color.
     * @param {draw2d.util.Color} color
     * @param {Number} pc the fade percentage in [0..1]
     * @returns {draw2d.util.Color}
     * 
     * @since 2.1.0
     */
    fadeTo: function(color, pc){

        var r= Math.floor(this.red+(pc*(color.red-this.red)) + .5);
        var g= Math.floor(this.green+(pc*(color.green-this.green)) + .5);
        var b= Math.floor(this.blue+(pc*(color.blue-this.blue)) + .5);

        return new draw2d.util.Color(r,g,b);   
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.util.ArrayList
 * An ArrayList stores a variable number of objects. This is similar to making an array of 
 * objects, but with an ArrayList, items can be easily added and removed from the ArrayList 
 * and it is resized dynamically. This can be very convenient, but it's slower than making
 * an array of objects when using many elements. 
 */
draw2d.util.ArrayList = Class.extend({

    /**
     * @constructor
     * Initializes a new instance of the ArrayList class that is empty and has
     * the default initial capacity.
     * 
     */
    init: function( a) {
        this.increment = 10;
        this.size = 0;
        this.data = new Array(this.increment); 
        if(typeof a !=="undefined"){
            $.each(a,$.proxy(function(i,e){this.add(e);},this));
        }
    },
    
    
     /**
      * @method
      * Reverses the order of the elements in the ArrayList. The array will be modified!
      * 
     * @return {draw2d.util.ArrayList} self
      */
     reverse:function()
     {
        var newData = new Array(this.size);
        for (var i=0; i<this.size; i++)
        {
           newData[i] = this.data[this.size-i-1];
        }
        this.data = newData;
        
        return this;
     },
    
     /**
      * @method
      * Returns the allocated/reserved entries. Not all entries are filled with an valid element.
      * 
      * @return {Number} the size of the allocated entries
      */
     getCapacity:function() 
     {
        return this.data.length;
     },
    
     /**
      * @method
      * The size/count of the stored objects.
      *
      * @return {Number}
      */
     getSize:function() 
     {
        return this.size;
     },
     
    
     /**
      * @method 
      * checks to see if the Vector has any elements.
      * 
      * @return {Boolean} true if the list is empty
      **/
     isEmpty:function() 
     {
        return this.getSize() === 0;
     },
    
     /**
      * @method
      * return the last element.
      * 
      * @return {Object}
      */
     last:function() 
     {
        if (this.data[this.getSize() - 1] !== null) 
        {
           return this.data[this.getSize() - 1];
        }
        return null;
     },
     /* @deprecated */
     getLastElement:function(){return this.last();},
     
    
     /**
      * @method
      * Return a reference to the internal javascript native array.
      * 
      * @return {Array}
      */
     asArray:function() 
     {
       this.trimToSize();
       return this.data;
     },
    
     /**
      * @method
      * returns the first element
      * 
      * @return {Object}
      */
     first:function() 
     {
        if (this.data[0] !== null && typeof this.data[0] !=="undefined") 
        {
           return this.data[0];
        }
        return null;
     },
     /* @deprecated */
     getFirstElement: function(){return this.first();},
    
     
     /**
      * @method
      * returns an element at a specified index
      *
      * @param {Number} i
      * @return {Object}
      */
     get:function(i)
     {
        return this.data[i];
     },

    /**
     * @method
     * Adds a element at the end of the Vector.
     *
     * @param {Object} obj the object to add
     * @return {draw2d.util.ArrayList} self
     */
     add:function(obj)
     {
        if(this.getSize() == this.data.length) 
        {
           this.resize();
        }
        this.data[this.size++] = obj;
        
        return this;
     },

     /**
      * @method
      * 
      * The method removes items from an array as necessary so that all remaining items pass a 
      * provided test. The test is a function that is passed an array item and the index of the 
      * item within the array. Only if the test returns true will the item stay in the array.
      * 
      * @param {Function} func the filter function
      * @since 2.0.0
      */
     grep: function(func){
         this.trimToSize();
         this.data = $.grep(this.data, func);
         this.data = $.grep(this.data, function(e){
             return (typeof e !=="undefined");
         });
         this.size = this.data.length;
 
         return this;
     },
     
     /**
      * @method
      * Translate all items in the array into new items. The array list is modified after this call. 
      * You must clone the array before if you want avoid this.
      * 
      *     var labels = this.commands.clone().map(function(e){
      *          return e.getLabel();
      *     });
      *
      * @param {Function} func The function to process each item against. The first argument to the function is the value; the second argument is the index or key of the array or object property.
      * @since 4.0.0
      */
     map: function(func){
         this.trimToSize();
         this.data = $.map(this.data, func);
         this.data = $.grep(this.data, function(e){
             return (typeof e !=="undefined");
         });
         this.size = this.data.length;
 
         return this;
     },
 
     /**
      * @method
      * Removes any duplicate elements from the array. The array is modified after this call. You
      * must clone the array before if you want avoid this
      * 
     * @since 4.0.0
      */
     unique: function(){
         this.trimToSize();
         this.data = $.unique(this.data);
         this.data = $.grep(this.data, function(e){
             return (typeof e !=="undefined");
         });
         this.size = this.data.length;
 
         return this;
     },

     
    /**
     * @method
     * Add all elements into this array.
     *
     * @param {draw2d.util.ArrayList} list
     * @param {boolean} [avoidDuplicates] checks whenever the new elements exists before insert if the parameter is to [true] 
     * 
     * @return {draw2d.util.ArrayList} self
     */
     addAll:function(list, avoidDuplicates)
     {
        if(!(list instanceof draw2d.util.ArrayList)){
          throw "Unable to handle unknown object type in ArrayList.addAll";
        }
        var _this=this; // just to avoid $.proxy;
        if(typeof avoidDuplicates==="undefined" || avoidDuplicates===false){
            list.each(function(i,e){
                _this.add(e); 
            });
        }
        else{
            list.each(function(i,e){
                if(!_this.contains(e)){
                    _this.add(e); 
                }
            });
        }
        return this;
     },

     /**
      * @method
      * You can use the Array list as Stack as well. this is the pop method to remove one element
      * at the top of the stack.
      * 
      * @returns
      */
     pop:function() {
         return this.removeElementAt(this.getSize() - 1);
     },
     
     /**
      * @method
      * Push one element at the top of the stack/array
      * 
      * @param path
      */
     push: function( path) {
         this.add(path);
     },
     
     /**
      * @method
      * Remove the element from the list
      *
      * @param {Object} obj the object to remove
      * @return {Object} the removed object or null
      */
     remove:function( obj)
     {
        var index = this.indexOf(obj);
        if(index>=0){
           return this.removeElementAt(index);
        }
        
        return null;
     },


    /**
     * @method
     * Inserts an element at a given position. Existing elements will be shifted
     * to the right.
     *
     * @param {Object} obj the object to insert.
     * @param {Number} index the insert position.
     * 
     * @return {draw2d.util.ArrayList} self
    */
     insertElementAt:function(obj, index) 
     {
        if (this.size == this.capacity) 
        {
           this.resize();
        }
        
        for (var i=this.getSize(); i > index; i--) 
        {
           this.data[i] = this.data[i-1];
        }
        this.data[index] = obj;
        this.size++;
        
        return this;
     },

    /**
     * @method
     * removes an element at a specific index.
     *
     * @param {Number} index the index of the element to remove
     * @return {Object} the removed object
     */
     removeElementAt:function(index)
     {
        var element = this.data[index];
    
        for(var i=index; i<(this.size-1); i++)
        {
           this.data[i] = this.data[i+1];
        }
    
        this.data[this.size-1] = null;
        this.size--;
        
        return element;
     },

    /**
     * @method
     * removes all given elements in the Vector
     * 
     * @param {draw2d.util.ArrayList} elements The elements to remove
     * @return {draw2d.util.ArrayList} self
     */
     removeAll:function(elements)
     {
         $.each(elements, $.proxy(function(i,e){
             this.remove(e);
         },this));
        
        return this;
     },
    
     /**
      * @method
      * Return the zero based index of the given element or -1 if the element
      * not in the list.
      *
      * @param {Object} obj the element to check
      * 
      * @return {Number} the index of the element or -1
      */
     indexOf:function(obj)
     {
        for (var i=0; i<this.getSize(); i++) 
        {
           if (this.data[i] == obj) 
           {
              return i;
           }
        }
        return -1;
     },

    /**
     * @method
     * returns true if the element is in the Vector, otherwise false.
     *
     * @param {Object} obj the object to check
     * @return {boolean}
     */
     contains:function(obj) 
     {
        for (var i=0; i<this.getSize(); i++) 
        {
           if (this.data[i] == obj)
           {
              return true;
           }
        }
        return false;
     },
    
     // resize() -- increases the size of the Vector
     resize:function()
     {
        newData = new Array(this.data.length + this.increment);
    
        for   (var i=0; i< this.data.length; i++) 
        {
           newData[i] = this.data[i];
        }
    
        this.data = newData;
        
        return this;
     },
    
    
     // trimToSize() -- trims the vector down to it's size
     trimToSize:function()
     {
        // nothing to do
        if(this.data.length == this.size)
           return this;
    
        var temp = new Array(this.getSize());
    
        for (var i = 0; i < this.getSize(); i++) 
        {
           temp[i] = this.data[i];
        }
        this.size = temp.length;
        this.data = temp;
        
        return this;
     }, 
    
     /**
      * @method
      * Sorts the collection based on a field name - f
      * 
      * @param {String} the fieldname for the sorting
      * 
      * @return {draw2d.util.ArrayList} self
      */
     sort:function(f) 
     {
        var i, j;
        var currentValue;
        var currentObj;
        var compareObj;
        var compareValue;
    
        for(i=1; i<this.getSize();i++) 
        {
           currentObj = this.data[i];
           currentValue = currentObj[f];
    
           j= i-1;
           compareObj = this.data[j];
           compareValue = compareObj[f];
    
           while(j >=0 && compareValue > currentValue) 
           {
              this.data[j+1] = this.data[j];
              j--;
              if (j >=0) {
                       compareObj = this.data[j];
                       compareValue = compareObj[f];
              }
           }
           this.data[j+1] = currentObj;
        }
        
        return this;
     },
    
     /** 
      * @method
      * copies the contents of a Vector to another Vector returning the new Vector.
      * 
      * @returns {draw2d.util.ArrayList} the new ArrayList
      */
     clone:function() 
     {
        var newVector = new draw2d.util.ArrayList();
    
        for (var i=0; i<this.size; i++) {
           newVector.add(this.data[i]);
        }
    
        return newVector;
     },
    
     
     /**
      * @method
      * Iterate over the array and call the callback method with the given index and element
      *
      * @param {Function} func the callback function to call for each element
      * @param {boolean} reveser optional parameter. Iterate the collection if it set to <b>true</b>
      * @return {boolean}
      */
      each:function(func, reverse) 
      {

         if(typeof reverse !=="undefined" && reverse===true){
             for (var i=this.size-1; i>=0; i--) {
                 if(func(i, this.data[i])===false)
                     break;
              }
          }
         else{
             for (var i=0; i<this.size; i++) {
                if(func(i, this.data[i])===false)
                    break;
             }
         }
      },
     
     // overwriteElementAt() - overwrites the element with an object at the specific index.
     overwriteElementAt:function(obj, index) 
     {
        this.data[index] = obj;
        
        return this;
     },
    
     getPersistentAttributes:function()
     {
        return {
               data: this.data,
               increment: this.increment,
               size: this.getSize()
               };
     },
     
     /**
      * @method 
      * Read all attributes from the serialized properties and transfer them into the shape.
      * 
      * @param {Object} memento
      * @returns 
      */
     setPersistentAttributes : function(memento)
     {
         this.data = memento.data;
         this.increment = memento.increment;
         this.size = memento.size;
     }
     

});

draw2d.util.ArrayList.EMPTY_LIST = new draw2d.util.ArrayList();



/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
// extending raphael with a polygon function
Raphael.fn.polygon = function(pointString) {
  var poly  = ['M'];
  var point = pointString.split(' ');
      
  for(var i=0; i < point.length; i++) {
     var c = point[i].split(',');
     for(var j=0; j < c.length; j++) {
        var d = parseFloat(c[j]);
        if (!isNaN(d))
          poly.push(d);
     };
     if (i == 0)
      poly.push('L');
  }
  poly.push('Z');
  
  return this.path(poly);
};
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.util.UUID
 * Generates a (pseudo) UUID's
 * 
 *      // a UUID in the format 
 *      // xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx (8-4-4-4-12) 
 *      var id = draw2d.util.UUID.create();
 * 
 * @author Andreas Herz
 * @constructor
 * @private
 */
draw2d.util.UUID=function()
{
};


/**
 * @method
 * Generates a unique id.<br>
 * But just for the correctness: <strong>this is no Global Unique Identifier</strong>, it is just a random generator 
 * with the output that looks like a GUID. <br>
 * But may be also useful.
 *
 * @returns {String} the  UUID in the format xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx (8-4-4-4-12) 
 **/
draw2d.util.UUID.create=function()
{
  var segment=function() 
  {
     return (((1+Math.random())*0x10000)|0).toString(16).substring(1);
  };
  return (segment()+segment()+"-"+segment()+"-"+segment()+"-"+segment()+"-"+segment()+segment()+segment());
};

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.util.spline.Spline
 * 
 *  An abstract class defining a general spline object.
 */
draw2d.util.spline.Spline = Class.extend({

    NAME : "draw2d.util.spline.Spline",
    
    /**
     * @constructor 
     */
    init : function()
    {
    },
    
   /**
    * Create a spline based on the given control points.
    * The generated curve starts in the first control point and ends
    * in the last control point.
    * 
    * @param {Array} controlPoints  Control points of spline (x0,y0,z0,x1,y1,z1,...).
    * @param {Number} parts Number of parts to divide each leg into.
    **/
    generate: function(controlPoints, parts){
        throw "inherit classes must implement the method 'draw2d.util.spline.Spline.generate()'";
    }

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.util.spline.CubicSpline
 * 
 * A cubic spline object.
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.util.spline.Spline
 */
draw2d.util.spline.CubicSpline = draw2d.util.spline.Spline.extend(
{
    NAME : "draw2d.util.spline.CubicSpline",
    
    /**
     * @constructor 
     */
    init : function()
    {
        this._super();
    },

    
    /**
    * Create a spline based on the given control points.
    * The generated curve starts in the first control point and ends
    * in the last control point.
    * 
    * @param {Array} controlPoints  Control points of spline (x0,y0,z0,x1,y1,z1,...).
    * @param {Number} parts Number of parts to divide each leg into.
    * 
    * @returns {Array} the new generated array with new draw2d.geo.Point
    */
    generate: function(controlPoints, parts)
    {
        // Endpoints are added twice to get them include in the
        // generated array    
        var cp = new draw2d.util.ArrayList();
        cp.add(controlPoints.get(0));
        cp.addAll(controlPoints);
        cp.add(controlPoints.get(controlPoints.getSize()-1));

      var n = cp.getSize();
      var spline = new draw2d.util.ArrayList();
      spline.add(controlPoints.get(0));
      spline.add( this.p(1, 0, cp) );
        
      for (var i = 1; i < n - 2; i++) {
        for (var j = 1; j <= parts; j++) {
          spline.add( this.p(i, j / parts, cp));
        }
      }
      spline.add(controlPoints.get(controlPoints.getSize()-1));
        
      return spline;      
    },
    
    
      p :function( i,  t,  cp)
      {
        var x = 0.0;
        var y = 0.0;
          
        var k = i-1;
        for (var j = -2; j <= 1; j++) {
          var b = this.blend (j, t);
          var p = cp.get(k++);
          x += b * p.x;
          y += b * p.y;
        }
          
       return new draw2d.geo.Point(x, y);
      },

    
        
      blend : function(i, t) 
      {
        if (i === -2)
            return (((-t + 3) * t - 3) * t + 1) / 6;
        else if (i === -1)
            return (((3 * t - 6) * t) * t + 4) / 6;
        else if (i === 0)
            return (((-3 * t + 3) * t + 3) * t + 1) / 6;
    
        return (t * t * t) / 6;
      }
  
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.util.spline.CatmullRomSpline
 * 
 * A catmull-rom spline object.
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.util.spline.CubicSpline
 */
draw2d.util.spline.CatmullRomSpline = draw2d.util.spline.CubicSpline.extend(
{
    NAME : "draw2d.util.spline.CatmullRomSpline",
    
    /**
     * @constructor 
     */
    init : function()
    {
        this._super();
    },

    
    blend : function(i, t) {
        if (i == -2)
            return ((-t + 2) * t - 1) * t / 2;
        else if (i == -1)
            return (((3 * t - 5) * t) * t + 2) / 2;
        else if (i == 0)
            return ((-3 * t + 4) * t + 1) * t / 2;
        else
            return ((t - 1) * t * t) / 2;
    }
  
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.util.spline.BezierSpline
 * 
 * A bezier spline object.
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.util.spline.Spline
 */
draw2d.util.spline.BezierSpline = draw2d.util.spline.Spline.extend(
{
    NAME : "draw2d.util.spline.BezierSpline",
    
    /**
     * @constructor 
     */
    init : function()
    {
        this._super();
    },


    
    /**
     * Create a spline based on the given control points.
     * The generated curve starts in the first control point and ends
     * in the last control point.
     * 
     * @param {Array} controlPoints  Control points of spline (x0,y0,z0,x1,y1,z1,...).
     * @param {Number} parts Number of parts to divide each leg into.
     * 
     * @returns {Array} the new generated array with new draw2d.geo.Point
     */
    generate: function(controlPoints, parts)
    {
      var n = controlPoints.getSize();
      var spline = new draw2d.util.ArrayList();

      spline.add(this.p(0, 0, controlPoints));
        
      for (var i = 0; i < n - 3; i += 3) {
        for (var j = 1; j <= parts; j++) {
           spline.add(this.p (i, j /  parts, controlPoints));
        }
      }
      
  //    spline.add(controlPoints.get(controlPoints.getSize()-1));
      
      return spline;      
    },

      
      
    p :function( i,  t,  cp)
    {
      var x = 0.0;
      var y = 0.0;
        
      var k = i;
      for (var j = 0; j <= 3; j++) {
        var b = this.blend (j, t);
        var p = cp.get(k++);
        x += b * p.x;
        y += b * p.y;
     }
        
      return new draw2d.geo.Point( x, y);
    },



    blend: function ( i,  t)
    {
      if      (i == 0) return (1 - t) * (1 - t) * (1 - t);
      else if (i == 1) return 3 * t * (1 - t) * (1 - t);
      else if (i == 2) return 3 * t * t * (1 - t);
      else             return t * t * t;
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.geo.PositionConstants
 * Static values for point orientation.
 * 
 */
draw2d.geo.PositionConstants=function()
{
};

draw2d.geo.PositionConstants.NORTH =  1;
draw2d.geo.PositionConstants.SOUTH =  4;
draw2d.geo.PositionConstants.WEST  =  8;
draw2d.geo.PositionConstants.EAST  = 16;

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.geo.Point Util class for geometrie handling.
 */
draw2d.geo.Point = Class.extend({

    NAME : "draw2d.geo.Point",
    
    /**
     * @constructor 
     * Creates a new Point object with the hands over coordinates.
     * @param {Number} x
     * @param {Number} y
     */
    init : function(x, y)
    {
        if(x instanceof draw2d.geo.Point){
            this.x = x.x;
            this.y = x.y;
        }
        else{
            this.x = x;
            this.y = y;
        }

        // limit for the maxi/minimum boundary of this rectangle
        // It is not possible that the rect leave the boundary if set.
        this.bx = null;
        this.by = null;
        this.bw = null;
        this.bh = null;
    },

    
    /**
     * @method
     * Set the boundary of the rectangle. If set, the rectangle is always inside
     * the boundary. A setX or setY will always be adjusted.
     * 
     */
    setBoundary:function(bx, by, bw, bh){
        if(bx instanceof draw2d.geo.Rectangle){
            this.bx = bx.x;
            this.by = bx.y;
            this.bw = bx.w;
            this.bh = bx.h;
        }else
        {
            this.bx = bx;
            this.by = by;
            this.bw = bw;
            this.bh = bh;
        }
        this.adjustBoundary();

        return this;
    },
    

    /**
     * @method
     * @private
     */
    adjustBoundary:function(){
        if(this.bx===null){
            return;
        }
        this.x = Math.min(Math.max(this.bx, this.x), this.bw);
        this.y = Math.min(Math.max(this.by, this.y), this.bh);
        
        return this;
    },
    
    /**
     * @method
     * Moves this Rectangle horizontally by dx and vertically by dy, then returns 
     * this Rectangle for convenience.<br>
     * <br>
     * The method return the object itself. This allows you to do command chaining, where 
     * you can perform multiple methods on the same elements.
     *
     * @param {Number} dx  Shift along X axis
     * @param {Number} dy  Shift along Y axis
     * 
     **/
    translate:function( dx,  dy)
    {
      this.x +=dx;
      this.y +=dy;
      this.adjustBoundary();
      
      return this;
    },
        
    /**
     * @method 
     * The X value of the point
     * @since 0.1
     * @return {Number}
     */
    getX : function()
    {
        return this.x;
    },

    /**
     * @method 
     * The y value of the point
     * 
     * @return {Number}
     */
    getY : function()
    {
        return this.y;
    },

    /**
     * @method 
     * Set the new X value of the point
     * 
     * @param {Number} x the new value
     */
    setX : function(x)
    {
        this.x = x;
        this.adjustBoundary();
        
        return this;
    },

    /**
     * @method 
     * Set the new Y value of the point
     * 
     * @param {Number}y the new value
     */
    setY : function(y)
    {
        this.y = y;
        this.adjustBoundary();
        
        return this;
    },

    /**
     * @method
     * Set the new x/y coordinates of this point
     * 
     * @param {Number|draw2d.geo.Point} x
     * @param {Number} [y]
     */
    setPosition:function(x,y){
    	if(x instanceof draw2d.geo.Point){
     	   this.x=x.x;
    	   this.y=x.y;
    	}
    	else{
    	   this.x=x;
    	   this.y=y;
    	}
        this.adjustBoundary();

        return this;
    },
    
    /**
     * @method 
     * Calculates the relative position of the specified Point to this Point.
     * 
     * @param {draw2d.geo.Point} p The reference Point
     * @return {draw2d.geo.PositionConstants} NORTH, SOUTH, EAST, or WEST, as defined in {@link draw2d.geo.PositionConstants}
     */
    getPosition : function(p)
    {
        var dx = p.x - this.x;
        var dy = p.y - this.y;
        if (Math.abs(dx) > Math.abs(dy))
        {
            if (dx < 0)
                return draw2d.geo.PositionConstants.WEST;
            return draw2d.geo.PositionConstants.EAST;
        }
        if (dy < 0)
            return draw2d.geo.PositionConstants.NORTH;
        return draw2d.geo.PositionConstants.SOUTH;
    },

    /**
     * @method 
     * Compares two points and return [true] if x and y are equals.
     * 
     * @param {draw2d.geo.Point} p the point to compare with
     * @return boolean
     */
    equals : function(p)
    {
        return this.x === p.x && this.y === p.y;
    },

    /**
     * @method 
     * Return the distance between this point and the hands over.
     * 
     * @param {draw2d.geo.Point} other the point to use
     * @return {Number}
     */
    getDistance : function(other)
    {
        return Math.sqrt((this.x - other.x) * (this.x - other.x) + (this.y - other.y) * (this.y - other.y));
    },

    /**
     * @method 
     * Return the th of the vector from [0,0]
     * 
     * @return {Number}
     * @since 2.10.0
     */
    length : function()
    {
        return Math.sqrt(this.x  * this.x  + this.y * this.y);
    },

    /**
     * @method 
     * Return a new Point translated with the x/y values of the hands over point.
     * 
     * @param {draw2d.geo.Point} other the offset to add for the new point.
     * @return {draw2d.geo.Point} The new translated point.
     */
    getTranslated : function(other)
    {
        return new draw2d.geo.Point(this.x + other.x, this.y + other.y);
    },

    /**
     * @method 
     * Return a new Point scaled with the x/y values of the hands over point.
     * 
     * @param {Number} factor the factor to scaled the new point.
     * @return {draw2d.geo.Point} The new translated point.
     */
    getScaled : function(factor)
    {
        return new draw2d.geo.Point(this.x * factor, this.y * factor);
    },

    /**
     * @method 
     * Return an objects with all important attributes for XML or JSON serialization
     * 
     * @returns {Object}
     */
    getPersistentAttributes : function()
    {
        return {
            x : this.x,
            y : this.y
        };
    },
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     * @returns 
     */
    setPersistentAttributes : function(memento)
    {
        this.x    = memento.x;
        this.y    = memento.y;
    },
    
    /**
     * @method
     * substract the given point and return the new point.
     * 
     * @param that
     * @returns {draw2d.geo.Point}
     */
    subtract:function(that)
    {
    	return new draw2d.geo.Point(this.x-that.x,this.y-that.y);
    },
    
    
    dot:function(that)
    {
    	return this.x*that.x+this.y*that.y;
    },

    cross:function(that)
    {
    	return this.x*that.y-this.y*that.x;
    },

    
    lerp:function(that,t)
    {
    	return new draw2d.geo.Point(this.x+(that.x-this.x)*t,this.y+(that.y-this.y)*t);
    },
    

    /**
     * @method 
     * Clone the Point and return them
     * 
     * @returns 
     */
    clone : function()
    {
       return new draw2d.geo.Point(this.x,this.y);
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.geo.Rectangle
 * 
 * Util class for geometrie handling.
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.geo.Point
 */
draw2d.geo.Rectangle = draw2d.geo.Point.extend({

    NAME : "draw2d.geo.Rectangle",
    
    /**
     * @constructor 
     * Creates a new Point object with the hands over coordinates.
     * @param {Number} x
     * @param {Number} y
     * @param {Number} w
     * @param {Number} h
     */
    init : function( x, y,  w, h)
    {
    	this._super(x,y);
        this.w = w;
        this.h = h;
    },


    /**
     * @method
     * @private
     */
    adjustBoundary:function(){
        if(this.bx===null){
            return;
        }
        this.x = Math.min(Math.max(this.bx, this.x), this.bw-this.w);
        this.y = Math.min(Math.max(this.by, this.y), this.bh-this.h);
        this.w = Math.min(this.w, this.bw);
        this.h = Math.min(this.h, this.bh);
    },
    
	/**
	 * @method
	 * Resizes this Rectangle by the values supplied as input and returns this for 
	 * convenience. This Rectangle's width will become this.width + dw. This 
	 * Rectangle's height will become this.height + dh.
	 * <br>
	 * The method return the object itself. This allows you to do command chaining, where 
	 * you can perform multiple methods on the same elements.
	 *
	 *
	 * @param {Number} dw  Amount by which width is to be resized
	 * @param {Number} dh  Amount by which height is to be resized
	 * 
	 * @return  {draw2d.geo.Rectangle} The method return the object itself
	 **/
	resize:function(/*:int*/ dw, /*:int*/ dh)
	{
	  this.w +=dw;
	  this.h +=dh;
      this.adjustBoundary();
	  return this;
	},
	
    /**
     * @method
     * Scale this Rectangle by the values supplied as input and returns this for 
     * convenience. This Rectangle's width will become this.width + dw. This 
     * Rectangle's height will become this.height + dh. The top left corner moves
     * -dw/2, -dh/2
     * <br>
     * The method return the object itself. This allows you to do command chaining, where 
     * you can perform multiple methods on the same elements.
     *
     *
     * @param {Number} dw  Amount by which width is to be resized
     * @param {Number} dh  Amount by which height is to be resized
     * 
     * @return  {draw2d.geo.Rectangle} The method return the object itself
     **/
    scale:function( dw, dh)
    {
        
      this.w +=(dw);
      this.h +=(dh);
      this.x -=(dw/2);
      this.y -=(dh/2);
      this.adjustBoundary();
      return this;
    },
    
    /**
	 * Sets the parameters of this Rectangle from the Rectangle passed in and
	 * returns this for convenience.<br>
	 * <br>
	 * The method return the object itself. This allows you to do command chaining, where 
	 * you can perform multiple methods on the same elements.
	 *
	 * @param {draw2d.geo.Rectangle} Rectangle providing the bounding values
	 * 
	 * @return  {draw2d.geo.Rectangle} The method return the object itself
	 */
	setBounds:function( rect)
	{
	    this.setPosition(rect.x,rect.y);

	    this.w = rect.w;
	    this.h = rect.h;
	    
  	   return this;
	},
	
	/**
	 * @method
	 * Returns <code>true</code> if this Rectangle's width or height is less than or
	 * equal to 0.
	 * 
	 * @return {Boolean}
	 */
	isEmpty:function()
	{
	  return this.w <= 0 || this.h <= 0;
	},
	
	/**
	 * @method
	 * The width of the dimension element.
	 * 
	 * @return {Number}
	 **/
	getWidth:function()
	{
	  return this.w;
	},
	
	/**
	 * @method
	 * Set the new width of the rectangle.
	 * 
	 * @param {Number} w the new width of the rectangle
	 */
	setWidth: function(w){
      this.w = w;
      this.adjustBoundary();
      return this;
	},
	
	/**
	 * @method
	 * The height of the dimension element.
	 * 
	 * @return {Number}
	 **/
	getHeight:function()
	{
	  return this.h;
	},
    /**
     * @method
     * Set the new height of the rectangle.
     * 
     * @param {Number} h the new height of the rectangle
     */
    setHeight: function(h){
      this.h = h;
      this.adjustBoundary();
      return this;
    },	
    
    /**
     * @method
     * The x coordinate of the left corner.
     * 
     * @return {Number}
     **/
    getLeft:function()
    {
      return this.x;
    },
    
	/**
	 * @method
	 * The x coordinate of the right corner.
	 * 
	 * @return {Number}
	 **/
	getRight:function()
	{
	  return this.x+this.w;
	},
	
    /**
     * @method
     * The y coordinate of the top.
     * 
     *@return {Number}
     **/
    getTop:function()
    {
      return this.y;
    },
    
    /**
	 * @method
	 * The y coordinate of the bottom.
	 * 
	 *@return {Number}
	 **/
	getBottom:function()
	{
	  return this.y+this.h;
	},
	
	/**
	 * @method
	 * The top left corner of the dimension object.
	 * 
	 * @return {draw2d.geo.Point} a new point objects which holds the coordinates
	 **/
	getTopLeft:function()
	{
	  return new draw2d.geo.Point(this.x,this.y);
	},
	
    /**
     * @method
     * The top center coordinate of the dimension object.
     * 
     * @return {draw2d.geo.Point} a new point objects which holds the coordinates
     **/
    getTopCenter:function()
    {
      return new draw2d.geo.Point(this.x+(this.w/2),this.y);
    },

    /**
	 * @method
	 * The top right corner of the dimension object.
	 * 
	 * @return {draw2d.geo.Point} a new point objects which holds the coordinates
	 **/
	getTopRight:function()
	{
	  return new draw2d.geo.Point(this.x+this.w,this.y);
	},
		
	/**
	 * @method
	 * The bottom left corner of the dimension object.
	 * 
	 * @return {draw2d.geo.Point} a new point objects which holds the coordinates
	 **/
	getBottomLeft:function()
	{
	  return new draw2d.geo.Point(this.x,this.y+this.h);
	},
	
	/**
     * @method
     * The bottom center coordinate of the dimension object.
     * 
     * @return {draw2d.geo.Point} a new point objects which holds the coordinates
     **/
    getBottomCenter:function()
    {
      return new draw2d.geo.Point(this.x+(this.w/2),this.y+this.h);
    },
    
	/**
	 * @method
	 * The center of the dimension object
	 * 
	 * @return {draw2d.geo.Point} a new point which holds the center of the object
	 **/
	getCenter:function()
	{
	  return new draw2d.geo.Point(this.x+this.w/2,this.y+this.h/2);
	},
	
	
	/**
	 * @method
	 * Bottom right corner of the object
	 * 
	 * @return {draw2d.geo.Point} a new point which holds the bottom right corner
	 **/
	getBottomRight:function()
	{
	  return new draw2d.geo.Point(this.x+this.w,this.y+this.h);
	},
	
	/**
	 * @method
	 * Return all points of the rectangle as array. Starting at topLeft and the
	 * clockwise.
	 * 
	 * @return {draw2d.util.ArrayList} the points starting at top/left and the clockwise
	 */
	getVertices:function()
	{
	    var result = new draw2d.util.ArrayList();
        result.add(this.getTopLeft());
        result.add(this.getTopRight());
        result.add(this.getBottomRight());
        result.add(this.getBottomLeft());

        return result;
	},
	/* @deprecated */
	getPoints: function(){return this.getVertices();},
	
	/**
	 * @method
	 * Return a new rectangle which fits into this rectangle. <b>ONLY</b> the x/y coordinates
	 * will be changed. Not the dimension of the given rectangle.
	 * 
	 * @param {draw2d.geo.Rectangle} rect the rectangle to adjust
	 * @return the new shifted rectangle
	 */
	moveInside: function(rect){
	    var newRect = new draw2d.geo.Rectangle(rect.x,rect.y,rect.w,rect.h);
	    // shift the coordinate right/down if coordinate not inside the rect
	    //
	    newRect.x= Math.max(newRect.x,this.x);
	    newRect.y= Math.max(newRect.y,this.y);
	    
	    // ensure that the right border is inside this rect (if possible). 
	    //
	    if(newRect.w<this.w){
	        newRect.x = Math.min(newRect.x+newRect.w, this.x+this.w)-newRect.w; 
	    }
	    else{
	        newRect.x = this.x;
	    }
	    
	    // ensure that the bottom is inside this rectangle
	    //
        if(newRect.h<this.h){
            newRect.y = Math.min(newRect.y+newRect.h, this.y+this.h)-newRect.h; 
        }
        else{
            newRect.y = this.y;
        }

        return newRect;
	},
	
	/**
	 * @method
	 * Return the minimum distance of this rectangle to the given {@link draw2d.geo.Point} or 
	 * {link draw2d.geo.Rectangle}.
	 * 
	 * @param {draw2d.geo.Point} pointOrRectangle the reference point/rectangle for the distance calculation
	 */
	getDistance: function (pointOrRectangle){
		var cx = this.x;
		var cy = this.y;
		var cw = this.w;
		var ch = this.h;
		
		var ox = pointOrRectangle.getX();
		var oy = pointOrRectangle.getY();
		var ow = 1;
		var oh = 1;
		
		if(pointOrRectangle instanceof draw2d.geo.Rectangle){
			ow = pointOrRectangle.getWidth();
			oh = pointOrRectangle.getHeight();
		}
		var oct=9;

		// Determin Octant
		//
		// 0 | 1 | 2
		// __|___|__
		// 7 | 9 | 3
		// __|___|__
		// 6 | 5 | 4

		if(cx + cw <= ox){
			if((cy + ch) <= oy){
				oct = 0;
			}
			else if(cy >= (oy + oh)){
				oct = 6;
			}
			else{
				oct = 7;
			}
	    }
		else if(cx >= ox + ow){
			if(cy + ch <= oy){
				oct = 2;
			}
			else if(cy >= oy + oh){
				oct = 4;
			}
			else{
				oct = 3;
			}
		}
		else if(cy + ch <= oy){
			oct = 1;
		}
		else if(cy >= oy + oh){
			oct = 5;
		}
		else{
			return 0;
		}


		// Determin Distance based on Quad
		//
		switch( oct){
			case 0:
				cx = (cx + cw) - ox;
				cy = (cy + ch) - oy;
				return -(cx + cy) ;
			case 1:
				return -((cy + ch) - oy);
			case 2:
				cx = (ox + ow) - cx;
				cy = (cy + ch) - oy;
				return -(cx + cy);
			case 3:
				return -((ox + ow) - cx);
			case 4:
				cx = (ox + ow) - cx;
				cy = (oy + oh) - cy;
				return -(cx + cy);
			case 5:
				return -((oy + oh) - cy);
			case 6:
				cx = (cx + cw) - ox;
				cy = (oy + oh) - cy;
				return -(cx + cy);
			case 7:
				return -((cx + cw) - ox);
		}

		throw "Unknown data type of parameter for distance calculation in draw2d.geo.Rectangle.getDistnace(..)";
	},
	
    
    /**
     * @method
     * Determin Octant
     *
     *    0 | 1 | 2
     *    __|___|__
     *    7 | 8 | 3
     *    __|___|__
     *    6 | 5 | 4
     *
     * @param r1
     * @param r2

     * @returns {Number}
     */
    determineOctant: function( r2){
        
        var HISTERESE= 3; // Tolleranz um diese vermieden wird, dass der Octant "8" zur?�ckgegeben wird
        
        var ox = this.x+HISTERESE;
        var oy = this.y+HISTERESE;
        var ow = this.w-(HISTERESE*2);
        var oh = this.h-(HISTERESE*2);
         
        var cx = r2.x;
        var cy = r2.y;
        var cw = 2;
        var ch = 2;
        if(r2 instanceof draw2d.geo.Rectangle){
            cw = r2.w;
            ch = r2.h;
        }
 
        var oct =0;

        if(cx + cw <= ox){
            if((cy + ch) <= oy){
                oct = 0;
            }
            else if(cy >= (oy + oh)){
                oct = 6;
            }
            else{
                oct = 7;
            }
        }
        else if(cx >= ox + ow){
            if(cy + ch <= oy){
                oct = 2;
            }
            else if(cy >= oy + oh){
                oct = 4;
            }
            else{
                oct = 3;
            }
        }
        else if(cy + ch <= oy){
            oct = 1;
        }
        else if(cy >= oy + oh){
            oct = 5;
        }
        else{
            oct= 8;
        }
        
        return oct;
    },
  
    
    /**
     * @method
     * Returns the direction the point <i>p</i> is in relation to the given rectangle.
     * Util method for inherit router implementations.
     * 
     * <p>
     * Possible values:
     * <ul>
     *   <li>up -&gt; 0</li>
     *   <li>right -&gt; 1</li>
     *   <li>down -&gt; 2</li>
     *   <li>left -&gt; 3</li>
     * </ul>
     * <p>
     * 
     * @param {draw2d.geo.Point} p the point in relation to the given rectangle
     * 
     * @return {Number} the direction from <i>r</i> to <i>p</i>
     */
    getDirection:function(other) 
    {
        var current = this.getTopLeft();
        switch(this.determineOctant(other)){
            case 0:
                if((current.x-other.x)<(current.y-other.y))
                    return draw2d.geo.Rectangle.DIRECTION_UP;
                return draw2d.geo.Rectangle.DIRECTION_LEFT;
            case 1:
                return draw2d.geo.Rectangle.DIRECTION_UP;
            case 2:
                current = this.getTopRight();
                if((other.x-current.x)<(current.y-other.y))
                    return draw2d.geo.Rectangle.DIRECTION_UP;
                return draw2d.geo.Rectangle.DIRECTION_RIGHT;
            case 3:
                return draw2d.geo.Rectangle.DIRECTION_RIGHT;
            case 4:
                current = this.getBottomRight();
                if((other.x-current.x)<(other.y-current.y))
                    return draw2d.geo.Rectangle.DIRECTION_DOWN;
                return draw2d.geo.Rectangle.DIRECTION_RIGHT;
            case 5:
                return draw2d.geo.Rectangle.DIRECTION_DOWN;
            case 6:
                current = this.getBottomLeft();
                if((current.x-other.x)<(other.y-current.y))
                    return draw2d.geo.Rectangle.DIRECTION_DOWN;
                return draw2d.geo.Rectangle.DIRECTION_LEFT;
            case 7:
                return draw2d.geo.Rectangle.DIRECTION_LEFT;
            case 8: 
                if(other.y>this.y){
                    return draw2d.geo.Rectangle.DIRECTION_DOWN;
                }
                return draw2d.geo.Rectangle.DIRECTION_UP;
            
        }
        return draw2d.geo.Rectangle.DIRECTION_UP;
    },
    
    
	/**
	 * @method
	 * Compares two Dimension objects
	 * 
	 * @param {draw2d.geo.Rectangle} o
	 *@return {Boolean}
	 **/
	equals:function( o)
	{
	  return this.x==o.x && this.y==o.y && this.w==o.w && this.h==o.h;
	},
	
    /**
     * @method
     * Detect whenever the hands over coordinate is inside the figure.
     *
     * @param {Number/draw2d.geo.Point} iX
     * @param {Number} iY
     * @returns {Boolean}
     */
    hitTest : function ( iX , iY)
    {
    	if(iX instanceof draw2d.geo.Point){
    		iY = iX.y;
    		iX = iX.x;
    	}
        var iX2 = this.x + this.getWidth();
        var iY2 = this.y + this.getHeight();
        return (iX >= this.x && iX <= iX2 && iY >= this.y && iY <= iY2);
    },
    
    /**
     * @method
     * return true if the this rectangle inside the hand over rectangle
     * 
     *
     * @param {draw2d.geo.Rectangle} rect
     * @returns {Boolean}
     */
    isInside : function ( rect)
    {
       	return    rect.hitTest(this.getTopLeft()) 
    	       && rect.hitTest(this.getTopRight())
    	       && rect.hitTest(this.getBottomLeft()) 
    	       && rect.hitTest(this.getBottomRight());
    },
    
    intersects: function (rect)
    {
        x11 = rect.x,
        y11 = rect.y,
        x12 = rect.x + rect.w,
        y12 = rect.y + rect.h,
        x21 = this.x,
        y21 = this.y,
        x22 = this.x + this.w,
        y22 = this.y + this.h;
  
        x_overlap = Math.max(0, Math.min(x12,x22) - Math.max(x11,x21));
        y_overlap = Math.max(0, Math.min(y12,y22) - Math.max(y11,y21));
 
        return x_overlap*y_overlap!==0;
    },
    
    toJSON : function(){
        return  { 
              width:this.w,
              height:this.h,
              x : this.x,
              y :this.y
          };
      }


});

/**
 * ENUM for Direction
 */
draw2d.geo.Rectangle.DIRECTION_UP    =0;
draw2d.geo.Rectangle.DIRECTION_RIGHT =1;
draw2d.geo.Rectangle.DIRECTION_DOWN  =2;
draw2d.geo.Rectangle.DIRECTION_LEFT  =3;
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/


draw2d.geo.Ray = draw2d.geo.Point.extend({

    NAME : "draw2d.geo.Ray",
    
    /**
     * @constructor 
     * Creates a new Point object with the hands over coordinates.
     * @param {Number} x
     * @param {Number} y
     */
    init : function( x, y)
    {
        this._super(x,y);
    },
    
    
    isHorizontal:function() 
    {
       return this.x != 0;
    },
    
    similarity:function( otherRay) 
    {
       return Math.abs(this.dot(otherRay));
    },
    
    getAveraged:function( otherRay) 
    {
        return new draw2d.geo.Ray((this.x + otherRay.x) / 2, (this.y + otherRay.y) / 2);
    }

});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/

/**
 * @class draw2d.command.CommandType
 * 
 * EditPolicies should determine an Figures editing capabilities. 
 * It is possible to implement an Figure such that it handles all editing 
 * responsibility.<br> 
 * However, it is much more flexible and object-oriented to use 
 * EditPolicies. Using policies, you can pick and choose the editing behavior for 
 * an Figure without being bound to its class hierarchy. Code reuse is increased, 
 * and code management is easier. 
 * 
 * @author Andreas Herz
 */
draw2d.command.CommandType = Class.extend({
	
    NAME : "draw2d.command.CommandType",

    /**
     * @constructor
     * Create a new edit policy object
     * 
     * @param {String} policy 
     */
    init: function( policy) {
       this.policy = policy;
    },

    /**
     * @method
     * Return the String representation of the policy
     * 
     * @return {String}
     **/
    getPolicy:function()
    {
       return this.policy;
    }
});
 
draw2d.command.CommandType.DELETE               = "DELETE";
draw2d.command.CommandType.MOVE                 = "MOVE";
draw2d.command.CommandType.CONNECT              = "CONNECT";
draw2d.command.CommandType.MOVE_BASEPOINT       = "MOVE_BASEPOINT";
draw2d.command.CommandType.MOVE_VERTEX          = "MOVE_VERTEX";
draw2d.command.CommandType.MOVE_VERTICES        = "MOVE_VERTICES";
draw2d.command.CommandType.MOVE_GHOST_VERTEX    = "MOVE_GHOST_VERTEX";
draw2d.command.CommandType.RESIZE               = "RESIZE";
draw2d.command.CommandType.RESET                = "RESET";



/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.Command
 * 
 * Commands are passed around throughout editing. They are used to encapsulate and combine 
 * changes to the application's model. An application has a single command stack. Commands must
 * be executed using the command stack rather than directly calling execute.
 * <br> 
 * This is requried for a deneric support for the undo/redo concept within draw2d.<br>
 * 
 * @inheritable
 * @author Andreas Herz
 */
draw2d.command.Command = Class.extend({

    NAME : "draw2d.command.Command", 

    /**
     * @constructor
     * Create a new Command objects which can be execute via the CommandStack.
     * 
     * @param {String} label
     */
    init: function( label) {
        this.label = label;
    },
    
    
    /**
     * @method
     * Returns a label of the Command. e.g. "move figure".
     *
     * @return {String} the label for this command
     **/
    getLabel:function()
    {
       return this.label;
    },
    
    
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modifies the model. e.g.: a CommandMove with [startX,startX] == [endX,endY] should
     * return false. The execution of this Command doesn't modify the model.
     *
     * @return {boolean} return try if the command modify the model or make any relevant changes
     **/
    canExecute:function()
    {
      return true;
    },
    
    /**
     * @method
     * Execute the command the first time.
     * Sup-classes must implement this method.
     *
     * @template
     **/
    execute:function()
    {
    },
    
    /**
     * @method
     * Will be called if the user cancel the operation.
     *
     * @template
     **/
    cancel:function()
    {
    },
    
    /**
     * @method
     * Undo the command.
     * Sup-classes must implement this method.
     *
     * @template
     **/
    undo:function()
    {
    },
    
    /** 
     * @method
     * Redo the command after the user has undo this command.
     * Sup-classes must implement this method.
     *
     * @template
     **/
    redo:function()
    {
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandCollection
 * 
 * A CommandCollection works as a single command. You can add more than one
 * Command to this CommandCollection and execute/undo them onto the CommandStack as a
 * single Command.
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandCollection = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandCollection", 
    
    /**
     * @constructor
     * Create a new CommandConnect objects which can be execute via the CommandStack.
     * 
     * @param {String} commandLabel the label to show on the command stack for the undo/redo operation
     */
    init : function(commandLabel)
     {
       this._super((typeof commandLabel === 'undefined') ? draw2d.Configuration.i18n.command.collection : commandLabel);
       
       this.commands = new draw2d.util.ArrayList();
    },
    
    /**
     * @method
     * Returns a label of the Command. e.g. "move figure".
     *
     * @return {String} the label for this command
     **/
    getLabel:function()
    {
        //return the label of the one and only command
        //
        if(this.commands.getSize()===1){
           return this.commands.first().getLabel();
        }
        
        // return a common label if all commands have the same label.
        //
        if(this.commands.getSize()>1){
            var labels = this.commands.clone().map(function(e){
                return e.getLabel();
            });
            labels.unique();
            if(labels.getSize()===1){
                return labels.first();
            }
        }
        
        // return the all purpose label.
        return this._super();
    },
    

    
    /**
     * @method
     * Add a command to the collection.
     * 
     * @param {draw2d.command.Command} command
     */
    add: function(command){
    	this.commands.add(command);
    },
    
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modifies the model. e.g.: a CommandMove with [startX,startX] == [endX,endY] should
     * return false. The execution of this Command doesn't modify the model.
     *
     * @return boolean
     **/
    canExecute:function()
    {
        // We ask all cmd's if they make any changes.
        // Keep in mind: the command will be execute if at least ONE command return [true]!!!!
        // doesn't matter if the other commands return [false].
        // The method should be renamed into: modifiesTheModel()....design flaw.
        var canExec = false;
        this.commands.each(function(i,cmd){
            canExec = canExec|| cmd.canExecute();
        });
        return canExec;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
    	this.commands.each(function(i,cmd){
    	    cmd.execute();
    	});
    },
    
    /**
     * @method
     * Redo the command after the user has undo this command.
     *
     **/
    redo:function()
    {
        this.commands.each(function(i,cmd){
            cmd.redo();
        });
    },
    
    /** 
     * @method
     * Undo the command.
     *
     **/
    undo:function()
    {
        // execute the undo operation in reverse direction.
        
        this.commands.reverse();
        this.commands.each(function(i,cmd){
            cmd.undo();
        });
        this.commands.reverse();
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandStack
 * Stack for undo/redo operations
 */
draw2d.command.CommandStack = Class.extend({
    NAME : "draw2d.command.CommandStack", 


    /**
     * @constructor
     * Create a new CommandStack objects which can be execute via the CommandStack.
     * 
     */
    init : function( ) {
       this.undostack = [];
       this.redostack = [];
       this.maxundo = 50;
       this.transactionCommand = null;
       this.eventListeners = new draw2d.util.ArrayList();
       
       window.onpopstate = $.proxy(function(event) {
    	   if( event.state === null){
    		   return;
    	   }
    	   
    	   /*
    	   var stackLegth = event.state.length;
    	   console.log(stackLegth +"<="+ this.undostack.length );
    	   if(stackLegth <= this.undostack.length ){
    		   console.log("Back");
    		   this.undo();
    	   }
    	   else{
    		   console.log("Forward");
    		   this.redo();
    	   }
    	   console.log("location: " + document.location + ", state: " + JSON.stringify(event.state));
    	   */
       },this);
    },
    
 
    /**
     * @method
     * Set the maximal undo stack size. Entries will be remove if the max. stack 
     * size has been reached.
     *
     * @param {Number} count The maximal undo stack size.
     * 
     **/
    setUndoLimit:function( count)
    {
      this.maxundo = count;
    },
    
    /**
     * @method
     * Remove the undo / redo history. This is useful if the user has been save the 
     * document.
     *
     **/
    markSaveLocation:function()
    {
       this.undostack = [];
       this.redostack = [];

       // fire an empty command to inform all listener that the stack has been changed
       this.notifyListeners(new draw2d.command.Command(), draw2d.command.CommandStack.POST_EXECUTE);
    },
    
    /**
     * @method
     * 
     * Executes the specified Command if possible. Prior to executing the command, a
     * draw2d.command.CommandStackEvent for {@link #PRE_EXECUTE} will be fired to event listeners. 
     * Similarly, after attempting to execute the command, an event for {@link #POST_EXECUTE}
     * will be fired.
     *
     * @param {draw2d.command.Command} command The command to execute.
     * 
     **/
    execute:function(command)
    {
        if(typeof command === "undefined")
            throw "Missing parameter [command] for method call CommandStack.execute";
        
       // nothing to do
       if(command===null)
          return; //silently
    
       // return if the command can't execute or it doesn't change the model
       // => Empty command
       if(command.canExecute()===false)
          return;
    
       // A command stack transaction is open.
       // The execution will be postpone until the transaction will commit
       //
       if(this.transactionCommand!==null){
           this.transactionCommand.add(command);
           return;
       }
       
       this.notifyListeners(command, draw2d.command.CommandStack.PRE_EXECUTE);
    
       this.undostack.push(command);
       command.execute();
    
       // cleanup the redo stack if the user execute a new command.
       // I think this will create a "clean" behaviour of the unde/redo mechanism.
       //
       this.redostack = [];
    
       // monitor the max. undo stack size
       //
       if(this.undostack.length > this.maxundo)
       {
          this.undostack = this.undostack.slice(this.undostack.length-this.maxundo);
       }
       this.notifyListeners(command, draw2d.command.CommandStack.POST_EXECUTE);
       
//       window.history.pushState({length: this.undostack.length}, "title 1", "?undo="+this.undostack.length);
    },
    
    /**
     * @method
     * Opens a transaction for further multiple commands. If you execute a command all
     * {@ #execute} calls will be ignored until you commit the current transaction.
     * 
     * @param {String} commandLabel the label to show for the undo/redo operation
     * 
     * @since 4.0.0
     */
    startTransaction: function(commandLabel){
        this.transactionCommand = new draw2d.command.CommandCollection(commandLabel);
    },
    
    /**
     * @method
     * Commit the running transaction. All commands between the start/end of a transaction
     * can be undo/redo in a single step.
     * 
     * @since 4.0.0
     */
    commitTransaction: function(){
        if(this.transactionCommand===null){
            return;//silently
        }
        
        var cmd = this.transactionCommand;
        this.transactionCommand =null;
        this.execute(cmd);
    },
    
    /**
     * @method
     * Undo on command from the stack and store them on the redo command stack.
     *
     **/
    undo:function()
    {
       var command = this.undostack.pop();
       if(command)
       {
          this.notifyListeners(command, draw2d.command.CommandStack.PRE_UNDO);
          this.redostack.push(command);
          command.undo();
          this.notifyListeners(command, draw2d.command.CommandStack.POST_UNDO);
       }
    },
    
    /** 
     * @method
     * Redo a command after the user has undo a command
     *
     **/
    redo:function()
    {
       var command = this.redostack.pop();
    
       if(command)
       {
          this.notifyListeners(command, draw2d.command.CommandStack.PRE_REDO);
          this.undostack.push(command);
          command.redo();
          this.notifyListeners(command, draw2d.command.CommandStack.POST_REDO);
       }
    },
    
    /**
     * @method
     * Return the label of the next REDO command.
     *
     * @return {String}
    **/
    getRedoLabel:function()
    {
       if(this.redostack.length===0)
         return "";
         
       var command = this.redostack[this.redostack.length-1];
    
       if(command)
       {
          return command.getLabel();
       }
       return "";
    },
    
    
    /**
     * @method
     * Return the label of the next UNDO command.
     *
     * @return {String}
     **/
    getUndoLabel:function()
    {
       if(this.undostack.length===0)
         return "";
         
       var command = this.undostack[this.undostack.length-1];
    
       if(command)
       {
          return command.getLabel();
       }
       return "";
    },
    
    
    /**
     * @method
     * Indicates whenever a REDO is possible.
     * 
     * @return boolean <code>true</code> if it is appropriate to call {@link #redo()}.
     */
    canRedo:function()
    {
       return this.redostack.length>0;
    },
    
    /**
     * @method 
     * indicator whenever a undo is possible.
     * 
     * @return {boolean} <code>true</code> if {@link #undo()} can be called
     **/ 
    canUndo:function()
    {
       return this.undostack.length>0;
    },
    
    /**
     * @method
     * Adds a listener to the command stack, which will be notified whenever a command has been processed on the stack.
     * 
     * @param {draw2d.command.CommandStackEventListener} listener the listener to add.
     */
    addEventListener:function( listener)
    {
        if(listener instanceof draw2d.command.CommandStackEventListener){
          this.eventListeners.add(listener);
        }
        else if(typeof listener.stackChanged ==="function"){
          this.eventListeners.add(listener);
        }
        else if(typeof listener === "function"){
          this.eventListeners.add( {  stackChanged : listener });
        }
        else{
          throw "Object doesn't implement required callback interface [draw2d.command.CommandStackListener]";
        }
    },
    
    /**
     * @method
     * Removes a listener from the command stack.
     * 
     * @param {draw2d.command.CommandStackEventListener} listener the listener to remove.
     */
    removeEventListener:function(listener)
    {
        for (var i = 0; i < size; i++){
            var entry = this.eventListeners.get(i);
            if(entry ===listener || entry.stackChanged === listener){
                this.eventListeners.remove(entry);
                return;
            }
         }
    },
        
    /**
     * @method
     * Notifies command stack event listeners that the command stack has changed to the
     * specified state.
     * 
     * @param {draw2d.command.Command} command the command
     * @param {Number} state the current stack state
     *
     **/
    notifyListeners:function(command,  state)
    {
      var event = new draw2d.command.CommandStackEvent(this, command, state);
      var size = this.eventListeners.getSize();
      for (var i = 0; i < size; i++)
         this.eventListeners.get(i).stackChanged(event);
    }
});


/** Constant indicating notification prior to executing a command (value is 1).*/
draw2d.command.CommandStack.PRE_EXECUTE=1;
/** Constant indicating notification prior to redoing a command (value is 2).*/
draw2d.command.CommandStack.PRE_REDO=2;
/** Constant indicating notification prior to undoing a command (value is 4).*/
draw2d.command.CommandStack.PRE_UNDO=4;
/**  Constant indicating notification after a command has been executed (value is 8).*/
draw2d.command.CommandStack.POST_EXECUTE=8;
/** Constant indicating notification after a command has been redone (value is 16).*/
draw2d.command.CommandStack.POST_REDO=16;
/** Constant indicating notification after a command has been undone (value is 32).*/
draw2d.command.CommandStack.POST_UNDO=32;
/** Constant indicating notification after the stack has been (re)init (value is 64).*/
draw2d.command.CommandStack.POST_INIT=64;

draw2d.command.CommandStack.POST_MASK = draw2d.command.CommandStack.POST_EXECUTE | draw2d.command.CommandStack.POST_UNDO | draw2d.command.CommandStack.POST_REDO;
draw2d.command.CommandStack.PRE_MASK  = draw2d.command.CommandStack.PRE_EXECUTE  | draw2d.command.CommandStack.PRE_UNDO  |draw2d.command.CommandStack.PRE_REDO;



/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandStackEvent
 * Event class which will be fired for every CommandStack operation. Required for CommandStackListener.
 */
draw2d.command.CommandStackEvent = Class.extend({
    NAME : "draw2d.command.CommandStackEvent", 

    /**
     * @constructor
     * Create a new CommandStack objects which can be execute via the CommandStack.
     * @param {draw2d.command.Command} command the related command
     * @param {Number} details the current state of the command execution
     * 
     */
    init : function(stack, command, details)
    {
    	this.stack = stack;
        this.command = command;
        this.details = details;
    },
    
    
    /**
     * @method
     * Return the corresponding stack of the event.
     * 
     * @return {draw2d.command.CommandStack}
     **/
    getStack:function()
    {
       return this.stack;
    },
    
    
    /**
     * @method
     * Returns null or a Command if a command is relevant to the current event.
     * 
     * @return {draw2d.command.Command}
     **/
    getCommand:function()
    {
       return this.command;
    },
    
    /**
     * @method
     * Returns an integer identifying the type of event which has occurred.
     * Defined by {@link draw2d.command.CommandStack}.
     * 
     * @return {Number}
     **/
    getDetails:function()
    {
       return this.details;
    },
    
    
    /**
     * @method
     * Returns true if this event is fired after the stack having changed.
     *
     * @return {boolean} true if post-change event
     **/
    isPostChangeEvent:function()
    {
       return 0 != (this.getDetails() & draw2d.command.CommandStack.POST_MASK);
    },
    
    /**
     * @method
     * Returns true if this event is fired prior to the stack changing.
     * 
     * @return {boolean} true if pre-change event
     **/
    isPreChangeEvent:function()
    {
       return 0 != (this.getDetails() & draw2d.command.CommandStack.PRE_MASK);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandStackEventListener
 * 
 * Event class which will be fired for every CommandStack operation. Required for CommandStackListener.
 * 
 * @author Andreas Herz
 */
draw2d.command.CommandStackEventListener = Class.extend({
    NAME : "draw2d.command.CommandStackEventListener", 

    /**
     * @constructor
     * Creates a new Listener Object
     * 
     */
    init : function()
    {
    },
    
    /**
     * @method
     * Sent when an event occurs on the command stack. draw2d.command.CommandStackEvent.getDetail() 
     * can be used to identify the type of event which has occurred.
     * 
     * @template
     * 
     * @param {draw2d.command.CommandStackEvent} event
     **/
    stackChanged:function(event)
    {
    }

});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandMove
 * 
 * Command for the movement of figures.
 *
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandMove = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandMove", 
  
    /**
     * @constructor
     * Create a new Command objects which can be execute via the CommandStack.
     *
     * @param {draw2d.Figure} figure the figure to move
     * @param {Number} [x] the current x position
     * @param {Number} [y] the current y position
     */
    init : function(figure, x, y)
    {
        this._super(draw2d.Configuration.i18n.command.moveShape);
        this.figure = figure;
        if (typeof x === "undefined")
        {
            this.oldX = figure.getX();
            this.oldY = figure.getY();
        }
        else
        {
            this.oldX = x;
            this.oldY = y;
        }
   },
    
  
    /**
     * @method
     * Set the initial position of the element
     *
     * @param {Number} x the new initial x position
     * @param {Number} y the new initial y position
     **/
    setStartPosition:function( x,  y)
    {
       this.oldX = x;
       this.oldY = y;
    },
    
    /**
     * @method
     * Set the target/final position of the figure move command.
     *
     * @param {Number} x the new x position
     * @param {Number} y the new y position
     **/
    setPosition:function( x,  y)
    {
       this.newX = x;
       this.newY = y;
    },

    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return this.newX!=this.oldX || this.newY!=this.oldY;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     *
     * Undo the move command
     *
     **/
    undo:function()
    {
       this.figure.setPosition(this.oldX, this.oldY);
    },
    
    /**
     * @method
     * 
     * Redo the move command after the user has undo this command
     *
     **/
    redo:function()
    {
       this.figure.setPosition(this.newX, this.newY);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandMoveLine
 * 
 * Command for the movement of figures.
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandMoveLine = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandMoveLine", 
  
    /**
     * @constructor
     * Create a new Command objects which can be execute via the CommandStack.
     *
     * @param {draw2d.Figure} figure the figure to move
     */
    init : function(figure)
    {
        this._super(draw2d.Configuration.i18n.command.moveLine);
        this.line = figure;
        this.dx = 0;
        this.dy = 0;
    },
   
    /**
     * @method
     * set the offset of the line translation
     * 
     * @param {Number} dx
     * @param {Number} dy
     */
    setTranslation: function(dx, dy){
        this.dx = dx;
        this.dy = dy;
    },
    
   /**
    * Returns [true] if the command can be execute and the execution of the
    * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
    * return false. <br>
    * the execution of the Command doesn't modify the model.
    *
    * @type boolean
    **/
   canExecute:function()
   {
     // return false if we doesn't modify the model => NOP Command
     return this.dx !==0 && this.dy !==0;
   },

   /**
    * Execute the command the first time
    * 
    **/
   execute:function()
   {
      this.redo();
   },

   /**
    * Undo the command
    *
    **/
   undo:function()
   {
       this.line.getVertices().each($.proxy(function(i,e){
           e.translate(-this.dx, -this.dy);
       },this));
       this.line.svgPathString = null;
       // required to update resize handles and the painting of the line
       this.line.setPosition(this.line.getStartPoint());
       
       //this.line.repaint();
   },

   /** 
    * Redo the command after the user has undo this command
    *
    **/
   redo:function()
   {
       this.line.getVertices().each($.proxy(function(i,e){
           e.translate(this.dx, this.dy);
       },this));
       this.line.svgPathString = null;       
       
       // required to update resize handles and the painting of the line
       this.line.setPosition(this.line.getStartPoint());

       //this.line.repaint();
   }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandMoveVertex
 * 
 * Command for the vertex movement of a polyline/polygon.
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandMoveVertex = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandMoveVertex", 
  
    /**
     * @constructor
     * Create a new Command objects which can be execute via the CommandStack.
     *
     * @param {draw2d.shape.basic.PolyLine} line the related line
     */
    init : function(line)
    {
        this._super(draw2d.Configuration.i18n.command.moveVertex);
        
        this.line = line;
        this.index = -1;
        this.newPoint = null;
    },
    
  
    /**
     * @method
     * Set the index of the vertex of the polyline/polygon to modify.
     *
     * @param {Number} index the related index of the vertex
     **/
    setIndex:function( index)
    {
       this.index = index;
       this.origPoint = this.line.getVertices().get(this.index).clone();
    },
    
    updatePosition: function(x,y){
        this.newPoint = new draw2d.geo.Point(x,y);
    },
    
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return this.index!==-1 && this.newPoint!==null;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     *
     * Undo the move command
     *
     **/
    undo:function()
    {
        this.line.setVertex(this.index, this.origPoint.x, this.origPoint.y);
    },
    
    /**
     * @method
     * 
     * Redo the move command after the user has undo this command
     *
     **/
    redo:function()
    {
        this.line.setVertex(this.index, this.newPoint.x, this.newPoint.y);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandMoveVertices
 * 
 * Command for the vertices movement of a polyline/polygon.
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandMoveVertices = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandMoveVertices", 
  
    /**
     * @constructor
     * Create a new Command objects which can be execute via the CommandStack.
     *
     * @param {draw2d.shape.basic.PolyLine} line the related line
     */
    init : function(line)
    {
        this._super(draw2d.Configuration.i18n.command.moveVertices);
        
        this.line = line;
        this.oldVertices = line.getVertices().clone();
        this.newVertices = null;
    },
    
  
    
    updateVertices: function(newVertices){
       this.newVertices = newVertices;
    },
    
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return this.newVertices!==null;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     *
     * Undo the move command
     *
     **/
    undo:function()
    {
        this.line.setVertices(this.oldVertices);
    },
    
    /**
     * @method
     * 
     * Redo the move command after the user has undo this command
     *
     **/
    redo:function()
    {
        this.line.setVertices(this.newVertices);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandResize
 * Resize command for figures. Can be execute/undo/redo via a CommandStack.
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.command.Command
 */
draw2d.command.CommandResize = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandResize", 

    /**
     * @constructor
     * Create a new resize Command objects which can be execute via the CommandStack.
     *
     * @param {draw2d.Figure} figure the figure to resize
     * @param {Number} [width] the current width
     * @param {Number} [height] the current height
     */
    init : function(figure, width, height)
    {
        this._super(draw2d.Configuration.i18n.command.resizeShape);
        this.figure = figure;
        
        if (typeof width === "undefined")
        {
            this.oldWidth = figure.getWidth();
            this.oldHeight = figure.getHeight();
        }
        else
        {
            this.oldWidth = width;
            this.oldHeight = height;
        }
    },
  
    /**
     * @method
     * Set the new dimension of the element.
     *
     * @param {Number} width the new width.
     * @param {Number} height the new height of the element.
     **/
    setDimension:function( width, height)
    {
       this.newWidth  = width|0;
       this.newHeight = height|0;
    },

    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return this.newWidth!=this.oldWidth || this.newHeight!=this.oldHeight;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     * Undo the command
     *
     **/
    undo:function()
    {
       this.figure.setDimension(this.oldWidth, this.oldHeight);
    },
    
    /**
     * @method
     * Redo the command after the user has undo this command
     *
     **/
    redo:function()
    {
       this.figure.setDimension(this.newWidth, this.newHeight);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandConnect
 * 
 * Connects two ports with a connection.
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandConnect = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandConnect", 
    
    /**
     * @constructor
     * Create a new CommandConnect objects which can be execute via the CommandStack.
     *
     * @param {draw2d.Canvas} canvas the canvas to user
     * @param {draw2d.Port} source the source port for the connection to create
     * @param {draw2d.Port} target the target port for the connection to create
     * @param {draw2d.Port} [dropTarget] the port who has initiate the connection creation. mainly the drop target
     */
    init : function(canvas, source, target, dropTarget)
     {
       this._super(draw2d.Configuration.i18n.command.connectPorts);
       this.canvas = canvas;
       this.source   = source;
       this.target   = target;
       this.connection = null;
       this.dropTarget= dropTarget; // optional
    },
    
    setConnection:function(connection)
    {
       this.connection=connection;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
        var optionalCallback = $.proxy(function(conn){
            this.connection = conn;
            this.connection.setSource(this.source);
            this.connection.setTarget(this.target);
            this.canvas.addFigure(this.connection);
        },this);
        
        // the createConnection must return either a connection or "undefined". If the method return "undefined"
        // the asynch callback must be called. Usefull if the createConnection shows a selection dialog
        //
        if(this.connection===null){
          var result = draw2d.Connection.createConnection(this.source, this.target, optionalCallback, this.dropTarget);
          // well be handeld by the optional callback
          if(typeof result==="undefined"){
              return;
          }

          this.connection = result;
        }
       
        optionalCallback(this.connection);
    },
    
    /**
     * @method
     * Redo the command after the user has undo this command.
     *
     **/
    redo:function()
    {
       this.canvas.addFigure(this.connection);
       this.connection.reconnect();
    },
    
    /** 
     * @method
     * Undo the command.
     *
     **/
    undo:function()
    {
        this.canvas.removeFigure(this.connection);
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandReconnect
 * 
 * Reconnects two ports. This command is used during the DragDrop operation of a draw2d.Connection.
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandReconnect = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandReconnect", 


    /**
     * @constructor
     * Create a new Command objects which can be execute via the CommandStack.
     *
     * @param {draw2d.Connection} con the related Connection which is currently in the drag&drop operation
     */
    init : function(con){
       this._super(draw2d.Configuration.i18n.command.connectPorts);
       this.con      = con;
       this.oldSourcePort  = con.getSource();
       this.oldTargetPort  = con.getTarget();
//      this.oldRouter      = con.getRouter();
   },
    
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return true;
    },
    
    /**
     * @method
     * The new ports to use during the execute of this command.
     * 
     * @param {draw2d.Port} source
     * @param {draw2d.Port} target
     */
    setNewPorts:function(source,  target)
    {
      this.newSourcePort = source;
      this.newTargetPort = target;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    cancel:function()
    {
        this.con.setSource(this.oldSourcePort);
        this.con.setTarget(this.oldTargetPort);
        
        // force a routing of the connection and DON'T set the old reouter again because this reset all manual added
        // vertices
        this.con.routingRequired =true;
        this.con.repaint();
        
//       this.con.setRouter(this.oldRouter);
    },
    
    /**
     * @method
     * Undo the command
     *
     **/
    undo:function()
    {
      this.con.setSource(this.oldSourcePort);
      this.con.setTarget(this.oldTargetPort);
      // force a routing of the connection and DON'T set the old reouter again because this reset all manual added
      // vertices
      this.con.routingRequired =true;
      this.con.repaint();
//      this.con.setRouter(this.oldRouter);
    },
    
    /** 
     * @method
     * Redo the command after the user has undo this command
     *
     **/
    redo:function()
    {
      this.con.setSource(this.newSourcePort);
      this.con.setTarget(this.newTargetPort);
      // force a routing of the connection and DON'T set the old reouter again because this reset all manual added
      // vertices
      this.con.routingRequired =true;
      this.con.repaint();
//      this.con.setRouter(this.oldRouter);
    }

});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandDelete
 * Command to remove a figure with CommandStack support.
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandDelete = draw2d.command.Command.extend({
    
    /**
     * @constructor
     * Create a delete command for the given figure.
     * 
     * @param {draw2d.Figure} figure
     */
    init: function( figure)
    {
       this._super(draw2d.Configuration.i18n.command.deleteShape);
       this.parent   = figure.getParent();
       this.figure   = figure;
       this.canvas = figure.getCanvas();
       this.connections = null;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     * Undo the command
     *
     **/
    undo:function()
    {
        this.canvas.addFigure(this.figure);
        if(this.figure instanceof draw2d.Connection)
           this.figure.reconnect();
    
        this.canvas.setCurrentSelection(this.figure);
        if(this.parent!==null)
          this.parent.addChild(this.figure);
        for (var i = 0; i < this.connections.getSize(); ++i)
        {
           this.canvas.addFigure(this.connections.get(i));
           this.connections.get(i).reconnect();
        }
    },
    
    /** 
     * @method
     * Redo the command after the user has undo this command
     *
     **/
    redo:function()
    {
        this.canvas.setCurrentSelection(null);
        if(this.figure instanceof draw2d.shape.node.Node && this.connections===null)
        {
          this.connections = new draw2d.util.ArrayList();
          var ports = this.figure.getPorts();
          for(var i=0; i<ports.getSize(); i++)
          {
            var port = ports.get(i);
            // Do NOT add twice the same connection if it is linking ports from the same node
            for (var c = 0, c_size = port.getConnections().getSize() ; c< c_size ; c++)
            {
                if(!this.connections.contains(port.getConnections().get(c)))
                {
                  this.connections.add(port.getConnections().get(c));
                }
            }
          }
          for(var i=0; i<ports.getSize(); i++)
          {
            var port = ports.get(i);
            port.setCanvas(null);
          }
        }
        this.canvas.removeFigure(this.figure);
    
       if(this.connections===null)
          this.connections = new draw2d.util.ArrayList();
    
        // remove this figure from the parent 
        //
        if(this.parent!==null)
          this.parent.removeChild(this.figure);
    
       for (var i = 0; i < this.connections.getSize(); ++i)
       {
          this.canvas.removeFigure(this.connections.get(i));
       }
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandAdd
 * 
 * Command to add a figure with CommandStack support.
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandAdd = draw2d.command.Command.extend({
    
    /**
     * @constructor
     * Create a add command for the given figure.
     * 
     * @param {draw2d.Canvas} canvas the canvas to use
     * @param {draw2d.Figure} figure the figure to add
     * @param {Number|draw2d.geo.Point} x the x-coordinate or a complete point where to place the figure
     * @param {Number} [y] the y-coordinate if x is a number and not a complete point
     */
    init: function(canvas, figure, x,y)
    {
       this._super(draw2d.Configuration.i18n.command.addShape);
       this.figure = figure;
       this.canvas = canvas;
       this.pos = new draw2d.geo.Point(x,y);
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.canvas.addFigure(this.figure, this.pos.x, this.pos.y);
    },
    
    /** 
     * @method
     * Redo the command after the user has undo this command
     *
     **/
    redo:function()
    {
        this.execute();
    },
    
    /**
     * @method
     * Undo the command
     *
     **/
    undo:function()
    {
        this.canvas.removeFigure(this.figure);
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandAddVertex
 * 
 * Add a vertex to a polyline or polygon
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandAddVertex = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandAddVertex", 
  
    /**
     * @constructor
     * Create a new Command objects which add a vertex to a PolyLine / Polygon.
     *
     * @param {draw2d.shape.basic.PolyLine} line the related line
     * @param {Number} index the index where to add
     * @param {Number} x the x coordinate for the new vertex
     * @param {Number} y the y coordinate for the new vertex
     */
    init : function(line, index, x ,y)
    {
        this._super(draw2d.Configuration.i18n.command.addVertex);
        
        this.line = line;
        this.index = index;
        this.newPoint = new draw2d.geo.Point(x,y);
    },
    
  
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return true;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     *
     * Undo the move command
     *
     **/
    undo:function()
    {
        this.line.removeVertexAt(this.index);
    },
    
    /**
     * @method
     * 
     * Redo the move command after the user has undo this command
     *
     **/
    redo:function()
    {
        this.line.insertVertexAt(this.index, this.newPoint.x, this.newPoint.y);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.command.CommandRemoveVertexPoint
 * 
 * Remove a vertex from a polyline or polygon
 *
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.command.Command
 */
draw2d.command.CommandRemoveVertex = draw2d.command.Command.extend({
    NAME : "draw2d.command.CommandRemoveVertex", 
  
    /**
     * @constructor
     * Create a new Command objects which add a vertex to a PloyLine.
     *
     * @param {draw2d.shape.basic.PolyLine} line the related line
     * @param {Number} index the index where to add
     * @param {Number} x the x coordinate for the new vertex
     * @param {Number} y the y coordinate for the new vertex
     */
    init : function(line, index)
    {
        this._super(draw2d.Configuration.i18n.command.deleteVertex);
        
        this.line = line;
        this.index = index;
        this.oldPoint = line.getVertices().get(index).clone();
    },
    
  
    /**
     * @method
     * Returns [true] if the command can be execute and the execution of the
     * command modify the model. A CommandMove with [startX,startX] == [endX,endY] should
     * return false. <br>
     * the execution of the Command doesn't modify the model.
     *
     * @return {boolean}
     **/
    canExecute:function()
    {
      // return false if we doesn't modify the model => NOP Command
      return true;
    },
    
    /**
     * @method
     * Execute the command the first time
     * 
     **/
    execute:function()
    {
       this.redo();
    },
    
    /**
     * @method
     *
     * Undo the move command
     *
     **/
    undo:function()
    {
    	this.line.insertVertexAt(this.index, this.oldPoint.x, this.oldPoint.y);
    },
    
    /**
     * @method
     * 
     * Redo the move command after the user has undo this command
     *
     **/
    redo:function()
    {
    	this.line.removeVertexAt(this.index);
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.ConnectionRouter
 * Routes a {@link draw2d.Connection}, possibly using a constraint.
 *
 * @author Andreas Herz
 */
draw2d.layout.connection.ConnectionRouter = Class.extend({
    NAME : "draw2d.layout.connection.ConnectionRouter",

	/**
	 * @constructor 
	 * Creates a new Router object
	 */
    init: function(){
    },
    
    
    /**
     * @method
     * Routes the Connection.
     * 
     * @param {draw2d.Connection} connection The Connection to route
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
     * @template
     */
    route:function( connection, oldVertices)
    {
    	throw "subclasses must implement the method [ConnectionRouter.route]";
    },
    
    _paint: function(conn){
        // calculate the path string for the SVG rendering
        // Important: to avoid subpixel error rendering we add 0.5 to each coordinate
        //            With this offset the canvas can paint the line on a "full pixel" instead
        //            of subpixel rendering.
        var ps = conn.getVertices();
        var p = ps.get(0);
        var distance = conn.getRadius();
        var path = ["M",(p.x|0)+0.5," ",(p.y|0)+0.5];
        var i=1;
        if(distance>0){
            var lastP = p;
            var length = (ps.getSize()-1);
            for(  ;i<length;i++){
                  p = ps.get(i);
                  inset = this.insetPoint(p,lastP, distance);
                  path.push("L", (inset.x|0)+0.5, ",", (inset.y|0)+0.5);
    
                  p2 = ps.get(i+1);
                  inset = this.insetPoint(p,p2,distance);
                  
                  path.push("Q",p.x,",",p.y," ", (inset.x|0)+0.5, ", ", (inset.y|0)+0.5);
                  lastP = p;
            }
            p = ps.get(i);
            path.push("L", (p.x|0)+0.5, ",", (p.y|0)+0.5);
       }
        else{
            var length = ps.getSize();
            for( ;i<length;i++){
                p = ps.get(i);
                path.push("L", (p.x|0)+0.5, ",", (p.y|0)+0.5);
          }
        }
         conn.svgPathString = path.join("");
     },
     
     insetPoint: function(start, end, distanceFromStart){
         if(start.equals(end)){
             return start;
         }
         var vx = start.x-end.x;
         var vy = start.y-end.y;
         var length = Math.sqrt(vx*vx + vy*vy);
         var localDistance = Math.min(length/2,distanceFromStart);
         return {x: end.x + vx/length * (length - localDistance),
                y: end.y + vy/length * (length - localDistance)};

     },
     

    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection)
    {
        
    },
    
    /**
     * @method
     * Callback method if the router has been removed from the connection.
     * 
     * @param {draw2d.Connection} connection The related connection
     * @template
     * @since 2.7.2
     */
    onUninstall: function(connection)
    {
        
    },
    
    
    /**
     * @method 
     * Tweak or enrich the polyline persistence data with routing information
     * 
     * @since 2.10.0
     * @param {draw2d.shape.basic.PolyLine} line
     * @param {Object} memento The memento data of the polyline
     * @returns {Object}
     */
    getPersistentAttributes : function(line, memento)
    {   
        return memento;
    },
    
    /**
     * @method 
     * set the attributes for the polyline with routing information
     * 
     * @since 2.10.0
     * @param {Object} memento the JSON data to read
     */
    setPersistentAttributes : function(line, memento)
    {
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.DirectRouter
 * Router for direct connections between two ports. Beeline
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     // Override the default connection type. This is used during drag&drop operations of ports.
 *     //
 *     draw2d.Connection.createConnection=function(sourcePort, targetPort){
 *        // return my special kind of connection
 *        var con = new draw2d.Connection();
 *        con.setRouter(new draw2d.layout.connection.DirectRouter());
 *        return con;
 *     };
 *     
 *     // create and add two nodes which contains Ports (In and OUT)
 *     //
 *     var start = new draw2d.shape.node.Start();
 *     var end   = new draw2d.shape.node.End();
        
 *     // ...add it to the canvas 
 *     canvas.addFigure( start, 50,50);
 *     canvas.addFigure( end, 230,80);
 *          
 *     // first Connection
 *     //
 *     var c = draw2d.Connection.createConnection();
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *     canvas.addFigure(c);
 * 
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ConnectionRouter
 */
draw2d.layout.connection.DirectRouter = draw2d.layout.connection.ConnectionRouter.extend({

    NAME : "draw2d.layout.connection.DirectRouter",

    /**
	 * @constructor 
	 * Creates a new Router object
	 */
    init: function(){
        this._super();
    },
    
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    },
 
    /**
     * @method
     * Invalidates the given Connection
     */
    invalidate:function()
    {
    },
    
    /**
     * @method
     * Routes the Connection in air line (beeline).
     * 
     * @param {draw2d.Connection} connection The Connection to route
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
     */
    route:function( connection, oldVertices)
    {
       var start =connection.getStartPoint();
       var end = connection.getEndPoint();
       
       // required for hit tests
       //
       connection.addPoint(start);
       connection.addPoint(end);
       
       // calculate the path
       var path = ["M",start.x," ",start.y];
       path.push("L", end.x, " ", end.y);

       connection.svgPathString = path.join("");

    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.VertexRouter
 * Router for direct connections between two ports. Beeline
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     // Override the default connection type. This is used during drag&drop operations of ports.
 *     //
 *     draw2d.Connection.createConnection=function(sourcePort, targetPort){
 *        // return my special kind of connection
 *        var con = new draw2d.Connection();
 *        con.setRouter(new draw2d.layout.connection.DirectRouter());
 *        return con;
 *     };
 *     
 *     // create and add two nodes which contains Ports (In and OUT)
 *     //
 *     var start = new draw2d.shape.node.Start();
 *     var end   = new draw2d.shape.node.End();
        
 *     // ...add it to the canvas 
 *     canvas.addFigure( start, 50,50);
 *     canvas.addFigure( end, 230,80);
 *          
 *     // first Connection
 *     //
 *     var c = draw2d.Connection.createConnection();
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *     canvas.addFigure(c);
 * 
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ConnectionRouter
 */
draw2d.layout.connection.VertexRouter = draw2d.layout.connection.ConnectionRouter.extend({

    NAME : "draw2d.layout.connection.VertexRouter",

    /**
	 * @constructor 
	 * Creates a new Router object
	 */
    init: function(){
        this._super();
    },
    
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.VertexSelectionFeedbackPolicy());
    },
 
    /**
     * @method
     * Invalidates the given Connection
     */
    invalidate:function()
    {
    },
    
    /**
     * @method
     * Routes the Connection in air line (beeline).
     * 
     * @param {draw2d.Connection} connection The Connection to route
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
     */
    route:function( connection, oldVertices)
    {
        
       // required for hit tests
       //
       var count = oldVertices.getSize();
       for(var i=0; i<count;i++){
           connection.addPoint(oldVertices.get(i));
       }

       var ps = connection.getVertices();
       
       // respect the calculated anchor position if the start/end port has set any Anchor impl.
       var startAnchor = connection.getStartPoint(ps.get(1));
       var endAnchor   = connection.getEndPoint(ps.get(ps.getSize()-2));
       ps.first().setPosition(startAnchor);
       ps.last().setPosition(endAnchor);
       
       this._paint(connection);
    },
    
    /**
     * @method 
     * Tweak or enrich the polyline persistence data with routing information
     * 
     * @since 2.10.0
     * @param {draw2d.shape.basic.PolyLine} line
     * @param {Object} memento The memento data of the polyline
     * @returns {Object}
     */
    getPersistentAttributes : function(line, memento)
    {   
        memento.vertex = [];
        
        line.getVertices().each(function(i,e){
            memento.vertex.push({x:e.x, y:e.y});
        });
        
        return memento;
    },
    
    /**
     * @method 
     * set the attributes for the polyline with routing information
     * 
     * @since 2.10.0
     * @param {Object} memento
     */
    setPersistentAttributes : function(line, memento)
    {
        // restore the points from the JSON data and add them to the polyline
        //
        if(typeof memento.vertex !=="undefined"){
            
            line.oldPoint=null;
            line.lineSegments = new draw2d.util.ArrayList();
            line.vertices     = new draw2d.util.ArrayList();

            $.each(memento.vertex, $.proxy(function(i,e){
                line.addPoint(e.x, e.y);
            },this));
        }
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.ManhattanConnectionRouter
 * Provides a {@link draw2d.Connection} with an orthogonal route between the Connection's source 
 * and target anchors.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     // Override the default connection type. This is used during drag&drop operations of ports.
 *     //
 *     draw2d.Connection.createConnection=function(sourcePort, targetPort){
 *        // return my special kind of connection
 *        var con = new draw2d.Connection();
 *        con.setRouter(new draw2d.layout.connection.ManhattanConnectionRouter());
 *        return con;
 *     };
 *     
 *     // create and add two nodes which contains Ports (In and OUT)
 *     //
 *     var start = new draw2d.shape.node.Start();
 *     var end   = new draw2d.shape.node.End();
        
 *     // ...add it to the canvas 
 *     canvas.addFigure( start, 50,50);
 *     canvas.addFigure( end, 230,80);
 *          
 *     // first Connection
 *     //
 *     var c = draw2d.Connection.createConnection();
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *     canvas.addFigure(c);
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ConnectionRouter
 */
draw2d.layout.connection.ManhattanConnectionRouter = draw2d.layout.connection.ConnectionRouter.extend({
    NAME : "draw2d.layout.connection.ManhattanConnectionRouter",

	MINDIST : 20,
	TOL     : 0.1,
	TOLxTOL : 0.01,
    TOGGLE_DIST : 5,
    
	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    init: function(){
        this._super();
    },
    
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    },
 
	/**
	 * @method
	 * Layout the hands over connection in a manhattan like layout
	 * 
	 * @param {draw2d.Connection} conn
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
	 */
	route:function( conn, oldVertices)
	{
	   var fromPt  = conn.getStartPoint();
	   var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

       var toPt    = conn.getEndPoint();
	   var toDir   = conn.getTarget().getConnectionDirection(conn, conn.getSource());
	
	   // calculate the lines between the two points.
	   //
	   this._route(conn,toPt, toDir, fromPt, fromDir);
	   this._paint(conn);
	},
	
	/**
	 * @method
	 * Internal routing algorithm.
	 * 
	 * @private
	 * @param {draw2d.Connection} conn
	 * @param {draw2d.geo.Point} fromPt
	 * @param {Number} fromDir
	 * @param {draw2d.geo.Point} toPt
	 * @param {Number} toDir
	 */
	_route:function( conn, fromPt, fromDir, toPt, toDir)
	{
	   // fromPt is an x,y to start from.  
	   // fromDir is an angle that the first link must 
	   //
	   var UP   = draw2d.geo.Rectangle.DIRECTION_UP;
	   var RIGHT= draw2d.geo.Rectangle.DIRECTION_RIGHT;
	   var DOWN = draw2d.geo.Rectangle.DIRECTION_DOWN;
	   var LEFT = draw2d.geo.Rectangle.DIRECTION_LEFT;
	
	   var xDiff = fromPt.x - toPt.x;
	   var yDiff = fromPt.y - toPt.y;
	   var point;
	   var dir;
	
	   if (((xDiff * xDiff) < (this.TOLxTOL)) && ((yDiff * yDiff) < (this.TOLxTOL))) 
	   {
	      conn.addPoint(new draw2d.geo.Point(toPt.x, toPt.y));
	      return;
	   }
	
	   if (fromDir === LEFT) 
	   {
	      if ((xDiff > 0) && ((yDiff * yDiff) < this.TOL) && (toDir === RIGHT))
	      {
	         point = toPt;
	         dir = toDir;
	      } 
	      else 
	      {
	         if (xDiff < 0) 
	         {
	            point = new draw2d.geo.Point(fromPt.x - this.MINDIST, fromPt.y);
	         }
	         else if (((yDiff > 0) && (toDir === DOWN)) || ((yDiff < 0) && (toDir === UP))) 
	         {
	            point = new draw2d.geo.Point(toPt.x, fromPt.y);
	         }
	         else if (fromDir == toDir)
	         {
	            var pos = Math.min(fromPt.x, toPt.x) - this.MINDIST;
	            point = new draw2d.geo.Point(pos, fromPt.y);
	         }
	         else
	         {
	            point = new draw2d.geo.Point(fromPt.x - (xDiff / 2), fromPt.y);
	         }
	
	         if (yDiff > 0) 
	         {
	            dir = UP;
	         }
	         else
	         {
	            dir = DOWN;
	         }
	      }
	   }
	   else if (fromDir === RIGHT) 
	   {
	      if ((xDiff < 0) && ((yDiff * yDiff) < this.TOL)&& (toDir === LEFT)) 
	      {
	         point = toPt;
	         dir = toDir;
	      } 
	      else 
	      {
	         if (xDiff > 0) 
	         {
	           point = new draw2d.geo.Point(fromPt.x + this.MINDIST, fromPt.y);
	         } 
	         else if (((yDiff > 0) && (toDir === DOWN)) || ((yDiff < 0) && (toDir === UP))) 
	         {
	            point = new draw2d.geo.Point(toPt.x, fromPt.y);
	         } 
	         else if (fromDir === toDir) 
	         {
	            var pos = Math.max(fromPt.x, toPt.x) + this.MINDIST;
	            point = new draw2d.geo.Point(pos, fromPt.y);
	         } 
	         else 
	         {
	               point = new draw2d.geo.Point(fromPt.x - (xDiff / 2), fromPt.y);
	         }
	
	         if (yDiff > 0)
	         {
	            dir = UP;
	         }
	         else
	         {
	            dir = DOWN;
	         }
	      }
	   } 
	   else if (fromDir === DOWN) 
	   {
	      if (((xDiff * xDiff) < this.TOL) && (yDiff < 0)&& (toDir === UP)) 
	      {
	         point = toPt;
	         dir = toDir;
	      } 
	      else 
	      {
	         if (yDiff > 0) 
	         {
	            point = new draw2d.geo.Point(fromPt.x, fromPt.y + this.MINDIST);
	         } 
	         else if (((xDiff > 0) && (toDir === RIGHT)) || ((xDiff < 0) && (toDir === LEFT))) 
	         {
	           point = new draw2d.geo.Point(fromPt.x, toPt.y);
	         } 
	         else if (fromDir === toDir) 
	         {
	            var pos = Math.max(fromPt.y, toPt.y) + this.MINDIST;
	            point = new draw2d.geo.Point(fromPt.x, pos);
	         } 
	         else 
	         {
	            point = new draw2d.geo.Point(fromPt.x, fromPt.y - (yDiff / 2));
	         }
	
	         if (xDiff > 0) 
	         {
	            dir = LEFT;
	         }
	         else 
	         {
	            dir = RIGHT;
	         }
	      }
	   } 
	   else if (fromDir === UP) 
	   {
	      if (((xDiff * xDiff) < this.TOL) && (yDiff > 0) && (toDir === DOWN))
	      {
	         point = toPt;
	         dir = toDir;
	      } 
	      else 
	      {
	         if (yDiff < 0) 
	         {
	            point = new draw2d.geo.Point(fromPt.x, fromPt.y - this.MINDIST);
	         } 
	         else if (((xDiff > 0) && (toDir === RIGHT)) || ((xDiff < 0) && (toDir === LEFT))) 
	         {
	            point = new draw2d.geo.Point(fromPt.x, toPt.y);
	         } 
	         else if (fromDir === toDir) 
	         {
	            var pos = Math.min(fromPt.y, toPt.y) - this.MINDIST;
	            point = new draw2d.geo.Point(fromPt.x, pos);
	         } 
	         else 
	         {
	            point = new draw2d.geo.Point(fromPt.x, fromPt.y - (yDiff / 2));
	         }
	
	         if (xDiff > 0)
	         {
	            dir = LEFT;
	         }
	         else
	         {
	            dir = RIGHT;
	         }
	      }
	   }
	   this._route(conn,point, dir, toPt, toDir);
	   conn.addPoint(fromPt);
	}

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.ManhattanBridgedConnectionRouter
 * Provides a {@link draw2d.Connection} with an orthogonal route between the Connection's source 
 * and target anchors.
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ManhattanConnectionRouter
 */
draw2d.layout.connection.ManhattanBridgedConnectionRouter = draw2d.layout.connection.ManhattanConnectionRouter.extend({
    NAME : "draw2d.layout.connection.ManhattanBridgedConnectionRouter",

	BRIDGE_HORIZONTAL_LR : " r 0 0 3 -4 7 -4 10 0 13 0 ", // Left to right
    BRIDGE_HORIZONTAL_RL : " r 0 0 -3 -4 -7 -4 -10 0 -13 0 ", // right to left
 
	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    init: function(){
        this._super();
    },
    
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    },
 
	/**
	 * @method
	 * Layout the hands over connection in a manhattan like layout
	 * 
	 * @param {draw2d.Connection} conn the connection to layout
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
	 */
	route : function(conn, oldVertices) {
		var fromPt  = conn.getStartPoint();
		var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

		var toPt  = conn.getEndPoint();
		var toDir = conn.getTarget().getConnectionDirection(conn, conn.getSource());

		// calculate the lines between the two points.
		//
		this._route(conn, toPt, toDir, fromPt, fromDir);

		// calculate the path string for the SVG rendering
		//
        var intersectionsASC = conn.getCanvas().getIntersection(conn).sort("x");
        var intersectionsDESC= intersectionsASC.clone().reverse();
        
        var intersectionForCalc = intersectionsASC;
		var i = 0;

		// ATTENTION: we cast all x/y coordinates to int and add 0.5 to avoid subpixel rendering of
		//            the connection. The 1px or 2px lines look much clearer than before.
		//
		var ps = conn.getVertices();
		var p = ps.get(0);
		var path = [ "M", (p.x|0)+0.5, " ", (p.y|0)+0.5 ];
		var oldP = p;
		for (i = 1; i < ps.getSize(); i++) {
			p = ps.get(i);
       
			// check for intersection and paint a bridge if required
			// line goes from left to right
			//
			var bridgeWidth = 5;
			var bridgeCode = this.BRIDGE_HORIZONTAL_LR;

			// line goes from right->left. Inverse the bridge and the bridgeWidth
			//
			if (oldP.x > p.x) {
				intersectionForCalc=intersectionsDESC;
				bridgeCode = this.BRIDGE_HORIZONTAL_RL;
				bridgeWidth = -bridgeWidth;
			}

			intersectionForCalc.each(function(ii, interP) {
				if (interP.justTouching ==false && draw2d.shape.basic.Line.hit(1, oldP.x, oldP.y, p.x, p.y, interP.x, interP.y) === true) {
					// we draw only horizontal bridges. Just a design decision
					//
					if (p.y === interP.y) {
						path.push(" L", ((interP.x - bridgeWidth)|0)+0.5, " ", (interP.y|0)+0.5);
						path.push(bridgeCode);
					}
				}

			});

			path.push(" L", (p.x|0)+0.5, " ", (p.y|0)+0.5);
			oldP = p;
		}
		conn.svgPathString = path.join("");
	}

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.InteractiveManhattanConnectionRouter
 * 
 * 
 * @author Andreas Herz
 * @since 4.0.2
 * @extends  draw2d.layout.connection.ManhattanConnectionRouter
 */
draw2d.layout.connection.InteractiveManhattanConnectionRouter = draw2d.layout.connection.ManhattanConnectionRouter.extend({
    NAME : "draw2d.layout.connection.InteractiveManhattanConnectionRouter",


	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    init: function(){
        this._super();

    },
    
    onInstall: function(conn)
    {
        conn.installEditPolicy(new draw2d.policy.line.OrthogonalSelectionFeedbackPolicy());
        if(!conn._routingMetaData){
            conn._routingMetaData = {
                    routedByUserInteraction:false,
                    fromDir:-1,
                    toDir:-1
            };
        }
    },
    
    onUninstall: function(conn)
    {
        delete conn._routingMetaData;
    },
 
    /**
	 * @method
	 * Layout the hands over connection in a manhattan like layout
	 * 
	 * @param {draw2d.Connection} conn the connection to layout
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
	 */
	route : function(conn, oldVertices) {
	    if(oldVertices.getSize()===0 || conn._routingMetaData.routedByUserInteraction===false){
	        this._super(conn, oldVertices);
	        conn._routingMetaData.fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());
	        conn._routingMetaData.toDir   = conn.getTarget().getConnectionDirection(conn, conn.getSource());
	    }
	    else{
	        this.halfRoute(conn, oldVertices);
	        this._paint(conn);
	    }
 	},
	
    /**
     * @method
     * The routing algorithm if the user has changed at least on of the vertices manually.
     * This kind of routing just align the start and end vertices to the new source/target port
     * location.
     * The vertices between keep untouched. Modification of this vertices are done by the
     * draw2d.policy.line.OrthogonalSelectionFeedbackPolicy
     * 
     * @param {draw2d.Connection} conn the connection to route
     * @param {draw2d.util.ArrayList} oldVertices the vertices of the routing before
     */	
    halfRoute:function(conn, oldVertices){

       var vertexCount  = oldVertices.getSize();
       
       var fromPt  = conn.getStartPoint();
       var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

       var toPt    = conn.getEndPoint();
       var toDir   = conn.getTarget().getConnectionDirection(conn, conn.getSource());
       
       var max = Math.max;
       var min = Math.min;

       // the port orientation has been changed. This can happen if the node rotates. In this case
       // we must recalculate the routing.
       if(conn._routingMetaData.fromDir !== fromDir || conn._routingMetaData.toDir !== toDir){
           conn._routingMetaData.routedByUserInteraction = false;
           this.route(conn, oldVertices);
       }
       
	   //  go back to the default if no routing is possible anymore
	   //
	   if((fromDir===1 ) && (toDir === 3) && (fromPt.x > toPt.x) && (vertexCount<=4)){
	       conn._routingMetaData.routedByUserInteraction = false;
	       this.route(conn, oldVertices);
	   }

       // transfer the old vertices into the connection
       //
       oldVertices.each(function(i,vertex){
           conn.addPoint(vertex);
       });

	    
	    // The SOURCE port (labeled with p0) has been moved/changed.
	    //
	    if(!fromPt.equals(oldVertices.get(0))){
	        var p1 = oldVertices.get(1);
	        var p2 = oldVertices.get(2);
	        conn.setVertex(0,fromPt);
	        switch(fromDir){
    	       //          .
    	       //   p0     . p1
    	       //   +------+
    	       //          .
    	       //          .
    	       //
	           case draw2d.geo.Rectangle.DIRECTION_RIGHT:
    	         conn.setVertex(1,max(fromPt.x+10,p1.x),fromPt.y);// p1
    	         conn.setVertex(2,max(fromPt.x+10,p1.x),p2.y);    // p2
    	         break;
    	       //   .       
    	       //   . p1     p0
    	       //   +------+
    	       //   .       
    	       //   .       
    	       //
	           case draw2d.geo.Rectangle.DIRECTION_LEFT:
    	         conn.setVertex(1,min(fromPt.x-10,p1.x),fromPt.y);// p1
    	         conn.setVertex(2,min(fromPt.x-10,p1.x),p2.y);    // p2
    	         break;
    	       //     ...+....
    	       //     p1 |      
    	       //        |  
    	       //        |  
    	       //     p0 +  
    	       //
	           case draw2d.geo.Rectangle.DIRECTION_UP:
	             conn.setVertex(1,fromPt.x, min(fromPt.y-10,p1.y)); // p1
                 conn.setVertex(2,p2.x, min(fromPt.y-10,p1.y)); // p2
                 break;
               //        +
               //     p0 |      
               //        |  
               //     p1 |  
               //    ....+....  
               //
	           case draw2d.geo.Rectangle.DIRECTION_DOWN:
                 conn.setVertex(1,fromPt.x, max(fromPt.y+10,p1.y)); // p1
                 conn.setVertex(2,p2.x, max(fromPt.y+10,p1.y));     // p2
    	         break;
	       }
	    }
        //////////////////////////////////////////////////////////////////
	    // the TARGET port ( labeled with p0) has been moved
	    //
	    if(!toPt.equals(oldVertices.get(vertexCount-1))){
            var p1 = oldVertices.get(vertexCount-2);
            var p2 = oldVertices.get(vertexCount-3);
            conn.setVertex(vertexCount-1,toPt);                        // p0
	        
    	      switch(toDir){
    	      //               .
    	      //      p0       . p1
    	      //    +----------+ 
    	      //               .
    	      //               .
    	      case draw2d.geo.Rectangle.DIRECTION_RIGHT:
    	         conn.setVertex(vertexCount-2,max(toPt.x+10,p1.x),toPt.y);  // p1
    	         conn.setVertex(vertexCount-3,max(toPt.x+10,p1.x),p2.y);    // p2
    	         break;
    	         
    	      //    .
    	      //    .
    	      //    . p1         p0
    	      //    +----------+ 
    	      //    .
    	      //    .
    	      //
    	      case draw2d.geo.Rectangle.DIRECTION_LEFT:
    	         conn.setVertex(vertexCount-2,min(toPt.x-10,p1.x),toPt.y);  // p1
    	         conn.setVertex(vertexCount-3,min(toPt.x-10,p1.x),p2.y);    // p2
    	         break;
    	         
    	      //     ...+....
    	      //     p1 |      
    	      //        |  
    	      //        |  
    	      //     p0 +  
    	      //
    	      case draw2d.geo.Rectangle.DIRECTION_UP:
    	         conn.setVertex(vertexCount-2, toPt.x,max(toPt.y+10,p1.y));  // p1
    	         conn.setVertex(vertexCount-3, p2.x  ,max(toPt.y+10,p1.y));  // p2
    	         break;
    	         
    	      //        +    
    	      //     p0 |      
    	      //        |  
    	      //     p1 |  
    	      //     ...+...
    	      //
    	      case draw2d.geo.Rectangle.DIRECTION_DOWN:
    	         conn.setVertex(vertexCount-2,toPt.x,max(toPt.y+10,p1.y));  // p1
    	         conn.setVertex(vertexCount-3,p2.x  ,max(toPt.y+10,p1.y));  // p2
    	         break;
    	      }
	   }
	},
	
    /**
     * @method 
     * Tweak or enrich the polyline persistence data with routing information
     * 
     * @since 2.10.0
     * @param {draw2d.shape.basic.PolyLine} line
     * @param {Object} memento The memento data of the polyline
     * @returns {Object}
     */
    getPersistentAttributes : function(line, memento)
    {   
        memento.vertex = [];
        
        line.getVertices().each(function(i,e){
            memento.vertex.push({x:e.x, y:e.y});
        });
        memento.routingMetaData = $.extend({},line._routingMetaData);
        
        return memento;
    },
    
    /**
     * @method 
     * set the attributes for the polyline with routing information of the intractive mannhattan router.
     * 
     * @since 4..0.0
     * @param {Object} memento
     */
    setPersistentAttributes : function(line, memento)
    {
        // restore the points from the JSON data and add them to the polyline
        //
        if(typeof memento.vertex !=="undefined"){
            
            line.oldPoint=null;
            line.lineSegments = new draw2d.util.ArrayList();
            line.vertices     = new draw2d.util.ArrayList();

            $.each(memento.vertex, $.proxy(function(i,e){
                line.addPoint(e.x, e.y);
            },this));
        }
        
        if(typeof memento.routingMetaData !== "undefinied"){
            line._routingMetaData = $.extend({},memento.routingMetaData);
        }
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.CircuitConnectionRouter
 * Provides a {@link draw2d.Connection} with an orthogonal route between the Connection's source 
 * and target anchors.
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ManhattanConnectionRouter
 */
draw2d.layout.connection.CircuitConnectionRouter = draw2d.layout.connection.ManhattanConnectionRouter.extend({
    NAME : "draw2d.layout.connection.CircuitConnectionRouter",
 
	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    init: function(){
        this._super();

        this.setBridgeRadius(4);
        this.setVertexRadius(2);
        
        // experimental
        this.abortRoutingOnFirstVertexNode=false;
    },
    
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    },
    
    /**
     * @method
     * Callback method if the router has been removed from the connection. In the case of the CircuitRouter
     * all vertex nodes will be removed from the canvas.
     * 
     * @param {draw2d.Connection} connection The related connection
     * @template
     * @since 2.7.2
     */
    onUninstall: function(connection){
        if(typeof connection.vertexNodes!=="undefined" && connection.vertexNodes!==null){
            connection.vertexNodes.remove();
            connection.vertexNodes = null;
        }
    },
    
    /**
     * @method
     * Set the radius of the vertex circle.
     * 
     * @param {Number} radius
     * @deprecated
     */
    setVertexRadius: function(radius){
        this.vertexRadius=radius;
    },
    /** deprecated 
     * @private
     * **/
    setJunctionRadius: function(radius){ this.vertexRadius=radius;},
    
    /**
     * @method
     * Set the radius or span of the bridge. A bridge will be drawn if two connections are crossing and didn't have any
     * common port.
     * 
     * @param {Number} radius
     */
    setBridgeRadius: function(radius){
        this.bridgeRadius=radius;
        this.bridge_LR = [" r", 0.5, -0.5, radius-(radius/2), -(radius-radius/4), radius, -radius,radius+(radius/2), -(radius-radius/4), radius*2, "0 "].join(" ");
        this.bridge_RL = [" r", -0.5, -0.5, -(radius-(radius/2)), -(radius-radius/4), -radius, -radius,-(radius+(radius/2)), -(radius-radius/4), -radius*2, "0 "].join(" ");
    },
    
	/**
	 * @method
	 * Layout the hands over connection in a manhattan like layout
	 * 
	 * @param {draw2d.Connection} conn the connection to layout
     * @param {draw2d.util.ArrayList} oldVertePoints old/existing vertex of the Connection
	 */
	route : function(conn, oldVertexPoints) {
		var fromPt  = conn.getStartPoint();
		var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

		var toPt  = conn.getEndPoint();
		var toDir = conn.getTarget().getConnectionDirection(conn, conn.getSource());

		// calculate the lines between the two points with the standard ManhattanRouter.
		//
		this._route(conn, toPt, toDir, fromPt, fromDir);

        // get the intersections to the other connections
        //
        var intersectionsASC = conn.getCanvas().getIntersection(conn).sort("x");
        var intersectionsDESC= intersectionsASC.clone().reverse();
        
        var intersectionForCalc = intersectionsASC;
        var i = 0;

        // add a ArrayList of all added vertex nodes to the connection
        //
        if(typeof conn.vertexNodes!=="undefined" && conn.vertexNodes!==null){
            conn.vertexNodes.remove();
        }
        conn.vertexNodes = conn.canvas.paper.set();

        // ATTENTION: we cast all x/y coordinates to integer and add 0.5 to avoid subpixel rendering of
		//            the connection. The 1px or 2px lines look much clearer than before.
		//
		var ps = conn.getVertices();
		var p = ps.get(0);
        var path = [ "M", (p.x|0)+0.5, " ", (p.y|0)+0.5 ];

        var oldP = p;
        var bridgeWidth = null;
        var bridgeCode  = null;
        
        var lastVertexNode=null;
		
        for (i = 1; i < ps.getSize(); i++) {
			p = ps.get(i);

			// line goes from right->left.
            if (oldP.x > p.x) {
                intersectionForCalc=intersectionsDESC;
                bridgeCode = this.bridge_RL;
                bridgeWidth = -this.bridgeRadius;
            }
            // line goes from left->right
            else{
                intersectionForCalc=intersectionsASC;
                bridgeCode = this.bridge_LR;
                bridgeWidth = this.bridgeRadius;
            }
            
            // add a bridge or a vertex node depending to the intersection connection
            //
            // bridge   => the connections didn't have a common port
            // vertex => the connections did have a common source or target port
            //
            intersectionForCalc.each($.proxy(function(ii, interP) {
                if (draw2d.shape.basic.Line.hit(1, oldP.x, oldP.y, p.x, p.y, interP.x, interP.y) === true) {
                    
                    // It is a vertex node..
                    //
    			    if(conn.sharingPorts(interP.other)){
    			        var other = interP.other;
                        var otherZ = other.getZOrder();
                        var connZ = conn.getZOrder();
                        if(connZ<otherZ){
                            var vertexNode=conn.canvas.paper.ellipse(interP.x,interP.y, this.vertexRadius, this.vertexRadius).attr({fill:conn.lineColor.hash()});
        			        conn.vertexNodes.push(vertexNode);
        				    // we found a vertex node. In this case an already existing connection did draw the connection.
        				    //
        			        if(this.abortRoutingOnFirstVertexNode===true){
            				    if(conn.getSource()==other.getSource()|| conn.getSource()==other.getTarget()){
            				        path = [ "M", (interP.x|0)+0.5, " ", (interP.y|0)+0.5 ];
            				        if(lastVerteNode!==null){
                                        lastVerteNode.remove();
            				            conn.vertexNodes.exclude(lastVerteNode);
            				        }
            				    }
                                lastVerteNode = vertexNode;
        			        }
                        }
    			    }
                    // ..or a bridge. We draw only horizontal bridges. Just a design decision
                    //
    			    else if (p.y === interP.y) {
                        path.push(" L", ((interP.x - bridgeWidth)|0)+0.5, " ", (interP.y|0)+0.5);
                        path.push(bridgeCode);
    			    }
                }
			},this));

			path.push(" L", (p.x|0)+0.5, " ", (p.y|0)+0.5);
			oldP = p;
		}
		conn.svgPathString = path.join("");
	}
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.connection.SplineConnectionRouter 
 * 
 * A MannhattanConnectionRouter with an spline interpolation between the bend points.
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.layout.connection.ManhattanConnectionRouter
 */
draw2d.layout.connection.SplineConnectionRouter = draw2d.layout.connection.ManhattanConnectionRouter.extend({

	NAME : "draw2d.layout.connection.SplineConnectionRouter",

    /**
     * @constructor Creates a new Router object
     */
    init : function()
    {
        this._super();

//        this.spline = new draw2d.util.spline.CatmullRomSpline();
        this.spline = new draw2d.util.spline.CubicSpline();
//        this.spline = new draw2d.util.spline.BezierSpline();
        
        this.MINDIST = 50,
        this.cheapRouter = null;
    },

    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
    },
 
	/**
	 * @method
	 * Layout the hands over connection with the cubic spline calculation and manhattan routing
	 * 
	 * @param {draw2d.Connection} conn
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
	 */
    route : function(conn, oldVertices)
    {
    	var i=0;
		var fromPt  = conn.getStartPoint();
		var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

		var toPt  = conn.getEndPoint();
		var toDir = conn.getTarget().getConnectionDirection(conn, conn.getSource());

		// calculate the manhatten bend points between start/end.
		//
		this._route(conn, toPt, toDir, fromPt, fromDir);

        var ps = conn.getVertices();

        conn.oldPoint=null;
        conn.lineSegments = new draw2d.util.ArrayList();
        conn.vertices     = new draw2d.util.ArrayList();
 
        var splinePoints = this.spline.generate(ps,8);
        splinePoints.each(function(i,e){
            conn.addPoint(e);
        });
        
        // calculate the path string for the SVG rendering
        //
        var ps = conn.getVertices();
        length = ps.getSize();
        var p = ps.get(0);
        var path = ["M",p.x," ",p.y];
        for( i=1;i<length;i++){
              p = ps.get(i);
              path.push("L", p.x, " ", p.y);
        }
        conn.svgPathString = path.join("");
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.connection.FanConnectionRouter
 * 
 * Automatic router that spreads its  {@link draw2d.Connection Connections} in a fan-like fashion upon collision. 
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     // Override the default connection type. This is used during drag&drop operations of ports.
 *     //
 *     draw2d.Connection.createConnection=function(sourcePort, targetPort){
 *        // return my special kind of connection
 *        var con = new draw2d.Connection();
 *        con.setRouter(new draw2d.layout.connection.FanConnectionRouter());
 *        return con;
 *     };
 *     
 *     // create and add two nodes which contains Ports (In and OUT)
 *     //
 *     var start = new draw2d.shape.node.Start();
 *     var end   = new draw2d.shape.node.End();
        
 *     // ...add it to the canvas 
 *     canvas.addFigure( start, 50,50);
 *     canvas.addFigure( end, 230,80);
 *          
 *     // first Connection
 *     //
 *     var c = draw2d.Connection.createConnection();
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *     canvas.addFigure(c);
 *     
 *     // second Connection
 *     //
 *     c = draw2d.Connection.createConnection();
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *     canvas.addFigure(c);
 *     
 *     // third Connection
 *     //
 *     c = draw2d.Connection.createConnection();
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *     canvas.addFigure(c);
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends draw2d.layout.connection.DirectRouter
 */
draw2d.layout.connection.FanConnectionRouter = draw2d.layout.connection.DirectRouter.extend({
    NAME : "draw2d.layout.connection.FanConnectionRouter",

    /**
     * @constructor Creates a new Router object.
     */
    init : function()
    {
        this._super();
        
    },

    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    },
 
    /**
     * @method 
     * Layout the hands over connection in a manhattan like layout
     * 
     * @param {draw2d.Connection}  conn
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
     */
    route : function(conn, oldVertices)
    {
        var lines = conn.getSource().getConnections();
        var connections = new draw2d.util.ArrayList();
        var index = 0;
        for ( var i = 0; i < lines.getSize(); i++) 
        {
            var figure = lines.get(i);
            if (figure.getTarget() === conn.getTarget() || figure.getSource() === conn.getTarget()) 
            {
                connections.add(figure);
                if (conn === figure){
                    index = connections.getSize();
                }
            }
        }
        if (connections.getSize() > 1){
            this.routeCollision(conn, index);
        }
        else{
            this._super(conn);
        }
    },

    routeCollision : function(/* :draw2d.Connection */conn, /* :int */index)
    {
        var start = conn.getStartPoint();
        var end = conn.getEndPoint();

        var separation = 15;

        var midPoint = new draw2d.geo.Point((end.x + start.x) / 2, (end.y + start.y) / 2);
        var position = end.getPosition(start);
        var ray;
        if (position == draw2d.geo.PositionConstants.SOUTH || position == draw2d.geo.PositionConstants.EAST){
            ray = new draw2d.geo.Point(end.x - start.x, end.y - start.y);
        }
        else{
            ray = new draw2d.geo.Point(start.x - end.x, start.y - end.y);
        }

        var length = Math.sqrt(ray.x * ray.x + ray.y * ray.y);

        var xSeparation = separation * ray.x / length;
        var ySeparation = separation * ray.y / length;

        var bendPoint;

        if (index % 2 === 0){
            bendPoint = new draw2d.geo.Point(midPoint.x + (index / 2) * (-1 * ySeparation), midPoint.y + (index / 2) * xSeparation);
        }
        else{
            bendPoint = new draw2d.geo.Point(midPoint.x + (index / 2) * ySeparation, midPoint.y + (index / 2) * (-1 * xSeparation));
        }

        // required for hit tests
        conn.addPoint(start);
        conn.addPoint(bendPoint);
        conn.addPoint(end);

        // calculate the path string for the SVG rendering
        //
        var ps = conn.getVertices();
        var p = ps.get(0);
        var path = ["M",p.x," ",p.y];
        for( var i=1;i<ps.getSize();i++){
              p = ps.get(i);
              path.push("L", p.x, " ", p.y);
        }
        conn.svgPathString = path.join("");
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.connection.MazeConnectionRouter
 * <b>BETA VERSION. Not for production!!!<br></b>
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ConnectionRouter
 */
var ROUTER_RECTS = null;

draw2d.layout.connection.MazeConnectionRouter = draw2d.layout.connection.ConnectionRouter.extend({
    NAME : "draw2d.layout.connection.MazeConnectionRouter",

    
	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    init: function(){
        this._super();

        this.useSpline = false;
    	this.useSimplify = true;
    	this.useSimplifyValue=2;
    	this.useDebug = false;
    	this.useShift = 4;
    	this.portOutletOffset = 15;
    	
   
//    	this.finder = new PF.AStarFinder();
//      this.finder = new PF.AStarFinder({ allowDiagonal: true, dontCrossCorners: true});
//      this.finder = new PF.AStarFinder({ allowDiagonal: false});
//      this.finder = new PF.BiBreadthFirstFinder({ allowDiagonal: false});
//      this.finder = new PF.BreadthFirstFinder({ allowDiagonal: false});
        this.finder = new PF.JumpPointFinder({allowDiagonal: false, dontCrossCorners: true});
    },
    
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    },
 
	/**
	 * @method
	 * Layout the hands over connection in a manhattan like layout
	 * 
	 * @param {draw2d.Connection} conn
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
	 */
	route:function( conn, oldVertices)
	{
	   var fromPt  = conn.getStartPoint();
	   var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

       var toPt    = conn.getEndPoint();
	   var toDir   = conn.getTarget().getConnectionDirection(conn, conn.getSource());
	
	   // calculate the lines between the two points.
	   //
	   this._route(conn,toPt, toDir, fromPt, fromDir);
       this._paint(conn);
	},
	
	/**
	 * @method
	 * Internal routing algorithm.
	 *      * <p>
     * Possible values:
     * <ul>
     *   <li>up -&gt; 0</li>
     *   <li>right -&gt; 1</li>
     *   <li>down -&gt; 2</li>
     *   <li>left -&gt; 3</li>
     * </ul>
     * <p>

	 * @private
	 * @param {draw2d.Connection} conn
	 * @param {draw2d.geo.Point} fromPt
	 * @param {Number} fromDir
	 * @param {draw2d.geo.Point} toPt
	 * @param {Number} toDir
	 */
	_route:function( conn, fromPt, fromDir, toPt, toDir)
	{
        var shift     = this.useShift ; 
        
        oldToPt   = toPt;
        oldFromPt = fromPt;
	    
        // move the points with an offset in the prefered routing direction of the ports
        // to avoid that the routed connection is sticking on one side of the figure.
        //
	    fromPt = this.getAddjustedPoint(fromPt, fromDir, this.portOutletOffset);
	    toPt   = this.getAddjustedPoint(toPt,   toDir,   this.portOutletOffset);
	    
	    var grid = this.generateNoGoGrid(conn, fromPt, fromDir, toPt, toDir);
 
        // 4. Calculate the shortest path from source to target based on the grid
        //
        var path = this.finder.findPath(
        		                   fromPt.x>>shift, fromPt.y>>shift, 
                                   toPt.x>>shift,   toPt.y>>shift, 
                                   grid);
        
        // transfer the path from the grid based coordinates back to the real coordinates
        //
        $.each(path,function(i,e){
            e.x=e[0]=e[0]<<shift;
            e.y=e[1]=e[1]<<shift;
        });
     
        // 5. paint the "no go" area in read if we are in debug mode
        //
        if(this.useDebug) {
            if(ROUTER_RECTS!==null){
                ROUTER_RECTS.remove();
            }
            ROUTER_RECTS = conn.canvas.paper.set();

            for(var i=0;i<grid.width;i++ ){
                for(var j=0;j<grid.height;j++ ){
                    if(!grid.isWalkableAt(i,j))
                    ROUTER_RECTS.push( conn.canvas.paper.rect(i<<shift,j<<shift,1<<shift,1<<shift).attr({"fill":"red","opacity": "0.1"}));
                }
            }
            ROUTER_RECTS.push( conn.canvas.paper.rect(fromPt.x-3,fromPt.y-3,6,6).attr({"fill":"#ff0000","opacity": "0.8"}));
            ROUTER_RECTS.push( conn.canvas.paper.rect(toPt.x-3,toPt.y-3,6,6).attr({"fill":"#ff0000","opacity": "0.8"}));

            // paint the original calculated path without any simplification in BLUE
            $.each(path,function(i,e){
                ROUTER_RECTS.push( conn.canvas.paper.rect(e.x-3,e.y-3,6,6).attr({"fill":"#0000ff","opacity": "0.8"}));
            });
            var p= path[0];
            var svgPathBefore = ["M",p.x," ",p.y];
            for( var i=1;i<path.length;i++){
                  p = path[i];
                  svgPathBefore.push("L", p.x, " ", p.y);
            }
            svgPathBefore = svgPathBefore.join("");
            ROUTER_RECTS.push(conn.canvas.paper.path(svgPathBefore).attr({"stroke":"#0000ff"}));
        }

        
        this.adjustPath(fromPt, path, fromDir);
        path.reverse();
        this.adjustPath(toPt, path, toDir);
        path.reverse();
        
        $.each(path,function(i,e){
            e.x=e[0];
            e.y=e[1];
        });
        

        if(this.useSpline){
	        var p  =new draw2d.util.ArrayList();
	        p.add(oldFromPt);
	        $.each(path,function(i,e){
	            p.add(new draw2d.geo.Point(e[0], e[1]));
	        });
	        p.add(oldToPt);
	        
            if(this.useDebug){
                $.each(path,function(i,e){
                    ROUTER_RECTS.push( conn.canvas.paper.rect(e.x-3,e.y-3,6,6).attr({"fill":"#00ff00","opacity": "0.8"}));
                });
                var pt= path[0];
                var svgPathBefore = ["M",pt.x," ",pt.y];
                for( var i=1;i<path.length;i++){
                      pt = path[i];
                      svgPathBefore.push("L", pt.x, " ", pt.y);
                }
                svgPathBefore = svgPathBefore.join("");
                ROUTER_RECTS.push(conn.canvas.paper.path(svgPathBefore).attr({"stroke":"#00ff00"}));
            }

            this.spline = new draw2d.util.spline.CubicSpline();
	        var splinePoints = this.spline.generate(p,8);

	        if(this.useSimplify){
		        path=[];
		        splinePoints.each(function(i,e){
		        	path.push({x:e.x,y:e.y});
		        });
		        path = this.simplify(path,this.useSimplifyValue ,true);
		        
		        $.each(path,function(i,e){
		            conn.addPoint(e.x,e.y);
		        });
	        }
	        else{
	        	splinePoints.each(function(i,e){
		            conn.addPoint(e);
		        });
	        }
	    }
        else{
        	if(this.useSimplify){
        		path = this.simplify(path,this.useSimplifyValue ,true);
        	}
            
        	if(this.useDebug){
                $.each(path,function(i,e){
                    ROUTER_RECTS.push( conn.canvas.paper.rect(e.x-3,e.y-3,6,6).attr({"fill":"#00ff00","opacity": "0.8"}));
                });
                var p= path[0];
                var svgPathBefore = ["M",p.x," ",p.y];
                for( var i=1;i<path.length;i++){
                      p = path[i];
                      svgPathBefore.push("L", p.x, " ", p.y);
                }
                svgPathBefore = svgPathBefore.join("");
                ROUTER_RECTS.push(conn.canvas.paper.path(svgPathBefore).attr({"stroke":"#00ff00"}));
            }

            conn.addPoint(oldFromPt);
	        $.each(path,function(i,e){
	            conn.addPoint(e[0], e[1]);
	        });
	        conn.addPoint(oldToPt);

        }

	},
	
	/**
	 * @method
	 * Generate a grid base no go map required for the path finding algorithm
	 * 
	 * @param conn
	 * @returns {PF.Grid}
	 */
	generateNoGoGrid: function(conn, fromPt, fromDir, toPt, toDir){
        var shift     = this.useShift ; 
        var oneShift2 = (1<<shift)/2;

	    // 1. generate a map with all "no go" areas. The bounding box of the shapes defines
        //    the no go areas.
        //
        var canvasWidth  = conn.getCanvas().paper.width>>shift;
        var canvasHeight = conn.getCanvas().paper.height>>shift;
        var grid = new PF.Grid(canvasWidth, canvasHeight); 
        var figures = conn.getCanvas().getFigures();
        figures.each(function(i,e){
            var box = e.getBoundingBox();
            // remove shapes which are hit by the input or output ports. It is not possible to route
            // out from a not walkable area
            if(box.hitTest(fromPt.x, fromPt.y)===true || box.hitTest(toPt.x, toPt.y)){
                return;
            }
            
            var x =box.x>>shift;
            var y =box.y>>shift;
            if(x<1 || y<1 ){
                return;
            }
            var r_orig = (box.x+box.w+oneShift2)>>shift;
            var b_orig = (box.y+box.h+oneShift2)>>shift;
            for(var i=x;i<=r_orig;i++ ){
                for(var j=y;j<=b_orig;j++ ){
                    grid.setWalkableAt(i, j, false);
                }
            }
        });
       

        // 3. make the are walkable on the edge of the port side. Otherwise we a 
        //    an enclosed area around the port if we are very close to another shape
        //
        var box = conn.getSource().getParent().getBoundingBox();
        if(toDir===1 || toDir===3){
            var y =box.y>>shift;
            if(y>0){
                var b_orig = box.y+box.h;
                var i = (toPt.x>>shift);
                
                for(var j=y-1;j<<shift<=b_orig;j++ ){
                    grid.setWalkableAt(i, j, true);
                }
            }
        }
        else{
            var x =box.x>>shift;
            if(x>0){
                var r_orig = box.x+box.w;
                var j = (toPt.x>>shift);
                for(var i=x-1;i<<shift<=r_orig;i++ ){
                    grid.setWalkableAt(i, j, true);
                }
            }
        }

        box = conn.getTarget().getParent().getBoundingBox();
        if(fromDir===1 || fromDir===3){
            var y =box.y>>shift;
            if(y>0){
                var b_orig = box.y+box.h;
                var i = (fromPt.x>>shift);
                for(var j=y-1;j<<shift<=b_orig;j++ ){
                    grid.setWalkableAt(i, j, true);
                }
            }
        }
        else{
            var x =box.x>>shift;
            if(x>0){
                var r_orig = box.x+box.w;
                var j = (fromPt.x>>shift);
                for(var i=x-1;i<<shift<=r_orig;i++ ){
                    grid.setWalkableAt(i, j, true);
                }
            }
        }
        
        return grid;
	},
	
	/**
	 * @method
	 * move the point in the given direction with the given offset
	 * 
	 * @param {draw2d.geo.Point} pt
	 * @param {Number} direction
	 * @param {Number} adjustment
	 * 
	 * @returns {draw2d.geo.Point}
	 */
	getAddjustedPoint: function(pt , direction, adjustment){

	    switch(direction){
	        case draw2d.geo.Rectangle.DIRECTION_UP:
	            return new draw2d.geo.Point(pt.x, pt.y-adjustment);
            case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                return new draw2d.geo.Point(pt.x+adjustment, pt.y);
            case draw2d.geo.Rectangle.DIRECTION_DOWN:
                return new draw2d.geo.Point(pt.x,pt.y+adjustment);
            case draw2d.geo.Rectangle.DIRECTION_LEFT:
                return new draw2d.geo.Point(pt.x-adjustment,pt.y);
	    }
	},
	
	adjustPath: function(pt , path, direction){
        var shift = this.useShift;
        var x = pt.x>>shift;
        var y = pt.y>>shift;
        $.each(path,function(i,e){
            if(y===(e[1]>>shift)){
                e[1]=pt.y;
            }
            else{
                return false;
            }
        });
        $.each(path,function(i,e){
            if(x===(e[0]>>shift)){
                e[0]=pt.x;
            }
            else{
                return false;
            }
        });
    },
    

    getSquareDistance: function(p1, p2) { // square distance between 2 points

    		var dx = p1.x - p2.x,
    		    dy = p1.y - p2.y;

    		return dx * dx +
    		       dy * dy;
    	},

   	getSquareSegmentDistance: function(p, p1, p2) { // square distance from a point to a segment

    		var x = p1.x,
    		    y = p1.y,

    		    dx = p2.x - x,
    		    dy = p2.y - y,

    		    t;

    		if (dx !== 0 || dy !== 0) {

    			t = ((p.x - x) * dx +
    			     (p.y - y) * dy) /
    			        (dx * dx +
    			         dy * dy);

    			if (t > 1) {
    				x = p2.x;
    				y = p2.y;

    			} else if (t > 0) {
    				x += dx * t;
    				y += dy * t;
    			}
    		}

    		dx = p.x - x;
    		dy = p.y - y;

    		return dx * dx +
    		       dy * dy;
    	},

    	simplifyRadialDistance: function(points, sqTolerance) { // distance-based simplification

    		var i,
    		    len = points.length,
    		    point =null,
    		    prevPoint = points[0],
    		    newPoints = [prevPoint];

    		for (i = 1; i < len; i++) {
    			point = points[i];

    			if (this.getSquareDistance(point, prevPoint) > sqTolerance) {
    				newPoints.push(point);
    				prevPoint = point;
    			}
    		}

    		if (prevPoint !== point) {
    			newPoints.push(point);
    		}

    		return newPoints;
    	},


    	// simplification using optimized Douglas-Peucker algorithm with recursion elimination

    	simplifyDouglasPeucker: function(points, sqTolerance) {

    		var len = points.length,

    		    MarkerArray = (typeof Uint8Array !== undefined + '')
    		                ? Uint8Array
    		                : Array,

    		    markers = new MarkerArray(len),

    		    first = 0,
    		    last  = len - 1,

    		    i,
    		    maxSqDist,
    		    sqDist,
    		    index,

    		    firstStack = [],
    		    lastStack  = [],

    		    newPoints  = [];

    		markers[first] = markers[last] = 1;

    		while (last) {

    			maxSqDist = 0;

    			for (i = first + 1; i < last; i++) {
    				sqDist = this.getSquareSegmentDistance(points[i], points[first], points[last]);

    				if (sqDist > maxSqDist) {
    					index = i;
    					maxSqDist = sqDist;
    				}
    			}

    			if (maxSqDist > sqTolerance) {
    				markers[index] = 1;

    				firstStack.push(first);
    				lastStack.push(index);

    				firstStack.push(index);
    				lastStack.push(last);
    			}

    			first = firstStack.pop();
    			last = lastStack.pop();
    		}

    		for (i = 0; i < len; i++) {
    			if (markers[i]) {
    				newPoints.push(points[i]);
    			}
    		}

    		return newPoints;
    	},



    	simplify : function (points, tolerance, highestQuality) {

    		var sqTolerance = (tolerance !== undefined)
    		                ? tolerance * tolerance
    		                : 1;

    		if (!highestQuality) {
    			points = this.simplifyRadialDistance(points, sqTolerance);
    		}
    		points = this.simplifyDouglasPeucker(points, sqTolerance);

    		return points;
    	}
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.MuteableManhattanConnectionRouter
 * 
 * JUST FOR RESEARCH AT THE MOMENT!!!!!!
 * 
 * @inheritable
 * @author Andreas Herz
 * 
 * @extends  draw2d.layout.connection.ManhattanConnectionRouter
 */

draw2d.layout.connection.MuteableManhattanConnectionRouter = draw2d.layout.connection.ManhattanConnectionRouter.extend({
    NAME : "draw2d.layout.connection.MuteableManhattanConnectionRouter",

	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    UP      : new draw2d.geo.Ray(0, -1),
    DOWN    : new draw2d.geo.Ray(0, 1),
    LEFT    : new draw2d.geo.Ray(-1, 0),
    RIGHT   : new draw2d.geo.Ray(1, 0),

    init: function(){
        this._super();
        
        this.rowsUsed     = {};//new HashMap<Integer, Integer>();
        this.colsUsed     = {};//new HashMap<Integer, Integer>();
        this.constraints  = {};//new HashMap<Connection, Object>();
        this.reservedInfo = {};//new HashMap<Connection, ReservedInfo>();
    },


    /**
     * @method
     * Layout the hands over connection in a manhattan like layout
     * 
     * @param {draw2d.Connection} conn
     * @param {draw2d.util.ArrayList} oldVertices old/existing vertices of the Connection
     */
    route:function( conn, oldVertices){
        this.rowsUsed     = {};//new HashMap<Integer, Integer>();
        this.colsUsed     = {};//new HashMap<Integer, Integer>();
        this.constraints  = {};//new HashMap<Connection, Object>();
        this.reservedInfo = {};//new HashMap<Connection, ReservedInfo>();

        var canvas = conn.getCanvas();
        var i;

        var startPoint= conn.getStartPoint();
        var endPoint= conn.getEndPoint();

        var start   = new draw2d.geo.Ray(startPoint);
        var end     = new draw2d.geo.Ray(endPoint);
        var average = new draw2d.geo.Ray((start.x+end.x)/2,(start.y+end.y)/2);

        var direction   = new draw2d.geo.Ray(end.x-start.x, end.y-start.y);
        var startNormal = this.getStartDirection(conn);
        var endNormal   = this.getEndDirection(conn);

        var positions  = new draw2d.util.ArrayList();
        var horizontal = startNormal.isHorizontal();

        if (horizontal){
            positions.add(start.y);
        }
        else{
            positions.add(start.x);
        }
        
        horizontal = !horizontal;

        // dot product is zero if the vector orthogonal (90�?)
        if (startNormal.dot(endNormal) === 0) {
            if ((startNormal.dot(direction) >= 0)  && (endNormal.dot(direction) <= 0)) {
                // 0
            } else {

                // 2
                if (startNormal.dot(direction) < 0)
                    i = startNormal.similarity(start.getTranslated(startNormal.getScaled(10)));
                else {
                    if (horizontal) 
                        i = average.y;
                    else 
                        i = average.x;
                }
                
                positions.add(i);
                horizontal = !horizontal;

                if (endNormal.dot(direction) > 0){
                    i = endNormal.similarity(end.getTranslated(endNormal.getScaled(10)));
                }
                else {
                    if (horizontal) {
                        i = average.y;
                    }
                    else {
                        i = average.x;
                    }
                }
                positions.add(i);
                horizontal = !horizontal;
            }
        } else {
            if (startNormal.dot(endNormal) > 0) {
                //1
                if (startNormal.dot(direction) >= 0)
                    i = startNormal.similarity(start.getTranslated(startNormal.getScaled(10)));
                else
                    i = endNormal.similarity(end.getTranslated(endNormal.getScaled(10)));
                positions.add( i);
                horizontal = !horizontal;
            } else {
                //3 or 1
                if (startNormal.dot(direction) < 0) {
                    i = startNormal.similarity(start.getTranslated(startNormal.getScaled(10)));
                    positions.add(i);
                    horizontal = !horizontal;
                }

                // my tweak to route SCA wires starts
                if (this.isCycle(conn)) {
                    if (horizontal)
                        i = conn.getSource().getParent().getBoundingBox().getTop() - 10;// * index;
                    else
                        i = conn.getSource().getParent().getBoundingBox().getRight() + 10;// * index;
                } else {
                    if (horizontal) {
                        var j = average.y;

                        var next = endNormal.similarity(end.getTranslated(endNormal.getScaled(10)));

                        var trial = new draw2d.geo.Ray((positions.get(positions.getSize() - 1)), j);
                        var figure = this.findFirstFigureAtStraightLine(canvas, trial, this.LEFT, draw2d.util.ArrayList.EMPTY_LIST);

                        while (figure != null && figure.getBoundingBox().x + figure.getBoundingBox().width > next) {
                            j = figure.getBoundingBox().y + figure.getBoundingBox().height + 5;
                            trial.y = j;
                            figure = this.findFirstFigureAtStraightLine(canvas, trial, this.LEFT, Collections.EMPTY_LIST);                        
                        }

                        i = j;

                    } else {
                        var figure = this.findFirstFigureAtStraightLine(canvas, start, this.RIGHT, this.getExcludingFigures(conn));
                        if (figure == null)
                            i = average.x;
                        else {
                            i = Math.min(average.x, start.getTranslated(new draw2d.geo.Ray(3 * (figure.getBoundingBox().x - start.x) / 4, 0)).x);
                            i = Math.max(start.x, i);
                        }
                        i = this.adjust(conn, i);
                    }
                }
                // my tweak to route SCA wires ends
                positions.add(i);
                horizontal = !horizontal;
                /*

                if (startNormal.dot(direction) < 0) {
                    i = endNormal.similarity(end.getTranslated(endNormal.getScaled(10)));
                    positions.add( i);
                    horizontal = !horizontal;
                } else {
                    // my tweak to route SCA wires starts               
                    var reroute = false;

                    var j = end.y;

                    var figure = this.findFirstFigureAtStraightLine(canvas, new draw2d.geo.Ray(i, j), this.RIGHT, this.getExcludingFigures(conn));
                    while (figure != null && figure.getBoundingBox().x < end.x) {
                        reroute = true;
                        if (direction.dot(this.DOWN) > 0) 
                            j = figure.getBoundingBox().y - 5;                       
                        else 
                            j = figure.getBoundingBox().y + figure.getBoundingBox().height + 5; 

                        figure = this.findFirstFigureAtStraightLine(canvas, new draw2d.geo.Ray(i, j), this.RIGHT, this.getExcludingFigures(conn));                                        
                    }
                    if (reroute) {
                        i = j;
                        positions.add(i);
                        horizontal = !horizontal;

                        i = endNormal.similarity(end.getTranslated(endNormal.getScaled(10)));
                        positions.add( i);
                        horizontal = !horizontal;                       

                    }
                    // my tweak to route SCA wires ends
                    
                }
                */
            }
        }
        if (horizontal) 
            positions.add(end.y);
        else 
            positions.add( end.x);

        this.processPositions(start, end, positions, startNormal.isHorizontal(), conn);

    
        this._paint(conn);
    },

    /**
     * @method
     * 
     * @param {draw2d.Connection} connection
     * @param {Number} r
     * @param {Number} n
     * @param {Number} x
     * 
     * @private
     */
    getColumnNear:function (connection, r, n, x) {
        var min = Math.min(n, x);
        var max = Math.max(n, x);
        
        if (min > r) {
            max = min;
            min = r - (min - r);
        }
        if (max < r) {
            min = max;
            max = r + (r - max);
        }
        
        var proximity = 0;
        var direction = -1;
        if (r % 6 != 0){
            r = r - ( r % 6);
        }
        
        var i;
        while (proximity < r) {
            i = parseInt(r + proximity * direction);
            if (! (i in this.colsUsed)) {
                this.colsUsed[i]= i;
                this.reserveColumn(connection, i);
                return i;
            }
            
            if (i <= min){
                return i + 6;
            }
            
            if (i >= max){
                return i - 6;
            }
            
            if (direction === 1){
                direction = -1;
            }
            else {
                direction = 1;
                proximity += 6;
            }
        }
        return r;
    },
    
    getRowNear: function(connection, r, n, x) {
        var min = Math.min(n, x);
        var max = Math.max(n, x);
        
        if (min > r) {
            max = min;
            min = r - (min - r);
        }
        if (max < r) {
            min = max;
            max = r + (r - max);
        }

        var proximity = 0;
        var direction = -1;
        if (r % 6 != 0){
            r = r - ( r % 6);
        }
        
        var i;
        while (proximity < r) {
            i = parseInt(r + proximity * direction);
            if (! (i in this.rowsUsed)) {
                this.rowsUsed[i]= i;
                this.reserveRow(connection, i);
                return i;
            }
            if (i <= min)
                return i + 6;
            if (i >= max)
                return i - 6;
            if (direction == 1)
                direction = -1;
            else {
                direction = 1;
                proximity += 6;
            }
        }
        return r;
    },
    
    /**
    *   <li>up -&gt; 0</li>
    *   <li>right -&gt; 1</li>
    *   <li>down -&gt; 2</li>
    *   <li>left -&gt; 3</li>
    **/
    getEndDirection:function( conn) {
        var p    = conn.getEndPoint();
        var rect= conn.getTarget().getParent().getBoundingBox();
        return this.getDirection(rect, p);
    },
    
    
    /**
    *   <li>up -&gt; 0</li>
    *   <li>right -&gt; 1</li>
    *   <li>down -&gt; 2</li>
    *   <li>left -&gt; 3</li>
    **/
    getStartDirection:function( conn) {
        var p    = conn.getStartPoint();
        var rect= conn.getSource().getParent().getBoundingBox();
        return this.getDirection(rect, p);
    },
    
    /**
     * Returns the direction the point <i>p</i> is in relation to the given rectangle.
     * Possible values are LEFT (-1,0), RIGHT (1,0), UP (0,-1) and DOWN (0,1).
     * 
     * @param r the rectangle
     * @param p the point
     * @return the direction from <i>r</i> to <i>p</i>
     */
    getDirection:function( r,  p) {
        var i=Math.abs(r.y - p.y);
        var distance = Math.abs(r.x - p.x);
        var direction = this.LEFT;

        if (i <= distance) {
            distance = i;
            direction = this.UP;
        }

        i = Math.abs(r.getBottom() - p.y);
        if (i <= distance) {
          distance = i;
          direction = this.DOWN;
        }

        i = Math.abs(r.getRight() - p.x);
        if (i < distance) {
            distance = i;
            direction = this.RIGHT;
        }

        return direction;
    },

    processPositions: function(/*Ray*/ start, /*Ray*/ end, /*List*/ positions, /*boolean*/ horizontal, /*Connection*/ conn) {
        this.removeReservedLines(conn);

        var pos =  [];
        if (horizontal)
            pos.push(start.x);
        else
            pos.oush(start.y);
        var i;
        for (i = 0; i < positions.getSize(); i++) {
            pos.push(positions.get(i));
        }
        
        if (horizontal == (positions.getSize() % 2 == 1)){
            pos.push(end.x);
        }
        else{
            pos.push(end.y);
        }

        conn.addPoint(new draw2d.geo.Point(start.x, start.y));
        var p;
        var current, prev, min, max;
        var adjust;
        for (i = 2; i < pos.length - 1; i++) {
            horizontal = !horizontal;
            prev = pos[i - 1];
            current = pos[i];

            adjust = (i !== pos.length - 2);
            if (horizontal) {
                if (adjust) {
                    min = pos[i - 2];
                    max = pos[i + 2];
                    pos[i] = current = this.getRowNear(conn, current, min, max);
                }
                p = new draw2d.geo.Point(prev, current);
            } else {
                if (adjust) {
                    min = pos[i - 2];
                    max = pos[i + 2];
                    pos[i] = current = this.getColumnNear(conn, current, min, max);
                }
                p = new draw2d.geo.Point(current, prev);
            }
            conn.addPoint(p);
        }
        conn.addPoint(new draw2d.geo.Point(end.x, end.y));
    },
    

   removeReservedLines: function( connection) {
        var rInfo = this.reservedInfo[connection];
        if ( typeof rInfo ==="undefined" || rInfo === null) 
            return;

        for (var i = 0; i < rInfo.reservedRows.getSize(); i++) {
            delete this.rowsUsed[rInfo.reservedRows.get(i)];
        }
        for (var i = 0; i < rInfo.reservedCols.getSize(); i++) {
            delete this.colsUsed[rInfo.reservedCols.get(i)];
        }
        delete this.reservedInfo[connection];
    },

    reserveColumn: function( connection,  column) {
        var info = this.reservedInfo[connection];
        if ( typeof info ==="undefined" || info === null) {
           info = {reservedCols: new draw2d.util.ArrayList(), reservedRows: new draw2d.util.ArrayList()};
           this.reservedInfo[connection] = info;
        }
        info.reservedCols.add(column);
    },

    reserveRow:function(connection, row) {
        var info = this.reservedInfo[connection];
        if ( typeof info ==="undefined" || info === null) {
            info = {reservedCols: new draw2d.util.ArrayList(), reservedRows: new draw2d.util.ArrayList()};
            this.reservedInfo[connection] = info;
        }
        info.reservedRows.add(row);
    },

    getConstraint:function( connection) {
        return this.constraints[connection];
    },

    setConstraint:function( connection,  constraint) {
        this.constraints[connection]= constraint;
    },

    isCycle:function( conn) {
        var source = conn.getSource().getParent();
        var target = conn.getTarget().getParent();

        return source.id===target.id;
    },

    getExcludingFigures:function( conn) {
        var excluding = new draw2d.util.ArrayList();

        excluding.add(conn.getSource().getParent());
        excluding.add(conn.getTarget().getParent());

        return excluding;
    },

    findFirstFigureAtStraightLine:function(canvas, /*Ray*/ start, /*Ray*/ direction, /*List*/ excluding) {
        var figure = null;

        var figures = canvas.getFigures();
        figures.each($.proxy(function(i,child) {
            try{
                if (!excluding.contains(child)) {
                    var rect = child.getBoundingBox();
                    if (this.LEFT.equals(direction)) {
                        if (start.x > rect.x && start.y >= rect.y && start.y <= rect.y + rect.h) {                 
                            if (figure === null || rect.x > figure.getBoundingBox().x)
                                figure = child;
                        }
                    } else if (this.RIGHT.equals(direction)) {
                        if (start.x < rect.x + rect.w && start.y >= rect.y && start.y <= rect.y + rect.h) {                    
                            if (figure == null || rect.x < figure.getBoundingBox().x)
                                figure = child;
                        } 
                    } else if (this.UP.equals(direction)){
                        if (start.y > rect.y && start.x >= rect.x && start.x <= rect.x + rect.w) {
                            if (figure === null || rect.y > figure.getBoundingBox().y)
                                figure = child;
                        }           
                    } else if (this.DOWN.equals(direction)){
                        if (start.y < rect.y + rect.h && start.x >= rect.x && start.x <= rect.x + rect.w) {
                            if (figure === null || rect.y < figure.getBoundingBox().y)
                                figure = child;
                        }                   
                    }
                }
            }
            catch(exc){
                console.log(exc);
            }
        },this));
        return figure;
    },

    adjust:function( connection,  col) {
        var column = col;

        var start = connection.getSource().getPosition();

        var connections = connection.getCanvas().getLines();
        connections.each(function(i,conn) {
            try{
                if (conn===connection)
                    return;
    
                var end = conn.getTarget().getPosition();
                if (start.x < end.x && start.y == end.y) {
                    if (conn.getVertices().getMidpoint().x <= col)
                        column = conn.getVertices().getMidpoint().x - 5;
                }
            }
            catch(exc){
                console.log(exc);
            }
        });
        return column;
    }

    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.connection.SketchBridgedConnectionRouter
 * 
 * Provide a router which routes the connection in a hand drawn manner.
 * 
 * 
 * @inheritable
 * @author Andreas Herz
 * @since 2.7.2
 * @extends  draw2d.layout.connection.MazeConnectionRouter
 */
draw2d.layout.connection.SketchConnectionRouter = draw2d.layout.connection.MazeConnectionRouter.extend({
    NAME : "draw2d.layout.connection.SketchConnectionRouter",


	/**
	 * @constructor 
	 * Creates a new Router object.
	 * 
	 */
    init: function(){
    	this._super();
    	
    	this.useSpline = true;
    	this.useShift = 5;
    	this.useSimplifyValue=0.2;
        this.finder = new PF.JumpPointFinder({allowDiagonal: false,dontCrossCorners: true});
    },
    
    /**
     * @method
     * Callback method if the router has been assigned to a connection.
     * 
     * @param {draw2d.Connection} connection The assigned connection
     * @template
     * @since 2.7.2
     */
    onInstall: function(connection){
        connection.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());
       
    }
 
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.mesh.MeshLayouter
 * Layouter for a mesh or grid. 
 *
 * @author Andreas Herz
 */
draw2d.layout.mesh.MeshLayouter = Class.extend({

	/**
	 * @constructor 
	 * Creates a new layouter object.
	 */
    init: function(){
    },
    
    /**
     * @method
     * Return a changes list for an existing mesh/canvas to ensure that the element to insert 
     * did have enough space.
     * 
     * @param {draw2d.Canvas} canvas the canvas to use for the analytic
     * @param {draw2d.Figure} figure The figure to add to the exising canvas
     * 
     * 
     * @return {draw2d.util.ArrayList} a list of changes to apply if the user want to insert he figure.
     */
    add:function( canvas, figure)
    {
    	return new draw2d.util.ArrayList();
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.mesh.ExplodeLayouter
 * Routes a {@link draw2d.Connection}, possibly using a constraint.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.mesh.MeshLayouter
 */
draw2d.layout.mesh.ExplodeLayouter = draw2d.layout.mesh.MeshLayouter.extend({

	MIN_MARGIN : 40,
	
	/**
	 * @constructor 
	 * Creates a new layouter object.
	 */
    init: function(){
    },
    
    /**
     * @method
     * Return a changes list for an existing mesh/canvas to ensure that the element to insert 
     * did have enough space.
     * 
     * @param {draw2d.Canvas} canvas the canvas to use for the analytic
     * @param {draw2d.Figure} figure The figure to add to the exising canvas
     * @param {Number} x requested x-position for the figure
     * @param {Number} y requested y-position for the figure
     * 
     * 
     * @return {draw2d.util.ArrayList} a list of changes to apply if the user want to insert he figure.
     */
    add:function( canvas, figureToAdd)
    {
    	// changes for the differenct octant areas
    	var changes = [];
    	changes[0]= {x:0, y:0};
    	changes[1]= {x:0, y:0};
    	changes[2]= {x:0, y:0};
    	changes[3]= {x:0, y:0};
    	changes[4]= {x:0, y:0};
    	changes[5]= {x:0, y:0};
    	changes[6]= {x:0, y:0};
    	changes[7]= {x:0, y:0};
    	changes[8]= {x:0, y:0};

    	var boundingBox = figureToAdd.getBoundingBox();

    	var figures = canvas.getFigures();
    	var figure = null;
    	
    	var dis=0;
    	var oct =0;
    	var currentOctChanges =null;
    	var i=0;
    	for( i=0; i< figures.getSize();i++){
    		
    		figure = figures.get(i);
    		
    		// calculate the distance of all corners in relation to the requested x/y coordinate
    		//
    		if(figure !== figureToAdd ){
    			dis = figure.getBoundingBox().getDistance(boundingBox);
    			// other figure is to close
    			//
    			if(dis<this.MIN_MARGIN){
    				// determine the octant of the figure
    				oct = this.determineOctant(boundingBox, figure.getBoundingBox());
        			
    				// all other relevant segments must be arranged too!!
    				//
        			switch(oct){
        			case 2:
        				changes[2].x =  Math.max(changes[2].x,this.MIN_MARGIN-dis);
        				changes[3].x =  Math.max(changes[3].x,this.MIN_MARGIN-dis);
        				changes[4].x =  Math.max(changes[4].x,this.MIN_MARGIN-dis);
        				break;
        			case 3:
        				changes[2].x =  Math.max(changes[2].x,this.MIN_MARGIN-dis);
        				changes[3].x =  Math.max(changes[3].x,this.MIN_MARGIN-dis);
        				changes[4].x =  Math.max(changes[4].x,this.MIN_MARGIN-dis);
        				break;
        			case 4:
        				changes[2].x =  Math.max(changes[2].x,this.MIN_MARGIN-dis);
        				changes[3].x =  Math.max(changes[3].x,this.MIN_MARGIN-dis);
        				changes[4].x =  Math.max(changes[4].x,this.MIN_MARGIN-dis);
        				changes[4].y =  Math.max(changes[4].y,this.MIN_MARGIN-dis);
        				changes[5].y =  Math.max(changes[5].y,this.MIN_MARGIN-dis);
        				changes[6].y =  Math.max(changes[6].y,this.MIN_MARGIN-dis);
        				break;
        			case 5:
        				changes[4].y =  Math.max(changes[4].y,this.MIN_MARGIN-dis);
        				changes[5].y =  Math.max(changes[5].y,this.MIN_MARGIN-dis);
        				changes[6].y =  Math.max(changes[6].y,this.MIN_MARGIN-dis);
        				break;
        			case 6:
        				changes[4].y =  Math.max(changes[4].y,this.MIN_MARGIN-dis);
        				changes[5].y =  Math.max(changes[5].y,this.MIN_MARGIN-dis);
        				changes[6].y =  Math.max(changes[6].y,this.MIN_MARGIN-dis);
        				break;
        			case 8:
        				// overlapping
        				// we must determine the new distance with the border of the figures
        				dis = (boundingBox.getBottomRight().getDistance(figure.getBoundingBox().getTopLeft()))|0;
        				
        				changes[2].x =  Math.max(changes[2].x,this.MIN_MARGIN+dis);
        				changes[3].x =  Math.max(changes[3].x,this.MIN_MARGIN+dis);
        				changes[4].x =  Math.max(changes[4].x,this.MIN_MARGIN+dis);
        				changes[4].y =  Math.max(changes[4].y,this.MIN_MARGIN+dis);
        				changes[5].y =  Math.max(changes[5].y,this.MIN_MARGIN+dis);
        				changes[6].y =  Math.max(changes[6].y,this.MIN_MARGIN+dis);
        				changes[8].x =  Math.max(changes[8].x,this.MIN_MARGIN+dis);
//        				changes[8].y =  Math.max(changes[8].y,this.MIN_MARGIN+dis);
        			}
    			}
    		}
    		// Falls die minimale Distance zu den Objecten kleiner 80 ist, muss ein layout erfolgen
    	}

    	// calculate the adjustment for each figure
    	//
    	var result = new draw2d.util.ArrayList();
    	for( i=0; i< figures.getSize();i++){
    		figure = figures.get(i);
    		if(figure !== figureToAdd ){
				oct = this.determineOctant(boundingBox, figure.getBoundingBox());
				currentOctChanges = changes[oct];
				if(currentOctChanges.x!==0 || currentOctChanges.y!==0){
					result.add(new draw2d.layout.mesh.ProposedMeshChange(figure, currentOctChanges.x,currentOctChanges.y));
				}
    		}
    	}
    	
    	return result;
    },
    
    
    /**
     * @method
     * Determin Octant
	 *
	 *    0 | 1 | 2
	 *    __|___|__
	 *    7 | 8 | 3
     *    __|___|__
	 *    6 | 5 | 4
     *
     * @param cx
     * @param cy
     * @param cw
     * @param ch
     * @param ox
     * @param oy
     * @param ow
     * @param oh
     * @returns {Number}
     */
    determineOctant: function(r1, r2){
		var ox = r1.x;
		var oy = r1.y;
		var ow = r1.w;
		var oh = r1.h;
		
		var cx = r2.x;
		var cy = r2.y;
		var cw = r2.w;
		var ch = r2.h;
		var oct =0;

		if(cx + cw <= ox){
			if((cy + ch) <= oy){
				oct = 0;
			}
			else if(cy >= (oy + oh)){
				oct = 6;
			}
			else{
				oct = 7;
			}
	    }
		else if(cx >= ox + ow){
			if(cy + ch <= oy){
				oct = 2;
			}
			else if(cy >= oy + oh){
				oct = 4;
			}
			else{
				oct = 3;
			}
		}
		else if(cy + ch <= oy){
			oct = 1;
		}
		else if(cy >= oy + oh){
			oct = 5;
		}
		else{
			oct= 8;
		}
		
		return oct;
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.mesh.ProposedMeshChange
 * Change proposal for grid/mesh layout changes.
 *
 * @author Andreas Herz
 */
draw2d.layout.mesh.ProposedMeshChange = Class.extend({

	/**
	 * @constructor 
	 * Creates change object.
	 */
    init: function(figure, x, y){
    	this.figure = figure;
    	this.x = x;
    	this.y = y;
    },
    
    /**
     * @method
     * Return the related figure.
     * 
     * @return {draw2d.Figure} the figure to the related change proposal
     */
    getFigure:function( )
    {
    	return this.figure;
    },
    
    /**
     * @method
     * The proposed x-coordinate.
     * 
     * @return {Number}
     */
    getX: function(){
    	return this.x;
    },
    
    /**
     * @method
     * The proposed y-coordinate
     * 
     * @return {Number}
     */
    getY: function(){
    	return this.y;
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.Locator
 * 
 * Controls the location of an IFigure. 
 *
 * @author Andreas Herz
 */
draw2d.layout.locator.Locator= Class.extend({
    NAME : "draw2d.layout.locator.Locator",
   
    /**
     * @constructor
     * Initial Constructor
     * 
     * @param {draw2d.Figure} [parentShape] the parent or owner of the child 
     */
    init:function(parentShape )
    {
        this.parent = parentShape;
    },
    
    /**
     * @method
     * Returns the associated owner of the locator
     *
     * @return {draw2d.Figure}
     **/
    getParent:function(){
       return this.parent;
    },
    
    
    /**
     * @method
     * Set the associated owner of the locator
     *
     * @param {draw2d.Figure} parentShape
     **/
    setParent:function(parentShape){
        this.parent= parentShape;
    },
    
    /**
     * @method
     * Controls the location of an I{@link draw2d.Figure} 
     *
     * @param {Number} index child index of the figure
     * @param {draw2d.Figure} figure the figure to control
     * 
     * @template
     **/
    relocate:function(index, figure){
    	
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.PortLocator
 * 
 * Repositions a Figure attached to a Connection when the 
 * Connection is moved. Provides for alignment at the start 
 * (source), middle, or end (target) of the Connection.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.PortLocator = draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.PortLocator",
    
    /**
     * @constructor
     * Default constructor for a Locator which can layout a port in context of a 
     * {@link grapiti.shape.node.Node}
     * 
     */
    init:function( ){
      this._super();
    },
    
    applyConsiderRotation: function(port, x, y){
    	var parent = port.getParent();
    
    	var halfW = parent.getWidth()/2;
    	var halfH = parent.getHeight()/2;
    	var rotAngle = parent.getRotationAngle();
    	var m = Raphael.matrix();
    	m.rotate(rotAngle, halfW, halfH);
        if(rotAngle=== 90|| rotAngle===270){
            var ratio = parent.getHeight()/parent.getWidth();
            m.scale(ratio, 1/ratio, halfW, halfH);
        }

        port.setPosition( m.x(x,y), m.y(x,y));
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.XYAbsPortLocator
 * 
 * Create a locator for fixed x/y coordinate position.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.PortLocator
 * @since 4.0.0
 */
draw2d.layout.locator.XYAbsPortLocator = draw2d.layout.locator.PortLocator.extend({
    NAME : "draw2d.layout.locator.XYAbsPortLocator",
     
    /**
     * @constructor
     * 
     * {@link grapiti.shape.node.Node}
     * 
     * @param {Number} x the x coordinate of the port relative to the left of the parent
     * @param {Number} y the y coordinate of the port relative to the top of the parent
     */
    init:function(x ,y ){
      this._super();
      
      this.x = x;
      this.y = y;
    },    
   
   /**
    * @method
    * Controls the location of an I{@link draw2d.Figure} 
    *
    * @param {Number} index child index of the figure
    * @param {draw2d.Figure} figure the figure to control
    * 
    * @template
    **/
    relocate:function(index, figure){
        this.applyConsiderRotation( figure, this.x, this.y);
    }
    
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.XYRelPortLocator
 * 
 * Create a locator for a relative x/y coordinate position. The coordinates are named in percentage 
 * relative to the top/left corner of the parent node.
 * 
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.PortLocator
 * @since 4.0.0
 */
draw2d.layout.locator.XYRelPortLocator = draw2d.layout.locator.PortLocator.extend({
    NAME : "draw2d.layout.locator.XYRelPortLocator",
     
    /**
     * @constructor
     * 
     * {@link grapiti.shape.node.Node}
     * 
     * @param {Number} xPercentage the x coordinate in percent of the port relative to the left of the parent
     * @param {Number} xPercentage the y coordinate in percent of the port relative to the top of the parent
     */
    init:function(xPercentage ,yPercentage ){
      this._super();
      
      this.x = xPercentage;
      this.y = yPercentage;
    },    
   
   /**
    * @method
    * Controls the location of an I{@link draw2d.Figure} 
    *
    * @param {Number} index child index of the figure
    * @param {draw2d.Figure} figure the figure to control
    * 
    * @template
    **/
    relocate:function(index, figure){
        var node = figure.getParent();
        var x = node.getWidth()/100 * this.x;
        var y = node.getHeight()/100  * this.y;

        this.applyConsiderRotation( figure, x, y);
    }
    
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.InputPortLocator
 * 
 * Repositions a Figure attached to a Connection when the 
 * Connection is moved. Provides for alignment at the start 
 * (source), middle, or end (target) of the Connection.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.InputPortLocator = draw2d.layout.locator.PortLocator.extend({
    NAME : "draw2d.layout.locator.InputPortLocator",
     
    /**
     * @constructor
     * Default constructor for a Locator which can layout a port in context of a 
     * {@link grapiti.shape.node.Node}
     * 
     */
    init:function(  ){
      this._super();
    },    
   
   /**
    * @method
    * Controls the location of an I{@link draw2d.Figure} 
    *
    * @param {Number} index port index of the figure
    * @param {draw2d.Figure} figure the figure to control
    * 
    * @template
    **/
    relocate:function(index, figure){
        var node = figure.getParent();
        var dividerFactor = 1;
        var thisNAME = this.NAME;
        var portIndex =1;
        node.getPorts().each(function(i,p){
        	portIndex = (p===figure)?dividerFactor:portIndex;
        	dividerFactor += p.getLocator().NAME === thisNAME?1:0;
        });
        this.applyConsiderRotation( figure, 0, (node.getHeight()/dividerFactor)*portIndex);
    }
    
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.OutputPortLocator
 * 
 * Repositions a Figure attached to a Connection when the 
 * Connection is moved. Provides for alignment at the start 
 * (source), middle, or end (target) of the Connection.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.OutputPortLocator = draw2d.layout.locator.PortLocator.extend({
    NAME : "draw2d.layout.locator.OutputPortLocator",
    
    /**
     * @constructor
     * Default constructor for a Locator which can layout a port in context of a 
     * {@link grapiti.shape.node.Node}
     * 
     */
    init:function( ){
      this._super();
    },    
   
   /**
    * @method
    * Controls the location of an I{@link draw2d.Figure} 
    *
    * @param {Number} index child index of the figure
    * @param {draw2d.Figure} figure the figure to control
    * 
    * @template
    **/
    relocate:function(index, figure){
        var node = figure.getParent();
        var dividerFactor = 1;
        var thisNAME = this.NAME;
        var portIndex =1;
        node.getPorts().each(function(i,p){
        	portIndex = (p===figure)?dividerFactor:portIndex;
        	dividerFactor += p.getLocator().NAME === thisNAME?1:0;
        });
        this.applyConsiderRotation( figure, node.getWidth(), (node.getHeight()/dividerFactor)*portIndex);
    }
    
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.locator.ConnectionLocator
 * 
 * Repositions a Figure attached to a Connection when the 
 * Connection is moved. Provides for alignment at the start 
 * (source), middle, or end (target) of the Connection.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.ConnectionLocator= draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.ConnectionLocator",
    
    /**
     * @constructor
     * Default constructor for a Locator which can layout a figure in context of a 
     * {@link grapiti.Connector}
     * 
     * @param {draw2d.Figure} parentShape the base or parent figure for the locator
     */
    init:function( parentShape)
    {
      this._super(parentShape);
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.ManhattanMidpointLocator
 * 
 * A ManhattanMidpointLocator that is used to place figures at the midpoint of a Manhatten routed
 * connection. The midpoint is always in the center of an edge.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.ConnectionLocator
 */
draw2d.layout.locator.ManhattanMidpointLocator= draw2d.layout.locator.ConnectionLocator.extend({
    NAME : "draw2d.layout.locator.ManhattanMidpointLocator",
    
    /**
     * @constructor
     * Constructs a ManhattanMidpointLocator with associated Connection c.
     * 
     * @param {draw2d.Connection} c the connection associated with the locator
     */
    init: function(c)
    {
      this._super(c);
    },
    
    
    /**
     * @method
     * Relocates the given Figure always in the center of an edge.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var conn = this.getParent();
       var points = conn.getVertices();
       
       var segmentIndex = Math.floor((points.getSize() -2) / 2);
       if (points.getSize() <= segmentIndex+1)
          return; 
    
       var p1 = points.get(segmentIndex);
       var p2 = points.get(segmentIndex + 1);
    
       var x = ((p2.x - p1.x) / 2 + p1.x - target.getWidth()/2)|0;
       var y = ((p2.y - p1.y) / 2 + p1.y - target.getHeight()/2)|0;
    
       target.setPosition(x,y);
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.PolylineMidpointLocator
 * 
 * A PolylineMidpointLocator is used to place figures at the midpoint of a routed
 * connection. <br>
 * If the connection did have an odd count of points the figure is located in the center vertex of the polyline.<br>
 * On an even count of junction point, the figure will be center on the middle segment of the ploy line. 
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.ManhattanMidpointLocator
 */
draw2d.layout.locator.PolylineMidpointLocator= draw2d.layout.locator.ManhattanMidpointLocator.extend({
    NAME : "draw2d.layout.locator.PolylineMidpointLocator",
    
    /**
     * @constructor
     * Constructs a ManhattanMidpointLocator with associated Connection c.
     * 
     * @param {draw2d.Connection} c the connection associated with the locator
     */
    init: function(c)
    {
      this._super(c);
    },
    
    
    /**
     * @method
     * Relocates the given Figure.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var conn = this.getParent();
       var points = conn.getVertices();
       
       // it has an event count of points -> use the manhattan algorithm...this is working 
       // well in this case
       if(points.getSize()%2===0){
           this._super(index, target);
       }
       // odd count of points. take the center point as fulcrum
       else{

           var index = Math.floor(points.getSize() / 2);
        
           var p1 = points.get(index);
        
           
           target.setPosition(p1.x- (target.getWidth()/2),p1.y-(target.getHeight()/2));
       }      
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.TopLocator
 * 
 * A TopLocator  is used to place figures at the top/center of a parent shape.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.TopLocator= draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.TopLocator",
    
    /**
     * @constructor
     * Constructs a ManhattanMidpointLocator with associated Connection c.
     * 
     * @param {draw2d.Figure} parent the parent associated with the locator
     */
    init: function(parent)
    {
      this._super(parent);
    },
    
    
    /**
     * @method
     * Relocates the given Figure.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var parent = this.getParent();
       var boundingBox = parent.getBoundingBox();
    
       var targetBoundingBox = target.getBoundingBox();
       if(target instanceof draw2d.Port){
           target.setPosition(boundingBox.w/2,0);
       }
       else{
           target.setPosition(boundingBox.w/2-(targetBoundingBox.w/2),-(targetBoundingBox.h+2));
       }
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.BottomLocator
 * 
 * A TopLocator  is used to place figures at the top/center of a parent shape.
 *
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.BottomLocator= draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.BottomLocator",
    
    /**
     * @constructor
     * Constructs a ManhattanMidpointLocator with associated Connection c.
     * 
     * @param {draw2d.Figure} parent the parent associated with the locator
     */
    init: function(parent)
    {
      this._super(parent);
    },
    
    
    /**
     * @method
     * Relocates the given Figure.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var parent = this.getParent();
       var boundingBox = parent.getBoundingBox();
    
       var targetBoundingBox = target.getBoundingBox();
       if(target instanceof draw2d.Port){
           target.setPosition(boundingBox.w/2,boundingBox.h);
       }
       else{
           target.setPosition(boundingBox.w/2-targetBoundingBox.w/2,2+boundingBox.h);
       }
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.LeftLocator
 * 
 * A LeftLocator is used to place figures to the left of a parent shape.
 *
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *
 *     // create a basic figure and add a Label/child via API call
 *     //
 *     var start = new draw2d.shape.node.Start();
 *     start.addFigure(new draw2d.shape.basic.Label("Left Label"), new draw2d.layout.locator.LeftLocator(start));	
 *     canvas.addFigure( start, 100,50);
 *
 *     
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.LeftLocator= draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.LeftLocator",
    
    /**
     * @constructor
     * Constructs a locator with associated parent.
     * 
     * @param {draw2d.Figure} parent the parent associated with the locator
     */
    init: function(parent)
    {
      this._super(parent);
    },
    
    
    /**
     * @method
     * Relocates the given Figure.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var parent = this.getParent();
       var boundingBox = parent.getBoundingBox();

       var targetBoundingBox = target.getBoundingBox();
       target.setPosition(-targetBoundingBox.w-5,(boundingBox.h/2)-(targetBoundingBox.h/2));
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.RightLocator
 * 
 * A RightLocator is used to place figures to the right of a parent shape.
 *
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *
 *     // create a basic figure and add a Label/child via API call
 *     //
 *     var end = new draw2d.shape.node.End();
 *     end.addFigure(new draw2d.shape.basic.Label("Right Label"), new draw2d.layout.locator.RightLocator(end));	
 *     canvas.addFigure( end, 50,50);
 *
 *     
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.RightLocator = draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.RightLocator",
    
    /**
     * @constructor
     * Constructs a locator with associated parent.
     * 
     * @param {draw2d.Figure} parent the parent associated with the locator
     */
    init: function(parent)
    {
      this._super(parent);
    },
    
    
    /**
     * @method
     * Relocates the given Figure.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var parent = this.getParent();
       var boundingBox = parent.getBoundingBox();
      
       var targetBoundingBox = target.getBoundingBox();
       target.setPosition(boundingBox.w+5,(boundingBox.h/2)-(targetBoundingBox.h/2));
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.locator.CenterLocator
 * 
 * A CenterLocator is used to place figures in the center of a parent shape.
 *
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *
 *     // create a basic figure and add a Label/child via API call
 *     //
 *     var circle = new draw2d.shape.basic.Circle(120);
 *     circle.setStroke(3);
 *     circle.setColor("#A63343");
 *     circle.setBackgroundColor("#E65159");
 *     circle.addFigure(new draw2d.shape.basic.Label("Center Label"), new draw2d.layout.locator.CenterLocator(circle));	
 *     canvas.addFigure( circle, 100,50);
 *
 *     
 * @author Andreas Herz
 * @extend draw2d.layout.locator.Locator
 */
draw2d.layout.locator.CenterLocator= draw2d.layout.locator.Locator.extend({
    NAME : "draw2d.layout.locator.CenterLocator",
    
    /**
     * @constructor
     * Constructs a locator with associated parent.
     * 
     * @param {draw2d.Figure} parent the parent associated with the locator
     */
    init: function(parent)
    {
      this._super(parent);
    },
    
    
    /**
     * @method
     * Relocates the given Figure.
     *
     * @param {Number} index child index of the target
     * @param {draw2d.Figure} target The figure to relocate
     **/
    relocate:function(index, target)
    {
       var parent = this.getParent();
       var boundingBox = parent.getBoundingBox();

       // TODO: instanceof is always a HACK. ugly. Redirect the call to the figure instead of 
       // determine the position with a miracle.
       //
       if(target instanceof draw2d.Port){
           target.setPosition(boundingBox.w/2,boundingBox.h/2);
       }
       else{
           var targetBoundingBox = target.getBoundingBox();
           target.setPosition(boundingBox.w/2-targetBoundingBox.w/2,boundingBox.h/2-(targetBoundingBox.h/2));
       }
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.EditPolicy
 * 
 * A pluggable contribution implementing a portion of an behavior. 
 *
 *
 * EditPolicies should determine an Canvas or figure editing capabilities. It is possible to implement 
 * an figure such that it handles all editing responsibility. However, it is much more flexible 
 * and object-oriented to use EditPolicies. Using policies, you can pick and choose the editing 
 * behavior for an figure/canvas without being bound to its class hierarchy. Code management is easier. 
 * 
 * 
 * This interface is not intended to be implemented by clients. Clients should inherit from {@link draw2d.policy.figure.SelectionFeedbackPolicy}
 * or {@link draw2d.policy.canvas.SelectionPolicy}. 
 * 
 * @author Andreas Herz
 */
draw2d.policy.EditPolicy = Class.extend({

    NAME : "draw2d.policy.EditPolicy",
    
    /**
     * @constructor 
     * 
     */
    init: function(){
    },

    /**
     * @method
     * Called by the host if the policy has been installed.
     * 
     * @param {draw2d.Canvas/draw2d.Figure} host
     */
    onInstall: function( host){
    },
    
    /**
     * @method
     * Called by the host if the policy has been uninstalled.
     * 
     * @param {draw2d.Canvas/draw2d.Figure} host
     */
    onUninstall: function( host){
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.CanvasPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.EditPolicy
 */
draw2d.policy.canvas.CanvasPolicy = draw2d.policy.EditPolicy.extend({

    NAME : "draw2d.policy.canvas.CanvasPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    
    onInstall: function(canvas){
    },
    
    onUninstall: function(canvas){
    },
    
    /**
     * @method
     * Called by the canvas if the user click on a figure.
     * 
     * @param {draw2d.Figure} the figure under the click event. Can be null
     * @param {Number} mouseX the x coordinate of the mouse during the click event
     * @param {Number} mouseY the y coordinate of the mouse during the click event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * 
     * @since 3.0.0
     * 
     * @template
     */
    onClick: function(figure, mouseX, mouseY, shiftKey, ctrlKey){
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse event
     * @param {Number} y the y-coordinate of the mouse event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseMove:function(canvas, x, y, shiftKey, ctrlKey){
    },
    
    /**
     * @method
     * Called by the canvas if the user double click on a figure.
     * 
     * @param {draw2d.Figure} the figure under the double click event. Can be null
     * @param {Number} mouseX the x coordinate of the mouse during the click event
     * @param {Number} mouseY the y coordinate of the mouse during the click event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * 
     * @since 4.1.0
     * 
     * @template
     */
    onDoubleClick: function(figure, mouseX, mouseY, shiftKey, ctrlKey){
    },
    
    
    /**
     * @method
     * Adjust the coordinates to the given constraint.
     * 
     * @param figure
     * @param {draw2d.geo.Point} clientPos
     * @returns {draw2d.geo.Point} the contraint position of th efigure
     */
    snap: function(canvas, figure, clientPos){
        return clientPos;
    },

    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     */
    onMouseDown:function(canvas, x, y, shiftKey, ctrlKey){
        
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseUp: function(figure, x, y, shiftKey, ctrlKey){
        
    },
    
    
    /**
     * @method
     * Helper method to make an monochrome GIF image WxH pixels big, first create a properly sized array: var pixels = new Array(W*H);. 
     * Then, for each pixel X,Y that should be opaque, store a 1 at the proper location: pixels[X+Y*W] = 1;. 
     * Finally, create the image: var my_glif = createGif(W, H, pixels, color);
     * "0" pixels are transparent.
     * The <b>color</b> defines the foreground color.
     * 
     * Now, you can specify this image as the SRC attribute of an IMG tag: document.write("<IMG SRC=\"" + my_glif + "\">"); 
     * or for the canvas as background-image css attribute.
     *  
     * 
     * @param w
     * @param h
     * @param d
     * @param color
     * @returns {String}
     */
    createMonochromGif: function(w,h,d,color) {
        var r = String.fromCharCode(w%256) + String.fromCharCode(w/256) + String.fromCharCode(h%256) + String.fromCharCode(h/256) ;
        var gif = "GIF89a" + r + "\xf0\0\0\xff\xff\xff" + String.fromCharCode(color.red) + String.fromCharCode(color.green) + String.fromCharCode(color.blue) + "\x21\xf9\4\1\0\0\0\0,\0\0\0\0" + r + "\0\2";

        // help method to generate uncompressed in memory GIF data structur without the usage of a canvas or any other
        // heavy weight stuff.
        var b = { 
                bit: 1,
                byte_: 0,
                data : "",
    
            writeBit: function(b) {
                if(b) this.byte_ |= this.bit;
                this.bit <<= 1;
                if(this.bit == 256) {
                    this.bit = 1;
                    this.data += String.fromCharCode(this.byte_);
                    this.byte_ = 0;
                }
            },
            
            get:function() {
                var result = "";
                var data = this.data;
                if(this.bit != 1) { data += String.fromCharCode(this.byte_); }
                for(var i=0; i<data.length + 1; i+=255) {
                    chunklen = data.length - i; if(chunklen < 0) chunklen = 0;
                    if(chunklen > 255) chunklen=255;
                    result += String.fromCharCode(chunklen) + data.substring(i,i+255);
                }
                return result + "\0";
            }
        };
        
        for(var y=0; y<h; y++) {
            for(var x=0; x<w; x++) {
                b.writeBit(d[x+w*y]); b.writeBit(0); b.writeBit(0);
                b.writeBit(0); b.writeBit(0); b.writeBit(1);
            }
        }
        gif += b.get() + ";" ;

        return "data:image/gif;base64," + draw2d.util.Base64.encode(gif);
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.SelectionPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.CanvasPolicy
 */
draw2d.policy.canvas.SelectionPolicy = draw2d.policy.canvas.CanvasPolicy.extend({

    NAME : "draw2d.policy.canvas.SelectionPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
 

    unselect: function(canvas, figure){
        canvas.getSelection().remove(figure);

        figure.unselect();

        canvas.selectionListeners.each(function(i,w){
            w.onSelectionChanged(null);
        });    
   }

});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.SingleSelectionPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.SelectionPolicy
 */
draw2d.policy.canvas.SingleSelectionPolicy =  draw2d.policy.canvas.SelectionPolicy.extend({

    NAME : "draw2d.policy.canvas.SingleSelectionPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
        this.mouseMovedDuringMouseDown = false;
        this.mouseDraggingElement = null;
        this.mouseDownElement = null;
    },
   
    select: function(canvas, figure){
        if(canvas.getSelection().getAll().contains(figure)){
            return; // nothing to to
        }
        
        if(canvas.getSelection().getPrimary()!==null){
            this.unselect(canvas, canvas.getSelection().getPrimary());
        }
      
        if(figure !==null) {
            figure.select(true); // primary selection
        }
        
        canvas.getSelection().setPrimary(figure);

        // inform all selection listeners about the new selection.
        //
        canvas.selectionListeners.each(function(i,w){
            w.onSelectionChanged(figure);
        });
    },
    

    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     */
    onMouseDown:function(canvas, x, y, shiftKey, ctrlKey){
        this.mouseMovedDuringMouseDown  = false;
        var canDragStart = true;

        var figure = canvas.getBestFigure(x, y);

        // check if the user click on a child shape. DragDrop and movement must redirect
        // to the parent
        // Exception: Port's
        while((figure!==null && figure.getParent()!==null) && !(figure instanceof draw2d.Port)){
            figure = figure.getParent();
        }

        if (figure !== null && figure.isDraggable()) {
            canDragStart = figure.onDragStart(x - figure.getAbsoluteX(), y - figure.getAbsoluteY());
            // Element send a veto about the drag&drop operation
            if (canDragStart === false) {
                this.mouseDraggingElement = null;
                this.mouseDownElement = figure;
            }
            else {
                this.mouseDraggingElement = figure;
                this.mouseDownElement = figure;
            }
        }

        if (figure !== canvas.getSelection().getPrimary() && figure !== null && figure.isSelectable() === true) {
            this.select(canvas,figure);

            // its a line
            if (figure instanceof draw2d.shape.basic.Line) {
                // you can move a line with Drag&Drop...but not a connection.
                // A Connection is fixed linked with the corresponding ports.
                //
                if (!(figure instanceof draw2d.Connection)) {
                    canvas.draggingLineCommand = figure.createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE));
                    if (canvas.draggingLineCommand !== null) {
                        canvas.draggingLine = figure;
                    }
                }
            }
            else if (canDragStart === false) {
                figure.unselect();
            }
        }
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        this.mouseMovedDuringMouseDown = true;
        
        if (this.mouseDraggingElement !== null) {
            // it is only necessary to repaint all connections if we change the layout of any connection
            // This can only happen if we:
            //    - with at least one connection intersection
            //    - we move a "Node. Only a node can have ports and connections
            if(canvas.linesToRepaintAfterDragDrop.isEmpty()===true && (this.mouseDraggingElement instanceof draw2d.shape.node.Node)){
                var nodeConnections = this.mouseDraggingElement.getConnections();
                var newLineIntersections = canvas.lineIntersections.clone();
                canvas.lineIntersections.each($.proxy(function(i, inter){
                    
                    if(nodeConnections.contains(inter.line) || nodeConnections.contains(inter.other)){
                        newLineIntersections.remove(inter);
                        canvas.linesToRepaintAfterDragDrop.add(inter.line);
                        canvas.linesToRepaintAfterDragDrop.add(inter.other);
                    }
                },this));
                canvas.lineIntersections = newLineIntersections;
                canvas.linesToRepaintAfterDragDrop.each(function(i, line){
                    line.svgPathString=null;
                    line.repaint();
                });
            }
            
            // Can be a ResizeHandle or a normal Figure
            //
            var sel =canvas.getSelection().getAll();
            if(!sel.contains(this.mouseDraggingElement)){
                this.mouseDraggingElement.onDrag(dx, dy, dx2, dy2);
            }
            else{
                sel.each(function(i,figure){
                    figure.onDrag(dx, dy, dx2, dy2);
                });
            }
            
            var p = canvas.fromDocumentToCanvasCoordinate(canvas.mouseDownX + (dx/canvas.zoomFactor), canvas.mouseDownY + (dy/canvas.zoomFactor));           
            var target = canvas.getBestFigure(p.x, p.y,this.mouseDraggingElement);
            
            if (target !== canvas.currentDropTarget) {
                if (canvas.currentDropTarget !== null) {
                    canvas.currentDropTarget.onDragLeave(this.mouseDraggingElement);                     
                    canvas.currentDropTarget = null;
                }
                if (target !== null) {
                    canvas.currentDropTarget = target.onDragEnter(this.mouseDraggingElement);
                }
            }
       }
       // Connection didn't support panning at the moment. There is no special reason for that. Just an interaction
       // decision.
       //
       else if(this.mouseDownElement!==null && !(this.mouseDownElement instanceof draw2d.Connection)){
           this.mouseDownElement.onPanning(dx, dy, dx2, dy2);
       } 
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse up event
     * @param {Number} y the y-coordinate of the mouse up event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseUp: function(canvas, x, y, shiftKey, ctrlKey){
        if (this.mouseDraggingElement !== null) {
            var sel =canvas.getSelection().getAll();
            if(!sel.contains(this.mouseDraggingElement)){
                this.mouseDraggingElement.onDragEnd();
            }
            else{
                canvas.getCommandStack().startTransaction();
                canvas.getSelection().getAll().each(function(i,figure){
                     figure.onDragEnd();
                });
                canvas.getCommandStack().commitTransaction();
            }
            if(canvas.currentDropTarget!==null){
                this.mouseDraggingElement.onDrop(canvas.currentDropTarget);
                canvas.currentDropTarget.onDragLeave(this.mouseDraggingElement);
                canvas.currentDropTarget = null;
            }
            this.mouseDraggingElement = null;
        }
        
        // Reset the current selection if the user click in the blank canvas.
        // Don't reset the selection if the user pan the canvas
        //
        if (this.mouseDownElement === null && this.mouseMovedDuringMouseDown===false) {
            this.select(canvas,null);
        }

        this.mouseDownElement = null;
        this.mouseMovedDuringMouseDown  = false;
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.PanningSelectionPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.SingleSelectionPolicy
 */
draw2d.policy.canvas.PanningSelectionPolicy =  draw2d.policy.canvas.SingleSelectionPolicy.extend({

    NAME : "draw2d.policy.canvas.PanningSelectionPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
     },
   
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        
        this._super(canvas, dx,dy,dx2,dy2);
        
        if (this.mouseDraggingElement === null && this.mouseDownElement===null) {
           var area = canvas.getScrollArea();
           area.scrollTop(area.scrollTop()+dy2);
           area.scrollLeft(area.scrollLeft()+dx2);
       }
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.BoundingboxSelectionPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.SelectionPolicy
 */
draw2d.policy.canvas.BoundingboxSelectionPolicy =  draw2d.policy.canvas.SingleSelectionPolicy.extend({

    NAME : "draw2d.policy.canvas.BoundingboxSelectionPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
        
        this.boundingBoxFigure1 =null;
        this.boundingBoxFigure2 =null;
        this.x = 0;
        this.y = 0;
     },
   
     select: function(canvas, figure){
         if(canvas.getSelection().getAll().contains(figure)){
             return; // noting to to
         }
  
         if(figure !==null) {
             figure.select(true); // primary selection
         }
         
         canvas.getSelection().setPrimary(figure);

         // inform all selection listeners about the new selection.
         //
         canvas.selectionListeners.each(function(i,w){
             w.onSelectionChanged(figure);
         });
     },
     
     /**
      * @method
      * 
      * @param {draw2d.Canvas} canvas
      * @param {Number} x the x-coordinate of the mouse down event
      * @param {Number} y the y-coordinate of the mouse down event
      * @param {Boolean} shiftKey true if the shift key has been pressed during this event
      * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
      */
     onMouseDown:function(canvas, x, y, shiftKey, ctrlKey){
        this.x = x;
        this.y = y;

        var currentSelection = canvas.getSelection().getAll();
        
        // COPY_PARENT
        // this code part is copied from the parent implementation. The main problem is, that 
        // the sequence of unselect/select of elements is broken if we call the base implementation
        // in this case a wrong of events is fired if we select a figure if alread a figure is selected!
        // WRONG: selectNewFigure -> unselectOldFigure
        // RIGHT: unselectOldFigure -> selectNEwFigure
        // To ensure this I must copy the parent code and postpond the event propagation
        //
        this.mouseMovedDuringMouseDown  = false;
        var canDragStart = true;

        var figure = canvas.getBestFigure(x, y);

        // check if the user click on a child shape. DragDrop and movement must redirect
        // to the parent
        // Exception: Port's
        while((figure!==null && figure.getParent()!==null) && !(figure instanceof draw2d.Port)){
            figure = figure.getParent();
        }

        if (figure !== null && figure.isDraggable()) {
            canDragStart = figure.onDragStart(x - figure.getAbsoluteX(), y - figure.getAbsoluteY());
            // Element send a veto about the drag&drop operation
            if (canDragStart === false) {
                this.mouseDraggingElement = null;
                this.mouseDownElement = figure;
            }
            else {
                this.mouseDraggingElement = figure;
                this.mouseDownElement = figure;
            }
        }

        // we click on an element which are not part of the current selection
        // => reset the "old" current selection if we didn't press the shift key
        if(shiftKey === false){
            if(this.mouseDownElement!==null && this.mouseDownElement.isResizeHandle===false && !currentSelection.contains(this.mouseDownElement)){
                currentSelection.each($.proxy(function(i, figure){
                    this.unselect(canvas,figure);
                },this));
            }
        }

        if (figure !== canvas.getSelection().getPrimary() && figure !== null && figure.isSelectable() === true) {
            this.select(canvas,figure);

            // its a line
            if (figure instanceof draw2d.shape.basic.Line) {
                // you can move a line with Drag&Drop...but not a connection.
                // A Connection is fixed linked with the corresponding ports.
                //
                if (!(figure instanceof draw2d.Connection)) {
                    canvas.draggingLineCommand = figure.createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE));
                    if (canvas.draggingLineCommand !== null) {
                        canvas.draggingLine = figure;
                    }
                }
            }
            else if (canDragStart === false) {
                figure.unselect();
            }
        }
        // END_COPY FROM PARENT
        
     	
     	// inform all figures that they have a new ox/oy position for the relative
     	// drag/drop operation
        currentSelection = canvas.getSelection().getAll();
        currentSelection.each($.proxy(function(i,figure){
             var canDragStart= figure.onDragStart(figure.getAbsoluteX(),figure.getAbsoluteY());
             // its a line
             if (figure instanceof draw2d.shape.basic.Line) {
                 
             }
             else if(canDragStart===false){
                 this.unselect(canvas,figure);
             }
        },this));
     },
     
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        
        this._super(canvas, dx,dy,dx2,dy2);
        
        if (this.mouseDraggingElement === null && this.mouseDownElement===null && this.boundingBoxFigure1===null) {
            this.boundingBoxFigure1 = new draw2d.shape.basic.Rectangle(1,1);
            this.boundingBoxFigure1.setPosition(this.x,this.y);
            this.boundingBoxFigure1.setCanvas(canvas);
            this.boundingBoxFigure1.setBackgroundColor("#0f0f0f");
            this.boundingBoxFigure1.setAlpha(0.1);
            
            this.boundingBoxFigure2 = new draw2d.shape.basic.Rectangle(1,1);
            this.boundingBoxFigure2.setPosition(this.x,this.y);
            this.boundingBoxFigure2.setCanvas(canvas);
            this.boundingBoxFigure2.setDashArray("- ");
            this.boundingBoxFigure2.setStroke(1);
            this.boundingBoxFigure2.setColor(new draw2d.util.Color(84,151,220));
            this.boundingBoxFigure2.setBackgroundColor(null);
       }
        
        if (this.boundingBoxFigure1!==null) {
        	this.boundingBoxFigure1.setDimension(Math.abs(dx),Math.abs(dy));
        	this.boundingBoxFigure1.setPosition(this.x + Math.min(0,dx), this.y + Math.min(0,dy));
        	this.boundingBoxFigure2.setDimension(Math.abs(dx),Math.abs(dy));
        	this.boundingBoxFigure2.setPosition(this.x + Math.min(0,dx), this.y + Math.min(0,dy));
        }
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
      * @param {Boolean} shiftKey true if the shift key has been pressed during this event
      * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     */
    onMouseUp:function(canvas, x,y, shiftKey, ctrlKey){
        // delete the current selection if you have clicked in the empty
        // canvas.
        if(this.mouseDownElement===null){
            canvas.getSelection().getAll().each($.proxy(function(i,figure){
                this.unselect(canvas, figure);
            },this));
        }
        else if(this.mouseDownElement instanceof draw2d.ResizeHandle || (this.mouseDownElement instanceof draw2d.shape.basic.LineResizeHandle)){
            // Do nothing
            // A click on a resize handle didn't change the selection of the canvas
            //
        }
        // delete the current selection if you click on another figure than the current
        // selection and you didn't drag the complete selection.
        else if(this.mouseDownElement!==null && this.mouseMovedDuringMouseDown===false){
            var sel =canvas.getSelection().getAll();
            if(!sel.contains(this.mouseDownElement)){
               canvas.getSelection().getAll().each($.proxy(function(i,figure){
                        this.unselect(canvas, figure);
                },this));
            }   
        }
        this._super(canvas, x,y);
        
        if (this.boundingBoxFigure1!==null) {
        	// retrieve all figures which are inside the bounding box and select all of them
        	//
        	var selectionRect = this.boundingBoxFigure1.getBoundingBox();
         	canvas.getFigures().each($.proxy(function(i,figure){
        		if(figure.getBoundingBox().isInside(selectionRect)){
                    var canDragStart = figure.onDragStart(figure.getAbsoluteX(),figure.getAbsoluteY());
                    if(canDragStart===true){
                        this.select(canvas,figure,false);
                    }
        		}
        	},this));
         	
         	var selection = canvas.getSelection();
         	
         	// adding connections to the selection of the source and target port part of the current selection
            canvas.getLines().each($.proxy(function(i,line){
                if(line instanceof draw2d.Connection){
                    if(selection.contains(line.getSource().getParent()) && selection.contains(line.getTarget().getParent())){
                        this.select(canvas,line,false);
                    }
                }
            },this));
         	
    	  this.boundingBoxFigure1.setCanvas(null);
       	  this.boundingBoxFigure1 = null;
      	  this.boundingBoxFigure2.setCanvas(null);
       	  this.boundingBoxFigure2 = null;
        }
   }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.ReadOnlySelectionPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.SelectionPolicy
 */
draw2d.policy.canvas.ReadOnlySelectionPolicy = draw2d.policy.canvas.SelectionPolicy.extend({

    NAME : "draw2d.policy.canvas.ReadOnlySelectionPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    
    /**
     * @method
     * Called by the host if the policy has been installed.
     * 
     * @param {draw2d.Canvas/draw2d.Canvas} canvas
     */
    onInstall: function(canvas){
        canvas.getAllPorts().each($.proxy(function(i,port){
            port.setVisible(false);
        },this));
    },
    
    /**
     * @method
     * Called by the host if the policy has been uninstalled.
     * 
     * @param {draw2d.Canvas/draw2d.Canvas} canvas
     */
    onUninstall: function(canvas){
        canvas.getAllPorts().each($.proxy(function(i,port){
            port.setVisible(true);
        },this));
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        var area = canvas.getScrollArea();
        area.scrollTop(area.scrollTop()+dy2);
        area.scrollLeft(area.scrollLeft()+dx2);
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.DecorationPolicy 
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.CanvasPolicy
 */
draw2d.policy.canvas.DecorationPolicy = draw2d.policy.canvas.CanvasPolicy.extend({

    NAME : "draw2d.policy.canvas.DecorationPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.FadeoutDecorationPolicy
 * 
 * Install this edit policy in a canvas if you want fadeout all decorations like ports, resize handles 
 * if the user didn't move the mouse. This is good for a clean representation of your diagram.
 *  
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.DecorationPolicy
 */
draw2d.policy.canvas.FadeoutDecorationPolicy = draw2d.policy.canvas.DecorationPolicy.extend({

    NAME : "draw2d.policy.canvas.FadeoutDecorationPolicy",
    
    DEFAULT_FADEOUT_DURATION : 60,
    DEFAULT_ALPHA_DECREMENT: 0.05,
    
    /**
     * @constructor 
     * Creates a new fade out policy. Don't forget to install them into the canvas.
     * 
     */
    init: function(){
        this._super();
        this.alpha = 1.0;
        this.alphaDec = this.DEFAULT_ALPHA_DECREMENT;
        this.hidePortsCounter = this.DEFAULT_FADEOUT_DURATION;
        this.canvas = null;
        this.portDragging = false;
    },
    
    onInstall: function(canvas){
        this.canvas = canvas;
        this.timerId = window.setInterval($.proxy(this.onTimer,this), 50);
        
        // initial hide all decorations after install of this policy
        //
        this.hidePortsCounter=1;
        this.alpha = 0.1;
    },
    
    onUninstall: function(canvas){
        window.clearInterval(this.timerId);
        this.canvas.getAllPorts().each($.proxy(function(i,port){
            port.setAlpha(1.0);
        },this));
        
    },
    
    onTimer: function(){
        this.hidePortsCounter--;
        
        if(this.hidePortsCounter<=0 && this.alpha >0){
            this.alpha = Math.max(0,this.alpha-this.alphaDec);
            
            this.canvas.getAllPorts().each($.proxy(function(i,port){
                port.setAlpha(this.alpha);
            },this));
            
            this.canvas.getSelection().getAll().each($.proxy(function(i,figure){
                figure.selectionHandles.each($.proxy(function(i,handle){
                    handle.setAlpha(this.alpha);
                },this));
            },this));
        }
        else if(this.hidePortsCounter>0 && this.alpha!==1.0){
            this.alpha =1;// Math.min(1,this.alpha+0.1);
            this.alphaDec = this.DEFAULT_ALPHA_DECREMENT;
            this.duringHide = false;
            this.canvas.getAllPorts().each($.proxy(function(i,port){
                port.setAlpha(this.alpha);
            },this));
            this.canvas.getSelection().getAll().each($.proxy(function(i,figure){
                figure.selectionHandles.each($.proxy(function(i,handle){
                    handle.setAlpha(this.alpha);
                },this));
            },this));
        }
    },
    
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     */
    onMouseDown:function(canvas, x,y, shiftKey, ctrlKey){
        this.hidePortsCounter=this.DEFAULT_FADEOUT_DURATION;
        this.portDragging = (canvas.getBestFigure(x, y) instanceof draw2d.Port);
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse event
     * @param {Number} y the y-coordinate of the mouse event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseMove:function(canvas, x, y, shiftKey, ctrlKey){
        this.hidePortsCounter=this.DEFAULT_FADEOUT_DURATION;
        this.portDragging = false;
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        if(this.portDragging === false){
            this.hidePortsCounter=0;
            this.alphaDec = 0.1;
            this.onTimer();
        }
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseUp: function(figure, x, y, shiftKey, ctrlKey){
        this.hidePortsCounter=this.DEFAULT_FADEOUT_DURATION;
        this.portDragging = false;
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.CoronaDecorationPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.canvas.DecorationPolicy
 */
draw2d.policy.canvas.CoronaDecorationPolicy = draw2d.policy.canvas.DecorationPolicy.extend({

    NAME : "draw2d.policy.canvas.CoronaDecorationPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
        
        this.startDragX = 0;
        this.startDragY = 0;
   },
    
    onInstall: function(canvas){
        var figures = canvas.getFigures();
        figures.each(function(i,figure){
            figure.getPorts().each(function(i,p){
                p.setVisible(false);
            });
        });
    },
    
    onUninstall: function(canvas){
        var figures = canvas.getFigures();
        figures.each(function(i,figure){
            figure.getPorts().each(function(i,p){
                p.setVisible(true);
            });
        });
    },
    
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     */
    onMouseDown:function(canvas, x, y, shiftKey, ctrlKey){
        this.startDragX = x;
        this.startDragY = y;
   },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse event
     * @param {Number} y the y-coordinate of the mouse event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseMove:function(canvas, x, y, shiftKey, ctrlKey){
        this.updatePorts(canvas, x, y);
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} dx The x diff between start of dragging and this event
     * @param {Number} dy The y diff between start of dragging and this event
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onMouseDrag:function(canvas, dx, dy, dx2, dy2){
        this.updatePorts(canvas, this.startDragX+dx, this.startDragY+dy);
    },
    
    
    updatePorts:function(canvas,x,y){
        // 3.) Check now the common objects
        //
        var figures = canvas.getFigures();
        figures.each(function(i,figure){
            if (figure.isVisible()===true && figure.hitTest(x, y, 50) === true && figure instanceof draw2d.shape.node.Node){
                figure.getPorts().each(function(i,p){
                    p.setVisible(true);
                });
            }
            else{
                figure.getPorts().each(function(i,p){
                    p.setVisible(false);
                });
            }
        });
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.SnapToEditPolicy
 * 
 * A helper used by Tools for snapping certain mouse interactions. 
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.canvas.CanvasPolicy
 */
draw2d.policy.canvas.SnapToEditPolicy = draw2d.policy.canvas.CanvasPolicy.extend({

    NAME : "draw2d.policy.canvas.SnapToEditPolicy",
    
    /**
     * @constructor 
     * Creates a new constraint policy for snap to grid
     * 
     * @param {Number} grid the grid width of the canvas
     */
    init: function( grid){
        this._super();
        this.grid = grid;
    },


    /**
     * @method
     * Adjust the coordinates to the given grid size.
     * 
     * @param figure
     * @param {draw2d.geo.Point} clientPos
     * @returns {draw2d.geo.Point} the contraint position of th efigure
     */
    snap: function(canvas, figure, clientPos){
        return clientPos;
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.SnapToGridEditPolicy
 * 
 * A helper used to perform snapping to a grid, which is specified on the canvas via the various 
 * properties defined in this class. 
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.canvas.SnapToEditPolicy
 */
draw2d.policy.canvas.SnapToGridEditPolicy = draw2d.policy.canvas.SnapToEditPolicy.extend({

    NAME : "draw2d.policy.canvas.SnapToGridEditPolicy",
    
    GRID_COLOR  : "#e0e0f0",
    GRID_WIDTH  : 10,
    
    /**
     * @constructor 
     * Creates a new constraint policy for snap to grid
     * 
     * @param {Number} grid the grid width of the canvas
     */
    init: function( grid){
        this._super();
        this.canvas = null;
        this.lines = null;
        
        if(grid){
            this.grid = grid;
        }
        else{
            this.grid = this.GRID_WIDTH;
        }
    },

    onInstall: function(canvas){
        this.canvas = canvas;
        this.showGrid();
    },
    
    onUninstall: function(canvas){
        this.canvas = null;
        this.lines.remove();
    },
    
    /**
     * @method
     * Applies a snapping correction to the given result. 
     * 
     * @param figure
     * @param {draw2d.geo.Point} pos
     * @returns {draw2d.geo.Point} the contraint position of the figure
     * @since 2.3.0
     */
    snap: function(canvas, figure, pos){
        
        var snapPoint = figure.getSnapToGridAnchor();

        pos.x= pos.x+snapPoint.x;
        pos.y= pos.y+snapPoint.y;

       
        pos.x = this.grid*Math.floor(((pos.x + this.grid/2.0) / this.grid));
        pos.y = this.grid*Math.floor(((pos.y + this.grid/2.0) / this.grid));
        
        pos.x= pos.x-snapPoint.x;
        pos.y= pos.y-snapPoint.y;
        
        return pos;
    },
    
    /**
     * @method
     * paint the grid into the canvas
     * 
     * @private
     * @since 2.3.0
     */
    showGrid: function(){
        // vertical lines
        var w = this.canvas.initialWidth;
        var h = this.canvas.initialHeight;
        this.lines = this.canvas.paper.set();
        
        for(var x = this.grid; x < w; x += this.grid){
            this.lines.push( this.canvas.paper.path( "M " + x + " 0 l 0 " + h));
        }
        // horizontal lines
        for(var y = this.grid; y < h; y += this.grid){
            this.lines.push( this.canvas.paper.path("M 0 " + y + " l " + w + " 0"));
        }
        this.lines.attr({"stroke":this.GRID_COLOR});
        this.lines.toBack();
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.ShowGridEditPolicy
 * 
 * Just to paint a grid in the background. 
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.canvas.CanvasPolicy
 */
draw2d.policy.canvas.ShowGridEditPolicy = draw2d.policy.canvas.CanvasPolicy.extend({

    NAME : "draw2d.policy.canvas.ShowGridEditPolicy",
    
    GRID_COLOR  : "#e0e0f0",
    GRID_WIDTH  : 10,
    
    /**
     * @constructor 
     * Creates a new constraint policy for snap to grid
     * 
     * @param {Number} grid the grid width of the canvas
     */
    init: function( grid){
        this._super();
        this.canvas = null;
        this.lines = null;
        if(grid){
            this.grid = grid;
        }
        else{
            this.grid = this.GRID_WIDTH;
        }
    },

    onInstall: function(canvas){
        this.canvas = canvas;
        this.showGrid();
    },
    
    onUninstall: function(canvas){
        this.canvas = null;
        this.lines.remove();
    },
    

    /**
     * @method
     * paint the grid into the canvas
     * 
     * @private
     * @since 2.3.0
     */
    showGrid: function(){
        // vertical lines
        var w = this.canvas.initialWidth;
        var h = this.canvas.initialHeight;
        this.lines = this.canvas.paper.set();
        
        for(var x = this.grid; x < w; x += this.grid){
            this.lines.push( this.canvas.paper.path( "M " + x + " 0 l 0 " + h));
        }
        // horizontal lines
        for(var y = this.grid; y < h; y += this.grid){
            this.lines.push( this.canvas.paper.path("M 0 " + y + " l " + w + " 0"));
        }
        this.lines.attr({"stroke":this.GRID_COLOR});
        this.lines.toBack();
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.ShowDotEditPolicy
 * 
 * Paint a dotted pattern in the background of the canvas.
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.canvas.CanvasPolicy
 * @since 4.0.1
 */
draw2d.policy.canvas.ShowDotEditPolicy = draw2d.policy.canvas.CanvasPolicy.extend({

    NAME : "draw2d.policy.canvas.ShowDotEditPolicy",
    
    DOT_COLOR  : "#999999",
    DOT_RADIUS  : 1,
    DOT_DISTANCE : 20,
    
    /**
     * @constructor 
     * Creates a new constraint policy for snap to grid
     * 
     * @param {Number} [dotDistance] the distance or grid width between the dots.
     * @param {Number} [dotRadius] the radius of the dots.
     * @param {draw2d.util.Color|String} [dotColor] the color for the dots.
     */
    init: function( dotDistance, dotRadius, dotColor){
        this._super();
        this.canvas = null;

        this.dotDistance = dotDistance ? dotDistance : this.DOT_DISTANCE;
        this.dotRadius = dotRadius ? dotRadius : this.DOT_RADIUS;
        this.dotColor = new draw2d.util.Color(dotColor ? dotColor : this.DOT_COLOR);
               
        // generate the background pattern with an data URL GIF image. This is much faster than draw
        // the pattern via the canvas and the raphael.circle method
        //
        var mypixels = Array(this.dotDistance*this.dotDistance);
        // set the pixel at the coordinate [0,0] as opaque.       
        mypixels[0] = 1;
        this.imageDataURL = this.createMonochromGif(this.dotDistance, this.dotDistance, mypixels, this.dotColor);
    },

    onInstall: function(canvas){
        this.canvas = canvas;
        this.oldBg =  this.canvas.html.css("background-image");
        $(canvas.paper.canvas).css({"background-image": "url('"+this.imageDataURL+"')"});
    },
    
    onUninstall: function(canvas){
        this.canvas = null;
        $(canvas.paper.canvas).css({"background-image": this.oldBg});
    }
    

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.canvas.ShowChessboardEditPolicy
 * 
 * Just to paint a grid in the background. 
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.canvas.CanvasPolicy
 */
draw2d.policy.canvas.ShowChessboardEditPolicy = draw2d.policy.canvas.CanvasPolicy.extend({

    NAME : "draw2d.policy.canvas.ShowChessboardEditPolicy",
    
    GRID_COLOR  : "#e0e0e0",
    GRID_WIDTH  :20,
    
    /**
     * @constructor 
     * Creates a new constraint policy for snap to grid
     * 
     * @param {Number} grid the grid width of the canvas
     */
    init: function( grid){
        this._super();
        this.canvas = null;
        this.cells  = null;
        if(grid){
            this.grid = grid;
        }
        else{
            this.grid = this.GRID_WIDTH;
        }
    },

    onInstall: function(canvas){
        this.canvas = canvas;
        this.showGrid();
    },
    
    onUninstall: function(canvas){
        this.canvas = null;
        this.cells.remove();
    },
    

    /**
     * @method
     * paint the grid into the canvas
     * 
     * @private
     * @since 2.3.0
     */
    showGrid: function(){
        // vertical lines
        var w = this.canvas.initialWidth;
        var h = this.canvas.initialHeight;
        this.cells = this.canvas.paper.set();
        
        var even = false;
        var xEven = even;
        for(var x = 0; x < w; x += this.grid) {
           for(var y = 0; y < h; y+= this.grid) {
               if(even) {
                   var crect = this.canvas.paper.rect(x, y, this.grid, this.grid);
                   crect.attr({fill: this.GRID_COLOR, "stroke-width":0});
                   this.cells.push(crect);
               }
               even = !even;
           }
           xEven = !xEven;
           even = xEven;
       }
        
       this.cells.toBack();
    }
  
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
draw2d.SnapToHelper = {};

draw2d.SnapToHelper.NORTH =  1;
draw2d.SnapToHelper.SOUTH =  4;
draw2d.SnapToHelper.WEST  =  8;
draw2d.SnapToHelper.EAST  = 16;
draw2d.SnapToHelper.CENTER= 32;

draw2d.SnapToHelper.NORTH_EAST  = draw2d.SnapToHelper.NORTH | draw2d.SnapToHelper.EAST;
draw2d.SnapToHelper.NORTH_WEST  = draw2d.SnapToHelper.NORTH | draw2d.SnapToHelper.WEST;
draw2d.SnapToHelper.SOUTH_EAST  = draw2d.SnapToHelper.SOUTH | draw2d.SnapToHelper.EAST;
draw2d.SnapToHelper.SOUTH_WEST  = draw2d.SnapToHelper.SOUTH | draw2d.SnapToHelper.WEST;
draw2d.SnapToHelper.NORTH_SOUTH = draw2d.SnapToHelper.NORTH | draw2d.SnapToHelper.SOUTH;
draw2d.SnapToHelper.EAST_WEST   = draw2d.SnapToHelper.EAST | draw2d.SnapToHelper.WEST;
draw2d.SnapToHelper.NSEW        = draw2d.SnapToHelper.NORTH_SOUTH | draw2d.SnapToHelper.EAST_WEST;

/**
 * @class draw2d.policy.canvas.SnapToGeometryEditPolicy
 * 
 * Snapping is based on the existing children of a container. When snapping a shape, 
 * the edges of the bounding box will snap to edges of other rectangles generated 
 * from the children of the given canvas. 
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.canvas.SnapToEditPolicy
 */
draw2d.policy.canvas.SnapToGeometryEditPolicy = draw2d.policy.canvas.SnapToEditPolicy.extend({

    NAME : "draw2d.policy.canvas.SnapToGeometryEditPolicy",
    
    SNAP_THRESHOLD   : 3,
    LINE_COLOR       : "#1387E6",
    FADEOUT_DURATION : 300,
    
    /**
     * @constructor 
     * Creates a new constraint policy for snap to grid
     * 
     * @param {Number} grid the grid width of the canvas
     */
    init: function( ){
        this._super();
        
        this.rows=null;
        this.cols=null;
        this.vline = null;
        this.hline = null;
        this.canvas = null;
    },

    onInstall: function(canvas){
        this.canvas = canvas;
    },
    
    onUninstall: function(canvas){
        this.canvas = null;
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Canvas} canvas
     * @param {Number} x the x-coordinate of the mouse down event
     * @param {Number} y the y-coordinate of the mouse down event
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @template
     */
    onMouseUp: function(figure, x, y, shiftKey, ctrlKey){
        this.rows=null;
        this.cols=null;
        this.hideVerticalLine();
        this.hideHorizontalLine();
    },
    
    /**
     * @method
     * Adjust the coordinates to the canvas neighbours
     * 
     * @param figure
     * @param {draw2d.geo.Point} pos
     * @returns {draw2d.geo.Point} the contraint position of th efigure
     */
    snap: function(canvas, figure, pos){
        
        if(figure instanceof draw2d.ResizeHandle)
        {
           var snapPoint = figure.getSnapToGridAnchor();
           pos.x+= snapPoint.x;
           pos.y+= snapPoint.y;
           var result = new draw2d.geo.Point(pos.x,pos.y);

           var snapDirections = figure.getSnapToDirection();
           var direction = this.snapPoint(snapDirections, pos,result);

           // Show a vertical line if the snapper has modified the inputPoint
           //
           if((snapDirections & draw2d.SnapToHelper.EAST_WEST) && !(direction & draw2d.SnapToHelper.EAST_WEST))
              this.showVerticalLine(result.x);
           else
              this.hideVerticalLine();

           // Show a horizontal line if the snapper has modified the inputPoint
           //
           if((snapDirections & draw2d.SnapToHelper.NORTH_SOUTH) && !(direction & draw2d.SnapToHelper.NORTH_SOUTH))
              this.showHorizontalLine(result.y);
           else
              this.hideHorizontalLine();

           result.x-= snapPoint.x;
           result.y-= snapPoint.y;
           return result;
        }

        // The user drag&drop a normal figure
        var inputBounds = new draw2d.geo.Rectangle(pos.x,pos.y, figure.getWidth(), figure.getHeight());
        var result = new draw2d.geo.Rectangle(pos.x,pos.y, figure.getWidth(), figure.getHeight());

        var snapDirections = draw2d.SnapToHelper.NSEW;
        var direction = this.snapRectangle( inputBounds, result);

        // Show a vertical line if the snapper has modified the inputPoint
        //
        if((snapDirections & draw2d.SnapToHelper.WEST) && !(direction & draw2d.SnapToHelper.WEST))
           this.showVerticalLine(result.x);
        else if((snapDirections & draw2d.SnapToHelper.EAST) && !(direction & draw2d.SnapToHelper.EAST))
           this.showVerticalLine(result.getX()+result.getWidth());
        else
           this.hideVerticalLine();


        // Show a horizontal line if the snapper has modified the inputPoint
        //
        if((snapDirections & draw2d.SnapToHelper.NORTH) && !(direction & draw2d.SnapToHelper.NORTH))
           this.showHorizontalLine(result.y);
        else if((snapDirections & draw2d.SnapToHelper.SOUTH) && !(direction & draw2d.SnapToHelper.SOUTH))
           this.showHorizontalLine(result.getY()+result.getHeight());
        else
           this.hideHorizontalLine();

        return result.getTopLeft();
    },
    
    
    snapRectangle:function( /*:draw2d.Dimension*/ inputBounds,  /*:draw2d.Dimension*/ resultBounds)
    {
        var topLeftResult     = inputBounds.getTopLeft();
        var bottomRightResult = inputBounds.getBottomRight();

        var snapDirectionsTopLeft = this.snapPoint(draw2d.SnapToHelper.NORTH_WEST, inputBounds.getTopLeft(), topLeftResult);
        resultBounds.x = topLeftResult.x;
        resultBounds.y = topLeftResult.y;

        var snapDirectionsBottomRight = this.snapPoint(draw2d.SnapToHelper.SOUTH_EAST, inputBounds.getBottomRight(), bottomRightResult);
        // the first test (topLeft) has not modified the point. so we can modify them with the bottomRight adjustment
        //
        if(snapDirectionsTopLeft & draw2d.SnapToHelper.WEST)
          resultBounds.x = bottomRightResult.x-inputBounds.getWidth();

        // the first test (topLeft) has not modified the point. so we can modify them with the bottomRight adjustment
        //
        if(snapDirectionsTopLeft & draw2d.SnapToHelper.NORTH)
           resultBounds.y = bottomRightResult.y-inputBounds.getHeight();


        return snapDirectionsTopLeft |snapDirectionsBottomRight;
    },
    
    snapPoint:function(/*:int*/ snapOrientation, /*:draw2d.Point*/ inputPoint,  /*:draw2d.Point*/ resultPoint)
    {
       if(this.rows===null || this.cols===null)
         this.populateRowsAndCols();

       if ((snapOrientation & draw2d.SnapToHelper.EAST) !== 0) 
       {
          var rightCorrection = this.getCorrectionFor(this.cols, inputPoint.getX() -1, 1);
          if (rightCorrection !== this.SNAP_THRESHOLD) 
          {
             snapOrientation &= ~draw2d.SnapToHelper.EAST;
             resultPoint.x += rightCorrection;
          }
       }

       if ((snapOrientation & draw2d.SnapToHelper.WEST) !== 0) 
       {
          var leftCorrection = this.getCorrectionFor(this.cols, inputPoint.getX(), -1);
          if (leftCorrection !== this.SNAP_THRESHOLD) 
          {
             snapOrientation &= ~draw2d.SnapToHelper.WEST;
             resultPoint.x += leftCorrection;
          }
       }

       if ((snapOrientation & draw2d.SnapToHelper.SOUTH) !== 0) 
       {
          var bottomCorrection = this.getCorrectionFor(this.rows,  inputPoint.getY() - 1, 1);
          if (bottomCorrection !== this.SNAP_THRESHOLD) 
          {
             snapOrientation &= ~draw2d.SnapToHelper.SOUTH;
             resultPoint.y += bottomCorrection;
          }
       }

       if ((snapOrientation & draw2d.SnapToHelper.NORTH) !== 0) 
       {
          var topCorrection = this.getCorrectionFor(this.rows, inputPoint.getY(), -1);
          if (topCorrection !== this.SNAP_THRESHOLD) 
          {
             snapOrientation &= ~draw2d.SnapToHelper.NORTH;
             resultPoint.y += topCorrection;
          }
       }

      return snapOrientation;
    },
    
    populateRowsAndCols:function()
    {
       var selection = this.canvas.getSelection();
       this.rows = [];
       this.cols = [];
       
       var figures = this.canvas.getFigures();
       var index =0;
       for (var i = 0; i < figures.getSize();i++ )
       {
          var figure = figures.get(i);
          if(!selection.contains(figure))
          {
             var bounds = figure.getBoundingBox();
             this.cols[index * 3]     = {type:-1, location: bounds.getX()};
             this.rows[index * 3]     = {type:-1, location: bounds.getY()};
             this.cols[index * 3 + 1] = {type:0 , location: bounds.x + (bounds.getWidth() - 1) / 2};
             this.rows[index * 3 + 1] = {type:0 , location: bounds.y + (bounds.getHeight() - 1) / 2};
             this.cols[index * 3 + 2] = {type:1 , location: bounds.getRight() - 1};
             this.rows[index * 3 + 2] = {type:1 , location: bounds.getBottom() - 1};
             index++;
         }
       }
    },

    getCorrectionFor:function(/*:Array*/ entries, /*:double*/ value, /*:int*/ side) 
    {
       var resultMag = this.SNAP_THRESHOLD;
       var result = this.SNAP_THRESHOLD;

       for (var i = 0; i < entries.length; i++) 
       {
          var entry = entries[i];
          var magnitude;

          if (entry.type === -1 && side !== 0) 
          {
             magnitude = Math.abs(value - entry.location);
             if (magnitude < resultMag)
             {
                   resultMag = magnitude;
                   result = entry.location - value;
             }
          }
          else if (entry.type === 0 && side === 0) 
          {
             magnitude = Math.abs(value - entry.location);
             if (magnitude < resultMag)
             {
                resultMag = magnitude;
                result = entry.location - value;
             }
          }
          else if (entry.type === 1 && side !== 0) 
          {
             magnitude = Math.abs(value - entry.location);
             if (magnitude < resultMag)
             {
                resultMag = magnitude;
                result = entry.location - value;
             }
          }
       }
       return result;
    },
    
    showVerticalLine:function(x){
        if(this.vline!=null){
            return; //silently
        }
        this.vline = this.canvas.paper
                        .path("M " + x + " 0 l 0 " + this.canvas.getHeight())
                        .attr({"stroke":this.LINE_COLOR,"stroke-width":1});
    },
    
    hideVerticalLine:function(){
        if(this.vline==null){
            return;
        }
        var tmp = this.vline;
        tmp.animate({
            opacity: 0.1
        }, this.FADEOUT_DURATION,function(){
            tmp.remove();
        });
        
        this.vline = null;
    },
    
    showHorizontalLine:function(y){
        if(this.hline!=null){
            return;
        }
        
        this.hline = this.canvas.paper
                      .path("M 0 " + y + " l " + this.canvas.getWidth() + " 0")
                      .attr({"stroke":this.LINE_COLOR,"stroke-width":1});
    },

    hideHorizontalLine:function(){
        if(this.hline==null){
            return; //silently
        }
        var tmp = this.hline;
        tmp.animate({
            opacity: 0.1
        }, this.FADEOUT_DURATION,function(){
            tmp.remove();
        });
        this.hline = null;
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.DragDropEditPolicy
 * 
 * Called by the framework if the user edit the position of a figure with a drag drop operation.
 * Sub class like SelectionEditPolicy or RegionEditPolicy cam adjust th e position of the figure or the selections handles.
 * 
 * @author  Andreas Herz
 * @extends draw2d.policy.EditPolicy
 */
draw2d.policy.figure.DragDropEditPolicy = draw2d.policy.EditPolicy.extend({

    NAME : "draw2d.policy.figure.DragDropEditPolicy",

    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    
   
    /**
     * @method
     * Called by the framework if the related shape has init a drag&drop
     * operation
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDragStart: function(canvas, figure){
    	figure.shape.attr({cursor:"move"});
    	figure.isMoving = false;
    	figure.originalAlpha = figure.getAlpha();
    },
    
    /**
     * @method
     * Called by the framework during drag a figure.
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDrag: function(canvas, figure){
        
        // enable the alpha blending of the first real move of the object
        //
        if(figure.isMoving===false)
        {
            figure.isMoving = true;
            figure.setAlpha(figure.originalAlpha*0.4);
        }    	
    },
    
    /**
     * @method
     * Called by the framework if the drag drop operation ends.
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDragEnd: function(canvas, figure){
        figure.shape.attr({cursor:"default"});
        figure.isMoving = false;
    },
    
    /**
     * @method
     * Adjust the coordinates to the rectangle/region of this constraint.
     * 
     * @param figure
     * @param {Number|draw2d.geo.Point} x
     * @param {number} [y]
     * @returns {draw2d.geo.Point} the constraint position of the figure
     * 
     * @template
     */
    adjustPosition: function(figure, x,y){
        // do nothing per default implementation
        if(x instanceof draw2d.geo.Point){
            return x;
        }
        
        return new draw2d.geo.Point(x,y);
    },

    /**
     * @method
     * ensure that the dimension didn't goes outside the given restrictions
     * 
     * @param figure
     * @param {Number} w
     * @param {number} h
     * @returns {draw2d.geo.Rectangle} the constraint position of the figure
     */
    adjustDimension : function(figure, w, h){
        return new draw2d.geo.Rectangle(0,0,w,h);
    },
    
    /**
     * @method
     * Callback if the figure has been moved
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * 
     * @template
     */
    moved: function(canvas,figure) {

    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.RegionConstraintPolicy
 * 
 * An EditPolicy for use with Figures. The constraint for RegionContraintPolicy is a Rectangle. It is
 * not possible to move the related figure outside this contrained area.
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.figure.RegionEditPolicy = draw2d.policy.figure.DragDropEditPolicy.extend({

    NAME : "draw2d.policy.figure.RegionEditPolicy",
    
    /**
     * @constructor 
     * Creates a new constraint object
     * 
     * @param {Number|draw2d.geo.Rectangle} x x coordinate or a rectangle as constraint for the assigned figure.
     * @param {Number} y
     * @param {Number} w
     * @param {Number} h
     */
    init: function( x,y,w,h){
        this._super();
        if(x instanceof draw2d.geo.Rectangle){
            this.constRect = x;
        }
        else if(typeof h === "number"){
            this.constRect = new draw2d.geo.Rectangle(x,y,w,h);
        }
        else{
            throw "Invalid parameter. RegionConstraintPolicy need a rectangle as parameter in the constructor";
        }
    },


    /**
     * @method
     * Adjust the coordinates to the rectangle/region of this contraint.
     * 
     * @param figure
     * @param {Number|draw2d.geo.Point} x
     * @param {number} [y]
     * @returns {draw2d.geo.Point} the contraint position of th efigure
     */
    adjustPosition : function(figure, x, y)
    {
        var r = null;
        if (x instanceof draw2d.geo.Point) {
            r = new draw2d.geo.Rectangle(x.x, x.y, figure.getWidth(), figure.getHeight());
        }
        else {
            r = new draw2d.geo.Rectangle(x, y, figure.getWidth(), figure.getHeight());
        }
        r = this.constRect.moveInside(r);
        return r.getTopLeft();
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.HorizontalEditPolicy
 * 
 * An EditPolicy for use with Figures. The constraint for RegionContraintPolicy is a Rectangle. It is
 * not possible to move the related figure outside this constrained area.
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.figure.HorizontalEditPolicy = draw2d.policy.figure.DragDropEditPolicy.extend({
    
    NAME : "draw2d.policy.figure.HorizontalEditPolicy",

    /**
     * @constructor 
     * Creates a new constraint object
     * 
     */
    init: function(){
        this._super();
    },


    /**
     * @method
     * It is only possible to drag&drop the element in a horizontal line
     * 
     * @param figure
     * @param {Number|draw2d.geo.Point} x
     * @param {number} [y]
     * @returns {draw2d.geo.Point} the constraint position of the figure
     */
    adjustPosition : function(figure, x, y)
    {
        return new draw2d.geo.Point(x,figure.getY());
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.VerticalEditPolicy
 * 
 * An EditPolicy for use with Figures. The constraint for RegionContraintPolicy is a Rectangle. It is
 * not possible to move the related figure outside this contrained area.
 * 
 * 
 * @author Andreas Herz
 * 
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.figure.VerticalEditPolicy = draw2d.policy.figure.DragDropEditPolicy.extend({

    NAME : "draw2d.policy.figure.VerticalEditPolicy",

    /**
     * @constructor 
     * Creates a new constraint object
     */
    init: function()
    {
        this._super();
    },


    /**
     * @method
     * It is only possible to drag&drop the element in a vertical line
     * 
     * @param figure
     * @param {Number|draw2d.geo.Point} x
     * @param {number} [y]
     * @returns {draw2d.geo.Point} the constraint position of the figure
     */
    adjustPosition : function(figure, x, y)
    {
        return new draw2d.geo.Point(figure.getX(),y);
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.SelectionFeedbackPolicy
 * 
 * A draw2d.policy.SelectionFeedbackPolicy that is sensitive to the canvas selection. Subclasses will typically 
 * decorate the {@link draw2d.Figure figure} with things like selection handles and/or focus feedback.
 * <br>
 * If you want to change the handle visibility for a figure, then you should use SelectionFeedbackPolicy to do that.
 * 
 * @author Andreas Herz
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.figure.SelectionFeedbackPolicy = draw2d.policy.figure.DragDropEditPolicy.extend({

    NAME : "draw2d.policy.figure.SelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * 
     * @template
     * @param figure
     * @param isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
    },
    
    
    /**
     * @method
     * 
     * @param {draw2d.Figure} figure the unselected figure
     */
    onUnselect: function(canvas, figure ){
      
        figure.selectionHandles.each(function(i,e){
            e.hide();
        });
        figure.selectionHandles = new draw2d.util.ArrayList();
    },
    
    /**
     * @method
     * Called by the host if the policy has been installed.
     * 
     * @param {draw2d.Canvas/draw2d.Figure} figure
     */
    onInstall: function( figure){
        this._super(figure);
        
        var canvas = figure.getCanvas();
        if(canvas!==null){
            if(canvas.getSelection().contains(figure)){
                this.onSelect(canvas, figure, true);
            }
        }
    },
    
    
    /**
     * @method
     * Called by the host if the policy has been uninstalled.
     * 
     * @param {draw2d.Canvas/draw2d.Figure} figure
     */
    onUninstall: function( figure){
        this._super(figure);

        if(typeof figure.selectionHandles ==="undefined"){
            return;
        }
        
        figure.selectionHandles.each(function(i,e){
            e.hide();
        });
        figure.selectionHandles = new draw2d.util.ArrayList();
    }
        
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.ResizeSelectionFeedbackPolicy
 * 
 * Selection feedback policy without "marching ant lines" or any other rectangle highlight. Just
 * some resize handles at each corner of the shape.
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.ResizeSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *
 * @author Andreas Herz
 * @since 4.0.0
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 * 
 */
draw2d.policy.figure.ResizeSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.ResizeSelectionFeedbackPolicy",
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
   },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        
        if(figure.selectionHandles.isEmpty())
        {
            // create standard Resize handles for the figure
            //
            var r1= new draw2d.ResizeHandle(figure,1); // 1 = LEFT TOP
            var r2= new draw2d.ResizeHandle(figure,2); // 2 = CENTER_TOP
            var r3= new draw2d.ResizeHandle(figure,3); // 3 = RIGHT_TOP
            var r4= new draw2d.ResizeHandle(figure,4); // 4 = RIGHT_MIDDLE
            var r5= new draw2d.ResizeHandle(figure,5); // 5 = RIGHT_BOTTOM
            var r6= new draw2d.ResizeHandle(figure,6); // 6 = CENTER_BOTTOM
            var r7= new draw2d.ResizeHandle(figure,7); // 7 = LEFT_BOTTOM
            var r8= new draw2d.ResizeHandle(figure,8); // 8 = LEFT_MIDDLE

            // and add them to the figure. We need the reference to the ResizeHandles
            // to remove the resize handles if the figure will be unselect. Just a simple
            // refrence store
            //
            figure.selectionHandles.add(r1);
            figure.selectionHandles.add(r2);
            figure.selectionHandles.add(r3);
            figure.selectionHandles.add(r4);
            figure.selectionHandles.add(r5);
            figure.selectionHandles.add(r6);
            figure.selectionHandles.add(r7);
            figure.selectionHandles.add(r8);
            
            // show the default top/left, top/right, bottom/right and bottom/left
            // resize handles 
            //
            r1.show(canvas);
            r3.show(canvas);
            r5.show(canvas);
            r7.show(canvas);

            // The corner ResizeHandles are only draggable fi the figure is
            // resizeable. But the Resize handles are visible
            //
            
            // change the look&feel of the corner resizehandles if the
            // figure isn't resizeable
            //
            if(figure.isResizeable()===false) {
              r1.setBackgroundColor(null);
              r3.setBackgroundColor(null);
              r5.setBackgroundColor(null);
              r7.setBackgroundColor(null);
              r1.setDraggable(false);
              r3.setDraggable(false);
              r5.setDraggable(false);
              r7.setDraggable(false);
            }

            // show only the additional resizehandles if the figure is resizeable
            //
            if((!figure.getKeepAspectRatio()) && figure.isResizeable()){
              r2.show(canvas);
              r4.show(canvas);
              r6.show(canvas);
              r8.show(canvas);
            }
        }
        this.moved(canvas, figure);
   },
  
    /**
     * @method
     * Callback if the figure has been moved. In this case we must update the position of the
     * resize handles.
     * 
     * @param figure
     * 
     * @template
     */
    moved: function(canvas, figure ){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        
        var objHeight   = figure.getHeight();
        var objWidth    = figure.getWidth();
        var xPos = figure.getX();
        var yPos = figure.getY();
        
        var r1= figure.selectionHandles.get(0);
        var r3= figure.selectionHandles.get(2);
        var r5= figure.selectionHandles.get(4);
        var r7= figure.selectionHandles.get(6); 
        r1.setPosition(xPos-r1.getWidth(),yPos-r1.getHeight());
        r3.setPosition(xPos+objWidth,yPos-r3.getHeight());
        r5.setPosition(xPos+objWidth,yPos+objHeight);
        r7.setPosition(xPos-r7.getWidth(),yPos+objHeight);
        
        if(!figure.getKeepAspectRatio())
        {
            var r2= figure.selectionHandles.get(1); 
            var r4= figure.selectionHandles.get(3); 
            var r6= figure.selectionHandles.get(5); 
            var r8= figure.selectionHandles.get(7); 
     
            r2.setPosition(xPos+(objWidth/2)-(r2.getWidth()/2),yPos-r2.getHeight());
            r4.setPosition(xPos+objWidth,yPos+(objHeight/2)-(r4.getHeight()/2));
            r6.setPosition(xPos+(objWidth/2)-(r6.getWidth()/2),yPos+objHeight);
            r8.setPosition(xPos-r8.getWidth(),yPos+(objHeight/2)-(r8.getHeight()/2));
        }
    }
    
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.RectangleSelectionFeedbackPolicy 
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.RectangleSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.RectangleSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.RectangleSelectionFeedbackPolicy",
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
   },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        
        if(figure.selectionHandles.isEmpty())
        {
            // Add a dotted line rectangle to the figure. Override the show/hide method of the standard
            // figure to avoid adding these element to the hit test of the canvas. In this case the element
            // is just visible but not part of the model or responsible for any drag/drop operation
            //
            var box = new draw2d.shape.basic.Rectangle();
            box.setBackgroundColor(null);
            box.setDashArray("- ");
            box.setColor("#2096fc");
            box.setStroke(0.5);
            box.hide= function(){
                // IMPORTANT
                // don't add/remove this rectangle to the canvas resizeHandles. This rect isn't responsible for any hitTest or
                // dragDrop operation
                //canvas.resizeHandles.remove(box);
                box.setCanvas(null);
            };
            box.show= function(canvas){
                box.setCanvas(canvas);
                // IMPORTANT
                // don't add/remove this rectangle to the canvas resizeHandles. This rect isn't responsible for any hitTest or
                // dragDrop operation
                //canvas.resizeHandles.remove(box);
                //canvas.resizeHandles.add(box);
                box.shape.toFront();
            };
            box.show(canvas);

            // create standard Resize handles for the figure
            //
            var r1= new draw2d.ResizeHandle(figure,1); // 1 = LEFT TOP
            var r2= new draw2d.ResizeHandle(figure,2); // 2 = CENTER_TOP
            var r3= new draw2d.ResizeHandle(figure,3); // 3 = RIGHT_TOP
            var r4= new draw2d.ResizeHandle(figure,4); // 4 = RIGHT_MIDDLE
            var r5= new draw2d.ResizeHandle(figure,5); // 5 = RIGHT_BOTTOM
            var r6= new draw2d.ResizeHandle(figure,6); // 6 = CENTER_BOTTOM
            var r7= new draw2d.ResizeHandle(figure,7); // 7 = LEFT_BOTTOM
            var r8= new draw2d.ResizeHandle(figure,8); // 8 = LEFT_MIDDLE

            // and add them to the figure. We need the reference to the ResizeHandles
            // to remove the resize handles if the figure will be unselect. Just a simple
            // reference store
            //
            figure.selectionHandles.add(r1);
            figure.selectionHandles.add(r2);
            figure.selectionHandles.add(r3);
            figure.selectionHandles.add(r4);
            figure.selectionHandles.add(r5);
            figure.selectionHandles.add(r6);
            figure.selectionHandles.add(r7);
            figure.selectionHandles.add(r8);
            
            // show the default top/left, top/right, bottom/right and bottom/left
            // resize handles 
            //
            r1.show(canvas);
            r3.show(canvas);
            r5.show(canvas);
            r7.show(canvas);

            
            // change the look&feel of the corner resizehandles if the
            // figure isn't resizeable
            //
            if(figure.isResizeable()===false) {
              r1.setBackgroundColor(null);
              r3.setBackgroundColor(null);
              r5.setBackgroundColor(null);
              r7.setBackgroundColor(null);
              r1.setDraggable(false);
              r3.setDraggable(false);
              r5.setDraggable(false);
              r7.setDraggable(false);
            }

            // show only the additional resizehandles if the figure is resizeable and didn't care about
            // the aspect ration
            //
            if((!figure.getKeepAspectRatio()) && figure.isResizeable())
            {
              r2.show(canvas);
              r4.show(canvas);
              r6.show(canvas);
              r8.show(canvas);
            }

            // add the reference of the "ant box" to the figure as well. But wee add them
            // to the end of the array because inherit classes expect the resizehandles
            // on index 0-7.
            //
            figure.selectionHandles.add(box);
        }
        this.moved(canvas, figure);
   },
  
    /**
     * @method
     * Callback if the figure has been moved. In this case we must update the position of the
     * resize handles and the "ant" box.
     * 
     * @param figure
     * 
     * @template
     */
    moved: function(canvas, figure ){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        
        var objHeight   = figure.getHeight();
        var objWidth    = figure.getWidth();
        var xPos = figure.getX();
        var yPos = figure.getY();
        
        var r1= figure.selectionHandles.get(0);
        var r3= figure.selectionHandles.get(2);
        var r5= figure.selectionHandles.get(4);
        var r7= figure.selectionHandles.get(6); 
        r1.setPosition(xPos-r1.getWidth(),yPos-r1.getHeight());
        r3.setPosition(xPos+objWidth,yPos-r3.getHeight());
        r5.setPosition(xPos+objWidth,yPos+objHeight);
        r7.setPosition(xPos-r7.getWidth(),yPos+objHeight);
        
        if(!figure.getKeepAspectRatio())
        {
            var r2= figure.selectionHandles.get(1); 
            var r4= figure.selectionHandles.get(3); 
            var r6= figure.selectionHandles.get(5); 
            var r8= figure.selectionHandles.get(7); 
     
            r2.setPosition(xPos+(objWidth/2)-(r2.getWidth()/2),yPos-r2.getHeight());
            r4.setPosition(xPos+objWidth,yPos+(objHeight/2)-(r4.getHeight()/2));
            r6.setPosition(xPos+(objWidth/2)-(r6.getWidth()/2),yPos+objHeight);
            r8.setPosition(xPos-r8.getWidth(),yPos+(objHeight/2)-(r8.getHeight()/2));
        }
        var box= figure.selectionHandles.get(8); 
        box.setPosition(figure.getPosition().translate(-2,-2));
        box.setDimension(figure.getWidth()+4, figure.getHeight()+4);
        box.setRotationAngle(figure.getRotationAngle());
    }
    
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.BigRectangleSelectionFeedbackPolicy 
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.figure.BigRectangleSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.BigRectangleSelectionFeedbackPolicy = draw2d.policy.figure.RectangleSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.BigRectangleSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
   },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        
        this._super(canvas, figure, isPrimarySelection);
        
        if(!figure.selectionHandles.isEmpty())
        {
            figure.selectionHandles.each(function(i,e){
               e.setDimension(15,15); 
            });
        }
        this.moved(canvas, figure);
   }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.RoundRectangleSelectionFeedbackPolicy 
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.RoundRectangleSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.RectangleSelectionFeedbackPolicy
 */
draw2d.policy.figure.RoundRectangleSelectionFeedbackPolicy = draw2d.policy.figure.RectangleSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.RoundRectangleSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
   },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas,figure, isPrimarySelection){
        
        this._super(canvas,figure, isPrimarySelection);
        
        if(!figure.selectionHandles.isEmpty())
        {
            figure.selectionHandles.each(function(i,e){
               e.setDimension(12,12);
               e.setRadius(4);
            });
        }
        this.moved(canvas,figure);
   }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.BusSelectionFeedbackPolicy 
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.BusSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.BusSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        if (figure.selectionHandles.isEmpty()) {
            var r2 = new draw2d.ResizeHandle(figure, 2); // 2 = CENTER_TOP
            var r4 = new draw2d.ResizeHandle(figure, 4); // 4 = RIGHT_MIDDLE
            var r6 = new draw2d.ResizeHandle(figure, 6); // 6 = CENTER_BOTTOM
            var r8 = new draw2d.ResizeHandle(figure, 8); // 8 = LEFT_MIDDLE

            figure.selectionHandles.add(r2);
            figure.selectionHandles.add(r4);
            figure.selectionHandles.add(r6);
            figure.selectionHandles.add(r8);

            r2.setDraggable(figure.isResizeable());
            r4.setDraggable(figure.isResizeable());
            r6.setDraggable(figure.isResizeable());
            r8.setDraggable(figure.isResizeable());
            
            r2.show(canvas);
            r4.show(canvas);
            r6.show(canvas);
            r8.show(canvas);
        }
        this.moved(canvas, figure);
   },
    
    
    /**
     * @method
     * Callback if the figure has been moved
     * 
     * @param figure
     * 
     * @template
     */
    moved: function(canvas, figure){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        var r2= figure.selectionHandles.get(0); 
        var r4= figure.selectionHandles.get(1); 
        var r6= figure.selectionHandles.get(2); 
        var r8= figure.selectionHandles.get(3); 

        var objHeight   = figure.getHeight();
        var objWidth    = figure.getWidth();
        
        var xPos = figure.getX();
        var yPos = figure.getY();
        r2.setPosition(xPos+(objWidth/2)-(r2.getWidth()/2),yPos-r2.getHeight());
        r4.setPosition(xPos+objWidth,yPos+(objHeight/2)-(r4.getHeight()/2));
        r6.setPosition(xPos+(objWidth/2)-(r6.getWidth()/2),yPos+objHeight);
        r8.setPosition(xPos-r8.getWidth(),yPos+(objHeight/2)-(r8.getHeight()/2));
     }
    
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.WidthSelectionFeedbackPolicy
 * This selection shows only selection handles for the width. It is only possible to chang ethe width
 * of an shaped. The height stays always the same or is recalculated by the figure itself. 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.WidthSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.BusSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        if (figure.selectionHandles.isEmpty()) {
            var r4 = new draw2d.ResizeHandle(figure, 4); // 4 = RIGHT_MIDDLE
            var r8 = new draw2d.ResizeHandle(figure, 8); // 8 = LEFT_MIDDLE

            r4.installEditPolicy(new draw2d.policy.figure.HorizontalEditPolicy());
            r8.installEditPolicy(new draw2d.policy.figure.HorizontalEditPolicy());
            figure.selectionHandles.add(r4);
            figure.selectionHandles.add(r8);

            r4.setDraggable(figure.isResizeable());
            r8.setDraggable(figure.isResizeable());
            
            r4.show(canvas);
            r8.show(canvas);
        }
        this.moved(canvas, figure);
   },
    
    
    /**
     * @method
     * Callback if the figure has been moved
     * 
     * @param figure
     * 
     * @template
     */
    moved: function(canvas, figure){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        var r4= figure.selectionHandles.first(); 
        var r8= figure.selectionHandles.last(); 

        var objWidth    = figure.getWidth();
        
        var xPos = figure.getX();
        var yPos = figure.getY();
        r4.setDimension(r4.getWidth(), figure.getHeight());
        r8.setDimension(r8.getWidth(), figure.getHeight());
        r4.setPosition(xPos+objWidth     , yPos);
        r8.setPosition(xPos-r8.getWidth(), yPos);
     }
    
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.VBusSelectionFeedbackPolicy
 *  
 * Selection feedback policy for vertical bus figures. 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.BusSelectionFeedbackPolicy
 */
draw2d.policy.figure.VBusSelectionFeedbackPolicy = draw2d.policy.figure.BusSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.VBusSelectionFeedbackPolicy",
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * Callback if the figure has been moved
     * 
     * @param figure
     * 
     **/
    moved: function(canvas,figure){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        var r2= figure.selectionHandles.get(0); 
        var r6= figure.selectionHandles.get(2); 
        var objWidth    = figure.getWidth();
        // adjust the resize handles on the left/right to the new dimension of the shape
        //
        r2.setDimension(objWidth, r2.getHeight());
        r6.setDimension(objWidth, r6.getHeight());
        
        this._super(canvas,figure);
     }
    
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.HBusSelectionFeedbackPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.BusSelectionFeedbackPolicy
 */
draw2d.policy.figure.HBusSelectionFeedbackPolicy = draw2d.policy.figure.BusSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.HBusSelectionFeedbackPolicy",
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    
    /**
     * @method
     * Callback if the figure has been moved
     * 
     * @param figure
     * 
     * @template
     */
    moved: function(canvas, figure){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        var r4= figure.selectionHandles.get(1); 
        var r8= figure.selectionHandles.get(3); 

        r4.setDimension(r4.getWidth(), figure.getHeight());
        r8.setDimension(r4.getWidth(), figure.getHeight());
        
        this._super(canvas,figure);
     }
    
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.AntSelectionFeedbackPolicy 
 * 
 * Provide support for selecting and positioning a non-resizable figure. 
 * Selection is indicated via rectangular handle that outlines the figure with a 1-pixel black 
 * dotted line. 
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.figure.AntSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *       
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.AntSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.AntSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {draw2d.Figure} figure the figure to decorate with a selection feedback
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        if (figure.selectionHandles.isEmpty()) {
            var box = new draw2d.shape.basic.Rectangle();
            box.setBackgroundColor(null);
            box.setDashArray("- ");
            box.setColor("#00bdee");
            box.hide= function(){
                // IMPORTANT
                // don't add/remove this rectangle to the canvas resizeHandles. This rect isn't responsible for any hitTest or
                // dragDrop operation
                //canvas.resizeHandles.remove(box);
                box.setCanvas(null);
            };
            box.show= function(canvas){
                box.setCanvas(canvas);
                // IMPORTANT
                // don't add/remove this rectangle to the canvas resizeHandles. This rect isn't responsible for any hitTest or
                // dragDrop operation
                //canvas.resizeHandles.remove(box);
                //canvas.resizeHandles.add(box);
                box.shape.toFront();
            };
            box.show(canvas);
            figure.selectionHandles.add(box);
        }
        this.moved(canvas, figure);
   },
    
    
    /**
     * @method
     * Callback if the figure has been moved
     * 
     * @param figure
     * 
     * @template
     */
    moved: function(canvas, figure){
        if(figure.selectionHandles.isEmpty()){
            return; // silently
        }
        var box= figure.selectionHandles.get(0); 
        box.setPosition(figure.getPosition().translate(-2,-2));
        box.setDimension(figure.getWidth()+4, figure.getHeight()+4);
        box.setRotationAngle(figure.getRotationAngle());
     }
}); 
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.GlowSelectionFeedbackPolicy
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.figure.GlowSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.GlowSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.GlowSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        figure.setGlow(true);
        this.moved(canvas, figure);
   },
   
   
   /**
    * @method
    * 
    * @param {draw2d.Figure} figure the unselected figure
    */
   onUnselect: function(canvas, figure ){
		this._super(canvas, figure);
		figure.setGlow(false);
   }
     
}); 
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.SlimSelectionFeedbackPolicy
 * 
 * See the example:
 *
 *     @example preview small frame
 *       circle =new draw2d.shape.basic.Circle();
 *       circle.installEditPolicy(new draw2d.policy.SlimSelectionFeedbackPolicy());
 *       canvas.addFigure(circle,90,50);
 *
 *       canvas.addFigure(new draw2d.shape.basic.Label("Click on the circle to see the selection feedback"),20,10);
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.RectangleSelectionFeedbackPolicy
 */
draw2d.policy.figure.SlimSelectionFeedbackPolicy = draw2d.policy.figure.RectangleSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.SlimSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
   },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas,figure, isPrimarySelection){
        
        this._super(canvas,figure, isPrimarySelection);
        
        if(!figure.selectionHandles.isEmpty())
        {
            // resize the standard resize handles to the half on the normal size
            //
            figure.selectionHandles.each(function(i,e){
                e.setDimension(6,6);
                e.setRadius(0);
             });
        }
        this.moved(canvas,figure);
   }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.figure.VertexSelectionFeedbackPolicy
 * 
 * Called by the framework if the user edit the position of a figure with a drag drop operation.
 * Sub class like SelectionEditPolicy or RegionEditPolicy cam adjust th e position of the figure or the selections handles.
 * 
 * @author  Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.figure.VertexSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.figure.VertexSelectionFeedbackPolicy",

    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * 
     * @template
     * @param {draw2d.Connection} connection the selected figure
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, connection, isPrimarySelection){
//    	this._super(canvas, connection, isPrimarySelection);
    	
    	var points = connection.getVertices();
    	for(var i=0 ; i<points.getSize(); i++){
    		var handle = new draw2d.shape.basic.VertexResizeHandle(connection, i);
            connection.selectionHandles.add( handle);         
            handle.setDraggable(connection.isResizeable());
            handle.show(canvas);

            if(i!==0){
        		var handle = new draw2d.shape.basic.GhostVertexResizeHandle(connection, i-1);
                connection.selectionHandles.add( handle);         
                handle.setDraggable(connection.isResizeable());
                handle.show(canvas);
            }
        }
 
        this.moved(canvas, connection);
    },
    
    /**
     * @method
     * Callback method if the figure has been moved.
     * 
     * @template
     */
    moved: function(canvas,figure){
        figure.selectionHandles.each(function(i,e){
            e.relocate();
        });
    }
    

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.line.LineSelectionFeedbackPolicy
 * 
 *
 * @author Andreas Herz
 * @extends draw2d.policy.figure.SelectionFeedbackPolicy
 */
draw2d.policy.line.LineSelectionFeedbackPolicy = draw2d.policy.figure.SelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.line.LineSelectionFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new selection feedback policy for a line or connection
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * Called by the framework of the Policy should show a resize handle for the given shape
     * 
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
        if(figure.selectionHandles.isEmpty()){
            figure.selectionHandles.add( new draw2d.shape.basic.LineStartResizeHandle(figure));
            figure.selectionHandles.add( new draw2d.shape.basic.LineEndResizeHandle(figure));

            figure.selectionHandles.each(function(i,e){
                e.setDraggable(figure.isResizeable());
                e.show(canvas);
            });
        }
        this.moved(canvas, figure);
    },
    
    /**
     * @method
     * Callback method if the figure has been moved.
     * 
     * @template
     */
    moved: function(canvas,figure){
    	figure.selectionHandles.each(function(i,e){
            e.relocate();
        });
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.line.VertexSelectionFeedbackPolicy
 * 
 * Feedback and edit policy for the VertexRouter.
 * 
 * @author  Andreas Herz
 * @extends draw2d.policy.line.LineSelectionFeedbackPolicy
 */
draw2d.policy.line.VertexSelectionFeedbackPolicy = draw2d.policy.line.LineSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.line.VertexSelectionFeedbackPolicy",

    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },
    

    /**
     * @method
     * 
     * @template
     * @param {draw2d.Connection} connection the selected figure
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, figure, isPrimarySelection){
    	
        var startHandle =  new draw2d.shape.basic.LineStartResizeHandle(figure);
        var endHandle = new draw2d.shape.basic.LineEndResizeHandle(figure);
        figure.selectionHandles.add(startHandle);
        figure.selectionHandles.add( endHandle);

    	var points = figure.getVertices();
    	var count = points.getSize()-1;
    	var i=1;
    	for( ; i<count; i++){
    	    figure.selectionHandles.add( new draw2d.shape.basic.VertexResizeHandle(figure, i));         
    	    figure.selectionHandles.add( new draw2d.shape.basic.GhostVertexResizeHandle(figure, i-1));         
        }
    	
    	figure.selectionHandles.add( new draw2d.shape.basic.GhostVertexResizeHandle(figure, i-1));         

    	figure.selectionHandles.each(function(i,e){
            e.setDraggable(figure.isResizeable());
            e.show(canvas);
        });
        
        this.moved(canvas, figure);
    }   

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.line.OrthogonalSelectionFeedbackPolicy
 * 
 * Feedback and edit policy for the InteractiveMannhattanRouter.
 * 
 * @author  Andreas Herz
 * @extends draw2d.policy.line.LineSelectionFeedbackPolicy
 */
draw2d.policy.line.OrthogonalSelectionFeedbackPolicy = draw2d.policy.line.LineSelectionFeedbackPolicy.extend({

    NAME : "draw2d.policy.line.OrthogonalSelectionFeedbackPolicy",

    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
        
        // The ResizeHandle for the Policy. This is inline to avoid that a user want to use them without
        // the right installed policy.
        //
        this.ResizeHandle = draw2d.ResizeHandle.extend({
            NAME : "#ResizeHandle",

             init: function( figure, index) {
                this._super(figure);
                this.index = index;
            },
           
            
            /**
             * @method
             * Will be called after a drag and drop action.<br>
             *
             * @private
             **/
            onDragStart : function()
            {
                this._super();
                this.command = this.getCanvas().getCurrentSelection().createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE_VERTICES));
                
                // Vertex is a reference and not a copy of the point
                this.vertex = this.owner.getVertices().get(this.index).clone();
            },
            
            /**
             * @method
             * Called from the framework during a drag&drop operation of the ResizeHandles
             * 
             * @param {Number} dx the x difference between the start of the drag drop operation and now
             * @param {Number} dy the y difference between the start of the drag drop operation and now
             * @param {Number} dx2 The x diff since the last call of this dragging operation
             * @param {Number} dy2 The y diff since the last call of this dragging operation
             * @return {boolean}
             **/
            onDrag : function(dx, dy, dx2, dy2) 
            {
                if (this.command == null) {
                    return false;
                }
                
                var fromDir = this.owner.getSource().getConnectionDirection(this.owner, this.owner.getTarget());
                var toDir   = this.owner.getTarget().getConnectionDirection(this.owner, this.owner.getSource());
                
                this.vertex.translate(dx2, dy2);
                
                var vertices = this.owner.getVertices();
                var   count  = vertices.getSize();
                //shortcut for math operations
                var max = Math.max;
                var min = Math.min;
                
                
                // Keep in mind: "p1" is always the dragged handle in the coding below
                //               marked with an '*' in the diagram 
                //
                
                // FIRST handle of the connection
                //
                if(this.index === 1){
                    var p0 = vertices.get(this.index-1); // first vertex of the connection
                    var p1 = vertices.get(this.index  ); // dragged vertex
                    var p2 = vertices.get(this.index+1); // additional neighbor

                    // vertex alignment to handle:
                    //
                    //      p0 +-----* p1       p1 *------+ p0          
                    //               |             |          
                    //               |             |         
                    //               + p2       p2 +
                    if((p1.x == p2.x) && (p0.y == p1.y)){
                       switch(fromDir){
                       case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                          // p0 is on the left of p1
                          //
                          this.owner.setVertex(1,max(p0.x+10,this.vertex.x), p1.y); // p1
                          this.owner.setVertex(2,max(p0.x+10,this.vertex.x), p2.y); // p2
                          break;
                          // p0 is on the right of p2
                          //
                       case draw2d.geo.Rectangle.DIRECTION_LEFT:
                          this.owner.setVertex(1,min(p0.x-10,this.vertex.x), p1.y); // p1
                          this.owner.setVertex(2,min(p0.x-10,this.vertex.x), p2.y); // p2
                          break;
                       }
                    }
                    
                    // vertices alignment to handle:
                    //
                    //      p0 +              p1 *--------+ p2
                    //         |                 |
                    //         |                 |
                    //      p1 *-----+ p2     p0 +
                    else{
                       switch(fromDir){
                       case draw2d.geo.Rectangle.DIRECTION_UP:
                          // p0 is below of p1
                          //
                          this.owner.setVertex(1,p1.x,min(p0.y-10,this.vertex.y)); // p1
                          this.owner.setVertex(2,p2.x,min(p0.y-10,this.vertex.y)); // p2
                          break;
                          // p0 is above of p2
                          //
                       case draw2d.geo.Rectangle.DIRECTION_DOWN:
                          this.owner.setVertex(1,p1.x,max(p0.y+10,this.vertex.y)); // p1
                          this.owner.setVertex(2,p2.x,max(p0.y+10,this.vertex.y)); // p2
                          break;
                       }
                    }
                 }
                
                // LAST handle: Only the left hand side sibling can be changed
                //
                else if(this.index === (count-2)){
                   var p2 = vertices.get(this.index-1);  // neighbor of the dragged vertex
                   var p1 = vertices.get(this.index  );  // dragged vertex 
                   var p0 = vertices.get(this.index+1);  // last vertex of the connection 

                   // vertices with this alignment.
                   //
                   //      p2 +-----* p1                 + p0
                   //               |                    |
                   //               |                    |
                   //               + p0     p2 +--------* p1
                   if((p0.x === p1.x) && (p2.y === p1.y)){
                      switch(toDir){
                      // p0 is below of p1
                      case draw2d.geo.Rectangle.DIRECTION_UP:
                         this.owner.setVertex(count - 2,p1.x, min(p0.y-10,this.vertex.y)); // p1
                         this.owner.setVertex(count - 3,p2.x, min(p0.y-10,this.vertex.y)); // p2
                         break;
                      // p0 is above p2
                      case draw2d.geo.Rectangle.DIRECTION_DOWN:
                          this.owner.setVertex(count - 2,p1.x, max(p0.y+10,this.vertex.y)); // p1
                          this.owner.setVertex(count - 3,p2.x, max(p0.y+10,this.vertex.y)); // p2
                         break;
                      }
                   }
                   
                   // vertices with this alignment. 
                   //
                   //      p2 +              p0 +--------* p1
                   //         |                          |
                   //         |                          |
                   //      p1 *-----+ p0              p2 +
                   else{
                      switch(toDir){
                      case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                         // p0 is on the left of p1
                         //
                          this.owner.setVertex(count -2,max(p0.x+10,this.vertex.x),p1.y); // p1
                          this.owner.setVertex(count -3,max(p0.x+10,this.vertex.x),p2.y); // p2
                         break;
                         // p0 is on the right of p2
                         //
                      case draw2d.geo.Rectangle.DIRECTION_LEFT:
                          this.owner.setVertex(count -2,min(p0.x-10,this.vertex.x),p1.y); // p1
                          this.owner.setVertex(count -3,min(p0.x-10,this.vertex.x),p2.y); // p2
                         break;
                      }
                   }
                }
                // The resize handle is in the middle of the connection.
                // -> In this case the connection MUST HAVE at least 5 vertices
                //
                else{
                   var p_m1= vertices.get(this.index-2);
                   var p0  = vertices.get(this.index-1);
                   var p1  = vertices.get(this.index);
                   var p2  = vertices.get(this.index+1);
                   var p3  = vertices.get(this.index+2);
                   
                   // vertices alignment to handle
                   //
                   //               .              .
                   //               .              .
                   //   p1 *------->+  p0      p0  +<---------* p1
                   //      |        .              .          |
                   //      |        .              .          |
                   //   p2 |                                  | p2
                   //   ...+...                         ......+.....
                   //
                   if((p1.x=== p2.x) && (p1.y === p0.y)){
                      // Exception handling if the dragged handle (p1) is near by the start of the connection
                      // p_m1 is the start of the connection 
                      // p0 must be the immediate neighbor of p_m1 
                      //
                      if(this.index-2 === 0) {
                         switch(fromDir){
                         case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                             this.owner.setVertex(this.index-1,p0.x,max(this.vertex.y,p_m1.y-10));          // p0
                             this.owner.setVertex(this.index  ,this.vertex.x,max(this.vertex.y,p_m1.y-10)); // p1
                             this.owner.setVertex(this.index+1,this.vertex.x,p2.y);                         // p2
                            break;
                         case draw2d.geo.Rectangle.DIRECTION_LEFT:
                             this.owner.setVertex(this.index-1,p0.x,min(this.vertex.y,p_m1.y+10));          // p0
                             this.owner.setVertex(this.index  ,this.vertex.x,this.vertex.y); // p1
                             this.owner.setVertex(this.index+1,this.vertex.x,p2.y);                         // p2
                            break;
                         case draw2d.geo.Rectangle.DIRECTION_UP:
                             this.owner.setVertex(this.index-1,p0.x,min(this.vertex.y,p_m1.y-10));          // p0
                             this.owner.setVertex(this.index  ,this.vertex.x,min(this.vertex.y,p_m1.y-10)); // p1
                             this.owner.setVertex(this.index+1,this.vertex.x,p2.y);                         // p2
                            break;
                         case draw2d.geo.Rectangle.DIRECTION_DOWN:
                             this.owner.setVertex(this.index-1,p0.x,max(this.vertex.y,p_m1.y+10));          // p0
                             this.owner.setVertex(this.index  ,this.vertex.x,max(this.vertex.y,p_m1.y+10)); // p1
                             this.owner.setVertex(this.index+1,this.vertex.x, p2.y);                        // p2
                            break;
                         }
                      }
                      // Exception handling if the dragged handle (p1L) near by the end of the connection
                      // p3 is the end of the connection
                      //
                      else if((this.index-count+3) === 0) {
                         switch(toDir){
                         case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                             this.owner.setVertex(this.index-1,p0.x,this.vertex.y);                       // p0
                             this.owner.setVertex(this.index  ,max(this.vertex.x,p3.x+10),this.vertex.y); // p1
                             this.owner.setVertex(this.index+1,max(this.vertex.x,p3.x+10),p2.y);          // p2
                            break;
                         case draw2d.geo.Rectangle.DIRECTION_LEFT:
                             this.owner.setVertex(this.index-1,p0.x,this.vertex.y);                       // p0
                             this.owner.setVertex(this.index  ,min(this.vertex.x,p3.x-10),this.vertex.y); // p1
                             this.owner.setVertex(this.index+1,min(this.vertex.x,p3.x-10),p2.y);          // p2
                            break;
                         }
                      }
                      else{
                          this.owner.setVertex(this.index-1,p0.x,this.vertex.y);                          // p0
                          this.owner.setVertex(this.index  ,this.vertex);                                 // p1
                          this.owner.setVertex(this.index+1,this.vertex.x,p2.y);                          // p2
                      }
                   }
                   // vertices alignment to handle
                   //
                   //  ...+...                            ...+...
                   //  p0 |                        .         | p0
                   //     |          .             .         |         
                   //     |          .             .         |        
                   //  p1 *----------+ p2      p2  +---------* p1 
                   //                .             .                    
                   //                .             .                    
                   else if((p0.x === p1.x) && (p1.y===p2.y)){
                      // p_m1 is the start of the analyzed segment
                       // p0 must be the immediate neighbor of p_m1 
                      //
                      if(this.index-2 === 0) {
                         switch(fromDir){
                         case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                             this.owner.setVertex(this.index-1,max(this.vertex.x,p_m1.x+10),p0.y);          // p0
                             this.owner.setVertex(this.index  ,max(this.vertex.x,p_m1.x+10),this.vertex.y); // p1
                             this.owner.setVertex(this.index+1,p2.x,this.vertex.y);                         // p2
                            break;
                         case draw2d.geo.Rectangle.DIRECTION_LEFT:
                             this.owner.setVertex(this.index-1,min(this.vertex.x,p_m1.x-10),p0.y);          // p0
                             this.owner.setVertex(this.index  ,min(this.vertex.x,p_m1.x-10),this.vertex.y); // p1
                             this.owner.setVertex(this.index+1,p2.x,this.vertex.y);                         // p2
                            break;
                         }
                      }
                      // p3 ist der Endpunkt
                      //
                      else if((this.index-count+3) === 0) {
                         switch(toDir){
                         case draw2d.geo.Rectangle.DIRECTION_RIGHT:
                             this.owner.setVertex(this.index-1,p0.x,min(this.vertex.y,p3.y+10));            // p0
                             this.owner.setVertex(this.index  ,this.vertex.x,min(this.vertex.y,p3.y+10));   // p1
                             this.owner.setVertex(this.index+1,this.vertex.x,p2.y);                         // p2
                            break;
                         case draw2d.geo.Rectangle.DIRECTION_LEFT:
                             this.owner.setVertex(this.index-1,p0.x,max(this.vertex.y,p3.y-10));            // p0
                             this.owner.setVertex(this.index  ,this.vertex.x,max(this.vertex.y,p3.y-10));   // p1
                             this.owner.setVertex(this.index+1,this.vertex.x,p2.y);                         // p2
                            break;
                         }
                      }
                      else{
                          this.owner.setVertex(this.index-1,this.vertex.x,p0.y);                            // p0
                          this.owner.setVertex(this.index  ,this.vertex);                                   // p1
                          this.owner.setVertex(this.index+1,p2.x,this.vertex.y);                            // p2
                      }
                   }
                }

                this.relocate();

                // update the command for the undo/redo stuff
                //
                if(this.command!==null){
                    this.command.updateVertices(this.owner.getVertices().clone());                   
                }
                
                // note that the user has changed the routing manually.
                // This skips the automatic routing.
                this.owner._routingMetaData.routedByUserInteraction = true;             
                return true;
            },
            
            /**
             * @method Called after a drag and drop action.<br>
             *         Sub classes can override this method to implement additional stuff. Don't forget to call the super implementation via <code>this._super();</code>
             * @return {boolean}
             */
            onDragEnd : function()
            {
                var stack = this.getCanvas().getCommandStack();
                
                stack.execute(this.command);
                this.command = null;
                
                return true;
            },
            
            
            /**
             * @method
             * Controls the location of the resize handle 
             *
             * @template
             **/
            relocate:function(){

                var resizeWidthHalf = this.getWidth()/2;
                var resizeHeightHalf= this.getHeight()/2;

                var anchor = this.owner.getVertices().get(this.index);
                    
                this.setPosition(anchor.x-resizeWidthHalf,anchor.y-resizeHeightHalf);
            }
            
        });
    },
    

    /**
     * @method
     * 
     * @template
     * @param {draw2d.Connection} connection the selected figure
     * @param {boolean} isPrimarySelection
     */
    onSelect: function(canvas, connection, isPrimarySelection){
    	this._super(canvas, connection, isPrimarySelection);
    	
    	var points = connection.getVertices();
    	var i=1;
    	for( ; i<(points.getSize()-1); i++){
    		var handle = new this.ResizeHandle(connection, i);
            connection.selectionHandles.add( handle);         
            handle.setDraggable(connection.isResizeable());
            handle.show(canvas);
/*
    		var handle = new draw2d.shape.basic.GhostVertexResizeHandle(connection, i-1);
            connection.selectionHandles.add( handle);         
            handle.setDraggable(connection.isResizeable());
            handle.show(canvas);
            */
        }
    	/*
		var handle = new draw2d.shape.basic.GhostVertexResizeHandle(connection, i-1);
        connection.selectionHandles.add( handle);         
        handle.setDraggable(connection.isResizeable());
        handle.show(canvas);
        */
    	
        this.moved(canvas, connection);
    },
    
    

    /**
     * @method
     * 
     */
    isSegmentDeleteable:function(conn, segmentIndex){

       var segmentCount = conn.getSegments().getSize();
       
       // Das erste und das letzte Segment kann nicht geloescht werden
       //
       if( (segmentIndex<=0) || (segmentIndex>= segmentCount-2))
          return false;

       // A Connection needs at least 3 segments
       //
       if((segmentCount<=4))
          return false;

       var fromPt  = conn.getStartPoint();
       var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

       var toPt    = conn.getEndPoint();
       var toDir   = conn.getTarget().getConnectionDirection(conn, conn.getSource());

       // 
       // Falls die Leitungsfuehrung sich     ___
       // wie nebenan aufgezeigt aufbaut,    |   |        From
       // mussen mindestens 5 Segmente er-   | 1 |-----+
       // halten beleiben, damit ueberhaupt  |___|     |
       // eine Leitungsfuehrung m??glich ist.           |
       // Der Ausgang von Objekt 1 ist      +----------+
       // so zu dem Eingang von 2 versetzt, |
       // das man nicht mehr mit 3 Segment  |    ___
       // auskommt.                         |   |   |
       //                                   +---| 2 |      To
       //                                       |___|
       //
       if( (fromDir === 0) && ( toDir === 180) && (fromPt.x >= toPt.x) && (segmentCount < 6) )
          return false;


       // 
       // Falls die Leitungsfuehrung sich     ___
       // wie nebenan aufgezeigt aufbaut,    |   |        To
       // mussen mindestens 5 Segmente er-   | 2 |-----+
       // halten beleiben, damit ueberhaupt  |___|     |
       // eine Leitungsfuehrung m??glich ist.           |
       // Der Ausgang von Objekt 1 ist      +----------+
       // so zu dem Eingang von 2 versetzt, |
       // das man nicht mehr mit 3 Segment  |    ___
       // auskommt.                         |   |   |
       //                                   +---| 1 |     From
       //                                       |___|
       //
       if( (fromDir == 180) & ( toDir == 0) && (fromPt.x <= toPt.x) && (NumPoints() < 6) )
          return false;

       return true;
    },
    
    removeSegment: function(conn, segmentIndex){

       if(!this.isSegmentDeleteable(conn, segmentIndex))
          return;
       
       var vertexCount  = conn.getVertices().getSize();
             
       var fromPt  = conn.getStartPoint();
       var fromDir = conn.getSource().getConnectionDirection(conn, conn.getTarget());

       var toPt    = conn.getEndPoint();
       var toDir   = conn.getTarget().getConnectionDirection(conn, conn.getSource());

       var p0 = conn.getVertex(segmentIndex -1);
       var p1 = conn.getVertex(segmentIndex   );
       var p2 = conn.getVertex(segmentIndex +1);
       var p3 = conn.getVertex(segmentIndex +2);
       
       //                                             p0 .
       // Es wird ein Horizontales Segment               .
       // geloescht. Es muessen somit die Punkte         .
       // p0 und p3 neu gesetzt werden.               p1 +------*-----+ p2
       // Ihre neue X-Koordinate ist somit in der               ^     .
       // Mitte des geloeschten Segmentes                      newX   .
       //                                                             . p3
       //  
       if(p1.y == p2.y){
          var newX = (p1.x + p2.x) / 2;
          // Die neue X-Koordinate muss auf jeden Falls zwischen p-1 und p4 liegen
          //
          if(fromDir == 0 && segmentIndex == 2)
             newX = Math.max(newX ,fromPt.x);
          else if(fromDir == 180 && segmentIndex == 2)
             newX = Math.min(newX ,fromPt.x);
          
          if(toDir == 0 && segmentIndex == vertexCount-4)
             newX = max(newX ,toPt.x);
          else if(toDir == 180 && segmentIndex == vertexCount-4)
             newX = Math.min(newX ,toPt.x);

          conn.setVertex(segmentIndex -1, new draw2d.geo.Point(newX,p0.y));
          conn.setVertex(segmentIndex +2, new draw2d.geo.Point(newX,p3.y));
          
          conn.removeSegment(segmentIndex);
       }
       
       //                                                         p2       p3
       // Es wird ein vertikales Segment                        +..........+
       // geloescht. Es muessen somit die Punkte                |
       // p0 und p3 neu gesetzt werden.                         |             
       // Ihre neue Y-Koordinate ist somit in der               |     
       // Mitte des geloeschten Segmentes              p0       | p1     
       //                                              +........+     
       //   
       else if(p1.x == p2.x){
          // Das erste senkrechte Segment wird geloescht
          // p0 ist der Startpunkt und darf somit nicht verschoben werden
          //
          if(fromDir == 0 && segmentIndex == 1)
             SetPoint(segmentIndex +2, CPoint(p3.x,p1.y));
          else if(fromDir == 180 && segmentIndex == 1)
             SetPoint(segmentIndex +2, CPoint(p3.x,p1.y));
          // Das letzte Segment welche senkrecht ist, wird geloescht
          // p3 ist der Endpunkt und darf somit nicht verschoben werden
          //
          else if(toDir == 0 && segmentIndex == vertexCount-3)
             SetPoint(segmentIndex -1, CPoint(p0.x,p2.y));
          else if(toDir == 180 && segmentIndex == vertexCount-3)
             SetPoint(segmentIndex -1, CPoint(p0.x,p2.y));
          // Es ist ein Segment in der Mitte
          // p0 und p3 duerfen somit verschoben werden
          //
          else{ 
             SetPoint(segmentIndex -1, CPoint(p0.x,(p1.y+p2.y)/2));
             SetPoint(segmentIndex +2, CPoint(p3.x,(p1.y+p2.y)/2));
          }
          RemoveSegment(segmentIndex);
          return;
       }
    },
    

    splitSegment: function(conn, segmentIndex){
       var p1 = conn.getVertex(segmentIndex   );
       var p2 = conn.getVertex(segmentIndex +1);
       var length= 40;

       // Das einzufuegende Segment ist horizontal
       //       p2 +
       //          .
       // np1 +----+ np2
       //     .
       //     .
       //     + p1
       //
       if(p1.x == p2.x){
          var np1 = new util.geo.Point(p1.x-(length/2), (p1.y + p2.y ) /2);
          var np2 = new util.geo.Point(p2.x+(length/2), (p1.y + p2.y ) /2);

          conn.setVertex(segmentIndex  , new util.geo.Point(np1.x,p1.y));
          conn.setVertex(segmentIndex+1, new util.geo.Point(np2.x,p2.y));
          conn.insertVertexAt(segmentIndex+1, np1);
          conn.insertVertexAt(segmentIndex+2, np2);
       }
       // Das eizufuegende Segment ist senkrecht
       //     p1        np1
       //   +.........+
       //             |
       //             |
       //             | np2       p2
       //             +.........+
       //
       else if(p1.y == p2.y){
          var np1 = new draw2d.util.Point(0,0);
          var np2 = new draw2d.util.Point(0,0);

          // p1 ist der Startpunkt und darf somit nicht verschoben werden
          //
          if(m_lastHitSegment==0){
             np1.x = (p1.x + p2.x ) /2;
             np1.y = p1.y;
             np2.x = (p1.x + p2.x ) /2;
             np2.y = p2.y+length;
             conn.setVertex(segmentIndex+1, new draw2d.util.Point(p2.x,np2.y));
          }
          // p2 ist der Schlusspunkt und darf somit nicht veaendert werden
          //
          else if(m_lastHitSegment == NumPoints()-2){
             np1.x = (p1.x + p2.x ) /2;
             np1.y = p1.y-length;
             np2.x = (p1.x + p2.x ) /2;
             np2.y = p2.y;
             conn.setVertex(segmentIndex  , new draw2d.util.Point(p1.x,np1.y));
          }
          else {
             np1.x = (p1.x + p2.x ) /2;
             np1.y = p1.y - (length/2);
             np2.x = (p1.x + p2.x ) /2;
             np2.y = p2.y + (length/2);
             conn.setVertex(segmentIndex  , new draw2d.util.Point(p1.x,np1.y));
             conn.setVertex(segmentIndex+1, new draw2d.util.Point(p2.x,np2.y));
          }
          conn.insertVertexAt(segmentIndex+1, np1);
          conn.insertVertexAt(segmentIndex+2, np2);
       }

       // Die Leitungsfuehrung wurde mit Hand veraendert
       // -> Es darf spaeter kein vollstaendiges Autorouting mehr
       // gemacht werden.
       //
    //   m_routingHasChanged = true;
    }
    


});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.port.PortFeedbackPolicy
 * 
 * A draw2d.policy.SelectionFeedbackPolicy that is sensitive to the canvas selection. Subclasses will typically 
 * decorate the {@link draw2d.Figure figure} with things like selection handles and/or focus feedback.
 * <br>
 * If you want to change the handle visibility for a figure, then you should use SelectionFeedbackPolicy to do that.
 * 
 * @author Andreas Herz
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.port.PortFeedbackPolicy = draw2d.policy.figure.DragDropEditPolicy.extend({

	
    NAME : "draw2d.policy.port.PortFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
    },

    /**
     * @method
     * Called if the dragged port hove another port
     * 
     * @param {draw2d.Canvas} canvas
     * @param {draw2d.Port}   draggedFigure
     * @param {draw2d.Figure} hoverFigure
     */
    onHoverEnter: function(canvas, draggedFigure, hoverFigure){
    },
    
    onHoverLeave: function(canvas, draggedFigure, hoverFigure){
    }

        
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.port.ElasticStrapFeedbackPolicy
 * 
 * A draw2d.policy.SelectionFeedbackPolicy that is sensitive to the canvas selection. Subclasses will typically 
 * decorate the {@link draw2d.Figure figure} with things like selection handles and/or focus feedback.
 * <br>
 * If you want to change the handle visibility for a figure, then you should use SelectionFeedbackPolicy to do that.
 * 
 * @author Andreas Herz
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.port.ElasticStrapFeedbackPolicy = draw2d.policy.port.PortFeedbackPolicy.extend({

    NAME : "draw2d.policy.port.ElasticStrapFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
        this.connectionLine = null;
    },
    
    /**
     * @method
     * Called by the framework if the related shape has init a drag&drop
     * operation
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDragStart: function(canvas, figure){
        this.connectionLine = new draw2d.shape.basic.Line();
        this.connectionLine.setCanvas(canvas);
        this.connectionLine.getShapeElement();
        
        this.onDrag(canvas, figure);
    },
    
    
    /**
     * @method
     * Called by the framework during drag a figure.
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDrag: function(canvas, figure){
        var x1 = figure.ox+figure.getParent().getAbsoluteX();
        var y1 = figure.oy+figure.getParent().getAbsoluteY();
        
        this.connectionLine.setStartPoint(x1,y1);
        this.connectionLine.setEndPoint(figure.getAbsoluteX(),figure.getAbsoluteY());
    },
    
    /**
     * @method
     * Called by the framework if the drag drop operation ends.
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDragEnd: function(canvas, figure){
        this.connectionLine.setCanvas(null);
        this.connectionLine = null;
    },
    
    onHoverEnter: function(canvas, draggedFigure, hoverFiger){
    	this.connectionLine.setGlow(true);
    	hoverFiger.setGlow(true);
    },
    
    onHoverLeave: function(canvas, draggedFigure, hoverFiger){
    	hoverFiger.setGlow(false);
    	this.connectionLine.setGlow(false);
    }

        
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.policy.port.IntrusivePortsFeedbackPolicy
 * 
 * A draw2d.policy.SelectionFeedbackPolicy that is sensitive to the canvas selection. Subclasses will typically 
 * decorate the {@link draw2d.Figure figure} with things like selection handles and/or focus feedback.
 * <br>
 * If you want to change the handle visibility for a figure, then you should use SelectionFeedbackPolicy to do that.
 * 
 * @author Andreas Herz
 * @extends draw2d.policy.figure.DragDropEditPolicy
 */
draw2d.policy.port.IntrusivePortsFeedbackPolicy = draw2d.policy.port.PortFeedbackPolicy.extend({

    NAME : "draw2d.policy.port.IntrusivePortsFeedbackPolicy",
    
    /**
     * @constructor 
     * Creates a new Router object
     */
    init: function(){
        this._super();
        this.connectionLine = null;
        this.tweenable = null;
    },
    
    /**
     * @method
     * Called by the framework if the related shape has init a drag&drop
     * operation
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDragStart: function(canvas, figure){
        var start = 0;
    	figure.getDropTargets().each(function(i, element){
	        element.__beforeInflate = element.getWidth();
	        start = element.__beforeInflate;
    	});

    	// animate the resize of the ports
    	//
    	var portsToGrow = figure.getDropTargets();
    	portsToGrow.grep(function(p){
    	    return (p.NAME != figure.NAME && p.parent!==figure.parent) || (p instanceof draw2d.HybridPort) || (figure instanceof draw2d.HybridPort);
    	});
        this.tweenable = new Tweenable();
        this.tweenable.tween({
          from:     { 'size': start/2  },
          to:       { 'size': start },
          duration: 200,
          easing : "easeOutSine",
          step: function (params) {
              portsToGrow.each(function(i, element){
                  // IMPORTANT shortcut to avoid rendering errors!!
                  // performance shortcut to avoid a lot of fireMoveEvent and recalculate/routing of all related connections
                  // for each setDimension call. Additional the connection is following a port during Drag&Drop operation
                  element.shape.attr({rx : params.size, ry :params.size});
                  element.width = element.height = params.size*2;
                  //element.setDimension(params.size, params.size);
              });
          }
        });
    	
        this.connectionLine = new draw2d.shape.basic.Line();
        this.connectionLine.setCanvas(canvas);
        this.connectionLine.getShapeElement();
        this.connectionLine.setDashArray("- ");
        this.connectionLine.setColor("#30c48a");
       
        this.onDrag(canvas, figure);
    },
    
    
    /**
     * @method
     * Called by the framework during drag a figure.
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDrag: function(canvas, figure){
        var x1 = figure.ox+figure.getParent().getAbsoluteX();
        var y1 = figure.oy+figure.getParent().getAbsoluteY();
        
        this.connectionLine.setStartPoint(x1,y1);
        this.connectionLine.setEndPoint(figure.getAbsoluteX(),figure.getAbsoluteY());
    },
    
    /**
     * @method
     * Called by the framework if the drag drop operation ends.
     * 
     * @param {draw2d.Canvas} canvas The host canvas
     * @param {draw2d.Figure} figure The related figure
     * @template
     */
    onDragEnd: function(canvas, figure){
        this.tweenable.stop(false);
        this.tweenable = null;
    	figure.getDropTargets().each(function(i, element){
            // IMPORTANT shortcut to avoid rendering errors!!
            // performance shortcut to avoid a lot of fireMoveEvent and recalculate/routing of all related connections
    	    // for each setDimension call. Additional the connection is following a port during Drag&Drop operation
    	    element.shape.attr({rx : element.__beforeInflate/2, ry :element.__beforeInflate/2});
            element.width = element.height = element.__beforeInflate;
    		//element.setDimension(element.__beforeInflate, element.__beforeInflate);
    	});
        this.connectionLine.setCanvas(null);
        this.connectionLine = null;
    },
    
    onHoverEnter: function(canvas, draggedFigure, hoverFiger){
    	this.connectionLine.setGlow(true);
    	hoverFiger.setGlow(true);
    },
    
    onHoverLeave: function(canvas, draggedFigure, hoverFiger){
    	hoverFiger.setGlow(false);
    	this.connectionLine.setGlow(false);
    }

        
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/draw2d.Configuration = {
        version : "4.3.2",
        i18n : {
            command : {
                move : "Move Shape",
                deleteShape  : "Delete Shape",
                moveShape: "Move Shape",
                moveLine : "Move Line",
                addShape : "Add Shape",
                moveVertex : "Move Vertex",
                moveVertices : "Move Vertices",
                deleteVertex : "Delete Vertex",
                resizeShape : "Resize Shape",
                collection: "Execute Commands",
                addVertex: "Add Vertex",
                connectPorts: "Connect Ports"
            },
            dialog:{
                filenamePrompt: "Enter Filename:"
            }
        },
        color:{
            resizeHandle : "#5bcaff",
            vertexHandle : "#5bcaff"
        }
};
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.Canvas
 * Interactive paint area of the draw2d library.
 * <br>
 * <strong>Usage</strong>
 *      <script type="text/javascript">
 *      
 *      $(window).load(function () {
 *          
 *          var canvas = new draw2d.Canvas("gfx_holder");
 *      
 *          var figure1 = new draw2d.shape.basic.Oval();
 *          var figure2 = new draw2d.shape.basic.Rectangle();
 *          canvas.addFigure(figure1,100,100);
 *          canvas.addFigure(figure2,120,150);
 *      });
 *      </script>
 *      
 * @inheritable
 * @author Andreas Herz
 */
draw2d.Canvas = Class.extend(
{
    NAME : "draw2d.Canvas",

    /**
     * @constructor
     * Create a new canvas with the given HTML DOM references.
     * 
     * @param {String} canvasId the id of the DOM element to use a parent container
     */
    init : function(canvasId, width, height)
    {
        // Hook the canvas calculation for IE8
        //
        if (navigator.appName == 'Microsoft Internet Explorer')
        {
          var ua = navigator.userAgent;
          var re  = new RegExp("MSIE ([0-9]{1,}[\.0-9]{0,})");
          if (re.exec(ua) != null){
            rv = parseInt( RegExp.$1 );
            if(rv===8){
                this.fromDocumentToCanvasCoordinate = this._fromDocumentToCanvasCoordinate_IE8_HACK;
            }
          }
        }

        this.setScrollArea(document.body);
        this.canvasId = canvasId;
        this.html = $("#"+canvasId);
        this.html.css({"cursor":"default"});
        if(typeof width!=="undefined"){
            this.initialWidth = width;
            this.initialHeight = height;
        }
        else{
            this.initialWidth = this.getWidth();
            this.initialHeight = this.getHeight();
        }
        
        // avoid the "highlighting" in iPad, iPhone if the user tab/touch on the canvas.
        // .... I don't like this.
        this.html.css({"-webkit-tap-highlight-color": "rgba(0,0,0,0)"});
        
        // Drag&Drop Handling from foreign DIV into the Canvas
        // Only available in combination with jQuery-UI
        //
        // Create the droppable area for the css class "draw2d_droppable"
        // This can be done by a palette of toolbar or something else.
        // For more information see : http://jqueryui.com/demos/droppable/
        //
        if(typeof this.html.droppable !=="undefined"){
            this.html.droppable({
                accept: '.draw2d_droppable',
                over: $.proxy(function(event, ui) {
                    this.onDragEnter(ui.draggable);
                },this),
                out: $.proxy(function(event, ui) {
                    this.onDragLeave(ui.draggable);
                },this),
                drop:$.proxy(function(event, ui){
                    event = this._getEvent(event);
                    var pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
                    this.onDrop(ui.draggable, pos.getX(), pos.getY());
                },this)
            });
        
            // Create the jQuery-Draggable for the palette -> canvas drag&drop interaction
            //
            $(".draw2d_droppable").draggable({
                appendTo:"body",
                stack:"body",
                zIndex: 27000,
                helper:"clone",
                drag: $.proxy(function(event, ui){
                    event = this._getEvent(event);
                    var pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
                    this.onDrag(ui.draggable, pos.getX(), pos.getY());
                },this),
                stop: function(e, ui){
                    this.isInExternalDragOperation=false;
                },
                start: function(e, ui){
                    this.isInExternalDragOperation=true;
                    $(ui.helper).addClass("shadow");
                }
           });
        }

        // painting stuff
        //
        if(typeof height!== "undefined"){
            this.paper = Raphael(canvasId,width, height);
        }
        else{
            this.paper = Raphael(canvasId, this.getWidth(), this.getHeight());
        }
        this.paper.canvas.style.position="absolute";
        
        // Status handling
        //
        this.zoomFactor = 1.0; // range [0.001..10]
        this.selection  = new draw2d.Selection();
        this.currentDropTarget = null;
        this.isInExternalDragOperation=false;
        this.currentHoverFigure = null;
        
        this.editPolicy = new draw2d.util.ArrayList();

        // internal document with all figures, ports, ....
        //
        this.figures     = new draw2d.util.ArrayList();
        this.lines       = new draw2d.util.ArrayList(); // crap - why are connections not just figures. Design by accident
        this.commonPorts = new draw2d.util.ArrayList();
        this.dropTargets = new draw2d.util.ArrayList();
        
        // all visible resize handles which can be drag&drop around. Selection handles like AntRectangleSelectionFeedback
        // are not part of this collection
        this.resizeHandles = new draw2d.util.ArrayList();
        
        // listener for selection handling
        // called if any selection events happens in the canvas
        //
        this.selectionListeners = new draw2d.util.ArrayList();

        // The CommandStack for undo/redo operations
        // 
        this.commandStack = new draw2d.command.CommandStack();
       
        // INTERSECTION/CROSSING handling for connections and lines
        //
        this.linesToRepaintAfterDragDrop =  new draw2d.util.ArrayList();
        this.lineIntersections = new draw2d.util.ArrayList();
       
        this.installEditPolicy( new draw2d.policy.canvas.BoundingboxSelectionPolicy());
//        this.installEditPolicy( new draw2d.policy.canvas.FadeoutDecorationPolicy());

        // Calculate all intersection between the different lines
        //
        this.commandStack.addEventListener($.proxy(function(event){
            if(event.isPostChangeEvent()===true){
                this.calculateConnectionIntersection();
                this.linesToRepaintAfterDragDrop.each(function(i,line){
                    line.svgPathString=null;
                    line.repaint();
                });
                this.linesToRepaintAfterDragDrop =  new draw2d.util.ArrayList();
            }
        },this));
        
        // DragDrop status handling
        //
        this.mouseDown  = false;
        this.mouseDownX = 0;
        this.mouseDownY = 0;
        this.mouseDragDiffX =0;
        this.mouseDragDiffY =0;

        this.html.bind("mouseup touchend", $.proxy(function(event)
        {
            if (this.mouseDown === false){
                return;
            }

            event = this._getEvent(event);
            this.calculateConnectionIntersection();

            this.mouseDown = false;
            var pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
            this.editPolicy.each($.proxy(function(i,policy){
                policy.onMouseUp(this, pos.x, pos.y, event.shiftKey, event.ctrlKey);
            },this));
            
            this.mouseDragDiffX = 0;
            this.mouseDragDiffY = 0;
        }, this));

        this.html.bind("mousemove touchmove", $.proxy(function(event)
        {
            event = this._getEvent(event);
            if (this.mouseDown === false){
               var pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
               // mouseEnter/mouseLeave events for Figures. Don't use the Raphael or DOM native functions.
               // Raphael didn't work for Rectangle with transparent fill (events only fired for the border line)
               // DOM didn't work well for lines. No eclipse area - you must hit the line exact to retrieve the event.
               // In this case I implement my own stuff...again and again.
               //
               // don't break the main event loop if one element fires an error during enter/leave event.
               try{
	               var hover = this.getBestFigure(pos.x,pos.y);
	               if(hover !== this.currentHoverFigure && this.currentHoverFigure!==null){
	            	   this.currentHoverFigure.onMouseLeave();
	               }
	               if(hover !== this.currentHoverFigure && hover!==null){
	            	   hover.onMouseEnter();
	               }
	               this.currentHoverFigure = hover;
               }
               catch(exc){
            	   // just write it to the console
            	   console.log(exc);
               }

               this.editPolicy.each($.proxy(function(i,policy){
                   policy.onMouseMove(this, pos.x, pos.y, event.shiftKey, event.ctrlKey);
               },this));
            }
            else{
               var diffXAbs = (event.clientX - this.mouseDownX)*this.zoomFactor;
               var diffYAbs = (event.clientY - this.mouseDownY)*this.zoomFactor;
               this.editPolicy.each($.proxy(function(i,policy){
                   policy.onMouseDrag(this,diffXAbs, diffYAbs, diffXAbs-this.mouseDragDiffX, diffYAbs-this.mouseDragDiffY);
               },this));
               this.mouseDragDiffX = diffXAbs;
               this.mouseDragDiffY = diffYAbs;
           }
        }, this));
        
        this.html.bind("mousedown touchstart", $.proxy(function(event)
        {
            var pos = null;
            switch (event.which) {
            case 1: //touch pressed
            case 0: //Left mouse button pressed
                event.preventDefault();
                event = this._getEvent(event);
                this.mouseDownX = event.clientX;
                this.mouseDownY = event.clientY;
                this.mouseDragDiffX = 0;
                this.mouseDragDiffY = 0;
                pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
                this.mouseDown = true;
                this.editPolicy.each($.proxy(function(i,policy){
                    policy.onMouseDown(this,pos.x,pos.y, event.shiftKey, event.ctrlKey);
                },this));
                break;
            case 3: //Right mouse button pressed             
                event.preventDefault();
                event = this._getEvent(event);
                pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
                this.onRightMouseDown(pos.x, pos.y, event.shiftKey, event.ctrlKey);
                break;
            case 2:
                //Middle mouse button pressed
                break;
             default:
                //You have a strange mouse
            }
        }, this));
        
        
        // Catch the dblclick and route them to the Canvas hook.
        //
        $(document).bind("dblclick",$.proxy(function(event)
        {
            event = this._getEvent(event);

            this.mouseDownX = event.clientX;
            this.mouseDownY = event.clientY;
            var pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
            this.onDoubleClick(pos.x, pos.y, event.shiftKey, event.ctrlKey);
        },this));

        
        // Catch the keyDown and CTRL-key and route them to the Canvas hook.
        //
        $(document).bind("click",$.proxy(function(event)
        {
            event = this._getEvent(event);

            // fire only the click event if we didn't move the mouse (drag&drop)
            //
            if(this.mouseDownX === event.clientX ||  this.mouseDownY === event.clientY){
                var pos = this.fromDocumentToCanvasCoordinate(event.clientX, event.clientY);
                this.onClick(pos.x, pos.y, event.shiftKey, event.ctrlKey);
            }
        },this));

        // Catch the keyDown and CTRL-key and route them to the Canvas hook.
        //
        $(document).bind("keydown",$.proxy(function(event)
        {
          // don't initiate the delete command if the event comes from an INPUT field. In this case the user want delete
          // a character in the input field and not the related shape
          if(!$(event.target).is("input")){
             var ctrl = event.ctrlKey;
             this.onKeyDown(event.keyCode, ctrl);
           }
        },this));

    },

    /**
     * @method
     * Callback for any kind of image export tools to trigger the canvas to hide all unwanted
     * decorations. The method is called e.g. from the draw2d.io.png.Writer
     * 
     * @since 4.0.0
     * @template
     */
    hideDecoration: function(){
    	
    },

    /**
     * @method
     * callback method for any image export writer to reactivate the decoration
     * of the canvas. e.g. grids, rulers,...
     * 
     * 
     * @since 4.0.0
     * @template
     */
    showDecoration: function(){
    	
    },

    /**
     * @method
     * Calculate all connection intersection of the canvas.
     * Required for "bridging" or "crossing decoration"
     * 
     * @private
     */
    calculateConnectionIntersection: function(){

        this.lineIntersections = new draw2d.util.ArrayList();
        var lines = this.getLines().clone();
        while(lines.getSize()>0){
            var l1 = lines.removeElementAt(0);
            lines.each($.proxy(function(ii,l2){
                var partInter =l1.intersection(l2);
                if(partInter.getSize()>0){
                   this.lineIntersections.add({line:l1, other:l2, intersection:partInter});
                   this.lineIntersections.add({line:l2, other:l1, intersection:partInter});
                }
            },this));
        }
    },

    /**
     * @method
     * reset the canvas and delete all model elements.<br>
     * You can now reload another model to the canvas with a {@link draw2d.io.Reader}
     * 
     * @since 1.1.0
     */
    clear : function(){
        
        this.lines.clone().each($.proxy(function(i,e){
            this.removeFigure(e);
        },this));
        
         this.figures.clone().each($.proxy(function(i,e){
            this.removeFigure(e);
        },this));
        
        this.zoomFactor =1.0;
        this.selection.clear();
        this.currentDropTarget = null;
        this.isInExternalDragOperation=false;

        // internal document with all figures, ports, ....
        //
        this.figures = new draw2d.util.ArrayList();
        this.lines = new draw2d.util.ArrayList();
        this.commonPorts = new draw2d.util.ArrayList();
        this.dropTargets = new draw2d.util.ArrayList();
       
        this.commandStack.markSaveLocation();
        
        // INTERSECTION/CROSSING handling for connections and lines
        //
        this.linesToRepaintAfterDragDrop =  new draw2d.util.ArrayList();
        this.lineIntersections = new draw2d.util.ArrayList();
        
        // Inform all listener that the selection has been cleanup. Normally this will be done
        // by the edit policies of the canvas..but exceptional this is done in the clear method as well -
        // Design flaw.
        this.selectionListeners.each(function(i,w){
            w.onSelectionChanged(null);
        });
    },
    
    /**
     * @method
     * 
     * Install a new selection and edit policy into the canvas
     * 
     * @since 2.2.0
     * @param {draw2d.policy.EditPolicy} policy
     */
    installEditPolicy: function(policy){
        // a canvas can handle only one selection policy
        //
        if(policy instanceof draw2d.policy.canvas.SelectionPolicy){
            // reset old selection before install new selection strategy
            this.getSelection().getAll().each(function(i,figure){
                figure.unselect();
            });
            
            // remove existing selection policy
            this.editPolicy.grep($.proxy(function(p){
                var stay = !(p instanceof draw2d.policy.canvas.SelectionPolicy); 
                if(stay===false){
                    p.onUninstall(this);
                }
                return stay;
            },this));
        }
        // only one SnapToXYZ edit policy at once
        else if (policy instanceof draw2d.policy.canvas.SnapToEditPolicy){
            // remove existing snapTo policy
            this.editPolicy.grep($.proxy(function(p){
                var stay = !(p instanceof draw2d.policy.canvas.SnapToEditPolicy); 
                if(stay===false){
                    p.onUninstall(this);
                }
                return stay;
            },this));
        }
        
        policy.onInstall(this);
        this.editPolicy.add(policy);    
    },
    
    /**
     * @method
     * 
     * UnInstall the selection and edit policy from the canvas.
     * 
     * @since 2.2.0
     * @param {draw2d.policy.EditPolicy} policy
     */
    uninstallEditPolicy: function(policy){
        if(!(policy instanceof draw2d.policy.EditPolicy)){
            return; // silently
        }
        
        this.editPolicy.grep($.proxy(function(p){
            if(p === policy || (p.NAME === policy.NAME)){
                p.onUninstall(this);
                return false;
            }
            return true;
        },this));
    },
    
    /**
     * @method
     * Set the new zoom factor for the canvas. The value must be between [0.01..10]
     * 
     * @param {Number} zoomFactor new zoom factor.
     * @param {boolean} [animated] set it to true for smooth zoom in/out
     */
    setZoom : function(zoomFactor, animated)
    {
        var _zoom = $.proxy(function(z){
            this.zoomFactor = Math.min(Math.max(0.01,z),10);
            
            var viewBoxWidth  = (this.initialWidth*(this.zoomFactor))|0;
            var viewBoxHeight = (this.initialHeight*(this.zoomFactor))|0;
            
            this.paper.setViewBox(0, 0, viewBoxWidth, viewBoxHeight);
            
            // BUG: raphael didn't handle setViewBox AND setSize correct
//            var paintArea =this.html.children(":first");
//            this.paper.setSize(this.html.width(), this.html.height());
            
            // didn't work too....   :-(
//            paintArea.width(this.initialWidth * this.zoomFactor);
//            paintArea.height(this.initialHeight * this.zoomFactor);
        },this);
        
       if(animated){
           var myTweenable = new Tweenable();
           myTweenable.tween({
             from:     { 'x': this.zoomFactor  },
             to:       { 'x': zoomFactor },
             duration: 300,
             easing : "easeOutSine",
             step: function (params) {
               _zoom(params.x);
             }
           });
       }
       else{
           _zoom(zoomFactor);
       }
    },

    /**
     * @method
     * Return the current zoom factor of the canvas.
     * 
     * @returns {Number}
     */
    getZoom: function(){
        return this.zoomFactor;
    },
    
    
    /**
     * @method
     * Transforms a document coordinate to canvas coordinate.
     * 
     * @param {Number} x the x coordinate relative to the window 
     * @param {Number} y the y coordinate relative to the window
     * 
     * @returns {draw2d.geo.Point} The coordinate in relation to the canvas [0,0] position
     */
    fromDocumentToCanvasCoordinate : function(x, y) {
        return new draw2d.geo.Point(
                (x - this.getAbsoluteX() + this.getScrollLeft())*this.zoomFactor,
                (y - this.getAbsoluteY() + this.getScrollTop())*this.zoomFactor);
    },
  
    _fromDocumentToCanvasCoordinate_IE8_HACK : function(x, y) {
   //     alert("dd");
            return new draw2d.geo.Point(
                    (x - this.getAbsoluteX())*this.zoomFactor,
                    (y - this.getAbsoluteY())*this.zoomFactor);
    },

    /**
     * @method
     * Transforms a canvas coordinate to document coordinate.
     * 
     * @param {Number} x the x coordinate in the canvas 
     * @param {Number} y the y coordinate in the canvas
     * 
     * @returns {draw2d.geo.Point} the coordinate in relation to the document [0,0] position
     */
    fromCanvasToDocumentCoordinate : function(x,y) {
        return new draw2d.geo.Point(
                ((x*(1/this.zoomFactor)) + this.getAbsoluteX() - this.getScrollLeft()),
                ((y*(1/this.zoomFactor)) + this.getAbsoluteY() - this.getScrollTop()));
    },
    
    /**
     * @method
     * The DOM host of the canvas
     * 
     * @returns {HTMLElement}
     */
    getHtmlContainer: function(){
       return this.html; 
    },
    
    
    /**
     * @method
     * Return a common event object independed if we run on an iPad or desktop.
     * 
     * @param event
     * @return
     * @private
     */
    _getEvent:function(event){
      // check for iPad, Android touch events
      //
      if(typeof event.originalEvent !== "undefined"){  
          if(event.originalEvent.touches && event.originalEvent.touches.length) {
               return event.originalEvent.touches[0];
          } else if(event.originalEvent.changedTouches && event.originalEvent.changedTouches.length) {
               return event.originalEvent.changedTouches[0];
          }
      }
      return event;
    },

    /**
     * @method
     * 
     * Set the area which are scrolling the canvas. This can be a jquery selector or 
     * a jQuery node.
     * 
     * @param {String/HTMLElement} elementSelector
     **/
    setScrollArea:function(elementSelector)
    {
       this.scrollArea= $(elementSelector);
    },

    /**
     * @method
     * 
     * return the scrolling area of the canvas. This is jQuery object
     * 
     * @return {HTMLElement} 
     **/
    getScrollArea:function()
    {
       return this.scrollArea;
    },

    /**
     * @method
     * The left scroll position.
     * 
     * @return {Number} the left scroll offset of the canvas
     **/
    getScrollLeft:function()
    {
      return this.scrollArea.scrollLeft();
    },

    /**
     * @method
     * The top scroll position
     * 
     * @return {Number} the top scroll offset of the cnavas.
     **/
    getScrollTop:function()
    {
      return this.scrollArea.scrollTop();
    },

    /**
     * @method
     * The absolute document x offset.
     *
     * @return {Number}
     **/
    getAbsoluteX:function()
    {
        return this.html.offset().left;
    },

    /**
     * @method
     * The absolute document y offset.
     * 
     * @return {Number} 
     **/
    getAbsoluteY:function()
    {
      return this.html.offset().top;
    },


    /**
     * @method
     * Return the width of the canvas
     * 
     * @return {Number}
     **/
    getWidth : function(){
        return this.html.width();
    },


    /**
     * @method
     * Return the height of the canvas.
     * 
     * @return {Number}
     **/
    getHeight:function() {
      return this.html.height();
    },
 

    /**
     * @method
     * Add a figure at the given x/y coordinate.
     *
     * @param {draw2d.Figure} figure The figure to add.
     * @param {Number} [x] The x position.
     * @param {Number} [y] The y position.
     **/
    addFigure:function( figure , x,  y)
    {
        if(figure.getCanvas()===this){
            return;
        }
        
      figure.setCanvas(this);

      // important inital 
      figure.getShapeElement();


      if(figure instanceof draw2d.shape.basic.Line){
        this.lines.add(figure);
        this.linesToRepaintAfterDragDrop = this.lines;
      }
      else{
        this.figures.add(figure);

        if(typeof y !== "undefined"){
            figure.setPosition(x,y);
        }
      }
      
      // init a repaint of the figure. This enforce that all properties
      // ( color, dim, stroke,...) will be set.
      figure.repaint();
      figure.fireMoveEvent();
    },

    /**
     * @method
     * Remove a figure from the Canvas.
     *
     * @param {draw2d.Figure} figure The figure to remove
     **/
    removeFigure:function(figure){
        // remove the figure froma selection handler as well and cleanup the 
        // selection feedback 
        this.editPolicy.each($.proxy(function(i,policy){
            if(typeof policy.unselect==="function"){
                policy.unselect(this,figure);
            }
        },this));
        

        if(figure instanceof draw2d.shape.basic.Line){
           this.lines.remove(figure);
         }
        else {
           this.figures.remove(figure);
        }
        figure.setCanvas(null);

        if(figure instanceof draw2d.Connection){
           figure.disconnect();
        }

    },
    
    /**
     * @method
     * Returns all lines/connections in this workflow/canvas.<br>
     *
     * @protected
     * @return {draw2d.util.ArrayList}
     **/
    getLines:function()
    {
      return this.lines;
    },

    /**
     * @method
     * Returns the internal figures container.<br>
     *
     * @protected
     * @return {draw2d.util.ArrayList}
     **/
    getFigures:function()
    {
      return this.figures;
    },

    /**
     * @method
     * Returns the line with the given id.
     *
     * @param {String} id The id of the line.
     * 
     * @type draw2d.shape.basic.Line
     **/
    getLine:function( id)
    {
      var count = this.lines.getSize();
      for(var i=0; i<count;i++)
      {
         var line = this.lines.get(i);
         if(line.getId()===id){
            return line;
         }
      }
      return null;
    },

    /**
     * @method
     * Returns the figure with the given id. 
     *
     * @param {String} id The id of the figure.
     * @return {draw2d.Figure}
     **/
    getFigure:function(/*:String*/ id)
    {
      var figure = null;
      this.figures.each(function(i,e){
          if(e.id===id){
              figure=e;
              return false;
           }
      });
      return figure;
    },

    /**
     * @method
     * Return all intersections draw2d.geo.Point between the given line and all other
     * lines in the canvas.
     * 
     * @param {draw2d.shape.basic.Line} line the line for the intersection test
     * @return {draw2d.util.ArrayList} 
     */
    getIntersection:function(line){
       var result = new draw2d.util.ArrayList();
       
       this.lineIntersections.each($.proxy(function(i, entry){
           if(entry.line ===line){
               entry.intersection.each(function(i,p){
                   result.add({x:p.x, y:p.y, justTouching:p.justTouching, other:entry.other});
               });
           }
       },this));
       
       return result;
    },
    



    /** 
     * @method
     *  Adjust the coordinate with the installed SnapToHelper.
     *
     * @param  {draw2d.Figure} figure The related figure
     * @param  {draw2d.geo.Point} pos The position to adjust
     * 
     * @return {draw2d.geo.Point} the adjusted position
     * @private
     **/
    snapToHelper:function(figure,  pos)
    {
        this.editPolicy.each($.proxy(function(i,policy){
            pos = policy.snap(this, figure, pos);
        },this));

        return pos;
    },


    /**
     * @method
     * Register a port to the canvas. This is required for other ports to find a valid drop target.
     * 
     * @param {draw2d.Port} port The new port which has been added to the Canvas.
     **/
    registerPort:function(port )
    {
      // All elements have the same drop targets.
      //
      port.targets= this.dropTargets;
      if(!this.commonPorts.contains(port)){
          this.commonPorts.add(port);
          this.dropTargets.add(port);
      }
    },

    /**
     * @method
     * Remove a port from the internal cnavas registration. Now other ports can't find the
     * port anymore as drop target. The port itself is still visible.
     * 
     * @param {draw2d.Port} p The port to unregister as potential drop target
     * @private
     **/
    unregisterPort:function(port )
    {
        port.targets=null;

        this.commonPorts.remove(port);
        this.dropTargets.remove(port);
    },

    /**
     * @method
     * Return all ports in the canvas
     * 
     */
    getAllPorts: function(){
        return this.commonPorts;
    },
    
    /**
     * @method
     * Returns the command stack for the Canvas. Required for undo/redo support.
     *
     * @return {draw2d.command.CommandStack}
     **/
    getCommandStack:function()
    {
      return this.commandStack;
    },

    /**
     * @method
     * Returns the current selected figure in the Canvas.
     *
     * @return {draw2d.Figure}
     * @deprecated
     **/
    getCurrentSelection:function()
    {
      return this.selection.getPrimary();
    },
    
    /**
     * @method
     * Returns the current selection.
     *
     * @return {draw2d.Selection}
     **/
    getSelection:function()
    {
      return this.selection;
    },

    /**
     * @method
     * Set the current selected figure in the workflow Canvas.
     *
     * @param {draw2d.Figure} figure The new selection.
     * @deprecated
     **/
    setCurrentSelection:function( figure )
    {
        this.selection.each($.proxy(function(i,e){
            this.editPolicy.each($.proxy(function(i,policy){
                if(typeof policy.unselect==="function"){
                    policy.unselect(this,e);
                }
            },this));
        },this));
 
        this.editPolicy.each($.proxy(function(i,policy){
            if(typeof policy.select==="function"){
                policy.select(this,figure);
            }
        },this));
    },

    /**
     * @method
     * Register a listener to the Canvas. The listener must provide a function "onSelectionChanged".
     * 
     * @param {Object/Function} w an object which implements the 'onSelectionChanged' method or a callback function
     **/
    addSelectionListener:function(w)
    {
      if(w!==null)
      {
        if(typeof w ==="function"){
          this.selectionListeners.add({onSelectionChanged: w});
        } 
        else if(typeof w.onSelectionChanged==="function"){
          this.selectionListeners.add(w);
        }
        else{
          throw "Object doesn't implement required callback method [onSelectionChanged]";
        }
      }
    },

    /**
     * @method
     * unregister the listener from the canvas.
     * 
     * @param {Object/Function} w The object which will be removed from the selection eventing
     **/
    removeSelectionListener:function(/*:Object*/ w )
    {
      this.selectionListeners = this.selectionListeners.grep(function(listener){
          return listener !== w && listener.onSelectionChanged!==w;
      });
    },


    /**
     * @method
     * Returns the best figure at the location [x,y]. It is a simple hit test. Keep in mind that only visible objects 
     * are returned.
     *
     * @param {Number} x The x position.
     * @param {Number} y The y position.
     * @param {draw2d.Figure} [figureToIgnore] The figure which should be ignored.
     **/
    getBestFigure : function(x, y, figureToIgnore)
    {
        var result = null;
        var testFigure = null;
        var i=0;
        var children = null;
        
        // ResizeHandles first
        for ( i = 0, len = this.resizeHandles.getSize(); i < len; i++)
        {
            testFigure = this.resizeHandles.get(i);
            if (testFigure.isVisible()===true && testFigure.hitTest(x, y) === true && testFigure !== figureToIgnore) 
            { 
                return testFigure; 
            }
        }

        // Checking ports
        for ( i = 0, len = this.commonPorts.getSize(); i < len; i++) 
        {
            testFigure = this.commonPorts.get(i);
            if(testFigure !== figureToIgnore)
            {
                if (testFigure.isVisible()===true && testFigure.hitTest(x, y) === true) 
                { 
                    return testFigure; 
                }
            }
        }

        // 2.) A line is the next option in the priority queue for a "Best" figure
        //
        result = this.getBestLine(x,y,figureToIgnore);
        if(result !==null){
            return result;
        }
        
        // 3.) Check now the common objects
        //     run from back to front to aware the z-oder of the figures
        for ( i = (this.figures.getSize()-1); i >=0; i--)
        {
            var figure = this.figures.get(i);
            // check first a children of the figure
            //
            var checkRecursive = function(children){
                children.each(function(i,e){
                    checkRecursive(e.getChildren());
                    if(result===null&&e.isVisible()===true && e.hitTest(x,y)===true){
                        result = e;
                    }
                    return result===null; // break the each-loop if we found an element
                });
            };
            checkRecursive( figure.getChildren());
            
            // ...and the figure itself
            //
            if (result ===null && figure.isVisible()===true && figure.hitTest(x, y) === true && figure !== figureToIgnore)
            {
                if (result === null){
                    result = figure;
                }
                else if(result.getZOrder()< figure.getZOrder())  {
                    result = figure;
                }
            }

            if(result !==null){
                return result;
            }
        }
        
        // 4.) Check the children of the lines as well
        //     Not selectable/draggable. But should receive onClick/onDoubleClick events 
        //      as well.
        var count = this.lines.getSize();
        for(i=0;i< count;i++)
        {
          var line = this.lines.get(i);
          children= line.getChildren();
          children.each(function(i,e){
              if(e.isVisible()===true && e.hitTest(x,y)===true){
                  result = e;
                  return false;
              }
              return true;
          });
        }
        
       return result;
    },


    /**
     * @method
     * Return the line which match the hands over coordinate
     *
     * @param {Number} x the x-coordinate for the hit test
     * @param {Number} y the x-coordinate for the hit test
     * @param {draw2d.shape.basic.Line} [lineToIgnore] a possible line which should be ignored for the hit test
     *
     * @private
     * @return {draw2d.shape.basic.Line}
     **/
    getBestLine:function( x,  y,  lineToIgnore)
    {
      var result = null;
      var count = this.lines.getSize();

      for(var i=0;i< count;i++)
      {
        var line = this.lines.get(i);
        if(line.isVisible()===true && line.hitTest(x,y)===true && line!==lineToIgnore)
        {
            if(result===null){
               result = line;
               break;
            }
        }
      }
      return result;
    }, 


    /**
     * @private
     **/
    hideSnapToHelperLines:function()
    {
      this.hideSnapToHelperLineHorizontal();
      this.hideSnapToHelperLineVertical();
    },

    /**
     * @private
     **/
    hideSnapToHelperLineHorizontal:function()
    {
    },

    /**
     * @private
     **/
    hideSnapToHelperLineVertical:function()
    {

    },


    /**
     * @method
     * Called by the framework during drag&drop operations.<br>
     * Droppable can be setup with:
     * <pre>
     *     $(".draw2d_droppable").draggable({
     *          appendTo:"#container",
     *          stack:"#container",
     *          zIndex: 27000,
     *          helper:"clone",
     *          start: function(e, ui){$(ui.helper).addClass("shadow");}
     *     });
     * </pre>
     * Graphiti use the jQuery draggable/droppable lib. Please inspect
     * http://jqueryui.com/demos/droppable/ for further information.
     * 
     * @param {HTMLElement} draggedDomNode The DOM element which is currently dragging
     * 
    * @template
     **/
    onDragEnter : function( draggedDomNode )
    {
    },
 
    
    /**
     * @method
     * Called if the DragDrop object is moving around.<br>
     * <br>
     * Graphiti use the jQuery draggable/droppable lib. Please inspect
     * http://jqueryui.com/demos/droppable/ for further information.
     * 
     * @param {HTMLElement} draggedDomNode The dragged DOM element.
     * @param {Number} x the x coordinate of the drag
     * @param {Number} y the y coordinate of the drag
     * 
     * @template
     **/
    onDrag:function(draggedDomNode, x, y )
    {
    },

        
    /**
     * @method
     * Called if the DragDrop object leaving the current hover figure.<br>
     * <br>
     * Graphiti use the jQuery draggable/droppable lib. Please inspect
     * http://jqueryui.com/demos/droppable/ for further information.
     * 
     * @param {HTMLElement} draggedDomNode The figure which is currently dragging
     * 
     * @template
     **/
    onDragLeave:function( draggedDomNode )
    {
    },

    
    /**
     * @method
     * Called if the user drop the droppedDomNode onto the canvas.<br>
     * <br>
     * Graphiti use the jQuery draggable/droppable lib. Please inspect
     * http://jqueryui.com/demos/droppable/ for further information.
     * 
     * @param {HTMLElement} droppedDomNode The dropped DOM element.
     * @param {Number} x the x coordinate of the drop
     * @param {Number} y the y coordinate of the drop
     * 
     * @template
     **/
    onDrop:function(droppedDomNode, x, y)
    {
    },
    
    /**
     * @method
     * Callback if the user press a key
     * 
     * @param {Number} keyCode the pressed key
     * @param {Boolean} ctrl true if the CTRL key is pressed as well
     * @private
     **/
    onKeyDown:function(keyCode, ctrl)
    {
      // Figure loescht sich selbst, da dies den KeyDown Event empfangen
      // kann. Bei einer Linie geht dies leider nicht, und muss hier abgehandelt werden.
      //
      if(keyCode==46 && this.selection.getPrimary()!==null){
         this.commandStack.execute(this.selection.getPrimary().createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.DELETE)));
      }
      /*
      else if(keyCode==90 && ctrl){
         this.commandStack.undo();
      }
      else if(keyCode==89 && ctrl){
         this.commandStack.redo();
      }
      else if(keyCode ===107){
          this.setZoom(this.zoomFactor*0.95);
      }
      else if(keyCode ===109){
          this.setZoom(this.zoomFactor*1.05);
      }
      */
    },

    /**
     * @private
     **/
    onDoubleClick : function(/* :int */x, /* :int */y, shiftKey, ctrlKey)
    {
        // check if a line has been hit
        //
        var figure = this.getBestFigure(x, y);

        if(figure!==null){
            figure.onDoubleClick();
        }
        // forward the event to all install policies as well.
        // (since 4.0.0)
        this.editPolicy.each($.proxy(function(i,policy){
            policy.onDoubleClick(figure, x,y, shiftKey, ctrlKey);
        },this));
    },

    /**
     * @param {Boolean} shiftKey true if the shift key has been pressed during this event
     * @param {Boolean} ctrlKey true if the ctrl key has been pressed during the event
     * @private
     **/
    onClick : function(/* :int */x, /* :int */y, shiftKey, ctrlKey)
    {
        // check if a figure has been hit
        //
        var figure = this.getBestFigure(x, y);
        
        // or a line/connection. May we should test the line before a figure..?
        // (since 4.0.0)
        if(figure===null){
            figure = this.getBestLine(x,y);
        }
        
        if(figure!==null){
            figure.onClick();
            
        }
        // forward the event to all install policies as well.
        // (since 3.0.0)
        this.editPolicy.each($.proxy(function(i,policy){
            policy.onClick(figure, x,y, shiftKey, ctrlKey);
        },this));

    },

    /**
     * @method
     * The user has triggered a right click. Redirect them to a responsible figure
     * 
     * @param {Number} x The x-coordinate of the click
     * @param {Number} y The y-coordinate of the click
     * 
     * @private
     * @since 1.1.0
     **/
    onRightMouseDown : function(x, y)
    {
       var figure = this.getBestFigure(x, y);
        if(figure!==null){
            figure.onContextMenu(x,y);
        }
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.Selection
 * 
 * Represents the current selection in the canvas. The selection element is a pure passive element which 
 * manage/store the selection.
 * 
 * 
 * @author Andreas Herz
 */
draw2d.Selection = Class.extend({
    
    NAME : "draw2d.Selection",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init: function() {
        this.primary = null;
        this.all = new draw2d.util.ArrayList();
    },

    /**
     * Reset the current selection
     * 
     */
    clear: function(){
        this.primary = null;
        this.all = new draw2d.util.ArrayList();
    },
    
    /**
     * @method
     * Return the primary selection. This can only one figure at once.
     * 
     * @return {draw2d.Figure} the primary selected figure
     */
    getPrimary: function(  )
    {
        return this.primary;
    },
 
    /**
     * @method
     * Set the primary selection.
     * 
     * @param {draw2d.Figure} figure The new primary selection
     */
    setPrimary: function(figure){
        this.primary = figure;
        this.add(figure);
    },
    
    /**
     * @method
     * Remove the given figure from the selection (primary,all)
     * 
     * @param {draw2d.Figure} figure
     */
    remove: function(figure){
        this.all.remove(figure);
        if(this.primary===figure){
            this.primary = null;
        }
    },

    add: function(figure){
        if(figure!==null && !this.all.contains(figure)){
            this.all.add(figure);
        }
    },
 
    
    /**
     * @method
     * return true if the given figure part of the selection
     * 
     * @param {draw2d.Figure} figure The figure to check
     * @since 2.2.0
     */
    contains: function(figure){
        return this.all.contains(figure);
    },
    
    /**
     * @method
     * Return the complete selection - including the primary selection.
     * 
     */
    getAll: function()
    {
        return this.all.clone();
    },
    
    /**
     * @method
     * @param func
     */
    each: function( func){
        this.all.each(func);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.Figure
 * A lightweight graphical object. Figures are rendered to a {@link draw2d.Canvas} object.
 * 
 * @inheritable
 * @author Andreas Herz
 */
draw2d.Figure = Class.extend({
    
	NAME : "draw2d.Figure",
    
	MIN_TIMER_INTERVAL: 50, // minimum timer interval in milliseconds
	
    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     * @param {Number} [width] initial width of the shape
     * @param {Number} [height] initial height of the shape
     */
    init: function( width, height ) {
        this.id = draw2d.util.UUID.create();
        
        // required in the SelectionEditPolicy to indicate the type of figure
        // which the user clicks
        this.isResizeHandle=false;
        
        // for undo/redo operation. It holds the command during a drag/drop operation
        // and execute it on the CommandStack if the user drop the figure.
        this.command = null;
        
        this.canvas = null;
        this.shape  = null;
        
        // possible decorations ( e.g. a Label) of the Connection
        // children are fixed bounded the figure. Most of the events of the child will bee
        // routed to the parent
        this.children = new draw2d.util.ArrayList();
            
        // behavior flags
        //
        this.selectable = true;
        this.deleteable = true;
        this.resizeable = true;
        this.draggable = true;
        this.visible = true;
        // since 4.1.0. 
        this.keepAspectRatio = false; 
        
        this.canSnapToHelper = true;
        this.snapToGridAnchor = new draw2d.geo.Point(0,0);    // hot spot for snap to grid  
        this.editPolicy = new draw2d.util.ArrayList();
        
        // timer for animation or automatic update
        //
        this.timerId = -1;
        this.timerInterval = 0;
        
        // possible parent of the figure. 
        // @see: this.children
        this.parent = null;
        
        // generic handle for the JSON read/write of user defined data
        this.userData = null;
       
        // appearance, position and dim properties
        //
        this.x = 0;
        this.y = 0;
        this.minHeight = 5;
        this.minWidth = 5;
        this.rotationAngle = 0;
        // add the name of the class to the css attribute
        this.cssClass = this.NAME.replace(new RegExp("[.]","g"), "_");
       
        if(typeof height !== "undefined"){
            this.width  = width;
            this.height = height;
        }
        else{
           this.width  = this.getMinWidth();
           this.height = this.getMinHeight();
        }
        this.alpha = 1.0;
        
        // internal status flags for the Drag&Drop operation handling and other stuff
        // 
        this.isInDragDrop =false;
        this.originalAlpha = this.alpha;
        this.ox = 0;
        this.oy = 0;
        this.repaintBlocked=false;
        this.selectionHandles = new draw2d.util.ArrayList();
        
        // listener for movement/resize. required for Ports or property panes in the UI
        //
        this.moveListener = new draw2d.util.ArrayList();
        this.resizeListener = new draw2d.util.ArrayList();

        
        this.installEditPolicy(new draw2d.policy.figure.RectangleSelectionFeedbackPolicy());
    },
    
    /**
     * @method
     * Add the figure to the current selection and propagate this to all edit policies.
     * 
     * @param {boolean} [isPrimarySelection] true if the element should be the primary selection
     * @final
     * @private
     */
    select: function(asPrimarySelection){
        if(typeof asPrimarySelection==="undefined"){
            asPrimarySelection=true;
        }
     
        // apply all EditPolicy for select Operations
        //
        this.editPolicy.each($.proxy(function(i,e){
              if(e instanceof draw2d.policy.figure.SelectionFeedbackPolicy){
                  e.onSelect(this.canvas, this,asPrimarySelection);
              }
        },this));

        if(this.canvas !==null){
            this.canvas.getSelection().add(this);
        }

        return this;
    },
    
    /**
     * @method
     * Unselect the figure and propagete this event to all edit policies.
     * 
     * @final
     * @private
     **/
    unselect:function()
    {
        // apply all EditPolicy for select Operations
        //
        this.editPolicy.each($.proxy(function(i,e){
              if(e instanceof draw2d.policy.figure.SelectionFeedbackPolicy){
                  e.onUnselect(this.canvas, this);
              }
        },this));

        if(this.canvas !==null){
            this.canvas.getSelection().remove(this);
        }
        
        return this;
    },
    
    /**
     * @method
     * Allows a user to attach (or remove) data to an element, without needing to create a custom figure or shape.
     * The data must be a valid JSON object.
     * 
     * @since 2.7.2
     * @param {Object} object
     */
    setUserData: function(object){
      this.userData = object;  
    },

    /**
     * @method
     * Returns any user data set previously on the given figure by setUserData.
     * 
     * @since 2.7.2
     * @returns {Object}
     */
    getUserData: function(){
        return this.userData;
    },
    
    /**
     * @method
     * Return the UUID of this element. 
     * 
     * @return {String}
     */
    getId: function()
    {
       return this.id; 
    },
    
    /**
     * @method
     * Set the id of this element. 
     * 
     * @param {String} id the new id for this figure
     */
    setId: function(id)
    {
        this.id = id; 

        return this;
    },
    

    /**
     * @method
     * Return the css styling class name of the element.
     * 
     * @return {String}
     */
    getCssClass: function()
    {
       return this.cssClass; 
    },
    
    /**
     * @method
     * Set the css class if the node.
     * 
     * @param {String} cssClass the new css class name of the node
     * @since 2.9.0
     */
    setCssClass: function(cssClass)
    {
        this.cssClass = cssClass===null?null:$.trim(cssClass);
        
        if(this.shape===null){
            return this;
        }
        
        if(this.cssClass===null){
            this.shape.node.removeAttribute("class");
        }
        else{
            this.shape.node.setAttribute("class", this.cssClass);
        }
                
        return this;
    },
    
    /**
     * @method
     * The method will return true if the class is assigned to the element, even if other classes also are.
     * 
     * @param {String} className the class name to check
     * @since 2.9.0
     */
    hasCssClass: function(className) {
        if(this.cssClass===null){
            return false;
        }
        
        return new RegExp(' ' + $.trim(className) + ' ').test(' ' + this.cssClass + ' ');
    },

    /**
     * @method
     * 
     * It's important to note that this method does not replace a class. It simply adds the class, 
     * appending it to any which may already be assigned to the elements.
     * 
     * @param {String} className
     * @since 2.9.0
     */
    addCssClass: function( className) {
        className = $.trim(className);
        if (!this.hasCssClass( className)) {
            if(this.cssClass===null){
                this.setCssClass(className);
            }
            else{
                this.setCssClass(this.cssClass + ' ' + className);
            }
        }
        
        return this;
    },

    /**
     * @method
     * 
     * Remove the given css class name from the figure
     * 
     * @param {String} className the css class name to add
     */
    removeCssClass:function(className) {
        className = $.trim(className);
        var newClass = ' ' + this.cssClass.replace( /[\t\r\n]/g, ' ') + ' ';
        if (this.hasCssClass(className)) {
            while (newClass.indexOf(' ' + className + ' ') >= 0 ) {
                newClass = newClass.replace(' ' + className + ' ', ' ');
            }
            this.setCssClass( newClass.replace(/^\s+|\s+$/g, ''));
        }
        
        return this;
    },
    
    /**
     * @method
     * 
     * Add or remove the given css class name from the figure
     * 
     * @param {String} className the class name to toggle
     */
    toggleCssClass:function( className) {
        className = $.trim(className);
        var newClass = ' ' + this.cssClass.replace( /[\t\r\n]/g, ' ' ) + ' ';
        if (this.hasCssClass( className)) {
            while (newClass.indexOf(' ' + className + ' ') >= 0 ) {
                newClass = newClass.replace( ' ' + className + ' ' , ' ' );
            }
            this.setCssClass( newClass.replace(/^\s+|\s+$/g, ''));
        } else {
            this.setCssClass(this.cssClass + ' ' + className);
        }
        
        return this;
    },

    /**
     * @method
     * Set the canvas element of this figures.
     * 
     * @param {draw2d.Canvas} canvas the new parent of the figure or null
     */
    setCanvas: function( canvas )
    {
      // remove the shape if we reset the canvas and the element
      // was already drawn
      if(canvas===null && this.shape!==null)
      {
         this.unselect();
         this.shape.remove();
         this.shape=null;
      }
    
      this.canvas = canvas;
      
      if(this.canvas!==null){
          this.getShapeElement();
      }

      if(canvas === null){
    	  this.stopTimer();
      }
      else{
    	  if(this.timerInterval>= this.MIN_TIMER_INTERVAL){
              this.startTimer(this.timerInterval);
    	  }
      }
      
      this.children.each(function(i,e){
          e.figure.setCanvas(canvas);
      });
      
      return this;
      
     },
     
     /**
      * @method
      * Return the current assigned canvas container.
      * 
      * @return {draw2d.Canvas}
      */
     getCanvas:function()
     {
         return this.canvas;
     },
     
    
     /**
      * @method
      * Start a timer which calles the onTimer method in the given interval.
      * 
      * @param {Number} milliSeconds
      */
     startTimer: function(milliSeconds)
     {
    	 this.stopTimer();
    	 this.timerInterval = Math.max(this.MIN_TIMER_INTERVAL, milliSeconds);
    	 
    	 if(this.canvas!==null){
    		 this.timerId = window.setInterval($.proxy(this.onTimer,this), this.timerInterval);
    	 }

    	 return this;
     },

     /**
      * @method
      * Stop the internal timer.
      * 
      */
     stopTimer: function()
     {
    	if(this.timerId>=0){
  		  window.clearInterval(this.timerId);
		  this.timerId=-1;
    	} 
    	
    	return this;
     },

     /**
      * @method
      * Callback method for the internal timer handling<br>
      * Inherit classes must override this method if they want use the timer feature.
      * 
      * @template
      */
     onTimer: function()
     {
    	
     },
     
     /**
      * @method
      * Moves the element so it is the closest to the viewer?��s eyes, on top of other elements. Additional
      * the internal model changed as well.
      * 
      * @since 3.0.0
      */
     toFront: function(){
         this.getShapeElement().toFront();
         if(this.canvas!==null){
             var figures = this.canvas.getFigures();
             var lines = this.canvas.getLines();
             if(figures.remove(this)!==null){
                 figures.add(this);
             }else if(lines.remove(this)!==null){
                 lines.add(this);
             }
         }
         
         // bring all children figures in front of the parent
         //
         this.children.each(function(i,child){
             child.figure.toFront();
         });
         return this;
     },
     
     
     /**
      * Install a new edit policy to the figure. Each editpolicy is able to focus on a single editing 
      * task or group of related tasks. This also allows editing behavior to be selectively reused across 
      * different figure implementations. Also, behavior can change dynamically, such as when the layouts 
      * or routing methods change.
      * Example for limited DragDrop behavior can be a draw2d.layout.constraint.RegionConstriantPolicy.
      * 
      * @aside example buildin_selection_feedback
      * @param {draw2d.policy.EditPolicy} policy
      */
     installEditPolicy: function(policy)
     {
         // it is only possible to install one SelectionFeedbackPolicy at once
         //
         if(policy instanceof draw2d.policy.figure.SelectionFeedbackPolicy){
             this.editPolicy.grep($.proxy(function(p){
                 var stay = !(p instanceof draw2d.policy.figure.SelectionFeedbackPolicy); 
                 if(!stay){
                     p.onUninstall(this);
                 }
                 return stay;
             },this));
         }
         policy.onInstall(this);
         this.editPolicy.add(policy);
         
         return this;
     },
     
     /**
      * @method
      * Add a child figure to the figure. The hands over figure doesn't support drag&drop 
      * operations. It's only a decorator for the connection.<br>
      * Mainly for labels or other fancy decorations :-)
      *
      * @param {draw2d.Figure} figure the figure to add as decoration to the connection.
      * @param {draw2d.layout.locator.Locator} locator the locator for the child. 
     **/
     addFigure : function(child, locator)
     {
         if(typeof locator ==="undefined" || locator ===null){
             throw "Second parameter 'locator' is requried for method 'Figure#addFigure'";
         }
         
         // the child is now a slave of the parent
         //
         child.setDraggable(false);
         child.setSelectable(false);
         child.setParent(this);
         
         this.children.add({figure:child, locator:locator});
         
         if(this.canvas!==null){
             child.setCanvas(this.canvas);
         }
         
         this.repaint();

         return this;
     },

     /**
      * @method
      * Remove the child figure from the this figure.
      *
      * @param {draw2d.Figure} figure the figure to remove.
      * @since 4.0.0
     **/
     removeFigure : function(child)
     {
         if(typeof child ==="undefined" || child ===null){
             debug.warn("The parameter child is required for Figure.removeFigure");
             return; // silently
         }
         
         child.setParent(null);
         child.setCanvas(null);
         
         var oldSize= this.children.getSize();
         this.children.grep(function(e){
             return e.figure!==child;
         });
         
         if(oldSize === this.children.getSize()){
             debug.warn("Child figure is not part of the shape. Remove not possible.");
         }
         
         this.repaint();

         return this;
     },

     
     /**
      * @method
      * Return all children/decorations of this shape
      */
     getChildren : function(){
         var shapes = new draw2d.util.ArrayList();
         this.children.each(function(i,e){
             shapes.add(e.figure);
         });
         
         return shapes;
     },
     
     
     /**
      * @method
      * Remove all children/decorations of this shape
      * 
      */
     resetChildren : function(){
         this.children.each(function(i,e){
             e.figure.setCanvas(null);
         });
         this.children= new draw2d.util.ArrayList();
         this.repaint();
         
         return this;
     },
     

     /**
     * @method
     * return the current SVG shape element or create it on demand.
     * 
     * @final
     */
    getShapeElement:function()
    {
       if(this.shape!==null){
         return this.shape;
       }

      this.shape=this.createShapeElement();
      
      // add CSS class to enable styling of the element with CSS rules/files
      //
      if(this.cssClass!==null){
          this.shape.node.setAttribute("class",this.cssClass);
      }
      
      return this.shape;
    },


    /**
     * @method
     * Inherited classes must override this method to implement it's own draw functionality.
     * 
     * @template
     * @abstract
     */
    createShapeElement : function()
    {
        throw "Inherited class ["+this.NAME+"] must override the abstract method createShapeElement";
    },


    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element
     * 
     **/
     repaint : function(attributes)
     {
// PERFORMANCE         
//         if (this.repaintBlocked===true || this.shape === null){
//             return;
//         }

         
         if(this.visible===true){
        	 this.shape.show();
         }
         else{
        	 this.shape.hide();
        	 return;
         }
         
         
         // enrich with common properties
         attributes.opacity = this.alpha;

         this.shape.attr(attributes);

         this.applyTransformation();

         /* moved to setDimension.
          * Locator is only called if the dimension of the figure has been changed
          * Performance
          */
        // Relocate all children of the figure.
        // This means that the Locater can calculate the new Position of the child.
        //
        this.children.each(function(i,e){
            e.locator.relocate(i, e.figure);
        });
        
     },
     
     /**
      * @private
      */
     applyTransformation:function(){
     },
     
     /**
      * @method
      * Highlight the element or remove the highlighting
      * 
      * @param {Boolean} flag indicates glow/noGlow
      * @template
      */
     setGlow: function(flag){
    	 // do nothing in the base class. 
    	 // Subclasses must implement this method.

         return this;
     },
     

    /**
     * @method
     * Will be called if the drag and drop action begins. You can return [false] if you
     * want avoid that the figure can be move.
     * 
     * @param {Number} relativeX the x coordinate within the figure
     * @param {Number} relativeY the y-coordinate within the figure.
     * 
     * @return {boolean} true if the figure accepts dragging
     **/
    onDragStart : function(relativeX, relativeY )
    {
      this.isInDragDrop =false;

      this.command = this.createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE));

      if(this.command!==null){
         this.ox = this.getX();
         this.oy = this.getY();
         this.isInDragDrop =true;
         
         // notify all installed policies
         //
         this.editPolicy.each($.proxy(function(i,e){
             if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
                 e.onDragStart(this.canvas, this);
             }
         },this));

         return true;
      }
      
      return false;
    },

    /**
     * @method
     * Don't call them manually. This will be done by the framework.<br>
     * Will be called if the object are moved via drag and drop.
     * Sub classes can override this method to implement additional stuff. Don't forget to call
     * the super implementation via <code>this._super(dx, dy, dx2, dy2);</code>
     * 
     * @private
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     **/
    onDrag : function( dx,  dy, dx2, dy2)
    {
      // apply all EditPolicy for DragDrop Operations
      //
      this.editPolicy.each($.proxy(function(i,e){
            if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
                var newPos = e.adjustPosition(this,this.ox+dx,this.oy+dy);
                dx = newPos.x-this.ox;
                dy = newPos.y-this.oy;
            }
      },this));
        
      this.x = this.ox+dx;
      this.y = this.oy+dy;

      // Adjust the new location if the object can snap to a helper
      // like grid, geometry, ruler,...
      //
      if(this.getCanSnapToHelper())
      {
        var p = new draw2d.geo.Point(this.x,this.y);
        p = this.getCanvas().snapToHelper(this, p);
        this.x = p.x;
        this.y = p.y;
      }

      this.setPosition(this.x, this.y);
      
      // notify all installed policies
      //
      this.editPolicy.each($.proxy(function(i,e){
          if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
              e.onDrag(this.canvas, this);
          }
      },this));
    },

    /**
     * @method
     * Called by the framework if the figure returns false for the drag operation. In this
     * case we send a "panning" event - mouseDown + mouseMove. This is very useful for
     * UI-Widget like slider, spinner,...
     * 
     * @param {Number} dx the x difference between the mouse down operation and now
     * @param {Number} dy the y difference between the mouse down operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @template
     */
    onPanning: function(dx, dy, dx2, dy2){
        
    },
    
    
    /**
     * @method
     * Will be called after a drag and drop action.<br>
     * Sub classes can override this method to implement additional stuff. Don't forget to call
     * the super implementation via <code>this._super();</code>
     * 
     * @template
     **/
    onDragEnd : function()
    {
      this.setAlpha(this.originalAlpha);
  
      // Element ist zwar schon an seine Position, das Command muss aber trotzdem
      // in dem CommandStack gelegt werden damit das Undo funktioniert.
      //
      this.command.setPosition(this.x, this.y);
      this.isInDragDrop = false;

      this.canvas.getCommandStack().execute(this.command);
      this.command = null;
     
      // notify all installed policies
      //
      this.editPolicy.each($.proxy(function(i,e){
          if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
              e.onDragEnd(this.canvas, this);
          }
      },this));

      this.fireMoveEvent();
    },

    /**
     * @method
     * Called by the framework during drag&drop operations.
     * 
     * @param {draw2d.Figure} draggedFigure The figure which is currently dragging
     * 
     * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didn't want a drop event
     * @template
     **/
    onDragEnter : function( draggedFigure )
    {
    	return null;
    },
 
    /**
     * @method
     * Called if the DragDrop object leaving the current hover figure.
     * 
     * @param {draw2d.Figure} figure The figure which is currently dragging
     * @template
     **/
    onDragLeave:function( draggedFigure )
    {
    },

    
    /**
     * @method
     * Called if the user drop this element onto the dropTarget
     * 
     * @param {draw2d.Figure} dropTarget The drop target.
     * @private
     **/
    onDrop:function(dropTarget)
    {
    },
   

    /**
     * @method
     * Callback method for the mouse enter event. Usefull for mouse hover-effects.
     * Override this method for your own effects. Don't call them manually.
     *
     * @template
     **/
    onMouseEnter:function()
    {
    },
    
    
    /**
     * @method
     * Callback method for the mouse leave event. Usefull for mouse hover-effects.
     * 
     * @template
     **/
    onMouseLeave:function()
    {
    },

    /**
     * @method
     * Called when a user dbl clicks on the element
     * 
     * @template
     * @aside example interaction_dblclick
     */
    onDoubleClick: function(){
    },
    
    
    /**
     * @method
     * Called when a user clicks on the element.
     * 
     * @template
     */
    onClick: function(){
    },
   
    /**
     * @method
     * called by the framework if the figure should show the contextmenu.<br>
     * The strategy to show the context menu depends on the plattform. Either loooong press or
     * right click with the mouse.
     * 
     * @param {Number} x the x-coordinate to show the menu
     * @param {Number} y the y-coordinate to show the menu
     * @since 1.1.0
     * @template
     */
    onContextMenu:function(x,y){

    },

    /**
     * @method
     * Set the alpha blending of this figure. 
     *
     * @param {Number} percent Value between [0..1].
     * @template
     **/
    setAlpha:function( percent)
    {
      percent = Math.min(1,Math.max(0,parseFloat(percent)));
      if(percent===this.alpha){
         return;
      }

      this.alpha =percent;
      this.repaint();

      return this;
    },

        
    /**
     * @method 
     * Return the alpha blending of the figure
     * 
     * @return {Number} the current alpha blending
     */
    getAlpha : function()
    {
        return this.alpha;
    },
    
    
    /**
     * @method
     * set the rotation angle in degree [0..356]<br>
     * <br>
     * <b>NOTE: this method is pre alpha and not for production.</b>
     * <br>
     * @param {Number} angle the rotation angle in degree
     */
    setRotationAngle: function(angle){
        this.rotationAngle = angle;
        

        // Update the resize handles if the user change the position of the element via an API call.
        //
        this.editPolicy.each($.proxy(function(i,e){
            if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
                e.moved(this.canvas, this);
            }
        },this));

        this.repaint();

        return this;
    },
    
    getRotationAngle : function(){
        return this.rotationAngle;
    },
    
    
    /**
     * @method
     * Show/hide the element. The element didn't receive any mouse events (click, dblclick) if you hide the
     * figure.
     * 
     * @param {Boolean} flag
     * @since 1.1.0
     */
    setVisible: function(flag){
    	this.visible = !!flag;
    	
    	this.repaint();

    	return this;
    },
    
    /**
     * @method
     * Return true if the figure visible.
     * 
     * @return {Boolean}
     * @since 1.1.0
     */
    isVisible: function(){
        return this.visible && this.shape!==null;
    },
    
    /**
     * @method
     * Guarantee, that the figure width/height will not be distorted. Applicable before calling setDimension().
     * It is false by default.
     * 
     * @since 4.1.0
     * @param {Boolean} flag boolean flag if the figure should respect the aspect ratio
     */
    setKeepAspectRatio: function( flag){
        this.keepAspectRatio = flag;
        return this;
    },
    
    /**
     * @method
     * Return the flag if the shape keep the aspect ratio.
     * 
     * @since 4.1.0
     */
    getKeepAspectRatio: function(){
        return this.keepAspectRatio;
    },
    
    /**
     * @method
     * Return the current z-index of the element. Currently this is an expensive method. The index will be calculated
     * all the time. Caching is not implemented at the moment.
     * 
     * @return {Number}
     */
    getZOrder: function(){
        if(this.shape===null){
            return -1;
        }
        
        var i = 0;
        var child = this.shape.node;
        while( (child = child.previousSibling) !== null ) {
          i++;
        }
        return i;
    },
    
    /**
     * @method
     * Set the flag if this object can snap to grid or geometry.
     * A window of dialog should set this flag to false.
     * 
     * @param {Boolean} flag The snap to grid/geometry enable flag.
     *
     **/
    setCanSnapToHelper:function(flag)
    {
      this.canSnapToHelper = !!flag;

      return this;
    },

    /**
     * @method
     * Returns true if the figure can snap to any helper like a grid, guide, geometrie
     * or something else.
     *
     * @return {boolean}
     **/
    getCanSnapToHelper:function()
    {
      return this.canSnapToHelper;
    },

    /**
     *
     * @return {draw2d.geo.Point}
     **/
    getSnapToGridAnchor:function()
    {
      return this.snapToGridAnchor;
    },

    /**
     * @method
     * Set the hot spot for all snapTo### operations.
     * 
     * @param {draw2d.geo.Point} point
     **/
    setSnapToGridAnchor:function(point)
    {
      this.snapToGridAnchor = point;
    },

    /**
     * @method
     * The current width of the figure.
     * 
     * @type {Number}
     **/
    getWidth:function()
    {
      return this.width;
    },

    /**
     * @method 
     * The current height of the figure.
     * 
     * @type {Number}
     **/
    getHeight:function()
    {
      return this.height;
    },


    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. width of this object.
     */
    getMinWidth:function()
    {
      return this.minWidth;
    },

    /**
     * @method
     * Set the minimum width of this figure
     * 
     * @param {Number} w
     */
    setMinWidth: function(w){
      this.minWidth = parseFloat(w);

      return this;
    },
    
    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. height of this object.
     */
    getMinHeight:function()
    {
      return this.minHeight;

      return this;
    },

    /**
     * @method
     * Set the minimum heigth of the figure.
     * 
     * @param {Number} h
     */
    setMinHeight:function(h){
        this.minHeight =parseFloat(h);

        return this;
    },
    
    /**
     * @method
     * The x-offset related to the parent figure or canvas.
     * 
     * @return {Number} the x-offset to the parent figure
     **/
    getX :function()
    {
        return this.x;
    },

    /**
     * @method
     * The y-offset related to the parent figure or canvas.
     * 
     * @return {Number} The y-offset to the parent figure.
     **/
    getY :function()
    {
        return this.y;
    },

    
    /**
     * @method
     * The x-offset related to the canvas.
     * 
     * @return {Number} the x-offset to the parent figure
     **/
    getAbsoluteX :function()
    {
        if(this.parent===null){
            // provide some good defaults if the figure not placed
            return this.getX()||0;
        }
        return this.getX() + this.parent.getAbsoluteX();  
    },


    /**
     * @method
     * The y-offset related to the canvas.
     * 
     * @return {Number} The y-offset to the parent figure.
     **/
    getAbsoluteY :function()
    {
        if(this.parent ===null){
            // provide some good defaults of the figure not placed
            return this.getY()||0;
        }
        return this.getY() + this.parent.getAbsoluteY();  
    },


    /**
     * @method
     * Returns the absolute y-position of the port.
     *
     * @type {draw2d.geo.Point}
     **/
    getAbsolutePosition:function()
    {
      return new draw2d.geo.Point(this.getAbsoluteX(), this.getAbsoluteY());
    },
    
    /**
     * @method
     * Returns the absolute y-position of the port.
     *
     * @return {draw2d.geo.Rectangle}
     **/
    getAbsoluteBounds:function()
    {
      return new draw2d.geo.Rectangle(this.getAbsoluteX(), this.getAbsoluteY(),this.getWidth(),this.getHeight());
    },
    

    /**
     * @method
     * Set the position of the object.
     *
     * @param {Number/draw2d.geo.Point} x The new x coordinate of the figure
     * @param {Number} [y] The new y coordinate of the figure 
     **/
    setPosition : function(x, y) {
        if (x instanceof draw2d.geo.Point) {
            this.x = x.x;
            this.y = x.y;
        }
        else {
            this.x = x;
            this.y = y;
        }
        
       this.repaint();
        
        // Update the resize handles if the user change the position of the
        // element via an API call.
        //
        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        this.fireMoveEvent();

        return this;
    },
    
    /**
     * @method
     * Get the current position of the figure 
     * 
     * @return {draw2d.geo.Point}
     * @since 2.0.0
     */
    getPosition: function(){
        return new draw2d.geo.Point(this.getX(), this.getY());
    },
    
    /**
     * @method
     * Translate the figure with the given x/y offset.
     *
     * @param {Number} dx The new x translate offset
     * @param {Number} dy The new y translate offset
     **/
    translate:function(dx , dy )
    {
    	this.setPosition(this.getX()+dx,this.getY()+dy);
    	
    	return this;
    },
    
    
    /**
     * @method
     * Set the new width and height of the figure. 
     *
     * @param {Number} w The new width of the figure
     * @param {Number} h The new height of the figure
     **/
    setDimension:function(w, h)
    {
        w = Math.max(this.getMinWidth(),w);
        h = Math.max(this.getMinHeight(),h);

        if(this.width === w && this.height ===h){
            return;
        }

        
    	// apply all EditPolicy to adjust/modify the new dimension
        //
        this.editPolicy.each($.proxy(function(i,e){
              if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
                  var newDim = e.adjustDimension(this,w,h);
                  w = newDim.w;
                  h = newDim.h;
              }
        },this));

        // respect the aspect ratio if required
        //
        if(this.keepAspectRatio===true){

            // resize the icon and keep the aspect ratio
            //
            var cw = this.getWidth();
            var ch = this.getHeight();
            if(w!==cw && h!==ch){
                h= ch;
            }

            // width is the same and height changed
            //
            if(h!==ch){
                w = w*(h/ch);
            }
            // width has changed and height is the same
            //
            else if(w !==cw){
                h = h*(w/cw);
            }
        }

        this.width = Math.max(this.getMinWidth(),w);
		this.height= Math.max(this.getMinHeight(),h);
		  
		this.repaint();

        this.fireResizeEvent();
        // just to be backward compatible....cost a lot of performance...still
		this.fireMoveEvent();
		
		// Update the resize handles if the user change the position of the element via an API call.
		//
		this.editPolicy.each($.proxy(function(i,e){
		   if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
		       e.moved(this.canvas, this);
		   }
		},this));

		return this;
    },


    /**
     * @method
     * Return the bounding box of the figure in absolute position to the canvas.
     * 
     * @return {draw2d.geo.Rectangle}
     **/
    getBoundingBox:function()
    {
      return new draw2d.geo.Rectangle(this.getAbsoluteX(),this.getAbsoluteY(),this.getWidth(),this.getHeight());
    },

    /**
     * @method
     * Detect whenever the hands over coordinate is inside the figure.
     * The default implementation is a simple bounding box test. 
     * 
     * @param {Number} iX
     * @param {Number} iY
     * @param {Number} [corona]
     * 
     * @returns {Boolean}
     */
    hitTest : function ( iX , iY, corona)
    {
        if(typeof corona === "number"){
            return this.getBoundingBox().scale(corona,corona).hitTest(iX,iY);
        }
        return this.getBoundingBox().hitTest(iX,iY);
    },


    /**
     * @method
     * Switch on/off the drag drop behaviour of this object
     *
     * @param {Boolean} flag The new drag drop indicator
     **/
    setDraggable:function(flag)
    {
      this.draggable= !!flag;

      return this;
    },

    /**
     * @method
     * Get the Drag drop enable flag
     *
     * @return {boolean} The new drag drop indicator
     **/
    isDraggable:function()
    {
      return this.draggable;
    },


    /**
     * @method
     * Returns the true if the figure can be resized.
     *
     * @return {boolean}
     **/
    isResizeable:function()
    {
      return this.resizeable;
    },

    /**
     * @method
     * You can change the resizeable behaviour of this object. Hands over [false] and
     * the figure has no resizehandles if you select them with the mouse.<br>
     *
     * @param {boolean} flag The resizeable flag.
     **/
    setResizeable:function(flag)
    {
      this.resizeable=!!flag;

      return this;
    },

    /**
     * @method
     * Indicates whenever the element is selectable by user interaction or API.
     * 
     * @return {boolean}
     **/
    isSelectable:function()
    {
      return this.selectable;
    },


    /**
     * @method
     * You can change the selectable behavior of this object. Hands over [false] and
     * the figure has no selection handles if you try to select them with the mouse.<br>
     *
     * @param {boolean} flag The selectable flag.
     **/
    setSelectable:function(flag)
    {
      this.selectable=!!flag;

      return this;
    },

    /**
     * @method
     * Return true if the object doesn't care about the aspect ratio.
     * You can change the height and width independent.<br>
     * 
     * Replaced with "getKeepAspectRatio"
     * @return {boolean}
     * @deprecated
     */
    isStrechable:function()
    {
      return !this.getKeepAspectRatio();
    },

    /**
     * @method
     * Return false if you avoid that the user can delete your figure.
     * Sub class can override this method.
     * 
     * @return {boolean}
     **/
    isDeleteable:function()
    {
      return this.deleteable;
    },

    /**
     * @method
     * Return false if you avoid that the user can delete your figure.
     * 
     * @param {boolean} flag Enable or disable flag for the delete operation
     **/
    setDeleteable:function(flag)
    {
      this.deleteable = !!flag;

      return this;
    },

    /**
     * @method
     * Set the parent of this figure.
     * Don't call them manually.

     * @param {draw2d.Figure} parent The new parent of this figure
     * @private
     **/
    setParent:function( parent)
    {
      this.parent = parent;

      return this;
    },

    /**
     * @method
     * Get the parent of this figure.
     *
     * @return {draw2d.Figure}
     **/
    getParent:function()
    {
      return this.parent;
    },

    /**
     * @method
     * Register the hands over object as a moveListener of this figure.<br>
     * All position changes will be broadcast to all move listener. This is
     * useful for Connectors and Layouter for router handling.
     *
     * @param {Object} listener the listener to call
     * @since 2.5.1
     **/
     attachResizeListener:function(listener) {
        if (listener === null) {
            return;
        }

        this.resizeListener.add(listener);

        return this;
     },
 

    /**
     * @method
     * Remove the hands over figure from notification queue.
     *
     * @param {draw2d.Figure} figure The figure to remove the monitor
     * @since 2.5.1
     **/
    detachResizeListener:function(figure) 
    {
      if(figure===null){
        return;
      }

      this.resizeListener.remove(figure);

      return this;
    },
    
    /**
     * @method
     * Called from the figure itself when any position changes happens. All listener
     * will be informed.
     * 
     * @private
     * @since 2.5.1
     **/
    fireResizeEvent: function()
    {
         this.resizeListener.each($.proxy(function(i, item){
            item.onOtherFigureIsResizing(this);
        },this));

        return this;
    },
    
    /**
     * @method
     * Fired if a figure is moving.
     *
     * @param {draw2d.Figure} figure The figure which has changed its position
     * @template
     * @since 2.5.1
     */
    onOtherFigureIsResizing:function(figure){
    },
    
    /**
     * @method
     * Register the hands over object as a moveListener of this figure.<br>
     * All position changes will be broadcast to all move listener. This is
     * useful for Connectors and Layouter for router handling.
     *
     * @param {Object} listener the listener to call
     *
     **/
     attachMoveListener:function(listener) {
		if (listener === null) {
			return;
		}

		if(!this.moveListener.contains(listener)){
	        this.moveListener.add(listener);
		}

		return this;
 	 },
 

    /**
     * @method
     * Remove the hands over figure from notification queue.
     *
     * @param {draw2d.Figure} figure The figure to remove the monitor
     *
     **/
    detachMoveListener:function(figure) 
    {
      if(figure===null){
        return;
      }

      this.moveListener.remove(figure);

      return this;
    },

    /**
     * @method
     * Called from the figure itself when any position changes happens. All listener
     * will be informed.
     * 
     * @private
     **/
    fireMoveEvent: function()
    {
        // first call. Reured for connections to update the routing,...
        //
        this.moveListener.each($.proxy(function(i, item){
            item.onOtherFigureIsMoving(this);
        },this));
        
        return this;
   },
    
    /**
     * @method
     * Fired if a figure is moving.
     *
     * @param {draw2d.Figure} figure The figure which has changed its position
     * @template
     */
    onOtherFigureIsMoving:function(figure){
    },
    
    /**
     * @method
     * Returns the Command to perform the specified Request or null.
      *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * @return {draw2d.command.Command} null or a Command
     **/
    createCommand:function( request)
    {
      if(request===null){
          return null;
      }
      
      if(request.getPolicy() === draw2d.command.CommandType.MOVE)
      {
        if(!this.isDraggable()){
          return null;
        }
        return new draw2d.command.CommandMove(this);
      }

      if(request.getPolicy() === draw2d.command.CommandType.DELETE)
      {
        if(!this.isDeleteable()){
           return null;
        }
        return new draw2d.command.CommandDelete(this);
      }
      
      if(request.getPolicy() === draw2d.command.CommandType.RESIZE)
      {
        if(!this.isResizeable()){
           return null;
        }
        return new draw2d.command.CommandResize(this);
      }
      
      return null;
    },
    
    /**
     * @method
     * Clone the figure. <br> 
     * You must override and implement the methods <b>getPersistentAttributes</b> and <b>setPersistentAttributes</b> for your custom
     * figures if the have special attributes.
     * 
     * @since 4.1.0
     */
    clone: function(){
        var clone = eval("new "+this.NAME+"();");
        var initialId = clone.id;

        clone.setPersistentAttributes( this.getPersistentAttributes());
        
        clone.id = initialId;
        return clone;
    },
    
    /**
     * @method 
     * Return an objects with all important attributes for XML or JSON serialization
     * 
     * @return
     */
    getPersistentAttributes : function(){
        // force deep copy of userData to avoid side effects in the clone method.
        //
        var memento= {
            type  : this.NAME,
            id    : this.id,
            x     : this.getX(),
            y     : this.getY(),
            width : this.width,
            height: this.height,
            userData: $.extend(true,{},this.userData)
        };

        
        if(this.cssClass!==null){
            memento.cssClass= this.cssClass;
        }
        
        return memento;
    },
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     */
    setPersistentAttributes : function(memento)
    {
        this.id    = memento.id;
        this.x     = parseFloat(memento.x);
        this.y     = parseFloat(memento.y);
        
        // width and height are optional parameter for the JSON stuff.
        // We use the defaults if the attributes not present
        if(typeof memento.width !== "undefined"){
            this.width = parseFloat(memento.width);
        }
        
        if(typeof memento.height !== "undefined"){
            this.height= parseFloat(memento.height);
        }
        
        if(typeof memento.userData !== "undefined"){
            this.userData= memento.userData;
        }

        if(typeof memento.cssClass !== "undefined"){
            this.setCssClass(memento.cssClass);
        }
        
        return this;
    }  

});



/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.Node
 * 
 * A Node is the base class for all figures which can have {@link draw2d.Port}s. A {@link draw2d.Port} is the
 * anchor for a {@link draw2d.Connection} line.<br><br>A {@link draw2d.Port} is a green dot which can 
 * be dragged and dropped over another port.<br>
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.Figure 
 */
draw2d.shape.node.Node = draw2d.Figure.extend({
 
	NAME : "draw2d.shape.node.Node",

   /**
     * @constructor
     * Creates a new Node element which are not assigned to any canvas.
     * 
     * @param {Number} [width] initial width of the shape
     * @param {Number} [height] initial height of the shape
    */
    init: function( width, height ) {
    
      this.inputPorts = new draw2d.util.ArrayList();
      this.outputPorts= new draw2d.util.ArrayList();
      this.hybridPorts= new draw2d.util.ArrayList();
      
      // Flags just for performance reasons
      //
      this.portRelayoutRequired = true;
      
      // just for performance reasons
      //
      this.cachedPorts = null;
      
      this._super( width, height);
    },
    

    onDoubleClick:function(){
        var w = this.getWidth();
        var h = this.getHeight();
        // rotate in 90�? increment steps..
        this.setRotationAngle((this.getRotationAngle()+90)%360);
        // ..and toggle the orientation of the shape (portrait / landscape)
        this.setDimension(h,w);
        
        this.portRelayoutRequired=true;
    },
    
    /**
     * @method
     * Moves the element so it is the closest to the viewer?��s eyes, on top of other elements. Additional
     * the internal model changed as well.
     * 
     * @since 3.0.0
     */
    toFront: function(){
        this._super();
        
        this.getPorts().each(function(i,port){
            port.getConnections().each(function(i,connection){
                connection.toFront();
            });
            port.toFront();
        });
        
        return this;
    },
    
    /**
     * @method
     * Return all ports of the node.
     *
     * @return  {draw2d.util.ArrayList}
     **/
    getPorts: function()
    {
      // TODO: expensive! Find another solution.
      if(this.cachedPorts===null ){
          this.cachedPorts = new draw2d.util.ArrayList();
          this.cachedPorts.addAll(this.inputPorts);
          this.cachedPorts.addAll(this.outputPorts);
          this.cachedPorts.addAll(this.hybridPorts);
          
          this.children.each($.proxy(function(i,e){
              this.cachedPorts.addAll( e.figure.getPorts());
          },this));
      }
              
      return this.cachedPorts;
    },
    
    
    /**
     * @method
     * Return all input ports of the node.
     *
     * @return {draw2d.util.ArrayList}
     **/
    getInputPorts: function()
    {
      return this.inputPorts
               .clone()
               .addAll(this.hybridPorts);
    },
    
    /**
     * @method
     * Return all output ports of the node.
     *
     * @return {draw2d.util.ArrayList}
     **/
    getOutputPorts: function()
    {
      return this.outputPorts
          .clone()
          .addAll(this.hybridPorts);
    },
    
    /**
     * @method
     * Return the port with the corresponding name.
     *
     * 
     * @param {String} portName The name of the port to return.
     * @return {draw2d.Port} Returns the port with the hands over name or null.
     **/
    getPort: function( portName)
    {
    	var port = null;
    	
        this.getPorts().each(function(i,e){
            
            if (e.getName() === portName) {
                port = e;
         		return false;
            }
        });
        
        return port;
    },
    
    /**
     * @method
     * Return the input port with the corresponding name.
     *
     * 
     * @param {String/Number} portNameOrIndex The name or numeric index of the port to return.
     * @return {draw2d.InputPort} Returns the port with the hands over name or null.
     **/
    getInputPort: function( portNameOrIndex)
    {
        if(typeof portNameOrIndex === "number"){
            return this.inputPorts.get(portNameOrIndex);
        }
        
        for ( var i = 0; i < this.inputPorts.getSize(); i++) {
            var port = this.inputPorts.get(i);
            if (port.getName() === portNameOrIndex) {
                return port;
            }
        }
      
        return null;
    },
    
    /**
     * @method
     * Return the output port with the corresponding name.
     *
     * @param {String/Number} portNameOrIndex The name or the numeric index of the port to return.
     * @return {draw2d.OutputPort} Returns the port with the hands over name or null.
     **/
    getOutputPort: function( portNameOrIndex)
    {
        if(typeof portNameOrIndex === "number"){
            return this.outputPorts.get(portNameOrIndex);
        }
        
         for ( var i = 0; i < this.outputPorts.getSize(); i++) {
            var port = this.outputPorts.get(i);
            if (port.getName() === portNameOrIndex) {
                return port;
            }
        }

        return null;
    },
    
    /**
     * @method
     * Return the input port with the corresponding name.
     *
     * 
     * @param {String/Number} portNameOrIndex The name or numeric index of the port to return.
     * @return {draw2d.InputPort} Returns the port with the hands over name or null.
     **/
    getHybridPort: function( portNameOrIndex)
    {
        if(typeof portNameOrIndex === "number"){
            return this.hybridPorts.get(portNameOrIndex);
        }
        
        for ( var i = 0; i < this.hybridPorts.getSize(); i++) {
            var port = this.hybridPorts.get(i);
            if (port.getName() === portNameOrIndex) {
                return port;
            }
        }
      
        return null;
    },
    
    /**
     * @method
     * Add a port to this node at the given position.<br>
     *
     * @param {draw2d.Port} port The new port to add.
     * @param {draw2d.layout.locator.Locator} locator The layouter for the port.
     **/
    addPort: function(port, locator)
    {
        if(!(port instanceof draw2d.Port)){
            throw "Argument is not typeof 'draw2d.Port'. \nFunction: draw2d.shape.node.Node#addPort";
        }

        // add to the internal cache if already build
        if(this.cachedPorts !== null){
        	this.cachedPorts.add(port);
        };
        
        this.portRelayoutRequired=true;
        
        
        if (port instanceof draw2d.InputPort) {
            this.inputPorts.add(port);
        }
        else if(port instanceof draw2d.OutputPort){
            this.outputPorts.add(port);
        }
        else if(port instanceof draw2d.HybridPort){
            this.hybridPorts.add(port);
        }

        if((typeof locator !== "undefined") && (locator instanceof draw2d.layout.locator.Locator)){
            port.setLocator(locator);
        }
        
        port.setParent(this);
        port.setCanvas(this.canvas);

        // You can't delete a port with the [DEL] key if a port is a child of a node
        port.setDeleteable(false);

        if (this.canvas !== null) {
            port.getShapeElement();
            this.canvas.registerPort(port);
        }
    },
    
    /**
     * @method
     * Removes a port and all related connections from this node.<br>
     *
     * @param {draw2d.Port} port The port to remove.
     **/
    removePort : function(port)
    {
        this.portRelayoutRequired=true;

        this.inputPorts.remove(port);
        this.outputPorts.remove(port);
        this.hybridPorts.remove(port);

        if (port.getCanvas() !== null) {
            port.getCanvas().unregisterPort(port);
            // remove the related connections of the port too.
            var connections = port.getConnections();
            for ( var i = 0; i < connections.getSize(); ++i) {
                port.getCanvas().removeFigure(connections.get(i));
            }
        }

        port.setCanvas(null);
    },
    
    /**
     * @method
     * Create a standard Port for this element. Inherited class can override this
     * method to create its own type of ports.
     * 
     * @param {String} type the type of the requested port. possible ["input", "output"]
     * @param {draw2d.layout.locator.Locator} [locator] the layouter to use for this port
     * @template
     */
    createPort: function(type, locator){
        var newPort = null;
        var count =0;
        
    	switch(type){
    	case "input":
    		newPort= new draw2d.InputPort();
    		count = this.inputPorts.getSize();
    		break;
    	case "output":
    		newPort= new draw2d.OutputPort();
            count = this.outputPorts.getSize();
    		break;
        case "hybrid":
            newPort= new draw2d.HybridPort();
            count = this.hybridPorts.getSize();
            break;
    	default:
            throw "Unknown type ["+type+"] of port requested";
    	}
    	
   	    newPort.setName(type+count);
    	
    	this.addPort(newPort, locator);
    	// relayout the ports
    	this.setDimension(this.width,this.height);
    	
        this.layoutPorts();

    	return newPort;
    },
    
    /**
     * @method
     * Return all connections related to this node.
     * 
     * @returns {draw2d.util.ArrayList}
     */
    getConnections: function(){
        var connections = new draw2d.util.ArrayList();
        var ports = this.getPorts();
        for(var i=0; i<ports.getSize(); i++)
        {
          var port = ports.get(i);
          // Do NOT add twice the same connection if it is linking ports from the same node
          for (var c = 0, c_size = port.getConnections().getSize() ; c< c_size ; c++)
          {
              if(!connections.contains(port.getConnections().get(c)))
              {
                connections.add(port.getConnections().get(c));
              }
          }
        }
        return connections;
    },

    /**
     * @private
     **/
    setCanvas : function(canvas)
    {
        var oldCanvas = this.canvas;
        this._super(canvas);
       
        var ports = this.getPorts();
        if (oldCanvas !== null) {
            ports.each(function(i,port){
                oldCanvas.unregisterPort(port);
            });
        }

        if (canvas !== null) {
            ports.each(function(i,port){
                port.setCanvas(canvas);
                canvas.registerPort(port);
            });
            // relayout the ports
            this.setDimension(this.width,this.height);
        }
        else {
            ports.each(function(i,port){
                port.setCanvas(null);
            });
        }
    },
    
    setRotationAngle: function(angle){
        this.portRelayoutRequired=true;
        this._super(angle);
        
        this.layoutPorts();
    },
    
    setDimension: function(w,h){
        this.portRelayoutRequired=true;
        this._super(w,h);
    },
    
    /**
     * @method
     * Called if the value of any port has been changed
     * 
     * @param {draw2d.Port} relatedPort
     * @template
     */
    onPortValueChanged: function(relatedPort){
    
    },
    
    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element
     * 
     **/
     repaint : function(attributes){
         if (this.repaintBlocked===true || this.shape === null){
             return;
         }
         this._super(attributes);
         this.layoutPorts();
     },
     
    /**
     * @method
     * 
     * @private
     */
     layoutPorts: function(){

         if(this.portRelayoutRequired===false){
             return;//silently
         }
         this.portRelayoutRequired=false;
         
         // layout the ports
         //
         this.outputPorts.each(function(i, port){
             port.locator.relocate(i,port);
         });
         
         this.inputPorts.each(function(i, port){
             port.locator.relocate(i,port);
         });
         
         this.hybridPorts.each(function(i, port){
             port.locator.relocate(i,port);
         });
     }
    
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.VectorFigure
 * The base class for all vector based figures like {@link draw2d.shape.basic.Rectangle}  or {@link draw2d.shape.basic.Oval} 
 * inside a canvas.
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.node.Node
 */
draw2d.VectorFigure = draw2d.shape.node.Node.extend({
    NAME : "draw2d.VectorFigure",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init : function( width, height)
    {
        this.stroke = 1;
        this.bgColor   = new  draw2d.util.Color(255,255,255);
        this.lineColor = new  draw2d.util.Color(128,128,255);
        this.color     = new  draw2d.util.Color(128,128,128);
           
        // memento for the stroke if we reset the glow effect of this shape
        //
        this.strokeBeforeGlow = this.stroke;
        this.glowIsActive = false;
        
        this._super( width, height);
    },
      
    
    /**
     * @method
     * Highlight the element or remove the highlighting
     * 
     * @param {Boolean} flag indicates glow/noGlow
     * @template
     */
    setGlow: function(flag){
        
        if(flag === this.glowIsActive) {
            return this;
        }
        
        this.glowIsActive = flag;
        if(flag===true){
            this.strokeBeforeGlow = this.getStroke();
            this.setStroke(this.strokeBeforeGlow*2.5);
        }
        else {
            this.setStroke(this.strokeBeforeGlow);
        }
        
        return this;
    },
    
   /**
    * @method
    * propagate all attributes like color, stroke,... to the shape element
    **/
    repaint : function(attributes)
    {
        if (this.repaintBlocked===true || this.shape === null){
            return;
        }

        if(typeof attributes === "undefined" ){
            attributes = {};
        }

        attributes.x = this.getAbsoluteX();
        attributes.y = this.getAbsoluteY();
        
        if(typeof attributes.stroke==="undefined"){
            if(this.color === null || this.stroke ===0){
                attributes.stroke = "none";
            }
            else {
                attributes.stroke = this.color.hash();
            }
        }
        
        if(typeof attributes["stroke-width"]==="undefined"){
            attributes["stroke-width"] = this.stroke;
        }
        
        if(typeof attributes.fill === "undefined"){
       	   attributes.fill = this.bgColor.hash();
        }

        this._super(attributes);
        
        return this;
    },


   /**
    * @method
    * Set the new background color of the figure. It is possible to hands over
    * <code>null</code> to set the background transparent.
    *
    * @param {draw2d.util.Color} color The new background color of the figure
    **/
    setBackgroundColor : function(color)
    {
        this.bgColor = new draw2d.util.Color(color);

        this.repaint();
        
        return this;
    },

   /**
    * @method
    * The current used background color.
    * 
    * @return {draw2d.util.Color}
    */
   getBackgroundColor:function()
   {
     return this.bgColor;
   },

   /**
    * @method
    * Set the stroke to use.
    * 
    * @param {Number} w The new line width of the figure
    **/
   setStroke:function( w )
   {
     this.stroke=w;
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * The used line width.
    * 
    * @type {Number}
    **/
   getStroke:function( )
   {
     return this.stroke;
   },

   /**
    * @method
    * Set the color of the line.
    * This method fires a <i>document dirty</i> event.
    * 
    * @param {draw2d.util.Color} color The new color of the line.
    **/
   setColor:function( color)
   {
     this.color = new draw2d.util.Color(color);
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * The current used foreground color
    * 
    * @returns {draw2d.util.Color}
    */
   getColor:function()
   {
     return this.color;
   },
   
   
   /**
    * @method 
    * Return an objects with all important attributes for XML or JSON serialization
    * 
    * @returns {Object}
    */
   getPersistentAttributes : function()
   {
       var memento= this._super();

       memento.bgColor = this.bgColor.hash();
       memento.color   = this.color.hash();
       memento.stroke  = this.stroke;
       memento.alpha   = this.alpha;
       
       return memento;
   },
   
   /**
    * @method 
    * Read all attributes from the serialized properties and transfer them into the shape.
    * 
    * @param {Object} memento
    * @return
    */
   setPersistentAttributes : function(memento)
   {
       this._super(memento);
       
       if(typeof memento.bgColor !== "undefined"){
           this.setBackgroundColor(memento.bgColor);
       }
       
       if(typeof memento.color !== "undefined"){
           this.setColor(memento.color);
       }
       
       if(typeof memento.stroke !== "undefined"){
           this.setStroke(parseFloat(memento.stroke));
       }
        
       if(typeof memento.alpha !== "undefined"){
           this.setAlpha(parseFloat(memento.alpha));
       }
        
       return this;
   }  


});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Rectangle
 * A Rectangle Figure.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var rect1 =  new draw2d.shape.basic.Rectangle();
 *     var rect2 =  new draw2d.shape.basic.Rectangle();
 *     
 *     canvas.addFigure(rect1,10,10);
 *     canvas.addFigure(rect2,100,10);
 *     
 *     rect2.setBackgroundColor("#f0f000");
 *     rect2.setAlpha(0.7);
 *     rect2.setDimension(100,60);
 *     rect2.setRadius(10);
 *     
 *     canvas.setCurrentSelection(rect2);
 *     
 * @author Andreas Herz
 * @extends draw2d.VectorFigure
 */
draw2d.shape.basic.Rectangle = draw2d.VectorFigure.extend({
    NAME : "draw2d.shape.basic.Rectangle",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init: function( width, height) {
       // corner radius
       this.radius = 2;
       this.dasharray = null;
       
      this._super();

      this.setBackgroundColor( new draw2d.util.Color(100,100,100));
      this.setColor("#1B1B1B");

      // set some good defaults
      //
      if(typeof width === "undefined"){
        this.setDimension(50, 90);
      }
      else{
        this.setDimension(width, height);
      }
    },

    /**
     * @inheritdoc
     **/
    repaint : function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }
        
        if (typeof attributes === "undefined") {
            attributes = {};
        }

        if(this.dasharray!==null){
            attributes["stroke-dasharray"]=this.dasharray;
        }
        attributes.width = this.getWidth();
        attributes.height = this.getHeight();
        attributes.r = this.radius;
        this._super(attributes);
    },
    
    /**
     * @private
     */
    applyTransformation:function(){
        this.shape.transform(
                "R"+
                this.rotationAngle);
        
        if(this.getRotationAngle()=== 90|| this.getRotationAngle()===270){
            var ratio = this.getHeight()/this.getWidth();
            var rs = "...S"+ratio+","+1/ratio+","+(this.getAbsoluteX() +this.getWidth()/2)+","+(this.getAbsoluteY() +this.getHeight()/2);
            this.shape.transform(rs);
        }
    },

    /**
     * @method
     * 
     * @inheritdoc
     */
    createShapeElement : function(){
       return this.canvas.paper.rect(this.getAbsoluteX(),this.getAbsoluteY(),this.getWidth(), this.getHeight());
    },

    /**
     * @method
     * Sets the corner radius for rectangles with round corners. 
     * 
     * @param {Number} radius
     */
     setRadius: function(radius){
        this.radius = radius;
        this.repaint();
        
        return this;
    },
    
    /**
     * @method
     * Indicates the corner radius for rectangles with round corners. The default is 2. 
     * 
     * @return {Number}
     */
    getRadius:function() {
        return this.radius;
    },
    
    /**
     * @method
     * set the line style for dot/dash styling. Possible values are
     * 
     * ["", "-", ".", "-.", "-..", ". ", "- ", "--", "- .", "--.", "--.."]
     * experimental only.
     * @param dash
     */
    setDashArray: function(dash){
        this.dasharray = dash;
        this.repaint();
    },
    
    /**
     * @method 
     * Return an objects with all important attributes for XML or JSON serialization
     * 
     * @returns {Object}
     */
    getPersistentAttributes : function(){
        var memento = this._super();
        
        memento.radius = this.radius;
        
        return memento;
    },
   
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     * @returns 
     */
    setPersistentAttributes : function(memento) {
        this._super(memento);
        
        if(typeof memento.radius ==="number")
        {
            this.radius = memento.radius;
        }
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.SetFigure
 * 
 * A SetFigure is a composition of different SVG elements.
 * 
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.SetFigure = draw2d.shape.basic.Rectangle.extend({
    
    NAME : "draw2d.SetFigure",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init: function( width, height) {
      // collection of SVG DOM nodes
      this.svgNodes=null;
      
      this.originalWidth = null;
      this.originalHeight= null;
      
      this.scaleX = 1;
      this.scaleY = 1;

      this._super( width, height);

      this.setStroke(0);
      this.setBackgroundColor(null); 
    },
    
    /**
     * @method
     * Set/Reset the canvas for the element.
     * 
     * @param {draw2d.Canvas} canvas the canvas to use
     */
    setCanvas: function( canvas )
    {
      // remove the shape if we reset the canvas and the element
      // was already drawn
      if(canvas===null && this.svgNodes!==null){
         this.svgNodes.remove();
         this.svgNodes=null;
      }
      
      this._super(canvas);
     },
 

     
     /**
      * @method
      * Set the css class if the node.
      * 
      * @param {String} cssClass the new css class name of the node
      * @since 2.9.0
      */
     setCssClass: function(cssClass)
     {
         this._super(cssClass);
         
         if(this.svgNodes===null){
             return this;
         }
         
         if(this.cssClass===null){
             this.svgNodes.forEach(function(e){
                 e.node.removeAttribute("class");
             });
         }
         else{
             this.svgNodes.forEach(function(e){
                 e.node.setAttribute("class", cssClass);
             });
         }
                 
         return this;
     },
     
     
    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element and
     * repaint them.
     * 
     **/
    repaint : function(attributes)
    {

        // repaint can be blocked during deserialization and if the shape
        // not part of any canvas.
        //
        if (this.repaintBlocked === true || this.shape === null) {
            return;
        }

        if (this.originalWidth !== null) {
        	this.scaleX = this.width / this.originalWidth;
        	this.scaleY = this.height / this.originalHeight;
        }
        
        if (typeof attributes === "undefined") {
            attributes = {};
        }

        this.applyAlpha();
        
        if(this.visible===true){
            this.svgNodes.show();
        }
        else{
            this.svgNodes.hide();
        }
        
        this._super(attributes);
    },


    /**
     * @method
     * Apply the opacity to all child set elements. Override this if you want to avoid opacity changes.
     * @private
     * 
     */
    applyAlpha: function(){
        this.svgNodes.attr({opacity: this.alpha});
    },
    
    /**
     * @private
     */
    applyTransformation:function(){
        var s = 
        	"S"+this.scaleX+","+this.scaleY+",0,0 "+
        	"R"+this.rotationAngle+","+((this.getWidth()/2)|0)+","+((this.getHeight()/2)|0)+
        	"T" + this.getAbsoluteX() + "," + this.getAbsoluteY()+
            "";
    	this.svgNodes.transform(s);
        if(this.rotationAngle===90 || this.rotationAngle===270){
            var before  = this.svgNodes.getBBox(true);
            var ratio = before.height/before.width;
            var reverseRatio = before.width/before.height;
            var rs = "...S"+ratio+","+reverseRatio+","+(this.getAbsoluteX() +this.getWidth()/2)+","+(this.getAbsoluteY() +this.getHeight()/2);
        	this.svgNodes.transform(rs);
        }
    },
    
    /**
     * @method
     * Moves the element so it is the closest to the viewer?��s eyes, on top of other elements. Additional
     * the internal model changed as well.
     * 
     * @since 3.0.0
     */
    toFront: function(){

        this._super();
 
        if(this.svgNodes!==null){
            this.svgNodes.toFront();
        }
         
        // the ports must always the top most
        //
        this.getPorts().each(function(i,port){
            port.getConnections().each(function(i,connection){
                connection.toFront();
            });
            port.toFront();
        });

        
        return this;
    },
    
    
    
    /**
     * @private
     */
    createShapeElement : function()
    {
       // NOTE: don't change the order of the two calls. This defines the z-oder in the canvas.
       // The "set" must always be on top.
       var shape= this.canvas.paper.rect(this.getX(),this.getY(),this.getWidth(), this.getHeight());
       this.svgNodes = this.createSet();
       
       // check if the element is a "set" or a simple raphael shape. otherwise we wrap them into a set 
       //
       if(typeof this.svgNodes.forEach==="undefined"){
           var set = this.canvas.paper.set();
           set.push(this.svgNodes);
           this.svgNodes = set;
       }
       
       // propagate the CSS style to all set elements
       this.setCssClass(this.cssClass);
       
       var bb = this.svgNodes.getBBox();
       this.originalWidth = bb.width;
       this.originalHeight= bb.height;
      
       return shape;
    },
    
    /**
     * @method
     * Override this method to add your own SVG elements. See {@link draw2d.shape.basic.Label} as example.
     * 
     * @template
     */
    createSet: function(){
    	return this.canvas.paper.set(); // return empty set as default;
    }
   
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.SVGFigure
 * Abstract class which can handle plain SVG content. Inherit class must override the method
 * <code>getSVG()</code>.
 * 
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.SVGFigure = draw2d.SetFigure.extend({
    
    NAME : "draw2d.SVGFigure",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
		return this.importSVG(this.canvas, this.getSVG());
	},
    
    /**
     * @private
     */
    importSVG : function (canvas, rawSVG) {
      
      var set = canvas.paper.set();
       
      try {
        if (typeof rawSVG === 'undefined'){
          throw 'No data was provided.';
        }
        
        // Override the dimension from the JSON if the SVG contains any
        //
        var svgDOM= $(rawSVG);
        
        // set the dimension of the element if the JSON import didn't provide
        // a dimension already
        //
        if(typeof this._dimensionReadFromJSON ==="undefined"){
            if(svgDOM.attr("width") && svgDOM.attr("height")){
                this.setDimension(parseFloat(svgDOM.attr("width")), parseFloat(svgDOM.attr("height")));
            }
            delete this._dimensionReadFromJSON;
        }
        
        var findStyle = new RegExp('([a-z0-9\-]+) ?: ?([^ ;]+)[ ;]?','gi');
        
        svgDOM.children().each(function(i,element){
          //element = $(element);
          var shape=null;
          var style=null;
          var attr = { };
          var node = element.tagName;
          
          // remove the namespace of the node if existing. This can happen in IE8
          //
          var index = node.indexOf(":");
          if(index != -1)
              node = node.substr(index+1);
          
          // map some element to Raphael specifix attributes or ignore some unknown attributes
          //
          $(element.attributes).each(function() {
            switch(this.nodeName) {
              case 'stroke-dasharray':
                attr[this.nodeName] = '- ';
              break;
              case 'style':
                style = this.nodeValue;
              break;
              case 'id':
              case 'xml:space':
                  // just to ignore
                  break;
              default:
                attr[this.nodeName] = this.nodeValue;
              break;
            }
          });
          
          
          if ( style !== null){
            while(findStyle.exec(style)){
              attr[RegExp.$1] = RegExp.$2;
            }
          }
          
          // set some good defaults if the element didn't provide a stroke-width but has a "stroke" attribute
          //
          if (typeof attr['stroke-width'] === 'undefined'){
              attr['stroke-width'] = (typeof attr.stroke === 'undefined' ? 0 : 1.2);
          }
          
          switch(node) {
            case 'rect':
              shape = canvas.paper.rect();
              break;
            case 'circle':
              shape = canvas.paper.circle();
              break;
            case 'ellipse':
              shape = canvas.paper.ellipse();
              break;
            case 'path':
              attr.fill ="none";
              shape = canvas.paper.path(attr.d);
              break;
            case 'line':
              attr.d= "M "+attr.x1+" "+attr.y1+"L"+attr.x2+" "+attr.y2;
              attr.fill ="none";
              shape = canvas.paper.path(attr.d);
             break;
            case 'polyline':
              var path = attr.points;
              attr.d = "M "+path.replace(" "," L");
              shape = canvas.paper.path(attr.d);
              break;
            case 'polygon':
              shape = canvas.paper.polygon(attr.points);
              break;
            case 'image':
              shape = canvas.paper.image();
              break;
            case 'tspan':
            case 'text':
                if(element.childNodes.length>0){
                    var child = element.firstChild;
                    do {
                       switch(child.nodeType){
                            case 2://ATTRIBUTE_NODE 
                            case 3://TEXT_NOD   
                            case 4://CDATA_SECTION_NODE
                            case 5://ENTITY_REFERENCE_NODE
                            case 6://ENTITY_NODE
                            case 7://PROCESSING_INSTRUCTION_NODE
                            case 8://COMMENT_NODE
                            case 9://DOCUMENT_NODE
                            case 10://DOCUMENT_TYPE_NODE
                            case 11://DOCUMENT_FRAGMENT_NODE
                            case 12://NOTATION_NODE
                                return;
                            case 1://ELEMENT_NODE 
                        }
                        var subShape = canvas.paper.text(0,0,$(child).text());
                        var subAttr ={"x":parseFloat(child.attributes.x.value), "y":parseFloat(child.attributes.y.value)};
                        subAttr["text-anchor"] = "start";
                    
                        if(typeof child.attributes["font-size"]!=="undefined"){
                            subAttr["font-size"] = parseInt(child.attributes["font-size"].value);
                        }
                        else if(typeof attr["font-size"]!=="undefined"){
                            // inherit the font size from the parent node
                            subAttr["font-size"] = parseInt(attr["font-size"]);
                        }
                        
                        if(typeof child.attributes["font-family"]!=="undefined"){
                            subAttr["font-family"] = child.attributes["family-family"].value;
                        }
                        else if(typeof attr["font-family"]!=="undefined"){
                            // inherit the font size from the parent node
                            subAttr["font-family"] = attr["font-family"];
                        }
                        
                        subAttr["fill"] = "#000000";
                        if(typeof child.attributes["fill"]!=="undefined"){
                            subAttr["fill"] = child.attributes["fill"].value;
                        }
                        else if(typeof attr["fill"]!=="undefined"){
                            // inherit the font size from the parent node
                            subAttr["fill"] = attr["fill"];
                        }
                        
                        subAttr.y= subAttr.y+subShape.getBBox().height/2;
                       
                        subShape.attr(subAttr);
                        set.push(subShape);
                    }while(child= child.nextSibling);
                }
                else{
                  shape = canvas.paper.text(0,0,$(element).html());
                  if(typeof attr["fill"]==="undefined")
                      attr["fill"] = "#000000";
                  if(typeof attr["text-anchor"]==="undefined")
                      attr["text-anchor"] = "start";
                  if(typeof attr["font-size"]!=="undefined")
                      attr["font-size"] = parseInt(attr["font-size"]);
                  if(typeof attr["font-family"]!=="undefined")
                      attr["font-family"] = parseInt(attr["font-family"]);
                  attr.y= parseFloat(attr.y)+shape.getBBox().height/2;
                }
              break;
          }
          if(shape!==null){
              shape.attr(attr);
              set.push(shape);
          }
        });
      } catch (error) {
        alert('The SVG data you entered was invalid! (' + error + ')');
      }
      
      
      return set;
    },
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     * @return
     */
    setPersistentAttributes : function(memento)
    {
        this._super(memento);
       
        // keep a temp flag to indicate the we have read the dimension of the
        // element from the JSON struct. In this case we didn't care about the dimension 
        // from the SVG data
        //
        if(typeof memento.width !== "undefined"){
            this._dimensionReadFromJSON=true;
        }
        else if(typeof memento.height !== "undefined"){
            this._dimensionReadFromJSON=true;
        }
        
        return this;
    }  


});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.Hub
 * 
 * A hub is a shape with a special kind of port handling. The hole figure is a hybrid port. You can drag&drop a Port directly on
 * the figure.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.node.Hub();
 *     
 *     canvas.addFigure(figure,50,10);
 *     
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.shape.node.Hub = draw2d.shape.basic.Rectangle.extend({

    NAME : "draw2d.shape.node.Hub",

    DEFAULT_COLOR : new draw2d.util.Color("#4DF0FE"),
    BACKGROUND_COLOR : new draw2d.util.Color("#29AA77"),

	/**
	 * 
	 * @param {Number} width initial width of the bus shape
	 * @param {Number} height height of the bus
	 */
	init : function(width, height, labelString)
    {
 	    this.label = null;
	    
        this._super(width,height);
        
        this.port = this.createPort("hybrid", new draw2d.layout.locator.CenterLocator(this));

        this.CONNECTION_DIR_STRATEGY= [ $.proxy(function(conn, relatedPort){ return this.getParent().getBoundingBox().getDirection(relatedPort.getAbsolutePosition());},this.port),
                                        $.proxy(function(conn, relatedPort){ return this.getAbsoluteY()>relatedPort.getAbsoluteY()?0:2;},this.port),
                                        $.proxy(function(conn, relatedPort){ return this.getAbsoluteX()>relatedPort.getAbsoluteX()?3:1;},this.port)];

        // redirect the glow effect and the hitTest for the port to the parent node
        //
        this.port.setGlow = $.proxy(this.setGlow,this);
        this.port._orig_hitTest = this.port.hitTest;
        this.port.hitTest = $.proxy(this.hitTest,this);
       
        
        // provide a special connection anchor for this port. We use the bounding box of the
        // parent as connection border
        //
        this.port.setConnectionAnchor(new draw2d.layout.anchor.ShortesPathConnectionAnchor(this.port));
        this.port.setVisible(false);
        
        // set some good defaults
        //
        if(typeof height ==="undefined"){
            this.setDimension(150, 50);
        }
        
        this.setConnectionDirStrategy(0);
        
        // set the border of the rectangle a little bit darker than the 
        // inner part
        //
        this.setColor(this.DEFAULT_COLOR.darker());
        this.setBackgroundColor(this.BACKGROUND_COLOR);
        if(typeof labelString !== "undefined"){
            this.setLabel(labelString);
        }
    },
    
      /**
      * @method
      * 
      * @param draggedFigure
      * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didnt want a drop event
      */
     onDragEnter : function( draggedFigure )
     {
         // redirect the dragEnter handling to the hybrid port
         //
  		 return this.getHybridPort(0).onDragEnter(draggedFigure);
     },
     
     /**
      * @method
      * This value is relevant for the interactive resize of the figure.
      *
      * @return {Number} Returns the min. width of this object.
      */
     getMinWidth:function()
     {
         if(this.label!==null){
             return Math.max(this.label.getMinWidth(), this._super());
         }
         return this._super();
     },
     

     /**
      * @inheritdoc
      * 
      * @param attributes
      */
     repaint:function(attributes)
     {
         if(this.repaintBlocked===true || this.shape===null){
             return;
         }

         if(typeof attributes === "undefined"){
             attributes = {};
         }
         
         // set some good defaults if the parent didn't
         if(typeof attributes.fill ==="undefined"){
             if(this.bgColor!==null){
                 attributes.fill="90-"+this.bgColor.hash()+":5-"+this.bgColor.lighter(0.3).hash()+":95";
             }
             else{
                 attributes.fill ="none";
             }
        }
         
        this._super(attributes);
     },
     
     /**
      * @method
      * set the label for the Hub
      * 
      * @param {String} labelString
      * @since 3.0.4
      */
     setLabel: function( labelString){
         // Create any Draw2D figure as decoration for the connection
         //
         if(this.label===null){
             this.label = new draw2d.shape.basic.Label(labelString);
             this.label.setColor("#0d0d0d");
             this.label.setFontColor("#0d0d0d");
             this.label.setStroke(0);
             // add the new decoration to the connection with a position locator.
             //
             this.addFigure(this.label, new draw2d.layout.locator.CenterLocator(this));
         }
         else{
             this.label.setText();
         }
         
     },
     
     /**
      * @method
      * Set the strategy for the connection direction calculation.<br>
      * <br>
      * 
      * <ul>
      * <li>0 - Use the best/shortest direction (UP/RIGHT/DOWN/LEFT) for the connection routing (default)</li>
      * <li>1 - Use UP/DOWN for the connection direction</li>
      * <li>2 - Use LEFT/RIGHT</li>
      * </ul>
      * @param {Number} strategy the connection routing strategy to use
      * @since 2.4.3
      */
     setConnectionDirStrategy: function(strategy){
         switch(strategy){
             case 0:
             case 1:
             case 2:
                 this.port.getConnectionDirection= this.CONNECTION_DIR_STRATEGY[strategy];
                 break;
         }
     },
     
     /**
      * @method 
      * Return an objects with all important attributes for XML or JSON serialization
      * 
      * @returns {Object}
      */
     getPersistentAttributes : function(){
         var memento = this._super();
         
         memento.dirStrategy = this.CONNECTION_DIR_STRATEGY.indexOf(this.port.getConnectionDirection);
         if(this.label !==null){
             memento.label = this.label.getText();
         }
         
         return memento;
     },
     
     /**
      * @method 
      * Read all attributes from the serialized properties and transfer them into the shape.
      * 
      * @param {Object} memento
      * @returns 
      */
     setPersistentAttributes : function(memento) {
         this._super(memento);
         
         if(typeof memento.dirStrategy ==="number") {
             this.setConnectionDirStrategy( memento.dirStrategy);
         }
         
         if(typeof memento.label !== "undefined"){
             this.setLabel(memento.label);
         }
     }
     
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.HorizontalBus
 * 
 * A horizontal bus shape with a special kind of port handling. The hole figure is a hybrid port.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.node.HorizontalBus(300,20,"Horizontal Bus");
 *     
 *     canvas.addFigure(figure,50,10);
 *     
 * @extends draw2d.shape.node.Hub
 */
draw2d.shape.node.HorizontalBus = draw2d.shape.node.Hub.extend({

    NAME : "draw2d.shape.node.HorizontalBus",

	/**
	 * 
	 * @param {Number} width initial width of the bus shape
	 * @param {Number} height height of the bus
	 */
	init : function(width, height, label)
    {
        this._super(width,height,label);
        
        this.setConnectionDirStrategy(1);

        this.installEditPolicy(new draw2d.policy.figure.HBusSelectionFeedbackPolicy());
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.VerticalBus
 * 
 * A horizontal bus shape with a special kind of port handling. The hole figure is a hybrid port.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.node.VerticalBus(40,300,"Vertical Bus");
 *     
 *     canvas.addFigure(figure,50,10);
 *     
 * @extends draw2d.shape.node.Hub
 */
draw2d.shape.node.VerticalBus = draw2d.shape.node.Hub.extend({

    NAME : "draw2d.shape.node.VerticalBus",

	/**
	 * 
	 * @param {Number} width initial width of the bus shape
	 * @param {Number} height height of the bus
	 */
	init : function(width, height, labelString)
    {
        this._super(width,height,labelString);
        

        this.setConnectionDirStrategy(2);
        this.installEditPolicy(new draw2d.policy.figure.VBusSelectionFeedbackPolicy());
    },
    
    
    /**
     * @method
     * set the label for the Hub
     * 
     * @param {String} labelString
     * @since 3.0.4
     */
    setLabel: function( labelString){
        var mustAdjustTheAngel = this.label === null;
        
        this._super(labelString);
        
        if(mustAdjustTheAngel===true && this.label !==null){
            this.label.setRotationAngle(90);
        }
    },
    
    
    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. height of this object.
     */
    getMinHeight:function()
    {
        if (this.shape === null && this.label === null) {
            return 0;
        }
        
        if(this.label!==null){
            return this.label.getMinWidth();
        }
        
        return this._super();
    },
    
    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. height of this object.
     */
    getMinWidth:function()
    {
        if (this.shape === null && this.label === null) {
            return 0;
        }
        
        if(this.label!==null){
            return this.label.getMinHeight();
        }
        
        return this._super();
    }

    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.Fulcrum
 * 
 * A horizontal bus shape with a special kind of port handling. The hole figure is a hybrid port.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     canvas.addFigure( new draw2d.shape.node.Fulcrum(),50,10);
 *     canvas.addFigure( new draw2d.shape.node.Fulcrum(),80,100);
 *     canvas.addFigure( new draw2d.shape.node.Fulcrum(),150,50);
 *     
 * @extends draw2d.shape.node.Hub
 */
draw2d.shape.node.Fulcrum = draw2d.shape.node.Hub.extend({

    NAME : "draw2d.shape.node.Fulcrum",
    
	/**
	 * 
	 */
	init : function()
    {
        this._super(40,40);

        
        this.port.setConnectionAnchor(new draw2d.layout.anchor.ConnectionAnchor(this.port));
        this.port.setVisible(true);
        this.port.hitTest = this.port._orig_hitTest;
        
        this.setConnectionDirStrategy(0);
        this.setColor(null);
        this.setRadius(10);
        this.setBackgroundColor(null);
        this.setStroke(0);
        this.installEditPolicy(new draw2d.policy.figure.AntSelectionFeedbackPolicy());
   },
   
   /**
     * @inheritdoc
     * 
     * @param attributes
     */
    repaint:function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }
    
        if(typeof attributes === "undefined"){
            attributes = {};
        }
        
        // set some good defaults if the parent didn't
        if(typeof attributes.fill ==="undefined"){
            attributes.fill=this.bgColor.hash();
        }
        
       this._super(attributes);
    }
    
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Oval
 * Oval figure.
 * 
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var oval =  new draw2d.shape.basic.Oval();
 *     oval.setDimension(150,100);
 *     canvas.addFigure(oval,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.VectorFigure
 */
draw2d.shape.basic.Oval = draw2d.VectorFigure.extend({
    NAME : "draw2d.shape.basic.Oval",

    /**
     * 
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width,height ) {
        this._super( );
        this.setBackgroundColor("#C02B1D");
        this.setColor("#1B1B1B");
        
        if((typeof height === "number") &&(typeof width === "number")){
            this.setDimension(width,height);
        }
        else{
            this.setDimension(50,50);
        }
    },
      

   /** 
    * @template
    **/
   createShapeElement : function()
   {
     var halfW = this.getWidth()/2;
     var halfH = this.getHeight()/2;
     
     return this.canvas.paper.ellipse(this.getAbsoluteX()+halfW, this.getAbsoluteY()+halfH, halfW, halfH);
   },

   /**
    * @inheritdoc
    * 
    * @template
    **/
   repaint: function(attributes)
   {
       if(this.repaintBlocked===true || this.shape===null){
           return;
       }

       if(typeof attributes === "undefined"){
           attributes = {};
       }
       
       
       // don't override cx/cy if inherited class has set the center already.
       if(typeof attributes.rx === "undefined"){
           attributes.rx = this.width/2;
           attributes.ry = this.height/2;
       }
 
       // don't override cx/cy if inherited class has set the center already.
       if(typeof attributes.cx === "undefined"){
           attributes.cx = this.getAbsoluteX()+attributes.rx;
           attributes.cy = this.getAbsoluteY()+attributes.ry;
       }
       
       this._super(attributes);
   },
   
   /*****
   *
   *   intersectEllipseLine
   *   
   *   NOTE: Rotation will need to be added to this function
   *
   *****/
   intersectionWithLine : function(a1, a2) {
	   var rx = this.getWidth()/2;
	   var ry = this.getHeight()/2;
       
	   var result= new draw2d.util.ArrayList();
       
       var origin = new draw2d.geo.Point(a1.x, a1.y);
       var dir    = a2.subtract(a1);
       var center = new draw2d.geo.Point(this.getAbsoluteX()+rx, this.getAbsoluteY()+ry);
       var diff   = origin.subtract(center);
       var mDir   = new draw2d.geo.Point( dir.x/(rx*rx),  dir.y/(ry*ry)  );
       var mDiff  = new draw2d.geo.Point( diff.x/(rx*rx), diff.y/(ry*ry) );

       var a = dir.dot(mDir);
       var b = dir.dot(mDiff);
       var c = diff.dot(mDiff) - 1.0;
       var d = b*b - a*c;

       if ( d < 0 ) {
           // "Outside"
       } else if ( d > 0 ) {
           var root = Math.sqrt(d);
           var t_a  = (-b - root) / a;
           var t_b  = (-b + root) / a;

           if ( (t_a < 0 || 1 < t_a) && (t_b < 0 || 1 < t_b) ) {
               if ( (t_a < 0 && t_b < 0) || (t_a > 1 && t_b > 1) ){
                   //"Outside";
               }
               else{            	   
                   ;//"Inside";
               }
           } else {
               if ( 0 <= t_a && t_a <= 1 )
                   result.add( a1.lerp(a2, t_a) );
               if ( 0 <= t_b && t_b <= 1 )
                   result.add( a1.lerp(a2, t_b) );
           }
       } else {
           var t = -b/a;
           if ( 0 <= t && t <= 1 ) {
               result.add( a1.lerp(a2, t) );
           } else {
               //"Outside";
           }
       }
       
       return result;
   }	 
    
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/

/**
 * @class draw2d.shape.basic.Circle
 * A circle figure with basic background and stroke API. <br>
 * A circle can not be streched. <strong>The aspect ration is always 1:1</strong>.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var shape =  new draw2d.shape.basic.Circle();
 *     shape.setStroke(3);
 *     shape.setColor("#3d3d3d");
 *     shape.setBackgroundColor("#3dff3d");
 *     
 *     canvas.addFigure(shape,40,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Oval
 */
draw2d.shape.basic.Circle = draw2d.shape.basic.Oval.extend({
    
    NAME : "draw2d.shape.basic.Circle", 
    
    /**
     * @constructor
     * Create a new circle figure.
     * 
     * @param {Number} [radius] the initial radius for the circle
     */
    init:function( radius)
    {
      this._super();
      this.setKeepAspectRatio(true);
      
      if(typeof radius === "number"){
        this.setDimension(radius,radius);
      }
      else {
         this.setDimension(40,40);
      }
    },
    
    /**
     * @method
     * Set the diameter of the circle. The center of the circle will be retained.
     * 
     * @param {Number} d The new diameter of the circle.
     * @since 4.0.0
     **/
    setDiameter:function(d)
    {
        var center = this.getCenter();
        this.setDimension(d,d);
        this.setCenter(center); 
        
        return this;
    },

    /**
     * @method
     * Get the diameter of the circle.
     * 
     * @since 4.0.0
     **/
    getDiameter:function()
    {
        return this.getWidth();
    },

    
    /**
     * @method
     * Set the radius of the circle. The center of the circle will be retained.
     * 
     * @param {Number} d The new radius of the circle.
     * @since 4.0.0
     **/
    setRadius:function(r)
    {
        this.setDiameter(r*2);
        
        return this;
    },

    
    /**
     * @method
     * Get the center of the circle
     * 
     */
    getCenter: function(){
        var d2= this.getDiameter()/2;
        return this.getPosition().translate(d2,d2);
    },

    /**
     * @method
     * Set the center of the circle.
     * 
     * @param {Number|draw2d.geo.Point} x the new x coordinate of the center or a draw2d.geo.Point object with the center
     * @param {Number} y the y coordinate of the new center of the first argument isn't a draw2d.geo.Point object
     */
    setCenter: function(x, y){
        var pos = new draw2d.geo.Point(x,y);
        var d2  = this.getDiameter()/2;
        pos.translate(-d2,-d2);
        this.setPosition(pos);
        
        return this;
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Label
 * Implements a simple text label.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var shape =  new draw2d.shape.basic.Label("This is a simple label");
 *          
 *     canvas.addFigure(shape,40,10);
 *     
 * @author Andreas Herz
 * 
 * @extends draw2d.SetFigure
 */
draw2d.shape.basic.Label= draw2d.SetFigure.extend({

	NAME : "draw2d.shape.basic.Label",
    FONT_FALLBACK:  {
      'Georgia'            :'Georgia, serif',
      'Palatino Linotype'  :'"Palatino Linotype", "Book Antiqua", Palatino, serif',
      'Times New Roman'    :'"Times New Roman", Times, serif',
      'Arial'              :'Arial, Helvetica, sans-serif',
      'Arial Black'        :'"Arial Black", Gadget, sans-serif',   
      'Comic Sans MS'      :'"Comic Sans MS", cursive, sans-serif',    
      'Impact'             :'Impact, Charcoal, sans-serif',
      'Lucida Sans Unicode':'"Lucida Sans Unicode", "Lucida Grande", sans-serif',  
      'Tahoma, Geneva'     :'Tahoma, Geneva, sans-seri',
      'Trebuchet MS'       :'"Trebuchet MS", Helvetica, sans-serif',
      'Verdana'            :'Verdana, Geneva, sans-serif',
      'Courier New'        :'"Courier New", Courier, monospace',
      'Lucida Console'     :'"Lucida Console", Monaco, monospace'},
      

    /**
     * @constructor
     * Creates a new text element.
     * 
     * @param {String} [text] the text to display
     */
    init : function(text)
    {
        this._super();
        
        if(typeof text === "string"){
    		this.text = text;
    	}
    	else{
    		this.text = "";
    	}
    	// for performance reasons
        //
        this.cachedWidth  = null;
        this.cachedHeight = null;
        this.cachedMinWidth  = null;
        this.cachedMinHeight = null;
        
        // appearance of the shape
        //
        this.fontSize = 12;
        this.fontColor = new draw2d.util.Color("#080808");
        this.fontFamily = null;
        this.padding = 4;
        
        this.outlineStroke = 0;
        this.outlineColor = new draw2d.util.Color(null);
        
        this.bold = false;
        
        
        // set some good defaults
        //
        this.setStroke(1);
        this.setDimension(1,1);

        // behavior of the shape
        //
        this.editor = null;
        
        this.installEditPolicy(new draw2d.policy.figure.AntSelectionFeedbackPolicy());
    },
    
    /** 
     * @method
     * Creates the shape object for a text node.
     * 
     * @template
     **/
    createSet : function()
    {
    	return this.canvas.paper.text(0, 0, this.text);
    },

    /**
     * @method
     * Set the canvas element of this figures.
     * 
     * @param {draw2d.Canvas} canvas the new parent of the figure or null
     */
    setCanvas: function( canvas )
    {
        this.clearCache();
        this._super(canvas);
        this.clearCache();
    },
    
    
    /**
     * @method
     * Trigger the repaint of the element and transport all style properties to the visual representation.<br>
     * Called by the framework.
     * 
     * @template
     **/
    repaint: function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }

        // style the label
        var lattr = this.calculateTextAttr();
        lattr.text = this.text;        
        this.svgNodes.attr(lattr);
        // set of the x/y must be done AFTER the font-size and bold has been set.
        // Reason: the getHeight method needs the font-size for calculation because
        //         it redirects the calculation to the SVG element.
        this.svgNodes.attr({x:this.padding,y: this.getHeight()/2});

        this._super(attributes);
    },
    

    /**
     * 
     * @private
     */
    calculateTextAttr:function(){
        var lattr={"text-anchor":"start",
                   "font-size":this.fontSize,
                   "font-weight":(this.bold===true)?"bold":"normal",
                   fill: this.fontColor.hash(),
                   stroke : this.outlineColor.hash(),
                   "stroke-width": this.outlineStroke
                   };
        if(this.fontFamily!==null){
            lattr["font-family"] = this.fontFamily;
        }
        return lattr;
    },

    /**
     * @private
     */
    applyTransformation:function(){
        this.shape.transform(
                "R"+
                this.rotationAngle);
        this.svgNodes.transform(
                "R"+
                this.rotationAngle+
                "T" + this.getAbsoluteX() + "," + this.getAbsoluteY());
    },

    
    /**
     * @method
     * A Label is not resizeable. In this case this method returns always <b>false</b>.
     * 
     * @returns Returns always false in the case of a Label.
     * @type boolean
     **/
    isResizeable:function()
    {
      return false;
    },
       
    
    /**
     * @method
     * Set the new font size in [pt].
     *
     * @param {Number} size The new font size in <code>pt</code>
     **/
    setFontSize: function( size)
    {
      this.clearCache();
      this.fontSize = size;
      
      this.repaint();

      this.fireResizeEvent();
      // just to be backward compatible....cost a lot of performance...still
      this.fireMoveEvent();
      
      // Update the resize handles if the user change the position of the element via an API call.
      //
      this.editPolicy.each($.proxy(function(i,e){
         if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
             e.moved(this.canvas, this);
         }
      },this));
      
    
      return this;
    },
    
    /**
     * @method
     * Return the current used font size in px.
     * 
     * @returns {Number}
     * @since 4.0.1
     */
    getFontSize: function( )
    {
      return this.fontSize;
    },
    

    /**
     * @method
     * Set the label to <b>bold</b> or <b>normal</b> font weight.
     *
     * @param {Boolean} bold The bold flag for the label
     * @since 2.4.1
     **/
    setBold: function( bold)
    {
      this.clearCache();
      this.bold = bold;
      this.repaint();
      
      return this;
    },
    
    /**
     * @method
     * Set the outline color of the font.
     * 
     * @param {draw2d.util.Color/String} color The new color of the line.
     * @since 4.2.1
     **/
    setOutlineColor:function( color)
    {
      this.outlineColor = new draw2d.util.Color(color);
      this.repaint();
      
      return this;
    },

    /**
     * @method
     * The outlien color of the text
     * 
     * @returns {draw2d.util.Color}
     * @since 4.2.1
     */
    getOutlineColor:function()
    {
      return this.outlineColor;
    },
    
    /**
     * @method
     * Set the stroke of the text to use.
     * 
     * @param {Number} w The new line width of the figure
     * @since 4.2.1
     **/
    setOutlineStroke:function( w )
    {
      this.outlineStroke=w;
      this.repaint();
      
      return this;
    },

    /**
     * @method
     * The used outline line width.
     * 
     * @type {Number}
     * @since 4.2.1
     **/
    getOutlineStroke:function( )
    {
      return this.outlineStroke;
    },

    /**
     * @method
     * Set the color of the font.
     * 
     * @param {draw2d.util.Color/String} color The new color of the line.
     **/
    setFontColor:function( color)
    {
      this.fontColor = new draw2d.util.Color(color);
      this.repaint();
      
      return this;
    },

    /**
     * @method
     * The current used font color
     * 
     * @returns {draw2d.util.Color}
     */
    getFontColor:function()
    {
      return this.fontColor;
    },
    
    /**
     * @method
     * Set the padding of the element
     *
     * @param {Number} padding The new padding
     **/
    setPadding: function( padding)
    {
      this.clearCache();
      this.padding = padding;
      this.repaint();
      
      return this;
    },

    
    /**
     * @method
     * Get the padding of the element.
     *
     * @since 4.0.1
     **/
    getPadding: function( )
    {
      return this.padding;
    },

    /**
     * @method
     * Set the font family to use. If you use the <b>bold</b> font names the typical fallback 
     * font are installed as well.
     * 
     * <b>Serif Fonts</b>
     * <ul>
     *  <li><b>Georgia</b>, serif   </li>
     *  <li><b>Palatino Linotype</b>, "Book Antiqua", Palatino, serif    </li>
     *  <li><b>Times New Roman</b>, Times, serif  </li>   
     * </ul>
     * 
     * <b>Sans-Serif Fonts</b>
     * <ul>
     *  <li><b>Arial</b>, Helvetica, sans-serif   </li> 
     *  <li><b>Arial Black</b>, Gadget, sans-serif </li>  
     *  <li><b>Comic Sans MS</b>, cursive, sans-serif   </li> 
     *  <li><b>Impact, Charcoal</b>, sans-serif   </li> 
     *  <li><b>Lucida Sans Unicode</b>, "Lucida Grande", sans-serif </li> 
     *  <li><b>Tahoma, Geneva</b>, sans-serif  </li>
     *  <li><b>Trebuchet MS</b>, Helvetica, sans-serif </li>  
     *  <li><b>Verdana</b>, Geneva, sans-serif   </li>  
     * </ul>
     * 
     * <b>Monospace Fonts</b>
     * <ul>
     *  <li><b>Courier New</b>, Courier, monospace   </li>
     *  <li><b>Lucida Console</b>, Monaco, monospace</li>
     * </ul>
     *
     * @param {String} font The font to use
     **/
    setFontFamily: function( font)
    {
      this.clearCache();
      
      // check for fallback
      //
      if((typeof font!=="undefined") && font!==null && typeof this.FONT_FALLBACK[font] !== "undefined"){
          font=this.FONT_FALLBACK[font];
      }
      
      this.fontFamily = font;
      this.repaint();
      
      return this;
    },
    
    /**
     * @method
     * A Label did have "autosize". Do nothing at all.
     *
     **/
    setDimension:function( w, h)
    {
        this.clearCache();
        
        this._super(w,h);
        
        return this;
    },
    
    /**
     * @method
     * clear the internal cache for width/height precalculation
     * @private
     */
    clearCache:function(){
        this.cachedMinWidth  = null;
        this.cachedMinHeight = null;
        this.cachedWidth=null;
        this.cachedHeight=null;
    },
    
    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. width of this object.
     */
    getMinWidth:function()
    {
        if (this.shape === null) {
            return 0;
        }
        
        if(this.cachedMinWidth=== null){
            this.cachedMinWidth=this.svgNodes.getBBox(true).width+2*this.padding+2*this.getStroke();
        }
        
        return this.cachedMinWidth;
    },
    
    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. width of this object.
     */
    getMinHeight:function()
    {
        if (this.shape === null) {
            return 0;
        }
        
        if(this.cachedMinHeight=== null){
            this.cachedMinHeight=this.svgNodes.getBBox(true).height+2*this.padding+2*this.getStroke();
        }
        
        return this.cachedMinHeight;
    },
    
    /**
     * @method
     * Return the calculate width of the set. This calculates the bounding box of all elements.
     * 
     * @returns the calculated width of the label
     * @return {Number}
     **/
    getWidth : function() {
        
        if (this.shape === null) {
            return 0;
        }
        
        if(this.cachedWidth===null){
            this.cachedWidth = Math.max(this.width, this.getMinWidth());
        }
        
        
        return this.cachedWidth;
    },
    
    /**
     * @method
     * Return the calculated height of the set. This calculates the bounding box of all elements.
     * 
     * @returns the calculated height of the label
     * @return {Number}
     */
    getHeight:function()
    {
        if (this.shape === null) {
            return 0;
        }
        
        if(this.cachedHeight===null){
            this.cachedHeight = Math.max(this.height, this.getMinHeight());
        }
        
        return this.cachedHeight;
    },

    /**
     * @method
     * Set an editor for the label. This can be a dialog or inplace editor for the 
     * Text.<br>
     * The editor will be activated if you doubleClick on the label.
     * 
     * @param {draw2d.ui.LabelEditor} editor
     */
    installEditor: function( editor ){
      this.editor = editor;  
      
      return this;
    },
    
    /**
     * @method
     * Called when a user dbl clicks on the element
     * 
     * @template
     */
    onDoubleClick: function(){
        if(this.editor!==null){
            this.editor.start(this);
        }
    },
    
    
    /**
     * @method
     * Returns the current text of the label.
     *
     * @returns the current display text of the label
     * @type String
     **/
    getText:function()
    {
      return this.text;
    },
    
    /**
     * @method
     * Set the text for the label. Use \n for multiline text.
     * 
     * @param {String} text The new text for the label.
     **/
    setText:function( text )
    {
      this.clearCache();
      this.text = text;
      
      this.repaint();
      // Update the resize handles if the user change the position of the element via an API call.
      //
      this.editPolicy.each($.proxy(function(i,e){
         if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
             e.moved(this.canvas, this);
         }
      },this));

      this.fireResizeEvent();
      
      if(this.parent!==null){
          this.parent.repaint();
      }
      
      return this;
    },
    

    hitTest: function(x, y) {
        // apply a simple bounding box test if the label isn'T rotated
        //
        if( this.rotationAngle === 0){
            return this._super(x,y); 
        }
        
        // rotate the box with the current matrix of the
        // shape
        var matrix = this.shape.matrix;
        var points = this.getBoundingBox().getVertices();
        points.each(function(i,point){
            var x = matrix.x(point.x,point.y);
            var y = matrix.y(point.x,point.y);
            point.x=x;
            point.y=y;
        });

        var polySides=4;
        var i=0;
        var j=polySides-1 ;
        var oddNodes=false;

        for (i=0; i<polySides; i++) {
            var pi = points.get(i);
            var pj = points.get(j);
            if ((pi.y< y && pj.y>=y
            ||   pj.y< y && pi.y>=y)
            &&  (pi.x<=x || pj.x<=x)) {
              if (pi.x+(y-pi.y)/(pj.y-pi.y)*(pj.x-pi.x)<x) {
                oddNodes=!oddNodes; }}
            j=i; }
        return oddNodes; 
     },
     

     /**
      * @method 
      * Return an objects with all important attributes for XML or JSON serialization
      * 
      * @returns {Object}
      */
     getPersistentAttributes : function()
     {
         var memento = this._super();
         
         memento.text = this.text;
         memento.outlineStroke = this.outlineStroke;
         memento.outlineColor = this.outlineColor.hash();
         memento.fontSize = this.fontSize;
         memento.fontColor = this.fontColor.hash();
         memento.fontFamily = this.fontFamily;

         return memento;
     },
     
     /**
      * @method 
      * Read all attributes from the serialized properties and transfer them into the shape.
      * 
      * @param {Object} memento
      * @returns 
      */
     setPersistentAttributes : function(memento)
     {
         this._super(memento);
         if(typeof memento.text !=="undefined"){
             this.setText(memento.text);
         }
         if(typeof memento.outlineStroke !=="undefined"){
             this.setOutlineStroke(memento.outlineStroke);
         }
         if(typeof memento.outlineColor !=="undefined"){
             this.setOutlineColor(memento.outlineColor);
         }
         if(typeof memento.fontFamily !=="undefined"){
             this.setFontFamily(memento.fontFamily);
         }
         if(typeof memento.fontSize !=="undefined"){
             this.setFontSize(memento.fontSize);
         }
         if(typeof memento.fontColor !=="undefined"){
             this.setFontColor(memento.fontColor);
         }
     }

});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Label
 * Implements a simple text with word wrapping.<br>
 * <br>
 * This very first implementation is pretty inchoate and must be revise fro performance reason.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var shape =  new draw2d.shape.basic.Text("This is a simple text with some loooooong word in.");
 *          
 *     canvas.addFigure(shape,40,10);
 *     
 * @author Andreas Herz
 * @since 4.2.3
 * @extends draw2d.SetFigure
 */
draw2d.shape.basic.Text= draw2d.shape.basic.Label.extend({

	NAME : "draw2d.shape.basic.Text",

    /**
     * @constructor
     * Creates a new text element.
     * 
     * @param {String} [text] the text to display
     */
    init : function(text)
    {
        this._super(text);
        
        this.cachedWrappedAttr = null;
        
        // set some good defaults
        this.setDimension(100,50);
        this.installEditPolicy(new draw2d.policy.figure.WidthSelectionFeedbackPolicy());
    },
    
    
    
    /**
     * @method
     * Trigger the repaint of the element and transport all style properties to the visual representation.<br>
     * Called by the framework.
     * 
     * @template
     **/
    repaint: function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }

        // style the label
       this.svgNodes.attr($.extend({},this.calculateTextAttr(),this.wrappedTextAttr(this.text, this.getWidth()-2*this.padding)));
        
        // set of the x/y must be done AFTER the font-size and bold has been set.
        // Reason: the getHeight method needs the font-size for calculation because
        //         it redirects the calculation to the SVG element.
        this.svgNodes.attr({x:this.padding,y: this.getHeight()/2});

        draw2d.SetFigure.prototype.repaint.call(this,attributes);
    },
    
    
    /**
     * @method
     * Set the dimension of the text box. The height parameter are ignored and adjusted with the calculated height.
     *
     **/
    setDimension:function( w, h)
    {
        this.clearCache();
        h = this.wrappedTextAttr(this.text, w).height;
        this._super(w,h);
        
        return this;
    },
    
    /**
     * @method
     * clear the internal cache for width/height precalculation
     * @private
     */
    clearCache:function(){
        this._super();
        this.cachedWrappedAttr = null;
    },
 

    
    /**
     * @method
     * A Label is not resizeable. In this case this method returns always <b>false</b>.
     * 
     * @returns Returns always false in the case of a Label.
     * @type boolean
     **/
    isResizeable:function()
    {
      return true;
    },
       
    
    /**
     * @method
     * This value is relevant for the interactive resize of the figure.
     *
     * @return {Number} Returns the min. width of this object.
     */
    getMinWidth:function()
    {
        if (this.shape === null) {
            return 0;
        }
        
        if(this.cachedMinWidth=== null){
            // get the longest word in the text
            //
            var longestWord = this.text.split(" ").reduce(function(arg1,arg2){ return arg1.length > arg2.length ? arg1 : arg2; });
            var svgText = this.canvas.paper
                                     .text(0, 0, longestWord)
                                     .attr($.extend({},this.calculateTextAttr(),{text:longestWord}));
            this.cachedMinWidth= svgText.getBBox(true).width+2*this.padding+2*this.getStroke();
            svgText.remove();
        }
        
        return this.cachedMinWidth;
    },
    

    /**
     * @method
     * calculates the attributes (wrapped text and width, height) with the given parameter
     * 
     * @private
     */
    wrappedTextAttr: function(text, width) {
        if(this.canvas ===null){
            return {text:text, width:width, height:20};
        }
        
        if(this.cachedWrappedAttr===null){
            var abc = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
            var svgText = this.canvas.paper.text(0, 0, "").attr($.extend({},this.calculateTextAttr(),{text:abc}));
            
            // get a good estimation of a letter width...not correct but this is working for the very first draft implementation
            var letterWidth = svgText.getBBox().width / abc.length;
    
            var words = this.text.split(" ");
            var x = 0, s = [];
            for ( var i = 0; i < words.length; i++) {
                var l = words[i].length;
                if (x + (l * letterWidth) > width) {
                    s.push("\n");
                    x = 0;
                }
                x += l * letterWidth;
                s.push(words[i] + " ");
            }
            var bbox = svgText.getBBox();
            svgText.remove();
            this.cachedWrappedAttr= {text: s.join(""), width:(bbox.width+this.padding*2), height: (bbox.height+this.padding*2)};
        }
        return this.cachedWrappedAttr;
     },

     /**
      * @method 
      * Return an objects with all important attributes for XML or JSON serialization
      * 
      * @returns {Object}
      */
     getPersistentAttributes : function()
     {
         var memento = this._super();
         

         return memento;
     },
     
     /**
      * @method 
      * Read all attributes from the serialized properties and transfer them into the shape.
      * 
      * @param {Object} memento
      * @returns 
      */
     setPersistentAttributes : function(memento)
     {
         this._super(memento);

     }

});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Line
 * The base class for all visible elements inside a canvas.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     // Create the line and modify the start/end after inserting them into 
 *     // the canvas
 *     var line1 =  new draw2d.shape.basic.Line();
 *     line1.setStartPoint(30,30);
 *     line1.setEndPoint(100,80);
 *       
 *     canvas.addFigure(line1);
 *     
 *     // Create the line with a given start/end coordinate in the constructor
 *     //
 *     var line2 = new draw2d.shape.basic.Line(20,80,200,150);
 *     line2.setStroke(3);
 *     line2.setColor("#1d1dff");
 *     canvas.addFigure(line2);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.Figure
 */
draw2d.shape.basic.Line = draw2d.Figure.extend({
    NAME : "draw2d.shape.basic.Line",

    DEFAULT_COLOR : new draw2d.util.Color(0,0,0),
    
    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     * @param {Number} startX the x-coordinate of the start
     * @param {Number} startY the y-coordinate of the start
     * @param {Number} endX   the x-coordinate of the end
     * @param {Number} endY   the y-coordinate of the end
     * 
     */
    init: function(startX, startY, endX, endY ) {
        
        // for performance reasons if we made some bulk changes to the object
        // For this case we can block the repaint and enable it after the bulk
        // Update of the properties
        this.repaintBlocked = false;
        
        // click area for the line hit test
        this.corona = 10;
        this.isGlowing = false;
        this.lineColor = this.DEFAULT_COLOR;
        this.stroke=1;
        this.outlineStroke = 0;
        this.outlineColor = new draw2d.util.Color(null);
        this.outlineVisible = false;

        this.dasharray = null;
        
        if(typeof endY ==="number"){
            this.start = new draw2d.geo.Point(startX,startY);
            this.end   = new draw2d.geo.Point(endX,endY);
        }
        else{
            this.start = new draw2d.geo.Point(30,30);
            this.end   = new draw2d.geo.Point(100,100);
        }

        this.vertices = new draw2d.util.ArrayList();
        this.vertices.add(this.start);
        this.vertices.add(this.end);
        
        this._super();
        
        // create the selections handles/decorations
        this.installEditPolicy(new draw2d.policy.line.LineSelectionFeedbackPolicy());

        this.setSelectable(true);
        this.setDeleteable(true);
   },
   
   /**
    * @method
    * Set the outline color of the line.
    * 
    * @param {draw2d.util.Color/String} color The new color of the line.
    * @since 4.2.1
    **/
   setOutlineColor:function( color)
   {
     this.outlineColor = new draw2d.util.Color(color);
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * The outlien color of the text
    * 
    * @returns {draw2d.util.Color}
    * @since 4.2.1
    */
   getOutlineColor:function()
   {
     return this.outlineColor;
   },
   
   /**
    * @method
    * Set the outline stroke of the line to use.
    * 
    * @param {Number} w The new outline width of the line
    * @since 4.2.1
    **/
   setOutlineStroke:function( w )
   {
     this.outlineStroke=w;
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * The used outline line width.
    * 
    * @type {Number}
    * @since 4.2.1
    **/
   getOutlineStroke:function( )
   {
     return this.outlineStroke;
   },

   /**
    * @method
    * Don't call them manually. This will be done by the framework.<br>
    * Will be called if the object are moved via drag and drop.
    * Sub classes can override this method to implement additional stuff. Don't forget to call
    * the super implementation via <code>this._super(dx, dy, dx2, dy2);</code>
    * @private
    * @param {Number} dx the x difference between the start of the drag drop operation and now
    * @param {Number} dy the y difference between the start of the drag drop operation and now
    * @param {Number} dx2 The x diff since the last call of this dragging operation
    * @param {Number} dy2 The y diff since the last call of this dragging operation
    **/
   onDrag : function( dx, dy, dx2, dy2)
   {
       if(this.command ===null){
           return;
       }
       
       this.command.setTranslation(dx,dy);
       
       this.vertices.each(function(i,e){
           e.translate(dx2, dy2);
       });
       this.start=this.vertices.first().clone();
       this.end=this.vertices.last().clone();

       this.svgPathString = null;
       this._super(dx, dy, dx2, dy2);
   },

   onDragEnd : function()
   {
       this.setAlpha(this.originalAlpha);
       // Element ist zwar schon an seine Position, das Command muss aber trotzdem
       // in dem CommandStack gelegt werden damit das Undo funktioniert.
       //
       this.isInDragDrop = false;

       if(this.command===null){
           return;
       }

	   // we must undo the interim drag/drop translation of the line. The real translation will be done
	   // by the execute of the command. Yes - you are right. This is a HACK or design flaw :-/
	   this.getVertices().each($.proxy(function(i,e){
           e.translate(-this.command.dx, -this.command.dy);
       },this));

	   

       this.canvas.getCommandStack().execute(this.command);
	   this.command = null;
	   this.isMoving = false;
	   
	   // notify all installed policies
	   //
	   this.editPolicy.each($.proxy(function(i,e){
    	   if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
    		   e.onDragEnd(this.canvas, this);
    	   }
	   },this));
	   
	   // inform all other listener
	   this.fireMoveEvent();
   },

   /**
    * @method
    * Called when a user clicks on the element.
    * 
    * @template
    * @since 4.0.0
    */
   onClick: function(){
   },

   /**
    * @method
    * Set the line style for this object.
    * 
    * @param dash can be one of this ["", "-", ".", "-.", "-..", ". ", "- ", "--", "- .", "--.", "--.."]
    */
   setDashArray: function(dash){
       this.dasharray = dash;
       
       return this;
   },
   
   
   /**
    * @method
    * Set the width for the click hit test of this line.
    *
    * @param {Number} width the width of the line hit test.
    **/
   setCoronaWidth:function( width)
   {
      this.corona = width;
      
      return this;
   },


   /**
    * @method
    * Called by the framework. Don't call them manually.
    * 
    * @private
    **/
   createShapeElement:function()
   {
     var set=  this.canvas.paper.set();

     // the drop shadow or border line
     set.push(this.canvas.paper.path("M"+this.start.x+" "+this.start.y+"L"+this.end.x+" "+this.end.y));
     // the main path
     set.push(this.canvas.paper.path("M"+this.start.x+" "+this.start.y+"L"+this.end.x+" "+this.end.y));
     set.node = set.items[1].node;
     
     return set;
   },

   /**
    * @method
    * Trigger the repaint of the element.
    * 
    */
   repaint:function(attributes)
   {
       if(this.repaintBlocked===true || this.shape===null){
           return;
       }

       // don't override existing values
       //
       if(typeof attributes === "undefined"){
           attributes = {"stroke":this.lineColor.hash(),
                         "stroke-width":this.stroke,
                         "path":["M",this.start.x,this.start.y,"L",this.end.x,this.end.y].join(" ")};
       }
       else{
    	   if(typeof attributes.path ==="undefined"){
    		   attributes.path =["M",this.start.x,this.start.y,"L",this.end.x,this.end.y].join(" ");
    	   }
    	   attributes.stroke = this.lineColor.hash();
    	   attributes["stroke-width"]=this.stroke;
       }
       
       if(this.dasharray!==null){
           attributes["stroke-dasharray"]=this.dasharray;
       }
       
       this._super(attributes);

       if(this.outlineStroke>0){
           this.shape.items[0].attr({"stroke-width":(this.outlineStroke+this.stroke), "stroke":this.outlineColor.hash()});
           if(this.outlineVisible===false)
               this.shape.items[0].show();
           this.outlineVisible = true;
       }
       else if(this.outlineVisible===true){
           // reset them once
           this.shape.items[0].attr({"stroke-width":0, "stroke":"none"});
           this.shape.items[0].hide();
       }
   },
   
   /**
    * @method
    * Highlight the element or remove the highlighting
    * 
    * @param {Boolean} flag indicates glow/noGlow
    * @template
    */
   setGlow: function(flag){
	   if(this.isGlowing===flag){
		   return;
	   }
	   
	   if(flag===true){
		   // store old values for restore
		   this._lineColor = this.lineColor;
		   this._stroke = this.stroke;
		   
	       this.setColor( new draw2d.util.Color("#3f72bf"));
	       this.setStroke((this.stroke*4)|0);
	   }
	   else{
	       this.setColor(this._lineColor);
	       this.setStroke(this._stroke);
	   }
	   
	   this.isGlowing = flag;
	   
	   return this;
   },


   /**
    * You can't drag&drop the resize handles if the line not resizeable.
    * @type boolean
    **/
   isResizeable:function()
   {
     return true;
   },


   /**
    * Set the line width. This enforce a repaint of the line.
    * This method fires a <i>document dirty</i> event.
    *
    * @param {Number} w The new line width of the figure.
    **/
   setStroke:function(w)
   {
     this.stroke=parseFloat(w);
     
     this.repaint();
     
     return this;
   },


   /**
    * @method
    * The used line width.
    * 
    * @type {Number}
    **/
   getStroke:function( )
   {
     return this.stroke;
   },


   /**
    * @method
    * Set the color of the line.
    * This method fires a <i>document dirty</i> event.
    * 
    * @param {draw2d.util.Color|String} color The new color of the line.
    **/
   setColor:function( color)
   {
     this.lineColor = new draw2d.util.Color(color);
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * Return the current paint color.
    * 
    * @return {draw2d.util.Color} The paint color of the line.
    **/
   getColor:function()
   {
     return this.lineColor;
   },

   /**
    * @method
    * Translate the line with the given x/y offset.
    *
    * @param {Number} dx The new x translate offset
    * @param {Number} dy The new y translate offset
    * @since 4.1.0
    **/
   translate:function(dx , dy )
   {
       this.vertices.each(function(i,e){
           e.translate(dx, dy);
       });
       this.start=this.vertices.first().clone();
       this.end=this.vertices.last().clone();

       this.editPolicy.each($.proxy(function(i,e){
           if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
               e.moved(this.canvas, this);
           }
       },this));

       
       this.svgPathString = null;
       this.repaint();

       return this;
   },

   /**
    * @method
    * Set the start point of the line.
    * This method fires a <i>document dirty</i> event.
    *
    * @param {Number} x the x coordinate of the start point
    * @param {Number} y the y coordinate of the start point
    **/
   setStartPoint:function( x, y)
   {
     if(this.start.x===x && this.start.y===y){
        return;
     }

     this.start.setPosition(x, y);
     this.repaint();

     this.editPolicy.each($.proxy(function(i,e){
         if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
             e.moved(this.canvas, this);
         }
     },this));
     
     return this;
  },

   /**
    * Set the end point of the line.
    * This method fires a <i>document dirty</i> event.
    *
    * @param {Number} x the x coordinate of the end point
    * @param {Number} y the y coordinate of the end point
    **/
   setEndPoint:function(x, y)
   {
     if(this.end.x===x && this.end.y===y){
        return;
     }

     this.end.setPosition(x, y);
     this.repaint();

     this.editPolicy.each($.proxy(function(i,e){
         if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
             e.moved(this.canvas, this);
         }
     },this));
     
     return this;
 },

   /**
    * @method
    * Return the x coordinate of the start.
    * @deprecated
    * @return {Number}
    **/
   getStartX:function()
   {
     return this.start.x;
   },

   /**
    * @method
    * Return the y coordinate of the start.
    * 
    * @deprecated
    * @return {Number}
    **/
   getStartY:function()
   {
     return this.start.y;
   },

   /**
    * @method
    * Return the start point.
    * 
    * @return {draw2d.geo.Point}
    **/
   getStartPoint:function()
   {
     return this.start.clone();
   },


   /**
    * @method
    * Return the x coordinate of the end point
    * 
    * @deprecated
    * @return {Number}
    **/
   getEndX:function()
   {
     return this.end.x;
   },

   /**
    * @method
    * Return the y coordinate of the end point.
    * 
    * @deprecated
    * @return {Number}
    **/
   getEndY:function()
   {
     return this.end.y;
   },

   /**
    * @method
    * Return the end point.
    * 
    * @return {draw2d.geo.Point}
    **/
   getEndPoint:function()
   {
     return this.end.clone();
   },


   /**
    * @method
    * Return the Vertex with the given index.
    *
    * @param {Number} index the index of the vertex to return
    */
   getVertex:function( index)
   {
       return this.vertices.get(index);
   },
 
   /**
    * @method
    * Returns the vertices of the connection
    *
    * @return {draw2d.util.ArrayList} an draw2d.util.ArrayList of type draw2d.Point
    **/
   getVertices:function()
   {
       return this.vertices;
   },
   /* @deprecated */
   getPoints:function (){return this.getVertices();},
   
   getSegments: function()
   {
       var result = new draw2d.util.ArrayList();
       result.add({start: this.getStartPoint(), end: this.getEndPoint()});
       
       return result;
   },
   
   /**
    * @method
    * Returns the length of the line.
    * 
    * @return {Number}
    **/
   getLength:function()
   {
     // call native path method if possible
     if(this.shape!==null){
       return this.shape.getTotalLength();
     }
       
     return Math.sqrt((this.start.x-this.end.x)*(this.start.x-this.end.x)+(this.start.y-this.end.y)*(this.start.y-this.end.y));
   },

   /**
    * @method
    * Returns the angle of the line in degree.
    *
    * <pre>
    *                                 270�?
    *                               |
    *                               |
    *                               |
    *                               |
    * 180�? -------------------------+------------------------> +X
    *                               |                        0�?
    *                               |
    *                               |
    *                               |
    *                               V +Y
    *                              90�?
    * </pre>
    * @return {Number}
    **/
   getAngle:function()
   {
     var length = this.getLength();
     var angle = -(180/Math.PI) *Math.asin((this.start.y-this.end.y)/length);

     if(angle<0)
     {
        if(this.end.x<this.start.x){
          angle = Math.abs(angle) + 180;
        }
        else{
          angle = 360- Math.abs(angle);
        }
     }
     else
     {
        if(this.end.x<this.start.x){
          angle = 180-angle;
        }
     }
     return angle;
   },

   /**
    * @method
    * Returns the Command to perform the specified Request or null if the shape want cancel the 
    * operation or it can't operate the command.
    *
    * @param {draw2d.command.CommandType} request describes the Command being requested
    * @return {draw2d.command.Command} null or a Command
    **/
   createCommand:function( request)
   {
     if(request.getPolicy() === draw2d.command.CommandType.MOVE)
     {
         if(this.isDraggable()){
             return new draw2d.command.CommandMoveLine(this);
          }
     }
     if(request.getPolicy() === draw2d.command.CommandType.DELETE)
     {
        if(this.isDeleteable()===false){
           return null;
        }
        return new draw2d.command.CommandDelete(this);
     }
     
     return null;
   },

   /**
    * @method
    * Checks if the hands over coordinate close to the line. The 'corona' is considered
    * for this test. This means the point isn't direct on the line. Is it only close to the
    * line!
    *
    * @param {Number} px the x coordinate of the test point
    * @param {Number} py the y coordinate of the test point
    * @return {boolean}
    **/
   hitTest: function( px, py)
   {
     return draw2d.shape.basic.Line.hit(this.corona+ this.stroke, this.start.x,this.start.y, this.end.x, this.end.y, px,py);
   },
   
   /**
    * @method
    * Return all intersection points between the given Line.
    * 
    * @param other
    * @returns {draw2d.util.ArrayList}
    */
   intersection: function (other){
       var result = new draw2d.util.ArrayList();
       
       // empty result. the lines are equal...infinit array
       if(other === this){
           return result;
       }
       
       var segments1= this.getSegments();
       var segments2= other.getSegments();
       
       segments1.each(function(i, s1){
           segments2.each(function(j, s2){
               var p= draw2d.shape.basic.Line.intersection(s1.start, s1.end, s2.start, s2.end);
               if(p!==null){
                   result.add(p);
               }
           });
       });
       return result;
   },
   
   
   /**
    * @method 
    * Return an objects with all important attributes for XML or JSON serialization
    * 
    * @returns {Object}
    */
   getPersistentAttributes : function()
   {
       var memento = this._super();
       delete memento.x;
       delete memento.y;
       delete memento.width;
       delete memento.height;

       memento.stroke = this.stroke;
       memento.color  = this.getColor().hash();
       memento.outlineStroke = this.outlineStroke;
       memento.outlineColor = this.outlineColor.hash();
       if(this.editPolicy.getSize()>0){
           memento.policy = this.editPolicy.getFirstElement().NAME;
       }
       
       return memento;
   },
   
   /**
    * @method 
    * Read all attributes from the serialized properties and transfer them into the shape.
    * 
    * @param {Object} memento
    * @returns 
    */
   setPersistentAttributes : function(memento)
   {
       this._super(memento);

       if(typeof memento.stroke !=="undefined"){
           this.setStroke(parseFloat(memento.stroke));
       }
       if(typeof memento.color !=="undefined"){
           this.setColor(memento.color);
       }
       if(typeof memento.outlineStroke !=="undefined"){
           this.setOutlineStroke(memento.outlineStroke);
       }
       if(typeof memento.outlineColor !=="undefined"){
           this.setOutlineColor(memento.outlineColor);
       }
       if(typeof memento.policy !=="undefined"){
           try{
               this.installEditPolicy(eval("new "+memento.policy +"()" ));
           }
           catch(exc){
               debug.warn("Unable to install edit policy '"+memento.policy+"' forced by "+this.NAME+".setPersistendAttributes. Using default.");
           }
       }
   }
});


/**
 * see: http://en.wikipedia.org/wiki/Line-line_intersection
 * 
 * @param {draw2d.geo.Point} a1
 * @param {draw2d.geo.Point} a2
 * @param {draw2d.geo.Point} b1
 * @param {draw2d.geo.Point} b2
 * @private
 * @returns
 */
draw2d.shape.basic.Line.intersection = function(a1, a2, b1, b2) {
    var result=null;
    
    var ua_t = (b2.x - b1.x) * (a1.y - b1.y) - (b2.y - b1.y) * (a1.x - b1.x);
    var ub_t = (a2.x - a1.x) * (a1.y - b1.y) - (a2.y - a1.y) * (a1.x - b1.x);
    var u_b  = (b2.y - b1.y) * (a2.x - a1.x) - (b2.x - b1.x) * (a2.y - a1.y);

    if ( u_b != 0 ) {
        var ua = ua_t / u_b;
        var ub = ub_t / u_b;

        if ( 0 <= ua && ua <= 1 && 0 <= ub && ub <= 1 ) {
//        if ( 0 < ua && ua < 1 && 0 < ub && ub < 1 ) {
            result = new draw2d.geo.Point((a1.x + ua * (a2.x - a1.x))|0, (a1.y + ua * (a2.y - a1.y))|0);
            
            // determine if the lines are crossing or just touching
            //
            result.justTouching=( 0 == ua || ua == 1 || 0 == ub || ub == 1 );
        }
    }
    /*
    else {
        if ( ua_t == 0 || ub_t == 0 ) {
            result = null;// Coincident
        } else {
            result = null; // Parallel
        }
    }
    */
    return result;
};

/**
 * Static util function to determine is a point(px,py) on the line(x1,y1,x2,y2)
 * A simple hit test.
 * 
 * @return {boolean}
 * @static
 * @private
 * @param {Number} coronaWidth the accepted corona for the hit test
 * @param {Number} X1 x coordinate of the start point of the line
 * @param {Number} Y1 y coordinate of the start point of the line
 * @param {Number} X2 x coordinate of the end point of the line
 * @param {Number} Y2 y coordinate of the end point of the line
 * @param {Number} px x coordinate of the point to test
 * @param {Number} py y coordinate of the point to test
 **/
draw2d.shape.basic.Line.hit= function( coronaWidth, X1, Y1,  X2,  Y2, px, py)
{
  // Adjust vectors relative to X1,Y1
  // X2,Y2 becomes relative vector from X1,Y1 to end of segment
  X2 -= X1;
  Y2 -= Y1;
  // px,py becomes relative vector from X1,Y1 to test point
  px -= X1;
  py -= Y1;
  var dotprod = px * X2 + py * Y2;
  var projlenSq;
  if (dotprod <= 0.0) {
      // px,py is on the side of X1,Y1 away from X2,Y2
      // distance to segment is length of px,py vector
      // "length of its (clipped) projection" is now 0.0
      projlenSq = 0.0;
  } else {
      // switch to backwards vectors relative to X2,Y2
      // X2,Y2 are already the negative of X1,Y1=>X2,Y2
      // to get px,py to be the negative of px,py=>X2,Y2
      // the dot product of two negated vectors is the same
      // as the dot product of the two normal vectors
      px = X2 - px;
      py = Y2 - py;
      dotprod = px * X2 + py * Y2;
      if (dotprod <= 0.0) {
          // px,py is on the side of X2,Y2 away from X1,Y1
          // distance to segment is length of (backwards) px,py vector
          // "length of its (clipped) projection" is now 0.0
          projlenSq = 0.0;
      } else {
          // px,py is between X1,Y1 and X2,Y2
          // dotprod is the length of the px,py vector
          // projected on the X2,Y2=>X1,Y1 vector times the
          // length of the X2,Y2=>X1,Y1 vector
          projlenSq = dotprod * dotprod / (X2 * X2 + Y2 * Y2);
      }
  }
    // Distance to line is now the length of the relative point
    // vector minus the length of its projection onto the line
    // (which is zero if the projection falls outside the range
    //  of the line segment).
    var lenSq = px * px + py * py - projlenSq;
    if (lenSq < 0) {
        lenSq = 0;
    }
    return Math.sqrt(lenSq)<coronaWidth;
};


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.PolyLine
 * 
 * A PolyLine is a line with more than 2 points.
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Line
 */
draw2d.shape.basic.PolyLine = draw2d.shape.basic.Line.extend({
    
	NAME : "draw2d.shape.basic.PolyLine",
	
    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     */
    init: function( router ) {
        
      // internal status handling for performance reasons
      //
      this.svgPathString = null;
      this.oldPoint=null;
    
      this.router = router || draw2d.shape.basic.PolyLine.DEFAULT_ROUTER;
      this.routingRequired = true;
  
      this.radius = 2;
      
      // all line segments with start/end as simple object member
      this.lineSegments = new draw2d.util.ArrayList();

      this._super();
    },
    
    /**
     * @method
     * Sets the corner radius or the edges. 
     * 
     * @param {Number} radius
     * @since 4.2.1
     */
     setRadius: function(radius){
        this.radius = radius;
        this.svgPathString =null;
        this.repaint();
        
        return this;
    },
    
    /**
     * @method
     * Get the corner radius of the edges.
     * 
     * @return {Number}
     * @since 4.2.1
     */
    getRadius:function() {
        return this.radius;
    },
    
    
    /**
     * @method
     * Set the start point of the line.
     *
     * @param {Number} x the x coordinate of the start point
     * @param {Number} y the y coordinate of the start point
     **/
    setStartPoint:function( x, y){
        this.repaintBlocked=true;
        this._super(x,y);
        this.calculatePath();
        
        this.repaintBlocked=false;
        this.repaint();
    },

    /**
     * Set the end point of the line.
     *
     * @param {Number} x the x coordinate of the end point
     * @param {Number} y the y coordinate of the end point
     **/
    setEndPoint:function(x, y)
    {
        this.repaintBlocked=true;
        this._super(x,y);
        this.calculatePath();
        
        this.repaintBlocked=false;
        this.repaint();
    },

    /**
     * @method
     * Inserts the draw2d.geo.Point object into the vertex list of the polyline just after the object with the given index.
     *  
     * @param {Number} index the insert index
     * @param {Number|draw2d.geo.Point} x the x coordinate or the draw2d.geo.Point object
     * @param {Number} [y] the y coordinate or undefined of the second argument is a point
     * 
     * @since 4.0.0
     */
    addVertex:function(x, y) 
    {
        this.vertices.add(new draw2d.geo.Point(x,y));
        
        this.start=this.vertices.first().clone();
        this.end=this.vertices.last().clone();
       
        this.svgPathString = null;
        this.repaint();

        if(!this.selectionHandles.isEmpty()){
            this.editPolicy.each($.proxy(function(i, e) {
                if (e instanceof draw2d.policy.figure.SelectionFeedbackPolicy) {
                    e.onUnselect(this.canvas, this);
                    e.onSelect(this.canvas, this);
                }
            }, this));
        }

        return this;
    },

    /**
     * @method
     * Update the vertex at the give position with the new coordinate
     * 
     * @param {Number} index the index of the vertex to update
     * @param {Number|draw2d.geo.Point} x the x coordinate or the draw2d.geo.Point object
     * @param {Number} [y] the y coordinate or undefined of the second argument is a point
     * 
     * @since 4.0.0
     */
    setVertex : function(index, x, y) 
    {
        if(x instanceof draw2d.geo.Point){
            y = x.y;
            x = x.x;
        }
        
        var vertex = this.vertices.get(index);

        // invalid point or nothing todo
        //
        if (vertex === null || (vertex.x === x && vertex.y === y)) {
            return;
        }

        vertex.x = parseFloat(x);
        vertex.y = parseFloat(y);
        
        this.start=this.vertices.first().clone();
        this.end=this.vertices.last().clone();

        this.svgPathString = null;
        this.routingRequired=true;
        this.repaint();

        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        return this;
    },

    /**
     * @method
     * Update the vertices of the object. The given array is copied and assigned.
     * 
     * @param {draw2d.util.ArrayList} vertices the new vertices of the polyline. 
     * 
     * @since 4.0.1
     */
    setVertices : function(vertices) 
    {
        this.vertices= vertices.clone();

        this.start=this.vertices.first().clone();
        this.end=this.vertices.last().clone();

        // update the UI and the segment parts
        this.svgPathString = null;
        this.repaint();

        // notify the listener about the changes
        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        return this;
    },

    /**
     * @method
     * Inserts the draw2d.geo.Point object into the vertex list of the polyline just after the object with the given index.
     *  
     * @param {Number} index the insert index
     * @param {Number|draw2d.geo.Point} x the x coordinate or the draw2d.geo.Point object
     * @param {Number} [y] the y coordinate or undefined of the second argument is a point
     * 
     * @since 4.0.0
     */
    insertVertexAt:function(index, x, y) 
    {
        var vertex = new draw2d.geo.Point(x,y);

        this.vertices.insertElementAt(vertex,index);

        this.start=this.vertices.first().clone();
        this.end=this.vertices.last().clone();

        this.svgPathString = null;
        this.repaint();

        if(!this.selectionHandles.isEmpty()){
	        this.editPolicy.each($.proxy(function(i, e) {
	            if (e instanceof draw2d.policy.figure.SelectionFeedbackPolicy) {
	                e.onUnselect(this.canvas, this);
	                e.onSelect(this.canvas, this);
	            }
	        }, this));
        }

        return this;
    },


    /**
     * @method
     * Remove a vertex from the polyline and return the removed point.
     * 
     * @param index
     * @returns {draw2d.geo.Point} the removed point
     * @since 4.0.0
     */
    removeVertexAt:function(index) 
    {
        var removedPoint = this.vertices.removeElementAt(index);
        
        this.start=this.vertices.first().clone();
        this.end=this.vertices.last().clone();

        this.svgPathString = null;
        this.repaint();

        if(!this.selectionHandles.isEmpty()){
	        this.editPolicy.each($.proxy(function(i, e) {
	            if (e instanceof draw2d.policy.figure.SelectionFeedbackPolicy) {
	                e.onUnselect(this.canvas, this);
	                e.onSelect(this.canvas, this);
	            }
	        }, this));
        }

        return removedPoint;
    },

    
    /**
     * @method
     * Set the router for this connection.
     * 
     * @param {draw2d.layout.connection.ConnectionRouter} [router] the new router for this connection or null if the connection should use the default routing
     **/
    setRouter:function(router)
    {
      if(this.router !==null){
          this.router.onUninstall(this);
      }
      
      if(typeof router ==="undefined" || router===null){
          this.router = new draw2d.layout.connection.NullRouter();
      }
      else{
          this.router = router;
      }
      
      this.router.onInstall(this);
      
      this.routingRequired =true;
    
      // repaint the connection with the new router
      this.repaint();
      
      return this;
    },
    
    /**
     * @method
     * Return the current active router of this connection.
     *
     * @type draw2d.layout.connection.ConnectionRouter
     **/
    getRouter:function()
    {
      return this.router;
    },
    
    /**
     * @method
     * Calculate the path of the polyline
     * 
     * @private
     */
    calculatePath: function()
    {
        
        if(this.shape===null){
            return;
        }
    
        this.svgPathString = null;
        
        var oldVertices = this.vertices;
        
        // cleanup the routing cache
        //
        this.oldPoint=null;
        this.lineSegments = new draw2d.util.ArrayList();
        this.vertices     = new draw2d.util.ArrayList();
    
        // Use the internal router
        //
        this.router.route(this, oldVertices);
        this.routingRequired=false;
    },
    
    /**
     * @private
     **/
    repaint : function(attributes)
    {
      if(typeof attributes ==="undefined")
        attributes = {};
      
      if(this.repaintBlocked===true || this.shape===null){
          return;
      }
      
      if(this.svgPathString===null || this.routingRequired===true){
          this.calculatePath();
      }
 
      $.extend(attributes, {path:this.svgPathString,"stroke-linecap":"round", "stroke-linejoin":"round"}, attributes);
       
      this._super(attributes);
    },
    
    
    
    /**
     * @method
     * Called by the framework during drag&drop operations.
     * 
     * @param {draw2d.Figure} draggedFigure The figure which is currently dragging
     * 
     * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didnt want a drop event
     * @template
     **/
    onDragEnter : function( draggedFigure ){
    	return this;
    },
 
    /**
     * @method
     * Called if the DragDrop object leaving the current hover figure.
     * 
     * @param {draw2d.Figure} draggedFigure The figure which is currently dragging
     * @template
     **/
    onDragLeave:function( draggedFigure ){
    },


    /**
     * @method
     * Return all line segments of the polyline.
     * 
     * @returns {draw2d.util.ArrayList}
     */
    getSegments: function(){
        return this.lineSegments;
    },
    
    /**
     * @method
     * used for the router to add the calculated points
     * 
     * @private
     *
     **/
    addPoint:function(/*:draw2d.geo.Point*/ p, y)
    {
      if(typeof y!=="undefined"){
          p = new draw2d.geo.Point(p, y);
      }
      this.vertices.add(p);

      if(this.oldPoint!==null){
        // store the painted line segment for the "mouse selection test"
        // (required for user interaction)
        this.lineSegments.add({start: this.oldPoint, end:p});
      }
      this.svgPathString=null;
      this.oldPoint = p;
    },

    /**
     * @see draw2d.Figure#onOtherFigureHasMoved
     **/
    onOtherFigureIsMoving:function(/*:draw2d.Figure*/ figure)
    {
      this.repaintBlocked=true;
      this._super(figure);
      this.calculatePath();
      
      this.repaintBlocked=false;
      this.repaint();
    },
    
    /**
     * @method
    * Checks if the hands over coordinate close to the line. The 'corona' is considered
    * for this test. This means the point isn't direct on the line. Is it only close to the
    * line!
    *
    * @param {Number} px the x coordinate of the test point
    * @param {Number} py the y coordinate of the test point
    * @return {boolean}
     **/
    hitTest:function( px, py)
    {
      for(var i = 0; i< this.lineSegments.getSize();i++){
         var segment = this.lineSegments.get(i);
         if(draw2d.shape.basic.Line.hit(this.corona, segment.start.x,segment.start.y,segment.end.x, segment.end.y, px,py)){
           return true;
         }
      }
      return false;
    },

    /**
     * @method
     * Returns the Command to perform the specified Request or null.
      *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * 
     * @return {draw2d.command.Command}
     **/
    createCommand:function(request) 
    {
 
      if(request.getPolicy() === draw2d.command.CommandType.DELETE){
        if(this.isDeleteable()===true){
          return new draw2d.command.CommandDelete(this);
        }
      }
      else if(request.getPolicy() === draw2d.command.CommandType.MOVE_VERTEX){
          if(this.isResizeable()===true){
              return new draw2d.command.CommandMoveVertex(this);
            }
      }
      else if(request.getPolicy() === draw2d.command.CommandType.MOVE_VERTICES){
          if(this.isResizeable()===true){
              return new draw2d.command.CommandMoveVertices(this);
            }
      }
    
      return this._super(request);
    },
    
    getPersistentAttributes : function()
    {   
        var memento = this._super();
        
        memento.router = this.router.NAME;
        memento.radius = this.radius;
      
        memento = this.router.getPersistentAttributes(this, memento);
        
        return memento;
    },
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     * @returns 
     */
    setPersistentAttributes : function(memento)
    {
        this._super(memento);

        if(typeof memento.router !=="undefined"){
            try{
                this.setRouter(eval("new "+memento.router+"()"));
            }
            catch(exc){
                debug.warn("Unable to install router '"+memento.router+"' forced by "+this.NAME+".setPersistendAttributes. Using default");
            }
        }
        
        if(typeof memento.radius !=="undefined")
            this.setRadius(memento.radius);

        this.router.setPersistentAttributes(this, memento);

        this.start=this.vertices.first().clone();
        this.end=this.vertices.last().clone();
    }
});

/**
 * The default ConnectionRouter for the running applicaiton. Set this to your wanted implementation
 * {@link draw2d.layout.connection.ConnectionRouter}
 */
draw2d.shape.basic.PolyLine.DEFAULT_ROUTER= new draw2d.layout.connection.ManhattanConnectionRouter();

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Diamond
 * A Diamond Figure.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var d1 =  new draw2d.shape.basic.Diamond();
 *     var d2 =  new draw2d.shape.basic.Diamond();
 *     
 *     canvas.addFigure(d1,10,10);
 *     canvas.addFigure(d2,100,10);
 *     
 *     d2.setBackgroundColor("#f0f000");
 *     d2.setAlpha(0.7);
 *     d2.setDimension(100,60);
 *     
 *     canvas.setCurrentSelection(d2);
 *     
 * @author Andreas Herz
 * @extends draw2d.VectorFigure
 */
draw2d.shape.basic.Diamond = draw2d.VectorFigure.extend({
    NAME : "draw2d.shape.basic.Diamond",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init: function( width, height) {
      this._super();
    
      this.setBackgroundColor("#00a3f6");
      this.setColor("#1B1B1B");

      // set some good defaults
      //
      if(typeof width === "undefined"){
        this.setDimension(50, 90);
      }
      else{
        this.setDimension(width, height);
      }
    },

    /**
     * @inheritdoc
     **/
    repaint : function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return this;
        } 
        
        if (typeof attributes === "undefined") {
            attributes = {};
        }
        
        var box = this.getBoundingBox();
        var path = ["M",box.x+(box.w/2)," ",box.y];         // Go to the top center..
        path.push("L", box.x+box.w, " ", box.y+box.h/2);    // ...draw line to the right middle
        path.push("L", box.x+box.w/2, " ", box.y+ box.h);   // ...bottom center...
        path.push("L", box.x, " ", box.y+box.h/2);          // ...left middle...
        path.push("L", box.x+box.w/2, " ", box.y);          // and close the path
        attributes.path = path.join("");

        this._super(attributes);
        
        return this;
    },


    /**
     * @method
     * Called by the framework. Don't call them manually.
     * 
     * @private
     **/
    createShapeElement:function()
    {
      // create dummy line
      return this.canvas.paper.path("M0 0L1 1");
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.Image
 * Simple Image shape.
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.node.Node
 */
draw2d.shape.basic.Image = draw2d.shape.node.Node.extend({
    NAME : "draw2d.shape.basic.Image",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     * @param {Number} path relative or absolute path of the image
     * @param {Number} width initial width of the shape
     * @param {Number} height initial height of the shape
     */
    init : function(path,  width, height)
    {
        this.path = path;
        this._super(width, height);
    },
      

    /**
     * @method
     * Set the image path attribute of the Image shape and repaint them.
     * The path can be relative or absolute
     * 
     * @param path
     * @since 2.8.0
     */
    setPath: function(path){
        this.path = path;
        this.repaint();
        
        return this;
    },
    
    /**
     * @method
     * Return the image path attribute of the shape.
     * 
     * @returns {String}
     * @since 2.8.0
     */
    getPath: function(){
        return this.path;
    },
    
   /**
    * @method
    * propagate all attributes like color, stroke,... to the shape element
    **/
    repaint : function(attributes)
    {
        if (this.repaintBlocked===true || this.shape === null){
            return this;
        }

        if(typeof attributes === "undefined" ){
            attributes = {};
        }

        attributes.x = this.getAbsoluteX();
        attributes.y = this.getAbsoluteY();
        attributes.width = this.getWidth();
        attributes.height = this.getHeight();
        attributes.src = this.path;
        
        this._super(attributes);
        
        return this;
    },

    /**
     * @method
     * 
     * @inheritdoc
     */
    createShapeElement : function()
    {
       return this.canvas.paper.image(this.path,this.getX(),this.getY(),this.getWidth(), this.getHeight());
    },
    

    /**
     * @method 
     * Return an objects with all important attributes for XML or JSON serialization
     * 
     * @returns {Object}
     */
    getPersistentAttributes : function()
    {
        var memento = this._super();
        
        memento.path = this.path;
        
        return memento;
    },
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     * @returns 
     */
    setPersistentAttributes : function(memento)
    {
        this._super(memento);
        if(typeof memento.path !=="undefined"){
            this.setPath(memento.path);
        }
    }

});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.shape.basic.Polygon
 * A Polygon figure.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var p1 =  new draw2d.shape.basic.Polygon();
 *     var p2 =  new draw2d.shape.basic.Polygon();
 *     
 *     canvas.addFigure(p1,10,10);
 *     canvas.addFigure(p2,100,10);
 *     
 *     p2.setBackgroundColor("#f0f000");
 *     p2.setAlpha(0.7);
 *     p2.setDimension(100,60);
 *     
 *     canvas.setCurrentSelection(p2);
 *     
 * @author Andreas Herz
 * @extends draw2d.VectorFigure
 */
draw2d.shape.basic.Polygon = draw2d.VectorFigure.extend({
    
    NAME: "draw2d.shape.basic.Polygon",
    
    init:function(w,h)
    {
      this._super();
      this.radius = 0;
      this.vertices   = new draw2d.util.ArrayList();
      
      if(typeof w==="undefined"){
          w = 50;
          h = 50;
      }
      this.addVertex(new draw2d.geo.Point(0,0) );
      this.addVertex(new draw2d.geo.Point(w,0) );
      this.addVertex(new draw2d.geo.Point(w,h) );
      
      this.svgPathString=null;
      
      this.installEditPolicy(new draw2d.policy.figure.VertexSelectionFeedbackPolicy());
    },
    
    /**
     * @method
     * Sets the corner radius or the edges. 
     * 
     * @param {Number} radius
     * @since 4.2.1
     */
     setRadius: function(radius){
        this.radius = radius;
        this.svgPathString =null;
        this.repaint();
        
        return this;
    },
    
    /**
     * @method
     * Get the corner radius of the edges.
     * 
     * @return {Number}
     * @since 4.2.1
     */
    getRadius:function() {
        return this.radius;
    },
    
 
    /**
     * @method
     * Called by the framework. Don't call them manually.
     * 
     * @private
     **/
    createShapeElement:function()
    {
        // return some good default...
        return this.canvas.paper.path("M0 10L100 100");
    },

    /**
     * @method
     * calculate the path of the polygon
     * 
     */
    calculatePath: function(){
        var radius = this.getRadius();
        var path = [];
        if(radius ===0){
            var length = this.vertices.getSize();
            var p = this.vertices.get(0);
            path.push("M",(p.x|0)+0.5," ",(p.y|0)+0.5);
            for(var i=1;i<length;i++){
                  p = this.vertices.get(i);
                  path.push("L", (p.x|0)+0.5, " ", (p.y|0)+0.5);
            }
            path.push("Z");
        }
        else{
            length = this.vertices.getSize();
            var start = this.vertices.first();
            var end   = this.vertices.last();
            var begin   = this.insetPoint(start,end, radius);
            path.push("M", (begin.x|0)+0.5, ",", (begin.y|0)+0.5);
            for( var i=0 ;i<length;i++){
                  start = this.vertices.get(i);
                  end   = this.vertices.get((i+1)%length);
                  modStart = this.insetPoint(start,end, radius);
                  modEnd   = this.insetPoint(end,start,radius);
                  path.push("Q",start.x,",",start.y," ", (modStart.x|0)+0.5, ", ", (modStart.y|0)+0.5);
                  path.push("L", (modEnd.x|0)+0.5, ",", (modEnd.y|0)+0.5);
            }
        }
        this.svgPathString = path.join("");
        return this;
    },
    
    insetPoint: function(start, end, distanceFromStart){
        if(start.equals(end)){
            return start;
        }
        var vx = start.x-end.x;
        var vy = start.y-end.y;
        var length = Math.sqrt(vx*vx + vy*vy);
        var localDistance = Math.min(length/2,distanceFromStart);
        return {x: end.x + vx/length * (length - localDistance),
               y: end.y + vy/length * (length - localDistance)};

    },
    
    /**
     * @method
     * Trigger the repaint of the element.
     * 
     */
    repaint:function(attributes)
    {
        if(this.shape===null){
            return;
        }

        if(this.svgPathString===null){
            this.calculatePath();
        }
        
        if(typeof attributes === "undefined"){
            attributes = {};
        }
        
        if(typeof attributes.path ==="undefined"){
            attributes.path = this.svgPathString;
        }
        
        this._super(attributes);
    },


    /**
     * @method
     * Translate the figure with the given x/y offset. This method modifies all
     * vertices and the bounding box.
     *
     * @param {Number} dx The new x translate offset
     * @param {Number} dy The new y translate offset
     **/
    translate:function(dx , dy )
    {
        this.vertices.each(function(i,e){
            e.translate(dx,dy);
        });
        this.svgPathString = null;
        this.repaint();
        
        this.updateBoundingBox();
        
        // Update the resize handles if the user change the position of the
        // element via an API call.
        //
        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        this.fireMoveEvent();

        return this;
    },
  
    /**
     * @method
     * Change the position of the polygon. This method call updates all vertices.
     * 
     * @param x
     * @param y
     */
    setPosition : function(x, y) {
        var dx = x-this.minX;
        var dy = y-this.minY;
        this.translate(dx,dy);
        
        return this;
    },
    
    /**
     * @method
     * Change the dimension of the polygon. Note - This is not done by view transformation. It
     * is done by modify all vertices.
     * 
     * @param w
     * @param h
     */
    setDimension:function(w, h){
        var oldWidth = this.width;
        var oldHeight= this.height;
        
        this._super(w,h);
        
        var fracWidth  = (1/oldWidth)*this.width;
        var fracHeight = (1/oldHeight)*this.height;
        
        this.vertices.each($.proxy(function(i,e){
            // calculate difference between point and figure origin
            var diffX = (e.getX()-this.x)*fracWidth;
            var diffY = (e.getY()-this.y)*fracHeight;
            e.setPosition(this.x+diffX,this.y+diffY);
        },this));

        this.svgPathString = null;
        this.repaint();
        
        return this;
    },
    
    /**
     * @method
     * Return all vertices of the polygon.
     * 
     * @returns {draw2d.util.ArrayList}
     */
    getVertices: function(){
        return this.vertices;
    },
    
    /**
     * @method
     * Update the vertex at the given index. The method call didn't have any effect 
     * if the vertex didn't exists.
     * 
     * @param index
     * @param x
     * @param y
     */
    setVertex : function(index, x, y) 
    {
        var vertex = this.vertices.get(index);

        // invalid point or nothing todo
        //
        if (vertex === null || (vertex.x === x && vertex.y === y)) {
            return;
        }

        vertex.x = parseFloat(x);
        vertex.y = parseFloat(y);
        
        this.svgPathString = null;
        this.repaint();

        this.updateBoundingBox();
        
        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        return this;
    },
    
    /**
     * @method
     * Append a new vertex to the polygon.
     * 
     * @param x
     * @param y
     */
    addVertex : function( x, y) 
    {
        var vertex = null;
        if(x instanceof draw2d.geo.Point){
            vertex =x.clone();
        }
        else{
            vertex =new draw2d.geo.Point(x,y);
        }
        this.vertices.add(vertex);
        
      
        this.svgPathString = null;
        this.repaint();

        this.updateBoundingBox();
        
        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        return this;
    },

    /**
     * Insert a new vertex at the given index. All vertices will be shifted to 
     * free the requested index.
     * 
     * @param index
     * @param x
     * @param y
     */
    insertVertexAt:function(index, x, y) 
    {
        var vertex = new draw2d.geo.Point(x,y);

        this.vertices.insertElementAt(vertex,index);
        
        this.svgPathString = null;
        this.repaint();

        this.updateBoundingBox();
        
        if(!this.selectionHandles.isEmpty()){
	        this.editPolicy.each($.proxy(function(i, e) {
	            if (e instanceof draw2d.policy.figure.SelectionFeedbackPolicy) {
	                e.onUnselect(this.canvas, this);
	                e.onSelect(this.canvas, this);
	            }
	        }, this));
        }

        return this;
    },


    /**
     * @method
     * Remove a vertex from the polygon and return the removed point.
     * 
     * @param index
     * @returns {draw2d.geo.Point} the removed point
     */
    removeVertexAt:function(index) 
    {
        // a polygon need at least 3 vertices
        //
        if(this.vertices.getSize()<=3){
            return null;
        }
        
        var vertex = this.vertices.removeElementAt(index);
        
        this.svgPathString = null;
        this.repaint();

        this.updateBoundingBox();
        
        if(!this.selectionHandles.isEmpty()){
	        this.editPolicy.each($.proxy(function(i, e) {
	            if (e instanceof draw2d.policy.figure.SelectionFeedbackPolicy) {
	                e.onUnselect(this.canvas, this);
	                e.onSelect(this.canvas, this);
	            }
	        }, this));
        }

        return vertex;
    },
    

    
    updateBoundingBox: function(){
        this.minX = this.x= Math.min.apply(Math,$.map(this.vertices.asArray(),function(n,i){return n.x;}));
        this.minY = this.y= Math.min.apply(Math,$.map(this.vertices.asArray(),function(n,i){return n.y;}));
        this.maxX = Math.max.apply(Math,$.map(this.vertices.asArray(),function(n,i){return n.x;}));
        this.maxY = Math.max.apply(Math,$.map(this.vertices.asArray(),function(n,i){return n.y;}));
        this.width = this.maxX - this.minX;
        this.height= this.maxY - this.minY;
    },
    
    
    /**
     * @method
     * Returns the Command to perform the specified Request or null.
      *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * 
     * @return {draw2d.command.Command}
     **/
    createCommand:function(request) 
    {
 
      if(request.getPolicy() === draw2d.command.CommandType.MOVE_VERTEX){
          if(this.isResizeable()===true){
              return new draw2d.command.CommandMoveVertex(this);
            }
      }
    
      return this._super(request);
    },
   
    
    getPersistentAttributes : function()
    {   
        var memento = this._super();
        
        memento.radius = this.radius;
        memento.vertices = [];
        
        this.vertices.each(function(i,e){
            memento.vertices.push({x:e.x, y:e.y});
        });
 
        return memento;
    },
    
    setPersistentAttributes : function( memento)
    {
        this._super(memento);
        
        if(typeof memento.radius !=="undefined")
           this.setRadius(memento.radius);
        
        // restore the points from the JSON data and add them to the polyline
        //
        if(typeof memento.vertices !=="undefined"){
            this.vertices = new draw2d.util.ArrayList();
            $.each(memento.vertices, $.proxy(function(i,e){
                this.addVertex(e.x,e.y);
            },this));
        }
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.Connection
 * See the example:
 *
 *     @example preview small frame
 *     
 *     // create and add two nodes which contains Ports (In and OUT)
 *     //
 *     var start = new draw2d.shape.node.Start();
 *     var end   = new draw2d.shape.node.End();
        
 *     // ...add it to the canvas 
 *     canvas.addFigure( start, 50,50);
 *     canvas.addFigure( end, 230,80);
 *          
 *     // Create a Connection and connect the Start and End node
 *     //
 *     var c = new draw2d.Connection();
 *      
 *     // Set the endpoint decorations for the connection
 *     //
 *     c.setSourceDecorator(new draw2d.decoration.connection.BarDecorator());
 *     c.setTargetDecorator(new draw2d.decoration.connection.DiamondDecorator());   
 *     // Connect the endpoints with the start and end port
 *     //
 *     c.setSource(start.getOutputPort(0));
 *     c.setTarget(end.getInputPort(0));
 *           
 *     // and finally add the connection to the canvas
 *     canvas.addFigure(c);
 *     
 * Connections figures are used to display a line between two points. The Connection interface extends 
 * {@link draw2d.shape.basic.PolyLine PolyLine}.<br>
 * The source and target endpoints of a connection are each defined using a {@link draw2d.layout.anchor.ConnectionAnchor ConnectionAnchor}. 
 * These endpoints, along with any other points on the connection, are set by the connection's  {@link draw2d.layout.connection.ConnectionRouter ConnectionRouter}. 
 * <br>
 * Usually every connection in a drawing has the same router instance. Connections with 
 * common endpoints can share anchor instances as well.
 * 
 * <h2>Connection Usage</h2>
 * 
 * Connections are created and added just like any other figure. Unlike normal figures, you must not set the 
 * bounds of a connection. Instead, you must provide the source and target port and let the connection router 
 * calculate the connection's points. The connection then determines its own bounding box.<br>
 * <br>
 * A connection has a simple router by default - one that can connect the source and target anchors. But additional routers 
 * are available and can be set on the connection. Some routers can handle constraints for the connection. Note that when 
 * setting a routing constraint on a connection, you must first set the router which will use that constraint.<br>
 * <br>
 * 
 * <b>TODO:<br></b>
 * <i>
 * A convenient way to share the router with all connections and to place connections above the drawing is to use a 
 * ConnectionLayer. The layer has a connection router property which it shares with every child that's a connection. 
 * You can update this property and easily change every connection's router at once.
 * </i>
 * <br>
 * <br>
 * <h2>Routing and Anchors</h2>
 * A connection always has a router and it must set at least two ports on the connection: the source and target 
 * endpoints. By default, or when set to null, the connection's routing will be performed by an internal default router. 
 * The ends are placed with the help of {@link draw2d.layout.anchor.ConnectionAnchor anchors}. An 
 * {@link draw2d.layout.anchor.ConnectionAnchor anchors} is a fixed or calculated location, usually associated with some 
 * figure. For example, the {@link draw2d.layout.anchor.ChopboxConnectionAnchor ChopboxAnchor} finds the point at which a 
 * line going to the reference point intersects a box, such as the bounds of a figure. The reference point is either 
 * the anchor at the opposite end, or a bendpoint or some other point nearest to the anchor. 
 * <br>
 * {@img jsdoc_chopbox.gif ChopboxAnchor}
 * <br>
 * The router calculates the endpoints and any other points in the middle of the connection. It then sets the points on the 
 * connection by calling {@link draw2d.shape.basic.PolyLine#addPoint Polyline.addPoint}. The connection's existing point list 
 * can be reused to reduce garbage, but the points must be set on the connection anyway so that it knows about any changes made.
 * <br>
 * <h2>Adding Decorations and Children to Connections</h2>
 * Like most figures, Connection supports the addition of children. The children might be a label that 
 * decorate the connection. The placement of each type of decoration can vary, so a {@link draw2d.layout.locator.ConnectionLocator ConnectionLocator} 
 * is used to delegate to each child's constraint object, a Locator. <br>
 * <br>
 * {@link draw2d.decoration.connection.Decorator Decorator} can be used to create and render a rotatable shape at 
 * the end or start of a connection like arrows or boxes. Examples are {@link draw2d.decoration.connection.ArrowDecorator ArrowDecorator}, {@link draw2d.decoration.connection.BarDecorator BarDecorator} or {@link draw2d.decoration.connection.CircleDecorator CircleDecorator}
 * <br>
 * <h2>Connection Layout</h2>
 * Connections extend the process of validation and layout to include routing. Since layout is the process of positioning children, routing must 
 * come first. This allows a child's locator to operate on the connection's newly-routed points.<br>
 * Check out [Class System Guide](#!/guide/class_system) for additional reading.
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.PolyLine
 */
draw2d.Connection = draw2d.shape.basic.PolyLine.extend({
    NAME : "draw2d.Connection",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * @param {draw2d.layout.connection.ConnectionRouter} router The router to use for this connection
     */
    init: function( router) {
      this._super(router );
      
      this.sourcePort = null;
      this.targetPort = null;
    
      this.oldPoint=null;
      
      this.sourceDecorator = null; /*:draw2d.ConnectionDecorator*/
      this.targetDecorator = null; /*:draw2d.ConnectionDecorator*/
      
      // decoration of the polyline
      //
      this.sourceDecoratorNode = null;
      this.targetDecoratorNode=null;
      
      this.setColor("#1B1B1B");
      this.setStroke(1);
      
      this.isMoving=false;
   },
    

    /**
     * @private
     **/
    disconnect : function()
    {
        if (this.sourcePort !== null) {
            this.sourcePort.detachMoveListener(this);
            this.fireSourcePortRouteEvent();
        }

        if (this.targetPort !== null) {
            this.targetPort.detachMoveListener(this);
            this.fireTargetPortRouteEvent();
        }
    },
    
    
    /**
     * @private
     **/
    reconnect : function()
    {
        if (this.sourcePort !== null) {
            this.sourcePort.attachMoveListener(this);
            this.fireSourcePortRouteEvent();
        }

        if (this.targetPort !== null) {
            this.targetPort.attachMoveListener(this);
            this.fireTargetPortRouteEvent();
        }
        this.routingRequired =true;
        this.repaint();
    },
    
    
    /**
     * You can't drag&drop the resize handles of a connector.
     * @type boolean
     **/
    isResizeable : function()
    {
        return this.isDraggable();
    },
    
   
    /**
     * @method
     * Add a child figure to the Connection. The hands over figure doesn't support drag&drop 
     * operations. It's only a decorator for the connection.<br>
     * Mainly for labels or other fancy decorations :-)
     *
     * @param {draw2d.Figure} figure the figure to add as decoration to the connection.
     * @param {draw2d.layout.locator.ConnectionLocator} locator the locator for the child. 
    **/
    addFigure : function(child, locator)
    {
        // just to ensure the right interface for the locator.
        // The base class needs only 'draw2d.layout.locator.Locator'.
        if(!(locator instanceof draw2d.layout.locator.ConnectionLocator)){
           throw "Locator must implement the class draw2d.layout.locator.ConnectionLocator"; 
        }
        
        this._super(child, locator);
    },
    

    /**
     * @method
     * Set the ConnectionDecorator for this object.
     *
     * @param {draw2d.decoration.connection.Decorator} the new source decorator for the connection
     **/
    setSourceDecorator:function( decorator)
    {
      this.sourceDecorator = decorator;
      this.routingRequired = true;
      if(this.sourceDecoratorNode!==null){
          this.sourceDecoratorNode.remove();
          this.sourceDecoratorNode=null;
      }
      this.repaint();
    },
    
    /**
     * @method
     * Get the current source ConnectionDecorator for this object.
     *
     * @returns draw2d.decoration.connection.Decorator
     **/
    getSourceDecorator:function()
    {
      return this.sourceDecorator;
    },
    
    /**
     * @method
     * Set the ConnectionDecorator for this object.
     *
     * @param {draw2d.decoration.connection.Decorator} the new target decorator for the connection
     **/
    setTargetDecorator:function( decorator)
    {
      this.targetDecorator = decorator;
      this.routingRequired =true;
      if(this.targetDecoratorNode!==null){
          this.targetDecoratorNode.remove();
          this.targetDecoratorNode=null;
      }      
      this.repaint();
    },
    
    /**
     * @method
     * Get the current target ConnectionDecorator for this object.
     *
     * @returns draw2d.decoration.connection.Decorator
     **/
    getTargetDecorator:function()
    {
      return this.targetDecorator;
    },
    

    /**
     * @method
     * Calculate the path of the polyline.
     * 
     * @private
     */
    calculatePath: function()
    {
        
        if(this.sourcePort===null || this.targetPort===null){
            return;
        }
        
        this._super();
    },
    
    /**
     * @private
     **/
    repaint:function(attributes)
    {
      if(this.repaintBlocked===true || this.shape===null){
          return;
      }
      
      if(this.sourcePort===null || this.targetPort===null){
          return;
      }
   
   
      this._super(attributes);

	    // paint the decorator if any exists
	    //
        if(this.targetDecorator!==null && this.targetDecoratorNode===null){
	      	this.targetDecoratorNode= this.targetDecorator.paint(this.getCanvas().paper);
	    }
	
	    if(this.sourceDecorator!==null && this.sourceDecoratorNode===null){
	      	this.sourceDecoratorNode= this.sourceDecorator.paint(this.getCanvas().paper);
	    }

	    
	    // translate/transform the decorations to the end/start of the connection 
	    // and rotate them as well
	    //
	    if(this.sourceDecoratorNode!==null){
	    	var start = this.getVertices().first();
	  	    this.sourceDecoratorNode.transform("r"+this.getStartAngle()+"," + start.x + "," + start.y +" t" + start.x + "," + start.y);
	  	    // propagate the color and the opacity to the decoration as well
	  	    this.sourceDecoratorNode.attr({"stroke":"#"+this.lineColor.hex(), opacity:this.alpha});
            this.sourceDecoratorNode.forEach($.proxy(function(shape){
                shape.node.setAttribute("class",this.cssClass!==null?this.cssClass:"");
            },this));
	    }
	    
        if(this.targetDecoratorNode!==null){
	    	var end = this.getVertices().last();
            this.targetDecoratorNode.transform("r"+this.getEndAngle()+"," + end.x + "," + end.y+" t" + end.x + "," + end.y);
            this.targetDecoratorNode.attr({"stroke":"#"+this.lineColor.hex(), opacity:this.alpha});
            this.targetDecoratorNode.forEach($.proxy(function(shape){
                shape.node.setAttribute("class",this.cssClass!==null?this.cssClass:"");
            },this));
        }

    },
    
    /**
     * @method
     * The x-offset related to the canvas.
     * Didn't provided by a connection. Return always '0'. This is required
     * for children position calculation. (e.g. Label decoration)
     * 
     * @return {Number} the x-offset to the parent figure
     **/
    getAbsoluteX :function()
    {
        return 0;
    },


    /**
     * @method
     * The y-offset related to the canvas.
     * Didn't provided by a connection. Return always '0'. This is required
     * for children position calculation. (e.g. Label decoration)
     * 
     * @return {Number} The y-offset to the parent figure.
     **/
    getAbsoluteY :function()
    {
        return 0;
    },


    postProcess: function(postProcessCache)
    {
    	this.router.postProcess(this, this.getCanvas(), postProcessCache);
    },
    
    

    /**
     * @method
     * Don't call them manually. This will be done by the framework.<br>
     * Will be called if the object are moved via drag and drop.
     * Sub classes can override this method to implement additional stuff. Don't forget to call
     * the super implementation via <code>this._super(dx, dy, dx2, dy2);</code>
     * @private
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     **/
    onDrag : function( dx, dy, dx2, dy2)
    {
        if(this.command ===null){
            return;
        }
        
        this.command.setTranslation(dx,dy);
        
        // don't drag start/end around. This Points are bounded to the related
        // ports.
        var count = this.getVertices().getSize()-1;
        for(var i=1; i<count;i++){
            this.getVertex(i).translate(dx2, dy2);
            
        }

        // notify all installed policies
        //
        this.editPolicy.each($.proxy(function(i,e){
            if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
                e.onDrag(this.canvas, this);
            }
        },this));
        
       this.svgPathString = null;
       this.repaint();
        
        // Update the resize handles if the user change the position of the
        // element via an API call.
        //
        this.editPolicy.each($.proxy(function(i, e) {
            if (e instanceof draw2d.policy.figure.DragDropEditPolicy) {
                e.moved(this.canvas, this);
            }
        }, this));

        this.fireMoveEvent();
    },

    
    /**
     * @method
     * Called by the framework during drag&drop operations.
     * 
     * @param {draw2d.Figure} draggedFigure The figure which is currently dragging
     * 
     * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didnt want a drop event
     * @template
     **/
    onDragEnter : function( draggedFigure )
    {
        if(draggedFigure instanceof draw2d.shape.basic.LineResizeHandle){
            return null;
        }
        
//    	this.setGlow(true);
    	
    	return this;
    },
 
    /**
     * @method
     * Called if the DragDrop object leaving the current hover figure.
     * 
     * @param {draw2d.Figure} draggedFigure The figure which is currently dragging
     * @template
     **/
    onDragLeave:function( draggedFigure )
    {
//    	this.setGlow(false);
    },


    /**
     * @method
     * Return the recalculated position of the start point with the usage of 
     * the installed connection anchor locator.
     * 
     * @return draw2d.geo.Point
     **/
    getStartPoint:function( refPoint)
     {
      if(this.isMoving===false){
          if(refPoint){
              return this.sourcePort.getConnectionAnchorLocation(refPoint);
          }
          return this.sourcePort.getConnectionAnchorLocation(this.targetPort.getConnectionAnchorReferencePoint());
      }

      return this._super();
     },
    
    
    /**
     * @method
     * Return the recalculated position of the start point with the usage of 
     * the installed connection anchor locator.
     *
     * @return draw2d.geo.Point
     **/
     getEndPoint:function(refPoint)
     {
      if(this.isMoving===false){
          if(refPoint){
              return this.targetPort.getConnectionAnchorLocation(refPoint);
          }
         return this.targetPort.getConnectionAnchorLocation(this.sourcePort.getConnectionAnchorReferencePoint());
      }
      
      return this._super();
     },
    
    /**
     * @method
     * Set the new source port of this connection. This enforce a repaint of the connection.
     *
     * @param {draw2d.Port} port The new source port of this connection.
     * 
     **/
    setSource:function( port)
    {
      if(this.sourcePort!==null){
        this.sourcePort.detachMoveListener(this);
        this.sourcePort.onDisconnect(this);
      }
    
      this.sourcePort = port;
      if(this.sourcePort===null){
        return;
      }
      this.routingRequired = true;
      this.fireSourcePortRouteEvent();
      this.sourcePort.attachMoveListener(this);
      if(this.canvas!==null){
          this.sourcePort.onConnect(this);
      }
      this.setStartPoint(port.getAbsoluteX(), port.getAbsoluteY());
    },
    
    /**
     * @method
     * Returns the source port of this connection.
     *
     * @type draw2d.Port
     **/
    getSource:function()
    {
      return this.sourcePort;
    },
    
    /**
     * @method
     * Set the target port of this connection. This enforce a repaint of the connection.
     * 
     * @param {draw2d.Port} port The new target port of this connection
     **/
    setTarget:function( port)
    {
      if(this.targetPort!==null){
        this.targetPort.detachMoveListener(this);
        this.targetPort.onDisconnect(this);
      }
    
      this.targetPort = port;
      if(this.targetPort===null){
        return;
      }
      
      this.routingRequired = true;
      this.fireTargetPortRouteEvent();
      this.targetPort.attachMoveListener(this);
      if(this.canvas!==null){
         this.targetPort.onConnect(this);
      }
      this.setEndPoint(port.getAbsoluteX(), port.getAbsoluteY());
    },
    
    /**
     * @method
     * Returns the target port of this connection.
     *
     * @type draw2d.Port
     **/
    getTarget:function()
    {
      return this.targetPort;
    },
    
    /**
     * @method
     * Method returns true if the connection has at least one common port with the given connection.
     * 
     * @param {draw2d.Connection} other
     * @returns {Boolean}
     */
    sharingPorts:function(other){
        return this.sourcePort== other.sourcePort ||
               this.sourcePort== other.targetPort ||
               this.targetPort== other.sourcePort ||
               this.targetPort== other.targetPort;
    },

    
    /**
     * @method
     * Set the canvas element of this figures.
     * 
     * @param {draw2d.Canvas} canvas the new parent of the figure or null
     */
    setCanvas: function( canvas )
    {
       this._super(canvas);
       

       if(this.sourceDecoratorNode!==null){
           this.sourceDecoratorNode.remove();
           this.sourceDecoratorNode=null;
       }
       
       if(this.targetDecoratorNode!==null){
           this.targetDecoratorNode.remove();
           this.targetDecoratorNode=null;
       }
       
       if(this.canvas===null){
           this.router.onUninstall(this);
           
           if(this.sourcePort!==null){
               this.sourcePort.detachMoveListener(this);
               this.sourcePort.onDisconnect(this);
           }
           if(this.targetPort!==null){
               this.targetPort.detachMoveListener(this);
               this.targetPort.onDisconnect(this);
           }
       }
       else{
           this.router.onInstall(this);
           
           if(this.sourcePort!==null){
               this.sourcePort.attachMoveListener(this);
               this.sourcePort.onConnect(this);
           }
           if(this.targetPort!==null){
               this.targetPort.attachMoveListener(this);
               this.targetPort.onConnect(this);
           }
       }

    },

    /**
     * @method
     * Fired by the given figure if the position/dimension of the figure has been changed. This
     * will be called if you have registered this event before.
     * 
     * @param {draw2d.Figure} figure The moved figure
     **/
    onOtherFigureIsMoving:function( figure)
    {
      if(figure===this.sourcePort){
        this.setStartPoint(this.sourcePort.getAbsoluteX(), this.sourcePort.getAbsoluteY());
      }
      else{
        this.setEndPoint(this.targetPort.getAbsoluteX(), this.targetPort.getAbsoluteY());
      }
      this._super(figure);
    },
    

    /**
     * Returns the angle of the connection at the output port (source)
     *
     **/
    getStartAngle:function()
    {
    	// return a good default value if the connection is not routed at the 
    	//  moment
    	if( this.lineSegments.getSize()===0){
    		return 0;
    	}
    	
      var p1 = this.lineSegments.get(0).start;
      var p2 = this.lineSegments.get(0).end;
      if(this.router instanceof draw2d.layout.connection.SplineConnectionRouter)
      {
       p2 = this.lineSegments.get(5).end;
      }
      var length = Math.sqrt((p1.x-p2.x)*(p1.x-p2.x)+(p1.y-p2.y)*(p1.y-p2.y));
      var angle = -(180/Math.PI) *Math.asin((p1.y-p2.y)/length);
    
      if(angle<0)
      {
         if(p2.x<p1.x){
           angle = Math.abs(angle) + 180;
         }
         else{
           angle = 360- Math.abs(angle);
         }
      }
      else
      {
         if(p2.x<p1.x){
           angle = 180-angle;
         }
      }
      return angle;
    },
    
    getEndAngle:function()
    {
      // return a good default value if the connection is not routed at the 
      //  moment
      if (this.lineSegments.getSize() === 0) {
        return 90;
      }
    
      var p1 = this.lineSegments.get(this.lineSegments.getSize()-1).end;
      var p2 = this.lineSegments.get(this.lineSegments.getSize()-1).start;
      if(this.router instanceof draw2d.layout.connection.SplineConnectionRouter)
      {
       p2 = this.lineSegments.get(this.lineSegments.getSize()-5).end;
      }
      var length = Math.sqrt((p1.x-p2.x)*(p1.x-p2.x)+(p1.y-p2.y)*(p1.y-p2.y));
      var angle = -(180/Math.PI) *Math.asin((p1.y-p2.y)/length);
    
      if(angle<0)
      {
         if(p2.x<p1.x){
           angle = Math.abs(angle) + 180;
         }
         else{
           angle = 360- Math.abs(angle);
         }
      }
      else
      {
         if(p2.x<p1.x){
           angle = 180-angle;
         }
      }
      return angle;
    },
    
    
    /**
     * @private
     **/
    fireSourcePortRouteEvent:function()
    {
        // enforce a repaint of all connections which are related to this port
        // this is required for a "FanConnectionRouter" or "ShortesPathConnectionRouter"
        //
       var connections = this.sourcePort.getConnections();
       for(var i=0; i<connections.getSize();i++)
       {
          connections.get(i).repaint();
       }
    },
    
    /**
     * @private
     **/
    fireTargetPortRouteEvent:function()
    {
        // enforce a repaint of all connections which are related to this port
        // this is required for a "FanConnectionRouter" or "ShortesPathConnectionRouter"
        //
       var connections = this.targetPort.getConnections();
       for(var i=0; i<connections.getSize();i++)
       {
          connections.get(i).repaint();
       }
    },
    
    
    /**
     * @method
     * Returns the Command to perform the specified Request or null.
      *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * @return {draw2d.command.Command} null or a Command
     **/
    createCommand:function( request)
    {
      if(request.getPolicy() === draw2d.command.CommandType.MOVE_BASEPOINT)
      {
        // DragDrop of a connection doesn't create a undo command at this point. This will be done in
        // the onDrop method
        return new draw2d.command.CommandReconnect(this);
      }

      return this._super(request);
    },
    
    
    /**
     * @method 
     * Return an objects with all important attributes for XML or JSON serialization
     * 
     * @returns {Object}
     */
    getPersistentAttributes : function()
    {
        var memento = this._super();

        var parentNode = this.getSource().getParent();
        while(parentNode.getParent()!==null){
        	parentNode = parentNode.getParent();
        }
        memento.source = {
                  node:parentNode.getId(),
                  port: this.getSource().getName()
                };
        
        var parentNode = this.getTarget().getParent();
        while(parentNode.getParent()!==null){
        	parentNode = parentNode.getParent();
        }
        memento.target = {
                  node:parentNode.getId(),
                  port:this.getTarget().getName()
                };
       
        return memento;
    },
    
    /**
     * @method 
     * Read all attributes from the serialized properties and transfer them into the shape.
     * 
     * @param {Object} memento
     * @returns 
     */
    setPersistentAttributes : function(memento)
    {
        this._super(memento);
        
        // nothing to to for the connection creation. This will be done in the draw2d.io.Reader 
        // implementation
        //
    }
});

/**
 * @method
 * Factory method to provide a default connection for all drag&drop connections. You
 * can override this method and customize this for your personal purpose.
 * 
 * Either you return a conection of "undefined". If "undefined" returned the "callback" must
 * be called by this method.
 * 
 * @param {draw2d.Port} sourcePort port of the source of the connection
 * @param {draw2d.Port} targetPort port of the target of the connection
 * @template
 * @returns {draw2d.Connection}
 */
draw2d.Connection.createConnection=function(sourcePort, targetPort, callback, dropTarget){
    
    return new draw2d.Connection();
};


/**
 * The default ConnectionRouter for the running applicaiton. Set this to your wanted implementation
 * {@link draw2d.layout.connection.ConnectionRouter}
 */
draw2d.Connection.DEFAULT_ROUTER= new draw2d.layout.connection.ManhattanConnectionRouter();
//draw2d.Connection.DEFAULT_ROUTER= new draw2d.layout.connection.DirectRouter();
//draw2d.Connection.DEFAULT_ROUTER= new draw2d.layout.connection.ManhattanBridgedConnectionRouter();
//draw2d.Connection.DEFAULT_ROUTER= new draw2d.layout.connection.FanConnectionRouter();
//draw2d.Connection.DEFAULT_ROUTER= new draw2d.layout.connection.SplineConnectionRouter();
        

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.VectorFigure
 * The base class for all vector based figures like {@link draw2d.shape.basic.Rectangle}  or {@link draw2d.shape.basic.Oval} 
 * inside a canvas.
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.node.Node
 */
draw2d.VectorFigure = draw2d.shape.node.Node.extend({
    NAME : "draw2d.VectorFigure",

    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * 
     */
    init : function( width, height)
    {
        this.stroke = 1;
        this.bgColor   = new  draw2d.util.Color(255,255,255);
        this.lineColor = new  draw2d.util.Color(128,128,255);
        this.color     = new  draw2d.util.Color(128,128,128);
           
        // memento for the stroke if we reset the glow effect of this shape
        //
        this.strokeBeforeGlow = this.stroke;
        this.glowIsActive = false;
        
        this._super( width, height);
    },
      
    
    /**
     * @method
     * Highlight the element or remove the highlighting
     * 
     * @param {Boolean} flag indicates glow/noGlow
     * @template
     */
    setGlow: function(flag){
        
        if(flag === this.glowIsActive) {
            return this;
        }
        
        this.glowIsActive = flag;
        if(flag===true){
            this.strokeBeforeGlow = this.getStroke();
            this.setStroke(this.strokeBeforeGlow*2.5);
        }
        else {
            this.setStroke(this.strokeBeforeGlow);
        }
        
        return this;
    },
    
   /**
    * @method
    * propagate all attributes like color, stroke,... to the shape element
    **/
    repaint : function(attributes)
    {
        if (this.repaintBlocked===true || this.shape === null){
            return;
        }

        if(typeof attributes === "undefined" ){
            attributes = {};
        }

        attributes.x = this.getAbsoluteX();
        attributes.y = this.getAbsoluteY();
        
        if(typeof attributes.stroke==="undefined"){
            if(this.color === null || this.stroke ===0){
                attributes.stroke = "none";
            }
            else {
                attributes.stroke = this.color.hash();
            }
        }
        
        if(typeof attributes["stroke-width"]==="undefined"){
            attributes["stroke-width"] = this.stroke;
        }
        
        if(typeof attributes.fill === "undefined"){
       	   attributes.fill = this.bgColor.hash();
        }

        this._super(attributes);
        
        return this;
    },


   /**
    * @method
    * Set the new background color of the figure. It is possible to hands over
    * <code>null</code> to set the background transparent.
    *
    * @param {draw2d.util.Color} color The new background color of the figure
    **/
    setBackgroundColor : function(color)
    {
        this.bgColor = new draw2d.util.Color(color);

        this.repaint();
        
        return this;
    },

   /**
    * @method
    * The current used background color.
    * 
    * @return {draw2d.util.Color}
    */
   getBackgroundColor:function()
   {
     return this.bgColor;
   },

   /**
    * @method
    * Set the stroke to use.
    * 
    * @param {Number} w The new line width of the figure
    **/
   setStroke:function( w )
   {
     this.stroke=w;
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * The used line width.
    * 
    * @type {Number}
    **/
   getStroke:function( )
   {
     return this.stroke;
   },

   /**
    * @method
    * Set the color of the line.
    * This method fires a <i>document dirty</i> event.
    * 
    * @param {draw2d.util.Color} color The new color of the line.
    **/
   setColor:function( color)
   {
     this.color = new draw2d.util.Color(color);
     this.repaint();
     
     return this;
   },

   /**
    * @method
    * The current used foreground color
    * 
    * @returns {draw2d.util.Color}
    */
   getColor:function()
   {
     return this.color;
   },
   
   
   /**
    * @method 
    * Return an objects with all important attributes for XML or JSON serialization
    * 
    * @returns {Object}
    */
   getPersistentAttributes : function()
   {
       var memento= this._super();

       memento.bgColor = this.bgColor.hash();
       memento.color   = this.color.hash();
       memento.stroke  = this.stroke;
       memento.alpha   = this.alpha;
       
       return memento;
   },
   
   /**
    * @method 
    * Read all attributes from the serialized properties and transfer them into the shape.
    * 
    * @param {Object} memento
    * @return
    */
   setPersistentAttributes : function(memento)
   {
       this._super(memento);
       
       if(typeof memento.bgColor !== "undefined"){
           this.setBackgroundColor(memento.bgColor);
       }
       
       if(typeof memento.color !== "undefined"){
           this.setColor(memento.color);
       }
       
       if(typeof memento.stroke !== "undefined"){
           this.setStroke(parseFloat(memento.stroke));
       }
        
       if(typeof memento.alpha !== "undefined"){
           this.setAlpha(parseFloat(memento.alpha));
       }
        
       return this;
   }  


});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.ResizeHandle
 * The Resizehandles for Figures.

 * <pre>
 * Possible Type:
 * 
 *     1             2               3
 *     O-----------O-------------O
 *     |                         |
 *     |                         |
 *   8 O           + 9           O 4
 *     |                         |
 *     |                         |
 *     O-----------O-------------O
 *   7             6               5
 * </pre>
 * 
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.ResizeHandle = draw2d.shape.basic.Rectangle.extend({
    NAME : "draw2d.ResizeHandle",
 
    /**
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     *
     * @param {draw2d.Canvas} canvas the related canvas element
     * @param {Number} type the type of the ResizeHandle.
     */
    init: function( figure , type) {
 
      this._super();

      // required in the SelectionEditPolicy to indicate the type of figure
      // which the user clicks
      this.isResizeHandle=true;
    
      this.owner = figure;
      this.type = type;
      this.command = null;
      this.commandMove=null;
      this.commandResize=null;
      
      this.setDimension();

      this.setBackgroundColor(new draw2d.util.Color(draw2d.Configuration.color.resizeHandle));
      this.setStroke(1);
      this.setSelectable(false);
      this.setRadius(1);
    },
    

    /**
     * @method
     *
     * @param asPrimarySelection
     * @private
     */
    select: function(asPrimarySelection){
        // it is not possible to select a resize handle. This makes no sense
        // return silently without any action
    },
    
    /**
     * @method
     * 
     * @private
     */
    unselect: function(){
        // it is not possible to unselect a resize handle. This makes no sense
        // return silently without any action
    },
    
    
    /**
     * @method
     * The edge of the rectangle for the snapTo mechanism.
     * 
     * @return
     */

    getSnapToDirection:function()
    {
      switch(this.type)
      {
        case 1:
         return draw2d.SnapToHelper.NORTH_WEST;
        case 2:
         return draw2d.SnapToHelper.NORTH;
        case 3:
         return draw2d.SnapToHelper.NORTH_EAST;
        case 4:
         return draw2d.SnapToHelper.EAST;
        case 5:
         return draw2d.SnapToHelper.SOUTH_EAST;
        case 6:
         return draw2d.SnapToHelper.SOUTH;
        case 7:
         return draw2d.SnapToHelper.SOUTH_WEST;
        case 8:
         return draw2d.SnapToHelper.WEST;
        case 9:
         return draw2d.SnapToHelper.NSEW;
        default :
         return draw2d.SnapToHelper.EAST;
      }
    },
    
    /**
     * @method
     * enrich the nomrla rectangle shape with the cursor information for resize
     * 
     * @inheritdoc
     */
    createShapeElement : function(){
       var shape= this._super();
       
       this.updateCursor(shape);
       
       return shape;
    },

    /**
     * @method
     * calculate and set the cursor of the reize handle
     * @private
     */
    updateCursor: function(shape){
        if(shape===null){
            return this;
        }
        
        if(this.isDraggable()===false){
            shape.attr({"cursor":"default"});
            return this;
        }
        
        switch(this.type)
        {
          case 1:
              shape.attr({"cursor":"nw-resize"});
              break;
          case 2:
              shape.attr({"cursor":"n-resize"});
              break;
          case 3:
              shape.attr({"cursor":"ne-resize"});
              break;
          case 4:
              shape.attr({"cursor":"e-resize"});
              break;
          case 5:
              shape.attr({"cursor":"se-resize"});
              break;
          case 6:
              shape.attr({"cursor":"s-resize"});
              break;
          case 7:
              shape.attr({"cursor":"sw-resize"});
              break;
          case 8:
              shape.attr({"cursor":"w-resize"});
              break;
          default:
              shape.attr({"cursor":"move"});
              break;
        }
        return this;
    },
    
    /**
     * @method
     * Adjust the draggable flag of the resize handle and update the cursor of the shape in relation
     * to the type of resize handle. north, south,west,..
     * 
     * @param flag
     * @returns 
     */
    setDraggable:function(flag)
    {
      this._super(flag);
      this.updateCursor(this.shape);
      return this;
    },

    /**
     * @method
     * Will be called if the drag and drop action beginns. You can return [false] if you
     * want avoid that the figure can be move.
     * 
    * @return {boolean} true whenever the drag drop operation is allowed.
     **/
    onDragStart : function()
    {
        // This happens if the selected figure has set the "nonResizeable" flag
        // In this case the ResizeHandle can't be dragged. => no resize
        //
        if (!this.isDraggable()) {
            return false;
        }

        this.ox = this.getAbsoluteX();
        this.oy = this.getAbsoluteY();

        this.commandMove = this.owner.createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE));
        this.commandResize = this.owner.createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.RESIZE));

        return true;
    },
    
    
    /**
     * @method 
     * Called by the framework if the figure is moved by user interaction.
     * 
     * @param {Number} dx the move x offset
     * @param {Number} dy the move y offset
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * 
     * @return {boolean}
     */
    onDrag : function(dx, dy, dx2, dy2)
    {
        if (this.isDraggable() === false) {
            return;
        }

        var oldX = this.getAbsoluteX();
        var oldY = this.getAbsoluteY();
        
        // call the super.drag method with all snapTo### handler and adjustments
        this._super(dx, dy, dx2, dy2);

        var diffX = this.getAbsoluteX() - oldX;
        var diffY = this.getAbsoluteY() - oldY;

        var obj = this.owner;
        var objPosX = obj.getAbsoluteX();
        var objPosY = obj.getAbsoluteY();
        var objWidth = obj.getWidth();
        var objHeight = obj.getHeight();

        switch (this.type) {
        case 1:
            obj.setDimension(objWidth - diffX, objHeight - diffY);
            obj.setPosition(objPosX + (objWidth - obj.getWidth()), objPosY + (objHeight - obj.getHeight()));
            break;
        case 2:
            obj.setDimension(objWidth, objHeight - diffY);
            obj.setPosition(objPosX, objPosY + (objHeight - obj.getHeight()));
            break;
        case 3:
            obj.setDimension(objWidth + diffX, objHeight - diffY);
            obj.setPosition(objPosX, objPosY + (objHeight - obj.getHeight()));
            break;
        case 4:
            obj.setDimension(objWidth + diffX, objHeight);
            break;
        case 5:
            obj.setDimension(objWidth + diffX, objHeight + diffY);
            break;
        case 6:
            obj.setDimension(objWidth, objHeight + diffY);
            break;
        case 7:
            obj.setDimension(objWidth - diffX, objHeight + diffY);
            obj.setPosition(objPosX + (objWidth - obj.getWidth()), objPosY);
            break;
        case 8:
            obj.setDimension(objWidth - diffX, objHeight);
            obj.setPosition(objPosX + (objWidth - obj.getWidth()), objPosY);
            break;
        }
    },

    /**
     * @method
     * Will be called after a drag and drop action.<br>
     *
     * @private
     **/
    onDragEnd : function()
    {
        if (!this.isDraggable()) {
            return;
        }

        // An non draggable resizeHandle doesn't create a move/resize command.
        // This happens if the selected figure has set the "nonResizeable" flag.
        //
        if (this.commandMove !== null) {
            this.commandMove.setPosition(this.owner.getX(), this.owner.getY());
            this.canvas.getCommandStack().execute(this.commandMove);
            this.commandMove = null;
        }

        if (this.commandResize !== null) {
            this.commandResize.setDimension(this.owner.getWidth(), this.owner.getHeight());
            this.canvas.getCommandStack().execute(this.commandResize);
            this.commandResize = null;
        }

        this.canvas.hideSnapToHelperLines();
    },

    /**
     * Set the position of the object.<br>
     * The ResizeHandle overrides the Figure.setPosition method. The base
     * class updates the resize handles during the update of the Dimension/Position. This
     * is not neccessary for the ResizeHandles. Performance issue.
     * 
     * @param {Number} x The new x coordinate of the figure
     * @param {Number} y The new y coordinate of the figure
     **/
    setPosition : function(x, y) {
        // don't call base implementation. Base implementation will show
        // ResizeHandles...but I'm the ResizeHandle
        if (x instanceof draw2d.geo.Point) {
            this.x = x.x;
            this.y = x.y;
        }
        else {
            this.x = x;
            this.y = y;
        }

        this.repaint();
    },
    
 
    /**
     * @method
     * Set the new dimension of the the ResizeHandle. IF you didn't pass any width/height the best default fo rthe
     * platform will be used.
     * 
     * @param {Number} [width] new width of the resize handle
     * @param {Number} [height] new width of the resize handle
     */
    setDimension: function(width, height)
    {
    	if(typeof height !=="undefined"){
    		this._super(width, height);
    	}
    	else{
	        if(draw2d.isTouchDevice){
	        	this._super(15,15);
	        }
	        else{
	        	this._super(8,8);
	        }
    	}
    	
        var offset= this.getWidth();
        var offset2 = offset/2;
        
        switch(this.type)
        {
          case 1:
            this.setSnapToGridAnchor(new draw2d.geo.Point(offset,offset));
            break;
          case 2:
            this.setSnapToGridAnchor(new draw2d.geo.Point(offset2,offset));
            break;
          case 3:
            this.setSnapToGridAnchor(new draw2d.geo.Point(0,offset));
            break;
          case 4:
            this.setSnapToGridAnchor(new draw2d.geo.Point(0,offset2));
            break;
          case 5:
            this.setSnapToGridAnchor(new draw2d.geo.Point(0,0));
            break;
          case 6:
            this.setSnapToGridAnchor(new draw2d.geo.Point(offset2,0));
            break;
          case 7:
            this.setSnapToGridAnchor(new draw2d.geo.Point(offset,0));
            break;
          case 8:
            this.setSnapToGridAnchor(new draw2d.geo.Point(offset,offset2));
            break;
          case 9:
            this.setSnapToGridAnchor(new draw2d.geo.Point(offset2,offset2));
            break;
        }
        
    },
    
    /**
     * @method
     * Show the ResizeHandle and add it to the canvas.<br>
     * Additional bring it in to the front of other figures.
     *
     * @param {draw2d.Canvas} canvas the canvas to use
     * @param {Number} x the x-positin
     * @param {Number} y the y-position
     **/
    show:function(canvas)
    {
      // don't call the parent function. The parent functions delete this object
      // and a resize handle can't be deleted.
      this.setCanvas(canvas);
    
      this.canvas.resizeHandles.add(this);
      this.shape.toFront();
    },
    
    /**
     * @method
     * Hide the resize handle and remove it from the cnavas.
     *
     **/
    hide:function()
    {
      // don't call the parent function. The parent functions delete this object
      // and a resize handle shouldn't be deleted.
      if(this.shape===null){
        return;
      }
        
      this.canvas.resizeHandles.remove(this);
      this.setCanvas(null);
    },
    
    /**
     * @method
     * Set the new background color of the figure. It is possible to hands over
     * <code>null</code> to set the background transparent.
     *
     * @param {draw2d.util.Color} color The new background color of the figure
     **/
     setBackgroundColor : function(color)
     {
         color = new draw2d.util.Color(color);
         
         this.bgGradient= "90-"+color.hash()+"-"+color.darker(0.2).hash();
         this._super(color);
         this.setColor(color.darker(0.3));
         
         return this;
     },
     
    /**
    * @inheritdoc
     * 
     * @param attributes
     */
    repaint:function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }

        if(typeof attributes === "undefined"){
            attributes= {};
        }
        
        if(this.bgColor.hash()==="none"){
            attributes.fill=this.bgColor.hash();
        }
        else if(this.getAlpha()<0.9){
            attributes.fill=this.bgColor.hash();
        }
        else{
            attributes.fill=this.bgGradient;
        }
        
         
        this._super(attributes);
    },
    

    /**
     * @method
     * return true if the element can be used in combination with the 
     * SnapToHelper feature.
     *
     * @return [boolean]
     **/
    supportsSnapToHelper:function()
    {
    	return true;
    },
    
    
    /**
     * @method
     * Override this method and redirect them to the cavas. A ResizeHandle didn't support
     * Keyboard interaction at the moment.
     *
     * @param {Number} keyCode the id of the pressed key
     * @param {boolean} ctrl true if the user has pressed the CTRL/STRG key as well.
     **/
    onKeyDown:function(keyCode, ctrl)
    {
      // don't call the parent function. The parent functions delete this object
      // and a resize handle can't be deleted.
      this.canvas.onKeyDown(keyCode,ctrl);
    },
    
    
    fireMoveEvent:function()
    {
      // A resizeHandle doesn't fire this event.
      // Normally this set the document dirty. This is not necessary for a ResizeHandle.
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.shape.basic.LineResizeHandle
 * Base class for selection handle for connections and normal lines.
 * 
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Circle
 */
draw2d.shape.basic.LineResizeHandle = draw2d.shape.basic.Circle.extend({
    NAME : "draw2d.shape.basic.LineResizeHandle",

    init : function(figure)
    {
        this._super();
        this.owner = figure;
        // required in the SelectionEditPolicy to indicate the type of figure
        // which the user clicks
        this.isResizeHandle=true;
         
        if (draw2d.isTouchDevice) {
            this.setDimension(20, 20);
        }
        else {
            this.setDimension(10, 10);
        }

        this.setBackgroundColor(new draw2d.util.Color(draw2d.Configuration.color.vertexHandle));
        this.setStroke(1);
        this.setSelectable(false);

        this.currentTarget = null;
    },

    createShapeElement : function(){
        var shape= this._super();
        
        shape.attr({"cursor":"move"});
        return shape;
     },
     
     /**
      * @method
      * Set the new background color of the figure. It is possible to hands over
      * <code>null</code> to set the background transparent.
      *
      * @param {draw2d.util.Color} color The new background color of the figure
      **/
      setBackgroundColor : function(color)
      {
          color = new draw2d.util.Color(color);
          
          this.bgGradient= "r(.4,.3)"+color.hash()+"-"+color.darker(0.1).hash()+":60-"+color.darker(0.2).hash();
          this._super(color);
          this.setColor(color.darker(0.3));
          
          return this;
      },
   

    /**
     * @method
     *
     * @param asPrimarySelection
     * @private
     */
    select: function(asPrimarySelection){
        // it is not possible to select a resize handle. This makes no sense
        // return silently without any action
    },
    
    /**
     * @method
     * 
     * @private
     */
    unselect: function(){
        // it is not possible to unselect a resize handle. This makes no sense
        // return silently without any action
    },
    
    /**
     * @method
     * Return the port below the ResizeHandle.
     * 
     * @template
     * @return {draw2d.Port}
     */
    getRelatedPort:function()
    {
      return null;
    },
    

    /**
     * @method
     * Return the port of the other side of the related connection.
     * 
     * @template
     * @return {draw2d.Port}
     */
    getOppositePort:function()
    {
      return null;
    },
    
    
    /**
     * @method
     * Trigger the repaint of the element
     * 
     * @template
     * @param attributes
     */
    repaint:function(attributes){
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }
        
        if(typeof attributes === "undefined"){
            attributes= {};
        }

        
        if(this.bgColor.hash()==="none"){
            attributes.fill=this.bgColor.hash();
        }
        else if(this.getAlpha()<0.9){
            attributes.fill=this.bgColor.hash();
        }
        else{
            attributes.fill=this.bgGradient;
        }
        
        
        this._super(attributes);
    },

    /**
     * Called if the drag and drop action beginns. You can return [false] if you
     * want avoid the that the figure can be move.
     *
     * @param {Number} x The x position where the mouse has been clicked in the figure
     * @param {Number} y The y position where the mouse has been clicked in the figure
     * @type {boolean}
     **/
    onDragStart : function()
    {
        this.command = this.getCanvas().getCurrentSelection().createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE_BASEPOINT));
        this.setAlpha(0.2);
        this.shape.attr({"cursor":"crosshair"});

        return true;
    },
    
     
    /**
     * @method
     * Called from the framework during a drag&drop operation
     * 
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @return {boolean}
     **/
    onDrag : function(dx, dy, dx2, dy2)
    {
        this.setPosition(this.x + dx2, this.y + dy2);

        var port = this.getOppositePort();

        var target = port!==null?port.getDropTarget(this.getX(), this.getY(), null): null;

        // the hovering element has been changed
        if (target !== this.currentTarget) {

            if (this.currentTarget !== null) {
                this.currentTarget.onDragLeave(port);
                this.currentTarget.setGlow(false);
            }

            if (target !== null) {
                this.currentTarget = target.onDragEnter(port);
                if(this.currentTarget!==null){
                    this.currentTarget.setGlow(true);
                }
            }
        }

        return true;
    },
    
    /**
     * @method Called after a drag and drop action.<br>
     *         Sub classes can override this method to implement additional stuff. Don't forget to call the super implementation via <code>this._super();</code>
     * @return {boolean}
     */
    onDragEnd : function()
    {
        if (!this.isDraggable()) {
            return false;
        }

        this.shape.attr({"cursor":"move"});
 
        var port = this.getOppositePort();
        if (port !== null) {
            if (this.currentTarget !== null) {
                
                this.onDrop(this.currentTarget);
                this.currentTarget.onDragLeave(port);
                this.currentTarget.setGlow(false);
                this.currentTarget = null;
            }
        }
        
        this.owner.isMoving=false;
        // A Connection is stuck to the corresponding ports. So we must reset the position
        // to the origin port if we doesn't drop the ResizeHandle on a other port.
        //
        if (this.getCanvas().getCurrentSelection() instanceof draw2d.Connection) {
            if (this.command !== null) {
                this.command.cancel();
            }
        }
        //
        else {
            // An non draggable resizeHandle doesn't create a move/resize command.
            // This happens if the selected figure has set "isResizeable=false".
            //
            if (this.command !== null) {
                this.getCanvas().getCommandStack().execute(this.command);
            }
        }
        this.command = null;
        this.getCanvas().hideSnapToHelperLines();

        this.setAlpha(1);

        return true;
    },


    /**
     * @method
     * Controls the location of the resize handle 
     *
     * @template
     **/
    relocate:function(){
    	
    },
    
    
    /**
     * @method
     * The LineResizeHandle didn't support the SnapToHelper feature if the
     * corresponding object is an Connection. A Connection is always bounded to
     * Port. In this case it makes no sense to use a Grid or Geometry for snapping.
     *
     * @return {boolean} return false if the corresponding object didn't support snapTo
     **/
    supportsSnapToHelper:function()
    {
      if(this.getCanvas().getCurrentSelection() instanceof draw2d.Connection){
        return false;
      }
        
      return true;
    },
    
    /**
     * @method
     * Show the ResizeHandle and add it to the canvas.<br>
     * Additional bring it in to the front of other figures.
     *
     * @param {draw2d.Canvas} canvas the canvas to use
     * @param {Number} x the x-position
     * @param {Number} y the y-position
     **/
    show:function(canvas, x, y)
    {
      // don't call the parent function. The parent functions make this object selectable/deleteable
      // and a resize handle can't be deleted.
      this.setCanvas(canvas);
      this.setPosition(x,y);
      this.shape.toFront();
      this.canvas.resizeHandles.add(this);
    },
    
    /**
     * @method
     * Hide the resize handle and remove it from the canvas.
     *
     **/
    hide:function()
    {
      // don't call the parent function. The parent functions delete this object
      // and a resize handle shouldn't be deleted.
      if(this.shape===null){
        return;
      }
      
      this.canvas.resizeHandles.remove(this);
      this.setCanvas(null);
    },
    
    /**
     * @method
     * Override this method and redirect them to the canvas. A ResizeHandle didn't support
     * Keyboard interaction at the moment.
     *
     * @param {Number} keyCode the id of the pressed key
     * @param {boolean} ctrl true if the user has pressed the CTRL/STRG key as well.
     **/
    onKeyDown:function(keyCode, ctrl)
    {
      // don't call the parent function. The parent functions delete this object
      // and a resize handle can't be deleted.
      this.canvas.onKeyDown(keyCode,ctrl);
    },
    
    
    fireMoveEvent:function()
    {
      // A resizeHandle doesn't fire this event.
      // Normally this set the document dirty. This is not necessary for a ResizeHandle.
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.shape.basic.LineStartResizeHandle
 * Selection handle for connections and normal lines.
 * 
 * TODO: Split the LineEndResizeHandle to ConnectionEndResizeHandle and LineEndResizeHandle!!!!
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.LineResizeHandle 
 */
draw2d.shape.basic.LineStartResizeHandle = draw2d.shape.basic.LineResizeHandle.extend({
    NAME : "draw2d.shape.basic.LineStartResizeHandle",

    init: function( figure) {
        this._super(figure);
    },
    
    /**
     * @method
     * Return the Port below the ResizeHandle
     * 
     * @return {draw2d.Port}
     */
    getRelatedPort:function()
    {
    	if(this.owner instanceof draw2d.Connection)
    		return this.owner.getSource();
    	
    	return null;
    },
    
    /**
     * @method
     * Return the Port on the opposite side of the ResizeHandle
     * 
     * @returns
     */
    getOppositePort:function()
    {
    	if(this.owner instanceof draw2d.Connection)
         return this.owner.getTarget();
    	
    	return null;
    },
    
    /**
     * @method
     * Called from the framework during a drag&drop operation
     * 
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @return {boolean}
     **/
    onDrag:function( dx, dy, dx2, dy2)
    {
      this._super(dx, dy, dx2, dy2);
     
      var objPos = this.owner.getStartPoint();
      objPos.translate(dx2, dy2);
      
      this.owner.setStartPoint(objPos.x, objPos.y);
      
      this.owner.isMoving = true;
      
      return true;
    },
    
    /**
     * @method
     * Resize handle has been drop on a InputPort/OutputPort.
     * 
     * @param {draw2d.Port} dropTarget
     **/
    onDrop:function( dropTarget)
    {
    	this.owner.isMoving=false;
    
      // The ResizeHandle of a Connection has been droped on a Port
      // This will enforce a ReconnectCommand
      if(this.owner instanceof draw2d.Connection && this.command !==null) {
         this.command.setNewPorts(dropTarget, this.owner.getTarget());
         this.getCanvas().getCommandStack().execute(this.command);
      }
      this.command = null;
    },
    
    /**
     * @method
     * Controls the location of the resize handle 
     *
     * @template
     **/
    relocate:function(){

        var resizeWidthHalf = this.getWidth()/2;
        var resizeHeightHalf= this.getHeight()/2;
        
        var anchor   = this.owner.getStartPoint();
        
        this.setPosition(anchor.x-resizeWidthHalf,anchor.y-resizeHeightHalf);
    }    
  
    

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.LineEndResizeHandle
 * Selection handle for connections and normal lines.
 * 
 * TODO: Split the LineEndResizeHandle to ConnectionEndResizeHandle and LineEndResizeHandle!!!!
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.LineResizeHandle 
 */
draw2d.shape.basic.LineEndResizeHandle = draw2d.shape.basic.LineResizeHandle.extend({
    NAME : "draw2d.shape.basic.LineEndResizeHandle",

    init: function( figure) {
        this._super(figure);
    },

    
    /**
     * @method
     * Return the Port below the ResizeHandle
     * 
     * @return {draw2d.Port}
     */
    getRelatedPort:function()
    {
    	if(this.owner instanceof draw2d.Connection){
         return this.owner.getTarget();
    	}
    	
    	return null;
    },
    
    /**
     * @method
     * Return the Port on the opposite side of the ResizeHandle
     * 
     * @returns
     */
    getOppositePort:function()
    {
    	if(this.owner instanceof draw2d.Connection) {
         return this.owner.getSource();
    	}
    	
    	return null;
    },
    
 
    /**
     * @method
     * Called from the framework during a drag&drop operation
     * 
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @return {boolean}
     **/
    onDrag:function( dx, dy, dx2, dy2)
    {
      this._super(dx,dy, dx2, dy2);
    
      var objPos = this.owner.end.clone();//getEndPoint();
     // objPos.translate(dx2,dy2);
      
      this.owner.setEndPoint(objPos.x+dx2, objPos.y+dy2);
      
      this.owner.isMoving = true;
    
      return true;
    },
    
    /**
     * Resizehandle has been drop on a InputPort/OutputPort.
     * @private
     **/
    onDrop:function( dropTarget)
    {
    	this.owner.isMoving=false;
      
      if(this.owner instanceof draw2d.Connection && this.command !==null){
         this.command.setNewPorts(this.owner.getSource(),dropTarget);
         this.getCanvas().getCommandStack().execute(this.command);
      }
      this.command = null;
    },
    
    /**
     * @method
     * Controls the location of the resize handle 
     *
     * @template
     **/
    relocate:function(){

        var resizeWidthHalf = this.getWidth()/2;
        var resizeHeightHalf= this.getHeight()/2;
        
        var anchor   = this.owner.getEndPoint();
        
        this.setPosition(anchor.x-resizeWidthHalf,anchor.y-resizeHeightHalf);
    }    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.VertexResizeHandle
 * 
 * Selection handle for polyline vertices.
 * 
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.ResizeHandle 
 */
draw2d.shape.basic.VertexResizeHandle = draw2d.ResizeHandle.extend({
    NAME : "draw2d.shape.basic.VertexResizeHandle",

    SNAP_THRESHOLD   : 3,
    LINE_COLOR       : "#1387E6",
    FADEOUT_DURATION : 300,

    init: function( figure, index) {
        this._super(figure);
        this.index = index;
        this.isDead = false;
        
        this.vline = null;
        this.hline = null;
    },
   
    
    /**
     * @method
     * Called when a user double clicks on the element
     * 
     * @template
     */
    onDoubleClick: function(){
       	var cmd  = new draw2d.command.CommandRemoveVertex(this.owner, this.index );
        this.getCanvas().getCommandStack().execute(cmd);

        this.isDead = true;
    },
    
    
    /**
     * @method
     * Will be called after a drag and drop action.<br>
     *
     * @private
     **/
    onDragStart : function()
    {
    	if(this.isDead===true){
    		return;
    	}
    	
        this._super();
        this.command = this.getCanvas().getCurrentSelection().createCommand(new draw2d.command.CommandType(draw2d.command.CommandType.MOVE_VERTEX));
        if(this.command!=null){
            this.command.setIndex(this.index);
            this.setAlpha(0.2);
            this.shape.attr({"cursor":"crosshair"});
        }
        
        // Vertex is a reference and not a copy of the point
        this.vertex = this.owner.getVertices().get(this.index).clone();
    },
    
    /**
     * @method
     * Called from the framework during a drag&drop operation
     * 
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @return {boolean}
     **/
    onDrag : function(dx, dy, dx2, dy2) 
    {
        if (this.isDead===true || this.command == null) {
            return false;
        }

        this.setPosition(this.x + dx2, this.y + dy2);

        // update the polyline for immediately  drag&drop feedback
        //
        this.vertex.translate(dx2, dy2);
        this.owner.setVertex(this.index, this.vertex.x, this.vertex.y);
        
        // update the command for the undo/redo stuff
        //
        this.command.updatePosition(this.vertex.x, this.vertex.y);
        
        // show snapTo lines
        //
        var points = this.owner.getVertices();
        var size = points.getSize();
        var left   = points.get((this.index-1 +size)%size); // % is just to ensure the [0, size] interval
        var right  = points.get((this.index+1)%size);       // % is just to ensure the [0, size] interval
        
        // horizontal guided line
        //
        if(Math.abs(left.x - this.vertex.x)<this.SNAP_THRESHOLD){
        	this.showVerticalLine(left.x);
        }
        else if( Math.abs(right.x - this.vertex.x)<this.SNAP_THRESHOLD){
        	this.showVerticalLine(right.x);
        }
        else{
        	this.hideVerticalLine();
        }
        
        // horizontal guided line
        //
        if(Math.abs(left.y - this.vertex.y)<this.SNAP_THRESHOLD){
        	this.showHorizontalLine(left.y);
        }
        else if( Math.abs(right.y - this.vertex.y)<this.SNAP_THRESHOLD){
        	this.showHorizontalLine(right.y);
        }
        else{
        	this.hideHorizontalLine();
        }

        
        return true;
    },
    
    /**
     * @method Called after a drag and drop action.<br>
     *         Sub classes can override this method to implement additional stuff. Don't forget to call the super implementation via <code>this._super();</code>
     * @return {boolean}
     */
    onDragEnd : function()
    {
        if (this.isDead===true || this.command===null) {
            return false;
        }

        this.shape.attr({"cursor":"move"});
        this.hideVerticalLine();
        this.hideHorizontalLine();
        
        // snapTo
        var points = this.owner.getVertices();
        var size = points.getSize();
        var left   = points.get((this.index-1 +size)%size); // % is just to ensure the [0, size] interval
        var right  = points.get((this.index+1)%size);       // % is just to ensure the [0, size] interval
        
        // Vertical snapTo
        //
        if(Math.abs(left.x - this.vertex.x)<this.SNAP_THRESHOLD){
            this.command.updatePosition(left.x, this.vertex.y);
        }
        else if( Math.abs(right.x - this.vertex.x)<this.SNAP_THRESHOLD){
            this.command.updatePosition(right.x, this.vertex.y);
        }

        // horizontal snapTo
        //
        if(Math.abs(left.y - this.vertex.y)<this.SNAP_THRESHOLD){
            this.command.updatePosition(this.vertex.x, left.y);
        }
        else if( Math.abs(right.y - this.vertex.y)<this.SNAP_THRESHOLD){
            this.command.updatePosition(this.vertex.x, right.y);
        }
        
        var stack = this.getCanvas().getCommandStack();
        
        stack.startTransaction();
        try{
	        stack.execute(this.command);
	        this.command = null;
	        this.getCanvas().hideSnapToHelperLines();
	
	        var angle = this.getEnclosingAngle();
	        if(angle>178){
	           	var cmd  = new draw2d.command.CommandRemoveVertex(this.owner, this.index );
	            stack.execute(cmd);
	        }
        }
        finally{
        	stack.commitTransaction();
        }
        
        this.setAlpha(1);

        return true;
    },
    
    
    /**
     * @method
     * Controls the location of the resize handle 
     *
     * @template
     **/
    relocate:function(){

        var resizeWidthHalf = this.getWidth()/2;
        var resizeHeightHalf= this.getHeight()/2;

        var anchor = this.owner.getVertices().get(this.index);
    		
  		this.setPosition(anchor.x-resizeWidthHalf,anchor.y-resizeHeightHalf);
    },
    
    /**
     * @method
     * Calculates the angle between the siblings 
     * 
     * @returns {Number}
     */
    getEnclosingAngle:function(){
        // calculate the angle between the siblings
        var points = this.owner.getVertices();
        var trans  = this.vertex.getScaled(-1);
        var size = points.getSize();
        var left   = points.get((this.index-1 +size)%size).getTranslated(trans); // % is just to ensure the [0, size] interval
        var right  = points.get((this.index+1)%size).getTranslated(trans);       // % is just to ensure the [0, size] interval
        
        var dot = left.dot(right);
        
        var acos = Math.acos(dot/(left.length() * right.length()));
        return acos*180/Math.PI;
    },
    
    showVerticalLine:function(x){
        if(this.vline!=null){
            return; //silently
        }
        this.vline = this.canvas.paper
                        .path("M " + x + " 0 l 0 " + this.canvas.getHeight())
                        .attr({"stroke":this.LINE_COLOR,"stroke-width":1});
    },
    
    hideVerticalLine:function(){
        if(this.vline==null){
            return;
        }
        var tmp = this.vline;
        tmp.animate({
            opacity: 0.1
        }, this.FADEOUT_DURATION,function(){
            tmp.remove();
        });
        
        this.vline = null;
    },

    
    showHorizontalLine:function(y){
        if(this.hline!=null){
            return;
        }
        
        this.hline = this.canvas.paper
                      .path("M 0 " + y + " l " + this.canvas.getWidth() + " 0")
                      .attr({"stroke":this.LINE_COLOR,"stroke-width":1});
    },

    hideHorizontalLine:function(){
        if(this.hline==null){
            return; //silently
        }
        var tmp = this.hline;
        tmp.animate({
            opacity: 0.1
        }, this.FADEOUT_DURATION,function(){
            tmp.remove();
        });
        this.hline = null;
    }

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.basic.GhostVertexResizeHandle
 * ResizeHandle for a vertex edit policy. Click of this kind of resize handles
 * adds a new vertex to the polyline or polygon.
 * 
 *
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.basic.LineResizeHandle 
 */
draw2d.shape.basic.GhostVertexResizeHandle = draw2d.shape.basic.LineResizeHandle.extend({
    NAME : "draw2d.shape.basic.GhostVertexResizeHandle",

    init: function( figure, precursorIndex) {
        this.maxOpacity = 0.35;
        
        this._super(figure);
        this.precursorIndex = precursorIndex;
        
        this.setAlpha(this.maxOpacity);
    },
   
    createShapeElement : function(){
        var shape= this._super();
        
        shape.attr({"cursor":"pointer"});
        
        return shape;
     },

     /**
      * @method
      * Set the alpha blending of this figure. 
      *
      * @param {Number} percent Value between [0..1].
      * @template
      **/
     setAlpha:function( percent)
     {
       percent = Math.min(this.maxOpacity,Math.max(0,parseFloat(percent)));
       this._super(percent);
       
       return this;
     },

    /**
     * @method
     * Called when a user clicks on the element
     * 
     * @template
     */
    onClick: function(){
    	var cmd  = new draw2d.command.CommandAddVertex(this.owner, this.precursorIndex+1,this.getAbsoluteX() + this.getWidth()/2, this.getAbsoluteY() + this.getHeight()/2 );
        this.getCanvas().getCommandStack().execute(cmd);
    },
    

    /**
     * @method
     * Called from the framework during a drag&drop operation
     * 
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     * @return {boolean}
     **/
    onDrag : function(dx, dy, dx2, dy2) 
    {
        return true;
    },
    
    /**
     * @method Called after a drag and drop action.<br>
     *         Sub classes can override this method to implement additional stuff. Don't forget to call the super implementation via <code>this._super();</code>
     * @return {boolean}
     */
    onDragEnd : function()
    {
        return true;
    },
    
    
    /**
     * @method
     * Controls the location of the resize handle 
     *
     * @template
     **/
    relocate:function(){
        var p1 = this.owner.getVertices().get(this.precursorIndex);
        var p2 = this.owner.getVertices().get(this.precursorIndex+1);
     
        var x = ((p2.x - p1.x) / 2 + p1.x - this.getWidth()/2)|0;
        var y = ((p2.y - p1.y) / 2 + p1.y - this.getHeight()/2)|0;

   		
  		this.setPosition(x,y);
    }    


});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.Port
 * A port is an object that is used to establish a connection between a node and a {@link draw2d.Connection}. The port can 
 * be placed anywhere within a node ( see {@link draw2d.layout.locator.PortLocator} for details)
 * 
 * 
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Circle
 */ 
draw2d.Port = draw2d.shape.basic.Circle.extend({
    NAME : "draw2d.Port",

    DEFAULT_BORDER_COLOR:new draw2d.util.Color("#1B1B1B"),
    
    /**
     * @constructor
     * Creates a new Node element which are not assigned to any canvas.
     * 
     * @param {String} [name] the name of the port.
     */
    init : function( name)
    {
        this.locator = null;
        this.lighterBgColor =null;
        
        this._super();
        
        if (draw2d.isTouchDevice) {
            this.setDimension(25, 25);
        }
        else {
            this.setDimension(10, 10);
        }

        // status var for user interaction
        //
        this.ox = this.x;
        this.oy = this.y;
        this.coronaWidth = 5; // the corona width for the hitTest method. Usefull during drag&drop of ports. Better SnapTo behaviour.
        this.corona = null; // Circle
        
        // currentTarget can be differ from the currentTargetPort. In this case
        // we must store booth of them for notifications hoverEnter/hoverLeft
        this.currentTargetPort = null; // port
        this.currentTarget = null; // Figure

        // visible representation
        //
        this.setBackgroundColor("#4f6870");
        this.setStroke(1);
        this.setColor(this.DEFAULT_BORDER_COLOR);
        this.setSelectable(false);
    
        // avoid "undefined" values. This breaks the code on iOS.
        if(typeof name ==="undefined"){
            this.name = null;
        }
        else{
            this.name = name;
        }
        
        
        this.connectionAnchor = new draw2d.layout.anchor.ConnectionAnchor(this);

        // for dynamic diagrams. A Port can have a value which is set by a connector
        //
        this.value = null; 
        this.maxFanOut = Number.MAX_VALUE;
        
        this.setCanSnapToHelper(false);
        
        this.installEditPolicy(new draw2d.policy.port.IntrusivePortsFeedbackPolicy());
    //    this.installEditPolicy(new draw2d.policy.port.ElasticStrapFeedbackPolicy());
    },

    /**
     * @method
     * set the maximal possible count of connections for this port
     * 
     * @param {Number} count
     */
    setMaxFanOut: function(count)
    {
        this.maxFanOut = Math.max(1,count);
        
        return this;
    },
    
    /**
     * @method
     * return the maximal possible connections (in+out) for this port.
     * 
     * @returns
     */
    getMaxFanOut: function()
    {
        return this.maxFanOut;
    },
    
    /**
     * @method
     * Set the Anchor for this object. An anchor is responsible for the endpoint calculation
     * of an connection. just visible representation.
     *
     * @param {draw2d.layout.anchor.ConnectionAnchor} [anchor] the new source anchor for the connection
     **/
    setConnectionAnchor:function( anchor)
    {
        // set some good defaults.
        if(typeof anchor ==="undefined" || anchor===null)
        {
    		anchor = new draw2d.layout.anchor.ConnectionAnchor( );
    	}
    	
        this.connectionAnchor = anchor;
        this.connectionAnchor.setOwner(this);

        return this;
    },
 
    getConnectionAnchorLocation:function(referencePoint)
    {
    	return this.connectionAnchor.getLocation(referencePoint);
    },
    
    getConnectionAnchorReferencePoint:function()
    {
    	return this.connectionAnchor.getReferencePoint();
    },
 
    
    /**
     * @method
     * Returns the **direction** for the connection in relation to the given port and it's parent.
     * 
     * <p>
     * Possible values:
     * <ul>
     *   <li>up -&gt; 0</li>
     *   <li>right -&gt; 1</li>
     *   <li>down -&gt; 2</li>
     *   <li>left -&gt; 3</li>
     * </ul>
     * <p>
     * 
     * @param {draw2d.Connection} conn the related Connection
     * @param {draw2d.Port} relatedPort the counterpart port
     * @return {Number} the direction.
     */
    getConnectionDirection:function(conn, relatedPort)
    {
       return this.getParent().getBoundingBox().getDirection(this.getAbsolutePosition());
    },
    
    /**
     * @method
     * Set the locator/layouter of the port. A locator is responsive for the x/y arrangement of the 
     * port in relation to the parent node.
     * 
     * @param {draw2d.layout.locator.Locator} locator
     */
    setLocator: function(locator)
    {
        this.locator = locator;

        return this;
    },
    
    /**
     * @method
     * Get the locator/layouter of the port. A locator is responsive for the x/y arrangement of the 
     * port in relation to the parent node.
     * 
     * @since 4.2.0
     */
    getLocator: function()
    {
        return this.locator;
    },
    

    /**
     * @method
     * Set the new background color of the figure. It is possible to hands over
     * <code>null</code> to set the background transparent.
     *
     * @param {draw2d.util.Color} color The new background color of the figure
     **/
     setBackgroundColor : function(color)
     {
        // delete cached colors. recalculated in the repaint method 
        this._super(color);
        this.lighterBgColor=this.bgColor.lighter(0.3).hash();;

        return this;
     },

    /**
     * @method
     * Set a value for the port. This is useful for interactive/dynamic diagrams like circuits, simulator,...
     *  
     * @param {Object} value the new value for the port 
     */
    setValue:function(value)
    {
        this.value = value;
        if(this.getParent()!==null){
           this.getParent().onPortValueChanged(this);
        }

        return this;
    },
    
    /**
     * @method
     * Return the user defined value of the port.
     * 
     * @returns {Object}
     */
    getValue:function()
    {
        return this.value;
    },
    
     /**
      * @inheritdoc
      * 
      * @param attributes
      */
     repaint:function(attributes)
     {
         if(this.repaintBlocked===true || this.shape===null){
             return;
         }

         if(typeof attributes === "undefined"){
             attributes= {};
         }
         
         // a port did have the 0/0 coordinate i the center and not in the top/left corner
         //
         attributes.cx = this.getAbsoluteX();
         attributes.cy = this.getAbsoluteY();
         attributes.rx = this.width/2;
         attributes.ry = attributes.rx;
         attributes.cursor = "move";
         
         if(this.getAlpha()<0.9){
             attributes.fill=this.bgColor.hash();
         }
         else{
             attributes.fill = ["90",this.bgColor.hash(),this.lighterBgColor].join("-");
         }
         
         this._super(attributes);
     },
     
    
    /**
     * @inheritdoc
     *
     **/
    onMouseEnter:function()
    {
        this.setStroke(2);
    },
    
    
    /**
     * @inheritdoc
     * 
     **/
    onMouseLeave:function()
    {
        this.setStroke(1);
    },


    /**
     * @method
     * Returns a {@link draw2d.util.ArrayList} of {@link draw2d.Connection}s of all related connections to this port.
     *
     * @type {draw2d.util.ArrayList}
     **/
    getConnections:function()
    {
      var result = new draw2d.util.ArrayList();
    
      // Return all Connections which are bounded to this port
      // In this case this are all movement listener
      var size= this.moveListener.getSize();
      for(var i=0;i<size;i++)
      {
        var target = this.moveListener.get(i);
        if(target instanceof draw2d.Connection){
           result.add(target);
        }
      }
      return result;
    },
    
    
    /**
     * @method
     * Set the parent of this port.
     * Call {@link draw2d.shape.node.Node#addPort} if you want to a port to node. Don't call this method directly.
     *
     * @private
     */
    setParent:function(parent)
    {
      this._super(parent);
      
      if(this.parent!==null){
        this.parent.detachMoveListener(this);
      }
      
      if(this.parent!==null) {
        this.parent.attachMoveListener(this);
      }
    },
    

    /**
     * @method
     * Returns the corona width of the Port. The corona width will be used during the
     * drag&drop of a port.
     *
     * @return {Number}
     **/
    getCoronaWidth:function()
    {
       return this.coronaWidth;
    },
    
    
    /**
     * @method
     * Set the corona width of the Port. The corona width will be used during the
     * drag&drop of a port. You can drop a port in the corona of this port to create
     * a connection. It is not neccessary to drop exactly on the port.
     *
     * @param {Number} width The new corona width of the port
     **/
    setCoronaWidth:function( width)
    {
       this.coronaWidth = width;
    },
    
    /**
     * @inheritdoc
     * 
     * @return {boolean}
     **/
    onDragStart : function()
    {
        // just allow the DragOperation if the port didn't have reached the max fanOut
        // limit.
        if(this.getConnections().getSize() >= this.maxFanOut){
            return false;
        }
        
        // this can happen if the user release the mouse button outside the window during a drag&drop
        // operation
        if(this.isInDragDrop ===true){
            this.onDragEnd();
        }
                
        this.getShapeElement().toFront();
       // don't call the super method. This creates a command and this is not necessary for a port
       this.ox = this.x;
       this.oy = this.y;

        // notify all installed policies
        //
        this.editPolicy.each($.proxy(function(i,e){
            if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
                e.onDragStart(this.canvas, this);
            }
        },this));

        return true;
    },
    
    /**
     * @inheritdoc
     * 
     * @param {Number} dx the x difference between the start of the drag drop operation and now
     * @param {Number} dy the y difference between the start of the drag drop operation and now
     **/
    onDrag:function(dx, dy)
    {
      this.isInDragDrop = true;

      this._super( dx, dy);

      var target=this.getDropTarget(this.getAbsoluteX(),this.getAbsoluteY(), this);
      // the hovering element has been changed
      if(target!==this.currentTarget){
          if(this.currentTarget!==null){
              this.currentTarget.onDragLeave(this);
              this.editPolicy.each($.proxy(function(i,e){
                  if(e instanceof draw2d.policy.port.PortFeedbackPolicy){
                      e.onHoverLeave(this.canvas, this, this.currentTarget);
                  }
              },this));
          }
          
          // possible hoverEnter event
          //
          if(target!==null){
              this.currentTarget= target.onDragEnter(this);
              if(this.currentTarget!==null){
            	  this.currentTargetPort = target;
                  this.editPolicy.each($.proxy(function(i,e){
                      if(e instanceof draw2d.policy.port.PortFeedbackPolicy){
                          e.onHoverEnter(this.canvas, this, this.currentTarget);
                      }
                  },this));
              }
          }
          else{
        	  this.currentTarget = null;
          }
          
      }
    },
    
    
    /**
     * @inheritdoc
     **/
    onDragEnd:function()
    {
      // Don't call the parent implementation. This will create an CommandMove object
      // and store them o the CommandStack for the undo operation. This makes no sense for a
      // port.
      // draw2d.shape.basic.Rectangle.prototype.onDragEnd.call(this); DON'T call the super implementation!!!
    
      this.setAlpha(1.0);
    
      // 1.) Restore the old Position of the node
      //
      this.setPosition(this.ox,this.oy);
    
      this.isInDragDrop =false;
      
      
      // notify all installed policies
      //
      if(this.currentTarget){
	      this.editPolicy.each($.proxy(function(i,e){
	          if(e instanceof draw2d.policy.port.PortFeedbackPolicy){
	              e.onHoverLeave(this.canvas, this, this.currentTarget);
	          }
	      },this));
      }
      
      this.editPolicy.each($.proxy(function(i,e){
          if(e instanceof draw2d.policy.figure.DragDropEditPolicy){
              e.onDragEnd(this.canvas, this);
          }
      },this));

      // Reset the drag&drop flyover information 
      //
      this.currentTarget = null;
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Figure} figure The figure which is currently dragging
     * 
     * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didn't want a drop event
     * @private
     **/
    onDragEnter : function( draggedFigure )
    {
    	// Ports accepts only Ports as DropTarget
    	//
    	if(!(draggedFigure instanceof draw2d.Port)){
    		return null;
    	}
    	
    	// consider the max possible connections for this port
    	//
    	if(this.getConnections().getSize() >= this.maxFanOut){
    	    return null;
    	}
        // Create a CONNECT Command to determine if we can show a Corona. Only valid
        // dropTarget did have a corona
        var request = new draw2d.command.CommandType(draw2d.command.CommandType.CONNECT);
        request.canvas = this.parent.getCanvas();
        request.source = this;
        request.target = draggedFigure;
        var command = draggedFigure.createCommand(request);

        if (command === null) {
            return null;
        }

        return this;
    },
    
    /**
     * @method
     * 
     * @param {draw2d.Figure} figure The figure which is currently dragging
     * @private
     **/
    onDragLeave:function( figure )
    {
		// Ports accepts only Ports as DropTarget
		//
		if(!(figure instanceof draw2d.Port)){
			return;
		}
    },
    
    /**
     * @method
     * Called if the user drop this element onto the dropTarget
     * 
     * @param {draw2d.Figure} dropTarget The drop target.
     * @private
     **/
    onDrop:function(dropTarget)
    {
    	// Ports accepts only Ports as DropTarget
    	//
    	if(!(dropTarget instanceof draw2d.Port)){
    		return;
    	}
 
    	var request = new draw2d.command.CommandType(draw2d.command.CommandType.CONNECT);
        request.canvas = this.parent.getCanvas();
        request.source = dropTarget;
        request.target = this;
        var command = this.createCommand(request);
        
        if(command!==null){
           this.parent.getCanvas().getCommandStack().execute(command);
        }
        this.setGlow(false);
    },
   

    /**
     * @method
     * Callback method if a new connection has created with this port
     * 
     * @param {draw2d.Connection} connection The connection which has been created
     * @since 2.5.1
     * @template
     **/
    onConnect: function(connection){
    },
    
    /**
     * @method
     * Callback method if a new connection has created with this port
     * 
     * @param {draw2d.Connection} connection The connection which has been deleted
     * @since 2.5.1
     * @template
     **/
    onDisconnect: function(connection){
    },
    
    /**
     * @method
     * Callback method of the movement of a figure
     * 
     * @param {draw2d.Figure} figure The figure which has been moved
     **/
    onOtherFigureIsMoving:function( figure)
    {
      this.repaint();
      
      // Falls sich der parent bewegt hat, dann muss der Port dies seinen
      // Connections mitteilen
      this.fireMoveEvent();
    },
    
    /**
     * @method
     * Return the name of this port.
     *
     * @return {String}
     **/
    getName:function()
    {
      return this.name;
    },
    
    /**
     * @method
     * Set the name of this port.
     *
     * @see draw2d.shape.node.Node#getPort
     * @param {String} name The new name of this port.
     **/
    setName:function( name)
    {
      this.name=name;
    },
    
    /**
     * @method
     * Hit test for ports. This method respect the corona diameter of the port for the hit test.
     * The corona width can be set with {@link draw2d.Port#setCoronaWidth}
     * @param {Number} iX
     * @param {Number} iY
     * @returns {Boolean}
     */
    hitTest:function ( iX , iY)
    {
        var x   = this.getAbsoluteX()-this.coronaWidth-this.getWidth()/2;
        var y   = this.getAbsoluteY()-this.coronaWidth-this.getHeight()/2;
        var iX2 = x + this.getWidth()  + (this.coronaWidth*2);
        var iY2 = y + this.getHeight() + (this.coronaWidth*2);

        return (iX >= x && iX <= iX2 && iY >= y && iY <= iY2);
    },
    
    /**
     * @method
     * Highlight this port
     * 
     * @param {boolean} flag indicator if the figure should glow.
     */
    setGlow:function ( flag)
    {
      if(flag===true && this.corona===null)
      {
    	  this.corona = new draw2d.Corona();
    	  this.corona.setDimension(this.getWidth()+(this.getCoronaWidth()*2),this.getWidth()+(this.getCoronaWidth()*2));
          this.corona.setPosition(this.getAbsoluteX()-this.getCoronaWidth()-this.getWidth()/2, this.getAbsoluteY()-this.getCoronaWidth()-this.getHeight()/2);
          
          this.corona.setCanvas(this.getCanvas());

          // important inital 
          this.corona.getShapeElement();
          this.corona.repaint();
          
          // DON'T add them to the document. The corona is just a visual feedback and not part
          // of the canvas document.
         // this.parent.getCanvas().addFigure(this.corona,this.getAbsoluteX()-this.getCoronaWidth()-this.getWidth()/2, this.getAbsoluteY()-this.getCoronaWidth()-this.getHeight()/2);
      }
      else if(flag===false && this.corona!==null)
      {
          this.corona.setCanvas(null);
    	  this.parent.getCanvas().removeFigure(this.corona);
    	  this.corona = null;
      }
      
      return this;
    },
    
    /**
     * @inheritdoc
     *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * @return {draw2d.command.Command} null or a valid command
     **/
    createCommand:function(request)
    {
       // the port has its own implementation of the CommandMove
       //
       if(request.getPolicy() === draw2d.command.CommandType.MOVE)
       {
         if(!this.isDraggable()){
            return null;
         }
         return new draw2d.command.CommandMovePort(this);
       }
       
       // Connect request between two ports
       //
       if(request.getPolicy() === draw2d.command.CommandType.CONNECT)
       {
         if(request.source.parent.id === request.target.parent.id){
            return null;
         }

         return new draw2d.command.CommandConnect(request.canvas,request.source,request.target, request.source);
       }
    
       return null;
    },
    
    /**
     * @method
     * Called from the figure itself when any positin changes happens. All listenter
     * will be informed.
     * DON'T fire this event if the Port is during a Drag&Drop operation. This can happen
     * if we try to connect two ports
     * 
     * @private
     **/
    fireMoveEvent : function()
    {
        if (this.isInDragDrop === true) {
            return;
        }

        this._super();
    },
 
    /**
     * @method 
     * Return a possible drop target which is under the hands over coordinate. This can be a 
     * 
     * @param {Number} x
     * @param {Number} y
     * @private
     * @return {draw2d.Figure}
     */
    getDropTarget: function (x , y, portToIgnore)
    {
      for(var i=0;i<this.targets.getSize();i++)
      {
        var target = this.targets.get(i);
        if (target!==portToIgnore)
        {
	        if (target.hitTest(x, y)===true)
	        {
	            return target;
	        }
        }
      }
      
      return null;
    },
    
    /**
     * @method 
     * Return a possible drop target which is under the hands over coordinate. This can be a 
     * 
     * @param {Number} x
     * @param {Number} y
     * @private
     * @return {draw2d.Figure}
     */
    getDropTargets: function ()
    {
      return this.targets.clone().grep($.proxy(function(element){
	    	  return element!==this;
	      },this));
    }
});


/**
 * @class draw2d.Corona
 * Glow effect for ports. Just for internal use.
 * 
 * @extend draw2d.shape.basic.Circle
 */
draw2d.Corona = draw2d.shape.basic.Circle.extend({

    /**
     * @constructor
     * Creates a new Node element which are not assigned to any canvas.
     * 
     */
    init : function()
    {
        this._super();
        this.setAlpha(0.3);
        this.setBackgroundColor(new  draw2d.util.Color(178,225,255));
        this.setColor(new draw2d.util.Color(102,182,252));
    },
    
    /**
     * @method
     * the the opacity of the element.
     * 
     * @param {Number} percent
     */
    setAlpha : function(percent)
    {
        this._super(Math.min(0.3, percent));
        this.setDeleteable(false);
        this.setDraggable(false);
        this.setResizeable(false);
        this.setSelectable(false);
        
        return this;
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.InputPort
 * A InputPort is the start anchor for a {@link draw2d.Connection}.
 * 
 * @author Andreas Herz
 * @extend draw2d.Port
 */ 
draw2d.InputPort = draw2d.Port.extend({

    NAME : "draw2d.InputPort",

    /**
     * @constructor
     * Create a new InputPort element
     * 
     * @param {String} [name] the name for the Port.
     */
    init : function( name)
    {
        this._super( name);
        
        // responsive for the arrangement of the port 
        // calculates the x/y coordinates in relation to the parent node
        this.locator=new draw2d.layout.locator.InputPortLocator();
    },

    
    /**
     * @inheritdoc
     **/
    onDragEnter : function(figure)
    {
        // User drag&drop a normal port
        if (figure instanceof draw2d.OutputPort) {
            return this._super(figure);
        }
        // User drag&drop a normal port
        if (figure instanceof draw2d.HybridPort) {
            return this._super(figure);
        }
        
        return null;
    },
    
    /**
     * @inheritdoc
     * 
     * @param {draw2d.Figure} figure
     */
    onDragLeave:function( figure)
    {
      if(figure instanceof draw2d.OutputPort){
        this._super( figure);
      }
      
      else if(figure instanceof draw2d.HybridPort){
          this._super( figure);
      }
    },
    
    
    /**
     * @method
     * Returns the Command to perform the specified Request or null.<br>
     * Inherited figures can override this method to return the own implementation
     * of the request.<br>
     *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * @return {draw2d.command.Command} null or a valid command
     **/
    createCommand:function( request)
    {
       // Connect request between two ports
       //
       if(request.getPolicy() ===draw2d.command.CommandType.CONNECT)
       {
         // loopback not supported at the moment
         if(request.source.getParent().getId() === request.target.getParent().getId()){
            return null;
         }
    
         // InputPort can only connect to an OutputPort. InputPort->InputPort make no sense
         if(request.source instanceof draw2d.OutputPort){
            // This is the different to the OutputPort implementation of createCommand
            return new draw2d.command.CommandConnect(request.canvas,request.source,request.target, request.source);
         }
         
         if(request.source instanceof draw2d.HybridPort){
             // This is the different to the OutputPort implementation of createCommand
             return new draw2d.command.CommandConnect(request.canvas,request.source,request.target, request.source);
         }
      
         return null;
       }
    
       // ...else call the base class
       return this._super(request);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.OutputPort
 * A OutputPort is the start anchor for a {@link draw2d.Connection}.
 * 
 * @author Andreas Herz
 * @extends draw2d.Port
 */ 
draw2d.OutputPort = draw2d.Port.extend({

    NAME : "draw2d.OutputPort",

    /**
     * @constructor
     * Create a new OutputPort element
     * 
     * @param {String} [name] the name for the Port. 
     */
    init : function(name)
    {
        this._super(name);
       
        // responsive for the arrangement of the port 
        // calculates the x/y coordinates in relation to the parent node
        this.locator=new draw2d.layout.locator.OutputPortLocator();
    },

    
    /**
     * @inheritdoc
     * 
     * @param {draw2d.Figure} figure The figure which is currently dragging
     * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didnt want a drop event
     */
    onDragEnter : function(figure)
    {
    	// Ports accepts only InputPorts as DropTarget
    	//
        if (figure instanceof draw2d.InputPort) {
            return this._super(figure);
        }
        
        if (figure instanceof draw2d.HybridPort) {
            return this._super(figure);
        }
        
        return null;
    },
    
    /**
     * @inheritdoc
     * 
     */
    onDragLeave:function( figure)
    {
	  // Ports accepts only InputPorts as DropTarget
	  //
      if(figure instanceof draw2d.InputPort){
        this._super( figure);
      }
      else if(figure instanceof draw2d.HybridPort){
        this._super( figure);
      }
    },

    /**
     * @inheritdoc
     *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * @return {draw2d.command.Command} null or a valid command
     **/
    createCommand:function(request)
    {
       // Connect request between two ports
       //
       if(request.getPolicy() === draw2d.command.CommandType.CONNECT)
       {
         if(request.source.getParent().getId() === request.target.getParent().getId()){
            return null;
         }
    
         if(request.source instanceof draw2d.InputPort){
            // This is the different to the InputPort implementation of createCommand.
            return new draw2d.command.CommandConnect(request.canvas,request.target,request.source, request.source);
         }
         if(request.source instanceof draw2d.HybridPort){
             // This is the different to the InputPort implementation of createCommand.
             return new draw2d.command.CommandConnect(request.canvas,request.target,request.source, request.source);
         }
    
         return null;
       }
    
       // ...else call the base class
       return this._super(request);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.HybridPort
 * A HybridPort can work as Input and as Output port in the same way for a {@link draw2d.Connection}.
 * 
 * @author Andreas Herz
 * @extends draw2d.Port
 */ 
draw2d.HybridPort = draw2d.Port.extend({

    NAME : "draw2d.HybridPort",

    /**
     * @constructor
     * Create a new HybridPort element
     * 
     * @param {String} [name] the name for the Port.
     */
    init : function(name)
    {
        this._super(name);

        // responsive for the arrangement of the port 
        // calculates the x/y coordinates in relation to the parent node
        this.locator=new draw2d.layout.locator.InputPortLocator();
    },

    
    /**
     * @inheritdoc
     * 
     * @param {draw2d.Figure} figure The figure which is currently dragging
     * @return {draw2d.Figure} the figure which should receive the drop event or null if the element didnt want a drop event
     */
    onDragEnter : function (figure)
    {
    	// Accept any kind of port
        if (figure instanceof draw2d.Port) {
            return this._super(figure);
        }
        
        return null;
    },
    
    /**
     * @inheritdoc
     * 
     */
    onDragLeave : function (figure)
    {
	  // Ports accepts only Ports as DropTarget
	  //
	  if(!(figure instanceof draw2d.Port)){
		 return;
	  }

	  // accept any kind of port
      if(figure instanceof draw2d.Port){
        this._super( figure);
      }
      
    },

    /**
     * @inheritdoc
     *
     * @param {draw2d.command.CommandType} request describes the Command being requested
     * @return {draw2d.command.Command} null or a valid command
     **/
    createCommand:function(request)
    {
       // Connect request between two ports
       //
       if(request.getPolicy() === draw2d.command.CommandType.CONNECT) {
           
         if(request.source.getParent().getId() === request.target.getParent().getId()){
            return null;
         }    

         if (request.source instanceof draw2d.InputPort) {
            // This is the difference to the InputPort implementation of createCommand.
            return new draw2d.command.CommandConnect(request.canvas, request.target, request.source, request.source);
         }
         else if (request.source instanceof draw2d.OutputPort) {
            // This is the different to the OutputPort implementation of createCommand
            return new draw2d.command.CommandConnect(request.canvas, request.source, request.target, request.source);
         }
         else if (request.source instanceof draw2d.HybridPort) {
            // This is the different to the OutputPort implementation of createCommand
            return new draw2d.command.CommandConnect(request.canvas, request.target,request.source, request.source);
         }
         
         return null;
       }
    
       // ...else call the base class
       return this._super(request);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.layout.anchor.ConnectionAnchor
 *  An object to which a {@link draw2d.Connection} will be anchored.
 *  
 * @inheritable
 * @author Andreas Herz
 */
draw2d.layout.anchor.ConnectionAnchor = Class.extend({
    NAME : "draw2d.layout.anchor.ConnectionAnchor",

    /**
     * @constructor
     * 
     * @param {draw2d.Figure} [owner] the figure to use for the anchor calculation 
     */
    init:function(owner){
        this.owner = owner;
    },

    /**
     * @method
     * Returns the location where the Connection should be anchored in absolute coordinates. 
     * The anchor may use the given reference Point to calculate this location.
     * 
     * @param
     * @return {Number} reference The reference Point in absolute coordinates
     */
    getLocation:function(reference)
    {
       // return the center of the owner/port.
       return this.getReferencePoint();
    },
    
    /**
     * @method
     * Returns the Figure that contains this ConnectionAnchor.
     * 
     * @return {draw2d.Figure} The Figure that contains this ConnectionAnchor
     */
    getOwner:function()
    {
       return this.owner;
    },
    
    /**
     * @method
     * Set the owner of the Anchor.
     * 
     * @param {draw2d.Figure} owner the new owner of the anchor locator
     */
    setOwner:function( owner)
    {
    	if(typeof owner ==="undefined"){
    		throw "Missing parameter for 'owner' in ConnectionAnchor.setOwner";
    	}
        this.owner=owner;
    },
    
    /**
     * @method
     * Returns the bounds of this Anchor's owner.  Subclasses can override this method
     * to adjust the box. Maybe you return the box of the port parent (the parent figure)
     *
     * @return {draw2d.geo.Rectangle} The bounds of this Anchor's owner
     */
    getBox:function()
    {
      return this.getOwner().getAbsoluteBounds();
    },
    
    /**
     * @method
     * Returns the reference point for this anchor in absolute coordinates. This might be used
     * by another anchor to determine its own location.
     * 
     * @return {draw2d.geo.Point} The reference Point
     */
    getReferencePoint:function()
    {
       return this.getOwner().getAbsolutePosition();
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.anchor.ChopboxConnectionAnchor
 * 
 * The ChopboxAnchor's location is found by calculating the intersection of a
 * line drawn from the center point of its owner's box (the parent of the
 * connection port) to a reference point on that box. A Connection using the
 * ChopBoxAnchor will be oriented such that they point to their port owner's
 * center.
 * 
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.layout.anchor.ConnectionAnchor
 */
draw2d.layout.anchor.ChopboxConnectionAnchor = draw2d.layout.anchor.ConnectionAnchor.extend({

	NAME : "draw2d.layout.anchor.ChopboxConnectionAnchor",

	/**
	 * @constructor
	 * 
	 * @param {draw2d.Figure} [owner] the figure to use for the anchor calculation
	 */
	init : function(owner) {
		this._super(owner);
	},

	/**
	 * @method
	 * 
	 * Returns the location where the Connection should be anchored in
	 * absolute coordinates. The anchor may use the given reference
	 * Point to calculate this location.
	 * 
	 * @param reference The reference Point in absolute coordinates
	 * @return The anchor's location
	 */
	getLocation : function(reference) {
		var r = new draw2d.geo.Rectangle(0,0);
		r.setBounds(this.getBox());
		r.translate(-1, -1);
		r.resize(1, 1);

		var center = r.getCenter();

		if (r.isEmpty()	|| (reference.x == center.x && reference.y == center.y))
			return center; // This avoids divide-by-zero

		var dx = reference.x - center.x;
		var dy = reference.y - center.y;

		// r.width, r.height, dx, and dy are guaranteed to be non-zero.
		var scale = 0.5 / Math.max(Math.abs(dx) / r.w, Math.abs(dy)	/ r.h);

		dx *= scale;
		dy *= scale;
		center.translate( dx, dy);

		return center;
	},

	/**
	 * Returns the bounds of this Anchor's owner. Subclasses can
	 * override this method to adjust the box. Maybe you return the box
	 * of the port parent (the parent figure)
	 * 
	 * @return The bounds of this Anchor's owner
	 */
	getBox : function() {
		return this.getOwner().getParent().getBoundingBox();
	},

	/**
	 * @method
	 * 
	 * Returns the bounds of this Anchor's owner. Subclasses can
	 * override this method to adjust the box. Maybe you return the box
	 * of the port parent (the parent figure)
	 * 
	 * @return The bounds of this Anchor's owner
	 */
	getReferencePoint : function() {
		return this.getBox().getCenter();
	}
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.anchor.ShortesPathConnectionAnchor
 * 
 * The ChopboxAnchor's location is found by calculating the intersection of a
 * line drawn from the center point of its owner's box (the parent of the
 * connection port) to a reference point on that box. A Connection using the
 * ChopBoxAnchor will be oriented such that they point to their port owner's
 * center.
 * 
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.layout.anchor.ConnectionAnchor
 */
draw2d.layout.anchor.ShortesPathConnectionAnchor = draw2d.layout.anchor.ConnectionAnchor.extend({

	NAME : "draw2d.layout.anchor.ShortesPathConnectionAnchor",

	/**
	 * @constructor
	 * 
	 * @param {draw2d.Figure} [owner] the figure to use for the anchor calculation
	 */
	init : function(owner) {
		this._super(owner);
	},

	/**
	 * @method
	 * 
	 * Returns the location where the Connection should be anchored in
	 * absolute coordinates. The anchor may use the given reference
	 * Point to calculate this location.
	 * 
	 * @param {draw2d.geo.Point} ref The reference Point in absolute coordinates
	 * @return {draw2d.geo.Point} The anchor's location
	 */
	getLocation : function(ref) {
		var r =  this.getOwner().getParent().getBoundingBox();
	    var center = r.getCenter();
	    
		// check if we can calculate with a circle/line intersection
		//
		if(this.getOwner().getParent() instanceof draw2d.shape.basic.Oval){	
			var result = this.getOwner().getParent().intersectionWithLine(ref,center);
			if(result.getSize()==1){
				return result.get(0);
			}
		}
		
		/*    0 | 1 | 2
	     *    __|___|__
	     *    7 | 8 | 3
	     *    __|___|__
	     *    6 | 5 | 4
	     */
	    var octant = r.determineOctant(new draw2d.geo.Rectangle(ref.x,ref.y,2,2));
		
		switch(octant)
		{
		case 0:
		    return r.getTopLeft();
		case 1:
		    return new draw2d.geo.Point(ref.x,r.getTop());
        case 2:
            return r.getTopRight();
        case 3:
            return new draw2d.geo.Point(r.getRight(),ref.y);
		case 4:
            return r.getBottomRight();
        case 5:
            return new draw2d.geo.Point(ref.x,r.getBottom());
		case 6:
            return r.getBottomLeft();
        case 7:
            return new draw2d.geo.Point(r.getLeft(),ref.y);
		}
		
		return r.getTopLeft();
	},

	/**
	 * Returns the bounds of this Anchor's owner. Subclasses can
	 * override this method to adjust the box. Maybe you return the box
	 * of the port parent (the parent figure)
	 * 
	 * @return The bounds of this Anchor's owner
	 */
	getBox : function() {
		return this.getOwner().getParent().getBoundingBox();
	},

	/**
	 * @method
	 * 
	 * Returns the bounds of this Anchor's owner. Subclasses can
	 * override this method to adjust the box. Maybe you return the box
	 * of the port parent (the parent figure)
	 * 
	 * @return The bounds of this Anchor's owner
	 */
	getReferencePoint : function() {
		return this.getBox().getCenter();
	}
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.layout.anchor.CenterEdgeConnectionAnchor
 * 
 * The ChopboxAnchor's location is found by calculating the intersection of a
 * line drawn from the center point of its owner's box (the parent of the
 * connection port) to a reference point on that box. A Connection using the
 * ChopBoxAnchor will be oriented such that they point to their port owner's
 * center.
 * 
 * 
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.layout.anchor.ConnectionAnchor
 */
draw2d.layout.anchor.CenterEdgeConnectionAnchor = draw2d.layout.anchor.ConnectionAnchor.extend({

	NAME : "draw2d.layout.anchor.CenterEdgeConnectionAnchor",

	/**
	 * @constructor
	 * 
	 * @param {draw2d.Figure} [owner] the figure to use for the anchor calculation
	 */
	init : function(owner) {
		this._super(owner);
	},

	/**
	 * @method
	 * 
	 * Returns the location where the Connection should be anchored in
	 * absolute coordinates. The anchor may use the given reference
	 * Point to calculate this location.
	 * 
	 * @param {draw2d.geo.Point} ref The reference Point in absolute coordinates
	 * @return {draw2d.geo.Point} The anchor's location
	 */
	getLocation : function(ref) {
		var r =  this.getOwner().getParent().getBoundingBox();
	    
	    var dir = r.getDirection(ref);
		var center = r.getCenter();
	
		switch(dir)
		{
		case 0:
		    center.y=r.y;
		    break;
		case 1:
		    center.x = r.x+r.w;
		    break;
        case 2:
            center.y = r.y+r.h;
            break;
        case 3:
            center.x = r.x;
		}
		
		return center;
	},

	/**
	 * Returns the bounds of this Anchor's owner. Subclasses can
	 * override this method to adjust the box. Maybe you return the box
	 * of the port parent (the parent figure)
	 * 
	 * @return The bounds of this Anchor's owner
	 */
	getBox : function() {
		return this.getOwner().getParent().getBoundingBox();
	},

	/**
	 * @method
	 * 
	 * Returns the bounds of this Anchor's owner. Subclasses can
	 * override this method to adjust the box. Maybe you return the box
	 * of the port parent (the parent figure)
	 * 
	 * @return The bounds of this Anchor's owner
	 */
	getReferencePoint : function() {
		return this.getBox().getCenter();
	}
});
// TODO GIOVANNI
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.arrow.CalligrapherArrowLeft
 * Hand drawn arrow to the left.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.arrow.CalligrapherArrowLeft();
 *     figure.setColor("#3d3d3d");
 *     
 *     canvas.addFigure(figure,10,10);
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.arrow.CalligrapherArrowLeft = draw2d.SVGFigure.extend({
   
    NAME:"draw2d.shape.arrow.CalligrapherArrowLeft",

    /**
     * @constructor 
     * Creates a new instance
     * 
     */
    init:function(){
        this._super();
    },
    

    getSVG: function(){
         return '<svg width="230" height="60" xmlns="http://www.w3.org/2000/svg" version="1.1">'+
                '  <path style="fill:#000000;fill-opacity:1;fill-rule:nonzero;stroke:none" id="path3024" d="m 218.87943,27.464763 c -1.21088,-0.0563 -2.42064,-0.14616 -3.63262,-0.16893 c -5.82495,-0.10948 -18.61676,-0.0226 -22.97385,0.0122 c -7.12848,0.057 -14.25673,0.14021 -21.38495,0.22333 c -9.03765,0.10539 -18.07511,0.22813 -27.11266,0.3422 c -10.2269,0.11878 -20.4538,0.23756 -30.6807,0.35634 c -35.488759,0.4089 -70.975849,0.82793 -106.4669238,0.95195 c 0,0 7.9718628,-5.70244 7.9718628,-5.70244 l 0,0 c 6.374241,0.28694 12.745594,0.64561 19.122722,0.86083 c 28.09499,0.94816 56.21338,0.92473 84.315959,0.32205 c 10.51273,-0.32805 21.0288,-0.56402 31.53819,-0.98412 c 27.47361,-1.09824 54.91405,-2.91665 82.28177,-5.53697 c 0,0 -12.9788,9.32351 -12.9788,9.32351 z" inkscape:connector-curvature="0" />'+
                '  <path style="fill:#000000;fill-opacity:1;fill-rule:nonzero;stroke:none" id="path3026" d="m 100.75066,1.6309831 c -7.165239,3.9571 -14.284929,7.47866 -22.036659,10.2707299 c -5.00195,1.80163 -10.10374,3.31886 -15.2402,4.79424 c -8.25878,2.37815 -16.55626,4.65805 -24.9012,6.79479 c -2.89107,0.71593 -5.74687,1.56407 -8.66266,2.20424 c -3.211679,0.70512 -6.49468,1.17333 -9.752959,1.6747 c -5.447101,0.92112 -10.9044008,1.81762 -16.3983488,2.50082 c -1.608931,0.0814 -0.850754,0.10697 -2.275834,-0.0365 C 20.004071,21.041553 19.256899,21.517873 32.515691,19.216243 c 6.21537,-1.05913 12.34875,-2.37668 18.3945,-4.03234 c 8.12719,-2.02803 16.23765,-4.1157 24.26421,-6.4321199 c 5.23574,-1.55053 10.41682,-3.15473 15.46857,-5.12875 c 1.38953,-0.54295 2.7579,-1.12682 4.12253,-1.71603 c 0.98421,-0.42496 3.86537,-1.81801999 2.92296,-1.32600999 C 93.642191,2.6934931 89.529511,4.7073031 85.450031,6.7704531 l 15.300629,-5.1394 z" inkscape:connector-curvature="0" sodipodi:nodetypes="csccsccccccsssccc" />'+
                '  <path style="fill:#000000;fill-opacity:1;fill-rule:nonzero;stroke:none" id="path3028" d="m 80.764281,58.068863 c -2.45498,-3.50762 -6.58178,-6.10525 -10.40324,-8.66732 c -4.30614,-2.72676 -7.93958,-6.28283 -12.6021,-8.28702 c -7.39912,-4.50257 -11.70055,-7.85592 -20.85866,-9.23429 c -4.9257,-0.85706 -17.294247,-1.32043 -22.226462,-2.15427 c -3.445882,-0.42869 -6.2035918,0.70541 -9.6845138,0.57715 c -1.496337,-0.0586 -2.99355,-0.0965 -4.491229,-0.12472 l 13.9525278,-6.24562 l 3.25,-1.17153 c 1.441459,0.0813 -1.116338,0.15309 0.325505,0.23016 c 3.574557,0.17902 7.211864,0.0695 10.712655,0.73822 c 4.723107,1.08097 9.443947,2.1624 14.234177,3.05317 c 2.76739,0.64203 3.92627,0.87082 6.64127,1.66289 c 4.42146,1.28993 8.60075,3.01513 12.86503,4.58129 c 1.90199,0.73446 5.05193,1.93181 6.89302,2.7216 c 4.92005,2.11061 9.5916,4.57045 13.9716,7.31023 c 4.16708,2.62011 8.48023,5.20033 11.72012,8.56863 z" inkscape:connector-curvature="0" sodipodi:nodetypes="ccccccccccccscsccc" />'+
                '</svg>';
    },
    
    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element
     **/
    repaint : function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }

        
        if(this.svgNodes!==null){
            this.svgNodes.attr({fill: this.color.hash()});
        }
        
        this._super(attributes);
    }

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.arrow.CalligrapherArrowDownLeft
 * Hand drawn arrow which points down left
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.arrow.CalligrapherArrowDownLeft();
 *     figure.setColor("#3d3d3d");
 *     
 *     canvas.addFigure(figure,10,10);
 *     
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.arrow.CalligrapherArrowDownLeft = draw2d.SVGFigure.extend({
   
    NAME:"draw2d.shape.arrow.CalligrapherArrowDownLeft",

    /**
     * @constructor
     * Create a new instance
     */
    init:function(){
        this._super();
    },
    

    getSVG: function(){
         return '<svg width="180" height="300" xmlns="http://www.w3.org/2000/svg" version="1.1">'+
                '     <path style="fill:#000000;fill-opacity:1;fill-rule:nonzero;stroke:none" id="path3084" d="m 159.63578,17.846597 c 0.43137,9.44641 -0.0636,18.88035 -0.8284,28.30165 c -1.73211,18.38336 -4.05381,36.71698 -6.08253,55.075313 c -1.61738,13.7075 -3.03402,27.43467 -3.97611,41.19113 c -1.09101,11.16584 -1.31019,22.36559 -1.28541,33.56466 c -0.1328,4.82188 0.3218,9.6468 0.14332,14.46812 c -0.0888,2.39977 -0.28315,3.73625 -0.55012,6.12095 c -0.85848,4.73147 -2.27416,9.40019 -4.7769,13.68272 c -1.47003,2.51544 -3.78493,5.6647 -5.47739,8.05048 c -5.02888,6.66256 -11.08555,12.65652 -18.10552,17.75963 c -4.23302,3.07716 -7.74942,5.12065 -12.22081,7.86298 c -13.253319,6.72606 -25.889792,15.11686 -40.84124,18.60576 c -3.016829,0.7039 -4.431417,0.8157 -7.450859,1.2076 c -6.983246,0.5774 -14.009174,0.3375 -21.010676,0.2509 c -3.278795,-0.033 -6.55749,0.01 -9.835897,0.045 c 0,0 20.838833,-13.2364 20.838833,-13.2364 l 0,0 c 3.147056,0.093 6.294483,0.1852 9.443646,0.2007 c 6.966697,0.011 13.971433,0.1301 20.897176,-0.6344 c 3.732439,-0.5577 7.321215,-1.2431 10.881203,-2.4145 c 1.517208,-0.4992 5.830867,-2.43339 4.487902,-1.6386 c -6.098183,3.6088 -25.104875,12.8748 -9.52514,5.223 c 4.40875,-2.5927 8.262057,-4.7459 12.425175,-7.65986 c 6.839117,-4.78709 12.633657,-10.50427 17.500607,-16.86761 c 2.53518,-3.56692 5.24684,-7.12748 7.07617,-11.03446 c 1.42357,-3.0404 2.21532,-6.28727 2.91146,-9.50152 c 0.91919,-6.88822 1.03991,-13.81392 1.25118,-20.74151 c 0.47683,-11.27871 0.96259,-22.55877 1.61689,-33.83062 c 1.21127,-14.03392 3.64191,-27.94339 5.46543,-41.92167 c 2.26899,-18.186603 4.6835,-36.384373 5.4487,-54.679643 c 0.0788,-2.46092 0.23808,-4.92087 0.23618,-7.38276 c -0.005,-6.45916 -0.62194,-13.00218 -2.13821,-19.32664 c 0,0 23.48134,-10.73998 23.48134,-10.73998 z" inkscape:connector-curvature="0" />'+
                '     <path style="fill:#000000;fill-opacity:1;fill-rule:nonzero;stroke:none" id="path3086" d="m 41.271518,252.40239 c 2.465518,-0.7264 4.879503,-1.7726 7.145328,-2.9859 c 0.955597,-0.5117 3.736822,-2.1986 2.791991,-1.6673 c -5.218817,2.9348 -10.409826,5.9187 -15.61474,8.878 c 5.366557,-3.4898 10.227818,-7.6685 14.119927,-12.75576 c 3.507157,-5.09382 4.097613,-11.17122 4.301158,-17.17644 c 0.02635,-3.95844 -0.31227,-7.90612 -0.635377,-11.84752 c 0,0 19.920693,-10.3059 19.920693,-10.3059 l 0,0 c 0.171761,4.05015 0.409899,8.09777 0.50079,12.15101 c -0.185739,6.23619 -0.347804,12.66862 -3.492579,18.24747 c -0.503375,0.75197 -0.961922,1.53596 -1.510126,2.25591 c -3.478528,4.56826 -8.226837,8.04586 -12.757403,11.47443 c -7.345206,4.3297 -14.671333,8.692 -22.035619,12.9891 c -3.551305,2.0723 -7.368692,3.8726 -11.394645,4.7773 c 0,0 18.660602,-14.0344 18.660602,-14.0344 z" inkscape:connector-curvature="0" />'+
                '     <path style="fill:#000000;fill-opacity:1;fill-rule:nonzero;stroke:none" id="path3088" d="m 37.815923,255.49919 c 3.41111,0.1581 6.814569,0.2213 10.182693,0.8184 c 6.92998,2.6928 13.533527,6.2357 20.043462,9.8162 c 3.912139,2.1362 7.91195,4.4644 10.690321,8.0298 c 1.039962,1.2802 1.510411,2.7604 1.893523,4.3313 c 0,0 -20.370847,10.9259 -20.370847,10.9259 l 0,0 c -0.225419,-1.2711 -0.55067,-2.4558 -1.329618,-3.5184 c -2.332229,-3.3633 -5.869056,-5.6279 -9.247191,-7.8233 c -6.335066,-3.7106 -12.98611,-7.1834 -20.232784,-8.6836 c -3.497247,-0.3814 -7.011372,-0.4307 -10.521829,-0.1703 c 0,0 18.89227,-13.726 18.89227,-13.726 z" inkscape:connector-curvature="0" />'+
                '</svg>';
    },
    
    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element
     **/
    repaint : function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }

        
        if(this.svgNodes!==null){
            this.svgNodes.attr({fill: this.color.hash()});
        }
        
        this._super(attributes);
    }

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.Start
 * 
 * A generic Node which has an OutputPort. Mainly used for demo and examples.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.node.Start();
 *     figure.setColor("#3d3d3d");
 *     
 *     canvas.addFigure(figure,50,10);
 *     
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.shape.node.Start = draw2d.shape.basic.Rectangle.extend({

    NAME : "draw2d.shape.node.Start",

	DEFAULT_COLOR : new draw2d.util.Color("#4D90FE"),

	init : function()
    {
        this._super();
        
        this.createPort("output");

        this.setDimension(50, 50);
        this.setBackgroundColor(this.DEFAULT_COLOR);
        this.setColor(this.DEFAULT_COLOR.darker());
    }
 
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.node.End
 * A simple Node which has a InputPort. Mainly used for demo and examples.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.node.End();
 *     figure.setColor("#3d3d3d");
 *     
 *     canvas.addFigure(figure,50,10);
 *     
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.shape.node.End = draw2d.shape.basic.Rectangle.extend({

    NAME : "draw2d.shape.node.End",

    DEFAULT_COLOR : new draw2d.util.Color("#4D90FE"),
	
    init : function()
    {
        this._super();

        this.createPort("input");

        this.setDimension(50, 50);
        this.setBackgroundColor(this.DEFAULT_COLOR);
        this.setColor(this.DEFAULT_COLOR.darker());
    }

});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************//**
 * @class draw2d.shape.node.Between
 * A simple Node which has a  InputPort and OutputPort. Mainly used for demo and examples.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.node.Between();
 *     figure.setColor("#3d3d3d");
 *     
 *     canvas.addFigure(figure,50,10);
 *     
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.shape.node.Between = draw2d.shape.basic.Rectangle.extend({

    NAME : "draw2d.shape.node.Between",

    DEFAULT_COLOR : new draw2d.util.Color("#4D90FE"),

	init : function()
    {
        this._super();
        
        this.setDimension(50, 50);
        this.setBackgroundColor(this.DEFAULT_COLOR);
        this.setColor(this.DEFAULT_COLOR.darker());

        this.createPort("output");
        this.createPort("input");
    }
});

/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.note.PostIt
 * 
 * Simple Post-it like figure with text. Can be used for annotations or documentation.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var shape =  new draw2d.shape.note.PostIt("This is a simple sticky note");
 *     shape.setColor("#000000");
 *     shape.setPadding(20);
 *          
 *     canvas.addFigure(shape,40,10);
 *     
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Label
 */
draw2d.shape.note.PostIt= draw2d.shape.basic.Label.extend({

	NAME : "draw2d.shape.note.PostIt",

    /**
     * @constructor
     * Creates a new PostIt element.
     * 
     * @param {String} [text] the text to display
     */
    init : function(text)
    {
        this._super(text);
         
        this.setStroke(1);
        this.setBackgroundColor("#5b5b5b");
        this.setColor("#FFFFFF");
        this.setFontColor("#ffffff");
        this.setFontSize(14);
        this.setPadding(5);
        this.setRadius(5);
       
    }
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.widget.Widget
 * Base class for all diagrams.
 * 
 * @extends draw2d.SetFigure
 */
draw2d.shape.widget.Widget = draw2d.SetFigure.extend({
    
    init: function( width, height){
        this._super( width, height);
    },
    

    /**
     * @method
     * Return the calculate width of the set. This calculates the bounding box of all elements.
     * 
     * @return {Number} the calculated width of the label
     **/
    getWidth : function() {
        return this.width;
    },
    
    /**
     * @method
     * Return the calculated height of the set. This calculates the bounding box of all elements.
     * 
     * @return {Number} the calculated height of the label
     */
    getHeight:function(){
       return this.height;
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.widget.Slider
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var slider = new draw2d.shape.widget.Slider(120,20);
 *     canvas.addFigure( slider,100,60);
 * 
 * @extends draw2d.shape.widget.Widget
 */
draw2d.shape.widget.Slider = draw2d.shape.widget.Widget.extend({
    
    NAME : "draw2d.shape.widget.Slider",
    
    DEFAULT_COLOR_THUMB : new draw2d.util.Color("#bddf69"),
//    DEFAULT_COLOR_THUMB : new draw2d.util.Color("#54b0d0"),
    DEFAULT_COLOR_BG : new draw2d.util.Color("#d3d3d3"),
    
    
    init: function( width, height){
        // create some good defaults for width/height
        if(typeof width === "undefined"){
            width=120;
            height=15;
        }
        
        this.currentValue = 0; // [0..100] 
        this.slideBoundingBox = new draw2d.geo.Rectangle(0,0,10,20);

        this._super( width, height);
        
        this.setBackgroundColor(this.DEFAULT_COLOR_BG);
        this.setColor(this.DEFAULT_COLOR_THUMB);
        this.setStroke(1);
        this.setRadius(4);
        this.setResizeable(true);
        
        this.setMinHeight(10);
        this.setMinWidth(80);
    },
    
    /**
     * @method
     * Create the additional elements for the figure
     * 
     */
    createSet: function(){
        var result = this.canvas.paper.set();
        var thumb= this.canvas.paper.rect(5,5,10,20);
        thumb.node.style.cursor=  "col-resize";
        result.push(thumb);

        return result;
    },
    
    setDimension:function(w,h){
        this._super(w,h);
        this.slideBoundingBox.setBoundary(0,0,this.getWidth()-10 , this.getHeight());
        this.slideBoundingBox.setHeight(this.getHeight()+1);
        
        // TODO: and repaint again.....two repaints for one "setDimension"....BAD
        //
        this.repaint();
    },

    /**
     * @method
     * Called if the value of the slider has been changed.
     * 
     * @param {Number} value The new value of the slider in percentage [0..100]
     * @template
     */
    onValueChange:function(value){
    },
    
    /**
     * @method
     * Will be called if the drag and drop action begins. You can return [false] if you
     * want avoid that the figure can be move.
     * 
     * @param {Number} relativeX the x coordinate within the figure
     * @param {Number} relativeY the y-coordinate within the figure.
     * 
     * @return {boolean} true if the figure accepts dragging
     **/
    onDragStart : function(relativeX, relativeY ){
        
        // check if the use has been clicked on the thumb
        //
        if(this.slideBoundingBox.hitTest(relativeX, relativeY)){
            this.origX=this.slideBoundingBox.getX();
            this.origY=this.slideBoundingBox.getY();
            return false;
        }
        
        return this._super(relativeX, relativeY);
    },
    
    /**
     * @method
     * Called by the framework if the figure returns false for the drag operation. In this
     * case we send a "panning" event - mouseDown + mouseMove. This is very usefull for
     * UI-Widget like slider, spinner,...
     * 
     * @param {Number} dx the x difference between the mouse down operation and now
     * @param {Number} dy the y difference between the mouse down operation and now
     * @param {Number} dx2 The x diff since the last call of this dragging operation
     * @param {Number} dy2 The y diff since the last call of this dragging operation
     */
    onPanning: function(dx, dy, dx2, dy2){
        this.slideBoundingBox.setPosition(this.origX+dx, this.origY+dy);
        // calculate the internal value of the slider
        this.setValue(100/(this.slideBoundingBox.bw-this.slideBoundingBox.getWidth())*this.slideBoundingBox.getX());
    },

    /**
     * @method
     * Set the current value of the slider. Valid values are [0..100]
     * 
     * @param {Number} value values between [0..100]
     */
    setValue:function(value){
        this.currentValue = Math.min(Math.max(0,(value|0)),100);
        this.repaint();
        this.onValueChange(this.currentValue);
    },

    
    /**
     * 
     * @param attributes
     */
    repaint: function(attributes){
        
        if (this.repaintBlocked === true || this.shape === null){
            return;
        }

        if (typeof attributes === "undefined") {
            attributes = {};
        }
     
        // adjust the slider to the current value and the new dimension of the widget
        //
        var thumbX = ((this.slideBoundingBox.bw-this.slideBoundingBox.getWidth())/100*this.currentValue)|0;
        this.slideBoundingBox.setX(thumbX);


        // update slider
        //
		if (this.svgNodes !== null) {
			var attr = this.slideBoundingBox.toJSON();
			attr.y = attr.y - 5;
			attr.height = attr.height + 10;
			attr.fill = this.getColor().hash();
			attr.stroke = this.getColor().darker(0.2).hash();
			attr.r = 4;
			this.svgNodes.attr(attr);
		}
 
        
        attributes.fill= "90-"+this.bgColor.hash()+":5-"+this.bgColor.lighter(0.3).hash()+":95";
        attributes.stroke = this.bgColor.darker(0.1).hash();

        this._super(attributes);
    },
    

    applyTransformation:function(){
        this.svgNodes.transform("T" + this.getAbsoluteX() + "," + this.getAbsoluteY());
    }

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.diagram.Diagram
 * Base class for all digrams.
 * 
 * @extends draw2d.SetFigure
 */
draw2d.shape.diagram.Diagram = draw2d.SetFigure.extend({
    
    init: function( width, height){
        
        this.data = [];
        this.padding = 5;
        this.cache = {}; 
        
        this._super( width, height);
        
        this.setBackgroundColor("#8dabf2");
        this.setStroke(1);
        this.setColor("#f0f0f0");
        this.setRadius(2);
        this.setResizeable(true);
    },
    
    /**
     * @method
     * Set the data for the chart/diagram element
     * 
     * @param {Array} data
     */
    setData:function( data){
        this.data = data;
        this.cache={};
        

       if (this.svgNodes !== null) {
            this.svgNodes.remove();
            this.svgNodes = this.createSet();
        }
        
       this.repaint();
    },

    setDimension:function(w,h){
        this.cache={};
        this._super(w,h);
    },

    
    /**
     * @method
     * Return the calculate width of the set. This calculates the bounding box of all elements.
     * 
     * @return {Number} the calculated width of the label
     **/
    getWidth:function() {
        return this.width;
    },
    
    /**
     * @method
     * Return the calculated height of the set. This calculates the bounding box of all elements.
     * 
     * @return {Number} the calculated height of the label
     */
    getHeight:function(){
       return this.height;
    },
    
    /**
     * 
     * @param attributes
     */
    repaint:function(attributes){
        if(this.repaintBlocked===true || this.shape==null){
            return;
        }
        
        if (typeof attributes === "undefined") {
            attributes = {};
        }

        if(typeof attributes.fill ==="undefined"){
            attributes.fill= "none";
        }
         
        this._super(attributes);
    },
    
    applyTransformation:function(){
        if (this.isResizeable()===true) {
            this.svgNodes.transform("S"+this.scaleX+","+this.scaleY+","+this.getAbsoluteX()+","+this.getAbsoluteY()+ "t"+ this.getAbsoluteX() + "," + this.getAbsoluteY());
        }
        else {
            this.svgNodes.transform("T" + this.getAbsoluteX() + "," + this.getAbsoluteY());
        }
    }
    

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.diagram.Pie
 * 
 * Small data pie chart.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var pie = new draw2d.shape.diagram.Pie(80,80);
 *     pie.setData([30,60,122,4]);
 *     canvas.addFigure( pie,100,60);
 *     
 * @extends draw2d.shape.diagram.Diagram
 */
draw2d.shape.diagram.Pie = draw2d.shape.diagram.Diagram.extend({
    
    COLORS: ['#00A8F0', '#b9dd69', '#f3546a', '#4DA74D', '#9440ED'],
    TWO_PI : Math.PI * 2,
    
    init: function( diameter){
        this._super( diameter, diameter);
        this.setStroke(0);
    },
    
    setData:function( data){
        
        // Normalize the Data.
        // The SUM must be == 1.
        this.sum = 0;
        $.each(data,$.proxy(function(i,val){this.sum +=val;},this));
        var _sum=1/this.sum;
        $.each(data,$.proxy(function(i,val){data[i] = _sum*val;},this));
        
        //  pass the normalize data to the base implementation
        //
        this._super(data);
    },
      
    /**
     * @method
     * Create the additional elements for the figure
     * 
     */
    createSet : function()
    {
        var radius = this.getWidth()/2;
        var length= this.data.length;

        var pie = this.canvas.paper.set();

        var offsetAngle = 0;

        for ( var i = 0; i < length; i++) {
            // angle is percent of TWO_PI
            var angle = this.TWO_PI * this.data[i];
            var color = this.COLORS[i%length];
            var seg = this.drawSegment(radius, angle, offsetAngle, 0.1);
            seg.attr({stroke: this.color.hash(),fill:color});
            pie.push(seg);
            offsetAngle += angle;
        }
        return pie;
    },
     
    setDimension:function(w,h){
        // keep the aspect ration
        //
        if(w>h){
            this._super(w,w);
         }
         else{
            this._super(h,h);
         }
        
        // we must recreate the diagram if we change the size.
        // low performance. Better: transfor/scale the set. Can be done in the next release
        //
        if (this.svgNodes !== null) {
            this.svgNodes.remove();
            this.svgNodes = this.createSet();
        }
                
        this.repaint();
    },

    polarPath:function(radius, theta, rotation){
        var x, y;
        x = radius * Math.cos(theta + rotation)+radius;
        y = radius * Math.sin(theta + rotation)+radius;
        return "L " + x + " " + y + " "; 
    },

    drawSegment:function(radius, value, rotation, resolution){
      var path = "M "+radius+" "+radius;

      for (var i = 0; i < value; i+=resolution){
        path += this.polarPath(radius, i, rotation);
      }
      path += this.polarPath(radius, value, rotation);

      path += "L "+radius+" "+radius;
      return this.getCanvas().paper.path(path);
    },
    
    applyTransformation:function(){
       this.svgNodes.transform("T" + this.getAbsoluteX() + "," + this.getAbsoluteY());
    }
    
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.diagram.Sparkline
 * 
 * Small data line diagram.
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var testData = [];
 *     for(var i=0;i<100;i++) {
 *       testData.push(Math.floor(Math.random() * 100));
 *     }
 *     
 *     var sparkline = new draw2d.shape.diagram.Sparkline();
 *     sparkline.setData(testData);
 *   
 *     canvas.addFigure( sparkline,100,60);
 *     
 * @extends draw2d.shape.diagram.Diagram
 */
draw2d.shape.diagram.Sparkline = draw2d.shape.diagram.Diagram.extend({
    
    init: function( width, height){
        this.min = 0;
        this.max = 10;

        // set some feasible default values
        //
        if(typeof width === "undefined"){
            width=180;
            height=50;
        }
        
        this._super( width, height);
    },
    
    
    setData:function( data){
        // get the min/max from an array and not only from two elements..
        this.min = Math.min.apply(Math, data);
        this.max = Math.max.apply(Math, data);

        if(this.max==this.min){
            this.max = this.min+1;
        }
        
        this._super(data);
    },
    
    /**
     * @method
     * Create the additional elements for the figure
     * 
     */
    createSet: function(){
        return this.canvas.paper.path("M0 0 l0 0");
    },
     
    /**
     * 
     * @param attributes
     */
    repaint: function(attributes){
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }
        
        if (typeof attributes === "undefined") {
            attributes = {};
        }

        attributes.fill= "90-#000:5-#4d4d4d:95";
        
        var padding = this.padding;
        var width = this.getWidth()- 2*+this.padding;
        var height= this.getHeight()- 2*+this.padding;
        var length= this.data.length;
        var min = this.min;
        var max = this.max;
        var toCoords = function(value, idx) {
            var step =1;
            // avoid divisionByZero
            if(length>1){
                step = (width/ (length-1));
            }

            return {
                y:  -((value-min)/(max-min) * height) + height+padding,
                x: padding+idx*step
            };
        };

        if(this.svgNodes!==null && (typeof this.cache.pathString ==="undefined")){
            var prev_pt=null;
            $.each(this.data, $.proxy(function(idx, item) {
                var pt = toCoords(item, idx);
                if(prev_pt===null) {
                    this.cache.pathString = [ "M", pt.x, pt.y].join(" ");
                }
                else{
                    this.cache.pathString = [ this.cache.pathString,"L", pt.x, pt.y].join(" ");
                }
                prev_pt = pt;
            },this));

            this.svgNodes.attr({path:this.cache.pathString, stroke: "#f0f0f0"});
            
        }
        this._super(attributes);
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.analog.OpAmp
 * Hand drawn arrow which points down left
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.analog.OpAmp();
 *     canvas.addFigure(figure,10,10);
 *     
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.analog.OpAmp = draw2d.SVGFigure.extend({

    NAME:"draw2d.shape.analog.OpAmp",
    
    // custom locator for the special design of the OpAmp Input area
    MyInputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, port){
        	var parent = port.getParent();
            var calcY = (8+18.5*index)*parent.scaleY;
            this.applyConsiderRotation(port, 1, calcY);
        }
    }),

    /**
     * @constructor
     * Create a new instance
     */
    init:function(width, height){
        this._super(width||50,height||50);
        
        this.inputLocator = new this.MyInputPortLocator();
        
        this.createPort("input", this.inputLocator);
        this.createPort("input", this.inputLocator);
        
        this.createPort("output");
        
        this.setBackgroundColor("#f0f0ff");
    },

    
    getSVG: function(){
         return '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50"  height="50">'+
                 '<path d="m8.2627,0l0,35.36035l31.23926,-17.76025l-31.23926,-17.60011l0,0l0,0.00001zm2.27832,27.36719l4.08105,0m-2.10449,-2.20703l0,4.27979m2.26367,-21.35938l-4.15918,0"  stroke="#1B1B1B" fill="none"/>'+
                 '<line x1="0.53516"  y1="8"  x2="8.21191"  y2="8"  stroke="#010101"/>'+
                 '<line x1="39.14941" y1="18" x2="45.81055" y2="18" stroke="#010101" />'+
                 '<line x1="0.53516"  y1="27" x2="8.21191"  y2="27" stroke="#010101" />'+
                '</svg>';
    },
    
    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element
     **/
     repaint : function(attributes)
     {
         if (this.repaintBlocked===true || this.shape === null){
             return;
         }

         if(typeof attributes === "undefined" ){
             attributes = {};
         }

         // redirect the backgroundColor to an internal SVG node.
         // In this case only a small part of the shape are filled with the background color
         // and not the complete rectangle/bounding box
         //
         attributes["fill"] = "none";
         if( this.bgColor!=null){
             this.svgNodes[0].attr({fill: this.bgColor.hash()});
         }
         
         this._super(attributes);
     }

});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.analog.ResistorBridge
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.analog.ResistorBridge();
 *     canvas.addFigure(figure,10,10);
 *     
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.analog.ResistorBridge = draw2d.SVGFigure.extend({

    NAME:"draw2d.shape.analog.ResistorBridge",
   
    // custom locator for the special design of the ResistorBridge Input area
    MyInputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            var h = figure.getParent().getHeight();
            this.applyConsiderRotation(figure,w/2+1, h*index);
        }
    }),

    // custom locator for the special design of the ResistorBridge Input area
    MyOutputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            var h = figure.getParent().getHeight();
            
            this.applyConsiderRotation(figure,w*(index-2), h/2);
        }
    }),


    /**
     * @constructor
     * Create a new instance
     */
    init:function(width, height){
        if(typeof width === "undefined"){
            width = 50;
            height= 50;
        }
        
        this._super(width,height);
        
        this.inputLocator = new this.MyInputPortLocator();
        this.outputLocator = new this.MyOutputPortLocator();
        
        this.createPort("hybrid",this.inputLocator);
        this.createPort("hybrid",this.inputLocator);
        
        this.createPort("hybrid",this.outputLocator);
        this.createPort("hybrid",this.outputLocator);
    },
    

    getSVG: function(){
         return '<svg  xmlns="http://www.w3.org/2000/svg" version="1.1">'+
                 '<path fill="#010101" stroke="#010101" stroke-miterlimit="14.3" id="path12322" d="m47.62207,22.71094l0,0c0.73145,0.73242 0.71777,1.93359 -0.03027,2.68164c-0.74805,0.74951 -1.94922,0.76123 -2.68073,0.0293c-0.73138,-0.73242 -0.71967,-1.93211 0.03033,-2.68115c0.74707,-0.74803 1.94727,-0.76219 2.68066,-0.02979l0,0z"/>'+
                 '<path fill="#010101" stroke="#010101" stroke-miterlimit="14.3" id="path12324" d="m25.84082,0.93115l0,0c0.73145,0.73096 0.71875,1.93359 -0.02832,2.68066c-0.75,0.74951 -1.94922,0.76123 -2.68164,0.0293c-0.73242,-0.73241 -0.71973,-1.93164 0.0293,-2.68065c0.74805,-0.74756 1.94922,-0.76172 2.68066,-0.0293l0,0l0,-0.00002z"/>'+
                 '<path fill="#010101" stroke="#010101" stroke-miterlimit="14.3" id="path12326" d="m25.75098,44.58203l0,0c0.73145,0.73193 0.71875,1.93311 -0.02832,2.68115c-0.75,0.74902 -1.94922,0.76074 -2.68262,0.0293c-0.73145,-0.73193 -0.71973,-1.93262 0.03033,-2.68164c0.74707,-0.74756 1.94922,-0.76123 2.68066,-0.02879l0,0l-0.00006,-0.00002z"/>'+
                 '<path fill="#010101" stroke="#010101" stroke-miterlimit="14.3" id="path12328" d="m3.9707,22.80127l0,0c0.73242,0.73193 0.71777,1.93359 -0.0293,2.68115c-0.74902,0.74951 -1.94922,0.76172 -2.68164,0.0293c-0.73145,-0.73242 -0.71973,-1.93164 0.03027,-2.68115c0.74707,-0.74707 1.94922,-0.76074 2.68066,-0.0293l0,0z"/>'+
                 '<polyline fill="none" stroke="#010101" id="polyline12334" points="24.908203125,45.49267578125 31.71875,38.68310546875 31.2119140625,36.98876953125 34.892578125,37.95703125 33.953125,34.22265625 37.6650390625,35.18359375 36.6767578125,31.52490234375 40.3759765625,32.47314453125 39.873046875,30.52783203125 45.884765625,24.51708984375 " stroke-miterlimit="14.3"/>'+
                 '<polyline fill="#010101" id="polyline12338" points="36.3408203125,23.98876953125 38.146484375,29.55810546875 33.630859375,29.55810546875 35.435546875,23.98779296875 "/>'+
                 '<line fill="none" stroke="#010101" id="line12340" y2="28.90967" x2="35.8877" y1="41.13428" x1="35.88867" stroke-miterlimit="14.3"/>'+
                 '<polyline fill="none" stroke="#010101" id="polyline12346" points="3.2109375,23.79248046875 10.01953125,16.98388671875 9.513671875,15.2890625 13.193359375,16.25732421875 12.251953125,12.5234375 15.9658203125,13.48486328125 14.9775390625,9.82568359375 18.6767578125,10.7734375 18.173828125,8.82958984375 24.185546875,2.81787109375 " stroke-miterlimit="14.3"/>'+
                 '<polyline fill="#010101" id="polyline12350" points="13.126953125,23.80419921875 11.3212890625,18.236328125 15.8369140625,18.236328125 14.0322265625,23.806640625 "/>'+
                 '<line fill="none" stroke="#010101" id="line12352" y2="18.8833" x2="13.58008" y1="6.65967" x1="13.5791" stroke-miterlimit="14.3"/>'+
                 '<polyline fill="none" stroke="#010101" id="polyline12358" points="46.65625,24.33642578125 39.84765625,17.52783203125 38.154296875,18.033203125 39.1220703125,14.353515625 35.3876953125,15.29345703125 36.34765625,11.58056640625 32.689453125,12.56884765625 33.6376953125,8.86865234375 31.6923828125,9.373046875 24.322265625,2.00341796875 " stroke-miterlimit="14.3"/>'+
                 '<polyline fill="#010101" id="polyline12362" points="36.578125,1.87109375 38.3828125,7.439453125 33.8681640625,7.439453125 35.6728515625,1.869140625 "/>'+
                 '<line fill="none" stroke="#010101" id="line12364" y2="6.7915" x2="36.125" y1="19.01758" x1="36.125" stroke-miterlimit="14.3"/>'+
                 '<polyline fill="none" stroke="#010101" id="polyline12370" points="24.494140625,46.49951171875 17.685546875,39.69091796875 15.9921875,40.1953125 16.958984375,36.515625 13.2265625,37.45556640625 14.185546875,33.7421875 10.52734375,34.73193359375 11.474609375,31.03125 9.529296875,31.53515625 2.1611328125,24.166015625 " stroke-miterlimit="14.3"/>'+
                 '<polyline fill="#010101" id="polyline12374" points="12.150390625,44.80029296875 10.34765625,39.23193359375 14.861328125,39.23095703125 13.0556640625,44.80224609375 "/>'+
                 '<line fill="none" stroke="#010101" id="line12376" y2="39.87891" x2="12.60352" y1="27.6543" x1="12.60352" stroke-miterlimit="14.3"/>'+
                '</svg>';
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.analog.ResistorVertical
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.analog.ResistorVertical();
 *     canvas.addFigure(figure,10,10);
 *     
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.analog.ResistorVertical = draw2d.SetFigure.extend({

    NAME:"draw2d.shape.analog.ResistorVertical",
    
    // custom locator for the special design of the Input area
    MyInputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            var h = figure.getParent().getHeight();
            this.applyConsiderRotation(figure,w/2, h);
        }
    }),
    
    // custom locator for the special design of the Output area
    MyOutputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            this.applyConsiderRotation(figure,w/2, 0);
        }
    }),

    /**
     * @constructor
     * Create a new instance
     */
    init:function(width, height){
        if(typeof width === "undefined"){
            width = 30;
            height= 50;
        }
        
        this._super(width,height);
        this.inputLocator = new this.MyInputPortLocator();
        this.outputLocator = new this.MyOutputPortLocator();

        this.createPort("hybrid", this.inputLocator); 
        this.createPort("hybrid", this.outputLocator);
    },
    

    createSet: function(){
        var set = this._super();
        
    	set.push( this.canvas.paper.path("M15,0 L15,5 L0,7.5 L30,10 L0,15 L30,20 L0,25 L30,30 L15,32.5 L15,40"));
    	
    	return set;
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.analog.VoltageSupplyHorizontal
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.analog.VoltageSupplyHorizontal();
 *     canvas.addFigure(figure,10,10);
 *     
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.analog.VoltageSupplyHorizontal = draw2d.SVGFigure.extend({

    NAME:"draw2d.shape.analog.VoltageSupplyHorizontal",
    
    // custom locator for the special design of the Input area
    MyInputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var h = figure.getParent().getHeight();
            this.applyConsiderRotation(figure,0, h/2);
        }
    }),
    
    // custom locator for the special design of the Output area
    MyOutputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            var h = figure.getParent().getHeight();
            this.applyConsiderRotation(figure,w, h/2);
        }
    }),

    /**
     * @constructor
     * Create a new instance
     */
    init:function(width, height){
        if(typeof width === "undefined"){
            width = 50;
            height= 30;
        }
        
        this._super(width,height);
        
        this.createPort("hybrid", new this.MyInputPortLocator());  // GND
        this.createPort("hybrid", new this.MyOutputPortLocator()); // VCC
    },
    

    getSVG: function(){
         return '<svg width="49" height="28" xmlns="http://www.w3.org/2000/svg" version="1.1">'+
                '<path d="m24.99933,18.95592l0,-9.54576m-5.78374,-9.40907l0,28.35939m-5.78718,-9.40457l0,-9.54576m-5.78374,-9.40907l0,28.35939" id="path10566" stroke-miterlimit="14.3" stroke="#010101" fill="none"/>'+
                '<path d="m26.79878,14.13039l6.90583,0m-33.22691,0l6.90583,0" id="path10568" stroke-miterlimit="14.3" stroke-linecap="square" stroke="#010101" fill="none"/>'+
                '</svg>';
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.analog.VoltageSupplyVertical
 * 
 * See the example:
 *
 *     @example preview small frame
 *     
 *     var figure =  new draw2d.shape.analog.VoltageSupplyVertical();
 *     canvas.addFigure(figure,10,10);
 *     
 *     
 * @extends draw2d.SVGFigure
 */
draw2d.shape.analog.VoltageSupplyVertical = draw2d.SVGFigure.extend({

    NAME:"draw2d.shape.analog.VoltageSupplyVertical",
    
    // custom locator for the special design of the Input area
    MyInputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            var h = figure.getParent().getHeight();
            this.applyConsiderRotation(figure,w/2, h);
        }
    }),
    
    // custom locator for the special design of the Output area
    MyOutputPortLocator : draw2d.layout.locator.PortLocator.extend({
        init:function( ){
          this._super();
        },    
        relocate:function(index, figure){
            var w = figure.getParent().getWidth();
            this.applyConsiderRotation(figure,w/2, 0);
        }
    }),

    /**
     * @constructor
     * Create a new instance
     */
    init:function(width, height){
        if(typeof width === "undefined"){
            width = 30;
            height= 50;
        }
        
        this._super(width,height);
        
        this.inputLocator = new this.MyInputPortLocator();
        this.outputLocator = new this.MyOutputPortLocator();

        this.createPort("hybrid", this.inputLocator); // GND
        this.createPort("hybrid", this.outputLocator);// VCC
    },
    

    getSVG: function(){
         return '<svg  xmlns="http://www.w3.org/2000/svg" version="1.1">'+
                '<path d="m19.62398,12.37594l-9.87926,0m-9.74355,8.22145l29.36289,0m-9.74007,8.22469l-9.87927,0m-9.74355,8.22145l29.36289,0" id="path10560" stroke-miterlimit="14.3" stroke="#010101" fill="none"/>'+
                '<path d="m14.63157,9.81646l0,-9.81646m0,47.2328l0,-9.81646" id="path10562" stroke-miterlimit="14.3" stroke-linecap="square" stroke="#010101" fill="none"/>'+
                '</svg>';
    }
});
/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.layout.Layout
 * 
 * A base class for positioning child figures and determining the ideal size for 
 * a figure with children. 
 * 
 *     
 * @author Andreas Herz
 * @extends draw2d.shape.basic.Rectangle
 */
draw2d.shape.layout.Layout= draw2d.shape.basic.Rectangle.extend({

	NAME : "draw2d.shape.layout.Layout",

    /**
     * @constructor
     * 
     * 
     */
    init : function()
    {
        this._super();
         
        this.setBackgroundColor(null);
        this.setRadius(0);
        this.setStroke(0);
        this.installEditPolicy(new draw2d.policy.figure.AntSelectionFeedbackPolicy());
    },
    
    /**
     * @method
     * Add a child figure to the shape and use the locator from the inherit class
     * 
     * @param {draw2d.Figure} child
     */
    addFigure : function(child)
    {
       this._super(child, this.locator);
       child.attachResizeListener(this);
       this.fireResizeEvent();
    },

    /**
     * @method
     * Remove the giben figure from the shape and recalculate the layout.
     * 
     * @param {drawd.Figure} child
     * @since 4.0.0
     */
    removeFigure : function(child)
    {
       this._super(child);
       child.detachResizeListener(this);
       this.setDimension(1,1);
    },

    
    onOtherFigureIsResizing:function(figure)
    {
        // propagate the event to the parent or other listener if existing
        //
        if(this.getParent() instanceof draw2d.shape.layout.Layout){
            this.fireResizeEvent();
        }
        // or we are the parent and must consume it self
        else {
            this.setDimension(1,1);
        }
    },
    

    onDoubleClick:function()
    {
    	// ignore them for the layout elements
        // Layout's can't rotate
    }
    
});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.layout.HorizontalLayout
 * The HorizontalLayout class arranges the layout elements in a horizontal sequence, 
 * left to right, with optional gaps between the elements. 
 * 
 * During the execution of the setDimension() method, the minimum width of the container is calculated 
 * by accumulating the minimum sizes of the elements, including stroke, gaps and padding. 
 * 
 * 
 * See the example below with and without gap and border settings
 * 
 *     
 *     @example preview small frame
 *     
 *     // first container without any gap and a border of the parent
 *     // container
 *     var label1 =  new draw2d.shape.basic.Label("Label 1");
 *     var label2 =  new draw2d.shape.basic.Label("Label 2");
 *     var label3 =  new draw2d.shape.basic.Label("Label 3");
 *     
 *     var container1 = new draw2d.shape.layout.HorizontalLayout();
 *     
 *     container1.addFigure(label1);
 *     container1.addFigure(label2);
 *     container1.addFigure(label3);
 *     container1.setGap(10);
 *     container1.setStroke(2);
 *     canvas.addFigure(container1,50,10);
 *     
 *     // second container without any gab or border
 *     //
 *     var label11 =  new draw2d.shape.basic.Label("Label 1");
 *     var label12 =  new draw2d.shape.basic.Label("Label 2");
 *     var label13 =  new draw2d.shape.basic.Label("Label 3");
 *     
 *     var container2 = new draw2d.shape.layout.HorizontalLayout();
 *     
 *     container2.addFigure(label11);
 *     container2.addFigure(label12);
 *     container2.addFigure(label13);
 *     
 *     canvas.addFigure(container2,50,90);
 *     
 *     
 * @author Andreas Herz
 * @extends draw2d.shape.layout.Layout
 * @since 2.5.1
 */
draw2d.shape.layout.HorizontalLayout= draw2d.shape.layout.Layout.extend({

	NAME : "draw2d.shape.layout.HorizontalLayout",

    /**
     * @constructor
     * Constructs a HorizontalLayout.
     * 
     */
    init : function()
    {
        this._super();
        var _this = this;
        this.locator ={ 
                relocate:function(index, target)
                {
                    var stroke = _this.getStroke();
                    var xPos = stroke; // respect the border of the parent
                    
                    for (var i=0;i<index;i++){
                        var child = _this.children.get(i).figure;
                        xPos+=child.getWidth()+_this.gap;
                    }
                    
                    target.setPosition(xPos,stroke);
                 }
        };

        this.setDimension(1,1);
        this.gap = 0;
   },

   /**
    * @method
    * Set the gap width between child components within this layout. 
    * This will only affect the space between components, not the space around all the components in the layout.
    * 
    * @param {Number} gap The space, in pixels, between items.
    * @since 2.5.1
    */
   setGap: function(gap){
       this.gap = gap;
       // this forces a relayout of the element
       this.setDimension(1,1);
   },
   
   
   /**
    * @method
    * 
    * @private
    * @returns
    */
    getMinWidth:function()
    {
        var width=this.stroke*2+ Math.max(0,this.children.getSize()-1)*this.gap;
        this.children.each(function(i,e){
            width += e.figure.getMinWidth();
            
        });
        return width;
    },

    /**
     * @method
     * 
     * @private
     * @returns
     */
    getMinHeight:function()
    {
        var height=10;
        this.children.each(function(i,e){
            height = Math.max(height, e.figure.getMinHeight());
        });
        return height+this.stroke*2;
    },
    
    /**
     * @method
     * Set the new dimension of the Layout. Forces a relayout of children.
     * 
     * @returns
     */
    setDimension:function( w, h)
    {
        this._super(w,h);

        var diff = this.width-this.getMinWidth();
        if(diff>0){
            diff = (diff/this.children.getSize())|0;
            this.children.each(function(i,e){
                e.figure.setDimension(e.figure.getMinWidth()+diff,e.figure.getHeight());
            });
        }
        else{
            this.children.each(function(i,e){
                e.figure.setDimension(1,1);
            });
        }
     }

});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.layout.VerticalLayout
 * The VerticalLayout class arranges the layout elements in a vertical sequence, 
 * left to right, with optional gaps between the elements. 
 * 
 * During the execution of the setDimension() method, the minimum height of the container is calculated 
 * by accumulating the minimum sizes of the elements, including stroke, gaps and padding. 
 *     
 * See the example below with and without gap and border settings
 * 
 *     
 *     @example preview small frame
 *     
 *     // first container without any gap and a border of the parent
 *     // container
 *     var label1 =  new draw2d.shape.basic.Label("Label 1");
 *     var label2 =  new draw2d.shape.basic.Label("Label 2");
 *     var label3 =  new draw2d.shape.basic.Label("Label 3");
 *     
 *     var container1 = new draw2d.shape.layout.VerticalLayout();
 *     
 *     container1.addFigure(label1);
 *     container1.addFigure(label2);
 *     container1.addFigure(label3);
 *     container1.setGap(10);
 *     container1.setStroke(2);
 *     canvas.addFigure(container1,50,10);
 *     
 *     // second container without any gab or border
 *     //
 *     var label11 =  new draw2d.shape.basic.Label("Label 1");
 *     var label12 =  new draw2d.shape.basic.Label("Label 2");
 *     var label13 =  new draw2d.shape.basic.Label("Label 3");
 *     
 *     var container2 = new draw2d.shape.layout.VerticalLayout();
 *     
 *     container2.addFigure(label11);
 *     container2.addFigure(label12);
 *     container2.addFigure(label13);
 *     
 *     canvas.addFigure(container2,150,10);
 *     
 * @author Andreas Herz
 * @extends draw2d.shape.layout.Layout
 */
draw2d.shape.layout.VerticalLayout= draw2d.shape.layout.Layout.extend({

	NAME : "draw2d.shape.layout.VerticalLayout",

    /**
     * @constructor
     * Constructs a VerticalLayout 
     * 
     */
    init : function()
    {
        this._super();
        
        // some layout parameter
        //
        this.gap = 0;

        // "this" shortcut to avoid $.proxy
        var _this = this;
        
        this.locator = {
            relocate:function(index, target)
            {
                var stroke = _this.getStroke()/2;
                var yPos =stroke; // respect the border of the shape
                
                for (var i=0;i<index;i++){
                    var child = _this.children.get(i).figure;
                    yPos=yPos+child.getHeight()+_this.gap;
                }
                
                target.setPosition(stroke,yPos);
             }
        };
        
        this.setDimension(10,10);
    },
    

    /**
     * @method
     * Set the gap width between child components within this layout. 
     * This will only affect the space between components, not the space around all the components in the layout.
     * 
     * @param {Number} gap The space, in pixels, between items.
     */
    setGap: function(gap){
        this.gap = gap;
        // this forces a relayout of the element
        this.setDimension(1,1);
    },
    
    getMinWidth:function()
    {
        var width=10;
        this.children.each(function(i,e){
            width = Math.max(width, e.figure.getMinWidth());
        });
        return width+this.stroke;
    },
    
    getMinHeight:function()
    {
        var height=+this.stroke+ Math.max(0,this.children.getSize()-1)*this.gap;
        this.children.each(function(i,e){
            height += e.figure.getMinHeight();
        });
        
        return height;
    },
    
    /**
     * @method
     *
     **/
    setDimension:function( w, h)
    {
        this._super(w,h);

        var width=this.width-this.stroke;
        this.children.each(function(i,e){
            e.figure.setDimension(width,e.figure.getHeight());
        });
    }
   

});




/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Icon
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.VectorFigure
 */
draw2d.shape.icon.Icon = draw2d.SetFigure.extend({
    NAME : "draw2d.shape.icon.Icon",

    /**
     * 
     * @constructor
     * Creates a new figure element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
      this.setBackgroundColor("#333333");
      this.keepAspectRatio = true;
    },

    /**
     * @method
     * propagate all attributes like color, stroke,... to the shape element
     **/
    repaint : function(attributes)
    {
        if(this.repaintBlocked===true || this.shape===null){
            return;
        }

        if (typeof attributes === "undefined") {
            attributes = {};
        }
        
        // redirect the bgColor to the inner set and not to the outer container
        //
        attributes.fill="none";
        if(this.svgNodes!==null) {
            this.svgNodes.attr({fill: this.bgColor.hash(), stroke:"none"});
        }
        
        this._super(attributes);
    },

    applyTransformation:function(){
        if (this.isResizeable()===true) {
            this.svgNodes.transform("S"+this.scaleX+","+this.scaleY+","+this.getAbsoluteX()+","+this.getAbsoluteY()+ "t"+ (this.getAbsoluteX()-this.offsetX) + "," + (this.getAbsoluteY()-this.offsetY));
        }
        else {
            this.svgNodes.transform("T" + (this.getAbsoluteX()-this.offsetX) + "," + (this.getAbsoluteY()-this.offsetY));
        }
    },
    
    /**
     * @private
     */
    createShapeElement : function()
    {
    	var shape = this._super();
    	
        var bb = this.svgNodes.getBBox();

        this.offsetX = bb.x;
        this.offsetY = bb.y;
       
        return shape;
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Thunder

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Thunder();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Thunder = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Thunder",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.371,7.306c-0.092-3.924-3.301-7.077-7.248-7.079c-2.638,0.001-4.942,1.412-6.208,3.517c-0.595-0.327-1.28-0.517-2.01-0.517C7.626,3.229,5.772,5.033,5.689,7.293c-2.393,0.786-4.125,3.025-4.127,5.686c0,3.312,2.687,6,6,6v-0.002h5.271l-2.166,3.398l1.977-0.411L10,30.875l9.138-10.102L17,21l2.167-2.023h4.269c3.312,0,6-2.688,6-6C29.434,10.34,27.732,8.11,25.371,7.306zM23.436,16.979H7.561c-2.209-0.006-3.997-1.792-4.001-4.001c-0.002-1.982,1.45-3.618,3.35-3.931c0.265-0.043,0.502-0.191,0.657-0.414C7.722,8.41,7.779,8.136,7.73,7.87C7.702,7.722,7.685,7.582,7.685,7.446C7.689,6.221,8.68,5.23,9.905,5.228c0.647,0,1.217,0.278,1.633,0.731c0.233,0.257,0.587,0.375,0.927,0.309c0.342-0.066,0.626-0.307,0.748-0.63c0.749-1.992,2.662-3.412,4.911-3.41c2.899,0.004,5.244,2.35,5.251,5.249c0,0.161-0.009,0.326-0.027,0.497c-0.049,0.517,0.305,0.984,0.815,1.079c1.86,0.344,3.274,1.966,3.271,3.923C27.43,15.186,25.645,16.973,23.436,16.979z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Snow

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Snow();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Snow = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Snow",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.372,6.912c-0.093-3.925-3.302-7.078-7.248-7.08c-2.638,0.002-4.942,1.412-6.208,3.518c-0.595-0.327-1.28-0.518-2.01-0.518C7.627,2.834,5.773,4.639,5.69,6.898c-2.393,0.786-4.125,3.025-4.127,5.686c0,3.312,2.687,6,6,6v-0.002h15.875c3.312,0,6-2.688,6-6C29.434,9.944,27.732,7.715,25.372,6.912zM23.436,16.584H7.562c-2.209-0.006-3.997-1.793-4.001-4c-0.002-1.983,1.45-3.619,3.35-3.933c0.265-0.043,0.502-0.19,0.657-0.414C7.723,8.015,7.78,7.74,7.731,7.475C7.703,7.326,7.686,7.187,7.686,7.051c0.004-1.225,0.995-2.217,2.22-2.219c0.647,0,1.217,0.278,1.633,0.731c0.233,0.257,0.587,0.375,0.927,0.31c0.342-0.066,0.626-0.308,0.748-0.631c0.749-1.992,2.662-3.412,4.911-3.41c2.898,0.004,5.244,2.351,5.251,5.25c0,0.16-0.009,0.325-0.026,0.496c-0.05,0.518,0.305,0.984,0.814,1.079c1.859,0.345,3.273,1.966,3.271,3.923C27.43,14.791,25.645,16.578,23.436,16.584zM16.667,24.09l1.119-1.119c0.389-0.391,0.389-1.025,0-1.416c-0.392-0.391-1.025-0.391-1.415,0l-1.119,1.119l-1.119-1.119c-0.391-0.391-1.025-0.391-1.415,0c-0.391,0.391-0.391,1.025,0,1.416l1.118,1.117l-1.12,1.121c-0.389,0.393-0.389,1.021,0,1.414c0.195,0.188,0.451,0.293,0.707,0.293c0.256,0,0.512-0.104,0.708-0.293l1.12-1.119l1.12,1.119c0.195,0.188,0.451,0.293,0.708,0.293c0.256,0,0.512-0.104,0.707-0.293c0.391-0.396,0.391-1.021,0-1.414L16.667,24.09zM25.119,21.817c-0.393-0.392-1.025-0.392-1.415,0l-1.12,1.121l-1.12-1.121c-0.391-0.392-1.022-0.392-1.414,0c-0.39,0.392-0.39,1.022,0,1.416l1.119,1.119l-1.119,1.119c-0.39,0.391-0.39,1.022,0,1.413c0.195,0.195,0.451,0.294,0.707,0.294c0.257,0,0.513-0.099,0.707-0.294l1.12-1.118l1.12,1.118c0.194,0.195,0.45,0.294,0.707,0.294c0.256,0,0.513-0.099,0.708-0.294c0.389-0.391,0.389-1.022,0-1.413l-1.12-1.119l1.12-1.119C25.507,22.842,25.507,22.209,25.119,21.817zM9.334,23.953l1.119-1.119c0.389-0.394,0.389-1.021,0-1.414c-0.391-0.394-1.025-0.394-1.415,0l-1.119,1.119l-1.12-1.121c-0.391-0.39-1.023-0.39-1.415,0c-0.391,0.396-0.391,1.024,0,1.418l1.119,1.117l-1.12,1.118c-0.391,0.394-0.391,1.025,0,1.414c0.196,0.195,0.452,0.293,0.708,0.293c0.256,0,0.511-0.098,0.707-0.293l1.12-1.119l1.121,1.121c0.195,0.195,0.451,0.293,0.707,0.293s0.513-0.098,0.708-0.293c0.389-0.391,0.389-1.022,0-1.416L9.334,23.953z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Hail

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Hail();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Hail = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Hail",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.372,6.912c-0.093-3.925-3.302-7.078-7.248-7.08c-2.638,0.002-4.942,1.412-6.208,3.518c-0.595-0.327-1.28-0.518-2.01-0.518C7.627,2.834,5.773,4.639,5.69,6.898c-2.393,0.786-4.125,3.025-4.127,5.686c0,3.312,2.687,6,6,6v-0.002h15.875c3.312,0,6-2.688,6-6C29.434,9.944,27.732,7.715,25.372,6.912zM23.436,16.584H7.562c-2.209-0.006-3.997-1.793-4.001-4c-0.002-1.983,1.45-3.619,3.35-3.933c0.265-0.043,0.502-0.19,0.657-0.414C7.723,8.015,7.78,7.74,7.731,7.475C7.703,7.326,7.686,7.187,7.686,7.051c0.004-1.225,0.995-2.217,2.22-2.219c0.647,0,1.217,0.278,1.633,0.731c0.233,0.257,0.587,0.375,0.927,0.31c0.342-0.066,0.626-0.308,0.748-0.631c0.749-1.992,2.662-3.412,4.911-3.41c2.898,0.004,5.244,2.351,5.251,5.25c0,0.16-0.009,0.325-0.026,0.496c-0.05,0.518,0.305,0.984,0.814,1.079c1.859,0.345,3.273,1.966,3.271,3.923C27.43,14.791,25.645,16.578,23.436,16.584zM11.503,23.709c-0.784-0.002-1.418-0.636-1.418-1.416c0-0.785,0.634-1.416,1.418-1.418c0.78,0.002,1.413,0.633,1.416,1.418C12.917,23.073,12.284,23.707,11.503,23.709zM19.002,23.709c-0.783-0.002-1.418-0.636-1.418-1.416c0-0.785,0.635-1.416,1.418-1.418c0.779,0.002,1.414,0.633,1.414,1.418C20.417,23.073,19.784,23.707,19.002,23.709zM7.503,28.771c-0.783-0.002-1.417-0.637-1.417-1.418s0.634-1.414,1.417-1.416c0.78,0.002,1.415,0.635,1.415,1.416C8.917,28.135,8.284,28.77,7.503,28.771zM15.001,28.771c-0.782-0.002-1.417-0.637-1.417-1.418s0.634-1.414,1.417-1.416c0.78,0.002,1.413,0.635,1.415,1.416C16.415,28.135,15.784,28.77,15.001,28.771zM22.5,28.771c-0.782-0.002-1.416-0.634-1.416-1.416c0-0.785,0.634-1.418,1.416-1.42c0.781,0.002,1.414,0.635,1.418,1.42C23.915,28.138,23.282,28.77,22.5,28.771z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Rain

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Rain();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Rain = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Rain",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.371,7.306c-0.092-3.924-3.301-7.077-7.248-7.079c-2.638,0.001-4.942,1.412-6.208,3.517c-0.595-0.327-1.28-0.517-2.01-0.517C7.626,3.229,5.772,5.033,5.689,7.293c-2.393,0.786-4.125,3.025-4.127,5.686c0,3.312,2.687,6,6,6v-0.002h15.874c3.312,0,6-2.688,6-6C29.434,10.34,27.732,8.11,25.371,7.306zM23.436,16.979H7.561c-2.209-0.006-3.997-1.792-4.001-4.001c-0.002-1.982,1.45-3.618,3.35-3.931c0.265-0.043,0.502-0.191,0.657-0.414C7.722,8.41,7.779,8.136,7.73,7.87C7.702,7.722,7.685,7.582,7.685,7.446C7.689,6.221,8.68,5.23,9.905,5.228c0.647,0,1.217,0.278,1.633,0.731c0.233,0.257,0.587,0.375,0.927,0.309c0.342-0.066,0.626-0.307,0.748-0.63c0.749-1.992,2.662-3.412,4.911-3.41c2.899,0.004,5.244,2.35,5.251,5.249c0,0.161-0.009,0.326-0.027,0.497c-0.049,0.517,0.305,0.984,0.815,1.079c1.86,0.344,3.274,1.966,3.271,3.923C27.43,15.186,25.645,16.973,23.436,16.979zM9.029,26.682c0-1.115,0.021-5.425,0.021-5.432c0.002-0.409-0.247-0.779-0.628-0.932c-0.38-0.152-0.815-0.059-1.099,0.24c-0.006,0.008-1.037,1.098-2.081,2.342c-0.523,0.627-1.048,1.287-1.463,1.896c-0.399,0.648-0.753,1.066-0.811,1.885C2.971,28.355,4.324,29.711,6,29.714C7.672,29.71,9.029,28.354,9.029,26.682zM4.971,26.727c0.091-0.349,1.081-1.719,1.993-2.764c0.025-0.029,0.051-0.061,0.076-0.089c-0.005,1.124-0.01,2.294-0.01,2.808c0,0.567-0.461,1.028-1.029,1.03C5.447,27.71,4.997,27.273,4.971,26.727zM16.425,26.682c0-1.115,0.021-5.424,0.021-5.43c0.002-0.41-0.247-0.779-0.628-0.934c-0.381-0.152-0.814-0.058-1.1,0.242c-0.006,0.008-1.035,1.094-2.08,2.342c-0.522,0.623-1.047,1.285-1.463,1.894c-0.399,0.649-0.753,1.068-0.809,1.888c0,1.672,1.354,3.028,3.029,3.028C15.068,29.711,16.425,28.354,16.425,26.682zM12.365,26.729c0.092-0.349,1.081-1.72,1.993-2.765c0.025-0.03,0.05-0.06,0.075-0.089c-0.005,1.123-0.011,2.294-0.011,2.807c-0.002,0.568-0.461,1.027-1.028,1.029C12.84,27.709,12.392,27.273,12.365,26.729zM23.271,20.317c-0.38-0.153-0.816-0.06-1.099,0.24c-0.009,0.008-1.037,1.097-2.08,2.342c-0.523,0.625-1.049,1.285-1.462,1.896c-0.402,0.649-0.754,1.067-0.812,1.886c0,1.672,1.354,3.029,3.03,3.029c1.673,0,3.027-1.357,3.027-3.029c0-1.115,0.022-5.425,0.022-5.431C23.9,20.84,23.651,20.47,23.271,20.317zM21.879,26.681c-0.004,0.568-0.463,1.027-1.031,1.029c-0.553-0.002-1.002-0.438-1.028-0.982c0.092-0.349,1.081-1.72,1.993-2.765c0.025-0.028,0.05-0.059,0.074-0.088C21.883,24.998,21.879,26.167,21.879,26.681z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Cloudy

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Cloudy();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Cloudy = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Cloudy",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M14.378,6.781c0.41,0.988,1.938,0.346,1.524-0.648C15.708,5.667,15.515,5.2,15.32,4.734c-0.289-0.695-0.875-3.233-2.042-2.747c-1.03,0.433-0.128,1.846,0.142,2.494C13.739,5.248,14.059,6.015,14.378,6.781M20.8,7.223c1.094,0.453,1.538-1.551,1.813-2.216c0.281-0.677,1.478-2.565,0.357-3.029c-1.092-0.453-1.537,1.548-1.813,2.216C20.876,4.872,19.68,6.757,20.8,7.223M18.137,6.692c1.183,0,0.829-2.019,0.829-2.742c0-0.732,0.383-2.935-0.829-2.935c-1.183,0-0.828,2.019-0.828,2.742C17.309,4.49,16.926,6.692,18.137,6.692M23.058,8.729c0.852,0.85,2.142-0.972,2.659-1.49c0.512-0.513,2.187-1.687,1.352-2.524c-0.834-0.836-2.013,0.843-2.522,1.353C24.028,6.585,22.198,7.874,23.058,8.729M24.565,10.986c0.448,1.091,2.183-0.01,2.849-0.286c0.676-0.28,2.858-0.771,2.394-1.89c-0.455-1.091-2.181,0.008-2.849,0.285C26.281,9.377,24.102,9.866,24.565,10.986M12.036,8.742c0.752,0.75,1.932-0.415,1.17-1.173c-0.253-0.347-0.646-0.645-0.949-0.946c-0.541-0.539-2.162-2.799-3.068-1.889c-0.79,0.791,0.586,1.755,1.083,2.25C10.859,7.57,11.447,8.156,12.036,8.742M29.365,17.397c-0.768-0.317-1.534-0.635-2.302-0.952c-0.646-0.268-2.07-1.169-2.495-0.135c-0.481,1.168,2.054,1.747,2.751,2.035c0.455,0.188,0.911,0.377,1.367,0.565C29.7,19.331,30.379,17.816,29.365,17.397M29.942,12.817c-0.83,0-1.66,0-2.49,0c-0.701,0-2.357-0.288-2.355,0.83c0,1.262,2.567,0.827,3.319,0.827c0.493,0,0.986,0,1.479-0.001C30.99,14.473,31.043,12.815,29.942,12.817M24.234,18.568c-0.673-0.673-1.773,0.189-1.281,1.007c-0.295-0.264-0.614-0.499-0.961-0.69c3.894-2.866,3.328-9.006-1.021-11.107c-2.024-0.978-4.481-0.828-6.368,0.394c-0.871,0.564-1.603,1.336-2.119,2.236c-0.262,0.456-0.468,0.943-0.612,1.449c-0.074,0.258-0.131,0.521-0.172,0.786c-0.083,0.534-0.109,0.553-0.553,0.871c-0.182-0.957-1.64-0.675-2.326-0.674c-0.815,0.001-1.963-0.217-2.752,0.046c-0.867,0.289-0.652,1.615,0.263,1.613c0.324,0.052,0.701-0.001,1.028-0.001c0.904-0.001,1.809-0.002,2.713-0.003c-0.308,0.352-0.496,0.969-0.94,0.77c-0.467-0.209-0.978-0.319-1.49-0.319c-0.951,0-1.877,0.375-2.561,1.036c-0.681,0.658-1.088,1.569-1.123,2.516c-0.944,0.31-1.791,0.891-2.421,1.658c-2.756,3.354-0.265,8.554,4.058,8.554v-0.002c3.597,0,7.194,0,10.792,0c1.341,0,2.843,0.167,4.168-0.113c3.652-0.772,5.361-5.21,3.133-8.229c0.548,0.547,1.096,1.094,1.644,1.641c0.183,0.183,0.364,0.424,0.575,0.574c0.552,0.552,1.524,0.066,1.403-0.713c-0.097-0.622-1.042-1.267-1.448-1.673C25.319,19.652,24.776,19.11,24.234,18.568M18.137,8.787c4.559,0.009,6.576,5.979,2.912,8.734c-0.637-3.505-4.161-5.824-7.629-5.03C13.943,10.367,15.852,8.792,18.137,8.787M22.895,24.08c-0.633,3.346-4.149,2.879-6.68,2.879c-3.017,0-6.033,0-9.049,0c-0.767,0-1.62,0.084-2.373-0.095c-2.274-0.538-3.416-3.242-2.172-5.235c0.678-1.087,1.568-1.19,2.626-1.67c0.604-0.273,0.456-0.807,0.456-1.331c0.002-0.597,0.284-1.169,0.756-1.533c0.787-0.608,1.943-0.497,2.611,0.234c1.098,1.205,1.96-1.346,2.507-1.893c2.025-2.025,5.475-1.708,7.068,0.684c0.344,0.516,0.581,1.102,0.693,1.712c0.097,0.529-0.115,1.341,0.188,1.796c0.291,0.47,0.943,0.463,1.397,0.68c0.508,0.23,0.963,0.591,1.304,1.034C22.834,22.125,23.064,23.107,22.895,24.08M6.906,9.917c0.881,0.364,1.763,0.727,2.644,1.091c0.353,0.146,0.707,0.292,1.06,0.437c0.997,0.412,1.637-1.119,0.642-1.526C10.47,9.441,9.456,9.177,8.609,8.828c-0.354-0.146-0.707-0.292-1.06-0.437C6.554,7.98,5.912,9.505,6.906,9.917");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Sun

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Sun();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Sun = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Sun",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M15.502,7.504c-4.35,0-7.873,3.523-7.873,7.873c0,4.347,3.523,7.872,7.873,7.872c4.346,0,7.871-3.525,7.871-7.872C23.374,11.027,19.85,7.504,15.502,7.504zM15.502,21.25c-3.244-0.008-5.866-2.63-5.874-5.872c0.007-3.243,2.63-5.866,5.874-5.874c3.242,0.008,5.864,2.631,5.871,5.874C21.366,18.62,18.744,21.242,15.502,21.25zM15.502,6.977c0.553,0,1-0.448,1-1.001V1.125c-0.002-0.553-0.448-1-1-1c-0.553,0-1.001,0.449-1,1.002v4.85C14.502,6.528,14.949,6.977,15.502,6.977zM18.715,7.615c0.125,0.053,0.255,0.076,0.382,0.077c0.394,0,0.765-0.233,0.925-0.618l1.856-4.483c0.21-0.511-0.031-1.095-0.541-1.306c-0.511-0.211-1.096,0.031-1.308,0.541L18.174,6.31C17.963,6.82,18.205,7.405,18.715,7.615zM21.44,9.436c0.195,0.194,0.451,0.293,0.707,0.293s0.512-0.098,0.707-0.293l3.43-3.433c0.391-0.39,0.39-1.023,0-1.415c-0.392-0.39-1.025-0.39-1.415,0.002L21.44,8.021C21.049,8.412,21.049,9.045,21.44,9.436zM23.263,12.16c0.158,0.385,0.531,0.617,0.923,0.617c0.127,0,0.257-0.025,0.383-0.078l4.48-1.857c0.511-0.211,0.753-0.797,0.541-1.307s-0.796-0.752-1.307-0.54l-4.481,1.857C23.292,11.064,23.051,11.65,23.263,12.16zM29.752,14.371l-4.851,0.001c-0.552,0-1,0.448-0.998,1.001c0,0.553,0.447,0.999,0.998,0.999l4.852-0.002c0.553,0,0.999-0.449,0.999-1C30.752,14.817,30.304,14.369,29.752,14.371zM29.054,19.899l-4.482-1.854c-0.512-0.212-1.097,0.03-1.307,0.541c-0.211,0.511,0.031,1.096,0.541,1.308l4.482,1.854c0.126,0.051,0.256,0.075,0.383,0.075c0.393,0,0.765-0.232,0.925-0.617C29.806,20.695,29.563,20.109,29.054,19.899zM22.86,21.312c-0.391-0.391-1.023-0.391-1.414,0.001c-0.391,0.39-0.39,1.022,0,1.413l3.434,3.429c0.195,0.195,0.45,0.293,0.706,0.293s0.513-0.098,0.708-0.293c0.391-0.392,0.389-1.025,0-1.415L22.86,21.312zM20.029,23.675c-0.211-0.511-0.796-0.752-1.307-0.541c-0.51,0.212-0.752,0.797-0.54,1.308l1.86,4.48c0.159,0.385,0.531,0.617,0.925,0.617c0.128,0,0.258-0.024,0.383-0.076c0.511-0.211,0.752-0.797,0.54-1.309L20.029,23.675zM15.512,23.778c-0.553,0-1,0.448-1,1l0.004,4.851c0,0.553,0.449,0.999,1,0.999c0.553,0,1-0.448,0.998-1l-0.003-4.852C16.511,24.226,16.062,23.777,15.512,23.778zM12.296,23.142c-0.51-0.21-1.094,0.031-1.306,0.543l-1.852,4.483c-0.21,0.511,0.033,1.096,0.543,1.307c0.125,0.052,0.254,0.076,0.382,0.076c0.392,0,0.765-0.234,0.924-0.619l1.853-4.485C13.051,23.937,12.807,23.353,12.296,23.142zM9.57,21.325c-0.392-0.391-1.025-0.389-1.415,0.002L4.729,24.76c-0.391,0.392-0.389,1.023,0.002,1.415c0.195,0.194,0.45,0.292,0.706,0.292c0.257,0,0.513-0.098,0.708-0.293l3.427-3.434C9.961,22.349,9.961,21.716,9.57,21.325zM7.746,18.604c-0.213-0.509-0.797-0.751-1.307-0.54L1.96,19.925c-0.511,0.212-0.752,0.798-0.54,1.308c0.16,0.385,0.531,0.616,0.924,0.616c0.127,0,0.258-0.024,0.383-0.076l4.479-1.861C7.715,19.698,7.957,19.113,7.746,18.604zM7.1,15.392c0-0.553-0.447-0.999-1-0.999l-4.851,0.006c-0.553,0-1.001,0.448-0.999,1.001c0.001,0.551,0.449,1,1,0.998l4.852-0.006C6.654,16.392,7.102,15.942,7.1,15.392zM1.944,10.869l4.485,1.85c0.125,0.053,0.254,0.076,0.381,0.076c0.393,0,0.766-0.232,0.925-0.618c0.212-0.511-0.032-1.097-0.544-1.306L2.708,9.021c-0.511-0.21-1.095,0.032-1.306,0.542C1.19,10.074,1.435,10.657,1.944,10.869zM8.137,9.451c0.195,0.193,0.449,0.291,0.705,0.291s0.513-0.098,0.709-0.295c0.391-0.389,0.389-1.023-0.004-1.414L6.113,4.609C5.723,4.219,5.088,4.221,4.699,4.612c-0.391,0.39-0.389,1.024,0.002,1.414L8.137,9.451zM10.964,7.084c0.16,0.384,0.532,0.615,0.923,0.615c0.128,0,0.258-0.025,0.384-0.077c0.51-0.212,0.753-0.798,0.54-1.307l-1.864-4.479c-0.212-0.51-0.798-0.751-1.308-0.539C9.129,1.51,8.888,2.096,9.1,2.605L10.964,7.084z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Undo

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Undo();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Undo = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Undo",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M12.981,9.073V6.817l-12.106,6.99l12.106,6.99v-2.422c3.285-0.002,9.052,0.28,9.052,2.269c0,2.78-6.023,4.263-6.023,4.263v2.132c0,0,13.53,0.463,13.53-9.823C29.54,9.134,17.952,8.831,12.981,9.073z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Detour

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Detour();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Detour = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Detour",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M29.342,15.5l-7.556-4.363v2.614H18.75c-1.441-0.004-2.423,1.002-2.875,1.784c-0.735,1.222-1.056,2.561-1.441,3.522c-0.135,0.361-0.278,0.655-0.376,0.817c-1.626,0-0.998,0-2.768,0c-0.213-0.398-0.571-1.557-0.923-2.692c-0.237-0.676-0.5-1.381-1.013-2.071C8.878,14.43,7.89,13.726,6.75,13.75H2.812v3.499c0,0,0.358,0,1.031,0h2.741c0.008,0.013,0.018,0.028,0.029,0.046c0.291,0.401,0.634,1.663,1.031,2.888c0.218,0.623,0.455,1.262,0.92,1.897c0.417,0.614,1.319,1.293,2.383,1.293H11c2.25,0,1.249,0,3.374,0c0.696,0.01,1.371-0.286,1.809-0.657c1.439-1.338,1.608-2.886,2.13-4.127c0.218-0.608,0.453-1.115,0.605-1.314c0.006-0.01,0.012-0.018,0.018-0.025h2.85v2.614L29.342,15.5zM10.173,14.539c0.568,0.76,0.874,1.559,1.137,2.311c0.04,0.128,0.082,0.264,0.125,0.399h2.58c0.246-0.697,0.553-1.479,1.005-2.228c0.252-0.438,0.621-0.887,1.08-1.272H9.43C9.735,14.003,9.99,14.277,10.173,14.539z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Merge

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Merge();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Merge = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Merge",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M29.342,15.5l-7.556-4.363v2.613h-1.411c-0.788-0.01-1.331-0.241-2.019-0.743c-1.021-0.745-2.094-2.181-3.551-3.568C13.367,8.06,11.291,6.73,8.5,6.749H2.812v3.5H8.5c2.231,0.012,3.441,1.185,5.07,2.934c0.697,0.753,1.428,1.58,2.324,2.323c-1.396,1.165-2.412,2.516-3.484,3.501c-1.183,1.081-2.202,1.723-3.912,1.741H2.813v3.5h5.716c3.752,0.001,6.035-2.319,7.619-4.066c0.817-0.895,1.537-1.691,2.209-2.191c0.686-0.502,1.23-0.732,2.017-0.742h1.412v2.614L29.342,15.5z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Split

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Split();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Split = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Split",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.786,20.698c-1.792-0.237-2.912-1.331-4.358-2.886c-0.697-0.751-1.428-1.577-2.324-2.319c1.396-1.165,2.411-2.519,3.483-3.503c1.01-0.92,1.901-1.519,3.199-1.688v2.574l7.556-4.363L21.786,4.15v2.652c-3.34,0.266-5.45,2.378-6.934,4.013c-0.819,0.896-1.537,1.692-2.212,2.192c-0.685,0.501-1.227,0.731-2.013,0.742c-0.001,0-0.002,0-0.003,0H2.812v3.5h0.001v0.001c0,0,0.046-0.001,0.136-0.001h7.677c0.786,0.011,1.33,0.241,2.017,0.743c1.021,0.743,2.095,2.181,3.552,3.568c1.312,1.258,3.162,2.46,5.592,2.649v2.664l7.556-4.36l-7.556-4.361V20.698z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Fork

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Fork();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Fork = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Fork",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M13.741,10.249h8.045v2.627l7.556-4.363l-7.556-4.363v2.598H9.826C11.369,7.612,12.616,8.922,13.741,10.249zM21.786,20.654c-0.618-0.195-1.407-0.703-2.291-1.587c-1.79-1.756-3.712-4.675-5.731-7.227c-2.049-2.486-4.159-4.972-7.451-5.091h-3.5v3.5h3.5c0.656-0.027,1.683,0.486,2.879,1.683c1.788,1.753,3.712,4.674,5.731,7.226c1.921,2.331,3.907,4.639,6.863,5.016v2.702l7.556-4.362l-7.556-4.362V20.654z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ForkAlt

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ForkAlt();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ForkAlt = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ForkAlt",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.786,12.873l7.556-4.361l-7.556-4.362v2.701c-2.929,0.374-4.905,2.64-6.809,4.952c0.545,0.703,1.08,1.418,1.604,2.127c0.192,0.26,0.383,0.514,0.573,0.77c0.802-1.043,1.584-1.999,2.341-2.74c0.884-0.885,1.673-1.393,2.291-1.588V12.873zM17.661,17.006c-1.893-2.371-3.815-5.354-6.009-7.537c-1.461-1.428-3.155-2.664-5.34-2.693h-3.5v3.5h3.5c0.971-0.119,2.845,1.333,4.712,3.771c1.895,2.371,3.815,5.354,6.011,7.537c1.326,1.297,2.847,2.426,4.751,2.645v2.646l7.556-4.363l-7.556-4.362v2.535C20.746,20.346,19.205,19.022,17.661,17.006z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Exchange

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Exchange();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Exchange = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Exchange",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.786,12.876l7.556-4.363l-7.556-4.363v2.598H2.813v3.5h18.973V12.876zM10.368,18.124l-7.556,4.362l7.556,4.362V24.25h18.974v-3.501H10.368V18.124z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Shuffle

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Shuffle();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Shuffle = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Shuffle",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.786,20.654c-0.618-0.195-1.407-0.703-2.291-1.587c-0.757-0.742-1.539-1.698-2.34-2.741c-0.191,0.256-0.382,0.51-0.574,0.77c-0.524,0.709-1.059,1.424-1.604,2.127c1.904,2.31,3.88,4.578,6.809,4.952v2.701l7.556-4.362l-7.556-4.362V20.654zM9.192,11.933c0.756,0.741,1.538,1.697,2.339,2.739c0.195-0.262,0.39-0.521,0.587-0.788c0.52-0.703,1.051-1.412,1.592-2.11c-2.032-2.463-4.133-4.907-7.396-5.025h-3.5v3.5h3.5C6.969,10.223,7.996,10.735,9.192,11.933zM21.786,10.341v2.535l7.556-4.363l-7.556-4.363v2.647c-1.904,0.219-3.425,1.348-4.751,2.644c-2.196,2.183-4.116,5.167-6.011,7.538c-1.867,2.438-3.741,3.888-4.712,3.771h-3.5v3.5h3.5c2.185-0.029,3.879-1.266,5.34-2.693c2.194-2.184,4.116-5.167,6.009-7.538C19.205,12.003,20.746,10.679,21.786,10.341z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Refresh

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Refresh();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Refresh = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Refresh",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.083,15.5c-0.009,4.739-3.844,8.574-8.583,8.583c-4.741-0.009-8.577-3.844-8.585-8.583c0.008-4.741,3.844-8.577,8.585-8.585c1.913,0,3.665,0.629,5.09,1.686l-1.782,1.783l8.429,2.256l-2.26-8.427l-1.89,1.89c-2.072-1.677-4.717-2.688-7.587-2.688C8.826,3.418,3.418,8.826,3.416,15.5C3.418,22.175,8.826,27.583,15.5,27.583S27.583,22.175,27.583,15.5H24.083z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Ccw

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Ccw();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Ccw = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Ccw",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.249,15.499c-0.009,4.832-3.918,8.741-8.75,8.75c-2.515,0-4.768-1.064-6.365-2.763l2.068-1.442l-7.901-3.703l0.744,8.694l2.193-1.529c2.244,2.594,5.562,4.242,9.26,4.242c6.767,0,12.249-5.482,12.249-12.249H24.249zM15.499,6.75c2.516,0,4.769,1.065,6.367,2.764l-2.068,1.443l7.901,3.701l-0.746-8.693l-2.192,1.529c-2.245-2.594-5.562-4.245-9.262-4.245C8.734,3.25,3.25,8.734,3.249,15.499H6.75C6.758,10.668,10.668,6.758,15.499,6.75z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Acw

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Acw();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Acw = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Acw",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M19.275,3.849l1.695,8.56l1.875-1.642c2.311,3.59,1.72,8.415-1.584,11.317c-2.24,1.96-5.186,2.57-7.875,1.908l-0.84,3.396c3.75,0.931,7.891,0.066,11.02-2.672c4.768-4.173,5.521-11.219,1.94-16.279l2.028-1.775L19.275,3.849zM8.154,20.232c-2.312-3.589-1.721-8.416,1.582-11.317c2.239-1.959,5.186-2.572,7.875-1.909l0.842-3.398c-3.752-0.93-7.893-0.067-11.022,2.672c-4.765,4.174-5.519,11.223-1.939,16.283l-2.026,1.772l8.26,2.812l-1.693-8.559L8.154,20.232z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Contract

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Contract();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Contract = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Contract",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.083,18.895l-8.428-2.259l2.258,8.428l1.838-1.837l7.053,7.053l2.476-2.476l-7.053-7.053L25.083,18.895zM5.542,11.731l8.428,2.258l-2.258-8.428L9.874,7.398L3.196,0.72L0.72,3.196l6.678,6.678L5.542,11.731zM7.589,20.935l-6.87,6.869l2.476,2.476l6.869-6.869l1.858,1.857l2.258-8.428l-8.428,2.258L7.589,20.935zM23.412,10.064l6.867-6.87l-2.476-2.476l-6.868,6.869l-1.856-1.856l-2.258,8.428l8.428-2.259L23.412,10.064z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Expand

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Expand();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Expand = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Expand",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.545,23.328,17.918,15.623,25.534,8.007,27.391,9.864,29.649,1.436,21.222,3.694,23.058,5.53,15.455,13.134,7.942,5.543,9.809,3.696,1.393,1.394,3.608,9.833,5.456,8.005,12.98,15.608,5.465,23.123,3.609,21.268,1.351,29.695,9.779,27.438,7.941,25.6,15.443,18.098,23.057,25.791,21.19,27.638,29.606,29.939,27.393,21.5z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Stop

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Stop();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Stop = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Stop",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M5.5,5.5h20v20h-20z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.End

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.End();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.End = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.End",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.167,5.5,21.167,13.681,6.684,5.318,6.684,25.682,21.167,17.318,21.167,25.5,25.5,25.5,25.5,5.5z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Start

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Start();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Start = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Start",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.316,5.318,9.833,13.682,9.833,5.5,5.5,5.5,5.5,25.5,9.833,25.5,9.833,17.318,24.316,25.682z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Ff

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Ff();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Ff = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Ff",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.5,15.5,15.2,9.552,15.2,15.153,5.5,9.552,5.5,21.447,15.2,15.847,15.2,21.447z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Rw

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Rw();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Rw = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Rw",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M5.5,15.499,15.8,21.447,15.8,15.846,25.5,21.447,25.5,9.552,15.8,15.152,15.8,9.552z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ArrowRight

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ArrowRight();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ArrowRight = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ArrowRight",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M6.684,25.682L24.316,15.5L6.684,5.318V25.682z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ArrowLeft

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ArrowLeft();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ArrowLeft = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ArrowLeft",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.316,5.318L6.684,15.5l17.632,10.182V5.318L24.316,5.318z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ArrowUp

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ArrowUp();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ArrowUp = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ArrowUp",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.682,24.316L15.5,6.684L5.318,24.316H25.682z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ArrowDown

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ArrowDown();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ArrowDown = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ArrowDown",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M5.318,6.684L15.5,24.316L25.682,6.684H5.318z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ArrowLeft2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ArrowLeft2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ArrowLeft2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ArrowLeft2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.871,9.814 15.684,16.001 21.871,22.188 18.335,25.725 8.612,16.001 18.335,6.276z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ArrowRight2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ArrowRight2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ArrowRight2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ArrowRight2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M10.129,22.186 16.316,15.999 10.129,9.812 13.665,6.276 23.389,15.999 13.665,25.725z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Smile2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Smile2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Smile2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Smile2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,1.466C7.973,1.466,1.466,7.973,1.466,16c0,8.027,6.507,14.534,14.534,14.534c8.027,0,14.534-6.507,14.534-14.534C30.534,7.973,24.027,1.466,16,1.466zM16,29.534C8.539,29.534,2.466,23.462,2.466,16C2.466,8.539,8.539,2.466,16,2.466c7.462,0,13.535,6.072,13.535,13.533C29.534,23.462,23.462,29.534,16,29.534zM11.104,14c0.932,0,1.688-1.483,1.688-3.312s-0.755-3.312-1.688-3.312s-1.688,1.483-1.688,3.312S10.172,14,11.104,14zM20.729,14c0.934,0,1.688-1.483,1.688-3.312s-0.756-3.312-1.688-3.312c-0.932,0-1.688,1.483-1.688,3.312S19.798,14,20.729,14zM8.143,21.189C10.458,24.243,13.148,26,16.021,26c2.969,0,5.745-1.868,8.11-5.109c-2.515,1.754-5.292,2.734-8.215,2.734C13.164,23.625,10.54,22.756,8.143,21.189z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Smile

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Smile();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Smile = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Smile",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,1.466C7.973,1.466,1.466,7.973,1.466,16c0,8.027,6.507,14.534,14.534,14.534c8.027,0,14.534-6.507,14.534-14.534C30.534,7.973,24.027,1.466,16,1.466zM20.729,7.375c0.934,0,1.688,1.483,1.688,3.312S21.661,14,20.729,14c-0.932,0-1.688-1.483-1.688-3.312S19.798,7.375,20.729,7.375zM11.104,7.375c0.932,0,1.688,1.483,1.688,3.312S12.037,14,11.104,14s-1.688-1.483-1.688-3.312S10.172,7.375,11.104,7.375zM16.021,26c-2.873,0-5.563-1.757-7.879-4.811c2.397,1.564,5.021,2.436,7.774,2.436c2.923,0,5.701-0.98,8.215-2.734C21.766,24.132,18.99,26,16.021,26z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Alarm

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Alarm();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Alarm = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Alarm",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M8.179,20.115c-0.478,0.277-0.642,0.889-0.365,1.366c0.275,0.479,0.889,0.642,1.365,0.366c0.479-0.275,0.643-0.888,0.367-1.367C9.27,20.004,8.658,19.84,8.179,20.115zM9.18,12.239c-0.479-0.276-1.09-0.112-1.366,0.366s-0.111,1.09,0.365,1.366c0.479,0.276,1.09,0.113,1.367-0.366C9.821,13.126,9.657,12.516,9.18,12.239zM8.625,17.043c-0.001-0.552-0.448-0.999-1.001-1c-0.553,0-1,0.448-1,1c0,0.553,0.449,1,1,1C8.176,18.043,8.624,17.596,8.625,17.043zM16.312,3.957V3.031h1c0.275,0,0.5-0.225,0.5-0.5v-0.5c0-0.275-0.225-0.5-0.5-0.5h-3.625c-0.275,0-0.5,0.225-0.5,0.5v0.5c0,0.275,0.225,0.5,0.5,0.5h1v0.926C7.819,4.381,2.376,10.068,2.374,17.042C2.376,24.291,8.251,30.166,15.5,30.169c7.249-0.003,13.124-5.878,13.125-13.127C28.624,10.067,23.181,4.38,16.312,3.957zM15.5,27.166C9.909,27.157,5.385,22.633,5.375,17.042C5.385,11.451,9.909,6.927,15.5,6.917c5.59,0.01,10.115,4.535,10.124,10.125C25.615,22.633,21.091,27.157,15.5,27.166zM12.062,22.998c-0.478-0.275-1.089-0.111-1.366,0.367c-0.275,0.479-0.111,1.09,0.366,1.365c0.478,0.277,1.091,0.111,1.365-0.365C12.704,23.887,12.54,23.275,12.062,22.998zM12.062,11.088c0.479-0.276,0.642-0.888,0.366-1.366c-0.276-0.478-0.888-0.642-1.366-0.366s-0.642,0.888-0.366,1.366C10.973,11.2,11.584,11.364,12.062,11.088zM22.822,13.971c0.478-0.275,0.643-0.888,0.366-1.366c-0.275-0.478-0.89-0.642-1.366-0.366c-0.479,0.278-0.642,0.89-0.366,1.367C21.732,14.083,22.344,14.247,22.822,13.971zM15.501,23.92c-0.552,0-1,0.447-1,1c0,0.552,0.448,1,1,1s1-0.448,1-1C16.501,24.367,16.053,23.92,15.501,23.92zM19.938,9.355c-0.477-0.276-1.091-0.111-1.365,0.366c-0.275,0.48-0.111,1.091,0.366,1.367s1.089,0.112,1.366-0.366C20.581,10.245,20.418,9.632,19.938,9.355zM23.378,16.042c-0.554,0.002-1.001,0.45-1.001,1c0.001,0.552,0.448,1,1.001,1c0.551,0,1-0.447,1-1C24.378,16.492,23.929,16.042,23.378,16.042zM22.823,20.115c-0.48-0.275-1.092-0.111-1.367,0.365c-0.275,0.479-0.112,1.091,0.367,1.367c0.477,0.275,1.089,0.112,1.365-0.366C23.464,21.004,23.3,20.391,22.823,20.115zM15.501,8.167c-0.552,0-1,0.448-1,1l-0.466,7.343l-3.004,1.96c-0.478,0.277-0.642,0.889-0.365,1.366c0.275,0.479,0.889,0.642,1.365,0.366l3.305-1.676c0.055,0.006,0.109,0.017,0.166,0.017c0.828,0,1.5-0.672,1.5-1.5l-0.5-7.876C16.501,8.614,16.053,8.167,15.501,8.167zM18.939,22.998c-0.479,0.276-0.643,0.888-0.366,1.367c0.275,0.477,0.888,0.642,1.366,0.365c0.478-0.276,0.642-0.889,0.366-1.365C20.028,22.886,19.417,22.723,18.939,22.998zM11.197,3.593c-0.836-1.04-2.103-1.718-3.541-1.718c-2.52,0-4.562,2.042-4.562,4.562c0,0.957,0.297,1.843,0.8,2.576C5.649,6.484,8.206,4.553,11.197,3.593zM27.106,9.014c0.503-0.733,0.8-1.619,0.8-2.576c0-2.52-2.043-4.562-4.562-4.562c-1.438,0-2.704,0.678-3.541,1.717C22.794,4.553,25.351,6.484,27.106,9.014z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Clock

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Clock();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Clock = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Clock",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M15.5,2.374C8.251,2.375,2.376,8.251,2.374,15.5C2.376,22.748,8.251,28.623,15.5,28.627c7.249-0.004,13.124-5.879,13.125-13.127C28.624,8.251,22.749,2.375,15.5,2.374zM15.5,25.623C9.909,25.615,5.385,21.09,5.375,15.5C5.385,9.909,9.909,5.384,15.5,5.374c5.59,0.01,10.115,4.535,10.124,10.125C25.615,21.09,21.091,25.615,15.5,25.623zM8.625,15.5c-0.001-0.552-0.448-0.999-1.001-1c-0.553,0-1,0.448-1,1c0,0.553,0.449,1,1,1C8.176,16.5,8.624,16.053,8.625,15.5zM8.179,18.572c-0.478,0.277-0.642,0.889-0.365,1.367c0.275,0.479,0.889,0.641,1.365,0.365c0.479-0.275,0.643-0.887,0.367-1.367C9.27,18.461,8.658,18.297,8.179,18.572zM9.18,10.696c-0.479-0.276-1.09-0.112-1.366,0.366s-0.111,1.09,0.365,1.366c0.479,0.276,1.09,0.113,1.367-0.366C9.821,11.584,9.657,10.973,9.18,10.696zM22.822,12.428c0.478-0.275,0.643-0.888,0.366-1.366c-0.275-0.478-0.89-0.642-1.366-0.366c-0.479,0.278-0.642,0.89-0.366,1.367C21.732,12.54,22.344,12.705,22.822,12.428zM12.062,21.455c-0.478-0.275-1.089-0.111-1.366,0.367c-0.275,0.479-0.111,1.09,0.366,1.365c0.478,0.277,1.091,0.111,1.365-0.365C12.704,22.344,12.54,21.732,12.062,21.455zM12.062,9.545c0.479-0.276,0.642-0.888,0.366-1.366c-0.276-0.478-0.888-0.642-1.366-0.366s-0.642,0.888-0.366,1.366C10.973,9.658,11.584,9.822,12.062,9.545zM22.823,18.572c-0.48-0.275-1.092-0.111-1.367,0.365c-0.275,0.479-0.112,1.092,0.367,1.367c0.477,0.275,1.089,0.113,1.365-0.365C23.464,19.461,23.3,18.848,22.823,18.572zM19.938,7.813c-0.477-0.276-1.091-0.111-1.365,0.366c-0.275,0.48-0.111,1.091,0.366,1.367s1.089,0.112,1.366-0.366C20.581,8.702,20.418,8.089,19.938,7.813zM23.378,14.5c-0.554,0.002-1.001,0.45-1.001,1c0.001,0.552,0.448,1,1.001,1c0.551,0,1-0.447,1-1C24.378,14.949,23.929,14.5,23.378,14.5zM15.501,6.624c-0.552,0-1,0.448-1,1l-0.466,7.343l-3.004,1.96c-0.478,0.277-0.642,0.889-0.365,1.365c0.275,0.479,0.889,0.643,1.365,0.367l3.305-1.676C15.39,16.99,15.444,17,15.501,17c0.828,0,1.5-0.671,1.5-1.5l-0.5-7.876C16.501,7.072,16.053,6.624,15.501,6.624zM15.501,22.377c-0.552,0-1,0.447-1,1s0.448,1,1,1s1-0.447,1-1S16.053,22.377,15.501,22.377zM18.939,21.455c-0.479,0.277-0.643,0.889-0.366,1.367c0.275,0.477,0.888,0.643,1.366,0.365c0.478-0.275,0.642-0.889,0.366-1.365C20.028,21.344,19.417,21.18,18.939,21.455z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.StopWatch

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.StopWatch();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.StopWatch = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.StopWatch",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M27.216,18.533c0-3.636-1.655-6.883-4.253-9.032l0.733-0.998l0.482,0.354c0.198,0.146,0.481,0.104,0.628-0.097l0.442-0.604c0.146-0.198,0.103-0.482-0.097-0.628l-2.052-1.506c-0.199-0.146-0.481-0.103-0.628,0.097L22.03,6.724c-0.146,0.199-0.104,0.482,0.096,0.628l0.483,0.354l-0.736,1.003c-1.28-0.834-2.734-1.419-4.296-1.699c0.847-0.635,1.402-1.638,1.403-2.778h-0.002c0-1.922-1.557-3.48-3.479-3.48c-1.925,0-3.48,1.559-3.48,3.48c0,1.141,0.556,2.144,1.401,2.778c-1.549,0.277-2.99,0.857-4.265,1.68L8.424,7.684l0.484-0.353c0.198-0.145,0.245-0.428,0.098-0.628l-0.44-0.604C8.42,5.899,8.136,5.855,7.937,6.001L5.881,7.5c-0.2,0.146-0.243,0.428-0.099,0.628l0.442,0.604c0.145,0.2,0.428,0.244,0.627,0.099l0.483-0.354l0.729,0.999c-2.615,2.149-4.282,5.407-4.282,9.057c0,6.471,5.245,11.716,11.718,11.716c6.47,0,11.716-5.243,11.718-11.716H27.216zM12.918,4.231c0.002-1.425,1.155-2.58,2.582-2.582c1.426,0.002,2.579,1.157,2.581,2.582c-0.002,1.192-0.812,2.184-1.908,2.482v-1.77h0.6c0.246,0,0.449-0.203,0.449-0.449V3.746c0-0.247-0.203-0.449-0.449-0.449h-2.545c-0.247,0-0.449,0.202-0.449,0.449v0.749c0,0.246,0.202,0.449,0.449,0.449h0.599v1.77C13.729,6.415,12.919,5.424,12.918,4.231zM15.5,27.554c-4.983-0.008-9.015-4.038-9.022-9.021c0.008-4.982,4.039-9.013,9.022-9.022c4.981,0.01,9.013,4.04,9.021,9.022C24.513,23.514,20.481,27.546,15.5,27.554zM15.5,12.138c0.476,0,0.861-0.385,0.861-0.86s-0.386-0.861-0.861-0.861s-0.861,0.386-0.861,0.861S15.024,12.138,15.5,12.138zM15.5,24.927c-0.476,0-0.861,0.386-0.861,0.861s0.386,0.861,0.861,0.861s0.861-0.386,0.861-0.861S15.976,24.927,15.5,24.927zM12.618,11.818c-0.237-0.412-0.764-0.553-1.176-0.315c-0.412,0.238-0.554,0.765-0.315,1.177l2.867,6.722c0.481,0.831,1.543,1.116,2.375,0.637c0.829-0.479,1.114-1.543,0.635-2.374L12.618,11.818zM18.698,24.07c-0.412,0.237-0.555,0.765-0.316,1.176c0.237,0.412,0.764,0.554,1.176,0.315c0.413-0.238,0.553-0.765,0.316-1.176C19.635,23.974,19.108,23.832,18.698,24.07zM8.787,15.65c0.412,0.238,0.938,0.097,1.176-0.315c0.237-0.413,0.097-0.938-0.314-1.176c-0.412-0.239-0.938-0.098-1.177,0.313C8.234,14.886,8.375,15.412,8.787,15.65zM22.215,21.413c-0.412-0.236-0.938-0.096-1.176,0.316c-0.238,0.412-0.099,0.938,0.314,1.176c0.41,0.238,0.937,0.098,1.176-0.314C22.768,22.178,22.625,21.652,22.215,21.413zM9.107,18.531c-0.002-0.476-0.387-0.86-0.861-0.86c-0.477,0-0.862,0.385-0.862,0.86c0.001,0.476,0.386,0.86,0.861,0.861C8.722,19.393,9.106,19.008,9.107,18.531zM21.896,18.531c0,0.477,0.384,0.862,0.859,0.86c0.476,0.002,0.862-0.382,0.862-0.859s-0.387-0.86-0.862-0.862C22.279,17.671,21.896,18.056,21.896,18.531zM8.787,21.413c-0.412,0.238-0.554,0.765-0.316,1.176c0.239,0.412,0.765,0.553,1.177,0.316c0.413-0.239,0.553-0.765,0.315-1.178C9.725,21.317,9.198,21.176,8.787,21.413zM21.352,14.157c-0.411,0.238-0.551,0.764-0.312,1.176c0.237,0.413,0.764,0.555,1.174,0.315c0.412-0.236,0.555-0.762,0.316-1.176C22.29,14.06,21.766,13.921,21.352,14.157zM12.304,24.067c-0.413-0.235-0.939-0.096-1.176,0.315c-0.238,0.413-0.098,0.939,0.312,1.178c0.413,0.236,0.939,0.096,1.178-0.315C12.857,24.832,12.715,24.308,12.304,24.067zM18.698,12.992c0.41,0.238,0.938,0.099,1.174-0.313c0.238-0.411,0.1-0.938-0.314-1.177c-0.414-0.238-0.937-0.097-1.177,0.315C18.144,12.229,18.286,12.755,18.698,12.992z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.History

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.History();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.History = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.History",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M10.666,18.292c0.275,0.479,0.889,0.644,1.365,0.367l3.305-1.677C15.39,16.99,15.444,17,15.501,17c0.828,0,1.5-0.671,1.5-1.5l-0.5-7.876c0-0.552-0.448-1-1-1c-0.552,0-1,0.448-1,1l-0.466,7.343l-3.004,1.96C10.553,17.204,10.389,17.816,10.666,18.292zM12.062,9.545c0.479-0.276,0.642-0.888,0.366-1.366c-0.276-0.478-0.888-0.642-1.366-0.366s-0.642,0.888-0.366,1.366C10.973,9.658,11.584,9.822,12.062,9.545zM8.179,18.572c-0.478,0.277-0.642,0.889-0.365,1.367c0.275,0.479,0.889,0.641,1.365,0.365c0.479-0.275,0.643-0.888,0.367-1.367C9.27,18.461,8.658,18.297,8.179,18.572zM9.18,10.696c-0.479-0.276-1.09-0.112-1.366,0.366s-0.111,1.09,0.365,1.366c0.479,0.276,1.09,0.113,1.367-0.366C9.821,11.584,9.657,10.973,9.18,10.696zM6.624,15.5c0,0.553,0.449,1,1,1c0.552,0,1-0.447,1.001-1c-0.001-0.552-0.448-0.999-1.001-1C7.071,14.5,6.624,14.948,6.624,15.5zM14.501,23.377c0,0.553,0.448,1,1,1c0.552,0,1-0.447,1-1s-0.448-1-1-1C14.949,22.377,14.501,22.824,14.501,23.377zM10.696,21.822c-0.275,0.479-0.111,1.09,0.366,1.365c0.478,0.276,1.091,0.11,1.365-0.365c0.277-0.479,0.113-1.09-0.365-1.367C11.584,21.18,10.973,21.344,10.696,21.822zM21.822,10.696c-0.479,0.278-0.643,0.89-0.366,1.367s0.888,0.642,1.366,0.365c0.478-0.275,0.643-0.888,0.365-1.366C22.913,10.584,22.298,10.42,21.822,10.696zM21.456,18.938c-0.274,0.479-0.112,1.092,0.367,1.367c0.477,0.274,1.089,0.112,1.364-0.365c0.276-0.479,0.112-1.092-0.364-1.367C22.343,18.297,21.73,18.461,21.456,18.938zM24.378,15.5c0-0.551-0.448-1-1-1c-0.554,0.002-1.001,0.45-1.001,1c0.001,0.552,0.448,1,1.001,1C23.93,16.5,24.378,16.053,24.378,15.5zM18.573,22.822c0.274,0.477,0.888,0.643,1.366,0.365c0.478-0.275,0.642-0.89,0.365-1.365c-0.277-0.479-0.888-0.643-1.365-0.367C18.46,21.732,18.296,22.344,18.573,22.822zM18.939,9.546c0.477,0.276,1.088,0.112,1.365-0.366c0.276-0.478,0.113-1.091-0.367-1.367c-0.477-0.276-1.09-0.111-1.364,0.366C18.298,8.659,18.462,9.27,18.939,9.546zM28.703,14.364C28.074,7.072,21.654,1.67,14.364,2.295c-3.254,0.281-6.118,1.726-8.25,3.877L4.341,4.681l-1.309,7.364l7.031-2.548L8.427,8.12c1.627-1.567,3.767-2.621,6.194-2.833c5.64-0.477,10.595,3.694,11.089,9.335c0.477,5.64-3.693,10.595-9.333,11.09c-5.643,0.476-10.599-3.694-11.092-9.333c-0.102-1.204,0.019-2.373,0.31-3.478l-3.27,1.186c-0.089,0.832-0.106,1.684-0.031,2.55c0.629,7.29,7.048,12.691,14.341,12.066C23.926,28.074,29.328,21.655,28.703,14.364z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Future

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Future();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Future = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Future",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M17.001,15.5l-0.5-7.876c0-0.552-0.448-1-1-1c-0.552,0-1,0.448-1,1l-0.466,7.343l-3.004,1.96c-0.478,0.277-0.642,0.89-0.365,1.365c0.275,0.479,0.889,0.644,1.365,0.367l3.305-1.677C15.39,16.99,15.444,17,15.501,17C16.329,17,17.001,16.329,17.001,15.5zM18.939,21.455c-0.479,0.277-0.644,0.889-0.366,1.367c0.274,0.477,0.888,0.643,1.366,0.365c0.478-0.275,0.642-0.89,0.365-1.365C20.027,21.344,19.417,21.18,18.939,21.455zM19.938,7.813c-0.477-0.276-1.09-0.111-1.364,0.366c-0.275,0.48-0.111,1.091,0.366,1.367c0.477,0.276,1.088,0.112,1.365-0.366C20.581,8.702,20.418,8.089,19.938,7.813zM21.823,20.305c0.477,0.274,1.089,0.112,1.364-0.365c0.276-0.479,0.112-1.092-0.364-1.367c-0.48-0.275-1.093-0.111-1.367,0.365C21.182,19.416,21.344,20.029,21.823,20.305zM22.822,12.428c0.478-0.275,0.643-0.888,0.365-1.366c-0.274-0.478-0.89-0.642-1.365-0.366c-0.479,0.278-0.643,0.89-0.366,1.367S22.344,12.705,22.822,12.428zM24.378,15.5c0-0.551-0.448-1-1-1c-0.554,0.002-1.001,0.45-1.001,1c0.001,0.552,0.448,1,1.001,1C23.93,16.5,24.378,16.053,24.378,15.5zM9.546,12.062c0.275-0.478,0.111-1.089-0.366-1.366c-0.479-0.276-1.09-0.112-1.366,0.366s-0.111,1.09,0.365,1.366C8.658,12.704,9.269,12.541,9.546,12.062zM6.624,15.5c0,0.553,0.449,1,1,1c0.552,0,1-0.447,1.001-1c-0.001-0.552-0.448-0.999-1.001-1C7.071,14.5,6.624,14.948,6.624,15.5zM9.179,20.305c0.479-0.275,0.643-0.888,0.367-1.367c-0.276-0.477-0.888-0.641-1.367-0.365c-0.478,0.277-0.642,0.889-0.365,1.367C8.089,20.418,8.703,20.58,9.179,20.305zM12.062,9.545c0.479-0.276,0.642-0.888,0.366-1.366c-0.276-0.478-0.888-0.642-1.366-0.366s-0.642,0.888-0.366,1.366C10.973,9.658,11.584,9.822,12.062,9.545zM14.501,23.377c0,0.553,0.448,1,1,1c0.552,0,1-0.447,1-1s-0.448-1-1-1C14.949,22.377,14.501,22.824,14.501,23.377zM10.696,21.822c-0.275,0.479-0.111,1.09,0.366,1.365c0.478,0.276,1.091,0.11,1.365-0.365c0.277-0.479,0.113-1.09-0.365-1.367C11.584,21.18,10.973,21.344,10.696,21.822zM28.674,14.087l-3.27-1.186c0.291,1.105,0.41,2.274,0.309,3.478c-0.492,5.639-5.449,9.809-11.091,9.333c-5.639-0.495-9.809-5.45-9.333-11.09c0.494-5.641,5.449-9.812,11.089-9.335c2.428,0.212,4.567,1.266,6.194,2.833l-1.637,1.377l7.031,2.548l-1.309-7.364l-1.771,1.492c-2.133-2.151-4.996-3.597-8.25-3.877C9.346,1.67,2.926,7.072,2.297,14.364c-0.625,7.291,4.777,13.71,12.066,14.339c7.293,0.625,13.713-4.776,14.342-12.066C28.779,15.771,28.762,14.919,28.674,14.087z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.GlobeAlt2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.GlobeAlt2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.GlobeAlt2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.GlobeAlt2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,1.466C7.973,1.466,1.466,7.973,1.466,16c0,8.027,6.507,14.534,14.534,14.534c8.027,0,14.534-6.507,14.534-14.534C30.534,7.973,24.027,1.466,16,1.466zM8.251,7.48c0.122,0.055,0.255,0.104,0.28,0.137C8.57,7.668,8.621,7.823,8.557,7.861C8.492,7.9,8.39,7.887,8.376,7.771c-0.013-0.115-0.026-0.128-0.18-0.18c-0.022-0.007-0.035-0.01-0.051-0.015C8.18,7.544,8.216,7.512,8.251,7.48zM7.733,7.974c0.031,0.087,0.113,0.125,0,0.17C7.673,8.168,7.611,8.172,7.559,8.165C7.617,8.102,7.672,8.035,7.733,7.974zM16,27.533C9.639,27.533,4.466,22.36,4.466,16c0-0.085,0.011-0.168,0.013-0.254c0.004-0.003,0.008-0.006,0.012-0.009c0.129-0.102,0.283-0.359,0.334-0.45c0.052-0.089,0.181-0.154,0.116-0.256c-0.059-0.096-0.292-0.23-0.407-0.261c0.01-0.099,0.032-0.195,0.045-0.294c0.063,0.077,0.137,0.17,0.208,0.194c0.115,0.038,0.501,0.052,0.566,0.052c0.063,0,0.334,0.014,0.386-0.064c0.051-0.077,0.09-0.077,0.154-0.077c0.064,0,0.18,0.231,0.271,0.257c0.089,0.026,0.257,0.013,0.244,0.181c-0.012,0.166,0.077,0.309,0.167,0.321c0.09,0.013,0.296-0.194,0.296-0.194s0,0.322-0.012,0.438C6.846,15.698,7,16.124,7,16.124s0.193,0.397,0.244,0.488c0.052,0.09,0.27,0.36,0.27,0.476c0,0.117,0.026,0.297,0.104,0.297s0.155-0.206,0.244-0.335c0.091-0.128,0.117-0.31,0.155-0.438c0.039-0.129,0.039-0.36,0.039-0.45c0-0.091,0.076-0.168,0.257-0.245c0.181-0.077,0.309-0.296,0.463-0.412c0.155-0.116,0.142-0.309,0.452-0.309c0.308,0,0.282,0,0.36-0.078c0.077-0.077,0.154-0.128,0.192,0.013c0.039,0.142,0.257,0.347,0.296,0.399c0.039,0.052,0.116,0.193,0.104,0.348c-0.013,0.153,0.012,0.334,0.077,0.334c0.064,0,0.193-0.219,0.193-0.219s0.283-0.192,0.27,0.014c-0.014,0.205,0.025,0.425,0.025,0.552c0,0.13,0.232,0.438,0.232,0.362c0-0.079,0.103-0.296,0.103-0.413c0-0.114,0.064-0.063,0.231,0.051c0.167,0.116,0.283,0.349,0.283,0.349s0.168,0.154,0.193,0.219c0.026,0.064,0.206-0.025,0.244-0.104c0.039-0.076,0.065-0.115,0.167-0.141c0.104-0.026,0.231-0.026,0.271-0.168c0.039-0.142,0.154-0.308,0-0.502c-0.154-0.193-0.232-0.321-0.347-0.412c-0.117-0.09-0.206-0.322-0.206-0.322s0.244-0.218,0.321-0.296c0.079-0.077,0.193-0.025,0.207,0.064c0.013,0.091-0.115,0.168-0.141,0.361c-0.026,0.192,0.154,0.257,0.206,0.192c0.051-0.065,0.18-0.219,0.18-0.257c0-0.039-0.089-0.026-0.102-0.167c-0.013-0.142,0.166-0.245,0.23-0.207c0.066,0.039,0.477-0.051,0.67-0.154s0.308-0.322,0.425-0.412c0.116-0.089,0.515-0.386,0.489-0.527c-0.026-0.142,0.012-0.334-0.09-0.515c-0.103-0.18-0.232-0.295-0.283-0.373c-0.051-0.077,0.219-0.09,0.347-0.206c0.129-0.116,0-0.219-0.064-0.206c-0.064,0.013-0.232,0.052-0.296,0.039c-0.064-0.013-0.103-0.077-0.206-0.155c-0.102-0.077,0.026-0.192,0.091-0.179c0.064,0.013,0.23-0.129,0.308-0.193c0.077-0.064,0.193-0.115,0.154-0.051c-0.038,0.064-0.128,0.296-0.026,0.309c0.104,0.013,0.348-0.193,0.388-0.18c0.038,0.013,0.102,0.18,0.064,0.257c-0.039,0.077-0.039,0.206,0.013,0.193c0.051-0.013,0.154-0.129,0.18-0.09c0.027,0.039,0.154,0.116,0.09,0.257c-0.063,0.142-0.193,0.193-0.039,0.284c0.154,0.089,0.206,0.012,0.322-0.052c0.115-0.064,0.193-0.347,0.128-0.438c-0.064-0.09-0.218-0.27-0.218-0.334c0-0.064,0.257-0.064,0.257-0.167s0.09-0.18,0.18-0.219c0.091-0.039,0.206-0.206,0.244-0.154c0.039,0.052,0.271,0.116,0.334,0.039c0.064-0.077,0.4-0.36,0.605-0.515c0.206-0.154,0.283-0.334,0.336-0.515c0.051-0.18,0.128-0.296,0.102-0.437v0c0.077,0.18,0.09,0.309,0.077,0.45c-0.013,0.142,0,0.438,0.026,0.476c0.025,0.039,0.129,0.128,0.192,0.103c0.064-0.025-0.025-0.283-0.025-0.334c0-0.052,0.09-0.129,0.142-0.142c0.052-0.013,0-0.231-0.065-0.322c-0.063-0.09-0.154-0.142-0.102-0.154c0.051-0.013,0.115-0.116,0.077-0.142c-0.039-0.025-0.014-0.116-0.103-0.09c-0.065,0.019-0.241-0.015-0.235,0.095c-0.037-0.11-0.116-0.183-0.216-0.172c-0.116,0.013-0.181,0.077-0.296,0.077s-0.025-0.18-0.077-0.18c-0.051,0-0.168,0.167-0.231,0.077c-0.064-0.09,0.18-0.206,0.373-0.27c0.192-0.064,0.514-0.438,0.644-0.451c0.128-0.013,0.45,0.026,0.733,0.013c0.283-0.013,0.373-0.129,0.463-0.064s0.283,0.142,0.399,0.129c0.116-0.014,0.064,0,0.244-0.129c0.18-0.129,0.348-0.193,0.438-0.296c0.09-0.103,0.335-0.18,0.348-0.077c0.014,0.103-0.026,0.206,0.077,0.206s0.258-0.103,0.386-0.154c0.129-0.051,0.231-0.116,0.231-0.116s-0.527,0.36-0.655,0.438c-0.129,0.077-0.438,0.129-0.567,0.283c-0.128,0.155-0.205,0.206-0.192,0.374c0.014,0.167,0.231,0.386,0.128,0.54c-0.103,0.154-0.141,0.373-0.141,0.373s0.154-0.219,0.373-0.36s0.348-0.334,0.425-0.412s0.309-0.091,0.309-0.181s0.064-0.206,0.104-0.309c0.038-0.103-0.077-0.078,0-0.206c0.076-0.129,0.064-0.232,0.45-0.232s0.257,0.026,0.566,0.013c0.309-0.013,0.424-0.167,0.72-0.245c0.296-0.077,0.527-0.128,0.618-0.089c0.09,0.038,0.232,0.012,0.141-0.078c-0.089-0.09-0.295-0.219-0.193-0.245c0.104-0.026,0.207-0.039,0.246-0.142c0.039-0.103-0.142-0.283-0.039-0.386c0.104-0.103-0.077-0.231-0.207-0.257c-0.128-0.025-0.63,0.026-0.731-0.025c-0.104-0.052-0.271-0.116-0.322-0.078c-0.052,0.039-0.168,0.245-0.168,0.245s-0.09,0.025-0.168-0.09c-0.076-0.116-0.5-0.103-0.629-0.103s-0.271,0.025-0.413,0.039c-0.141,0.013-0.219,0.052-0.322-0.039c-0.102-0.09-0.243-0.129-0.296-0.167c-0.051-0.039-0.334-0.039-0.553-0.012c-0.218,0.025-0.438,0.025-0.438,0.025s-0.104-0.039-0.257-0.129c-0.154-0.09-0.309-0.154-0.361-0.154c-0.051,0-0.449,0.064-0.539,0c-0.091-0.064-0.181-0.103-0.245-0.103s-0.115-0.103-0.038-0.103s0.437-0.103,0.437-0.103s-0.103-0.142-0.231-0.142c-0.128,0-0.359-0.064-0.424-0.064s-0.014,0.064-0.142,0.039c-0.13-0.026-0.258-0.078-0.335-0.026c-0.076,0.051-0.258,0.128-0.064,0.18c0.193,0.052,0.373,0,0.425,0.078c0.052,0.077,0,0.115,0,0.167s-0.103,0.193-0.167,0.219c-0.064,0.025-0.143-0.039-0.27,0.025c-0.129,0.064-0.451,0.013-0.49,0.052c-0.038,0.039-0.115-0.103-0.18-0.077c-0.064,0.025-0.232,0.193-0.322,0.18c-0.089-0.013-0.206-0.103-0.206-0.206s-0.038-0.232-0.077-0.258c-0.038-0.025-0.322-0.039-0.425-0.025c-0.103,0.013-0.424,0.038-0.477,0.09c-0.052,0.052-0.193,0.09-0.283,0.09s-0.167-0.09-0.36-0.116c-0.192-0.026-0.617-0.039-0.669-0.026s-0.218-0.025-0.155-0.077c0.065-0.051,0.257-0.219,0.143-0.295c-0.117-0.078-0.375-0.078-0.489-0.09c-0.117-0.013-0.232-0.039-0.413-0.013c-0.181,0.026-0.219,0.116-0.296,0.039c-0.077-0.077,0.193,0.039-0.077-0.077c-0.27-0.116-0.399-0.103-0.477-0.064c-0.077,0.039,0.013,0.025-0.192,0.103c-0.206,0.078-0.322,0.116-0.374,0.129c-0.051,0.012-0.372-0.065-0.411-0.091c-0.038-0.025-0.181,0.013-0.309,0.064S9.895,7.025,9.767,7C9.638,6.973,9.432,6.973,9.303,7.025C9.174,7.076,9.084,7.076,8.956,7.166c-0.13,0.09-0.373,0.142-0.373,0.142S8.522,7.305,8.448,7.301C10.474,5.541,13.111,4.466,16,4.466c6.361,0,11.534,5.173,11.534,11.534S22.36,27.533,16,27.533zM14.888,19.92c0,0,0.207-0.026,0.207-0.117c0-0.089-0.207-0.205-0.282-0.102c-0.078,0.102-0.219,0.205-0.207,0.296C14.625,20.138,14.888,19.92,14.888,19.92zM14.875,17.023c-0.181,0.233-0.167,0.182-0.296,0.128c-0.128-0.05-0.334,0.116-0.296,0.182c0.039,0.064,0.322-0.014,0.386,0.102c0.065,0.116,0.065,0.129,0.193,0.104c0.128-0.026,0.257-0.205,0.219-0.295C15.043,17.151,14.875,17.023,14.875,17.023zM14.837,18.245c-0.051,0-0.412,0.064-0.451,0.079c-0.039,0.013-0.27-0.025-0.27-0.025c-0.09,0.089-0.026,0.179,0.116,0.166s0.438-0.052,0.502-0.052C14.799,18.413,14.888,18.245,14.837,18.245zM14.284,14.668c-0.19,0.03-0.308,0.438-0.155,0.425C14.284,15.081,14.451,14.643,14.284,14.668zM14.734,16.959c-0.052-0.064-0.181-0.271-0.323-0.219c-0.042,0.017-0.153,0.245-0.012,0.245C14.541,16.985,14.786,17.023,14.734,16.959zM14.85,16.805c0.232-0.013,0.167-0.245-0.013-0.257C14.786,16.544,14.618,16.818,14.85,16.805zM17.591,18.928c-0.193-0.039-0.244-0.102-0.45-0.205c-0.207-0.103-0.67-0.103-0.682-0.039c-0.014,0.064,0,0-0.155-0.05c-0.153-0.054-0.271,0-0.309-0.091c-0.038-0.091-0.128-0.117-0.244-0.002c-0.097,0.097-0.142,0.104,0.078,0.143c0.218,0.039,0.283,0.039,0.192,0.141c-0.09,0.104-0.154,0.233-0.077,0.244c0.077,0.015,0.309-0.05,0.334,0c0.026,0.054-0.051,0.064,0.207,0.105c0.258,0.037,0.309,0.128,0.359,0.178c0.051,0.052,0.206,0.22,0.104,0.22c-0.104,0-0.219,0.128-0.142,0.143c0.077,0.013,0.309-0.039,0.321,0c0.014,0.037,0.143,0.283,0.271,0.271c0.129-0.013,0.206-0.244,0.27-0.31c0.065-0.064,0.322-0.104,0.349,0.012c0.026,0.116,0.104,0.233,0.257,0.311c0.154,0.076,0.335,0.154,0.348,0.089c0.013-0.064-0.077-0.309-0.181-0.346c-0.103-0.041-0.282-0.259-0.282-0.348c0-0.091-0.155-0.117-0.232-0.182C17.849,19.147,17.784,18.967,17.591,18.928zM8.042,17.023c-0.084,0.037-0.155,0.476,0,0.527c0.154,0.052,0.244-0.205,0.193-0.271C8.183,17.218,8.158,16.973,8.042,17.023zM15.429,18.117c-0.118-0.05-0.335,0.424-0.181,0.463C15.403,18.62,15.518,18.156,15.429,18.117zM15.687,13.703c0.077,0,0.18-0.051,0.18-0.193c0-0.142,0.18,0,0.27-0.013s0.141-0.103,0.18-0.206c0.005-0.013,0.008-0.021,0.009-0.027c-0.003,0.024-0.001,0.093,0.095,0.117c0.154,0.038,0.205-0.064,0.205-0.103s0.283-0.103,0.336-0.142c0.051-0.038,0.258-0.103,0.27-0.154c0.013-0.051,0-0.348,0.064-0.373c0.064-0.026,0.154-0.026,0.052-0.206c-0.104-0.181-0.104-0.348-0.232-0.271c-0.095,0.057-0.038,0.284-0.115,0.438s-0.142,0.296-0.193,0.296s-0.321,0.103-0.399,0.18c-0.076,0.077-0.45-0.064-0.501,0c-0.052,0.064-0.154,0.141-0.219,0.193c-0.065,0.051-0.245,0.013-0.207,0.167C15.518,13.562,15.609,13.703,15.687,13.703zM17.449,12.056c0.18-0.013,0.348-0.064,0.348-0.064s0.271,0.013,0.232-0.116c-0.04-0.128-0.322-0.141-0.375-0.128c-0.051,0.013-0.142-0.142-0.244-0.116c-0.096,0.023-0.128,0.155-0.128,0.193c0,0.039-0.36,0.115-0.245,0.219C17.153,12.146,17.27,12.069,17.449,12.056zM13.91,19.058c0.104,0.064,0.296-0.219,0.349-0.13c0.051,0.091-0.013,0.13,0.076,0.246c0.091,0.114,0.258,0.102,0.258,0.102s-0.013-0.309-0.155-0.387c-0.142-0.077-0.232-0.166-0.064-0.141c0.167,0.026,0.257-0.039,0.219-0.114c-0.039-0.078-0.283-0.039-0.361-0.026s-0.193-0.052-0.193-0.052c-0.077,0.024-0.063,0.089-0.09,0.219C13.923,18.902,13.807,18.992,13.91,19.058zM20.924,21.618c-0.231-0.052-0.077,0.039,0,0.154c0.077,0.116,0.232,0.176,0.258,0.05C21.193,21.759,21.155,21.67,20.924,21.618zM21.915,24.744c-0.077,0.064,0,0.091-0.219,0.22c-0.22,0.13-0.49,0.271-0.541,0.386c-0.052,0.116,0.051,0.181,0.258,0.192c0.206,0.013,0.154,0.053,0.296-0.103s0.271-0.244,0.438-0.373c0.168-0.128,0.168-0.322,0.168-0.322s-0.181-0.178-0.193-0.141C22.1,24.665,21.992,24.681,21.915,24.744zM18.504,21.618c0.014-0.116-0.219-0.116-0.334-0.207c-0.116-0.089-0.128-0.359-0.193-0.515c-0.064-0.153-0.192-0.257-0.322-0.397c-0.128-0.143-0.192-0.465-0.23-0.438c-0.039,0.025-0.154,0.399-0.064,0.515c0.09,0.116-0.039,0.348-0.103,0.503c-0.065,0.153-0.22-0.026-0.349-0.104c-0.129-0.078-0.308-0.128-0.398-0.219c-0.09-0.091,0.155-0.335,0.091-0.426c-0.065-0.09-0.412-0.013-0.45-0.013c-0.039,0-0.116-0.128-0.194-0.128c-0.077,0-0.064,0.258-0.064,0.258s-0.078-0.091-0.193-0.207c-0.117-0.115,0.012,0.077-0.103,0.193c-0.117,0.117-0.079,0.078-0.129,0.206c-0.051,0.129-0.167,0.077-0.283-0.052c-0.116-0.128-0.179-0.037-0.258,0c-0.077,0.039-0.141,0.259-0.18,0.309c-0.039,0.052-0.309,0.117-0.374,0.182c-0.064,0.062-0.09,0.27-0.09,0.322c0,0.05-0.271,0.023-0.361,0.089c-0.09,0.064-0.23,0.025-0.321,0.025c-0.09,0-0.399,0.244-0.502,0.308c-0.103,0.066-0.103,0.298-0.051,0.362c0.051,0.063,0.154,0.219,0.09,0.244c-0.064,0.026-0.104,0.206,0.051,0.359c0.154,0.155,0.103,0.194,0.115,0.271c0.014,0.077,0.078,0.104,0.181,0.232c0.102,0.128-0.181,0.231-0.219,0.31c-0.039,0.076,0.091,0.192,0.167,0.257c0.077,0.063,0.271,0.026,0.386-0.013c0.117-0.039,0.245-0.143,0.321-0.155c0.079-0.013,0.438-0.026,0.438-0.026s0.129-0.192,0.219-0.296c0.089-0.102,0.372-0.013,0.372-0.013s0.117-0.076,0.426-0.141c0.309-0.065,0.179,0.064,0.296,0.104c0.115,0.037,0.27,0.062,0.359,0.128c0.09,0.064,0,0.218-0.012,0.283c-0.014,0.064,0.219,0.038,0.23-0.026c0.014-0.064,0.077-0.128,0.207-0.205c0.128-0.078,0.025,0.114,0.076,0.231c0.052,0.116,0.129-0.157,0.129-0.026c0,0.039,0.039,0.078,0.051,0.116c0.014,0.039,0.181,0.052,0.181,0.18c0,0.13,0,0.207,0.039,0.231c0.038,0.026,0.244,0,0.335,0.155c0.089,0.154,0.154,0.013,0.205-0.052c0.052-0.064,0.231,0.026,0.283,0.078c0.052,0.05,0.193-0.104,0.387-0.155c0.192-0.051,0.167-0.039,0.219-0.115c0.051-0.078,0.09-0.283,0.205-0.438c0.115-0.153,0.271-0.424,0.271-0.631c0-0.206-0.014-0.682-0.155-0.899C18.761,21.953,18.492,21.733,18.504,21.618zM18.029,24.77c-0.065-0.013-0.207-0.062-0.207-0.062c-0.142,0.141,0.142,0.141,0.104,0.283c-0.039,0.141,0.193,0.089,0.257,0.064c0.063-0.027,0.22-0.323,0.193-0.399C18.351,24.577,18.093,24.783,18.029,24.77zM22.803,24.178c-0.052,0-0.077,0.064-0.192,0c-0.117-0.063-0.091-0.037-0.168-0.167c-0.077-0.127-0.091-0.296-0.219-0.23c-0.051,0.025,0,0.168,0.051,0.218c0.053,0.052,0.077,0.231,0.064,0.283c-0.012,0.052-0.231,0.116-0.129,0.18c0.104,0.064,0.297,0,0.271,0.078c-0.025,0.077-0.129,0.179-0.013,0.205c0.115,0.025,0.154-0.089,0.207-0.178c0.051-0.093,0.089-0.169,0.179-0.221C22.944,24.294,22.854,24.178,22.803,24.178zM22.815,21.18c0.168,0.064,0.464-0.231,0.347-0.27C23.047,20.871,22.815,21.18,22.815,21.18zM13.923,19.906c-0.029,0.115,0.193,0.167,0.206,0.039C14.141,19.816,13.949,19.803,13.923,19.906zM14.27,16.47c-0.064,0.065-0.257,0.193-0.283,0.31c-0.025,0.115,0.309-0.182,0.399-0.296c0.091-0.117,0.27-0.052,0.308-0.117c0.04-0.063,0.04-0.063,0.04-0.063s-0.142-0.025-0.257-0.063c-0.117-0.039-0.258,0.102-0.193-0.104c0.064-0.206,0.257-0.167,0.219-0.322c-0.039-0.154-0.168-0.193-0.207-0.193c-0.09,0,0.013,0.141-0.116,0.231c-0.128,0.09-0.271,0.128-0.193,0.283C14.064,16.29,14.334,16.405,14.27,16.47zM13.254,19.751c0.013-0.076-0.142-0.192-0.206-0.192c-0.065,0-0.386-0.077-0.386-0.077c-0.058,0.023-0.135,0.045-0.158,0.077c-0.007-0.011-0.022-0.024-0.049-0.039c-0.142-0.075-0.309,0-0.361-0.102c-0.05-0.104-0.127-0.104-0.179-0.039c-0.094,0.117,0.025,0.206,0.063,0.231c0.038,0.024,0.181,0.052,0.309,0.039c0.08-0.008,0.181-0.027,0.21-0.059c0.004,0.014,0.016,0.027,0.035,0.044c0.103,0.092,0.167,0.13,0.321,0.116C13.009,19.74,13.241,19.829,13.254,19.751zM12.881,18.992c0.065,0,0.193,0,0.283,0.026c0.09,0.025,0.386,0.05,0.373-0.064c-0.013-0.115-0.038-0.297,0.089-0.411c0.13-0.117,0.257-0.18,0.193-0.348c-0.063-0.167-0.193-0.271-0.103-0.349c0.09-0.076,0.192-0.102,0.192-0.166c0-0.065-0.217,0.18-0.244-0.246c-0.005-0.091-0.206,0.025-0.219,0.116c-0.012,0.091,0.142,0.167-0.103,0.167c-0.245,0-0.257,0.194-0.309,0.232c-0.052,0.039-0.103,0.051-0.207,0.076c-0.102,0.026-0.127,0.13-0.153,0.194c-0.025,0.063-0.206-0.116-0.257-0.064c-0.051,0.052-0.013,0.296,0.077,0.501C12.585,18.863,12.816,18.992,12.881,18.992zM11.979,18.928c0.065-0.077,0.038-0.192-0.063-0.18c-0.103,0.013-0.193-0.168-0.36-0.283c-0.168-0.114-0.296-0.194-0.451-0.36c-0.154-0.167-0.347-0.271-0.45-0.359c-0.104-0.091-0.257-0.13-0.322-0.116c-0.159,0.032,0.231,0.309,0.271,0.346c0.039,0.041,0.387,0.335,0.387,0.478s0.231,0.476,0.296,0.527c0.064,0.052,0.385,0.244,0.437,0.348c0.052,0.103,0.167,0.13,0.167-0.013C11.89,19.174,11.916,19.006,11.979,18.928zM11.002,17.474c0.064,0.232,0.193,0.464,0.244,0.555c0.052,0.089,0.271,0.217,0.348,0.281c0.077,0.064,0.192-0.024,0.143-0.102c-0.052-0.078-0.155-0.192-0.167-0.283c-0.013-0.091-0.078-0.233-0.181-0.387c-0.102-0.153-0.192-0.192-0.257-0.295c-0.064-0.104-0.296-0.297-0.296-0.297c-0.102,0.013-0.102,0.205-0.051,0.271C10.834,17.28,10.938,17.243,11.002,17.474z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.GlobeAlt

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.GlobeAlt();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.GlobeAlt = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.GlobeAlt",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,1.466C7.973,1.466,1.466,7.973,1.466,16c0,8.027,6.507,14.534,14.534,14.534c8.027,0,14.534-6.507,14.534-14.534C30.534,7.973,24.027,1.466,16,1.466zM27.436,17.39c0.001,0.002,0.004,0.002,0.005,0.004c-0.022,0.187-0.054,0.37-0.085,0.554c-0.015-0.012-0.034-0.025-0.047-0.036c-0.103-0.09-0.254-0.128-0.318-0.115c-0.157,0.032,0.229,0.305,0.267,0.342c0.009,0.009,0.031,0.03,0.062,0.058c-1.029,5.312-5.709,9.338-11.319,9.338c-4.123,0-7.736-2.18-9.776-5.441c0.123-0.016,0.24-0.016,0.28-0.076c0.051-0.077,0.102-0.241,0.178-0.331c0.077-0.089,0.165-0.229,0.127-0.292c-0.039-0.064,0.101-0.344,0.088-0.419c-0.013-0.076-0.127-0.256,0.064-0.407s0.394-0.382,0.407-0.444c0.012-0.063,0.166-0.331,0.152-0.458c-0.012-0.127-0.152-0.28-0.24-0.318c-0.09-0.037-0.28-0.05-0.356-0.151c-0.077-0.103-0.292-0.203-0.368-0.178c-0.076,0.025-0.204,0.05-0.305-0.015c-0.102-0.062-0.267-0.139-0.33-0.189c-0.065-0.05-0.229-0.088-0.305-0.088c-0.077,0-0.065-0.052-0.178,0.101c-0.114,0.153,0,0.204-0.204,0.177c-0.204-0.023,0.025-0.036,0.141-0.189c0.113-0.152-0.013-0.242-0.141-0.203c-0.126,0.038-0.038,0.115-0.241,0.153c-0.203,0.036-0.203-0.09-0.076-0.115s0.355-0.139,0.355-0.19c0-0.051-0.025-0.191-0.127-0.191s-0.077-0.126-0.229-0.291c-0.092-0.101-0.196-0.164-0.299-0.204c-0.09-0.579-0.15-1.167-0.15-1.771c0-2.844,1.039-5.446,2.751-7.458c0.024-0.02,0.048-0.034,0.069-0.036c0.084-0.009,0.31-0.025,0.51-0.059c0.202-0.034,0.418-0.161,0.489-0.153c0.069,0.008,0.241,0.008,0.186-0.042C8.417,8.2,8.339,8.082,8.223,8.082S8.215,7.896,8.246,7.896c0.03,0,0.186,0.025,0.178,0.11C8.417,8.091,8.471,8.2,8.625,8.167c0.156-0.034,0.132-0.162,0.102-0.195C8.695,7.938,8.672,7.853,8.642,7.794c-0.031-0.06-0.023-0.136,0.14-0.153C8.944,7.625,9.168,7.708,9.16,7.573s0-0.28,0.046-0.356C9.253,7.142,9.354,7.09,9.299,7.065C9.246,7.04,9.176,7.099,9.121,6.972c-0.054-0.127,0.047-0.22,0.108-0.271c0.02-0.015,0.067-0.06,0.124-0.112C11.234,5.257,13.524,4.466,16,4.466c3.213,0,6.122,1.323,8.214,3.45c-0.008,0.022-0.01,0.052-0.031,0.056c-0.077,0.013-0.166,0.063-0.179-0.051c-0.013-0.114-0.013-0.331-0.102-0.203c-0.089,0.127-0.127,0.127-0.127,0.191c0,0.063,0.076,0.127,0.051,0.241C23.8,8.264,23.8,8.341,23.84,8.341c0.036,0,0.126-0.115,0.239-0.141c0.116-0.025,0.319-0.088,0.332,0.026c0.013,0.115,0.139,0.152,0.013,0.203c-0.128,0.051-0.267,0.026-0.293-0.051c-0.025-0.077-0.114-0.077-0.203-0.013c-0.088,0.063-0.279,0.292-0.279,0.292s-0.306,0.139-0.343,0.114c-0.04-0.025,0.101-0.165,0.203-0.228c0.102-0.064,0.178-0.204,0.14-0.242c-0.038-0.038-0.088-0.279-0.063-0.343c0.025-0.063,0.139-0.152,0.013-0.216c-0.127-0.063-0.217-0.14-0.318-0.178s-0.216,0.152-0.305,0.204c-0.089,0.051-0.076,0.114-0.191,0.127c-0.114,0.013-0.189,0.165,0,0.254c0.191,0.089,0.255,0.152,0.204,0.204c-0.051,0.051-0.267-0.025-0.267-0.025s-0.165-0.076-0.268-0.076c-0.101,0-0.229-0.063-0.33-0.076c-0.102-0.013-0.306-0.013-0.355,0.038c-0.051,0.051-0.179,0.203-0.28,0.152c-0.101-0.051-0.101-0.102-0.241-0.051c-0.14,0.051-0.279-0.038-0.355,0.038c-0.077,0.076-0.013,0.076-0.255,0c-0.241-0.076-0.189,0.051-0.419,0.089s-0.368-0.038-0.432,0.038c-0.064,0.077-0.153,0.217-0.19,0.127c-0.038-0.088,0.126-0.241,0.062-0.292c-0.062-0.051-0.33-0.025-0.367,0.013c-0.039,0.038-0.014,0.178,0.011,0.229c0.026,0.05,0.064,0.254-0.011,0.216c-0.077-0.038-0.064-0.166-0.141-0.152c-0.076,0.013-0.165,0.051-0.203,0.077c-0.038,0.025-0.191,0.025-0.229,0.076c-0.037,0.051,0.014,0.191-0.051,0.203c-0.063,0.013-0.114,0.064-0.254-0.025c-0.14-0.089-0.14-0.038-0.178-0.012c-0.038,0.025-0.216,0.127-0.229,0.012c-0.013-0.114,0.025-0.152-0.089-0.229c-0.115-0.076-0.026-0.076,0.127-0.025c0.152,0.05,0.343,0.075,0.622-0.013c0.28-0.089,0.395-0.127,0.28-0.178c-0.115-0.05-0.229-0.101-0.406-0.127c-0.179-0.025-0.42-0.025-0.7-0.127c-0.279-0.102-0.343-0.14-0.457-0.165c-0.115-0.026-0.813-0.14-1.132-0.089c-0.317,0.051-1.193,0.28-1.245,0.318s-0.128,0.19-0.292,0.318c-0.165,0.127-0.47,0.419-0.712,0.47c-0.241,0.051-0.521,0.254-0.521,0.305c0,0.051,0.101,0.242,0.076,0.28c-0.025,0.038,0.05,0.229,0.191,0.28c0.139,0.05,0.381,0.038,0.393-0.039c0.014-0.076,0.204-0.241,0.217-0.127c0.013,0.115,0.14,0.292,0.114,0.368c-0.025,0.077,0,0.153,0.09,0.14c0.088-0.012,0.559-0.114,0.559-0.114s0.153-0.064,0.127-0.166c-0.026-0.101,0.166-0.241,0.203-0.279c0.038-0.038,0.178-0.191,0.014-0.241c-0.167-0.051-0.293-0.064-0.115-0.216s0.292,0,0.521-0.229c0.229-0.229-0.051-0.292,0.191-0.305c0.241-0.013,0.496-0.025,0.444,0.051c-0.05,0.076-0.342,0.242-0.508,0.318c-0.166,0.077-0.14,0.216-0.076,0.292c0.063,0.076,0.09,0.254,0.204,0.229c0.113-0.025,0.254-0.114,0.38-0.101c0.128,0.012,0.383-0.013,0.42-0.013c0.039,0,0.216,0.178,0.114,0.203c-0.101,0.025-0.229,0.013-0.445,0.025c-0.215,0.013-0.456,0.013-0.456,0.051c0,0.039,0.292,0.127,0.19,0.191c-0.102,0.063-0.203-0.013-0.331-0.026c-0.127-0.012-0.203,0.166-0.241,0.267c-0.039,0.102,0.063,0.28-0.127,0.216c-0.191-0.063-0.331-0.063-0.381-0.038c-0.051,0.025-0.203,0.076-0.331,0.114c-0.126,0.038-0.076-0.063-0.242-0.063c-0.164,0-0.164,0-0.164,0l-0.103,0.013c0,0-0.101-0.063-0.114-0.165c-0.013-0.102,0.05-0.216-0.013-0.241c-0.064-0.026-0.292,0.012-0.33,0.088c-0.038,0.076-0.077,0.216-0.026,0.28c0.052,0.063,0.204,0.19,0.064,0.152c-0.14-0.038-0.317-0.051-0.419,0.026c-0.101,0.076-0.279,0.241-0.279,0.241s-0.318,0.025-0.318,0.102c0,0.077,0,0.178-0.114,0.191c-0.115,0.013-0.268,0.05-0.42,0.076c-0.153,0.025-0.139,0.088-0.317,0.102s-0.204,0.089-0.038,0.114c0.165,0.025,0.418,0.127,0.431,0.241c0.014,0.114-0.013,0.242-0.076,0.356c-0.043,0.079-0.305,0.026-0.458,0.026c-0.152,0-0.456-0.051-0.584,0c-0.127,0.051-0.102,0.305-0.064,0.419c0.039,0.114-0.012,0.178-0.063,0.216c-0.051,0.038-0.065,0.152,0,0.204c0.063,0.051,0.114,0.165,0.166,0.178c0.051,0.013,0.215-0.038,0.279,0.025c0.064,0.064,0.127,0.216,0.165,0.178c0.039-0.038,0.089-0.203,0.153-0.166c0.064,0.039,0.216-0.012,0.331-0.025s0.177-0.14,0.292-0.204c0.114-0.063,0.05-0.063,0.013-0.14c-0.038-0.076,0.114-0.165,0.204-0.254c0.088-0.089,0.253-0.013,0.292-0.115c0.038-0.102,0.051-0.279,0.151-0.267c0.103,0.013,0.243,0.076,0.331,0.076c0.089,0,0.279-0.14,0.332-0.165c0.05-0.025,0.241-0.013,0.267,0.102c0.025,0.114,0.241,0.254,0.292,0.279c0.051,0.025,0.381,0.127,0.433,0.165c0.05,0.038,0.126,0.153,0.152,0.254c0.025,0.102,0.114,0.102,0.128,0.013c0.012-0.089-0.065-0.254,0.025-0.242c0.088,0.013,0.191-0.026,0.191-0.026s-0.243-0.165-0.331-0.203c-0.088-0.038-0.255-0.114-0.331-0.241c-0.076-0.127-0.267-0.153-0.254-0.279c0.013-0.127,0.191-0.051,0.292,0.051c0.102,0.102,0.356,0.241,0.445,0.33c0.088,0.089,0.229,0.127,0.267,0.242c0.039,0.114,0.152,0.241,0.19,0.292c0.038,0.051,0.165,0.331,0.204,0.394c0.038,0.063,0.165-0.012,0.229-0.063c0.063-0.051,0.179-0.076,0.191-0.178c0.013-0.102-0.153-0.178-0.203-0.216c-0.051-0.038,0.127-0.076,0.191-0.127c0.063-0.05,0.177-0.14,0.228-0.063c0.051,0.077,0.026,0.381,0.051,0.432c0.025,0.051,0.279,0.127,0.331,0.191c0.05,0.063,0.267,0.089,0.304,0.051c0.039-0.038,0.242,0.026,0.294,0.038c0.049,0.013,0.202-0.025,0.304-0.05c0.103-0.025,0.204-0.102,0.191,0.063c-0.013,0.165-0.051,0.419-0.179,0.546c-0.127,0.127-0.076,0.191-0.202,0.191c-0.06,0-0.113,0-0.156,0.021c-0.041-0.065-0.098-0.117-0.175-0.097c-0.152,0.038-0.344,0.038-0.47,0.19c-0.128,0.153-0.178,0.165-0.204,0.114c-0.025-0.051,0.369-0.267,0.317-0.331c-0.05-0.063-0.355-0.038-0.521-0.038c-0.166,0-0.305-0.102-0.433-0.127c-0.126-0.025-0.292,0.127-0.418,0.254c-0.128,0.127-0.216,0.038-0.331,0.038c-0.115,0-0.331-0.165-0.331-0.165s-0.216-0.089-0.305-0.089c-0.088,0-0.267-0.165-0.318-0.165c-0.05,0-0.19-0.115-0.088-0.166c0.101-0.05,0.202,0.051,0.101-0.229c-0.101-0.279-0.33-0.216-0.419-0.178c-0.088,0.039-0.724,0.025-0.775,0.025c-0.051,0-0.419,0.127-0.533,0.178c-0.116,0.051-0.318,0.115-0.369,0.14c-0.051,0.025-0.318-0.051-0.433,0.013c-0.151,0.084-0.291,0.216-0.33,0.216c-0.038,0-0.153,0.089-0.229,0.28c-0.077,0.19,0.013,0.355-0.128,0.419c-0.139,0.063-0.394,0.204-0.495,0.305c-0.102,0.101-0.229,0.458-0.355,0.623c-0.127,0.165,0,0.317,0.025,0.419c0.025,0.101,0.114,0.292-0.025,0.471c-0.14,0.178-0.127,0.266-0.191,0.279c-0.063,0.013,0.063,0.063,0.088,0.19c0.025,0.128-0.114,0.255,0.128,0.369c0.241,0.113,0.355,0.217,0.418,0.367c0.064,0.153,0.382,0.407,0.382,0.407s0.229,0.205,0.344,0.293c0.114,0.089,0.152,0.038,0.177-0.05c0.025-0.09,0.178-0.104,0.355-0.104c0.178,0,0.305,0.04,0.483,0.014c0.178-0.025,0.356-0.141,0.42-0.166c0.063-0.025,0.279-0.164,0.443-0.063c0.166,0.103,0.141,0.241,0.23,0.332c0.088,0.088,0.24,0.037,0.355-0.051c0.114-0.09,0.064-0.052,0.203,0.025c0.14,0.075,0.204,0.151,0.077,0.267c-0.128,0.113-0.051,0.293-0.128,0.47c-0.076,0.178-0.063,0.203,0.077,0.278c0.14,0.076,0.394,0.548,0.47,0.638c0.077,0.088-0.025,0.342,0.064,0.495c0.089,0.151,0.178,0.254,0.077,0.331c-0.103,0.075-0.28,0.216-0.292,0.47s0.051,0.431,0.102,0.521s0.177,0.331,0.241,0.419c0.064,0.089,0.14,0.305,0.152,0.445c0.013,0.14-0.024,0.306,0.039,0.381c0.064,0.076,0.102,0.191,0.216,0.292c0.115,0.103,0.152,0.318,0.152,0.318s0.039,0.089,0.051,0.229c0.012,0.14,0.025,0.228,0.152,0.292c0.126,0.063,0.215,0.076,0.28,0.013c0.063-0.063,0.381-0.077,0.546-0.063c0.165,0.013,0.355-0.075,0.521-0.19s0.407-0.419,0.496-0.508c0.089-0.09,0.292-0.255,0.268-0.356c-0.025-0.101-0.077-0.203,0.024-0.254c0.102-0.052,0.344-0.152,0.356-0.229c0.013-0.077-0.09-0.395-0.115-0.457c-0.024-0.064,0.064-0.18,0.165-0.306c0.103-0.128,0.421-0.216,0.471-0.267c0.051-0.053,0.191-0.267,0.217-0.433c0.024-0.167-0.051-0.369,0-0.457c0.05-0.09,0.013-0.165-0.103-0.268c-0.114-0.102-0.089-0.407-0.127-0.457c-0.037-0.051-0.013-0.319,0.063-0.345c0.076-0.023,0.242-0.279,0.344-0.393c0.102-0.114,0.394-0.47,0.534-0.496c0.139-0.025,0.355-0.229,0.368-0.343c0.013-0.115,0.38-0.547,0.394-0.635c0.013-0.09,0.166-0.42,0.102-0.497c-0.062-0.076-0.559,0.115-0.622,0.141c-0.064,0.025-0.241,0.127-0.446,0.113c-0.202-0.013-0.114-0.177-0.127-0.254c-0.012-0.076-0.228-0.368-0.279-0.381c-0.051-0.012-0.203-0.166-0.267-0.317c-0.063-0.153-0.152-0.343-0.254-0.458c-0.102-0.114-0.165-0.38-0.268-0.559c-0.101-0.178-0.189-0.407-0.279-0.572c-0.021-0.041-0.045-0.079-0.067-0.117c0.118-0.029,0.289-0.082,0.31-0.009c0.024,0.088,0.165,0.279,0.19,0.419s0.165,0.089,0.178,0.216c0.014,0.128,0.14,0.433,0.19,0.47c0.052,0.038,0.28,0.242,0.318,0.318c0.038,0.076,0.089,0.178,0.127,0.369c0.038,0.19,0.076,0.444,0.179,0.482c0.102,0.038,0.444-0.064,0.508-0.102s0.482-0.242,0.635-0.255c0.153-0.012,0.179-0.115,0.368-0.152c0.191-0.038,0.331-0.177,0.458-0.28c0.127-0.101,0.28-0.355,0.33-0.444c0.052-0.088,0.179-0.152,0.115-0.253c-0.063-0.103-0.331-0.254-0.433-0.268c-0.102-0.012-0.089-0.178-0.152-0.178s-0.051,0.088-0.178,0.153c-0.127,0.063-0.255,0.19-0.344,0.165s0.026-0.089-0.113-0.203s-0.192-0.14-0.192-0.228c0-0.089-0.278-0.255-0.304-0.382c-0.026-0.127,0.19-0.305,0.254-0.19c0.063,0.114,0.115,0.292,0.279,0.368c0.165,0.076,0.318,0.204,0.395,0.229c0.076,0.025,0.267-0.14,0.33-0.114c0.063,0.024,0.191,0.253,0.306,0.292c0.113,0.038,0.495,0.051,0.559,0.051s0.33,0.013,0.381-0.063c0.051-0.076,0.089-0.076,0.153-0.076c0.062,0,0.177,0.229,0.267,0.254c0.089,0.025,0.254,0.013,0.241,0.179c-0.012,0.164,0.076,0.305,0.165,0.317c0.09,0.012,0.293-0.191,0.293-0.191s0,0.318-0.012,0.433c-0.014,0.113,0.139,0.534,0.139,0.534s0.19,0.393,0.241,0.482s0.267,0.355,0.267,0.47c0,0.115,0.025,0.293,0.103,0.293c0.076,0,0.152-0.203,0.24-0.331c0.091-0.126,0.116-0.305,0.153-0.432c0.038-0.127,0.038-0.356,0.038-0.444c0-0.09,0.075-0.166,0.255-0.242c0.178-0.076,0.304-0.292,0.456-0.407c0.153-0.115,0.141-0.305,0.446-0.305c0.305,0,0.278,0,0.355-0.077c0.076-0.076,0.151-0.127,0.19,0.013c0.038,0.14,0.254,0.343,0.292,0.394c0.038,0.052,0.114,0.191,0.103,0.344c-0.013,0.152,0.012,0.33,0.075,0.33s0.191-0.216,0.191-0.216s0.279-0.189,0.267,0.013c-0.014,0.203,0.025,0.419,0.025,0.545c0,0.053,0.042,0.135,0.088,0.21c-0.005,0.059-0.004,0.119-0.009,0.178C27.388,17.153,27.387,17.327,27.436,17.39zM20.382,12.064c0.076,0.05,0.102,0.127,0.152,0.203c0.052,0.076,0.14,0.05,0.203,0.114c0.063,0.064-0.178,0.14-0.075,0.216c0.101,0.077,0.151,0.381,0.165,0.458c0.013,0.076-0.279,0.114-0.369,0.102c-0.089-0.013-0.354-0.102-0.445-0.127c-0.089-0.026-0.139-0.343-0.025-0.331c0.116,0.013,0.141-0.025,0.267-0.139c0.128-0.115-0.189-0.166-0.278-0.191c-0.089-0.025-0.268-0.305-0.331-0.394c-0.062-0.089-0.014-0.228,0.141-0.331c0.076-0.051,0.279,0.063,0.381,0c0.101-0.063,0.203-0.14,0.241-0.165c0.039-0.025,0.293,0.038,0.33,0.114c0.039,0.076,0.191,0.191,0.141,0.229c-0.052,0.038-0.281,0.076-0.356,0c-0.075-0.077-0.255,0.012-0.268,0.152C20.242,12.115,20.307,12.013,20.382,12.064zM16.875,12.28c-0.077-0.025,0.025-0.178,0.102-0.229c0.075-0.051,0.164-0.178,0.241-0.305c0.076-0.127,0.178-0.14,0.241-0.127c0.063,0.013,0.203,0.241,0.241,0.318c0.038,0.076,0.165-0.026,0.217-0.051c0.05-0.025,0.127-0.102,0.14-0.165s0.127-0.102,0.254-0.102s0.013,0.102-0.076,0.127c-0.09,0.025-0.038,0.077,0.113,0.127c0.153,0.051,0.293,0.191,0.459,0.279c0.165,0.089,0.19,0.267,0.088,0.292c-0.101,0.025-0.406,0.051-0.521,0.038c-0.114-0.013-0.254-0.127-0.419-0.153c-0.165-0.025-0.369-0.013-0.433,0.077s-0.292,0.05-0.395,0.05c-0.102,0-0.228,0.127-0.253,0.077C16.875,12.534,16.951,12.306,16.875,12.28zM17.307,9.458c0.063-0.178,0.419,0.038,0.355,0.127C17.599,9.675,17.264,9.579,17.307,9.458zM17.802,18.584c0.063,0.102-0.14,0.431-0.254,0.407c-0.113-0.027-0.076-0.318-0.038-0.382C17.548,18.545,17.769,18.529,17.802,18.584zM13.189,12.674c0.025-0.051-0.039-0.153-0.127-0.013C13.032,12.71,13.164,12.725,13.189,12.674zM20.813,8.035c0.141,0.076,0.339,0.107,0.433,0.013c0.076-0.076,0.013-0.204-0.05-0.216c-0.064-0.013-0.104-0.115,0.062-0.203c0.165-0.089,0.343-0.204,0.534-0.229c0.19-0.025,0.622-0.038,0.774,0c0.152,0.039,0.382-0.166,0.445-0.254s-0.203-0.152-0.279-0.051c-0.077,0.102-0.444,0.076-0.521,0.051c-0.076-0.025-0.686,0.102-0.812,0.102c-0.128,0-0.179,0.152-0.356,0.229c-0.179,0.076-0.42,0.191-0.509,0.229c-0.088,0.038-0.177,0.19-0.101,0.216C20.509,7.947,20.674,7.959,20.813,8.035zM14.142,12.674c0.064-0.089-0.051-0.217-0.114-0.217c-0.12,0-0.178,0.191-0.103,0.254C14.002,12.776,14.078,12.763,14.142,12.674zM14.714,13.017c0.064,0.025,0.114,0.102,0.165,0.114c0.052,0.013,0.217,0,0.167-0.127s-0.167-0.127-0.204-0.127c-0.038,0-0.203-0.038-0.267,0C14.528,12.905,14.65,12.992,14.714,13.017zM11.308,10.958c0.101,0.013,0.217-0.063,0.305-0.101c0.088-0.038,0.216-0.114,0.216-0.229c0-0.114-0.025-0.216-0.077-0.267c-0.051-0.051-0.14-0.064-0.216-0.051c-0.115,0.02-0.127,0.14-0.203,0.14c-0.076,0-0.165,0.025-0.14,0.114s0.077,0.152,0,0.19C11.117,10.793,11.205,10.946,11.308,10.958zM11.931,10.412c0.127,0.051,0.394,0.102,0.292,0.153c-0.102,0.051-0.28,0.19-0.305,0.267s0.216,0.153,0.216,0.153s-0.077,0.089-0.013,0.114c0.063,0.025,0.102-0.089,0.203-0.089c0.101,0,0.304,0.063,0.406,0.063c0.103,0,0.267-0.14,0.254-0.229c-0.013-0.089-0.14-0.229-0.254-0.28c-0.113-0.051-0.241-0.28-0.317-0.331c-0.076-0.051,0.076-0.178-0.013-0.267c-0.09-0.089-0.153-0.076-0.255-0.14c-0.102-0.063-0.191,0.013-0.254,0.089c-0.063,0.076-0.14-0.013-0.217,0.012c-0.102,0.035-0.063,0.166-0.012,0.229C11.714,10.221,11.804,10.361,11.931,10.412zM24.729,17.198c-0.083,0.037-0.153,0.47,0,0.521c0.152,0.052,0.241-0.202,0.191-0.267C24.868,17.39,24.843,17.147,24.729,17.198zM20.114,20.464c-0.159-0.045-0.177,0.166-0.304,0.306c-0.128,0.141-0.267,0.254-0.317,0.241c-0.052-0.013-0.331,0.089-0.242,0.279c0.089,0.191,0.076,0.382-0.013,0.472c-0.089,0.088,0.076,0.342,0.052,0.482c-0.026,0.139,0.037,0.229,0.215,0.229s0.242-0.064,0.318-0.229c0.076-0.166,0.088-0.331,0.164-0.47c0.077-0.141,0.141-0.434,0.179-0.51c0.038-0.075,0.114-0.316,0.102-0.457C20.254,20.669,20.204,20.489,20.114,20.464zM10.391,8.802c-0.069-0.06-0.229-0.102-0.306-0.11c-0.076-0.008-0.152,0.06-0.321,0.06c-0.168,0-0.279,0.067-0.347,0C9.349,8.684,9.068,8.65,9.042,8.692C9.008,8.749,8.941,8.751,9.008,8.87c0.069,0.118,0.12,0.186,0.179,0.178s0.262-0.017,0.288,0.051C9.5,9.167,9.569,9.226,9.712,9.184c0.145-0.042,0.263-0.068,0.296-0.119c0.033-0.051,0.263-0.059,0.263-0.059S10.458,8.861,10.391,8.802z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Globe

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Globe();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Globe = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Globe",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,1.466C7.973,1.466,1.466,7.973,1.466,16c0,8.027,6.507,14.534,14.534,14.534c8.027,0,14.534-6.507,14.534-14.534C30.534,7.973,24.027,1.466,16,1.466zM19.158,23.269c-0.079,0.064-0.183,0.13-0.105,0.207c0.078,0.078-0.09,0.131-0.09,0.17s0.104,0.246,0.052,0.336c-0.052,0.092-0.091,0.223-0.13,0.301c-0.039,0.077-0.131,0.155-0.104,0.272c0.025,0.116-0.104,0.077-0.104,0.194c0,0.116,0.116,0.065,0.09,0.208c-0.025,0.144-0.09,0.183-0.09,0.285c0,0.104,0.064,0.247,0.064,0.286s-0.064,0.17-0.155,0.272c-0.092,0.104-0.155,0.17-0.144,0.233c0.014,0.065,0.104,0.144,0.091,0.184c-0.013,0.037-0.129,0.168-0.116,0.259c0.014,0.09,0.129,0.053,0.155,0.116c0.026,0.065-0.155,0.118-0.078,0.183c0.078,0.064,0.183,0.051,0.156,0.208c-0.019,0.112,0.064,0.163,0.126,0.198c-0.891,0.221-1.818,0.352-2.777,0.352C9.639,27.533,4.466,22.36,4.466,16c0-2.073,0.557-4.015,1.518-5.697c0.079-0.042,0.137-0.069,0.171-0.062c0.065,0.013,0.079,0.104,0.183,0.13c0.104,0.026,0.195-0.078,0.26-0.117c0.064-0.039,0.116-0.195,0.051-0.182c-0.065,0.013-0.234,0-0.234,0s0.183-0.104,0.183-0.169s0.025-0.169,0.129-0.208C6.83,9.655,6.83,9.681,6.765,9.837C6.7,9.993,6.896,9.928,6.973,9.863s0.13-0.013,0.272-0.104c0.143-0.091,0.143-0.143,0.221-0.143c0.078,0,0.221,0.143,0.299,0.091c0.077-0.052,0.299,0.065,0.429,0.065c0.129,0,0.545,0.169,0.624,0.169c0.078,0,0.312,0.09,0.325,0.259c0.013,0.169,0.09,0.156,0.168,0.156s0.26,0.065,0.26,0.13c0,0.065-0.052,0.325,0.078,0.39c0.129,0.064,0.247,0.169,0.299,0.143c0.052-0.026,0-0.233-0.064-0.26c-0.065-0.026-0.027-0.117-0.052-0.169c-0.026-0.051,0.078-0.051,0.117,0.039c0.039,0.091,0.143,0.26,0.208,0.26c0.064,0,0.208,0.156,0.168,0.247c-0.039,0.091,0.039,0.221,0.156,0.221c0.116,0,0.26,0.182,0.312,0.195c0.052,0.013,0.117,0.078,0.117,0.117c0,0.04,0.065,0.26,0.065,0.351c0,0.09-0.04,0.454-0.053,0.597s0.104,0.39,0.234,0.52c0.129,0.13,0.246,0.377,0.324,0.429c0.079,0.052,0.13,0.195,0.247,0.182c0.117-0.013,0.195,0.078,0.299,0.26c0.104,0.182,0.208,0.48,0.286,0.506c0.078,0.026,0.208,0.117,0.142,0.182c-0.064,0.064-0.168,0.208-0.051,0.208c0.117,0,0.156-0.065,0.247,0.053c0.09,0.116,0.208,0.181,0.194,0.26c-0.013,0.077,0.104,0.103,0.156,0.116c0.052,0.013,0.169,0.247,0.286,0.143c0.117-0.104-0.155-0.259-0.234-0.326c-0.078-0.064,0-0.207-0.182-0.35c-0.182-0.143-0.156-0.247-0.286-0.351c-0.13-0.104-0.233-0.195-0.104-0.286c0.13-0.091,0.143,0.091,0.195,0.208c0.052,0.116,0.324,0.351,0.441,0.454c0.117,0.104,0.326,0.468,0.39,0.468s0.247,0.208,0.247,0.208s0.103,0.168,0.064,0.22c-0.039,0.052,0.053,0.247,0.144,0.299c0.09,0.052,0.455,0.22,0.507,0.247c0.052,0.027,0.155,0.221,0.299,0.221c0.142,0,0.247,0.014,0.286,0.053c0.039,0.038,0.155,0.194,0.234,0.104c0.078-0.092,0.09-0.131,0.208-0.131c0.117,0,0.168,0.091,0.233,0.156c0.065,0.065,0.247,0.235,0.338,0.222c0.091-0.013,0.208,0.104,0.273,0.064s0.169,0.025,0.22,0.052c0.054,0.026,0.234,0.118,0.222,0.272c-0.013,0.157,0.103,0.195,0.182,0.234c0.078,0.039,0.182,0.13,0.248,0.195c0.064,0.063,0.206,0.077,0.246,0.116c0.039,0.039,0.065,0.117,0.182,0.052c0.116-0.064,0.092-0.181,0.092-0.181s0.129-0.026,0.194,0.026c0.064,0.05,0.104,0.22,0.144,0.246c0.038,0.026,0.115,0.221,0.063,0.362c-0.051,0.145-0.038,0.286-0.091,0.286c-0.052,0-0.116,0.17-0.195,0.209c-0.076,0.039-0.285,0.221-0.272,0.286c0.013,0.063,0.131,0.258,0.104,0.35c-0.025,0.091-0.194,0.195-0.154,0.338c0.038,0.144,0.312,0.183,0.323,0.312c0.014,0.131,0.209,0.417,0.235,0.546c0.025,0.13,0.246,0.272,0.246,0.453c0,0.184,0.312,0.3,0.377,0.312c0.063,0.013,0.182,0.131,0.272,0.17s0.169,0.116,0.233,0.221s0.053,0.261,0.053,0.299c0,0.039-0.039,0.44-0.078,0.674C19.145,23.021,19.235,23.203,19.158,23.269zM10.766,11.188c0.039,0.013,0.117,0.091,0.156,0.091c0.04,0,0.234,0.156,0.286,0.208c0.053,0.052,0.053,0.195-0.013,0.208s-0.104-0.143-0.117-0.208c-0.013-0.065-0.143-0.065-0.208-0.104C10.805,11.344,10.66,11.152,10.766,11.188zM27.51,16.41c-0.144,0.182-0.13,0.272-0.195,0.286c-0.064,0.013,0.065,0.065,0.09,0.194c0.022,0.112-0.065,0.224,0.063,0.327c-0.486,4.619-3.71,8.434-8.016,9.787c-0.007-0.011-0.019-0.025-0.021-0.034c-0.027-0.078-0.027-0.233,0.064-0.285c0.091-0.053,0.312-0.233,0.363-0.272c0.052-0.04,0.13-0.221,0.091-0.247c-0.038-0.026-0.232,0-0.26-0.039c-0.026-0.039-0.026-0.092,0.104-0.182c0.13-0.091,0.195-0.222,0.247-0.26c0.052-0.039,0.155-0.117,0.195-0.209c0.038-0.09-0.041-0.039-0.118-0.039s-0.117-0.142-0.117-0.207s0.195,0.026,0.339,0.052c0.143,0.024,0.077-0.065,0.064-0.142c-0.013-0.078,0.026-0.209,0.105-0.17c0.076,0.039,0.479-0.013,0.531-0.026c0.052-0.013,0.194-0.246,0.246-0.312c0.053-0.065,0.064-0.129,0-0.168c-0.065-0.04-0.143-0.184-0.168-0.221c-0.026-0.041-0.039-0.274-0.013-0.34c0.025-0.063,0,0.377,0.181,0.43c0.183,0.052,0.286,0.078,0.455-0.078c0.169-0.155,0.298-0.26,0.312-0.363c0.013-0.104,0.052-0.209,0.117-0.246c0.065-0.039,0.104,0.103,0.182-0.065c0.078-0.17,0.156-0.157,0.234-0.299c0.077-0.144-0.13-0.325,0.024-0.43c0.157-0.103,0.43-0.233,0.43-0.233s0.078-0.039,0.234-0.078c0.155-0.038,0.324-0.014,0.376-0.09c0.052-0.079,0.104-0.247,0.182-0.338c0.079-0.092,0.169-0.234,0.13-0.299c-0.039-0.065,0.104-0.352,0.091-0.429c-0.013-0.078-0.13-0.261,0.065-0.416s0.402-0.391,0.416-0.454c0.012-0.065,0.169-0.338,0.154-0.469c-0.012-0.129-0.154-0.285-0.245-0.325c-0.092-0.037-0.286-0.05-0.364-0.154s-0.299-0.208-0.377-0.182c-0.077,0.026-0.208,0.051-0.312-0.015c-0.104-0.063-0.272-0.143-0.337-0.194c-0.066-0.051-0.234-0.09-0.312-0.09s-0.065-0.053-0.182,0.103c-0.117,0.157,0,0.209-0.208,0.182c-0.209-0.024,0.025-0.038,0.144-0.194c0.115-0.155-0.014-0.247-0.144-0.207c-0.13,0.039-0.039,0.117-0.247,0.156c-0.207,0.038-0.207-0.092-0.077-0.117c0.13-0.026,0.363-0.143,0.363-0.194c0-0.053-0.026-0.196-0.13-0.196s-0.078-0.129-0.233-0.297c-0.156-0.17-0.351-0.274-0.508-0.249c-0.154,0.026-0.272,0.065-0.35-0.076c-0.078-0.144-0.169-0.17-0.222-0.247c-0.051-0.078-0.182,0-0.221-0.039s-0.039-0.039-0.039-0.039s-0.169,0.039-0.077-0.078c0.09-0.117,0.129-0.338,0.09-0.325c-0.038,0.013-0.104,0.196-0.168,0.183c-0.064-0.013-0.014-0.04-0.144-0.117c-0.13-0.078-0.337-0.013-0.337,0.052c0,0.065-0.065,0.117-0.065,0.117s-0.039-0.038-0.078-0.117c-0.039-0.078-0.221-0.091-0.312-0.013c-0.09,0.078-0.142-0.196-0.207-0.196s-0.194,0.065-0.26,0.184c-0.064,0.116-0.038,0.285-0.092,0.272c-0.05-0.013-0.063-0.233-0.05-0.312c0.012-0.079,0.155-0.208,0.05-0.234c-0.103-0.026-0.259,0.13-0.323,0.143c-0.065,0.013-0.195,0.104-0.273,0.209c-0.077,0.103-0.116,0.168-0.195,0.207c-0.077,0.039-0.193,0-0.167-0.039c0.025-0.039-0.222-0.181-0.261-0.13c-0.04,0.052-0.155,0.091-0.272,0.144c-0.117,0.052-0.222-0.065-0.247-0.117s-0.079-0.064-0.091-0.234c-0.013-0.168,0.027-0.351,0.065-0.454c0.038-0.104-0.195-0.312-0.286-0.3c-0.091,0.015-0.182,0.105-0.272,0.091c-0.092-0.012-0.052-0.038-0.195-0.038c-0.143,0-0.026-0.025,0-0.143c0.025-0.116-0.052-0.273,0.092-0.377c0.142-0.104,0.091-0.351,0-0.363c-0.092-0.014-0.261,0.039-0.377,0.026c-0.116-0.014-0.208,0.091-0.169,0.207c0.039,0.117-0.065,0.195-0.104,0.183c-0.039-0.013-0.09-0.078-0.234,0.026c-0.142,0.103-0.194,0.064-0.337-0.052c-0.143-0.118-0.299-0.234-0.325-0.416c-0.026-0.182-0.04-0.364,0.013-0.468c0.051-0.104,0.051-0.285-0.026-0.312c-0.078-0.025,0.09-0.155,0.181-0.181c0.092-0.026,0.234-0.143,0.26-0.195c0.026-0.052,0.156-0.04,0.298-0.04c0.143,0,0.169,0,0.312,0.078c0.143,0.078,0.169-0.039,0.169-0.078c0-0.039,0.052-0.117,0.208-0.104c0.156,0.013,0.376-0.052,0.416-0.013s0.116,0.195,0.194,0.143c0.079-0.051,0.104-0.143,0.131,0.014c0.025,0.155,0.09,0.39,0.208,0.429c0.116,0.039,0.052,0.194,0.168,0.207c0.115,0.013,0.17-0.246,0.131-0.337c-0.04-0.09-0.118-0.363-0.183-0.428c-0.064-0.065-0.064-0.234,0.064-0.286c0.13-0.052,0.442-0.312,0.532-0.389c0.092-0.079,0.338-0.144,0.261-0.248c-0.078-0.104-0.104-0.168-0.104-0.247s0.078-0.052,0.117,0s0.194-0.078,0.155-0.143c-0.038-0.064-0.026-0.155,0.065-0.143c0.091,0.013,0.116-0.065,0.078-0.117c-0.039-0.052,0.091-0.117,0.182-0.091c0.092,0.026,0.325-0.013,0.364-0.065c0.038-0.052-0.078-0.104-0.078-0.208c0-0.104,0.155-0.195,0.247-0.208c0.091-0.013,0.207,0,0.221-0.039c0.012-0.039,0.143-0.143,0.155-0.052c0.014,0.091,0,0.247,0.104,0.247c0.104,0,0.232-0.117,0.272-0.129c0.038-0.013,0.286-0.065,0.338-0.078c0.052-0.013,0.363-0.039,0.325-0.13c-0.039-0.09-0.078-0.181-0.118-0.22c-0.039-0.039-0.077,0.013-0.13,0.078c-0.051,0.065-0.143,0.065-0.168,0.013c-0.026-0.051,0.012-0.207-0.078-0.156c-0.092,0.052-0.104,0.104-0.157,0.078c-0.052-0.026-0.103-0.117-0.103-0.117s0.129-0.064,0.038-0.182c-0.09-0.117-0.221-0.091-0.35-0.025c-0.13,0.064-0.118,0.051-0.273,0.09s-0.234,0.078-0.234,0.078s0.209-0.129,0.299-0.208c0.091-0.078,0.209-0.117,0.286-0.195c0.078-0.078,0.285,0.039,0.285,0.039s0.105-0.104,0.105-0.039s-0.027,0.234,0.051,0.234c0.079,0,0.299-0.104,0.21-0.131c-0.093-0.026,0.129,0,0.219-0.065c0.092-0.065,0.194-0.065,0.247-0.09c0.052-0.026,0.092-0.143,0.182-0.143c0.092,0,0.13,0.117,0,0.195s-0.143,0.273-0.208,0.325c-0.064,0.052-0.026,0.117,0.078,0.104c0.104-0.013,0.194,0.013,0.286-0.013s0.143,0.026,0.168,0.065c0.026,0.039,0.104-0.039,0.104-0.039s0.169-0.039,0.221,0.026c0.053,0.064,0.092-0.039,0.053-0.104c-0.039-0.064-0.092-0.129-0.13-0.208c-0.039-0.078-0.091-0.104-0.194-0.078c-0.104,0.026-0.13-0.026-0.195-0.064c-0.065-0.04-0.118,0.052-0.065-0.04c0.053-0.09,0.078-0.117,0.117-0.195c0.039-0.078,0.209-0.221,0.039-0.259c-0.169-0.04-0.222-0.065-0.247-0.143c-0.026-0.078-0.221-0.221-0.272-0.221c-0.053,0-0.233,0-0.247-0.065c-0.013-0.065-0.143-0.208-0.208-0.273c-0.064-0.065-0.312-0.351-0.351-0.377c-0.039-0.026-0.091-0.013-0.208,0.143c-0.116,0.157-0.22,0.183-0.312,0.144c-0.091-0.039-0.104-0.026-0.193-0.13c-0.093-0.104,0.09-0.117,0.051-0.182c-0.04-0.064-0.247-0.091-0.377-0.104c-0.13-0.013-0.221-0.156-0.416-0.169c-0.194-0.013-0.428,0.026-0.493,0.026c-0.064,0-0.064,0.091-0.09,0.234c-0.027,0.143,0.09,0.182-0.027,0.208c-0.116,0.026-0.169,0.039-0.052,0.091c0.117,0.052,0.273,0.26,0.273,0.26s0,0.117-0.092,0.182c-0.09,0.065-0.182,0.13-0.233,0.053c-0.053-0.079-0.195-0.065-0.155,0.013c0.038,0.078,0.116,0.117,0.116,0.195c0,0.077,0.117,0.272,0.039,0.337c-0.078,0.065-0.168,0.014-0.233,0.026s-0.131-0.104-0.078-0.13c0.051-0.026-0.014-0.221-0.014-0.221s-0.155,0.221-0.143,0.104c0.014-0.117-0.064-0.13-0.064-0.221c0-0.091-0.079-0.13-0.194-0.104c-0.118,0.026-0.26-0.04-0.482-0.079c-0.22-0.039-0.311-0.064-0.493-0.156c-0.182-0.091-0.247-0.026-0.338-0.013c-0.091,0.013-0.052-0.182-0.169-0.207c-0.116-0.027-0.181,0.025-0.207-0.144c-0.026-0.168,0.039-0.208,0.324-0.39c0.286-0.182,0.247-0.26,0.468-0.286c0.22-0.026,0.325,0.026,0.325-0.039s0.052-0.325,0.052-0.195S16.95,9.109,16.832,9.2c-0.116,0.091-0.052,0.104,0.04,0.104c0.091,0,0.259-0.091,0.259-0.091s0.208-0.091,0.26-0.013c0.053,0.078,0.053,0.156,0.144,0.156s0.285-0.104,0.116-0.195c-0.168-0.091-0.272-0.078-0.376-0.182s-0.078-0.065-0.195-0.039c-0.116,0.026-0.116-0.039-0.156-0.039s-0.104,0.026-0.13-0.026c-0.025-0.052,0.014-0.065,0.145-0.065c0.129,0,0.285,0.039,0.285,0.039s0.155-0.052,0.194-0.065c0.039-0.013,0.247-0.039,0.208-0.155c-0.04-0.117-0.169-0.117-0.208-0.156s0.078-0.09,0.143-0.117c0.065-0.026,0.247,0,0.247,0s0.117,0.013,0.117-0.039S17.897,8.2,17.976,8.239s0,0.156,0.117,0.13c0.116-0.026,0.143,0,0.207,0.039c0.065,0.039-0.013,0.195-0.077,0.221c-0.065,0.025-0.169,0.077-0.026,0.09c0.144,0.014,0.246,0.014,0.246,0.014s0.092-0.091,0.131-0.169c0.038-0.078,0.104-0.026,0.155,0c0.052,0.025,0.247,0.065,0.065,0.117c-0.183,0.052-0.221,0.117-0.26,0.182c-0.038,0.065-0.053,0.104-0.221,0.065c-0.17-0.039-0.26-0.026-0.299,0.039c-0.039,0.064-0.013,0.273,0.053,0.247c0.063-0.026,0.129-0.026,0.207-0.052c0.078-0.026,0.39,0.026,0.467,0.013c0.078-0.013,0.209,0.13,0.248,0.104c0.039-0.026,0.117,0.052,0.194,0.104c0.078,0.052,0.052-0.117,0.194-0.013c0.144,0.104,0.065,0.104,0.144,0.104c0.076,0,0.246,0.013,0.246,0.013s0.014-0.129,0.144-0.104c0.13,0.026,0.245,0.169,0.232,0.064c-0.012-0.103,0.013-0.181-0.09-0.259c-0.104-0.078-0.272-0.13-0.299-0.169c-0.026-0.039-0.052-0.091-0.013-0.117c0.039-0.025,0.221,0.013,0.324,0.079c0.104,0.065,0.195,0.13,0.273,0.078c0.077-0.052,0.17-0.078,0.208-0.117c0.038-0.04,0.13-0.156,0.13-0.156s-0.391-0.051-0.441-0.117c-0.053-0.065-0.235-0.156-0.287-0.156s-0.194,0.091-0.246-0.039s-0.052-0.286-0.105-0.299c-0.05-0.013-0.597-0.091-0.674-0.13c-0.078-0.039-0.39-0.13-0.507-0.195s-0.286-0.156-0.389-0.156c-0.104,0-0.533,0.052-0.611,0.039c-0.078-0.013-0.312,0.026-0.403,0.039c-0.091,0.013,0.117,0.182-0.077,0.221c-0.195,0.039-0.169,0.065-0.13-0.13c0.038-0.195-0.131-0.247-0.299-0.169c-0.169,0.078-0.442,0.13-0.377,0.221c0.065,0.091-0.012,0.157,0.117,0.247c0.13,0.091,0.183,0.117,0.35,0.104c0.17-0.013,0.339,0.025,0.339,0.025s0,0.157-0.064,0.182c-0.065,0.026-0.169,0.026-0.196,0.104c-0.025,0.078-0.155,0.117-0.155,0.078s0.065-0.169-0.026-0.234c-0.09-0.065-0.117-0.078-0.221-0.013c-0.104,0.065-0.116,0.091-0.169-0.013C16.053,8.291,15.897,8.2,15.897,8.2s-0.104-0.129-0.182-0.194c-0.077-0.065-0.22-0.052-0.234,0.013c-0.013,0.064,0.026,0.129,0.078,0.247c0.052,0.117,0.104,0.337,0.013,0.351c-0.091,0.013-0.104,0.026-0.195,0.052c-0.091,0.026-0.13-0.039-0.13-0.143s-0.04-0.195-0.013-0.234c0.026-0.039-0.104,0.027-0.234,0c-0.13-0.025-0.233,0.052-0.104,0.092c0.13,0.039,0.157,0.194,0.039,0.233c-0.117,0.039-0.559,0-0.702,0s-0.35,0.039-0.39-0.039c-0.039-0.078,0.118-0.129,0.208-0.129c0.091,0,0.363,0.012,0.467-0.13c0.104-0.143-0.13-0.169-0.233-0.169c-0.104,0-0.183-0.039-0.299-0.155c-0.118-0.117,0.078-0.195,0.052-0.247c-0.026-0.052-0.156-0.014-0.272-0.014c-0.117,0-0.299-0.09-0.299,0.014c0,0.104,0.143,0.402,0.052,0.337c-0.091-0.064-0.078-0.156-0.143-0.234c-0.065-0.078-0.168-0.065-0.299-0.052c-0.129,0.013-0.35,0.052-0.415,0.039c-0.064-0.013-0.013-0.013-0.156-0.078c-0.142-0.065-0.208-0.052-0.312-0.117C12.091,7.576,12.182,7.551,12,7.538c-0.181-0.013-0.168,0.09-0.35,0.065c-0.182-0.026-0.234,0.013-0.416,0c-0.182-0.013-0.272-0.026-0.299,0.065c-0.025,0.091-0.078,0.247-0.156,0.247c-0.077,0-0.169,0.091,0.078,0.104c0.247,0.013,0.105,0.129,0.325,0.117c0.221-0.013,0.416-0.013,0.468-0.117c0.052-0.104,0.091-0.104,0.117-0.065c0.025,0.039,0.22,0.272,0.22,0.272s0.131,0.104,0.183,0.13c0.051,0.026-0.052,0.143-0.156,0.078c-0.104-0.065-0.299-0.051-0.377-0.116c-0.078-0.065-0.429-0.065-0.52-0.052c-0.09,0.013-0.247-0.039-0.299-0.039c-0.051,0-0.221,0.13-0.221,0.13S10.532,8.252,10.494,8.2c-0.039-0.052-0.104,0.052-0.156,0.065c-0.052,0.013-0.208-0.104-0.364-0.052C9.818,8.265,9.87,8.317,9.649,8.304s-0.272-0.052-0.35-0.039C9.22,8.278,9.22,8.278,9.22,8.278S9.233,8.33,9.143,8.382C9.052,8.434,8.986,8.499,8.921,8.421C8.857,8.343,8.818,8.343,8.779,8.33c-0.04-0.013-0.118-0.078-0.286-0.04C8.324,8.33,8.064,8.239,8.013,8.239c-0.04,0-0.313-0.015-0.491-0.033c2.109-2.292,5.124-3.74,8.478-3.74c2.128,0,4.117,0.589,5.83,1.598c-0.117,0.072-0.319,0.06-0.388,0.023c-0.078-0.043-0.158-0.078-0.475-0.061c-0.317,0.018-0.665,0.122-0.595,0.226c0.072,0.104-0.142,0.165-0.197,0.113c-0.055-0.052-0.309,0.06-0.293,0.165c0.016,0.104-0.039,0.225-0.175,0.199c-0.134-0.027-0.229,0.06-0.237,0.146c-0.007,0.087-0.309,0.147-0.332,0.147c-0.024,0-0.412-0.008-0.27,0.095c0.097,0.069,0.15,0.027,0.27,0.052c0.119,0.026,0.214,0.217,0.277,0.243c0.062,0.026,0.15,0,0.189-0.052c0.04-0.052,0.095-0.234,0.095-0.234s0,0.173,0.097,0.208c0.095,0.035,0.331-0.026,0.395-0.017c0.064,0.008,0.437,0.061,0.538,0.112c0.104,0.052,0.356,0.087,0.428,0.199c0.071,0.113,0.08,0.503,0.119,0.546c0.04,0.043,0.174-0.139,0.205-0.182c0.031-0.044,0.198-0.018,0.254,0.042c0.056,0.061,0.182,0.208,0.175,0.269C21.9,8.365,21.877,8.459,21.83,8.425c-0.048-0.034-0.127-0.025-0.096-0.095c0.032-0.069,0.048-0.217-0.015-0.217c-0.064,0-0.119,0-0.119,0s-0.12-0.035-0.199,0.095s-0.015,0.26,0.04,0.26s0.184,0,0.184,0.034c0,0.035-0.136,0.139-0.128,0.2c0.009,0.061,0.11,0.268,0.144,0.312c0.031,0.043,0.197,0.086,0.244,0.096c0.049,0.008-0.111,0.017-0.07,0.077c0.04,0.061,0.102,0.208,0.189,0.243c0.087,0.035,0.333,0.19,0.363,0.26c0.032,0.069,0.222-0.052,0.262-0.061c0.04-0.008,0.032,0.182,0.143,0.191c0.11,0.008,0.15-0.018,0.245-0.096s0.072-0.182,0.079-0.26c0.009-0.078,0-0.138,0.104-0.113c0.104,0.026,0.158-0.018,0.15-0.104c-0.008-0.087-0.095-0.191,0.07-0.217c0.167-0.026,0.254-0.138,0.357-0.138c0.103,0,0.389,0.043,0.419,0c0.032-0.043,0.167-0.243,0.254-0.251c0.067-0.007,0.224-0.021,0.385-0.042c1.582,1.885,2.561,4.284,2.673,6.905c-0.118,0.159-0.012,0.305,0.021,0.408c0.001,0.03,0.005,0.058,0.005,0.088c0,0.136-0.016,0.269-0.021,0.404C27.512,16.406,27.512,16.408,27.51,16.41zM17.794,12.084c-0.064,0.013-0.169-0.052-0.169-0.143s-0.091,0.169-0.04,0.247c0.053,0.078-0.104,0.169-0.155,0.169s-0.091-0.116-0.078-0.233c0.014-0.117-0.077-0.221-0.221-0.208c-0.143,0.014-0.208,0.13-0.259,0.169c-0.053,0.039-0.053,0.259-0.04,0.312s0.013,0.235-0.116,0.221c-0.118-0.013-0.092-0.233-0.079-0.312c0.014-0.078-0.039-0.273,0.014-0.376c0.053-0.104,0.207-0.143,0.312-0.156s0.324,0.065,0.363,0.052c0.04-0.014,0.222-0.014,0.312,0C17.729,11.837,17.858,12.071,17.794,12.084zM18.027,12.123c0.04,0.026,0.311-0.039,0.364,0.026c0.051,0.065-0.054,0.078-0.183,0.13c-0.129,0.052-0.169,0.039-0.221,0.104s-0.221,0.09-0.299,0.168c-0.078,0.079-0.217,0.125-0.246,0.065c-0.04-0.078,0.013-0.039,0.025-0.078c0.013-0.039,0.245-0.129,0.245-0.129S17.988,12.097,18.027,12.123zM16.988,11.668c-0.038,0.013-0.182-0.026-0.3-0.026c-0.116,0-0.091-0.078-0.143-0.064c-0.051,0.013-0.168,0.039-0.247,0.078c-0.078,0.039-0.208,0.03-0.208-0.04c0-0.104,0.052-0.078,0.221-0.143c0.169-0.065,0.352-0.247,0.429-0.169c0.078,0.078,0.221,0.169,0.312,0.182C17.144,11.5,17.026,11.655,16.988,11.668zM15.659,7.637c-0.079,0.026-0.347,0.139-0.321,0.199c0.01,0.023,0.078,0.069,0.19,0.052c0.113-0.018,0.276-0.035,0.355-0.043c0.078-0.009,0.095-0.139,0.009-0.147C15.805,7.689,15.736,7.611,15.659,7.637zM14.698,7.741c-0.061,0.026-0.243-0.043-0.338,0.018c-0.061,0.038-0.026,0.164,0.07,0.172c0.095,0.009,0.259-0.06,0.276-0.008c0.018,0.052,0.078,0.286,0.234,0.208c0.156-0.078,0.147-0.147,0.19-0.156c0.043-0.009-0.008-0.199-0.078-0.243C14.983,7.689,14.758,7.715,14.698,7.741zM14.385,7.005c0.017,0.044-0.008,0.078,0.113,0.095c0.121,0.018,0.173,0.035,0.243,0.035c0.069,0,0.042-0.113-0.018-0.19c-0.061-0.078-0.043-0.069-0.199-0.113c-0.156-0.043-0.312-0.043-0.416-0.035c-0.104,0.009-0.217-0.017-0.243,0.104c-0.013,0.062,0.07,0.112,0.174,0.112S14.368,6.962,14.385,7.005zM14.611,7.481c0.043,0.095,0.043,0.051,0.165,0.061C14.896,7.551,14.991,7.421,15,7.378c0.009-0.044-0.061-0.13-0.225-0.113c-0.165,0.017-0.667-0.026-0.736,0.034c-0.066,0.058,0,0.233-0.026,0.251c-0.026,0.017,0.009,0.095,0.077,0.078c0.069-0.017,0.104-0.182,0.157-0.182C14.299,7.447,14.568,7.386,14.611,7.481zM12.982,7.126c0.052,0.043,0.183,0.008,0.173-0.035c-0.008-0.043,0.053-0.217-0.051-0.225C13,6.858,12.854,6.962,12.697,7.014c-0.101,0.033-0.078,0.13-0.009,0.13S12.931,7.083,12.982,7.126zM13.72,7.282c-0.087,0.043-0.114,0.069-0.191,0.052c-0.078-0.017-0.078-0.156-0.217-0.13c-0.138,0.026-0.164,0.104-0.207,0.139s-0.139,0.061-0.173,0.043c-0.034-0.017-0.234-0.129-0.234-0.129s-0.416-0.018-0.433-0.07c-0.017-0.052-0.086-0.138-0.277-0.121s-0.52,0.13-0.572,0.13c-0.052,0,0.062,0.104-0.009,0.104c-0.069,0-0.155-0.008-0.181,0.069c-0.018,0.053,0.078,0.052,0.189,0.052c0.112,0,0.295,0,0.347-0.026c0.052-0.026,0.312-0.087,0.303-0.009c-0.009,0.079,0.104,0.199,0.164,0.182c0.061-0.017,0.183-0.13,0.243-0.086c0.061,0.043,0.07,0.146,0.13,0.173c0.061,0.025,0.226,0.025,0.304,0c0.077-0.027,0.294-0.027,0.389-0.009c0.095,0.018,0.373,0.069,0.399,0.018c0.026-0.053,0.104-0.061,0.112-0.113s0.051-0.216,0.051-0.216S13.806,7.239,13.72,7.282zM18.105,16.239c-0.119,0.021-0.091,0.252,0.052,0.21C18.3,16.407,18.223,16.217,18.105,16.239zM19.235,15.929c-0.104-0.026-0.221,0-0.299,0.013c-0.078,0.013-0.299,0.208-0.299,0.208s0.143,0.026,0.233,0.026c0.092,0,0.144,0.051,0.221,0.09c0.078,0.04,0.221-0.052,0.272-0.052c0.053,0,0.118,0.156,0.131-0.013C19.508,16.032,19.339,15.955,19.235,15.929zM15.616,7.507c-0.043-0.104-0.259-0.139-0.304-0.035C15.274,7.563,15.659,7.611,15.616,7.507zM18.093,15.292c0.143-0.026,0.064-0.144-0.053-0.13C17.922,15.175,17.949,15.318,18.093,15.292zM19.82,16.095c-0.119,0.022-0.092,0.253,0.051,0.211C20.015,16.264,19.937,16.074,19.82,16.095zM18.247,15.708c-0.09,0.013-0.285-0.09-0.389-0.182c-0.104-0.091-0.299-0.091-0.377-0.091c-0.077,0-0.39,0.091-0.39,0.091c-0.013,0.13,0.117,0.091,0.273,0.091s0.429-0.026,0.479,0.039c0.053,0.064,0.286,0.168,0.352,0.221c0.064,0.052,0.272,0.065,0.285,0.013S18.338,15.695,18.247,15.708zM16.698,7.412c-0.13-0.009-0.295-0.009-0.399,0c-0.104,0.008-0.182-0.069-0.26-0.113c-0.077-0.043-0.251-0.182-0.354-0.199c-0.104-0.017-0.086-0.017-0.303-0.069c-0.11-0.027-0.294-0.061-0.294-0.086c0-0.026-0.052,0.121,0.043,0.165c0.095,0.043,0.251,0.121,0.363,0.164c0.114,0.043,0.329,0.052,0.399,0.139c0.069,0.086,0.303,0.156,0.303,0.156l0.277,0.026c0,0,0.191-0.043,0.39-0.026c0.199,0.017,0.493,0.043,0.659,0.035c0.163-0.008,0.189-0.061,0.208-0.095c0.016-0.035-0.304-0.104-0.383-0.095C17.271,7.42,16.827,7.42,16.698,7.412zM17.182,9.404c-0.034,0.039,0.157,0.095,0.191,0.043C17.407,9.396,17.271,9.309,17.182,9.404zM17.764,9.585c0.086-0.035,0.043-0.139-0.079-0.104C17.547,9.521,17.676,9.62,17.764,9.585z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Warning

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Warning();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Warning = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Warning",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M29.225,23.567l-3.778-6.542c-1.139-1.972-3.002-5.2-4.141-7.172l-3.778-6.542c-1.14-1.973-3.003-1.973-4.142,0L9.609,9.853c-1.139,1.972-3.003,5.201-4.142,7.172L1.69,23.567c-1.139,1.974-0.207,3.587,2.071,3.587h23.391C29.432,27.154,30.363,25.541,29.225,23.567zM16.536,24.58h-2.241v-2.151h2.241V24.58zM16.428,20.844h-2.023l-0.201-9.204h2.407L16.428,20.844z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Code

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Code();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Code = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Code",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M8.982,7.107L0.322,15.77l8.661,8.662l3.15-3.15L6.621,15.77l5.511-5.511L8.982,7.107zM21.657,7.107l-3.148,3.151l5.511,5.511l-5.511,5.511l3.148,3.15l8.662-8.662L21.657,7.107z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Pensil

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Pensil();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Pensil = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Pensil",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.31,2.872l-3.384-2.127c-0.854-0.536-1.979-0.278-2.517,0.576l-1.334,2.123l6.474,4.066l1.335-2.122C26.42,4.533,26.164,3.407,25.31,2.872zM6.555,21.786l6.474,4.066L23.581,9.054l-6.477-4.067L6.555,21.786zM5.566,26.952l-0.143,3.819l3.379-1.787l3.14-1.658l-6.246-3.925L5.566,26.952z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Pen

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Pen();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Pen = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Pen",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M13.587,12.074c-0.049-0.074-0.11-0.147-0.188-0.202c-0.333-0.243-0.803-0.169-1.047,0.166c-0.244,0.336-0.167,0.805,0.167,1.048c0.303,0.22,0.708,0.167,0.966-0.091l-7.086,9.768l-2.203,7.997l6.917-4.577L26.865,4.468l-4.716-3.42l-1.52,2.096c-0.087-0.349-0.281-0.676-0.596-0.907c-0.73-0.529-1.751-0.369-2.28,0.363C14.721,6.782,16.402,7.896,13.587,12.074zM10.118,25.148L6.56,27.503l1.133-4.117L10.118,25.148zM14.309,11.861c2.183-3.225,1.975-4.099,3.843-6.962c0.309,0.212,0.664,0.287,1.012,0.269L14.309,11.861z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Plus

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Plus();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Plus = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Plus",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.979,12.896 19.312,12.896 19.312,6.229 12.647,6.229 12.647,12.896 5.979,12.896 5.979,19.562 12.647,19.562 12.647,26.229 19.312,26.229 19.312,19.562 25.979,19.562z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Minus

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Minus();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Minus = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Minus",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.979,12.896,5.979,12.896,5.979,19.562,25.979,19.562z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.TShirt

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.TShirt();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.TShirt = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.TShirt",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M20.1,4.039c-0.681,1.677-2.32,2.862-4.24,2.862c-1.921,0-3.56-1.185-4.24-2.862L1.238,8.442l2.921,6.884l3.208-1.361V28h17.099V14.015l3.093,1.312l2.922-6.884L20.1,4.039z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Sticker

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Sticker();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Sticker = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Sticker",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M15.5,1.999c-1.042,0-1.916,0.377-2.57,1.088L2.895,13.138C2.302,13.784,1.999,14.58,1.999,15.5C1.999,22.943,8.057,29,15.5,29S29,22.943,29,15.5S22.943,1.999,15.5,1.999zM15.5,28C8.596,28,3,22.404,3,15.5c0-3.452,5.239-2.737,7.501-4.999C12.762,8.239,12.048,3,15.5,3C22.404,3,28,8.597,28,15.5S22.404,28,15.5,28z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Page2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Page2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Page2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Page2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M23.024,5.673c-1.744-1.694-3.625-3.051-5.168-3.236c-0.084-0.012-0.171-0.019-0.263-0.021H7.438c-0.162,0-0.322,0.063-0.436,0.18C6.889,2.71,6.822,2.87,6.822,3.033v25.75c0,0.162,0.063,0.317,0.18,0.435c0.117,0.116,0.271,0.179,0.436,0.179h18.364c0.162,0,0.317-0.062,0.434-0.179c0.117-0.117,0.182-0.272,0.182-0.435V11.648C26.382,9.659,24.824,7.49,23.024,5.673zM22.157,6.545c0.805,0.786,1.529,1.676,2.069,2.534c-0.468-0.185-0.959-0.322-1.42-0.431c-1.015-0.228-2.008-0.32-2.625-0.357c0.003-0.133,0.004-0.283,0.004-0.446c0-0.869-0.055-2.108-0.356-3.2c-0.003-0.01-0.005-0.02-0.009-0.03C20.584,5.119,21.416,5.788,22.157,6.545zM25.184,28.164H8.052V3.646h9.542v0.002c0.416-0.025,0.775,0.386,1.05,1.326c0.25,0.895,0.313,2.062,0.312,2.871c0.002,0.593-0.027,0.991-0.027,0.991l-0.049,0.652l0.656,0.007c0.003,0,1.516,0.018,3,0.355c1.426,0.308,2.541,0.922,2.645,1.617c0.004,0.062,0.005,0.124,0.004,0.182V28.164z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Page

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Page();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Page = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Page",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M23.024,5.673c-1.744-1.694-3.625-3.051-5.168-3.236c-0.084-0.012-0.171-0.019-0.263-0.021H7.438c-0.162,0-0.322,0.063-0.436,0.18C6.889,2.71,6.822,2.87,6.822,3.033v25.75c0,0.162,0.063,0.317,0.18,0.435c0.117,0.116,0.271,0.179,0.436,0.179h18.364c0.162,0,0.317-0.062,0.434-0.179c0.117-0.117,0.182-0.272,0.182-0.435V11.648C26.382,9.659,24.824,7.49,23.024,5.673zM25.184,28.164H8.052V3.646h9.542v0.002c0.416-0.025,0.775,0.386,1.05,1.326c0.25,0.895,0.313,2.062,0.312,2.871c0.002,0.593-0.027,0.991-0.027,0.991l-0.049,0.652l0.656,0.007c0.003,0,1.516,0.018,3,0.355c1.426,0.308,2.541,0.922,2.645,1.617c0.004,0.062,0.005,0.124,0.004,0.182V28.164z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Landscape1

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Landscape1();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Landscape1 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Landscape1",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M19.883,5.71H2.746c-0.163,0-0.319,0.071-0.435,0.188c-0.118,0.117-0.18,0.272-0.18,0.435v18.364c0,0.164,0.063,0.318,0.18,0.436c0.123,0.117,0.287,0.18,0.435,0.18h25.75c0.164,0,0.324-0.066,0.438-0.18c0.118-0.114,0.182-0.273,0.182-0.436V14.551c-0.002-0.102-0.01-0.188-0.021-0.271c-0.186-1.543-1.543-3.424-3.236-5.168C24.039,7.31,21.869,5.753,19.883,5.71zM26.914,12.314c-0.008-0.005-0.019-0.007-0.029-0.01c-1.092-0.293-2.33-0.355-3.199-0.355c-0.162,0-0.312,0.002-0.445,0.004c-0.037-0.604-0.129-1.604-0.356-2.625c-0.11-0.461-0.246-0.94-0.433-1.42c0.857,0.541,1.748,1.264,2.535,2.068C25.74,10.718,26.41,11.551,26.914,12.314zM3.365,6.947h16.517c0.058,0,0.12,0,0.183,0.004c0.694,0.105,1.307,1.221,1.616,2.646c0.335,1.484,0.354,2.997,0.354,3l0.007,0.656l0.651-0.051c0,0,0.398-0.027,0.99-0.025c0.809,0,1.977,0.062,2.871,0.312c0.939,0.275,1.352,0.635,1.326,1.051h0.002v9.542H3.365V6.951V6.947z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Landscape2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Landscape2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Landscape2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Landscape2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M19.883,5.71H2.746c-0.163,0-0.319,0.071-0.435,0.188c-0.118,0.117-0.18,0.272-0.18,0.435v18.364c0,0.164,0.063,0.318,0.18,0.436c0.123,0.117,0.287,0.18,0.435,0.18h25.75c0.164,0,0.324-0.066,0.438-0.18c0.118-0.114,0.182-0.273,0.182-0.436V14.551c-0.002-0.102-0.01-0.188-0.021-0.271c-0.186-1.543-1.543-3.424-3.236-5.168C24.039,7.31,21.869,5.753,19.883,5.71zM3.365,6.947h16.517c0.058,0,0.12,0,0.183,0.004c0.694,0.105,1.307,1.221,1.616,2.646c0.335,1.484,0.354,2.997,0.354,3l0.007,0.656l0.651-0.051c0,0,0.398-0.027,0.99-0.025c0.809,0,1.977,0.062,2.871,0.312c0.939,0.275,1.352,0.635,1.326,1.051h0.002v9.542H3.365V6.951V6.947z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Plugin

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Plugin();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Plugin = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Plugin",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M26.33,15.836l-3.893-1.545l3.136-7.9c0.28-0.705-0.064-1.505-0.771-1.785c-0.707-0.28-1.506,0.065-1.785,0.771l-3.136,7.9l-4.88-1.937l3.135-7.9c0.281-0.706-0.064-1.506-0.77-1.786c-0.706-0.279-1.506,0.065-1.785,0.771l-3.136,7.9L8.554,8.781l-1.614,4.066l2.15,0.854l-2.537,6.391c-0.61,1.54,0.143,3.283,1.683,3.895l1.626,0.646L8.985,26.84c-0.407,1.025,0.095,2.188,1.122,2.596l0.93,0.369c1.026,0.408,2.188-0.095,2.596-1.121l0.877-2.207l1.858,0.737c1.54,0.611,3.284-0.142,3.896-1.682l2.535-6.391l1.918,0.761L26.33,15.836z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Bookmark

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Bookmark();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Bookmark = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Bookmark",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M17.396,1.841L6.076,25.986l7.341-4.566l1.186,8.564l11.32-24.146L17.396,1.841zM19.131,9.234c-0.562-0.264-0.805-0.933-0.541-1.495c0.265-0.562,0.934-0.805,1.496-0.541s0.805,0.934,0.541,1.496S19.694,9.498,19.131,9.234z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Hammer

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Hammer();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Hammer = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Hammer",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M7.831,29.354c0.685,0.353,1.62,1.178,2.344,0.876c0.475-0.195,0.753-1.301,1.048-1.883c2.221-4.376,4.635-9.353,6.392-13.611c0-0.19,0.101-0.337-0.049-0.595c0.983-1.6,1.65-3.358,2.724-5.138c0.34-0.566,0.686-1.351,1.163-1.577l0.881-0.368c1.12-0.288,1.938-0.278,2.719,0.473c0.396,0.383,0.578,1.015,0.961,1.395c0.259,0.26,1.246,0.899,1.613,0.8c0.285-0.077,0.52-0.364,0.72-0.728l0.696-1.286c0.195-0.366,0.306-0.718,0.215-0.999c-0.117-0.362-1.192-0.84-1.552-0.915c-0.528-0.113-1.154,0.081-1.692-0.041c-1.057-0.243-1.513-0.922-1.883-2.02c-2.608-1.533-6.119-2.53-10.207-1.244c-1.109,0.349-2.172,0.614-2.901,1.323c-0.146,0.412,0.143,0.494,0.446,0.489c-0.237,0.216-0.62,0.341-0.399,0.848c2.495-1.146,7.34-1.542,7.669,0.804c0.072,0.522-0.395,1.241-0.682,1.835c-0.905,1.874-2.011,3.394-2.813,5.091c-0.298,0.017-0.366,0.18-0.525,0.287c-2.604,3.8-5.451,8.541-7.9,12.794c-0.326,0.566-1.098,1.402-1.002,1.906C5.961,28.641,7.146,29,7.831,29.354z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Users

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Users();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Users = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Users",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M21.053,20.8c-1.132-0.453-1.584-1.698-1.584-1.698s-0.51,0.282-0.51-0.51s0.51,0.51,1.02-2.548c0,0,1.414-0.397,1.132-3.68h-0.34c0,0,0.849-3.51,0-4.699c-0.85-1.189-1.189-1.981-3.058-2.548s-1.188-0.454-2.547-0.396c-1.359,0.057-2.492,0.792-2.492,1.188c0,0-0.849,0.057-1.188,0.397c-0.34,0.34-0.906,1.924-0.906,2.321s0.283,3.058,0.566,3.624l-0.337,0.113c-0.283,3.283,1.132,3.68,1.132,3.68c0.509,3.058,1.019,1.756,1.019,2.548s-0.51,0.51-0.51,0.51s-0.452,1.245-1.584,1.698c-1.132,0.452-7.416,2.886-7.927,3.396c-0.511,0.511-0.453,2.888-0.453,2.888h26.947c0,0,0.059-2.377-0.452-2.888C28.469,23.686,22.185,21.252,21.053,20.8zM8.583,20.628c-0.099-0.18-0.148-0.31-0.148-0.31s-0.432,0.239-0.432-0.432s0.432,0.432,0.864-2.159c0,0,1.199-0.336,0.959-3.119H9.538c0,0,0.143-0.591,0.237-1.334c-0.004-0.308,0.006-0.636,0.037-0.996l0.038-0.426c-0.021-0.492-0.107-0.939-0.312-1.226C8.818,9.619,8.53,8.947,6.947,8.467c-1.583-0.48-1.008-0.385-2.159-0.336C3.636,8.179,2.676,8.802,2.676,9.139c0,0-0.72,0.048-1.008,0.336c-0.271,0.271-0.705,1.462-0.757,1.885v0.281c0.047,0.653,0.258,2.449,0.469,2.872l-0.286,0.096c-0.239,2.783,0.959,3.119,0.959,3.119c0.432,2.591,0.864,1.488,0.864,2.159s-0.432,0.432-0.432,0.432s-0.383,1.057-1.343,1.439c-0.061,0.024-0.139,0.056-0.232,0.092v5.234h0.575c-0.029-1.278,0.077-2.927,0.746-3.594C2.587,23.135,3.754,22.551,8.583,20.628zM30.913,11.572c-0.04-0.378-0.127-0.715-0.292-0.946c-0.719-1.008-1.008-1.679-2.59-2.159c-1.584-0.48-1.008-0.385-2.16-0.336C24.72,8.179,23.76,8.802,23.76,9.139c0,0-0.719,0.048-1.008,0.336c-0.271,0.272-0.709,1.472-0.758,1.891h0.033l0.08,0.913c0.02,0.231,0.022,0.436,0.027,0.645c0.09,0.666,0.21,1.35,0.33,1.589l-0.286,0.096c-0.239,2.783,0.96,3.119,0.96,3.119c0.432,2.591,0.863,1.488,0.863,2.159s-0.432,0.432-0.432,0.432s-0.053,0.142-0.163,0.338c4.77,1.9,5.927,2.48,6.279,2.834c0.67,0.667,0.775,2.315,0.746,3.594h0.48v-5.306c-0.016-0.006-0.038-0.015-0.052-0.021c-0.959-0.383-1.343-1.439-1.343-1.439s-0.433,0.239-0.433-0.432s0.433,0.432,0.864-2.159c0,0,0.804-0.229,0.963-1.841v-1.227c-0.001-0.018-0.001-0.033-0.003-0.051h-0.289c0,0,0.215-0.89,0.292-1.861V11.572z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.User

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.User();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.User = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.User",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M20.771,12.364c0,0,0.849-3.51,0-4.699c-0.85-1.189-1.189-1.981-3.058-2.548s-1.188-0.454-2.547-0.396c-1.359,0.057-2.492,0.792-2.492,1.188c0,0-0.849,0.057-1.188,0.397c-0.34,0.34-0.906,1.924-0.906,2.321s0.283,3.058,0.566,3.624l-0.337,0.113c-0.283,3.283,1.132,3.68,1.132,3.68c0.509,3.058,1.019,1.756,1.019,2.548s-0.51,0.51-0.51,0.51s-0.452,1.245-1.584,1.698c-1.132,0.452-7.416,2.886-7.927,3.396c-0.511,0.511-0.453,2.888-0.453,2.888h26.947c0,0,0.059-2.377-0.452-2.888c-0.512-0.511-6.796-2.944-7.928-3.396c-1.132-0.453-1.584-1.698-1.584-1.698s-0.51,0.282-0.51-0.51s0.51,0.51,1.02-2.548c0,0,1.414-0.397,1.132-3.68H20.771z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Customer

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Customer();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Customer = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Customer",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M28.523,23.813c-0.518-0.51-6.795-2.938-7.934-3.396c-1.132-0.451-1.584-1.697-1.584-1.697s-0.51,0.282-0.51-0.51c0-0.793,0.51,0.51,1.021-2.548c0,0,1.414-0.397,1.133-3.68l-0.338,0.001c0,0,0.85-3.511,0-4.699c-0.854-1.188-1.188-1.981-3.062-2.548c-1.869-0.567-1.188-0.454-2.547-0.396c-1.359,0.057-2.492,0.793-2.492,1.188c0,0-0.849,0.057-1.188,0.397c-0.34,0.34-0.906,1.924-0.906,2.32s0.283,3.059,0.566,3.624l-0.337,0.112c-0.283,3.283,1.132,3.681,1.132,3.681c0.509,3.058,1.019,1.755,1.019,2.548c0,0.792-0.51,0.51-0.51,0.51s-0.452,1.246-1.584,1.697c-1.132,0.453-7.416,2.887-7.927,3.396c-0.511,0.521-0.453,2.896-0.453,2.896h12.036l0.878-3.459l-0.781-0.781l1.344-1.344l1.344,1.344l-0.781,0.781l0.879,3.459h12.035C28.977,26.709,29.039,24.332,28.523,23.813z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Employee

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Employee();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Employee = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Employee",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M28.523,23.813c-0.518-0.51-6.795-2.938-7.934-3.396c-1.132-0.451-1.584-1.697-1.584-1.697s-0.51,0.282-0.51-0.51c0-0.793,0.51,0.51,1.021-2.548c0,0,1.414-0.397,1.133-3.68l-0.338,0.001c0,0,0.85-3.511,0-4.699c-0.854-1.188-1.188-1.981-3.062-2.548c-1.869-0.567-1.188-0.454-2.547-0.396c-1.359,0.057-2.492,0.793-2.492,1.188c0,0-0.849,0.057-1.188,0.397c-0.34,0.34-0.906,1.924-0.906,2.32s0.283,3.059,0.566,3.624l-0.337,0.112c-0.283,3.283,1.132,3.681,1.132,3.681c0.509,3.058,1.019,1.755,1.019,2.548c0,0.792-0.51,0.51-0.51,0.51s-0.452,1.246-1.584,1.697c-1.132,0.453-7.416,2.887-7.927,3.396c-0.511,0.521-0.453,2.896-0.453,2.896h26.954C28.977,26.709,29.039,24.332,28.523,23.813zM22.188,26.062h-4.562v-1.25h4.562V26.062z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Anonymous

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Anonymous();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Anonymous = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Anonymous",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M28.523,23.813c-0.518-0.51-6.795-2.938-7.934-3.396c-1.132-0.451-1.584-1.697-1.584-1.697s-0.51,0.282-0.51-0.51c0-0.793,0.51,0.51,1.021-2.548c0,0,1.414-0.397,1.133-3.68l-0.338,0.001c0,0,0.85-3.511,0-4.699c-0.854-1.188-1.188-1.981-3.062-2.548c-1.869-0.567-1.188-0.454-2.547-0.396c-1.359,0.057-2.492,0.793-2.492,1.188c0,0-0.849,0.057-1.188,0.397c-0.34,0.34-0.906,1.924-0.906,2.32s0.283,3.059,0.566,3.624l-0.337,0.112c-0.283,3.283,1.132,3.681,1.132,3.681c0.509,3.058,1.019,1.755,1.019,2.548c0,0.792-0.51,0.51-0.51,0.51s-0.452,1.246-1.584,1.697c-1.132,0.453-7.416,2.887-7.927,3.396c-0.511,0.521-0.453,2.896-0.453,2.896h26.954C28.977,26.709,29.039,24.332,28.523,23.813zM16.618,13.693c-0.398-0.251-0.783-1.211-0.783-1.64c0-0.133,0-0.236,0-0.236c-0.105-0.106-0.574-0.096-0.67,0c0,0,0,0.104,0,0.236c0,0.429-0.385,1.389-0.783,1.64c-0.399,0.251-1.611,0.237-2.084-0.236c-0.473-0.473-0.524-1.663-0.643-1.78c-0.118-0.119-0.185-0.185-0.185-0.185l0.029-0.414c0,0,0.842-0.207,1.699-0.207s1.803,0.502,1.803,0.502c0.231-0.074,0.784-0.083,0.996,0c0,0,0.945-0.502,1.803-0.502s1.699,0.207,1.699,0.207l0.029,0.414c0,0-0.066,0.066-0.185,0.185c-0.118,0.118-0.169,1.308-0.643,1.78C18.229,13.93,17.018,13.944,16.618,13.693z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Skull

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Skull();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Skull = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Skull",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.947,11.14c0-5.174-3.979-9.406-10.613-9.406c-6.633,0-10.282,4.232-10.282,9.406c0,5.174,1.459,4.511,1.459,7.43c0,1.095-1.061,0.564-1.061,2.919c0,2.587,3.615,2.223,4.677,3.283c1.061,1.062,0.961,3.019,0.961,3.019s0.199,0.796,0.564,0.563c0,0,0.232,0.564,0.498,0.232c0,0,0.265,0.563,0.531,0.1c0,0,0.265,0.631,0.696,0.166c0,0,0.431,0.63,0.929,0.133c0,0,0.564,0.53,1.194,0.133c0.63,0.397,1.194-0.133,1.194-0.133c0.497,0.497,0.929-0.133,0.929-0.133c0.432,0.465,0.695-0.166,0.695-0.166c0.268,0.464,0.531-0.1,0.531-0.1c0.266,0.332,0.498-0.232,0.498-0.232c0.365,0.232,0.564-0.563,0.564-0.563s-0.1-1.957,0.961-3.019c1.062-1.061,4.676-0.696,4.676-3.283c0-2.354-1.061-1.824-1.061-2.919C24.488,15.651,25.947,16.314,25.947,11.14zM10.333,20.992c-1.783,0.285-2.59-0.215-2.785-1.492c-0.508-3.328,2.555-3.866,4.079-3.683c0.731,0.088,1.99,0.862,1.99,1.825C13.617,20.229,11.992,20.727,10.333,20.992zM16.461,25.303c-0.331,0-0.862-0.431-0.895-1.227c-0.033,0.796-0.63,1.227-0.961,1.227c-0.332,0-0.83-0.331-0.863-1.127c-0.033-0.796,1.028-4.013,1.792-4.013c0.762,0,1.824,3.217,1.791,4.013S16.794,25.303,16.461,25.303zM23.361,19.5c-0.195,1.277-1.004,1.777-2.787,1.492c-1.658-0.266-3.283-0.763-3.283-3.35c0-0.963,1.258-1.737,1.99-1.825C20.805,15.634,23.869,16.172,23.361,19.5z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Mail

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Mail();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Mail = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Mail",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M28.516,7.167H3.482l12.517,7.108L28.516,7.167zM16.74,17.303C16.51,17.434,16.255,17.5,16,17.5s-0.51-0.066-0.741-0.197L2.5,10.06v14.773h27V10.06L16.74,17.303z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Picture

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Picture();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Picture = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Picture",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M2.5,4.833v22.334h27V4.833H2.5zM25.25,25.25H6.75V6.75h18.5V25.25zM11.25,14c1.426,0,2.583-1.157,2.583-2.583c0-1.427-1.157-2.583-2.583-2.583c-1.427,0-2.583,1.157-2.583,2.583C8.667,12.843,9.823,14,11.25,14zM24.251,16.25l-4.917-4.917l-6.917,6.917L10.5,16.333l-2.752,2.752v5.165h16.503V16.25z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Bubble

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Bubble();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Bubble = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Bubble",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,5.333c-7.732,0-14,4.701-14,10.5c0,1.982,0.741,3.833,2.016,5.414L2,25.667l5.613-1.441c2.339,1.317,5.237,2.107,8.387,2.107c7.732,0,14-4.701,14-10.5C30,10.034,23.732,5.333,16,5.333z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.CodeTalk

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.CodeTalk();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.CodeTalk = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.CodeTalk",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,4.938c-7.732,0-14,4.701-14,10.5c0,1.981,0.741,3.833,2.016,5.414L2,25.272l5.613-1.44c2.339,1.316,5.237,2.106,8.387,2.106c7.732,0,14-4.701,14-10.5S23.732,4.938,16,4.938zM13.704,19.47l-2.338,2.336l-6.43-6.431l6.429-6.432l2.339,2.341l-4.091,4.091L13.704,19.47zM20.775,21.803l-2.337-2.339l4.092-4.09l-4.092-4.092l2.337-2.339l6.43,6.426L20.775,21.803z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Talkq

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Talkq();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Talkq = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Talkq",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,4.938c-7.732,0-14,4.701-14,10.5c0,1.981,0.741,3.833,2.016,5.414L2,25.272l5.613-1.44c2.339,1.316,5.237,2.106,8.387,2.106c7.732,0,14-4.701,14-10.5S23.732,4.938,16,4.938zM16.868,21.375h-1.969v-1.889h1.969V21.375zM16.772,18.094h-1.777l-0.176-8.083h2.113L16.772,18.094z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Talke

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Talke();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Talke = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Talke",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,4.938c-7.732,0-14,4.701-14,10.5c0,1.981,0.741,3.833,2.016,5.414L2,25.272l5.613-1.44c2.339,1.316,5.237,2.106,8.387,2.106c7.732,0,14-4.701,14-10.5S23.732,4.938,16,4.938zM16.982,21.375h-1.969v-1.889h1.969V21.375zM16.982,17.469v0.625h-1.969v-0.769c0-2.321,2.641-2.689,2.641-4.337c0-0.752-0.672-1.329-1.553-1.329c-0.912,0-1.713,0.672-1.713,0.672l-1.12-1.393c0,0,1.104-1.153,3.009-1.153c1.81,0,3.49,1.121,3.49,3.009C19.768,15.437,16.982,15.741,16.982,17.469z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Home

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Home();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Home = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Home",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M27.812,16l-3.062-3.062V5.625h-2.625v4.688L16,4.188L4.188,16L7,15.933v11.942h17.875V16H27.812zM16,26.167h-5.833v-7H16V26.167zM21.667,23.167h-3.833v-4.042h3.833V23.167z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Lock

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Lock();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Lock = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Lock",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M22.335,12.833V9.999h-0.001C22.333,6.501,19.498,3.666,16,3.666S9.666,6.502,9.666,10h0v2.833H7.375V25h17.25V12.833H22.335zM11.667,10C11.667,10,11.667,10,11.667,10c0-2.39,1.944-4.334,4.333-4.334c2.391,0,4.335,1.944,4.335,4.333c0,0,0,0,0,0v2.834h-8.668V10z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Clip

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Clip();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Clip = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Clip",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M23.898,6.135c-1.571-1.125-3.758-0.764-4.884,0.808l-8.832,12.331c-0.804,1.122-0.546,2.684,0.577,3.488c1.123,0.803,2.684,0.545,3.488-0.578l6.236-8.706l-0.813-0.583l-6.235,8.707h0c-0.483,0.672-1.42,0.828-2.092,0.347c-0.673-0.481-0.827-1.419-0.345-2.093h0l8.831-12.33l0.001-0.001l-0.002-0.001c0.803-1.119,2.369-1.378,3.489-0.576c1.12,0.803,1.379,2.369,0.577,3.489v-0.001l-9.68,13.516l0.001,0.001c-1.124,1.569-3.316,1.931-4.885,0.808c-1.569-1.125-1.93-3.315-0.807-4.885l7.035-9.822l-0.813-0.582l-7.035,9.822c-1.447,2.02-0.982,4.83,1.039,6.277c2.021,1.448,4.831,0.982,6.278-1.037l9.68-13.516C25.83,9.447,25.47,7.261,23.898,6.135z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Star

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Star();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Star = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Star",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,22.375L7.116,28.83l3.396-10.438l-8.883-6.458l10.979,0.002L16.002,1.5l3.391,10.434h10.981l-8.886,6.457l3.396,10.439L16,22.375L16,22.375z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.StarOff

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.StarOff();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.StarOff = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.StarOff",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,22.375L7.116,28.83l3.396-10.438l-8.883-6.458l10.979,0.002L16.002,1.5l3.391,10.434h10.981l-8.886,6.457l3.396,10.439L16,22.375L16,22.375zM22.979,26.209l-2.664-8.205l6.979-5.062h-8.627L16,4.729l-2.666,8.206H4.708l6.979,5.07l-2.666,8.203L16,21.146L22.979,26.209L22.979,26.209z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Star2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Star2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Star2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Star2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M14.615,4.928c0.487-0.986,1.284-0.986,1.771,0l2.249,4.554c0.486,0.986,1.775,1.923,2.864,2.081l5.024,0.73c1.089,0.158,1.335,0.916,0.547,1.684l-3.636,3.544c-0.788,0.769-1.28,2.283-1.095,3.368l0.859,5.004c0.186,1.085-0.459,1.553-1.433,1.041l-4.495-2.363c-0.974-0.512-2.567-0.512-3.541,0l-4.495,2.363c-0.974,0.512-1.618,0.044-1.432-1.041l0.858-5.004c0.186-1.085-0.307-2.6-1.094-3.368L3.93,13.977c-0.788-0.768-0.542-1.525,0.547-1.684l5.026-0.73c1.088-0.158,2.377-1.095,2.864-2.081L14.615,4.928z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Star2Off

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Star2Off();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Star2Off = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Star2Off",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M26.522,12.293l-5.024-0.73c-1.089-0.158-2.378-1.095-2.864-2.081l-2.249-4.554c-0.487-0.986-1.284-0.986-1.771,0l-2.247,4.554c-0.487,0.986-1.776,1.923-2.864,2.081l-5.026,0.73c-1.088,0.158-1.334,0.916-0.547,1.684l3.637,3.544c0.788,0.769,1.28,2.283,1.094,3.368l-0.858,5.004c-0.186,1.085,0.458,1.553,1.432,1.041l4.495-2.363c0.974-0.512,2.566-0.512,3.541,0l4.495,2.363c0.974,0.512,1.618,0.044,1.433-1.041l-0.859-5.004c-0.186-1.085,0.307-2.6,1.095-3.368l3.636-3.544C27.857,13.209,27.611,12.452,26.522,12.293zM22.037,16.089c-1.266,1.232-1.966,3.394-1.67,5.137l0.514,2.984l-2.679-1.409c-0.757-0.396-1.715-0.612-2.702-0.612s-1.945,0.216-2.7,0.61l-2.679,1.409l0.511-2.982c0.297-1.743-0.404-3.905-1.671-5.137l-2.166-2.112l2.995-0.435c1.754-0.255,3.592-1.591,4.373-3.175L15.5,7.652l1.342,2.716c0.781,1.583,2.617,2.92,4.369,3.173l2.992,0.435L22.037,16.089z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Star3

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Star3();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Star3 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Star3",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M22.441,28.181c-0.419,0-0.835-0.132-1.189-0.392l-5.751-4.247L9.75,27.789c-0.354,0.26-0.771,0.392-1.189,0.392c-0.412,0-0.824-0.128-1.175-0.384c-0.707-0.511-1-1.422-0.723-2.25l2.26-6.783l-5.815-4.158c-0.71-0.509-1.009-1.416-0.74-2.246c0.268-0.826,1.037-1.382,1.904-1.382c0.004,0,0.01,0,0.014,0l7.15,0.056l2.157-6.816c0.262-0.831,1.035-1.397,1.906-1.397s1.645,0.566,1.906,1.397l2.155,6.816l7.15-0.056c0.004,0,0.01,0,0.015,0c0.867,0,1.636,0.556,1.903,1.382c0.271,0.831-0.028,1.737-0.739,2.246l-5.815,4.158l2.263,6.783c0.276,0.826-0.017,1.737-0.721,2.25C23.268,28.053,22.854,28.181,22.441,28.181L22.441,28.181z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Star3Off

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Star3Off();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Star3Off = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Star3Off",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M28.631,12.359c-0.268-0.826-1.036-1.382-1.903-1.382h-0.015l-7.15,0.056l-2.155-6.816c-0.262-0.831-1.035-1.397-1.906-1.397s-1.645,0.566-1.906,1.397l-2.157,6.816l-7.15-0.056H4.273c-0.868,0-1.636,0.556-1.904,1.382c-0.27,0.831,0.029,1.737,0.74,2.246l5.815,4.158l-2.26,6.783c-0.276,0.828,0.017,1.739,0.723,2.25c0.351,0.256,0.763,0.384,1.175,0.384c0.418,0,0.834-0.132,1.189-0.392l5.751-4.247l5.751,4.247c0.354,0.26,0.771,0.392,1.189,0.392c0.412,0,0.826-0.128,1.177-0.384c0.704-0.513,0.997-1.424,0.721-2.25l-2.263-6.783l5.815-4.158C28.603,14.097,28.901,13.19,28.631,12.359zM19.712,17.996l2.729,8.184l-6.94-5.125L8.56,26.18l2.729-8.184l-7.019-5.018l8.627,0.066L15.5,4.82l2.603,8.225l8.627-0.066L19.712,17.996z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Chat

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Chat();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Chat = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Chat",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M15.985,5.972c-7.563,0-13.695,4.077-13.695,9.106c0,2.877,2.013,5.44,5.147,7.108c-0.446,1.479-1.336,3.117-3.056,4.566c0,0,4.015-0.266,6.851-3.143c0.163,0.04,0.332,0.07,0.497,0.107c-0.155-0.462-0.246-0.943-0.246-1.443c0-3.393,3.776-6.05,8.599-6.05c3.464,0,6.379,1.376,7.751,3.406c1.168-1.34,1.847-2.892,1.847-4.552C29.68,10.049,23.548,5.972,15.985,5.972zM27.68,22.274c0-2.79-3.401-5.053-7.599-5.053c-4.196,0-7.599,2.263-7.599,5.053c0,2.791,3.403,5.053,7.599,5.053c0.929,0,1.814-0.116,2.637-0.319c1.573,1.597,3.801,1.744,3.801,1.744c-0.954-0.804-1.447-1.713-1.695-2.534C26.562,25.293,27.68,23.871,27.68,22.274z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Quote

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Quote();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Quote = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Quote",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M14.505,5.873c-3.937,2.52-5.904,5.556-5.904,9.108c0,1.104,0.192,1.656,0.576,1.656l0.396-0.107c0.312-0.12,0.563-0.18,0.756-0.18c1.128,0,2.07,0.411,2.826,1.229c0.756,0.82,1.134,1.832,1.134,3.037c0,1.157-0.408,2.14-1.224,2.947c-0.816,0.807-1.801,1.211-2.952,1.211c-1.608,0-2.935-0.661-3.979-1.984c-1.044-1.321-1.565-2.98-1.565-4.977c0-2.259,0.443-4.327,1.332-6.203c0.888-1.875,2.243-3.57,4.067-5.085c1.824-1.514,2.988-2.272,3.492-2.272c0.336,0,0.612,0.162,0.828,0.486c0.216,0.324,0.324,0.606,0.324,0.846L14.505,5.873zM27.465,5.873c-3.937,2.52-5.904,5.556-5.904,9.108c0,1.104,0.192,1.656,0.576,1.656l0.396-0.107c0.312-0.12,0.563-0.18,0.756-0.18c1.104,0,2.04,0.411,2.808,1.229c0.769,0.82,1.152,1.832,1.152,3.037c0,1.157-0.408,2.14-1.224,2.947c-0.816,0.807-1.801,1.211-2.952,1.211c-1.608,0-2.935-0.661-3.979-1.984c-1.044-1.321-1.565-2.98-1.565-4.977c0-2.284,0.449-4.369,1.35-6.256c0.9-1.887,2.256-3.577,4.068-5.067c1.812-1.49,2.97-2.236,3.474-2.236c0.336,0,0.612,0.162,0.828,0.486c0.216,0.324,0.324,0.606,0.324,0.846L27.465,5.873z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Gear2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Gear2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Gear2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Gear2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M17.047,27.945c-0.34,0.032-0.688,0.054-1.046,0.054l0,0c-0.32,0-0.631-0.017-0.934-0.043l0,0l-2.626,3.375l-0.646-0.183c-0.758-0.213-1.494-0.48-2.202-0.8l0,0L8.979,30.07l0.158-4.24c-0.558-0.39-1.079-0.825-1.561-1.302l0,0L3.424,25.42l-0.379-0.557c-0.445-0.654-0.824-1.339-1.16-2.032l0,0l-0.292-0.605l2.819-3.12c-0.176-0.661-0.293-1.343-0.353-2.038l0,0l-3.736-1.975l0.068-0.669c0.08-0.801,0.235-1.567,0.42-2.303l0,0l0.165-0.653l4.167-0.577c0.297-0.627,0.647-1.221,1.041-1.78l0,0l-1.59-3.914l0.48-0.47c0.564-0.55,1.168-1.048,1.798-1.503l0,0l0.546-0.394l3.597,2.259c0.606-0.279,1.24-0.509,1.897-0.685l0,0l1.304-4.046l0.672-0.051c0.362-0.027,0.751-0.058,1.174-0.058l0,0c0.422,0,0.81,0.031,1.172,0.058l0,0l0.672,0.051l1.318,4.088c0.632,0.176,1.244,0.401,1.831,0.674l0,0l3.647-2.291l0.548,0.394c0.63,0.455,1.235,0.954,1.798,1.501l0,0l0.482,0.47l-1.639,4.031c0.357,0.519,0.679,1.068,0.954,1.646l0,0l4.297,0.595l0.167,0.653c0.188,0.735,0.342,1.501,0.42,2.303l0,0l0.068,0.669l-3.866,2.044c-0.058,0.634-0.161,1.258-0.315,1.866l0,0l2.913,3.218l-0.293,0.608c-0.335,0.695-0.712,1.382-1.159,2.034l0,0l-0.379,0.555l-4.255-0.912c-0.451,0.451-0.939,0.866-1.461,1.241l0,0l0.162,4.323l-0.615,0.278c-0.709,0.319-1.444,0.587-2.202,0.8l0,0l-0.648,0.183L17.047,27.945L17.047,27.945zM20.424,29.028c0.227-0.076,0.45-0.157,0.671-0.244l0,0l-0.152-4.083l0.479-0.307c0.717-0.466,1.37-1.024,1.95-1.658l0,0l0.386-0.423l4.026,0.862c0.121-0.202,0.238-0.409,0.351-0.62l0,0l-2.754-3.045l0.171-0.544c0.243-0.783,0.381-1.623,0.422-2.5l0,0l0.025-0.571l3.658-1.933c-0.038-0.234-0.082-0.467-0.132-0.7l0,0l-4.07-0.563l-0.219-0.527c-0.327-0.787-0.76-1.524-1.277-2.204l0,0l-0.342-0.453l1.548-3.808c-0.179-0.157-0.363-0.31-0.552-0.458l0,0l-3.455,2.169L20.649,7.15c-0.754-0.397-1.569-0.698-2.429-0.894l0,0l-0.556-0.127l-1.248-3.87c-0.121-0.006-0.239-0.009-0.354-0.009l0,0c-0.117,0-0.235,0.003-0.357,0.009l0,0l-1.239,3.845l-0.564,0.12c-0.875,0.188-1.709,0.494-2.486,0.896l0,0l-0.508,0.264L7.509,5.249c-0.188,0.148-0.372,0.301-0.55,0.458l0,0l1.507,3.708L8.112,9.869c-0.552,0.709-1.011,1.485-1.355,2.319l0,0l-0.218,0.529l-3.939,0.545c-0.05,0.233-0.094,0.466-0.131,0.7l0,0l3.531,1.867l0.022,0.575c0.037,0.929,0.192,1.82,0.459,2.653l0,0l0.175,0.548l-2.667,2.95c0.112,0.212,0.229,0.419,0.351,0.621l0,0l3.916-0.843l0.39,0.423c0.601,0.657,1.287,1.229,2.043,1.703l0,0l0.488,0.305l-0.149,4.02c0.221,0.087,0.445,0.168,0.672,0.244l0,0l2.479-3.188l0.566,0.07c0.427,0.054,0.843,0.089,1.257,0.089l0,0c0.445,0,0.894-0.039,1.353-0.104l0,0l0.571-0.08L20.424,29.028L20.424,29.028zM21.554,20.75l0.546,0.839l-3.463,2.253l-1.229-1.891l0,0c-0.447,0.109-0.917,0.173-1.406,0.173l0,0c-3.384,0-6.126-2.743-6.126-6.123l0,0c0-3.384,2.742-6.126,6.126-6.126l0,0c3.38,0,6.123,2.742,6.123,6.126l0,0c0,1.389-0.467,2.676-1.25,3.704l0,0L21.554,20.75M19.224,21.073l0.108-0.069l-0.987-1.519l0.572-0.572c0.748-0.75,1.207-1.773,1.207-2.912l0,0c-0.004-2.278-1.848-4.122-4.123-4.126l0,0c-2.28,0.004-4.122,1.846-4.126,4.126l0,0c0.004,2.275,1.848,4.119,4.126,4.123l0,0c0.509,0,0.999-0.104,1.473-0.286l0,0l0.756-0.29L19.224,21.073L19.224,21.073z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Gear

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Gear();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Gear = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Gear",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M26.974,16.514l3.765-1.991c-0.074-0.738-0.217-1.454-0.396-2.157l-4.182-0.579c-0.362-0.872-0.84-1.681-1.402-2.423l1.594-3.921c-0.524-0.511-1.09-0.977-1.686-1.406l-3.551,2.229c-0.833-0.438-1.73-0.77-2.672-0.984l-1.283-3.976c-0.364-0.027-0.728-0.056-1.099-0.056s-0.734,0.028-1.099,0.056l-1.271,3.941c-0.967,0.207-1.884,0.543-2.738,0.986L7.458,4.037C6.863,4.466,6.297,4.932,5.773,5.443l1.55,3.812c-0.604,0.775-1.11,1.629-1.49,2.55l-4.05,0.56c-0.178,0.703-0.322,1.418-0.395,2.157l3.635,1.923c0.041,1.013,0.209,1.994,0.506,2.918l-2.742,3.032c0.319,0.661,0.674,1.303,1.085,1.905l4.037-0.867c0.662,0.72,1.416,1.351,2.248,1.873l-0.153,4.131c0.663,0.299,1.352,0.549,2.062,0.749l2.554-3.283C15.073,26.961,15.532,27,16,27c0.507,0,1.003-0.046,1.491-0.113l2.567,3.301c0.711-0.2,1.399-0.45,2.062-0.749l-0.156-4.205c0.793-0.513,1.512-1.127,2.146-1.821l4.142,0.889c0.411-0.602,0.766-1.243,1.085-1.905l-2.831-3.131C26.778,18.391,26.93,17.467,26.974,16.514zM20.717,21.297l-1.785,1.162l-1.098-1.687c-0.571,0.22-1.186,0.353-1.834,0.353c-2.831,0-5.125-2.295-5.125-5.125c0-2.831,2.294-5.125,5.125-5.125c2.83,0,5.125,2.294,5.125,5.125c0,1.414-0.573,2.693-1.499,3.621L20.717,21.297z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Wrench

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Wrench();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Wrench = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Wrench",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M26.834,14.693c1.816-2.088,2.181-4.938,1.193-7.334l-3.646,4.252l-3.594-0.699L19.596,7.45l3.637-4.242c-2.502-0.63-5.258,0.13-7.066,2.21c-1.907,2.193-2.219,5.229-1.039,7.693L5.624,24.04c-1.011,1.162-0.888,2.924,0.274,3.935c1.162,1.01,2.924,0.888,3.935-0.274l9.493-10.918C21.939,17.625,24.918,16.896,26.834,14.693z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Wrench2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Wrench2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Wrench2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Wrench2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.946,9.721l-2.872-0.768l-0.771-2.874l3.188-3.231c-1.992-0.653-4.268-0.192-5.848,1.391c-1.668,1.668-2.095,4.111-1.279,6.172l-3.476,3.478l-3.478,3.478c-2.062-0.816-4.504-0.391-6.173,1.277c-1.583,1.581-2.043,3.856-1.39,5.849l3.231-3.188l2.874,0.77l0.769,2.872l-3.239,3.197c1.998,0.665,4.288,0.207,5.876-1.384c1.678-1.678,2.1-4.133,1.271-6.202l3.463-3.464l3.464-3.463c2.069,0.828,4.523,0.406,6.202-1.272c1.592-1.589,2.049-3.878,1.384-5.876L24.946,9.721z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Wrench3

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Wrench3();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Wrench3 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Wrench3",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M27.839,6.775l-3.197,3.239L21.77,9.246l-0.771-2.874l3.188-3.231c-1.992-0.653-4.268-0.192-5.848,1.391c-1.668,1.668-2.095,4.111-1.279,6.172L7.42,20.344c-0.204-0.032-0.408-0.062-0.621-0.062c-2.173,0-3.933,1.759-3.933,3.933c0,2.173,1.76,3.933,3.933,3.933c2.171,0,3.931-1.76,3.933-3.933c0-0.24-0.03-0.473-0.071-0.7l9.592-9.59c2.069,0.828,4.523,0.406,6.202-1.272C28.047,11.062,28.504,8.772,27.839,6.775zM6.799,25.146c-0.517,0-0.933-0.418-0.935-0.933c0.002-0.515,0.418-0.933,0.935-0.933c0.514,0,0.932,0.418,0.932,0.933S7.313,25.146,6.799,25.146z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ScrewDriver

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ScrewDriver();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ScrewDriver = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ScrewDriver",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M19.387,14.373c2.119-2.619,5.322-6.77,5.149-7.75c-0.128-0.729-0.882-1.547-1.763-2.171c-0.883-0.625-1.916-1.044-2.645-0.915c-0.98,0.173-3.786,4.603-5.521,7.49c-0.208,0.344,0.328,1.177,0.156,1.468c-0.172,0.292-1.052,0.042-1.18,0.261c-0.263,0.451-0.417,0.722-0.417,0.722s-0.553,0.823,1.163,2.163l-5.233,7.473c-0.267,0.381-1.456,0.459-1.456,0.459l-1.184,3.312l0.859,0.602l2.708-2.246c0,0-0.334-1.143-0.068-1.523l5.242-7.489c1.719,1,2.377,0.336,2.377,0.336s0.201-0.238,0.536-0.639c0.161-0.194-0.374-0.936-0.159-1.197C18.169,14.467,19.133,14.685,19.387,14.373z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.HammerAndScrewDriver

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.HammerAndScrewDriver();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.HammerAndScrewDriver = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.HammerAndScrewDriver",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M28.537,9.859c-0.473-0.259-1.127-0.252-1.609-0.523c-0.943-0.534-1.186-1.316-1.226-2.475c-2.059-2.215-5.138-4.176-9.424-4.114c-1.162,0.017-2.256-0.035-3.158,0.435c-0.258,0.354-0.004,0.516,0.288,0.599c-0.29,0.138-0.692,0.147-0.626,0.697c2.72-0.383,7.475,0.624,7.116,2.966c-0.08,0.521-0.735,1.076-1.179,1.563c-1.263,1.382-2.599,2.45-3.761,3.667l0.336,0.336c0.742-0.521,1.446-0.785,2.104-0.785c0.707,0,1.121,0.297,1.276,0.433c0.575-0.618,1.166-1.244,1.839-1.853c0.488-0.444,1.047-1.099,1.566-1.178l0.949-0.101c1.156,0.047,1.937,0.29,2.471,1.232c0.27,0.481,0.262,1.139,0.521,1.613c0.175,0.324,0.937,1.218,1.316,1.228c0.294,0.009,0.603-0.199,0.899-0.49l1.033-1.034c0.291-0.294,0.501-0.6,0.492-0.896C29.754,10.801,28.861,10.035,28.537,9.859zM13.021,15.353l-0.741-0.741c-3.139,2.643-6.52,5.738-9.531,8.589c-0.473,0.443-1.452,1.021-1.506,1.539c-0.083,0.781,0.95,1.465,1.506,2c0.556,0.533,1.212,1.602,1.994,1.51c0.509-0.043,1.095-1.029,1.544-1.502c2.255-2.374,4.664-4.976,6.883-7.509c-0.312-0.371-0.498-0.596-0.498-0.596C12.535,18.451,11.779,17.272,13.021,15.353zM20.64,15.643c-0.366-0.318-1.466,0.143-1.777-0.122c-0.311-0.266,0.171-1.259-0.061-1.455c-0.482-0.406-0.77-0.646-0.77-0.646s-0.862-0.829-2.812,0.928L7.44,6.569C7.045,6.173,7.203,4.746,7.203,4.746L3.517,2.646L2.623,3.541l2.1,3.686c0,0,1.428-0.158,1.824,0.237l7.792,7.793c-1.548,1.831-0.895,2.752-0.895,2.752s0.238,0.288,0.646,0.771c0.196,0.23,1.188-0.249,1.455,0.061c0.264,0.312-0.196,1.41,0.12,1.777c2.666,3.064,6.926,7.736,8.125,7.736c0.892,0,2.021-0.724,2.948-1.64c0.925-0.917,1.639-2.055,1.639-2.947C28.377,22.567,23.704,18.309,20.64,15.643z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Magic

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Magic();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Magic = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Magic",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M23.043,4.649l-0.404-2.312l-1.59,1.727l-2.323-0.33l1.151,2.045l-1.032,2.108l2.302-0.463l1.686,1.633l0.271-2.332l2.074-1.099L23.043,4.649zM26.217,18.198l-0.182-1.25l-0.882,0.905l-1.245-0.214l0.588,1.118l-0.588,1.118l1.245-0.214l0.882,0.905l0.182-1.25l1.133-0.56L26.217,18.198zM4.92,7.672L5.868,7.3l0.844,0.569L6.65,6.853l0.802-0.627L6.467,5.97L6.118,5.013L5.571,5.872L4.553,5.908l0.647,0.786L4.92,7.672zM10.439,10.505l1.021-1.096l1.481,0.219l-0.727-1.31l0.667-1.341l-1.47,0.287l-1.069-1.048L10.16,7.703L8.832,8.396l1.358,0.632L10.439,10.505zM17.234,12.721c-0.588-0.368-1.172-0.618-1.692-0.729c-0.492-0.089-1.039-0.149-1.425,0.374L2.562,30.788h6.68l9.669-15.416c0.303-0.576,0.012-1.041-0.283-1.447C18.303,13.508,17.822,13.09,17.234,12.721zM13.613,21.936c-0.254-0.396-0.74-0.857-1.373-1.254c-0.632-0.396-1.258-0.634-1.726-0.69l4.421-7.052c0.064-0.013,0.262-0.021,0.543,0.066c0.346,0.092,0.785,0.285,1.225,0.562c0.504,0.313,0.908,0.677,1.133,0.97c0.113,0.145,0.178,0.271,0.195,0.335c0.002,0.006,0.004,0.011,0.004,0.015L13.613,21.936z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Download

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Download();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Download = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Download",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,1.466C7.973,1.466,1.466,7.973,1.466,16c0,8.027,6.507,14.534,14.534,14.534c8.027,0,14.534-6.507,14.534-14.534C30.534,7.973,24.027,1.466,16,1.466zM16,28.792c-1.549,0-2.806-1.256-2.806-2.806s1.256-2.806,2.806-2.806c1.55,0,2.806,1.256,2.806,2.806S17.55,28.792,16,28.792zM16,21.087l-7.858-6.562h3.469V5.747h8.779v8.778h3.468L16,21.087z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.View

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.View();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.View = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.View",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,8.286C8.454,8.286,2.5,16,2.5,16s5.954,7.715,13.5,7.715c5.771,0,13.5-7.715,13.5-7.715S21.771,8.286,16,8.286zM16,20.807c-2.649,0-4.807-2.157-4.807-4.807s2.158-4.807,4.807-4.807s4.807,2.158,4.807,4.807S18.649,20.807,16,20.807zM16,13.194c-1.549,0-2.806,1.256-2.806,2.806c0,1.55,1.256,2.806,2.806,2.806c1.55,0,2.806-1.256,2.806-2.806C18.806,14.451,17.55,13.194,16,13.194z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Noview

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Noview();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Noview = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Noview",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M11.478,17.568c-0.172-0.494-0.285-1.017-0.285-1.568c0-2.65,2.158-4.807,4.807-4.807c0.552,0,1.074,0.113,1.568,0.285l2.283-2.283C18.541,8.647,17.227,8.286,16,8.286C8.454,8.286,2.5,16,2.5,16s2.167,2.791,5.53,5.017L11.478,17.568zM23.518,11.185l-3.056,3.056c0.217,0.546,0.345,1.138,0.345,1.76c0,2.648-2.158,4.807-4.807,4.807c-0.622,0-1.213-0.128-1.76-0.345l-2.469,2.47c1.327,0.479,2.745,0.783,4.229,0.783c5.771,0,13.5-7.715,13.5-7.715S26.859,13.374,23.518,11.185zM25.542,4.917L4.855,25.604L6.27,27.02L26.956,6.332L25.542,4.917z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Cloud

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Cloud();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Cloud = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Cloud",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.345,13.904c0.019-0.195,0.03-0.392,0.03-0.591c0-3.452-2.798-6.25-6.25-6.25c-2.679,0-4.958,1.689-5.847,4.059c-0.589-0.646-1.429-1.059-2.372-1.059c-1.778,0-3.219,1.441-3.219,3.219c0,0.21,0.023,0.415,0.062,0.613c-2.372,0.391-4.187,2.436-4.187,4.918c0,2.762,2.239,5,5,5h15.875c2.762,0,5-2.238,5-5C28.438,16.362,26.672,14.332,24.345,13.904z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Cloud2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Cloud2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Cloud2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Cloud2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M7.562,24.812c-3.313,0-6-2.687-6-6l0,0c0.002-2.659,1.734-4.899,4.127-5.684l0,0c0.083-2.26,1.937-4.064,4.216-4.066l0,0c0.73,0,1.415,0.19,2.01,0.517l0,0c1.266-2.105,3.57-3.516,6.208-3.517l0,0c3.947,0.002,7.157,3.155,7.248,7.079l0,0c2.362,0.804,4.062,3.034,4.064,5.671l0,0c0,3.313-2.687,6-6,6l0,0H7.562L7.562,24.812zM24.163,14.887c-0.511-0.095-0.864-0.562-0.815-1.079l0,0c0.017-0.171,0.027-0.336,0.027-0.497l0,0c-0.007-2.899-2.352-5.245-5.251-5.249l0,0c-2.249-0.002-4.162,1.418-4.911,3.41l0,0c-0.122,0.323-0.406,0.564-0.748,0.63l0,0c-0.34,0.066-0.694-0.052-0.927-0.309l0,0c-0.416-0.453-0.986-0.731-1.633-0.731l0,0c-1.225,0.002-2.216,0.993-2.22,2.218l0,0c0,0.136,0.017,0.276,0.045,0.424l0,0c0.049,0.266-0.008,0.54-0.163,0.762l0,0c-0.155,0.223-0.392,0.371-0.657,0.414l0,0c-1.9,0.313-3.352,1.949-3.35,3.931l0,0c0.004,2.209,1.792,3.995,4.001,4.001l0,0h15.874c2.209-0.006,3.994-1.792,3.999-4.001l0,0C27.438,16.854,26.024,15.231,24.163,14.887L24.163,14.887");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.CloudDown

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.CloudDown();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.CloudDown = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.CloudDown",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.345,13.904c0.019-0.195,0.03-0.392,0.03-0.591c0-3.452-2.798-6.25-6.25-6.25c-2.679,0-4.958,1.689-5.847,4.059c-0.589-0.646-1.429-1.059-2.372-1.059c-1.778,0-3.219,1.441-3.219,3.219c0,0.21,0.023,0.415,0.062,0.613c-2.372,0.391-4.187,2.436-4.187,4.918c0,2.762,2.239,5,5,5h3.404l-0.707-0.707c-0.377-0.377-0.585-0.879-0.585-1.413c0-0.533,0.208-1.035,0.585-1.412l0.556-0.557c0.4-0.399,0.937-0.628,1.471-0.628c0.027,0,0.054,0,0.08,0.002v-0.472c0-1.104,0.898-2.002,2-2.002h3.266c1.103,0,2,0.898,2,2.002v0.472c0.027-0.002,0.054-0.002,0.081-0.002c0.533,0,1.07,0.229,1.47,0.63l0.557,0.552c0.78,0.781,0.78,2.05,0,2.828l-0.706,0.707h2.403c2.762,0,5-2.238,5-5C28.438,16.362,26.672,14.332,24.345,13.904z M21.033,20.986l-0.556-0.555c-0.39-0.389-0.964-0.45-1.276-0.137c-0.312,0.312-0.568,0.118-0.568-0.432v-1.238c0-0.55-0.451-1-1-1h-3.265c-0.55,0-1,0.45-1,1v1.238c0,0.55-0.256,0.744-0.569,0.432c-0.312-0.313-0.887-0.252-1.276,0.137l-0.556,0.555c-0.39,0.389-0.39,1.024-0.001,1.413l4.328,4.331c0.194,0.194,0.451,0.291,0.707,0.291s0.512-0.097,0.707-0.291l4.327-4.331C21.424,22.011,21.423,21.375,21.033,20.986z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.CloudUp

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.CloudUp();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.CloudUp = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.CloudUp",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.345,13.904c0.019-0.195,0.03-0.392,0.03-0.591c0-3.452-2.798-6.25-6.25-6.25c-2.679,0-4.958,1.689-5.847,4.059c-0.589-0.646-1.429-1.059-2.372-1.059c-1.778,0-3.219,1.441-3.219,3.219c0,0.21,0.023,0.415,0.062,0.613c-2.372,0.391-4.187,2.436-4.187,4.918c0,2.762,2.239,5,5,5h2.312c-0.126-0.266-0.2-0.556-0.2-0.859c0-0.535,0.208-1.04,0.587-1.415l4.325-4.329c0.375-0.377,0.877-0.585,1.413-0.585c0.54,0,1.042,0.21,1.417,0.587l4.323,4.329c0.377,0.373,0.585,0.878,0.585,1.413c0,0.304-0.073,0.594-0.2,0.859h1.312c2.762,0,5-2.238,5-5C28.438,16.362,26.672,14.332,24.345,13.904z M16.706,17.916c-0.193-0.195-0.45-0.291-0.706-0.291s-0.512,0.096-0.707,0.291l-4.327,4.33c-0.39,0.389-0.389,1.025,0.001,1.414l0.556,0.555c0.39,0.389,0.964,0.449,1.276,0.137s0.568-0.119,0.568,0.432v1.238c0,0.549,0.451,1,1,1h3.265c0.551,0,1-0.451,1-1v-1.238c0-0.551,0.256-0.744,0.569-0.432c0.312,0.312,0.887,0.252,1.276-0.137l0.556-0.555c0.39-0.389,0.39-1.025,0.001-1.414L16.706,17.916z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Location

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Location();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Location = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Location",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M16,3.5c-4.142,0-7.5,3.358-7.5,7.5c0,4.143,7.5,18.121,7.5,18.121S23.5,15.143,23.5,11C23.5,6.858,20.143,3.5,16,3.5z M16,14.584c-1.979,0-3.584-1.604-3.584-3.584S14.021,7.416,16,7.416S19.584,9.021,19.584,11S17.979,14.584,16,14.584z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Volume0

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Volume0();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Volume0 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Volume0",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M4.998,12.127v7.896h4.495l6.729,5.526l0.004-18.948l-6.73,5.526H4.998z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Volume1

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Volume1();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Volume1 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Volume1",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M4.998,12.127v7.896h4.495l6.729,5.526l0.004-18.948l-6.73,5.526H4.998z M18.806,11.219c-0.393-0.389-1.024-0.389-1.415,0.002c-0.39,0.391-0.39,1.024,0.002,1.416v-0.002c0.863,0.864,1.395,2.049,1.395,3.366c0,1.316-0.531,2.497-1.393,3.361c-0.394,0.389-0.394,1.022-0.002,1.415c0.195,0.195,0.451,0.293,0.707,0.293c0.257,0,0.513-0.098,0.708-0.293c1.222-1.22,1.98-2.915,1.979-4.776C20.788,14.136,20.027,12.439,18.806,11.219z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Volume2

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Volume2();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Volume2 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Volume2",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M4.998,12.127v7.896h4.495l6.729,5.526l0.004-18.948l-6.73,5.526H4.998z M18.806,11.219c-0.393-0.389-1.024-0.389-1.415,0.002c-0.39,0.391-0.39,1.024,0.002,1.416v-0.002c0.863,0.864,1.395,2.049,1.395,3.366c0,1.316-0.531,2.497-1.393,3.361c-0.394,0.389-0.394,1.022-0.002,1.415c0.195,0.195,0.451,0.293,0.707,0.293c0.257,0,0.513-0.098,0.708-0.293c1.222-1.22,1.98-2.915,1.979-4.776C20.788,14.136,20.027,12.439,18.806,11.219z M21.101,8.925c-0.393-0.391-1.024-0.391-1.413,0c-0.392,0.391-0.392,1.025,0,1.414c1.45,1.451,2.344,3.447,2.344,5.661c0,2.212-0.894,4.207-2.342,5.659c-0.392,0.39-0.392,1.023,0,1.414c0.195,0.195,0.451,0.293,0.708,0.293c0.256,0,0.512-0.098,0.707-0.293c1.808-1.809,2.929-4.315,2.927-7.073C24.033,13.24,22.912,10.732,21.101,8.925z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Volume3

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Volume3();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Volume3 = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Volume3",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M4.998,12.127v7.896h4.495l6.729,5.526l0.004-18.948l-6.73,5.526H4.998z M18.806,11.219c-0.393-0.389-1.024-0.389-1.415,0.002c-0.39,0.391-0.39,1.024,0.002,1.416v-0.002c0.863,0.864,1.395,2.049,1.395,3.366c0,1.316-0.531,2.497-1.393,3.361c-0.394,0.389-0.394,1.022-0.002,1.415c0.195,0.195,0.451,0.293,0.707,0.293c0.257,0,0.513-0.098,0.708-0.293c1.222-1.22,1.98-2.915,1.979-4.776C20.788,14.136,20.027,12.439,18.806,11.219z M21.101,8.925c-0.393-0.391-1.024-0.391-1.413,0c-0.392,0.391-0.392,1.025,0,1.414c1.45,1.451,2.344,3.447,2.344,5.661c0,2.212-0.894,4.207-2.342,5.659c-0.392,0.39-0.392,1.023,0,1.414c0.195,0.195,0.451,0.293,0.708,0.293c0.256,0,0.512-0.098,0.707-0.293c1.808-1.809,2.929-4.315,2.927-7.073C24.033,13.24,22.912,10.732,21.101,8.925z M23.28,6.746c-0.393-0.391-1.025-0.389-1.414,0.002c-0.391,0.389-0.391,1.023,0.002,1.413h-0.002c2.009,2.009,3.248,4.773,3.248,7.839c0,3.063-1.239,5.828-3.246,7.838c-0.391,0.39-0.391,1.023,0.002,1.415c0.194,0.194,0.45,0.291,0.706,0.291s0.513-0.098,0.708-0.293c2.363-2.366,3.831-5.643,3.829-9.251C27.115,12.389,25.647,9.111,23.28,6.746z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Key

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Key();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Key = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Key",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M18.386,16.009l0.009-0.006l-0.58-0.912c1.654-2.226,1.876-5.319,0.3-7.8c-2.043-3.213-6.303-4.161-9.516-2.118c-3.212,2.042-4.163,6.302-2.12,9.517c1.528,2.402,4.3,3.537,6.944,3.102l0.424,0.669l0.206,0.045l0.779-0.447l-0.305,1.377l2.483,0.552l-0.296,1.325l1.903,0.424l-0.68,3.06l1.406,0.313l-0.424,1.906l4.135,0.918l0.758-3.392L18.386,16.009z M10.996,8.944c-0.685,0.436-1.593,0.233-2.029-0.452C8.532,7.807,8.733,6.898,9.418,6.463s1.594-0.233,2.028,0.452C11.883,7.6,11.68,8.509,10.996,8.944z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Ruler

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Ruler();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Ruler = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Ruler",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M6.63,21.796l-5.122,5.121h25.743V1.175L6.63,21.796zM18.702,10.48c0.186-0.183,0.48-0.183,0.664,0l1.16,1.159c0.184,0.183,0.186,0.48,0.002,0.663c-0.092,0.091-0.213,0.137-0.332,0.137c-0.121,0-0.24-0.046-0.33-0.137l-1.164-1.159C18.519,10.96,18.519,10.664,18.702,10.48zM17.101,12.084c0.184-0.183,0.48-0.183,0.662,0l2.156,2.154c0.184,0.183,0.184,0.48,0.002,0.661c-0.092,0.092-0.213,0.139-0.334,0.139s-0.24-0.046-0.33-0.137l-2.156-2.154C16.917,12.564,16.917,12.267,17.101,12.084zM15.497,13.685c0.184-0.183,0.48-0.183,0.664,0l1.16,1.161c0.184,0.183,0.182,0.48-0.002,0.663c-0.092,0.092-0.211,0.138-0.33,0.138c-0.121,0-0.24-0.046-0.332-0.138l-1.16-1.16C15.314,14.166,15.314,13.868,15.497,13.685zM13.896,15.288c0.184-0.183,0.48-0.181,0.664,0.002l1.158,1.159c0.183,0.184,0.183,0.48,0,0.663c-0.092,0.092-0.212,0.138-0.332,0.138c-0.119,0-0.24-0.046-0.332-0.138l-1.158-1.161C13.713,15.767,13.713,15.471,13.896,15.288zM12.293,16.892c0.183-0.184,0.479-0.184,0.663,0l2.154,2.153c0.184,0.184,0.184,0.481,0,0.665c-0.092,0.092-0.211,0.138-0.33,0.138c-0.121,0-0.242-0.046-0.334-0.138l-2.153-2.155C12.11,17.371,12.11,17.075,12.293,16.892zM10.302,24.515c-0.091,0.093-0.212,0.139-0.332,0.139c-0.119,0-0.238-0.045-0.33-0.137l-2.154-2.153c-0.184-0.183-0.184-0.479,0-0.663s0.479-0.184,0.662,0l2.154,2.153C10.485,24.036,10.485,24.332,10.302,24.515zM10.912,21.918c-0.093,0.093-0.214,0.139-0.333,0.139c-0.12,0-0.24-0.045-0.33-0.137l-1.162-1.161c-0.184-0.183-0.184-0.479,0-0.66c0.184-0.185,0.48-0.187,0.664-0.003l1.161,1.162C11.095,21.438,11.095,21.735,10.912,21.918zM12.513,20.316c-0.092,0.092-0.211,0.138-0.332,0.138c-0.119,0-0.239-0.046-0.331-0.138l-1.159-1.16c-0.184-0.184-0.184-0.48,0-0.664s0.48-0.182,0.663,0.002l1.159,1.161C12.696,19.838,12.696,20.135,12.513,20.316zM22.25,21.917h-8.67l8.67-8.67V21.917zM22.13,10.7c-0.09,0.092-0.211,0.138-0.33,0.138c-0.121,0-0.242-0.046-0.334-0.138l-1.16-1.159c-0.184-0.183-0.184-0.479,0-0.663c0.182-0.183,0.479-0.183,0.662,0l1.16,1.159C22.312,10.221,22.313,10.517,22.13,10.7zM24.726,10.092c-0.092,0.092-0.213,0.137-0.332,0.137s-0.24-0.045-0.33-0.137l-2.154-2.154c-0.184-0.183-0.184-0.481,0-0.664s0.482-0.181,0.664,0.002l2.154,2.154C24.911,9.613,24.909,9.91,24.726,10.092z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Power

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Power();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Power = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Power",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M25.542,8.354c-1.47-1.766-2.896-2.617-3.025-2.695c-0.954-0.565-2.181-0.241-2.739,0.724c-0.556,0.961-0.24,2.194,0.705,2.763c0,0,0.001,0,0.002,0.001c0.001,0,0.002,0.001,0.003,0.002c0.001,0,0.003,0.001,0.004,0.001c0.102,0.062,1.124,0.729,2.08,1.925c1.003,1.261,1.933,3.017,1.937,5.438c-0.001,2.519-1.005,4.783-2.64,6.438c-1.637,1.652-3.877,2.668-6.368,2.669c-2.491-0.001-4.731-1.017-6.369-2.669c-1.635-1.654-2.639-3.919-2.64-6.438c0.005-2.499,0.995-4.292,2.035-5.558c0.517-0.625,1.043-1.098,1.425-1.401c0.191-0.152,0.346-0.263,0.445-0.329c0.049-0.034,0.085-0.058,0.104-0.069c0.005-0.004,0.009-0.006,0.012-0.008s0.004-0.002,0.004-0.002l0,0c0.946-0.567,1.262-1.802,0.705-2.763c-0.559-0.965-1.785-1.288-2.739-0.724c-0.128,0.079-1.555,0.93-3.024,2.696c-1.462,1.751-2.974,4.511-2.97,8.157C2.49,23.775,8.315,29.664,15.5,29.667c7.186-0.003,13.01-5.892,13.012-13.155C28.516,12.864,27.005,10.105,25.542,8.354zM15.5,17.523c1.105,0,2.002-0.907,2.002-2.023h-0.001V3.357c0-1.118-0.896-2.024-2.001-2.024s-2.002,0.906-2.002,2.024V15.5C13.498,16.616,14.395,17.523,15.5,17.523z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Unlock

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Unlock();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Unlock = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Unlock",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M20.375,12.833h-2.209V10c0,0,0,0,0-0.001c0-2.389,1.945-4.333,4.334-4.333c2.391,0,4.335,1.944,4.335,4.333c0,0,0,0,0,0v2.834h2V9.999h-0.001c-0.001-3.498-2.836-6.333-6.334-6.333S16.166,6.502,16.166,10v2.833H3.125V25h17.25V12.833z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Flag

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Flag();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Flag = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Flag",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M26.04,9.508c0.138-0.533,0.15-1.407,0.028-1.943l-0.404-1.771c-0.122-0.536-0.665-1.052-1.207-1.146l-3.723-0.643c-0.542-0.094-1.429-0.091-1.97,0.007l-4.033,0.726c-0.542,0.098-1.429,0.108-1.973,0.023L8.812,4.146C8.817,4.165,8.826,4.182,8.83,4.201l2.701,12.831l1.236,0.214c0.542,0.094,1.428,0.09,1.97-0.007l4.032-0.727c0.541-0.097,1.429-0.107,1.973-0.022l4.329,0.675c0.544,0.085,0.906-0.288,0.807-0.829l-0.485-2.625c-0.1-0.541-0.069-1.419,0.068-1.952L26.04,9.508zM6.667,3.636C6.126,3.75,5.78,4.279,5.894,4.819l5.763,27.378H13.7L7.852,4.409C7.736,3.867,7.207,3.521,6.667,3.636z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Tag

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Tag();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Tag = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Tag",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M14.263,2.826H7.904L2.702,8.028v6.359L18.405,30.09l11.561-11.562L14.263,2.826zM6.495,8.859c-0.619-0.619-0.619-1.622,0-2.24C7.114,6,8.117,6,8.736,6.619c0.62,0.62,0.619,1.621,0,2.241C8.117,9.479,7.114,9.479,6.495,8.859z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Search

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Search();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Search = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Search",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M29.772,26.433l-7.126-7.126c0.96-1.583,1.523-3.435,1.524-5.421C24.169,8.093,19.478,3.401,13.688,3.399C7.897,3.401,3.204,8.093,3.204,13.885c0,5.789,4.693,10.481,10.484,10.481c1.987,0,3.839-0.563,5.422-1.523l7.128,7.127L29.772,26.433zM7.203,13.885c0.006-3.582,2.903-6.478,6.484-6.486c3.579,0.008,6.478,2.904,6.484,6.486c-0.007,3.58-2.905,6.476-6.484,6.484C10.106,20.361,7.209,17.465,7.203,13.885z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ZoomOut

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ZoomOut();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ZoomOut = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ZoomOut",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M22.646,19.307c0.96-1.583,1.523-3.435,1.524-5.421C24.169,8.093,19.478,3.401,13.688,3.399C7.897,3.401,3.204,8.093,3.204,13.885c0,5.789,4.693,10.481,10.484,10.481c1.987,0,3.839-0.563,5.422-1.523l7.128,7.127l3.535-3.537L22.646,19.307zM13.688,20.369c-3.582-0.008-6.478-2.904-6.484-6.484c0.006-3.582,2.903-6.478,6.484-6.486c3.579,0.008,6.478,2.904,6.484,6.486C20.165,17.465,17.267,20.361,13.688,20.369zM8.854,11.884v4.001l9.665-0.001v-3.999L8.854,11.884z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.ZoomIn

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.ZoomIn();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.ZoomIn = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.ZoomIn",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M22.646,19.307c0.96-1.583,1.523-3.435,1.524-5.421C24.169,8.093,19.478,3.401,13.688,3.399C7.897,3.401,3.204,8.093,3.204,13.885c0,5.789,4.693,10.481,10.484,10.481c1.987,0,3.839-0.563,5.422-1.523l7.128,7.127l3.535-3.537L22.646,19.307zM13.688,20.369c-3.582-0.008-6.478-2.904-6.484-6.484c0.006-3.582,2.903-6.478,6.484-6.486c3.579,0.008,6.478,2.904,6.484,6.486C20.165,17.465,17.267,20.361,13.688,20.369zM15.687,9.051h-4v2.833H8.854v4.001h2.833v2.833h4v-2.834h2.832v-3.999h-2.833V9.051z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Cross

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Cross();
 *     icon.setDimension(50,50);
 *     canvas.addFigure(icon,50,10);
 *     
 * @inheritable
 * @author Andreas Herz
 * @extends draw2d.shape.icon.Icon
 */
draw2d.shape.icon.Cross = draw2d.shape.icon.Icon.extend({
    NAME : "draw2d.shape.icon.Cross",

    /**
     * 
     * @constructor
     * Creates a new icon element which are not assigned to any canvas.
     * @param {Number} [width] the width of the Oval
     * @param {Number} [height] the height of the Oval
     */
    init: function(width, height) {
      this._super(width, height);
    },

    /**
     * @private
     * @returns
     */
    createSet : function() {
        return this.canvas.paper.path("M24.778,21.419 19.276,15.917 24.777,10.415 21.949,7.585 16.447,13.087 10.945,7.585 8.117,10.415 13.618,15.917 8.116,21.419 10.946,24.248 16.447,18.746 21.948,24.248z");
    }
});


/*****************************************
 *   Library is under GPL License (GPL)
 *   Copyright (c) 2012 Andreas Herz
 ****************************************/
/**
 * @class draw2d.shape.icon.Check

 * See the example:
 *
 *     @example preview small frame
 *     
 *     var icon =  new draw2d.shape.icon.Check();
 *     icon.setDimension(50,50);
 *     canvas.addF