from __future__ import absolute_import
from __future__ import unicode_literals

import os
import json
import logging
import threading
import traceback
from django.db import connections
from debug_toolbar.panels.sql.panel import SQLPanel
from debug_toolbar.panels.sql.utils import reformat_sql
from debug_toolbar.middleware import DebugToolbarMiddleware

from .render import render_sql_panel
from . import settings as dt_settings


class SQLDebugMiddleware(DebugToolbarMiddleware):
    RESET_SEQ = "\033[0m"
    BLUE = '\033[94m'

    def process_response(self, request, response):
        toolbar = self.__class__.debug_toolbars.get(
            threading.current_thread().ident)

        # This is Ajax Debug toolbar, can't touch it
        if toolbar is None:
            return super(
                SQLDebugMiddleware,
                self).process_response(request, response)
        # Toolbar not rendered cause Content-Type is not html
        if not toolbar or not toolbar.store_id:
            toolbar.store()
            logging.info('No debug toolbar rendered. Creating it.')
        # Toolbar has been rendered
        else:
            logging.info('Debug toolbar already created.')

        sql_panel = None
        for panel in toolbar.panels:
            if isinstance(panel, SQLPanel):
                sql_panel = panel
                break
        if sql_panel is None:
            raise Exception("Can't find SQL Panel")

        for query in sql_panel._queries:
            db = query[0]
            query = query[1]
            print('%s%s%s' % (self.BLUE, query['sql'], self.RESET_SEQ))
            print('  [db: %s] [duration: %s]' % (db, query['duration']))
            print(''.join(traceback.format_list(query['stacktrace'])))
        _response = super(
            SQLDebugMiddleware,
            self).process_response(request, response)

        path = dt_settings.CONFIG['HTML_REPORT_PATH']
        if path is None:
            return _response
        if not os.path.exists(path):
            os.mkdir(path)

        for query in sql_panel._queries:
            ##############
            # SQL Select #
            ##############
            try:
                cursor = connections[query[0]].cursor()
                params = json.loads(query[1]['params'])
                cursor.execute(
                    query[1]['raw_sql'], params)
                headers = [d[0] for d in cursor.description]
                result = cursor.fetchall()
                cursor.close()
                query[1]['sql_select'] = {
                    'result': result,
                    'sql': reformat_sql(query[1]['raw_sql'] % tuple(params)),
                    'duration': query[1]['duration'],
                    'headers': headers,
                    'alias': query[0],
                }
            except Exception:
                print('Error while generating SQL report')
            ###############
            # SQL Explain #
            ###############
            sql = query[1]['raw_sql']
            params = json.loads(query[1]['params'])
            connection = connections[query[0]]
            vendor = connection.vendor
            cursor = connections[query[0]].cursor()

            if vendor == 'sqlite':
                # SQLite's EXPLAIN dumps the low-level opcodes
                # generated for a query;
                # EXPLAIN QUERY PLAN dumps a more human-readable summary
                # See http://www.sqlite.org/lang_explain.html for details
                cursor.execute("EXPLAIN QUERY PLAN %s" % (sql,), params)
            elif vendor == 'postgresql':
                cursor.execute("EXPLAIN ANALYZE %s" % (sql,), params)
            else:
                cursor.execute("EXPLAIN %s" % (sql,), params)

            headers = [d[0] for d in cursor.description]
            result = cursor.fetchall()
            cursor.close()
            query[1]['sql_explain'] = {
                'result': result,
                'sql': reformat_sql(query[1]['raw_sql'] % tuple(params)),
                'duration': query[1]['duration'],
                'headers': headers,
                'alias': query[0],
            }
            #################
            # SQL Profiling #
            #################
            sql = query[1]['raw_sql']
            params = json.loads(query[1]['params'])
            cursor = connections[query[0]].cursor()
            result = None
            headers = None
            result_error = None
            try:
                cursor.execute("SET PROFILING=1")  # Enable profiling
                cursor.execute(sql, params)  # Execute SELECT
                cursor.execute("SET PROFILING=0")  # Disable profiling
                # The Query ID should always be 1 here but
                # I'll subselect to get the last one just in case...
                cursor.execute("""
  SELECT  *
    FROM  information_schema.profiling
   WHERE  query_id = (
          SELECT  query_id
            FROM  information_schema.profiling
        ORDER BY  query_id DESC
           LIMIT  1
        )
""")
                headers = [d[0] for d in cursor.description]
                result = cursor.fetchall()
            except Exception:
                result_error = (
                    "Profiling is either not available or not "
                    "supported by your database.")
            cursor.close()
            query[1]['sql_profiling'] = {
                'result': result,
                'result_error': result_error,
                'sql': reformat_sql(query[1]['raw_sql'] % tuple(params)),
                'duration': query[1]['duration'],
                'headers': headers,
                'alias': query[0],
            }

        with open(os.path.join(path, 'output.html'), 'w') as f:
            f.write(render_sql_panel(sql_panel, request))

        return _response
