from __future__ import absolute_import
from __future__ import unicode_literals

import os
import logging
import threading
import traceback
from debug_toolbar.panels.sql.panel import SQLPanel
from debug_toolbar.middleware import DebugToolbarMiddleware

from .sql import sql_select
from .sql import sql_explain
from .sql import sql_profiling
from .render import render_sql_panel
from . import settings as dt_settings


class SQLDebugMiddleware(DebugToolbarMiddleware):
    RESET_SEQ = "\033[0m"
    BLUE = '\033[94m'

    def process_response(self, request, response):
        toolbar = self.__class__.debug_toolbars.get(
            threading.current_thread().ident)

        # This is Ajax Debug toolbar, can't touch it
        if toolbar is None:
            return super(
                SQLDebugMiddleware,
                self).process_response(request, response)
        # Toolbar not rendered cause Content-Type is not html
        if not toolbar or not toolbar.store_id:
            toolbar.store()
            logging.info('No debug toolbar rendered. Creating it.')
        # Toolbar has been rendered
        else:
            logging.info('Debug toolbar already created.')

        sql_panel = None
        for panel in toolbar.panels:
            if isinstance(panel, SQLPanel):
                sql_panel = panel
                break
        if sql_panel is None:
            raise Exception("Can't find SQL Panel")

        for query in sql_panel._queries:
            db = query[0]
            query = query[1]
            print('%s%s%s' % (self.BLUE, query['sql'], self.RESET_SEQ))
            print('  [db: %s] [duration: %s]' % (db, query['duration']))
            print(''.join(traceback.format_list(query['stacktrace'])))
        _response = super(
            SQLDebugMiddleware,
            self).process_response(request, response)

        path = dt_settings.CONFIG['HTML_REPORT_PATH']
        if path is None:
            return _response
        if not os.path.exists(path):
            os.mkdir(path)

        for query in sql_panel._queries:
            # SQL Select
            query = sql_select(query)
            # SQL Explain #
            query = sql_explain(query)
            # SQL Profiling #
            query = sql_profiling(query)

        with open(os.path.join(path, 'output.html'), 'w') as f:
            f.write(render_sql_panel(sql_panel, request))

        return _response
