import os.path
import re
from string import Formatter
from django.core.validators import deconstructible


class Wrapper(object):
    """Wraps a model to provide access to attributes as dict values so that it
    can be used with formatter classes.

    """
    def __init__(self, wrapped):
        self._wrapped = wrapped

    def __getitem__(self, key):
        wrapped_item = getattr(self._wrapped, key, None)
        if not wrapped_item:
            wrapped_item = getattr(self, key, None)
        if wrapped_item:
            return wrapped_item


class FilenameFormatter(Formatter):
    """
    Formats values for use in a filename.

    """
    def __init__(self, lowercase, nonwordchars, word_delimiter='_'):
        self.lowercase = lowercase
        self.nonwordchars = nonwordchars
        self.word_delimiter = word_delimiter

    def format_field(self, value, format_spec):
        """Formats the fields according to the options provided to the
        constructor.

        """
        value = super(FilenameFormatter, self).format_field(value, format_spec)
        if self.lowercase:
            value = value.lower()
        if not self.nonwordchars:
            value = re.sub('[^\w\s]+', '', value)
        value = re.sub('\s+', self.word_delimiter, value)
        return value


class MultiFormatter(Formatter):
    def __init__(self, formatter_map, default=None):
        self.formatter_map = formatter_map
        self.default = default

    def get_field(self, field_name, *args, **kwargs):
        if field_name in self.formatter_map:
            formatter = self.formatter_map[field_name]
        else:
            formatter = self.default
        return formatter.get_field(field_name, *args, **kwargs)


@deconstructible
class FormatFilename(object):
    def __init__(self, pattern, add_extension=True, lowercase=True,
                 nonwordchars=False, word_delimiter='_'):
        self.pattern = pattern
        self.add_extension = add_extension
        self.lowercase = lowercase
        self.nonwordchars = nonwordchars
        self.word_delimiter = word_delimiter

    def __call__(self, instance, old_filename):
        _filename, _ext = os.path.splitext(os.path.basename(old_filename))

        replacers = Wrapper(instance)
        setattr(replacers, '__filename', _filename)
        setattr(replacers, '__ext', _ext)

        default_formatter = FilenameFormatter(
            lowercase=self.lowercase, nonwordchars=self.nonwordchars,
            word_delimiter=self.word_delimiter)
        formatter_map = {
            '__ext': FilenameFormatter(lowercase=self.lowercase,
                                       nonwordchars=False,
                                       word_delimiter=self.word_delimiter),
            '__filename': FilenameFormatter(
                lowercase=self.lowercase, nonwordchars=False,
                word_delimiter=self.word_delimiter),
        }

        formatter = MultiFormatter(formatter_map, default=default_formatter)
        filename = formatter.vformat(self.pattern, [], replacers)

        # For backwards compatibility
        if self.add_extension and _ext not in filename:
            filename += _ext

        return filename


def format_filename(pattern, add_extension=True, lowercase=True,
                    nonwordchars=False, word_delimiter='_'):
    """
    Creates a method to be used as a value for Django models' upload_to
    argument. The returned method will format a filename based on properties of
    the model.

    Usage:
        thumbnail = models.ImageField(
            upload_to=format_filename('profile_images/{last_name}_{first_name}'))
    """
    return FormatFilename(pattern, add_extension, lowercase,
                          nonwordchars, word_delimiter)
