from unittest.mock import patch

from django.contrib.auth.tokens import default_token_generator
from django.contrib.sites.models import Site
from django.core.management.color import no_style
from django.db import connection
from django.db.models.base import ModelBase
from django.db.utils import IntegrityError
from django.test import TestCase
from django.utils import timezone
from django.utils.encoding import force_bytes
from django.utils.http import urlsafe_base64_encode

from . import models
from .factories import UserFactory
from .. import mixins


class AbstractModelMixin(object):
    """
    Mixin class for tests of (abstract) model mixins. To use, subclass and specify
    the mixin class variable. A model using the mixin will be made available in self.model.

    From http://michael.mior.ca/2012/01/14/blog/unit-testing-django-model-mixins.html
    """
    def setUp(self):
        # Create a dummy model which extends the mixin
        self._model_name = '__TestModel__' + self.mixin.__name__
        self.model = ModelBase(
            self._model_name,
            (self.mixin,),
            {'__module__': self.mixin.__module__},
        )

        # Create the schema for our test model
        self._style = no_style()
        sql, _ = connection.creation.sql_create_model(self.model, self._style)

        self._cursor = connection.cursor()
        for statement in sql:
            self._cursor.execute(statement)

    def tearDown(self):
        # Delete the schema for the test model
        sql = connection.creation.sql_destroy_model(self.model, (), self._style)
        for statement in sql:
            self._cursor.execute(statement)


class TestUser(TestCase):
    """Test the "User" model"""
    model = models.User

    def test_fields(self):
        """Do we have the fields we expect?"""
        fields = self.model._meta.get_all_field_names()
        expected = {
            # On model
            'id',
            'name',
            'date_joined',
            'email',
            'verified_email',
            'is_active',
            'is_staff',
            'is_superuser',
            'last_login',
            'password',

            # Incoming
            'groups',  # Django permission groups
            'user_permissions',
            'logentry',  # Django admin logs
        }
        self.assertCountEqual(fields, expected)

    def test_str(self):
        """Does "User.__str__()" work as expected?"""
        expected = 'Test Name'
        user = self.model(name=expected)
        self.assertEqual(str(user), expected)

    def test_name_methods(self):
        """Do "User.get_full_name()" & "get_short_name()" work as expected?"""
        expected = 'Professor Chaos'
        user = self.model(name=expected)
        self.assertEqual(user.get_full_name(), expected)
        self.assertEqual(user.get_short_name(), expected)


class TestUserManager(TestCase):
    manager = models.User.objects

    def test_create_user_without_email(self):
        with self.assertRaises(ValueError):
            self.manager.create_user(email='')
        self.assertFalse(self.manager.count())

    def test_create_duplicate_email(self):
        existing_user = UserFactory.create()
        with self.assertRaises(IntegrityError):
            self.manager.create_user(email=existing_user.email.upper())

    def test_create_user(self):
        time_before = timezone.now()
        data = {
            'email': 'valid@example.com',
            'name': 'Mysterion',
            'password': 'I can N3ver DIE!'
        }

        # Call creation method of manager
        with self.assertNumQueries(1):
            # Only one query:
            #     INSERT INTO "users_user" ("fields",)
            #         VALUES ('blah') RETURNING "users_user"."id"
            result = self.manager.create_user(**data)

        # Check that user returned is the right one
        user = self.manager.get()
        self.assertEqual(user, result)

        # Check that the password is valid
        self.assertTrue(user.check_password(data['password']))

        # Check name/email is correct
        self.assertEqual(user.name, data['name'])
        self.assertEqual(user.email, data['email'])

        # Check defaults
        self.assertFalse(user.is_active)
        self.assertFalse(user.is_staff)
        self.assertFalse(user.is_superuser)
        self.assertFalse(user.verified_email)

        # Check that the time is correct (or at least, in range)
        time_after = timezone.now()
        self.assertTrue(time_before < user.date_joined < time_after)

    def test_create_user_uppercase_email(self):
        email = 'VALID@EXAMPLE.COM'

        user = self.manager.create_user(email)
        self.assertEqual(email.lower(), user.email)

    def test_create_superuser(self):
        email = 'valid@example.com'
        password = 'password'

        # Call creation method of manager:
        with self.assertNumQueries(1):
            # Only one query:
            #     INSERT INTO "users_user" ("fields",)
            #         VALUES ('blah') RETURNING "users_user"."id"
            result = self.manager.create_superuser(email, password)

        # Check that user returned is the right one
        user = self.manager.get()
        self.assertEqual(user, result)

        # Check that the password is valid
        self.assertTrue(user.check_password(password))

        # Check defaults
        self.assertTrue(user.is_active)
        self.assertTrue(user.is_staff)
        self.assertTrue(user.is_superuser)


class TestVeryifyEmailMixin(AbstractModelMixin, TestCase):
    mixin = mixins.VeryifyEmailMixin

    def test_save(self):
        user = self.model()
        with patch.object(self.model, 'send_validation_email') as send:
            user.save()
        self.assertFalse(user.is_active)
        self.assertFalse(user.verified_email)
        send.assert_called_once_with()

    def test_send_validation_email(self):
        site = Site.objects.get_current()
        subject = '{} account validate'.format(site.domain)
        template = 'user_management/account_validation_email.html'
        user = self.model()

        # send_validation_email requires user to have an email
        user.email = 'dummy@test.com'

        uid = urlsafe_base64_encode(force_bytes(user.pk))

        with patch.object(default_token_generator, 'make_token') as make_token:
            with patch('user_management.models.mixins.send') as send:
                user.send_validation_email()

        send.assert_called_once_with(
            to=[user.email],
            subject=subject,
            template_name=template,
            extra_context={'token': make_token(), 'uid': uid},
        )

    def test_verified_email(self):
        user = self.model(verified_email=True)

        with patch('user_management.models.mixins.send') as send:
            with self.assertRaises(ValueError):
                user.send_validation_email()

        self.assertFalse(send.called)
