import inspect
import warnings
from django.conf import settings
from django.utils import importlib

from widgets import Widget
from exceptions import DuplicateWidget, WidgetNotFound

available_widgets = {}


def load_widgets():
    # Setup ignore list to avoid duplication
    django_widget_members = get_app_widgets('django_widgets')
    ignore_widgets = [widget_name for widget_name, _ in django_widget_members]

    for app in settings.INSTALLED_APPS:
        if app == 'django_widgets':
            continue

        widget_members = get_app_widgets(app)

        for widget_name, widget in widget_members:
            if widget_name in ignore_widgets:
                continue

            if widget_name in available_widgets:
                raise DuplicateWidget(widget_name)

            if issubclass(widget, Widget):
                available_widgets[widget_name] = widget


def get_app_widgets(app):
    try:
        importlib.import_module(app)
    except ImportError:
        warnings.ward("Could not import app: %s" % app)

    try:
        widgets_module = importlib.import_module("%s.widgets" % app)
    except ImportError:
        return []

    return inspect.getmembers(widgets_module, inspect.isclass)


def get_widget(name):
    if name in available_widgets:
        return available_widgets[name]()
    else:
        raise WidgetNotFound(name)
