import logging

from django import shortcuts
from django.conf import settings
from django.contrib.auth import REDIRECT_FIELD_NAME
from django.contrib.auth.views import (login as django_login,
                                       logout_then_login as django_logout)
from django.contrib.auth.decorators import login_required
from django.views.decorators.debug import sensitive_post_parameters
from django.utils.functional import curry
from django.views.decorators.cache import never_cache
from django.views.decorators.csrf import csrf_protect

from keystoneclient.v2_0 import client as keystone_client
from keystoneclient import exceptions as keystone_exceptions

from .forms import Login
from .user import set_session_from_user, create_user_from_token


LOG = logging.getLogger(__name__)


@sensitive_post_parameters()
@csrf_protect
@never_cache
def login(request):
    """ Logs a user in using the :class:`~openstack_auth.forms.Login` form. """
    # Get our initial region for the form.
    initial = {}
    current_region = request.session.get('region_endpoint', None)
    requested_region = request.GET.get('region', None)
    regions = dict(getattr(settings, "AVAILABLE_REGIONS", []))
    if requested_region in regions and requested_region != current_region:
        initial.update({'region': requested_region})

    if request.method == "POST":
        form = curry(Login, request)
    else:
        form = curry(Login, initial=initial)

    extra_context = {'redirect_field_name': REDIRECT_FIELD_NAME}

    if request.is_ajax():
        template_name = 'auth/_login.html'
        extra_context['hide'] = True
    else:
        template_name = 'auth/login.html'

    res = django_login(request,
                       template_name=template_name,
                       authentication_form=form,
                       extra_context=extra_context)
    # Set the session data here because django's session key rotation
    # will erase it if we set it earlier.
    if request.user.is_authenticated():
        set_session_from_user(request, request.user)
        regions = dict(Login.get_region_choices())
        region = request.user.endpoint
        region_name = regions.get(region)
        request.session['region_endpoint'] = region
        request.session['region_name'] = region_name
    return res


def logout(request):
    """ Securely logs a user out. """
    return django_logout(request)


@login_required
def switch(request, tenant_id):
    """ Switches an authenticated user from one tenant to another. """
    LOG.debug('Switching to tenant %s for user "%s".'
              % (tenant_id, request.user.username))
    endpoint = request.user.endpoint
    client = keystone_client.Client(endpoint=endpoint)
    try:
        token = client.tokens.authenticate(tenant_id=tenant_id,
                                        token=request.user.token.id)
    except keystone_exceptions.ClientException:
        token = None
        LOG.exception('An error occurred while switching sessions.')
    if token:
        user = create_user_from_token(request, token, endpoint)
        set_session_from_user(request, user)
    return shortcuts.redirect(settings.LOGIN_REDIRECT_URL)
