/*
 * Copyright 2011 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * The first half of the angular json wrapper.
 * @author corbinrsmith@gmail.com (Cory Smith)
 */

jstestdriver.angular = (function (angular, JSON, jQuery) {
  angular = angular || {};
  var _null = null;
  var $null = 'null';
  var _undefined;
  var $undefined = 'undefined';
  var $function = 'function';
  
  // library functions for angular.
  var isNumber = function (obj) {
    return (typeof obj).toLowerCase() == 'number' || obj instanceof Number;
  };
  
  var isObject = function (obj) {
    return obj != null && (typeof obj).toLowerCase() == 'object';
  };

  var isString = function (obj) {
    return (typeof obj).toLowerCase() == 'string' || obj instanceof String;
  };

  var isArray = function (obj) {
    return obj instanceof Array;
  };

  var isFunction = function (obj) {
    return (typeof obj).toLowerCase() == 'function';
  }

  var isBoolean = function (obj) {
    return (typeof obj).toLowerCase() == 'boolean' || obj instanceof Boolean;
  };

  var isUndefined = function (obj) {
    return (typeof obj).toLowerCase() == 'undefined';
  };

  var isDate = function (obj) {
    return obj instanceof Date;
  };

  var forEach = function (coll, callback) {
    jQuery.each(coll, function (index, value){
      return callback(value, index);
    });
  }

  function includes(arr, obj) {
    for (var i = 0; i < arr.length; i++) {
      if (arr[i] === obj) {
        return true;
      }
    }
    return false;
  }

  // extracted from https://github.com/angular/angular.js/blob/master/src/filters.js
  // Lines 106..129, 
  function padNumber(num, digits, trim) {
    var neg = '';
    if (num < 0) {
      neg =  '-';
      num = -num;
    }
    num = '' + num;
    while(num.length < digits) num = '0' + num;
    if (trim)
      num = num.substr(num.length - digits);
    return neg + num;
  }
  
  // extracted from https://github.com/angular/angular.js/blob/master/src/apis.js
  // Lines 721..782, 
  var R_ISO8061_STR = /^(\d{4})-(\d\d)-(\d\d)(?:T(\d\d)(?:\:(\d\d)(?:\:(\d\d)(?:\.(\d{3}))?)?)?Z)?$/;

  angular['String'] = {
    'quote':function(string) {
      return '"' + string.replace(/\\/g, '\\\\').
                          replace(/"/g, '\\"').
                          replace(/\n/g, '\\n').
                          replace(/\f/g, '\\f').
                          replace(/\r/g, '\\r').
                          replace(/\t/g, '\\t').
                          replace(/\v/g, '\\v') +
               '"';
    },
    'quoteUnicode':function(string) {
      var str = angular['String']['quote'](string);
      var chars = [];
      for ( var i = 0; i < str.length; i++) {
        var ch = str.charCodeAt(i);
        if (ch < 128) {
          chars.push(str.charAt(i));
        } else {
          var encode = "000" + ch.toString(16);
          chars.push("\\u" + encode.substring(encode.length - 4));
        }
      }
      return chars.join('');
    },

    /**
     * Tries to convert input to date and if successful returns the date, otherwise returns the input.
     * @param {string} string
     * @return {(Date|string)}
     */
    'toDate':function(string){
      var match;
      if (isString(string) && (match = string.match(R_ISO8061_STR))){
        var date = new Date(0);
        date.setUTCFullYear(match[1], match[2] - 1, match[3]);
        date.setUTCHours(match[4]||0, match[5]||0, match[6]||0, match[7]||0);
        return date;
      }
      return string;
    }
  };

  angular['Date'] = {
      'toString':function(date){
        return !date ?
                  date :
                  date.toISOString ?
                    date.toISOString() :
                    padNumber(date.getUTCFullYear(), 4) + '-' +
                    padNumber(date.getUTCMonth() + 1, 2) + '-' +
                    padNumber(date.getUTCDate(), 2) + 'T' +
                    padNumber(date.getUTCHours(), 2) + ':' +
                    padNumber(date.getUTCMinutes(), 2) + ':' +
                    padNumber(date.getUTCSeconds(), 2) + '.' +
                    padNumber(date.getUTCMilliseconds(), 3) + 'Z';
      }
    };
  /*The MIT License

Copyright (c) 2010 Adam Abrons and Misko Hevery http://getangular.com

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

var array = [].constructor;

/**
 * @workInProgress
 * @ngdoc function
 * @name angular.toJson
 * @function
 *
 * @description
 * Serializes the input into a JSON formated string.
 *
 * @param {Object|Array|Date|string|number} obj Input to jsonify.
 * @param {boolean=} pretty If set to true, the JSON output will contain newlines and whitespace.
 * @returns {string} Jsonified string representing `obj`.
 */
function toJson(obj, pretty) {
  var buf = [];
  toJsonArray(buf, obj, pretty ? "\n  " : _null, []);
  return buf.join('');
}

/**
 * @workInProgress
 * @ngdoc function
 * @name angular.fromJson
 * @function
 *
 * @description
 * Deserializes a string in the JSON format.
 *
 * @param {string} json JSON string to deserialize.
 * @param {boolean} [useNative=false] Use native JSON parser if available
 * @returns {Object|Array|Date|string|number} Deserialized thingy.
 */
function fromJson(json, useNative) {
  if (!isString(json)) return json;

  var obj, p, expression;

  try {
    if (useNative && JSON && JSON.parse) {
      obj = JSON.parse(json);
      return transformDates(obj);
    }

    p = parser(json, true);
    expression =  p.primary();
    p.assertAllConsumed();
    return expression();

  } catch (e) {
    error("fromJson error: ", json, e);
    throw e;
  }

  // TODO make forEach optionally recursive and remove this function
  function transformDates(obj) {
    if (isString(obj) && obj.length === DATE_ISOSTRING_LN) {
      return angularString.toDate(obj);
    } else if (isArray(obj) || isObject(obj)) {
      forEach(obj, function(val, name) {
        obj[name] = transformDates(val);
      });
    }
    return obj;
  }
}

angular['toJson'] = toJson;
angular['fromJson'] = fromJson;

function toJsonArray(buf, obj, pretty, stack) {
  if (isObject(obj)) {
    if (obj === window) {
      buf.push(angular['String']['quote']('WINDOW'));
      return;
    }

    if (obj === document) {
      buf.push(angular['String']['quote']('DOCUMENT'));
      return;
    }

    if (includes(stack, obj)) {
      buf.push(angular['String']['quote']('RECURSION'));
      return;
    }
    stack.push(obj);
  }
  if (obj === _null) {
    buf.push($null);
  } else if (obj instanceof RegExp) {
    buf.push(angular['String']['quoteUnicode'](obj.toString()));
  } else if (isFunction(obj)) {
    return;
  } else if (isBoolean(obj)) {
    buf.push('' + obj);
  } else if (isNumber(obj)) {
    if (isNaN(obj)) {
      buf.push($null);
    } else {
      buf.push('' + obj);
    }
  } else if (isString(obj)) {
    return buf.push(angular['String']['quoteUnicode'](obj));
  } else if (isObject(obj)) {
    if (isArray(obj)) {
      buf.push("[");
      var len = obj.length;
      var sep = false;
      for(var i=0; i<len; i++) {
        var item = obj[i];
        if (sep) buf.push(",");
        if (!(item instanceof RegExp) && (isFunction(item) || isUndefined(item))) {
          buf.push($null);
        } else {
          toJsonArray(buf, item, pretty, stack);
        }
        sep = true;
      }
      buf.push("]");
    } else if (isDate(obj)) {
      buf.push(angular['String']['quoteUnicode'](angular['Date']['toString'](obj)));
    } else {
      buf.push("{");
      if (pretty) buf.push(pretty);
      var comma = false;
      var childPretty = pretty ? pretty + "  " : false;
      var keys = [];
      for(var k in obj) {
        if (obj[k] === _undefined)
          continue;
        keys.push(k);
      }
      keys.sort();
      for ( var keyIndex = 0; keyIndex < keys.length; keyIndex++) {
        var key = keys[keyIndex];
        var value = obj[key];
        if (typeof value != $function) {
          if (comma) {
            buf.push(",");
            if (pretty) buf.push(pretty);
          }
          buf.push(angular['String']['quote'](key));
          buf.push(":");
          toJsonArray(buf, value, childPretty, stack);
          comma = true;
        }
      }
      buf.push("}");
    }
  }
  if (isObject(obj)) {
    stack.pop();
  }
}
/*
 * Copyright 2011 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * The second half of the angular json wrapper.
 * @author corbinrsmith@gmail.com (Cory Smith)
 */

  angular.toJson = toJson;
  angular.fromJson = fromJson;
  return angular;
})(jstestdriver.angular, jstestdriver.JSON, jstestdriver.jQuery);
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * A simple Signal object used for communication of state.
 * @param {Object} initial value.
 * @constructor
 */
jstestdriver.Signal = function(initial) {
  /**
   * @type {Object}
   */
  this.value_ = initial;
}


jstestdriver.Signal.prototype.get = function() {
  return this.value_;
}


jstestdriver.Signal.prototype.set = function(value) {
  this.value_ = value;
}
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * Watches for unauthorized unload events, such as a script calling reload,
 * or navigating to another page.
 *
 * @constructor
 * @param {jstestdriver.StreamingService} streamingService
 *     Used for server communication.
 * @param {function():jstestdriver.BrowserInfo} getBrowserInfo A function that 
 *     returns the browser information.
 * @param {function():String} getCommand Provides current command.
 * @param {jstestdriver.Signal} unloadSignal Signals if the unload command is expected.
 */
jstestdriver.PageUnloadHandler =
    function(streamingService, getBrowserInfo, getCommand, unloadSignal) {
  this.streamingService_ = streamingService;
  this.getBrowserInfo_ = getBrowserInfo;
  this.getCommand_ = getCommand
  this.unloadSignal_ = unloadSignal;
};


jstestdriver.PageUnloadHandler.prototype.onUnload = function(e) {
  if (!this.unloadSignal_.get()) {
    var type;
    try {
      type = e.type;
    } catch (e) {
      type = '[error while trying to get event type: ' + e + ']';
    }
    this.streamingService_.synchClose(
        new jstestdriver.Response(
            jstestdriver.RESPONSE_TYPES.BROWSER_PANIC,
            "Page reloaded unexpectedly during or after " + this.getCommand_() +
            " triggered by " + type,
            this.getBrowserInfo_(),
            false));
  }
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * Resets the javascript state by reloading or replacing current window.
 * @param {window.location} location The location object.
 * @param {jstestdriver.Signal} signal Signals that the window will be reloaded.
 * @param {function():Number} now Returns the current time in ms.
 */
jstestdriver.ResetCommand = function(location, signal, now) {
  /**
   * @type {window.location}
   * @private
   */
  this.location_ = location;

  /**
   * @type {window.location}
   * @private
   */
  this.signal_ = signal;

  /**
   * @type {function():Number}
   * @private
   */
  this.now_ = now;
};


/**
 * @param {string} loadType method of loading: "load" or "preload", default: "preload"
 * @param {string} testCaseId id of the test case to be run.
 */
jstestdriver.ResetCommand.prototype.reset = function(args) {
  this.signal_.set(true);
  var loadType = args[0] ? args[0] : 'preload';
  var testCaseId = args[1];
  if (!testCaseId) {
    loadType = 'load'
  }

  var now = this.now_();
  var hostPrefixPageAndPath = this.location_.href.match(/^(.*)\/(slave|runner|bcr)\/(.*)/);
  var hostAndPrefix = hostPrefixPageAndPath[1];
  var page = hostPrefixPageAndPath[2];
  var urlParts = hostPrefixPageAndPath[3].split('/');
  var newUrlParts = [hostAndPrefix, page];
  for (var i = 0; urlParts[i]; i++) {
    if (urlParts[i]=='testcase_id' ||
            urlParts[i]=='refresh' ||
            urlParts[i] == 'load_type') {
      i++; //skip the value
      continue;
    }
    newUrlParts.push(urlParts[i]);
  }
  newUrlParts.push('refresh');
  newUrlParts.push(now);
  newUrlParts.push('load_type');
  newUrlParts.push(loadType);
  if (testCaseId) {
    newUrlParts.push('testcase_id');
    newUrlParts.push(testCaseId);
  }
  var newUrl = newUrlParts.join('/');
  jstestdriver.log('Replacing ' + newUrl);
  this.location_.replace(newUrl);
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * A command that does nothing but contact the server and return a response.
 * Used to retrieve new commands from the server.
 * 
 * @constructor
 * 
 * @author corbinrsmith@gmail.com (Cory Smith)
 */
jstestdriver.NoopCommand = function(streamStop, getBrowserInfo) {
  /**
   * Function used to contact the server.
   * @type {function(jstestdriver.Response):null}
   */
  this.streamStop_ = streamStop;
  /**
   * Function used to retrieve the jstestdriver.BrowserInfo.
   * @type {function():jstestdriver.BrowserInfo}
   */
  this.streamStop_ = streamStop;
  this.getBrowserInfo_ = getBrowserInfo;
};

jstestdriver.NoopCommand.prototype.sendNoop = function() {
  
  this.streamStop_(
      new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.NOOP,
          '{}',
          this.getBrowserInfo_()));
}

/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * @fileoverview A series of constants to correlate to the type of responses
 * between the browser and the server.
 * 
 * See com.google.jstestdriver.Response.GSON_TYPES.
 * @author corbinrsmith@gmail.com (Cory Smith)
 */

/**
 * @enum {string}
 */
jstestdriver.RESPONSE_TYPES = {
  FILE_LOAD_RESULT: 'FILE_LOAD_RESULT',
  REGISTER_RESULT: 'REGISTER_RESULT',
  TEST_RESULT: 'TEST_RESULT',
  TEST_QUERY_RESULT: 'TEST_QUERY_RESULT',
  RESET_RESULT: 'RESET_RESULT',
  COMMAND_RESULT: 'COMMAND_RESULT',
  BROWSER_READY: 'BROWSER_READY',
  BROWSER_PANIC: 'BROWSER_PANIC',
  NOOP: 'NOOP',
  LOG: 'LOG'
};


/**
 * Contains the state of a response.
 * This is the javascript twin to com.google.jstestdriver.Response.
 * 
 * @param {jstestdriver.RESPONSE_TYPES} type The type of the response.
 * @param {String} response The serialized contents of the response.
 * @param {jstestdriver.BrowserInfo} browser The browser information. 
 * @param {Boolean} start Is this the first response from the browser.
 * @constructor
 */
jstestdriver.Response = function(type, response, browser, start) {
  this.type = type;
  this.response = response;
  this.browser = browser;
  if (start) {
    this.start = true;
  }
};


jstestdriver.Response.prototype.toString = function() {
  return 'Response(\nresponse=' + this.response + ',\ntype' + this.type + ',\n browser=' + this.browser + ')';
};


/**
 * @param {String} done Indicates if this is the last streamed message.
 * @param {jstestdriver.Response} response The response.
 * @constructor
 */
jstestdriver.CommandResponse = function (done, response) {
  this.done = done;
  this.response = response;
};


/**
 * Represents the information about the browser.
 * @param {Number} id The unique id of this browser.
 * @constructor
 */
jstestdriver.BrowserInfo = function(id) {
  this.id = id;
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
function expectAsserts(count) {
  jstestdriver.expectedAssertCount = count;
}


var fail = function fail(msg) {
  var err = new Error(msg);
  err.name = 'AssertError';

  if (!err.message) {
    err.message = msg;
  }

  throw err;
};


function isBoolean_(bool) {
  if (typeof(bool) != 'boolean') {
    fail('Not a boolean: ' + prettyPrintEntity_(bool));
  }
}


var isElement_ = (function () {
  var div = document.createElement('div');

  function isNode(obj) {
    try {
      div.appendChild(obj);
      div.removeChild(obj);
    } catch (e) {
      return false;
    }

    return true;
  }

  return function isElement(obj) {
    return obj && obj.nodeType === 1 && isNode(obj);
  };
}());


function formatElement_(el) {
  var tagName;

  try {
    tagName = el.tagName.toLowerCase();
    var str = '<' + tagName;
    var attrs = el.attributes, attribute;

    for (var i = 0, l = attrs.length; i < l; i++) {
      attribute = attrs.item(i);

      if (!!attribute.nodeValue) {
        str += ' ' + attribute.nodeName + '=\"' + attribute.nodeValue + '\"';
      }
    }

    return str + '>...</' + tagName + '>';
  } catch (e) {
    return '[Element]' + (!!tagName ? ' ' + tagName : '');
  }
}


function prettyPrintEntity_(entity) {
  if (isElement_(entity)) {
    return formatElement_(entity);
  }

  var str;

  if (typeof entity == 'function') {
    try {
      str = entity.toString().match(/(function [^\(]+\(\))/)[1];
    } catch (e) {}

    return str || '[function]';
  }

  try {
    str = JSON.stringify(entity);
  } catch (e) {}

  return str || '[' + typeof entity + ']';
}


function argsWithOptionalMsg_(args, length) {
  var copyOfArgs = [];
  // make copy because it's bad practice to change a passed in mutable
  // And to ensure we aren't working with an arguments array. IE gets bitchy.
  for(var i = 0; i < args.length; i++) {
    copyOfArgs.push(args[i]);
  }
  var min = length - 1;

  if (args.length < min) {
    fail('expected at least ' + min + ' arguments, got ' + args.length);
  } else if (args.length == length) {
    copyOfArgs[0] += ' ';
  } else {
    copyOfArgs.unshift('');
  }
  return copyOfArgs;
}


function assertTrue(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  isBoolean_(args[1]);
  if (args[1] != true) {
    fail(args[0] + 'expected true but was ' + prettyPrintEntity_(args[1]));
  }
  return true;
}


function assertFalse(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  isBoolean_(args[1]);
  if (args[1] != false) {
    fail(args[0] + 'expected false but was ' + prettyPrintEntity_(args[1]));
  }
  return true;
}


function assertEquals(msg, expected, actual) {
  var args = argsWithOptionalMsg_(arguments, 3);
  jstestdriver.assertCount++;
  msg = args[0];
  expected = args[1];
  actual = args[2];

  if (!compare_(expected, actual)) {
    fail(msg + 'expected ' + prettyPrintEntity_(expected) + ' but was ' +
        prettyPrintEntity_(actual) + '');
  }
  return true;
}


function compare_(expected, actual) {
  if (expected === actual) {
    return true;
  }

  if (typeof expected != 'object' ||
      typeof actual != 'object' ||
      !expected || !actual) {
    return expected == actual;
  }

  if (isElement_(expected) || isElement_(actual)) {
    return false;
  }

  var key = null;
  var actualLength   = 0;
  var expectedLength = 0;

  try {
    // If an array is expected the length of actual should be simple to
    // determine. If it is not it is undefined.
    if (jstestdriver.jQuery.isArray(actual)) {
      actualLength = actual.length;
    } else {
      // In case it is an object it is a little bit more complicated to
      // get the length.
      for (key in actual) {
        if (actual.hasOwnProperty(key)) {
          ++actualLength;
        }
      }
    }

    // Arguments object
    if (actualLength == 0 && typeof actual.length == 'number') {
      actualLength = actual.length;

      for (var i = 0, l = actualLength; i < l; i++) {
        if (!(i in actual)) {
          actualLength = 0;
          break;
        }
      }
    }

    for (key in expected) {
      if (expected.hasOwnProperty(key)) {
        if (!compare_(expected[key], actual[key])) {
          return false;
        }

        ++expectedLength;
      }
    }

    if (expectedLength != actualLength) {
      return false;
    }

    return expectedLength == 0 ? expected.toString() == actual.toString() : true;
  } catch (e) {
    return false;
  }
}


function assertNotEquals(msg, expected, actual) {
  try {
    assertEquals.apply(this, arguments);
  } catch (e) {
    if (e.name == 'AssertError') {
      return true;
    }

    throw e;
  }

  var args = argsWithOptionalMsg_(arguments, 3);

  fail(args[0] + 'expected ' + prettyPrintEntity_(args[1]) +
      ' not to be equal to ' + prettyPrintEntity_(args[2]));
}


function assertSame(msg, expected, actual) {
  var args = argsWithOptionalMsg_(arguments, 3);
  jstestdriver.assertCount++;

  if (!isSame_(args[2], args[1])) {
    fail(args[0] + 'expected ' + prettyPrintEntity_(args[1]) + ' but was ' +
        prettyPrintEntity_(args[2]));
  }
  return true;
}


function assertNotSame(msg, expected, actual) {
  var args = argsWithOptionalMsg_(arguments, 3);
  jstestdriver.assertCount++;

  if (isSame_(args[2], args[1])) {
    fail(args[0] + 'expected not same as ' + prettyPrintEntity_(args[1]) +
        ' but was ' + prettyPrintEntity_(args[2]));
  }
  return true;
}


function isSame_(expected, actual) {
  return actual === expected;
}


function assertNull(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (args[1] !== null) {
    fail(args[0] + 'expected null but was ' + prettyPrintEntity_(args[1]));
  }
  return true;
}


function assertNotNull(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (args[1] === null) {
    fail(args[0] + 'expected not null but was null');
  }

  return true;
}


function assertUndefined(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (typeof args[1] != 'undefined') {
    fail(args[2] + 'expected undefined but was ' + prettyPrintEntity_(args[1]));
  }
  return true;
}


function assertNotUndefined(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (typeof args[1] == 'undefined') {
    fail(args[0] + 'expected not undefined but was undefined');
  }
  return true;
}


function assertNaN(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (!isNaN(args[1])) {
    fail(args[0] + 'expected to be NaN but was ' + args[1]);
  }

  return true;
}


function assertNotNaN(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (isNaN(args[1])) {
    fail(args[0] + 'expected not to be NaN');
  }

  return true;
}


function assertException(msg, callback, error) {
  if (arguments.length == 1) {
    // assertThrows(callback)
    callback = msg;
    msg = '';
  } else if (arguments.length == 2) {
    if (typeof callback != 'function') {
      // assertThrows(callback, type)
      error = callback;
      callback = msg;
      msg = '';
    } else {
      // assertThrows(msg, callback)
      msg += ' ';
    }
  } else {
    // assertThrows(msg, callback, type)
    msg += ' ';
  }

  jstestdriver.assertCount++;

  try {
    callback();
  } catch(e) {
    if (e.name == 'AssertError') {
      throw e;
    }

    if (error && e.name != error) {
      fail(msg + 'expected to throw ' + error + ' but threw ' + e.name);
    }

    return true;
  }

  fail(msg + 'expected to throw exception');
}


function assertNoException(msg, callback) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  try {
    args[1]();
  } catch(e) {
    fail(args[0] + 'expected not to throw exception, but threw ' + e.name +
        ' (' + e.message + ')');
  }
}


function assertArray(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  jstestdriver.assertCount++;

  if (!jstestdriver.jQuery.isArray(args[1])) {
    fail(args[0] + 'expected to be array, but was ' +
        prettyPrintEntity_(args[1]));
  }
}


function assertTypeOf(msg, expected, value) {
  var args = argsWithOptionalMsg_(arguments, 3);
  jstestdriver.assertCount++;
  var actual = typeof args[2];

  if (actual != args[1]) {
    fail(args[0] + 'expected to be ' + args[1] + ' but was ' + actual);
  }

  return true;
}


function assertBoolean(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  return assertTypeOf(args[0], 'boolean', args[1]);
}


function assertFunction(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  return assertTypeOf(args[0], 'function', args[1]);
}


function assertObject(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  return assertTypeOf(args[0], 'object', args[1]);
}


function assertNumber(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  return assertTypeOf(args[0], 'number', args[1]);
}


function assertString(msg, actual) {
  var args = argsWithOptionalMsg_(arguments, 2);
  return assertTypeOf(args[0], 'string', args[1]);
}


function assertMatch(msg, regexp, actual) {
  var args = argsWithOptionalMsg_(arguments, 3);
  var isUndef = typeof args[2] == 'undefined';
  jstestdriver.assertCount++;
  var _undef;

  if (isUndef || !args[1].test(args[2])) {
    actual = (isUndef ? _undef : prettyPrintEntity_(args[2]));
    fail(args[0] + 'expected ' + actual + ' to match ' + args[1]);
  }

  return true;
}


function assertNoMatch(msg, regexp, actual) {
  var args = argsWithOptionalMsg_(arguments, 3);
  jstestdriver.assertCount++;

  if (args[1].test(args[2])) {
    fail(args[0] + 'expected ' + prettyPrintEntity_(args[2]) +
        ' not to match ' + args[1]);
  }

  return true;
}


function assertTagName(msg, tagName, element) {
  var args = argsWithOptionalMsg_(arguments, 3);
  var actual = args[2] && args[2].tagName;

  if (String(actual).toUpperCase() != args[1].toUpperCase()) {
    fail(args[0] + 'expected tagName to be ' + args[1] + ' but was ' + actual);
  }
  return true;
}


function assertClassName(msg, className, element) {
  var args = argsWithOptionalMsg_(arguments, 3);
  var actual = args[2] && args[2].className;
  var regexp = new RegExp('(^|\\s)' + args[1] + '(\\s|$)');

  try {
    assertMatch(args[0], regexp, actual);
  } catch (e) {
    actual = prettyPrintEntity_(actual);
    fail(args[0] + 'expected class name to include ' +
        prettyPrintEntity_(args[1]) + ' but was ' + actual);
  }

  return true;
}


function assertElementId(msg, id, element) {
  var args = argsWithOptionalMsg_(arguments, 3);
  var actual = args[2] && args[2].id;
  jstestdriver.assertCount++;

  if (actual !== args[1]) {
    fail(args[0] + 'expected id to be ' + args[1] + ' but was ' + actual);
  }

  return true;
}


function assertInstanceOf(msg, constructor, actual) {
  jstestdriver.assertCount++;
  var args = argsWithOptionalMsg_(arguments, 3);
  var pretty = prettyPrintEntity_(args[2]);
  var expected = args[1] && args[1].name || args[1];

  if (args[2] == null) {
    fail(args[0] + 'expected ' + pretty + ' to be instance of ' + expected);
  }

  if (!(Object(args[2]) instanceof args[1])) {
    fail(args[0] + 'expected ' + pretty + ' to be instance of ' + expected);
  }

  return true;
}


function assertNotInstanceOf(msg, constructor, actual) {
  var args = argsWithOptionalMsg_(arguments, 3);
  jstestdriver.assertCount++;

  if (Object(args[2]) instanceof args[1]) {
    var expected = args[1] && args[1].name || args[1];
    var pretty = prettyPrintEntity_(args[2]);
    fail(args[0] + 'expected ' + pretty + ' not to be instance of ' + expected);
  }

  return true;
}

/**
 * Asserts that two doubles, or the elements of two arrays of doubles,
 * are equal to within a positive delta.
 */
function assertEqualsDelta(msg, expected, actual, epsilon) {
  var args = this.argsWithOptionalMsg_(arguments, 4);
  jstestdriver.assertCount++;
  msg = args[0];
  expected = args[1];
  actual = args[2];
  epsilon = args[3];

  if (!compareDelta_(expected, actual, epsilon)) {
    this.fail(msg + 'expected ' + epsilon + ' within ' +
              this.prettyPrintEntity_(expected) +
              ' but was ' + this.prettyPrintEntity_(actual) + '');
  }
  return true;
};

function compareDelta_(expected, actual, epsilon) {
  var compareDouble = function(e,a,d) {
    return Math.abs(e - a) <= d;
  }
  if (expected === actual) {
    return true;
  }

  if (typeof expected == "number" ||
      typeof actual == "number" ||
      !expected || !actual) {
    return compareDouble(expected, actual, epsilon);
  }

  if (isElement_(expected) || isElement_(actual)) {
    return false;
  }

  var key = null;
  var actualLength   = 0;
  var expectedLength = 0;

  try {
    // If an array is expected the length of actual should be simple to
    // determine. If it is not it is undefined.
    if (jstestdriver.jQuery.isArray(actual)) {
      actualLength = actual.length;
    } else {
      // In case it is an object it is a little bit more complicated to
      // get the length.
      for (key in actual) {
        if (actual.hasOwnProperty(key)) {
          ++actualLength;
        }
      }
    }

    // Arguments object
    if (actualLength == 0 && typeof actual.length == "number") {
      actualLength = actual.length;

      for (var i = 0, l = actualLength; i < l; i++) {
        if (!(i in actual)) {
          actualLength = 0;
          break;
        }
      }
    }

    for (key in expected) {
      if (expected.hasOwnProperty(key)) {
        if (!compareDelta_(expected[key], actual[key], epsilon)) {
          return false;
        }

        ++expectedLength;
      }
    }

    if (expectedLength != actualLength) {
      return false;
    }

    return expectedLength == 0 ? expected.toString() == actual.toString() : true;
  } catch (e) {
    return false;
  }
};

var assert = assertTrue;
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * Service for streaming information to the server.
 * @param {string} url The server url.
 * @param {function():Number} now Returns the current time in ms.
 * @param {function(String, Object, function():null)} post Posts to the server.
 * @param {function(String, Object)} synchPost Posts synchronously to the server.
 * @param {function(Function, Number)} The setTimeout for asynch xhrs.
 * @param {jstestdriver.Signal} unloadSignal
 * @constructor
 */
// TODO(corysmith): Separate the state from the service.
jstestdriver.StreamingService = function(url, now, post, synchPost, setTimeout, unloadSignal) {
  this.url_ = url;
  this.now_ = now;
  this.post_ = post;
  this.activeResponses_ = {};
  this.finishedResponses_ = {};
  this.completeFinalResponse = null;
  this.synchPost_ = synchPost;
  this.setTimeout_ = setTimeout;
  this.unloadSignal_ = unloadSignal;
};


jstestdriver.StreamingService.prototype.synchClose = function(response) {
  var data = new jstestdriver.CommandResponse(true, response);
  this.synchPost_(this.url_, data);
  this.unloadSignal_.set(true);
};


jstestdriver.StreamingService.prototype.stream = function(response, callback) {
  this.streamResponse(response, false, callback);
};


jstestdriver.StreamingService.prototype.streamResponse = function(response,
                                                                  done,
                                                                  callback) {
  var data = new jstestdriver.CommandResponse(done, response);
  if (!done && response != null) {
    data.responseId = this.now_();
    // no ack expected after the final response, and no ack expected on no response
    this.activeResponses_[data.responseId] = data;
  }
  var context = this;
  this.setTimeout_(function() {
    context.post_(context.url_, data, callback, 'text/plain');
  }, 1);
};


/**
 * Callback command for the stream acknowledge to a streamed responses.
 * @param {Array.<string>} received A list of received ids for the currently open stream.
 */
jstestdriver.StreamingService.prototype.streamAcknowledged = function(received) {
  for (var i = 0; received && received[i]; i++) {
    if (this.activeResponses_[received[i]]) {
      // cut down on memory goof ups....
      this.activeResponses_[received[i]] = null;
      delete this.activeResponses_[received[i]];
      this.finishedResponses_[received[i]] = true;
    }
  }

  // TODO(corysmith): This causes a extra traffic on close, as the service tries
  // to verify the received responses. Setup a timeout to reduce the queries to 
  // the server.
  if (this.completeFinalResponse) {
    this.completeFinalResponse()
  }
};


/**
 * Closes the current streaming session, sending the final response after all
 * other Responses are finished.
 * @param {!jstestdriver.Response} finalResponse The final response to send.
 * @param {!Function} callback The callback when the post is finished.
 */
jstestdriver.StreamingService.prototype.close =
    function(finalResponse, callback) {
  var context = this;
  this.completeFinalResponse = function() {
    if (context.hasOpenResponses()) {
      // have to query again, because these may be lost responses from a debug session.
      context.streamResponse(null, false, callback);
    } else {
      context.completeFinalResponse = null;
      context.activeResponses_ = {};
      context.finishedResponses_ = {};
      context.streamResponse(finalResponse, true, callback);
      this.unloadSignal_.set(true);
    }
  };

  this.completeFinalResponse();
};


/**
 * Indicates if there are currently open streamed response.
 * @return {Boolean} True for open responses, otherwise false.
 */
jstestdriver.StreamingService.prototype.hasOpenResponses = function() {
  for (var responseId in this.activeResponses_) {
    if (this.activeResponses_.hasOwnProperty(responseId)) {
      return true;
    }
  }
  return false;
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @param fileSrc
 * @param timestamp
 * @param basePath
 * @constructor
 */
jstestdriver.FileSource = function(fileSrc, timestamp, basePath) {
  this.fileSrc = fileSrc;
  this.timestamp = timestamp;
  this.basePath = basePath;
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @param file
 * @param success
 * @param message
 * @param elapsed
 * @constructor
 */
jstestdriver.FileResult = function(file, success, message, elapsed) {
  this.file = file;
  this.success = success;
  this.message = message;
  this.elapsed = elapsed
};

jstestdriver.FileResult.prototype.toString = function() {
  return ["FileResult(", this.file.fileSrc, this.success, this.message, ")"].join("");
}
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
/**
 * The PluginRegistrar allows developers to load their own plugins to perform certain actions.
 * A plugin must define methods with specific names in order for it to be used.
 * A plugin must define a name, by defining a property name.
 * 
 * A simple plugin supporting the loadSource method for example could look as follow:
 * 
 * var myPlugin = {
 *   name: 'myPlugin',
 *   loadSource: function(file, onSourceLoad) {
 *     // do some cool stuff
 *   }
 * };
 * 
 * To then register it one needs to call:
 * 
 * jstestdriver.pluginRegistrar.register(myPlugin);
 * 
 * The list of supported methods is:
 * - loadSource
 * - runTestConfiguration
 * 
 * For more information regarding the supported method just read the documentation for the method
 * in this class.
 * @constructor
 */
jstestdriver.PluginRegistrar = function() {
  this.plugins_ = [];
};


jstestdriver.PluginRegistrar.PROCESS_TEST_RESULT = 'processTestResult';


jstestdriver.PluginRegistrar.LOAD_SOURCE = 'loadSource';


jstestdriver.PluginRegistrar.RUN_TEST = 'runTestConfiguration';


jstestdriver.PluginRegistrar.IS_FAILURE = 'isFailure';


jstestdriver.PluginRegistrar.GET_TEST_RUN_CONFIGURATIONS = 'getTestRunsConfigurationFor';


jstestdriver.PluginRegistrar.ON_TESTS_START = 'onTestsStart';

jstestdriver.PluginRegistrar.ON_TESTS_FINISH = 'onTestsFinish';


jstestdriver.PluginRegistrar.prototype.register = function(plugin) {
  if (!plugin.name) {
    throw new Error("Plugins must define a name.");
  }
  var index = this.getIndexOfPlugin_(plugin.name);
  var howMany = 1;

  if (index == -1) {
    index = this.plugins_.length - 1;
    howMany = 0;
  }
  this.plugins_.splice(index, howMany, plugin);
};


jstestdriver.PluginRegistrar.prototype.unregister = function(plugin) {
  var index = this.getIndexOfPlugin_(plugin.name);

  if (index != -1) {
    this.plugins_.splice(index, 1);
  }
};


jstestdriver.PluginRegistrar.prototype.getPlugin = function(name) {
  var index = this.getIndexOfPlugin_(name);

  return index != -1 ? this.plugins_[index] : null;
};


jstestdriver.PluginRegistrar.prototype.getNumberOfRegisteredPlugins = function() {
  return this.plugins_.length;
};


jstestdriver.PluginRegistrar.prototype.dispatch_ = function(method, parameters) {
  var size = this.plugins_.length;

  for (var i = 0; i < size; i++) {
    var plugin = this.plugins_[i];

    if (plugin[method]) {
      if (plugin[method].apply(plugin, parameters)) {
        return true;
      }
    }
  }
  return false;
};


jstestdriver.PluginRegistrar.prototype.getIndexOfPlugin_ = function(name) {
  var size = this.plugins_.length;

  for (var i = 0; i < size; i++) {
    var plugin = this.plugins_[i];

    if (plugin.name == name) {
      return i;
    }
  }
  return -1;
};


/**
 * loadSource
 * 
 * By defining the method loadSource a plugin can implement its own way of loading certain types of
 * files.
 * 
 * loadSource takes 2 parameters:
 *  - file: A file object defined as -> { fileSrc: string, timestamp: number, basePath: string }
 *    fileSrc is the name of the file
 *    timestamp is the last modified date of the file
 *    basePath is defined if the file is a URL and the URL has been rewritten
 *  - onSourceLoad: A callback that must be called once the file has been loaded the callback takes
 *    1 parameter defined as -> { file: file object, success: boolean, message: string }
 *    file: A file object
 *    success: a boolean, true if the file was loaded successfully, false otherwise
 *    message: an error message if the file wasn't loaded properly
 *  
 *  loadSource must return a boolean:
 *  - true if the plugin knows how to and loaded the file
 *  - false if the plugin doesn't know how to load the file
 *  
 *  A simple loadSource plugin would look like:
 *  
 *  var myPlugin = {
 *    name: 'myPlugin',
 *    loadSource: function(file, onSourceLoad) {
 *      // load the file
 *      return true;
 *    }
 *  }
 */
jstestdriver.PluginRegistrar.prototype.loadSource = function(file, onSourceLoad) {
  this.dispatch_(jstestdriver.PluginRegistrar.LOAD_SOURCE, arguments);
};


/**
 * runTestConfiguration
 * 
 * By defining the method runTestConfiguration a plugin can implement its own way of running
 * certain types of tests.
 * 
 * runTestConfiguration takes 3 parameters:
 * - testRunConfiguration: A jstestdriver.TestRunConfiguration object.
 * - onTestDone: A callback that needs to be call when a test ran so that the results are properly
 *   sent back to the client. It takes 1 parameter a jstestdriver.TestResult.
 * - onTestRunConfigurationComplete: A callback that needs to be call when everything ran. It takes
 *   no parameter.
 *   
 * runTestConfiguration must return a boolean:
 * - true if the plugin can run the tests
 * - false if the plugin can not run the tests
 * 
 * A simple runTestConfiguration plugin would look like:
 * 
 * var myPlugin = {
 *   name: 'myPlugin',
 *   runTestConfiguration: function(testRunConfiguration, onTestDone,
 *       onTestRunConfigurationComplete) {
 *     // run the tests
 *     return true;
 *   }
 * }
 * 
 */
jstestdriver.PluginRegistrar.prototype.runTestConfiguration = function(testRunConfiguration,
    onTestDone, onTestRunConfigurationComplete) {
  this.dispatch_(jstestdriver.PluginRegistrar.RUN_TEST, arguments);
};


/**
 * processTestResult
 * 
 * By defining the method processTestResult a plugin can pass extra meta data about a test back to
 * the server.
 * 
 * processTestResult takes 1 parameter:
 * - testResult: The TestResult of the most recently run test.
 *              
 *   
 * processTestResult must return a boolean:
 * - true to allow other plugins to process the test result
 * - false if not further processing should be allowed.
 * 
 * A simple processTestResult plugin would look like:
 * 
 * var myPlugin = {
 *   name: 'myPlugin',
 *   processTestResult: function(testResult) {
 *     testResult.data.foo = 'bar';
 *     return true;
 *   }
 * }
 * 
 */
jstestdriver.PluginRegistrar.prototype.processTestResult = function(testResult) {
  this.dispatch_(jstestdriver.PluginRegistrar.PROCESS_TEST_RESULT, arguments);
};


/**
 * isFailure
 * 
 * By defining the method isFailure a plugin will determine if an exception thrown by an assertion
 * framework is considered a failure by the assertion framework.
 * 
 * isFailure takes 1 parameter:
 * - exception: The exception thrown by the test
 *              
 *   
 * processTestResult must return a boolean:
 * - true if the exception is considered a failure
 * - false if the exception is not considered a failure
 * 
 * A simple isFailure plugin would look like:
 * 
 * var myPlugin = {
 *   name: 'myPlugin',
 *   isFailure: function(exception) {
 *     return exception.name == 'AssertError';
 *   }
 * }
 * 
 */
jstestdriver.PluginRegistrar.prototype.isFailure = function(exception) {
  return this.dispatch_(jstestdriver.PluginRegistrar.IS_FAILURE, arguments);
};


/**
 * getTestRunsConfigurationFor
 * 
 * By defining the method getTestRunsConfigurationFor a plugin will be able to
 * modify the process in which a TestCaseInfo provides a TestRunConfiguration from an expression
 * such as fooCase.testBar.
 * 
 * getTestRunsConfigurationFor takes 3 parameters:
 * - testCaseInfos: The array of loaded TestCaseInfos.
 * - expressions: The array of expressions used to determine the TestRunConfiguration.
 * - testRunsConfiguration: An array to add test case TestRunConfigurations to.
 * 
 */
jstestdriver.PluginRegistrar.prototype.getTestRunsConfigurationFor =
    function(testCaseInfos, expressions, testRunsConfiguration) {
  return this.dispatch_(jstestdriver.PluginRegistrar.GET_TEST_RUN_CONFIGURATIONS, arguments);
};


/**
 * onTestsStart
 * 
 * A setup hook called before all tests are run.
 * 
 */
jstestdriver.PluginRegistrar.prototype.onTestsStart = function() {
  return this.dispatch_(jstestdriver.PluginRegistrar.ON_TESTS_START, []);
};


/**
 * onTestsFinish
 * 
 * A tear down hook called after all tests are run.
 * 
 */
jstestdriver.PluginRegistrar.prototype.onTestsFinish = function() {
  return this.dispatch_(jstestdriver.PluginRegistrar.ON_TESTS_FINISH, []);
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.LibLoader = function(files, dom, getScript) {
  this.files_ = files;
  this.dom_ = dom;
  this.getScript_ = getScript;
  this.remainingLibToLoad_ = this.files_.length;
  this.boundOnLibLoaded_ = jstestdriver.bind(this, this.onLibLoaded);
  this.savedDocumentWrite_ = dom.write;
  this.currentFile_ = 0;
};


jstestdriver.LibLoader.prototype.load = function(onAllLibLoaded, data) {
  if (this.files_.length == 0) {
    onAllLibLoaded(data);
  } else {
    this.dom_.write = function() {};
    this.onAllLibLoaded_ = onAllLibLoaded;
    this.data_ = data;
    this.getScript_(this.dom_, this.files_[this.currentFile_++], this.boundOnLibLoaded_);
  }
};


jstestdriver.LibLoader.prototype.onLibLoaded = function() {
  if (--this.remainingLibToLoad_ == 0) {
    var onAllLibLoaded = this.onAllLibLoaded_;
    var data = this.data_;

    this.onAllLibLoaded_ = null;
    this.data_ = null;
    this.dom_.write = this.savedDocumentWrite_;
    onAllLibLoaded(data);
  } else {
    this.getScript_(this.dom_, this.files_[this.currentFile_++], this.boundOnLibLoaded_);
  }
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.FileLoader = function(pluginRegistrar, onAllFilesLoaded) {
  this.pluginRegistrar_ = pluginRegistrar;
  this.onAllFilesLoaded_ = onAllFilesLoaded;
  this.boundOnFileLoaded = jstestdriver.bind(this, this.onFileLoaded_);
  this.boundLoadFile_ = jstestdriver.bind(this, this.onLoadFile_);
  this.loadedFiles_ = [];
};


/**
 * Load files.
 * 
 * files is an array containing jstestdriver.FileSource objects.
 */
jstestdriver.FileLoader.prototype.load = function(files) {
  this.files_ = files;
  if (this.files_.length > 0) {
    this.loadFile_(this.files_.shift());
  } else {
    this.onAllFilesLoaded_({ loadedFiles: [] });
  }
};


jstestdriver.FileLoader.prototype.loadFile_ = function(file) {
  this.pluginRegistrar_.loadSource(file, this.boundOnFileLoaded);
};

/**
 * This method is called once a file has been loaded. It then either load the next file or if none
 * are left sends back the list of loaded files to the server.
 * 
 * @param {Object} file A jstestdriver.FileResult object
 */
jstestdriver.FileLoader.prototype.onFileLoaded_ = function(fileLoaded) {
  this.loadedFiles_.push(fileLoaded);
  if (this.files_.length == 0) {
    this.onAllFilesLoaded_({
      loadedFiles: this.loadedFiles_
    });
  } else {
    this.loadFile_(this.files_.shift());
  }
};
/*
 * Copyright 2011 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

goog.provide('jstestdriver.TestRunFilter');

goog.require('jstestdriver');

/**
 * @constructor
 */
jstestdriver.TestRunFilter = function(testCaseInfo) {
  this.testCaseInfo_ = testCaseInfo;
};


jstestdriver.TestRunFilter.prototype.getDefaultTestRunConfiguration = function() {
  return this.createTestRunConfiguration_(this.testCaseInfo_.getTestNames());
};


/**
 * Includes and excludes tests based on the given expressions. Expressions are
 * of the form:
 *
 * Expr:
 *   "all" | RegExp | -RegExp
 *
 * RegExp:
 *   A JavaScript regular expression without the quoting characters.
 *
 * @param expressions {Array.<string>} The expression strings.
 */
jstestdriver.TestRunFilter.prototype.getTestRunConfigurationFor = function(expressions) {
  var positiveExpressions = this.filter_(expressions, this.regexMatcher_(/^[^-].*/));
  if (positiveExpressions.length < 1) {
    positiveExpressions.push('all');
  }
  var negativeExpressions = this.filter_(expressions, this.regexMatcher_(/^-.*/));
  var testMethodMap = this.buildTestMethodMap_();
  var excludedTestIds = this.getExcludedTestIds_(testMethodMap, negativeExpressions);
  var matchedTests = this.getMatchedTests_(testMethodMap, positiveExpressions, excludedTestIds);
  return matchedTests.length > 0 ? this.createTestRunConfiguration_(matchedTests) : null;
};


jstestdriver.TestRunFilter.prototype.createTestRunConfiguration_ = function(tests) {
  return new jstestdriver.TestRunConfiguration(this.testCaseInfo_, tests);
};


/**
 * @param regex {RegExp} The regular expression.
 * @return {function(string): boolean} A function that tests the given RegExp
 *     against the function's expression argument.
 * @private
 */
jstestdriver.TestRunFilter.prototype.regexMatcher_ = function(regex) {
  return function(expression) {
    return regex.test(expression);
  };
};


/**
 * @return {Object.<string, string>} A map from test method id to test method
 *     name, where a test method id is of the form TestCaseName#testMethodName.
 * @private
 */
jstestdriver.TestRunFilter.prototype.buildTestMethodMap_ = function() {
  var testMethodMap = {};
  var testMethods = this.testCaseInfo_.getTestNames();
  var testMethodsLength = testMethods.length;
  for (var i = 0; i < testMethodsLength; ++i) {
    var methodName = testMethods[i];
    if (this.isTestMethod_(methodName)) {
      testMethodMap[this.buildTestMethodId_(methodName)] = methodName;
    }
  }
  return testMethodMap;
};


/**
 * @param methodName {string} A name of a method of the test class.
 * @return {boolean} True if the method name begins with 'test'.
 * @private
 */
jstestdriver.TestRunFilter.prototype.isTestMethod_ = function(methodName) {
  return /^test.*/.test(methodName);
};


/**
 * @param testMethod {string} The name of the test method.
 * @return {string} A test method id which is of the form
 *     TestCaseName#testMethodName.
 * @private
 */
jstestdriver.TestRunFilter.prototype.buildTestMethodId_ = function(testMethod) {
  return this.testCaseInfo_.getTestCaseName() + '#' + testMethod;
};


/**
 * @param expressions {Array.<string>} The expression strings.
 * @param condition {function(string): boolean} A condition that applies to the
 *     expression strings.
 * @return {Array.<string>} Any expression strings for which the condition holds.
 * @private
 */
jstestdriver.TestRunFilter.prototype.filter_ = function(expressions, condition) {
  var result = [];
  for (var i = 0; i < expressions.length; ++i) {
    if (condition(expressions[i])) {
      result.push(expressions[i]);
    }
  }
  return result;
};


/**
 * @param testMethodMap {Object.<string, string>} A map from test method id to
 *     test method name.
 * @param negativeExpressions {Array.<string>} The negative expression strings.
 * @return {Object.<string, boolean>} A map from test method id to boolean that
 *     signals whether a test method should be excluded from this test run.
 * @private
 */
jstestdriver.TestRunFilter.prototype.getExcludedTestIds_ = function(
    testMethodMap, negativeExpressions) {
  var excludedTestIds = {};
  for (var i = 0; i < negativeExpressions.length; ++i) {
    var expr = negativeExpressions[i].substring(1);
    var pattern = new RegExp(expr);
    for (var testMethodId in testMethodMap) {
      if (pattern.test(testMethodId)) {
        excludedTestIds[testMethodId] = true;
      }
    }
  }
  return excludedTestIds;
};

/**
 * @param testMethodMap {Object.<string, string>} A map from test method id to
 *     test method name.
 * @param positiveExpressions {Array.<string>} The positive expression strings.
 * @param excludedTestIds {Object.<string, boolean>} A map from test method id to
 *     boolean that signals whether a test method should be excluded from this
 *     test run.
 * @return {Array.<string>} A list of test method names for test methods that
 *     should be run.
 * @private
 */
jstestdriver.TestRunFilter.prototype.getMatchedTests_ = function(
    testMethodMap, positiveExpressions, excludedTestIds) {
  var matchedTests = [];
  for (var i = 0; i < positiveExpressions.length; i++) {
    var expr = positiveExpressions[i];

    if (expr == 'all') {
      expr = '.*';
    }

    var pattern = new RegExp(expr);

    for (var testMethodId in testMethodMap) {
      if (pattern.test(testMethodId) && !excludedTestIds[testMethodId]) {
        matchedTests.push(testMethodMap[testMethodId]);
      }
    }
  }
  return matchedTests;
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

goog.provide('jstestdriver.TestCaseInfo');

goog.require('jstestdriver');
goog.require('jstestdriver.TestRunFilter');

/**
 * @param {string} testCaseName
 * @param {Function} template
 * @param {string} opt_type
 * @param {string} opt_fileName
 * @constructor
 */
jstestdriver.TestCaseInfo = function(testCaseName,
                                     template,
                                     opt_type,
                                     opt_fileName) {

  this.testCaseName_ = testCaseName;
  this.template_ = template;
  this.type_ = opt_type || jstestdriver.TestCaseInfo.DEFAULT_TYPE;
  this.fileName_ = opt_fileName || '';
};


jstestdriver.TestCaseInfo.DEFAULT_TYPE = 'default';


jstestdriver.TestCaseInfo.ASYNC_TYPE = 'async';


/**
 * @private
 * @type {string}
 */
jstestdriver.TestCaseInfo.prototype.testCaseName_;


/**
 * @private
 * @type {Function}
 */
jstestdriver.TestCaseInfo.prototype.template_;


/**
 * @private
 * @type {string}
 */
jstestdriver.TestCaseInfo.prototype.type_;


/**
 * @private
 * @type {string}
 */
jstestdriver.TestCaseInfo.prototype.fileName_;


/**
 * @return {string}
 */
jstestdriver.TestCaseInfo.prototype.getType = function() {
  return this.type_;
};


/**
 * @returns {string}
 */
jstestdriver.TestCaseInfo.prototype.getFileName = function() {
  return this.fileName_;
};


/**
 * @param {string} fileName
 */
jstestdriver.TestCaseInfo.prototype.setFileName = function(fileName) {
  this.fileName_ = fileName;
};


/**
 * @returns {string}
 */
jstestdriver.TestCaseInfo.prototype.getTestCaseName = function() {
  return this.testCaseName_;
};


/**
 * @returns {Function}
 */
jstestdriver.TestCaseInfo.prototype.getTemplate = function() {
  return this.template_;
};


/**
 * @returns {Array.<string>}
 */
jstestdriver.TestCaseInfo.prototype.getTestNames = function() {
  var testNames = [];

  for (var property in this.template_.prototype) {
    if (property.indexOf('test') == 0) {
      testNames.push(property);
    }
  }
  return testNames;
};


/**
 * @returns {jstestdriver.TestRunConfiguration}
 */
jstestdriver.TestCaseInfo.prototype.getDefaultTestRunConfiguration = function() {
  return new jstestdriver.TestRunFilter(this).getDefaultTestRunConfiguration();
};


/**
 * Includes and excludes tests based on the given expressions. Expressions are
 * of the form:
 *
 * Expr:
 *   "all" | RegExp | -RegExp
 *
 * RegExp:
 *   A JavaScript regular expression without the quoting characters.
 *
 * @param expressions {Array.<string>} The expression strings.
 */
jstestdriver.TestCaseInfo.prototype.getTestRunConfigurationFor = function(expressions) {
  return new jstestdriver.TestRunFilter(this).getTestRunConfigurationFor(expressions);
};

/**
 * @param {Object} obj
 * @returns {boolean}
 */
jstestdriver.TestCaseInfo.prototype.equals = function(obj) {
  return (!!obj) && typeof obj.getTestCaseName != 'undefined'
      && obj.getTestCaseName() == this.testCaseName_;
};


/**
 * @returns {string}
 */
jstestdriver.TestCaseInfo.prototype.toString = function() {
  return "TestCaseInfo(" +
    this.testCaseName_ +
    "," + this.template_ +
    "," + this.type_ + ")";
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

goog.provide('jstestdriver.TestResult');

goog.require('jstestdriver');


/**
 * @param {string} testCaseName
 * @param {string} testName
 * @param {jstestdriver.TestResult.RESULT} result
 * @param {string} message
 * @param {Array.<string>} log
 * @param {number} time
 * @param {Object.<string, Object>} opt_data A map of arbitrary value pairs
 *     representing test meta data.
 * @param {*} opt_argument An optional argument for a test fragment.
 * @constructor
 */
jstestdriver.TestResult = function(testCaseName,
    testName, result, message, log, time, opt_data, opt_argument) {
  this.testCaseName = testCaseName;
  this.testName = testName;
  this.result = result;
  this.message = message;
  this.log = log;
  this.time = time;
  this.data = opt_data || {};
  this.argument = opt_argument;
};


/**
 * @enum {string}
 */
jstestdriver.TestResult.RESULT = {
  PASSED : 'passed',
  ERROR : 'error',
  FAILED : 'failed'
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

goog.provide('jstestdriver.TestRunConfiguration');

goog.require('jstestdriver');
goog.require('jstestdriver.TestCaseInfo');

/**
 * Represents all of the necessary information to run a test case.
 * @param {jstestdriver.TestCaseInfo} testCaseInfo The test case information, containing
 * @param {Array.<string>} tests The names of all the tests to run.
 * @param {Object.<string, *>=} opt_args The arguments for the tests.
 * @constructor
 */
jstestdriver.TestRunConfiguration = function(testCaseInfo, tests, opt_args) {
  /**
   * @type {jstestdriver.TestCaseInfo}
   * @private
   */
  this.testCaseInfo_ = testCaseInfo;
  /**
   * @type {Array.<string>}
   * @private
   */
  this.tests_ = tests;
  /**
   * @type {Object.<string, *>}
   * @private
   */
  this.arguments_ = opt_args ? opt_args : null;
};


jstestdriver.TestRunConfiguration.prototype.getTestCaseInfo = function() {
  return this.testCaseInfo_;
};


jstestdriver.TestRunConfiguration.prototype.getTests = function() {
  return this.tests_;
};


/**
 * @return {Object.<string, *>} the arguments.
 */
jstestdriver.TestRunConfiguration.prototype.getArguments = function() {
  return this.arguments_;
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * Handles the TestCases
 * @constructor
 */
jstestdriver.TestCaseManager = function(pluginRegistrar) {
  this.testCasesInfo_ = [];
  this.fileToTestCaseMap_ = {};
  this.testCaseToFileMap_ = {};
  this.latestTestCaseInfo_ = null;
  this.pluginRegistrar_ = pluginRegistrar;
  this.recentCases_ = [];
};


/**
 * @param {jstestdriver.TestCaseInfo} testCaseInfo The testcase for the manager
 *   to track.
 */
jstestdriver.TestCaseManager.prototype.add = function(testCaseInfo) {
  var index = this.indexOf_(testCaseInfo);
  if (index != -1) {
    throw new Error('duplicate test case names! On ' +
        testCaseInfo + ' and ' + this.testCasesInfo_[index] +
        ' in ' + this.testCasesInfo_[index].getFileName());
  } else {
    this.testCasesInfo_.push(testCaseInfo);
    this.recentCases_.push(testCaseInfo);
  }
};


jstestdriver.TestCaseManager.prototype.updateLatestTestCase = function(filename) {
  if (this.recentCases_.length) {
    this.fileToTestCaseMap_[filename] = this.recentCases_;
    for (var i = 0; this.recentCases_[i]; i++) {
      // TODO(corysmith): find a way to keep the TestCaseInfo invariant.
      this.recentCases_[i].setFileName(filename);
    }
    this.recentCases_ = [];
  }
};


jstestdriver.TestCaseManager.prototype.removeTestCaseForFilename = function(filename) {
  var cases = this.fileToTestCaseMap_[filename] || [];
  this.fileToTestCaseMap_[filename] = null;
  delete this.fileToTestCaseMap_[filename];
  while (cases.length) {
    this.removeTestCase_(this.indexOf_(cases.pop()));
  }
};


jstestdriver.TestCaseManager.prototype.removeTestCase_ = function(index) {
  var testCase = this.testCasesInfo_.splice(index, 1);
};


jstestdriver.TestCaseManager.prototype.indexOf_ = function(testCaseInfo) {
  var size = this.testCasesInfo_.length;

  for (var i = 0; i < size; i++) {
    var currentTestCaseInfo = this.testCasesInfo_[i];

    if (currentTestCaseInfo.equals(testCaseInfo)) {
      return i;
    }
  }
  return -1;
};


jstestdriver.TestCaseManager.prototype.getDefaultTestRunsConfiguration = function() {
  var testRunsConfiguration = [];
  var size = this.testCasesInfo_.length;

  for (var i = 0; i < size; i++) {
    var testCaseInfo = this.testCasesInfo_[i];

    testRunsConfiguration.push(testCaseInfo.getDefaultTestRunConfiguration());
  }
  return testRunsConfiguration;
};


jstestdriver.TestCaseManager.prototype.getTestRunsConfigurationFor = function(expressions) {
  var testRunsConfiguration = [];
  this.pluginRegistrar_.getTestRunsConfigurationFor(this.testCasesInfo_,
                                                    expressions,
                                                    testRunsConfiguration);
  return testRunsConfiguration;
};


jstestdriver.TestCaseManager.prototype.getTestCasesInfo = function() {
  return this.testCasesInfo_;
};


jstestdriver.TestCaseManager.prototype.getCurrentlyLoadedTestCases = function() {
  var testCases = [];
  var size = this.testCasesInfo_.length;

  for (var i = 0; i < size; i++) {
    var testCaseInfo = this.testCasesInfo_[i];
    testCases.push({
      'name' : testCaseInfo.getTestCaseName(),
      'tests' : testCaseInfo.getTestNames()
    })
  }
  return {
    numTests: testCases.length,
    testCases: testCases
  };
};

jstestdriver.TestCaseManager.prototype.getCurrentlyLoadedTestCasesFor = function(expressions) {
  var testRunsConfiguration = this.getTestRunsConfigurationFor(expressions);
  var size = testRunsConfiguration.length;
  var testCases = [];

  for (var i = 0; i < size; i++) {
    var testRunConfiguration = testRunsConfiguration[i];
    var testCaseInfo = testRunConfiguration.getTestCaseInfo();
    var tests = testRunConfiguration.getTests();
    testCases.push({
      'name' : testCaseInfo.getTestCaseName(),
      'tests' : testCaseInfo.getTestNames(),
      'fileName' :  testCaseInfo.getFileName()
    })
  }
  return {
    numTests: testCases.length,
    testCases: testCases
  };
};


/** @deprecated */
jstestdriver.TestCaseManager.prototype.getCurrentlyLoadedTest = function() {
  var testNames = [];
  var size = this.testCasesInfo_.length;

  for (var i = 0; i < size; i++) {
    var testCaseInfo = this.testCasesInfo_[i];
    var testCaseName = testCaseInfo.getTestCaseName();
    var tests = testCaseInfo.getTestNames();
    var testsSize = tests.length;

    for (var j = 0; j < testsSize; j++) {
      testNames.push(testCaseName + '.' + tests[j]);
    }
  }
  return {
    numTests: testNames.length,
    testNames: testNames
  };
};


jstestdriver.TestCaseManager.prototype.getCurrentlyLoadedTestFor = function(expressions) {
  var testRunsConfiguration = this.getTestRunsConfigurationFor(expressions);
  var size = testRunsConfiguration.length;
  var testNames = [];

  for (var i = 0; i < size; i++) {
    var testRunConfiguration = testRunsConfiguration[i];
    var testCaseName = testRunConfiguration.getTestCaseInfo().getTestCaseName();
    var tests = testRunConfiguration.getTests();
    var testsSize = tests.length;

    for (var j = 0; j < testsSize; j++) {
      var testName = tests[j];

      testNames.push(testCaseName + '.' + testName);
    }
  }
  return {
    numTests: testNames.length,
    testNames: testNames
  };
};
/*
 * Copyright 2011 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @fileoverview
 * @author corysmith@google.com (Cory Smith)
 * @author rdionne@google.com (Robert Dionne)
 */



/**
 * Constructs a TestCaseBuilder.
 * @param {jstestdriver.TestCaseManager} testCaseManager The appropriate test
 *     case manager.
 * @constructor
 */
jstestdriver.TestCaseBuilder = function(testCaseManager) {
  this.testCaseManager_ = testCaseManager;
};


/**
 * Defines a test case.
 * @param {string} testCaseName The name of the test case.
 * @param {Object} opt_proto An optional prototype.
 * @param {Object} opt_type Either DEFAULT_TYPE or ASYNC_TYPE.
 * @return {Function} Base function that represents the test case class.
 */
jstestdriver.TestCaseBuilder.prototype.TestCase =
    function(testCaseName, opt_proto, opt_type) {
  this.checkNotBeginsWith_(testCaseName, '-');
  this.checkNotContains_(testCaseName, ',');
  this.checkNotContains_(testCaseName, '#');
  var testCaseClass = function() {};
  if (opt_proto) {
    testCaseClass.prototype = opt_proto;
  }
  if (typeof testCaseClass.prototype.setUp == 'undefined') {
    testCaseClass.prototype.setUp = function() {};
  }
  if (!testCaseClass.prototype.hasOwnProperty('toString')) {
    testCaseClass.prototype.toString = function() {
      return "TestCase(" + testCaseName +")";
    };
  }
  if (typeof testCaseClass.prototype.tearDown == 'undefined') {
    testCaseClass.prototype.tearDown = function() {};
  }
  this.testCaseManager_.add(
      new jstestdriver.TestCaseInfo(testCaseName, testCaseClass, opt_type));
  return testCaseClass;
};


jstestdriver.TestCaseBuilder.prototype.checkNotBeginsWith_ = function(
    testCaseName, illegalString) {
  if (testCaseName.indexOf(illegalString) == 0) {
    throw new Error('Test case names must not begin with \'' +
        illegalString + '\'');
  }
};


jstestdriver.TestCaseBuilder.prototype.checkNotContains_= function(
    testCaseName, illegalString) {
  if (testCaseName.indexOf(illegalString) > -1) {
    throw new Error('Test case names must not contain \'' + illegalString + '\'');
  }
};


/**
 * Defines an asynchronous test case.
 * @param {string} testCaseName The name of the test case.
 * @param {Object} opt_proto An optional prototype.
 * @return {Function} Base function that represents the asyncronous test case
 *     class.
 */
jstestdriver.TestCaseBuilder.prototype.AsyncTestCase =
    function(testCaseName, opt_proto) {
  return this.TestCase(
      testCaseName, opt_proto, jstestdriver.TestCaseInfo.ASYNC_TYPE);
};


/**
 * A TestCase that will only be executed when a certain condition is true.
 * @param {string} The name of the TestCase.
 * @param {function():boolean} A function that indicates if this case should be
 *     run.
 * @param {Object} opt_proto An optional prototype for the test case class.
 * @param {Object} opt_type Either DEFAULT_TYPE or ASYNC_TYPE.
 * @return {Function} Base function that represents the TestCase class.
 */
jstestdriver.TestCaseBuilder.prototype.ConditionalTestCase =
    function(testCaseName, condition, opt_proto, opt_type) {
  if (condition()) {
    return this.TestCase(testCaseName, opt_proto, opt_type);
  }
  this.testCaseManager_.add(
      new jstestdriver.TestCaseInfo(
          testCaseName,
          jstestdriver.TestCaseBuilder.PlaceHolderCase,
          opt_type));
  return function(){};
};


/**
 * An AsyncTestCase that will only be executed when a certain condition is true.
 * @param {String} The name of the AsyncTestCase.
 * @param {function():boolean} A function that indicates if this case should be
 *     run.
 * @param {Object} opt_proto An optional prototype for the test case class.
 * @return {Function} Base function that represents the TestCase class.
 */
jstestdriver.TestCaseBuilder.prototype.ConditionalAsyncTestCase =
    function(testCaseName, condition, opt_proto) {
  return this.ConditionalTestCase(
      testCaseName, condition, opt_proto, jstestdriver.TestCaseInfo.ASYNC_TYPE);
};


/**
 * Constructs a place holder test case.
 * @constructor
 */
jstestdriver.TestCaseBuilder.PlaceHolderCase = function() {};


/**
 * Ensures there is at least one test to demonstrate a correct exclusion.
 */
jstestdriver.TestCaseBuilder.PlaceHolderCase.prototype.testExcludedByCondition =
      jstestdriver.EMPTY_FUNC;

/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @param pluginRegistrar
 * @constructor
 */
jstestdriver.TestRunner = function(pluginRegistrar) {
  this.pluginRegistrar_ = pluginRegistrar;

  this.boundRunNextConfiguration_ =
      jstestdriver.bind(this, this.runNextConfiguration_);
};


/**
 * Runs all TestRunConfigurations.
 * @param {Array.<jstestdriver.TestRunConfiguration>} testRunsConfiguration Configurations to
 *      run. This array willl be modified...
 * @param {function(jstestdriver.TestResult):null} onTestDone
 * 
 */
jstestdriver.TestRunner.prototype.runTests = function(testRunsConfiguration,
                                                      onTestDone,
                                                      onComplete,
                                                      captureConsole) {

  this.pluginRegistrar_.onTestsStart();
  this.testRunsConfiguration_ = testRunsConfiguration;
  this.onTestDone_ = onTestDone;
  this.onComplete_ = onComplete;
  this.captureConsole_ = captureConsole;
  this.runNextConfiguration_();
};


jstestdriver.TestRunner.prototype.finish_ = function() {
  var onComplete = this.onComplete_;
  this.pluginRegistrar_.onTestsFinish();
  this.testRunsConfiguration_ = null;
  this.onTestDone_ = null;
  this.onComplete_ = null;
  this.captureConsole_ = false;
  onComplete();
};


jstestdriver.TestRunner.prototype.runNextConfiguration_ = function() {
  if (this.testRunsConfiguration_.length == 0) {
    this.finish_();
    return;
  }
  this.runConfiguration(
      this.testRunsConfiguration_.shift(),
      this.onTestDone_,
      this.boundRunNextConfiguration_);
}


/**
 * Runs a test configuration.
 * @param {jstestdriver.TestRunConfiguration} config
 * @param {function(jstestdriver.TestResult):null} onTestDone
 *     Function to be called when test is done.
 * @param {Function} onComplete Function to be called when all tests are done.
 */
jstestdriver.TestRunner.prototype.runConfiguration = function(config,
                                                              onTestDone,
                                                              onComplete) {
  if (this.captureConsole_) {
    this.overrideConsole_();
  }

  jstestdriver.log("running configuration " + config);
  this.pluginRegistrar_.runTestConfiguration(
      config,
      onTestDone,
      onComplete);

  if (this.captureConsole_) {
    this.resetConsole_();
  }
};


jstestdriver.TestRunner.prototype.overrideConsole_ = function() {
  this.logMethod_ = console.log;
  this.logDebug_ = console.debug;
  this.logInfo_ = console.info;
  this.logWarn_ = console.warn;
  this.logError_ = console.error;
  console.log = function() { jstestdriver.console.log.apply(jstestdriver.console, arguments); };
  console.debug = function() { jstestdriver.console.debug.apply(jstestdriver.console, arguments); };
  console.info = function() { jstestdriver.console.info.apply(jstestdriver.console, arguments); };
  console.warn = function() { jstestdriver.console.warn.apply(jstestdriver.console, arguments); };
  console.error = function() { jstestdriver.console.error.apply(jstestdriver.console, arguments); };
};


jstestdriver.TestRunner.prototype.resetConsole_ = function() {
  console.log = this.logMethod_;
  console.debug = this.logDebug_;
  console.info = this.logInfo_;
  console.warn = this.logWarn_;
  console.error = this.logError_;  
};



/**
 * A map to manage the state of running TestCases.
 * @constructor
 */
jstestdriver.TestRunner.TestCaseMap = function() {
  this.testCases_ = {};
};


/**
 * Start a TestCase.
 * @param {String} testCaseName The name of the test case to start.
 */
jstestdriver.TestRunner.TestCaseMap.prototype.startCase = function(testCaseName) {
  this.testCases_[testCaseName] = true;
};


/**
 * Stops a TestCase.
 * @param {String} testCaseName The name of the test case to stop.
 */
jstestdriver.TestRunner.TestCaseMap.prototype.stopCase = function(testCaseName) {
  this.testCases_[testCaseName] = false;
};


/**
 * Indicates if there are still cases running.
 */
jstestdriver.TestRunner.TestCaseMap.prototype.hasActiveCases = function() {
  for (var testCase in this.testCases_) {
    if (this.testCases_.hasOwnProperty(testCase) && this.testCases_[testCase]) {
      return true;
    }
  }
  return false;
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */



/**
 * Allows the browser to stop the test execution thread after a test when the
 * interval requires it to.
 * @param setTimeout {function(Function, number):null}
 * @param interval {number}
 * @return {function(Function):null}
 */
jstestdriver.testBreather = function(setTimeout, interval) {
  var lastBreath = new Date();
  function maybeBreathe(callback) {
    var now = new Date();
    if ((now - lastBreath) > interval) {
      setTimeout(callback, 1);
      lastBreath = now;
    } else {
      callback();
    }
  };
  return maybeBreathe;
};


jstestdriver.TIMEOUT = 500;

jstestdriver.NOOP_COMMAND = {
  command : 'noop',
  parameters : []
};

// TODO(corysmith): Extract the network streaming logic from the Executor logic.
/**
 * @param {jstestdriver.StreamingService} streamingService The service for
 *     streaming {@link jstestdriver.Reponse}s to the server.
 * @param {jstestdriver.TestCaseManager} testCaseManager Used to access the TestCaseInfo's for running.
 * @param {jstestdriver.TestRunner} testRunner Runs the tests...
 * @param {jstestdriver.PluginRegistrar} pluginRegistrar The plugin service,
 *     for post processing test results.
 * @param {jstestdriver.now} now
 * @param {function():number} getBrowserInfo
 * @param {jstestdriver.Signal} currentActionSignal
 * @param {jstestdriver.Signal} unloadSignal
 * @constructor
 */
jstestdriver.CommandExecutor = function(streamingService,
                                        testCaseManager,
                                        testRunner,
                                        pluginRegistrar,
                                        now,
                                        getBrowserInfo,
                                        currentActionSignal,
                                        unloadSignal) {
  this.streamingService_ = streamingService;
  this.__testCaseManager = testCaseManager;
  this.__testRunner = testRunner;
  this.__pluginRegistrar = pluginRegistrar;
  this.__boundExecuteCommand = jstestdriver.bind(this, this.executeCommand);
  this.__boundExecute = jstestdriver.bind(this, this.execute);
  this.__boundEvaluateCommand = jstestdriver.bind(this, this.evaluateCommand);
  this.boundOnFileLoaded_ = jstestdriver.bind(this, this.onFileLoaded);
  this.boundOnFileLoadedRunnerMode_ = jstestdriver.bind(this, this.onFileLoadedRunnerMode);
  this.commandMap_ = {};
  this.testsDone_ = [];
  this.debug_ = false;
  this.now_ = now;
  this.lastTestResultsSent_ = 0;
  this.getBrowserInfo = getBrowserInfo;
  this.currentActionSignal_ = currentActionSignal;
  this.currentCommand = null;
  this.unloadSignal_ = unloadSignal;
};


/**
 * Executes a command form the server.
 * @param jsonCommand {String}
 */
jstestdriver.CommandExecutor.prototype.executeCommand = function(jsonCommand) {
  var command;
  if (jsonCommand && jsonCommand.length) { //handling some odd IE errors.
    command = jsonParse(jsonCommand);
  } else {
    command = jstestdriver.NOOP_COMMAND;
  }
  this.currentCommand = command.command;
  jstestdriver.log('current command ' + command.command);
  try {
    this.unloadSignal_.set(false); // if the page unloads during a command, issue an error.
    this.commandMap_[command.command](command.parameters);
  } catch (e) {
    var message =  'Exception ' + e.name + ': ' + e.message +
        '\n' + e.fileName + '(' + e.lineNumber +
        '):\n' + e.stack;
    var response = new jstestdriver.Response(jstestdriver.RESPONSE_TYPES.LOG,
      jstestdriver.JSON.stringify(
          new jstestdriver.BrowserLog(1000,
              'jstestdriver.CommandExecutor',
              message,
              this.getBrowserInfo())),
      this.getBrowserInfo());
    if (top.console && top.console.log) {
      top.console.log(message);
    }
    this.streamingService_.close(response, this.__boundExecuteCommand);
    this.unloadSignal_.set(true); // reloads are possible between actions.
    // Propagate the exception.
    throw e;
  }
};


jstestdriver.CommandExecutor.prototype.execute = function(cmd) {
  var response = new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.COMMAND_RESULT,
          JSON.stringify(this.__boundEvaluateCommand(cmd)),
          this.getBrowserInfo());

  this.streamingService_.close(response, this.__boundExecuteCommand);
};


jstestdriver.CommandExecutor.prototype.evaluateCommand = function(cmd) {
  var res = '';
  try {
    var evaluatedCmd = eval('(' + cmd + ')');
    if (evaluatedCmd) {
      res = evaluatedCmd.toString();
    }
  } catch (e) {
    res = 'Exception ' + e.name + ': ' + e.message +
          '\n' + e.fileName + '(' + e.lineNumber +
          '):\n' + e.stack;
  }
  return res;
};


/**
 * Registers a command to the executor to handle incoming command requests.
 * @param {String} name The name of the command
 * @param {Object} context The context to call the command in.
 * @param {function(Array):null} func the command.
 */
jstestdriver.CommandExecutor.prototype.registerCommand =
    function(name, context, func) {
  this.commandMap_[name] = jstestdriver.bind(context, func);
};


/**
 * Registers a command to the executor to handle incoming command requests
 * @param {String} name The name of the command
 * @param {Object} context The context to call the command in.
 * @param {function(Array):null} func the command.
 */
jstestdriver.CommandExecutor.prototype.registerTracedCommand =
    function(name, context, func) {
  var bound = jstestdriver.bind(context, func);
  var signal = this.currentActionSignal_;
  this.commandMap_[name] = function() {
    signal.set(name);
    return bound.apply(null, arguments);
  };
};


jstestdriver.CommandExecutor.prototype.dryRun = function() {
  var response =
      new jstestdriver.Response(jstestdriver.RESPONSE_TYPES.TEST_QUERY_RESULT,
          JSON.stringify(this.__testCaseManager.getCurrentlyLoadedTestCases()),
          this.getBrowserInfo());
  
  this.streamingService_.close(response, this.__boundExecuteCommand);
};


jstestdriver.CommandExecutor.prototype.dryRunFor = function(args) {
  var expressions = jsonParse('{"expressions":' + args[0] + '}').expressions;
  var tests = JSON.stringify(
      this.__testCaseManager.getCurrentlyLoadedTestCasesFor(expressions))
  var response = new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.TEST_QUERY_RESULT,
          tests,
          this.getBrowserInfo());
  this.streamingService_.close(response, this.__boundExecuteCommand);
};


jstestdriver.CommandExecutor.prototype.listen = function(loadResults) {
  var response;
  if (window.location.href.search('refresh') != -1) {
    response =
        new jstestdriver.Response(jstestdriver.RESPONSE_TYPES.RESET_RESULT,
                                  '{"loadedFiles":' + JSON.stringify(loadResults) + '}',
                                  this.getBrowserInfo(),
                                  true);
    jstestdriver.log('Runner reset: ' + window.location.href);
  } else {
    jstestdriver.log('Listen: ' + window.location.href);
    response =
        new jstestdriver.Response(jstestdriver.RESPONSE_TYPES.BROWSER_READY,
                                  '{"loadedFiles":' + JSON.stringify(loadResults) + '}',
                                  this.getBrowserInfo(),
                                  true);

  }
  this.streamingService_.close(response, this.__boundExecuteCommand);
};
/*
 * Copyright 2011 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */



jstestdriver.ManualScriptLoader = function(win, testCaseManager, now) {
  this.win_ = win;
  this.testCaseManager_ = testCaseManager;
  this.now_ = now;
  this.onFileLoaded_ = null;
  this.started_ = -1;
  this.file_ = null;
  this.fileMap_ = {};
  this.errorHandler_ = this.createErrorHandler();
};


jstestdriver.ManualScriptLoader.prototype.beginLoad = function(file, onFileLoaded) {
  this.fileMap_[file.fileSrc] = file;
  this.testCaseManager_.removeTestCaseForFilename(file.fileSrc);
  this.file_ = file;
  this.win_.onerror = this.errorHandler_;
  this.started_ = this.now_();
  this.onFileLoaded_ = onFileLoaded;
  jstestdriver.log('loading ' + file.fileSrc);
};


jstestdriver.ManualScriptLoader.prototype.endLoad = function(file) {
  var elapsed = this.now_() - this.started_;
  if (elapsed > 50) {
    jstestdriver.log('slow load ' + this.file_.fileSrc + ' in ' + elapsed);
  }
  this.testCaseManager_.updateLatestTestCase(file.fileSrc);
  var result = new jstestdriver.FileResult(file,
                                           true,
                                           '',
                                           this.now_() - this.started_);
  this.onFileLoaded_(result);
};


jstestdriver.ManualScriptLoader.prototype.createErrorHandler = function() {
  var self = this;
  return function (msg, url, line) {
    var offset = url.indexOf('/test/')
    var fileSrc = offset > -1 ? url.substr(offset, url.length - offset) : url;
    var loadingFile = self.fileMap_[fileSrc];
    jstestdriver.log('failed load ' + fileSrc + ' in ' +
        (self.now_() - self.started_));
    var started = self.started_;
    self.started_ = -1;
    var loadMsg = 'error loading file: ' + fileSrc;

    if (line != undefined && line != null) {
      loadMsg += ':' + line;
    }

    if (msg != undefined && msg != null) {
      loadMsg += ': ' + msg;
    }
    self.win_.onerror = jstestdriver.EMPTY_FUNC;
    self.onFileLoaded_(new jstestdriver.FileResult(loadingFile, false, loadMsg, self.now_() - started));
  }
};
/*
 * Copyright 2011 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


jstestdriver.ManualResourceTracker = function(
    parse,
    serialize,
    pluginRegistrar,
    getBrowserInfo,
    manualScriptLoader) {
  this.parse_ = parse;
  this.serialize_ = serialize;
  this.getBrowserInfo_ = getBrowserInfo;
  this.manualScriptLoader_ = manualScriptLoader;
  this.boundOnComplete_ = jstestdriver.bind(this, this.onComplete_);
  this.results_ = [];
  this.resultsIndexMap_ = {};
};

/**
 * Starts the resource load tracking to catch errors and other statistics. 
 * @param {String} jsonFile A serialized jstestdriver.FileSrc
 */
jstestdriver.ManualResourceTracker.prototype.startResourceLoad =
    function(jsonFile) {
  var file = this.parse_(jsonFile);
  this.manualScriptLoader_.beginLoad(file, this.boundOnComplete_);
};

/**
 * Method to be called with the resource completes.
 * @param {jstestdriver.FileLoadResult} result
 */
jstestdriver.ManualResourceTracker.prototype.onComplete_ = function(result) {
  var fileSrc = result.file.fileSrc;
  var idx = this.resultsIndexMap_[fileSrc];
  if (idx != null) {
    // errors can arrive after the load is reported as complete. Apparently,
    // onError is not tied to the script resolution.
    if (!result.success) { // if it's successful, don't replace, as it could overwrite an error.
      this.results_[idx] = result;
    }
  } else {
    this.resultsIndexMap_[fileSrc] = this.results_.push(result) - 1;
  }
};

/**
 * Called after the resource has loaded (maybe, other times it will called immediately).
 */
jstestdriver.ManualResourceTracker.prototype.finishResourceLoad = function(jsonFile) {
  var file = this.parse_(jsonFile);
  this.manualScriptLoader_.endLoad(file);
};

/**
 * Returns the collected results from loading.
 * @return {Array.<jstestdriver.FileLoadResult>}
 */
jstestdriver.ManualResourceTracker.prototype.getResults = function() {
  return this.results_;
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * Loads script tags for the stand alone runner.
 * 
 * @param {function(String):Object} jsonParse A function to deserialize json objects.
 * @param {jstestdriver.PluginRegistrar} pluginRegistrar
 * @param {function():jstestdriver.BrowserInfo} getBrowserInfo Browser info factory.
 * @param {function(jstestdriver.Response):void} onLoadComplete The function to call when the loading is complete.
 * @param {jstestdriver.StandAloneTestReporter} results The reporter object for the stand alone runner.
 * 
 */
jstestdriver.StandAloneLoadTestsCommand =
    function(jsonParse, pluginRegistrar, getBrowserInfo, onLoadComplete, reporter, now) {
  this.jsonParse_ = jsonParse;
  this.pluginRegistrar_ = pluginRegistrar;
  this.boundOnFileLoaded_ = jstestdriver.bind(this, this.onFileLoaded);
  this.getBrowserInfo = getBrowserInfo;
  this.onLoadComplete_ = onLoadComplete;
  this.reporter_ = reporter;
  this.now_ = now;
}


jstestdriver.StandAloneLoadTestsCommand.prototype.loadTest = function(args) {
  var files = args[0];
  var fileSrcs = this.jsonParse_('{"f":' + files + '}').f;

  this.removeScripts(document, fileSrcs);
  var fileLoader = new jstestdriver.FileLoader(this.pluginRegistrar_,
    this.boundOnFileLoaded_);

  this.reporter_.startLoading(this.now_());
  fileLoader.load(fileSrcs);
};

jstestdriver.StandAloneLoadTestsCommand.prototype.onFileLoaded = function(status) {
  this.reporter_.addLoadedFileResults(status.loadedFiles);
  var response = new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.FILE_LOAD_RESULT,
          JSON.stringify(status),
          this.getBrowserInfo());
  this.reporter_.finishLoading(this.now_());
  this.onLoadComplete_(response);
};


jstestdriver.StandAloneLoadTestsCommand.prototype.findScriptTagsToRemove_ =
    function(dom, fileSrcs) {
  var scripts = dom.getElementsByTagName('script');
  var filesSize = fileSrcs.length;
  var scriptsSize = scripts.length;
  var scriptTagsToRemove = [];

  for (var i = 0; i < filesSize; i++) {
    var f = fileSrcs[i].fileSrc;

    for (var j = 0; j < scriptsSize; j++) {
      var s = scripts[j];

      if (s.src.indexOf(f) != -1) {
        scriptTagsToRemove.push(s);
        break;
      }
    }
  }
  return scriptTagsToRemove;
};


jstestdriver.StandAloneLoadTestsCommand.prototype.removeScriptTags_ =
    function(dom, scriptTagsToRemove) {
  var head = dom.getElementsByTagName('head')[0];
  var size = scriptTagsToRemove.length;

  for (var i = 0; i < size; i++) {
    var script = scriptTagsToRemove[i];
    head.removeChild(script);
  }
};


jstestdriver.StandAloneLoadTestsCommand.prototype.removeScripts = function(dom, fileSrcs) {
  this.removeScriptTags_(dom, this.findScriptTagsToRemove_(dom, fileSrcs));
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * Executes tests for the standalone runner.
 * 
 * @param {jstestdriver.TestCaseManager}
 * @param {jstestdriver.TestRunner}
 * @param {jstestdriver.PluginRegistrar}
 * @param {function():BrowserInfo}
 * @param {jstestdriver.StandAloneTestReporter} reporter The reporter object for the stand alone runner.
 * @param {function():Number}
 * @param {function(String):Object}
 */
jstestdriver.StandAloneRunTestsCommand = function(testCaseManager,
                                                  testRunner,
                                                  pluginRegistrar,
                                                  getBrowserInfo,
                                                  reporter,
                                                  now,
                                                  jsonParse,
                                                  streamContinue,
                                                  streamStop) {
  this.testCaseManager_ = testCaseManager;
  this.testRunner_ = testRunner;
  this.pluginRegistrar_ = pluginRegistrar;
  this.jsonParse_ = jsonParse;
  this.now_ = now;
  this.boundOnTestDone_ = jstestdriver.bind(this, this.onTestDone_);
  this.boundOnComplete_ = jstestdriver.bind(this, this.onComplete);
  this.testsDone_ = [];
  this.getBrowserInfo_ = getBrowserInfo;
  this.reporter_ = reporter;
  this.streamContinue_ = streamContinue;
  this.streamStop_ = streamStop;
};


jstestdriver.StandAloneRunTestsCommand.prototype.createLog_ = function(message) {
  return new jstestdriver.BrowserLog(0, 'jstestdriver.StandAloneRunTestsCommand',
    message, this.getBrowserInfo_());
};


jstestdriver.StandAloneRunTestsCommand.prototype.runAllTests = function(args) {
  this.streamContinue_(
      new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.LOG,
          jstestdriver.JSON.stringify(this.createLog_('all tests started.')),
          this.getBrowserInfo_()));
  var captureConsole = args[0];
  this.debug_ = Boolean(args[2]);

  this.runTestCases_(this.testCaseManager_.getDefaultTestRunsConfiguration(),
      captureConsole == "true" ? true : false);
};


jstestdriver.StandAloneRunTestsCommand.prototype.runTests = function(args) {
  this.streamContinue_(
      new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.LOG,
          jstestdriver.JSON.stringify(this.createLog_('started tests.')),
          this.getBrowserInfo_()));
  var expressions = jsonParse('{"expressions":' + args[0] + '}').expressions;
  var captureConsole = args[1];
  this.debug_ = Boolean(args[2]);

  this.runTestCases_(this.testCaseManager_.getTestRunsConfigurationFor(expressions),
                     captureConsole == "true" ? true : false,
                     false);
};


jstestdriver.StandAloneRunTestsCommand.prototype.runTestCases_ = function(testRunsConfiguration,
    captureConsole) {
  this.reporter_.startTests(this.now_());
  this.totaltestruns_ = testRunsConfiguration.length;
  this.testRunner_.runTests(testRunsConfiguration,
                            this.boundOnTestDone_,
                            this.boundOnComplete_,
                            captureConsole);
};


jstestdriver.StandAloneRunTestsCommand.prototype.onTestDone_ = function(result) {
  this.reporter_.updateIsSuccess(
      result.result == jstestdriver.TestResult.RESULT.PASSED);
  this.addTestResult(result);
};


jstestdriver.StandAloneRunTestsCommand.prototype.onComplete = function() {
  var serializedTests = jstestdriver.JSON.stringify(this.testsDone_);
  this.streamContinue_(new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.TEST_RESULT,
          serializedTests,
          this.getBrowserInfo_()));
  this.reporter_.setReport(serializedTests);
  this.testsDone_ = [];
  this.reporter_.finishTests(this.now_());
  this.reporter_.setIsFinished(true);
  this.streamStop_(
      new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.LOG,
          jstestdriver.JSON.stringify(this.createLog_(
              'testing complete, isSuccess:' +
              this.reporter_.isSuccess() +
              ', isFinished:' +
              this.reporter_.isFinished())),
          this.getBrowserInfo_()));
};


jstestdriver.StandAloneRunTestsCommand.prototype.addTestResult = function(testResult) {
  this.reporter_.addTestResult(testResult);
  this.pluginRegistrar_.processTestResult(testResult);
  this.testsDone_.push(testResult);
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * A command that does nothing but contact the server and return a response.
 * Used to retrieve new commands from the server.
 * 
 * @constructor
 * 
 * @author corbinrsmith@gmail.com (Cory Smith)
 */
jstestdriver.NoopCommand = function(streamStop, getBrowserInfo) {
  /**
   * Function used to contact the server.
   * @type {function(jstestdriver.Response):null}
   */
  this.streamStop_ = streamStop;
  /**
   * Function used to retrieve the jstestdriver.BrowserInfo.
   * @type {function():jstestdriver.BrowserInfo}
   */
  this.streamStop_ = streamStop;
  this.getBrowserInfo_ = getBrowserInfo;
};

jstestdriver.NoopCommand.prototype.sendNoop = function() {
  
  this.streamStop_(
      new jstestdriver.Response(
          jstestdriver.RESPONSE_TYPES.NOOP,
          '{}',
          this.getBrowserInfo_()));
}

/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * Reporter for test results when running in stand alone mode.
 * @param {Function} createNode
 * @param {Function} appendToBody
 * @constructor
 */
jstestdriver.VisualTestReporter = function(createNode, appendToBody, jQuery, parseJson_) {
  this.finished_ = false;
  this.success_ = 1;
  this.report_ = '';
  this.filesLoaded_ = [];
  this.testResults_ = [];
  this.createNode_ = createNode;
  this.appendToBody_ = appendToBody;
  this.jQuery_ = jQuery;
  this.parseJson_ = parseJson_;
  this.passed_ = 0;
  this.testTime_ = new jstestdriver.VisualTestReporter.Interval();
  this.loadTime_ = new jstestdriver.VisualTestReporter.Interval();
};


jstestdriver.VisualTestReporter.prototype.isFinished = function() {
  return this.finished_;
};


/**
 * @return {String} A json representation of the test results.
 */
jstestdriver.VisualTestReporter.prototype.getReport = function() {
  return this.report_;
};


jstestdriver.VisualTestReporter.prototype.getNumFilesLoaded = function() {
  return this.filesLoaded_.length;
};


jstestdriver.VisualTestReporter.prototype.setIsFinished = function(finished) {
  this.finished_ = finished;
  if (finished) {
    var resultsHolder = this.createNode_('div');
    this.jQuery_(resultsHolder).attr('id', 'JsTDVisualResults');
    resultsHolder.className = 'results';
    var details = this.createNode_('div');
    details.className = 'details';
    var cases = {};
    for (var i = 0, result = this.testResults_[i];
         result = this.testResults_[i];
         i++) {
      if (!cases[result.testCaseName] ||
          (cases[result.testCaseName] == jstestdriver.TestResult.RESULT.PASSED)) {
        cases[result.testCaseName] = result.result;
      }
      this.renderResult(result, details);
    }
    var caseSummaryHtml = [];
    for (caseName in cases) {
      caseSummaryHtml.push(
          jstestdriver.formatString(
              '<div class="%s">%s [%s]</div>',
              cases[caseName],
              caseName,
              cases[caseName].toUpperCase()));
    }
    caseSummaryHtml.sort(); // order: error, failed, passed

    var summary = resultsHolder.appendChild(this.createNode_('div'));
    summary.className = 'summary';
    summary.innerHTML = jstestdriver.formatString(
        '<h2>Summary:</h2><div class="cases">%s</div><p>%s out of %s tests passed in %ds.</p><h2>Details:</h2>',
        caseSummaryHtml.join(''),
        this.passed_,
        this.testResults_.length,
        this.testTime_.toSeconds()
    );
    this.renderFilesLoaded(resultsHolder);
    resultsHolder.appendChild(details);
    this.appendToBody_(resultsHolder);
  }
};

jstestdriver.VisualTestReporter.prototype.renderFilesLoaded = function(parent) {
  var filesLoaded = parent.appendChild(this.createNode_('div'));
  filesLoaded.className = 'load';
  filesLoaded.innerHTML = jstestdriver.formatString(
      '%s files loaded in %ds',
      this.filesLoaded_.length,
      this.loadTime_.toSeconds());
};


jstestdriver.VisualTestReporter.prototype.renderResult = function(result, parent) {
  var node = this.createNode_('div');
  node.className = result.result;
  var html = [jstestdriver.formatString(
      '%s.%s [%s] in %d ms',
      result.testCaseName,
      result.testName,
      result.result.toUpperCase(),
      result.time)];
  var errors;
  try {
    errors = result.message.length ? this.parseJson_(result.message) : [];
  } catch (e) {
    errors = 'Error parsing test result: ' + e.message + '\nmessage:' + result.message + '\n';
  }

  while(errors.length) {
    var error = errors.pop()
    html.push('<div class="message">' + error.message + '</div>');
    if (error.stack) {
      var frames = error.stack.split('\n');
      var stack = [];
      for (var i = 0; frames[i]; i++) {
        if (frames[i].indexOf('/test/') != -1) {
          stack.push(frames[i]);
        }
      }
      html.push('<div class="stack">' + stack.join("\n") + '</div>');
    }
  }
  if (result.result != jstestdriver.TestResult.RESULT.PASSED) {
    html.push('<div class="log">' + result.log + '</div>');
  }
  node.innerHTML = html.join('');
  parent.appendChild(node);
};


jstestdriver.VisualTestReporter.prototype.setIsSuccess = function(success) {
  this.success_ = success;
};


/**
 * Adds a test result to the current run.
 * @param {jstestdriver.TestResult}
 */
jstestdriver.VisualTestReporter.prototype.addTestResult = function(testResult) {
  this.testResults_.push(testResult);
};


jstestdriver.VisualTestReporter.prototype.isSuccess = function() {
  return !!this.success_;
};


jstestdriver.VisualTestReporter.prototype.updateIsSuccess = function(success) {
  this.passed_ += Number(success);
  this.success_ = success & this.success_;
};


jstestdriver.VisualTestReporter.prototype.setReport = function(report) {
  this.report_ = report;
};


jstestdriver.VisualTestReporter.prototype.addLoadedFileResults = function(filesLoaded) {
  this.filesLoaded_ = this.filesLoaded_.concat(filesLoaded);
};


jstestdriver.VisualTestReporter.prototype.startLoading = function(when) {
  this.loadTime_.start = when;
}


jstestdriver.VisualTestReporter.prototype.finishLoading = function(when) {
  this.loadTime_.stop = when;
}


jstestdriver.VisualTestReporter.prototype.finishTests = function(when) {
  this.testTime_.stop = when;
}


jstestdriver.VisualTestReporter.prototype.startTests = function(when) {
  this.testTime_.start = when;
}


jstestdriver.VisualTestReporter.prototype.toString = function() {
  return "VisualTestReporter(success=["
      + this.success_ + "], finished=["
      + this.finished_ + "], lastTestResult=["
      + this.lastTestResult_ + "], filesLoaded=["
      + this.filesLoaded_ + "])";
};



/**
 * A period of time defined by milliseconds.
 */
jstestdriver.VisualTestReporter.Interval = function() {
  this.start = 0;
  this.stop = 0;
};


/**
 * @return {Number} Representation of the interval is seconds.
 */
jstestdriver.VisualTestReporter.Interval.prototype.toSeconds = function() {
  return (this.stop - this.start)/1000.0;
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @fileoverview Configuration namespace for setting up JsTD runners.
 */
jstestdriver.config = (function(module) {
  var config = module || {};

  /**
   * Create a new runner.
   */
  config.createRunner = function(createCommandExecutor, opt_runTestLoop) {

    var runTestLoop = opt_runTestLoop || jstestdriver.plugins.defaultRunTestLoop;

    jstestdriver.pluginRegistrar = new jstestdriver.PluginRegistrar();
    jstestdriver.testCaseManager =
        new jstestdriver.TestCaseManager(jstestdriver.pluginRegistrar);

    jstestdriver.testRunner =
        new jstestdriver.TestRunner(jstestdriver.pluginRegistrar);

    jstestdriver.testCaseBuilder =
        new jstestdriver.TestCaseBuilder(jstestdriver.testCaseManager);

    jstestdriver.global.TestCase = jstestdriver.bind(jstestdriver.testCaseBuilder,
        jstestdriver.testCaseBuilder.TestCase);

    jstestdriver.global.AsyncTestCase = jstestdriver.bind(jstestdriver.testCaseBuilder,
        jstestdriver.testCaseBuilder.AsyncTestCase);

    jstestdriver.global.ConditionalTestCase = jstestdriver.bind(jstestdriver.testCaseBuilder,
        jstestdriver.testCaseBuilder.ConditionalTestCase);

    jstestdriver.global.ConditionalAsyncTestCase = jstestdriver.bind(
        jstestdriver.testCaseBuilder,
        jstestdriver.testCaseBuilder.ConditionalAsyncTestCase);

    // default plugin
    var scriptLoader = new jstestdriver.plugins.ScriptLoader(window, document,
          jstestdriver.testCaseManager, jstestdriver.now);
    var stylesheetLoader =
        new jstestdriver.plugins.StylesheetLoader(window, document,
              jstestdriver.jQuery.browser.mozilla || jstestdriver.jQuery.browser.safari);
    var fileLoaderPlugin = new jstestdriver.plugins.FileLoaderPlugin(
            scriptLoader,
            stylesheetLoader,
            jstestdriver.now);
    var testRunnerPlugin =
          new jstestdriver.plugins.TestRunnerPlugin(Date, function() {
        jstestdriver.log(jstestdriver.jQuery('body')[0].innerHTML);
        jstestdriver.jQuery('body').children().remove();
        jstestdriver.jQuery(document).unbind();
        jstestdriver.jQuery(document).die();
    }, runTestLoop);

    jstestdriver.pluginRegistrar.register(
        new jstestdriver.plugins.DefaultPlugin(
            fileLoaderPlugin,
            testRunnerPlugin,
            new jstestdriver.plugins.AssertsPlugin(),
            new jstestdriver.plugins.TestCaseManagerPlugin()));

    jstestdriver.pluginRegistrar.register(
        new jstestdriver.plugins.async.AsyncTestRunnerPlugin(Date, function() {

          jstestdriver.jQuery('body').children().remove();
          jstestdriver.jQuery(document).unbind();
          jstestdriver.jQuery(document).die();
        }, jstestdriver.utils.serializeErrors));

    // legacy
    jstestdriver.testCaseManager.TestCase = jstestdriver.global.TestCase;

    var id = parseInt(jstestdriver.extractId(top.location.toString()));

    function getBrowserInfo() {
      return new jstestdriver.BrowserInfo(id);
    }

    jstestdriver.manualResourceTracker = new jstestdriver.ManualResourceTracker(
        jstestdriver.JSON.parse,
        jstestdriver.JSON.stringify,
        jstestdriver.pluginRegistrar,
        getBrowserInfo,
        new jstestdriver.ManualScriptLoader(
            window,
            jstestdriver.testCaseManager,
            jstestdriver.now));

    return jstestdriver.executor = createCommandExecutor(
        jstestdriver.testCaseManager,
        jstestdriver.testRunner,
        jstestdriver.pluginRegistrar,
        jstestdriver.now,
        window.location.toString(),
        getBrowserInfo,
        id);
  };


  /**
   * Creates a CommandExecutor.
   * @static
   * 
   * @param {jstestdriver.TestCaseManager} testCaseManager
   * @param {jstestdriver.PluginRegistrar} pluginRegistrar
   * @param {function():Number} now
   * @param {String} location The current window location
   * 
   * @return {jstestdriver.CommandExecutor}
   */
  config.createExecutor = function(testCaseManager,
                                   testRunner,
                                   pluginRegistrar,
                                   now,
                                   location,
                                   getBrowserInfo,
                                   id) {
    var url = jstestdriver.createPath(top.location.toString(),
                                      jstestdriver.SERVER_URL + id);

    var unloadSignal = new jstestdriver.Signal(false);

    var streamingService = new jstestdriver.StreamingService(
            url,
            now,
            jstestdriver.convertToJson(jstestdriver.jQuery.post),
            jstestdriver.createSynchPost(jstestdriver.jQuery),
            jstestdriver.setTimeout,
            unloadSignal);

    var currentActionSignal = new jstestdriver.Signal(null);

    var executor = new jstestdriver.CommandExecutor(streamingService,
                                                    testCaseManager,
                                                    testRunner,
                                                    pluginRegistrar,
                                                    now,
                                                    getBrowserInfo,
                                                    currentActionSignal,
                                                    unloadSignal);

    var boundExecuteCommand = jstestdriver.bind(executor, executor.executeCommand);

    function streamStop(response) {
      streamingService.close(response, boundExecuteCommand)
    }

    function streamContinue(response) {
      streamingService.stream(response, boundExecuteCommand);
    }

    var loadTestsCommand = new jstestdriver.LoadTestsCommand(jsonParse,
            pluginRegistrar,
            getBrowserInfo,
            streamStop);

    var runTestsCommand = new jstestdriver.RunTestsCommand(
        testCaseManager,
        testRunner,
        pluginRegistrar,
        getBrowserInfo,
        jstestdriver.now,
        jsonParse,
        streamContinue,
        streamStop);
    var resetCommand = new jstestdriver.ResetCommand(
        window.location,
        unloadSignal,
        jstestdriver.now);

    var noopCommand = new jstestdriver.NoopCommand(streamStop, getBrowserInfo);

    executor.registerCommand('execute', executor, executor.execute);
    executor.registerCommand('noop', noopCommand, noopCommand.sendNoop);
    executor.registerCommand('runAllTests', runTestsCommand, runTestsCommand.runAllTests);
    executor.registerCommand('runTests', runTestsCommand, runTestsCommand.runTests);
    executor.registerCommand('loadTest', loadTestsCommand, loadTestsCommand.loadTest);
    executor.registerCommand('reset', resetCommand, resetCommand.reset);
    executor.registerCommand('dryRun', executor, executor.dryRun);
    executor.registerCommand('dryRunFor', executor, executor.dryRunFor);
    executor.registerCommand('unknownBrowser', null, function() {
      // TODO(corysmith): handle this better.
    });
    executor.registerCommand('stop', null, function() {
      if (window.console && window.console.log) {
        window.console.log('Stopping executor by server request.');
      }
    });
    executor.registerCommand('streamAcknowledged',
                              streamingService,
                              streamingService.streamAcknowledged);


    function getCommand() {
      return currentActionSignal.get();
    }

    var unloadHandler = new jstestdriver.PageUnloadHandler(
        streamingService,
        getBrowserInfo,
        getCommand,
        unloadSignal);
    
    jstestdriver.jQuery(window).bind('unload', jstestdriver.bind(unloadHandler, unloadHandler.onUnload));
    jstestdriver.jQuery(window).bind('beforeunload', jstestdriver.bind(unloadHandler, unloadHandler.onUnload));
    window.onbeforeunload = jstestdriver.bind(unloadHandler, unloadHandler.onUnload);

    return executor;
  }
  
  /**
   * Creates a visual stand alone CommandExecutor.
   * @static
   * 
   * @param {jstestdriver.TestCaseManager} testCaseManager
   * @param {jstestdriver.PluginRegistrar} pluginRegistrar
   * @param {function():Number} now
   * @param {String} location The current window location
   * 
   * @return {jstestdriver.CommandExecutor}
   */
  config.createVisualExecutor = function(testCaseManager,
      testRunner,
      pluginRegistrar,
      now,
      location,
      getBrowserInfo,
      id) {
    return config.createStandAloneExecutorWithReporter(
        testCaseManager,
        testRunner,
        pluginRegistrar,
        now,
        location,
        new jstestdriver.VisualTestReporter(
            function(tagName) {
              return document.createElement(tagName);
            },
            function(node) {
              return document.body.appendChild(node);
            },
            jstestdriver.jQuery,
            JSON.parse),
        getBrowserInfo,
        id);
  };

  /**
   * Creates a stand alone CommandExecutor.
   * @static
   * 
   * @param {jstestdriver.TestCaseManager} testCaseManager
   * @param {jstestdriver.PluginRegistrar} pluginRegistrar
   * @param {function():Number} now
   * @param {String} location The current window location
   * 
   * @return {jstestdriver.CommandExecutor}
   */
  config.createStandAloneExecutor =  function(
      testCaseManager,
      testRunner,
      pluginRegistrar,
      now,
      location,
      getBrowserInfo,
      id) {
    return config.createStandAloneExecutorWithReporter(testCaseManager,
        testRunner,
        pluginRegistrar,
        now,
        location,
        new jstestdriver.StandAloneTestReporter(),
        getBrowserInfo,
        id)
  };


  // TODO(corysmith): Factor out the duplicated code.
  /**
   * Creates a stand alone CommandExecutor configured with a reporter.
   * @static
   * 
   * @param {jstestdriver.TestCaseManager} testCaseManager
   * @param {jstestdriver.PluginRegistrar} pluginRegistrar
   * @param {function():Number} now
   * @param {String} location The current window location
   * 
   * @return {jstestdriver.CommandExecutor}
   */
  config.createStandAloneExecutorWithReporter = function(
      testCaseManager,
      testRunner,
      pluginRegistrar,
      now,
      location,
      reporter,
      getBrowserInfo,
      id) {
    var url =jstestdriver.createPath(top.location.toString(),
        jstestdriver.SERVER_URL + id);

    var unloadSignal = new jstestdriver.Signal(false);

    var streamingService = new jstestdriver.StreamingService(
            url,
            now,
            jstestdriver.convertToJson(jstestdriver.jQuery.post),
            jstestdriver.createSynchPost(jstestdriver.jQuery),
            jstestdriver.setTimeout,
            unloadSignal);

    window.top.G_testRunner = reporter;
    jstestdriver.reporter = reporter;

    var currentActionSignal = new jstestdriver.Signal(null);

    var executor = new jstestdriver.CommandExecutor(streamingService,
            testCaseManager,
            testRunner,
            pluginRegistrar,
            now,
            getBrowserInfo,
            currentActionSignal,
            unloadSignal);

    var boundExecuteCommand = jstestdriver.bind(executor,
                                                executor.executeCommand);

    function streamStop(response) {
      streamingService.close(response, boundExecuteCommand)
    }

    function streamContinue(response) {
      streamingService.stream(response, boundExecuteCommand);
    }

    var loadTestsCommand = new jstestdriver.StandAloneLoadTestsCommand(
        jsonParse,
        pluginRegistrar,
        getBrowserInfo,
        streamStop,
        reporter,
        jstestdriver.now);

    var runTestsCommand =
        new jstestdriver.StandAloneRunTestsCommand(
            testCaseManager,
            testRunner,
            pluginRegistrar,
            getBrowserInfo,
            reporter,
            now,
            jsonParse,
            streamContinue,
            streamStop);

    executor.registerTracedCommand('execute', executor, executor.execute);
    executor.registerTracedCommand('noop', null, streamStop);
    executor.registerTracedCommand('runAllTests', runTestsCommand, runTestsCommand.runAllTests);
    executor.registerTracedCommand('runTests', runTestsCommand, runTestsCommand.runTests);
    executor.registerTracedCommand('loadTest', loadTestsCommand, loadTestsCommand.loadTest);
    executor.registerTracedCommand('reset', executor, executor.reset);
    executor.registerTracedCommand('dryRun', executor, executor.dryRun);
    executor.registerTracedCommand('dryRunFor', executor, executor.dryRunFor);
    executor.registerCommand('streamAcknowledged',
            streamingService,
            streamingService.streamAcknowledged);
    executor.registerCommand('unknownBrowser', null, function() {
      // TODO(corysmith): handle this better.
    });
    executor.registerCommand('stop', null, function() {
      if (window.console && window.console.log) {
        window.console.log('Stopping executor by server request.');
      }
    });

    return executor;
  }

  return config;
})(jstestdriver.config);

/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @fileoverview Defines the FiniteUseCallback class, which decorates a
 * Javascript function by notifying the test runner about any exceptions thrown
 * when the function executes.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.CatchingCallback');

goog.require('jstestdriver');

/**
 * Constructs a CatchingCallback.
 *
 * @param {Object} testCase the testCase to use as 'this' when calling the
 *    wrapped function.
 * @param {jstestdriver.plugins.async.CallbackPool} pool the pool to which this
 *    callback belongs.
 * @param {Function} wrapped the wrapped callback function.
 * @constructor
 */
jstestdriver.plugins.async.CatchingCallback = function(
    testCase, pool, wrapped) {
  this.testCase_ = testCase;
  this.pool_ = pool;
  this.callback_ = wrapped;
};


/**
 * Invokes the wrapped callback, catching any exceptions and reporting the
 * status to the pool.
 * @return {*} The return value of the original callback.
 */
jstestdriver.plugins.async.CatchingCallback.prototype.invoke = function() {
  var result;
  var message;
  try {
    result = this.callback_.apply(this.testCase_, arguments);
    message = 'success.';
    return result;
  } catch (e) {
    this.pool_.onError(e);
    message = 'failure: ' + e;
    throw e;
  } finally {
    this.pool_.remove(message);
  }
};
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the ExpiringCallback class, which decorates a
 * Javascript function by restricting the length of time the asynchronous system
 * may delay before calling the function.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.ExpiringCallback');

goog.require('jstestdriver');

/**
 * Constructs an ExpiringCallback.
 *
 * @param {jstestdriver.plugins.async.CallbackPool} pool The pool to which this
 *     callback belongs.
 * @param {jstestdriver.plugins.async.FiniteUseCallback} callback A
 *     FiniteUseCallback.
 * @param {jstestdriver.plugins.async.Timeout} timeout A Timeout object.
 * @param {string} stepDescription A description of the current test step.
 * @constructor
 */
jstestdriver.plugins.async.ExpiringCallback = function(
    pool, callback, timeout, stepDescription, callbackDescription) {
  this.pool_ = pool;
  this.callback_ = callback;
  this.timeout_ = timeout;
  this.stepDescription_ = stepDescription;
  this.callbackDescription_ = callbackDescription;
};


/**
 * Arms this callback to expire after the given delay.
 *
 * @param {number} delay The amount of time (ms) before this callback expires.
 */
jstestdriver.plugins.async.ExpiringCallback.prototype.arm = function(delay) {
  var callback = this;
  this.timeout_.arm(function() {
    callback.pool_.onError(new Error('Callback \'' +
        callback.callbackDescription_ + '\' expired after ' + delay +
        ' ms during test step \'' + callback.stepDescription_ + '\''));
    callback.pool_.remove('expired.', callback.callback_.getRemainingUses());
    callback.callback_.deplete();
  }, delay);
};


/**
 * Invokes this callback.
 * @return {*} The return value of the FiniteUseCallback.
 */
jstestdriver.plugins.async.ExpiringCallback.prototype.invoke = function() {
  return this.callback_.invoke.apply(this.callback_, arguments);
};

/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the FiniteUseCallback class, which decorates a
 * Javascript function by restricting the number of times the asynchronous
 * system may call it.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.FiniteUseCallback');

goog.require('jstestdriver');

/**
 * Constructs a FiniteUseCallback.
 *
 * @param {jstestdriver.plugins.async.CatchingCallback} callback A
 *     CatchingCallback.
 * @param {Function} onDepleted a function to execute when this
 *     FiniteUseCallback depletes.
 * @param {?number} opt_remainingUses the number of permitted uses remaining;
 *     defaults to one.
 * @constructor
 */
jstestdriver.plugins.async.FiniteUseCallback = function(
    callback, onDepleted, opt_remainingUses) {
  this.callback_ = callback;
  this.onDepleted_ = onDepleted;
  this.remainingUses_ = opt_remainingUses || 1;
};


/**
 * Depletes the remaining permitted uses.  Calls onDepleted.
 */
jstestdriver.plugins.async.FiniteUseCallback.prototype.deplete = function() {
  this.remainingUses_ = 0;
  if (this.onDepleted_) {
    this.onDepleted_.apply();
  }
};


/**
 * @return {number} The number of remaining permitted uses.
 */
jstestdriver.plugins.async.FiniteUseCallback.prototype.getRemainingUses =
    function() {
  return this.remainingUses_;
};


/**
 * Invokes this callback if it is usable. Calls onDepleted if invoking this
 * callback depletes its remaining permitted uses.
 * @param {...*} var_args The original callback arguments.
 * @return {*} The return value of the CatchingCallback or null.
 */
jstestdriver.plugins.async.FiniteUseCallback.prototype.invoke =
    function(var_args) {
  if (this.isUsable()) {
    try {
      this.remainingUses_ -= 1;
      return this.callback_.invoke.apply(this.callback_, arguments);
    } finally {
      if (this.onDepleted_ && !this.isUsable()) {
        this.onDepleted_.apply();
      }
    }
  }
};


/**
 * @return {boolean} True if any permitted uses remain.
 */
jstestdriver.plugins.async.FiniteUseCallback.prototype.isUsable = function() {
  return this.remainingUses_ > 0;
};
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the Timeout class.  The arm() method is equivalent to
 * window.setTimeout() and maybeDisarm() is equivalent to window.clearTimeout().
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.Timeout');

goog.require('jstestdriver');

/**
 * Constructs a Timeout. Accepts alternate implementations of setTimeout and
 * clearTimeout.
 *
 * @param {Function} setTimeout The global setTimeout function to use.
 * @param {Function} clearTimeout The global clearTimeout function to use.
 * @constructor
 */
jstestdriver.plugins.async.Timeout = function(setTimeout, clearTimeout) {
  this.setTimeout_ = setTimeout;
  this.clearTimeout_ = clearTimeout;
  this.handle_ = null;
};


/**
 * Arms this Timeout to fire after the specified delay.
 *
 * @param {Function} callback The callback to call after the delay passes.
 * @param {number} delay The timeout delay in milliseconds.
 */
jstestdriver.plugins.async.Timeout.prototype.arm = function(callback, delay) {
  var self = this;
  this.handle_ = this.setTimeout_(function() {
    self.maybeDisarm();
    return callback.apply(null, arguments);
  }, delay);
};

/**
 * Explicitly disarms the timeout.
 * @private
 */
jstestdriver.plugins.async.Timeout.prototype.disarm_ = function() {
  this.clearTimeout_(this.handle_);
  this.handle_ = null;
};


/**
 * @return {boolean} True if the timeout is armed.
 */
jstestdriver.plugins.async.Timeout.prototype.isArmed = function() {
  return this.handle_ != null;
};


/**
 * Disarms the timeout if it is armed.
 */
jstestdriver.plugins.async.Timeout.prototype.maybeDisarm = function() {
  if (this.isArmed()) {
    this.disarm_();
  }
};
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the TestSafeCallbackBuilder class. It decorates a
 * Javascript function with several safeguards so that it may be safely executed
 * asynchronously within a test.
 *
 * The safeguards include:
 *   1) notifying the test runner about any exceptions thrown when the function
 *      executes
 *   2) restricting the number of times the asynchronous system may call the
 *      function
 *   3) restricting the length of time the asynchronous system may delay before
 *      calling the function
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.TestSafeCallbackBuilder');

goog.require('jstestdriver');
goog.require('jstestdriver.plugins.async.CatchingCallback');
goog.require('jstestdriver.plugins.async.ExpiringCallback');
goog.require('jstestdriver.plugins.async.FiniteUseCallback');
goog.require('jstestdriver.plugins.async.Timeout');

/**
 * Constructs a TestSafeCallbackBuilder.
 *
 * @param {Function} opt_setTimeout the global setTimeout function to use.
 * @param {Function} opt_clearTimeout the global clearTimeout function to use.
 * @param {Function} opt_timeoutConstructor a constructor for obtaining new the
 *     Timeouts.
 * @constructor
 */
jstestdriver.plugins.async.TestSafeCallbackBuilder = function(
    opt_setTimeout, opt_clearTimeout, opt_timeoutConstructor) {
  this.setTimeout_ = opt_setTimeout || jstestdriver.setTimeout;
  this.clearTimeout_ = opt_clearTimeout || jstestdriver.clearTimeout;
  this.timeoutConstructor_ = opt_timeoutConstructor ||
      jstestdriver.plugins.async.Timeout;
  this.callbackDescription = 'Unknown callback.';
  this.stepDescription_ = 'Unknown step.';
  this.pool_ = null;
  this.remainingUses_ = null;
  this.testCase_ = null;
  this.wrapped_ = null;
};


/**
 * Returns the original function decorated with safeguards.
 * @return {*} The return value of the original callback.
 */
jstestdriver.plugins.async.TestSafeCallbackBuilder.prototype.build =
    function() {
  var catchingCallback = new jstestdriver.plugins.async.CatchingCallback(
      this.testCase_, this.pool_, this.wrapped_);
  var timeout = new (this.timeoutConstructor_)(
      this.setTimeout_, this.clearTimeout_);
  var onDepleted = function() {
    timeout.maybeDisarm();
  };
  var finiteUseCallback = new jstestdriver.plugins.async.FiniteUseCallback(
      catchingCallback, onDepleted, this.remainingUses_);
  return new jstestdriver.plugins.async.ExpiringCallback(
      this.pool_, finiteUseCallback, timeout,
      this.stepDescription_, this.callbackDescription_);
};


jstestdriver.plugins.async.TestSafeCallbackBuilder.
    prototype.setCallbackDescription = function(callbackDescription) {
  this.callbackDescription_ = callbackDescription;
  return this;
};


jstestdriver.plugins.async.TestSafeCallbackBuilder.
    prototype.setStepDescription = function(stepDescription) {
  this.stepDescription_ = stepDescription;
  return this;
};


/**
 * @param {jstestdriver.plugins.async.CallbackPool} pool the CallbackPool to
 *     contain the callback.
 * @return {jstestdriver.plugins.async.TestSafeCallbackBuilder} This.
 */
jstestdriver.plugins.async.TestSafeCallbackBuilder.prototype.setPool = function(
    pool) {
  this.pool_ = pool;
  return this;
};


/**
 * @param {number} remainingUses The remaining number of permitted calls.
 * @return {jstestdriver.plugins.async.TestSafeCallbackBuilder} This.
 */
jstestdriver.plugins.async.TestSafeCallbackBuilder.prototype.setRemainingUses =
    function(remainingUses) {
  this.remainingUses_ = remainingUses;
  return this;
};


/**
 * @param {Object} testCase The test case instance available as 'this' within
 *     the function's scope.
 * @return {jstestdriver.plugins.async.TestSafeCallbackBuilder} This.
 */
jstestdriver.plugins.async.TestSafeCallbackBuilder.prototype.setTestCase =
    function(testCase) {
  this.testCase_ = testCase;
  return this;
};


/**
 * @param {Function} wrapped The function wrapped by the above safeguards.
 * @return {jstestdriver.plugins.async.TestSafeCallbackBuilder} This.
 */
jstestdriver.plugins.async.TestSafeCallbackBuilder.prototype.setWrapped =
    function(wrapped) {
  this.wrapped_ = wrapped;
  return this;
};
/*
 * Copyright 2011 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the CallbackPool class, which decorates given callback
 * functions with safeguards and tracks them until they execute or expire.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.CallbackPool');

goog.require('jstestdriver');
goog.require('jstestdriver.plugins.async.TestSafeCallbackBuilder');

/**
 * Constructs a CallbackPool.
 *
 * @param {Function} setTimeout The global setTimeout function.
 * @param {Object} testCase The test case instance.
 * @param {Function} onPoolComplete A function to call when the pool empties.
 * @param {string} stepDescription A description of the current test step.
 * @param {boolean} opt_pauseForHuman Whether or not to pause for debugging.
 * @param {Function} opt_callbackBuilderConstructor An optional constructor for
 *     a callback builder.
 * @constructor
 */
jstestdriver.plugins.async.CallbackPool = function(setTimeout, testCase,
      onPoolComplete, stepDescription, opt_pauseForHuman,
      opt_callbackBuilderConstructor) {
  this.setTimeout_ = setTimeout;
  this.testCase_ = testCase;
  this.onPoolComplete_ = onPoolComplete;
  this.stepDescription_ = stepDescription;
  this.pauseForHuman_ = !!opt_pauseForHuman;
  this.callbackBuilderConstructor_ = opt_callbackBuilderConstructor ||
      jstestdriver.plugins.async.TestSafeCallbackBuilder;
  this.errors_ = [];
  this.count_ = 0;
  this.callbackIndex_ = 1;
  this.active_ = false;
};


/**
 * The number of milliseconds to wait before expiring a delinquent callback.
 */
jstestdriver.plugins.async.CallbackPool.TIMEOUT = 30000;


/**
 * Calls onPoolComplete if the pool is active and empty.
 */
jstestdriver.plugins.async.CallbackPool.prototype.maybeComplete = function() {
  if (this.active_ && this.count_ == 0 && this.onPoolComplete_) {
    var pool = this;
    this.setTimeout_(function() {
      pool.active_ = false;
      pool.onPoolComplete_(pool.errors_);
    }, 0);
  }
};


/**
 * Activates the pool and calls maybeComplete.
 */
jstestdriver.plugins.async.CallbackPool.prototype.activate = function() {
    this.active_ = true;
    this.maybeComplete();
};


/**
 * @return {number} The number of outstanding callbacks in the pool.
 */
jstestdriver.plugins.async.CallbackPool.prototype.count = function() {
  return this.count_;
};


/**
 * Accepts errors to later report them to the test runner via onPoolComplete.
 * @param {Error} error The error to report.
 */
jstestdriver.plugins.async.CallbackPool.prototype.onError = function(error) {
  this.errors_.push(error);
  this.count_ = 0;
  this.maybeComplete();
};


/**
 * Adds a callback function to the pool, optionally more than once.
 *
 * @param {Function} wrapped The callback function to decorate with safeguards
 *     and to add to the pool.
 * @param {number} opt_n The number of permitted uses of the given callback;
 *     defaults to one.
 * @param {number} opt_timeout The timeout in milliseconds.
 * @return {Function} A test safe callback.
 */
jstestdriver.plugins.async.CallbackPool.prototype.addCallback = function(
    wrapped, opt_n, opt_timeout, opt_description) {
  this.count_ += opt_n || 1;
  var callback = new (this.callbackBuilderConstructor_)()
      .setCallbackDescription(opt_description || '#' + this.callbackIndex_++)
      .setStepDescription(this.stepDescription_)
      .setPool(this)
      .setRemainingUses(opt_n)
      .setTestCase(this.testCase_)
      .setWrapped(wrapped)
      .build();
  if (!this.pauseForHuman_) {
    callback.arm(opt_timeout ||
        jstestdriver.plugins.async.CallbackPool.TIMEOUT);
  }
  return function() {
    return callback.invoke.apply(callback, arguments);
  };
};


/**
 * Adds a callback function to the pool, optionally more than once.
 *
 * @param {Function} wrapped The callback function to decorate with safeguards
 *     and to add to the pool.
 * @param {number} opt_n The number of permitted uses of the given callback;
 *     defaults to one.
 * @deprecated Use CallbackPool#addCallback().
 */
jstestdriver.plugins.async.CallbackPool.prototype.add =
    jstestdriver.plugins.async.CallbackPool.prototype.addCallback;


/**
 * @return {Function} An errback function to attach to an asynchronous system so
 *     that the test runner can be notified in the event of error.
 * @param {string} message A message to report to the user upon error.
 */
jstestdriver.plugins.async.CallbackPool.prototype.addErrback = function(
    message) {
  var pool = this;
  return function() {
    pool.onError(new Error(
        'Errback ' + message + ' called with arguments: ' +
            Array.prototype.slice.call(arguments)));
  };
};


/**
 * Removes a callback from the pool, optionally more than one.
 *
 * @param {string} message A message to pass to the pool for logging purposes;
 *     usually the reason that the callback was removed from the pool.
 * @param {number} opt_n The number of callbacks to remove from the pool.
 */
jstestdriver.plugins.async.CallbackPool.prototype.remove = function(
    message, opt_n) {
  if (this.count_ > 0) {
    this.count_ -= opt_n || 1;
    this.maybeComplete();
  }
};
/*
 * Copyright 2011 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the CallbackPoolDelegate class. Encapsulates a
 * CallbackPool behind a narrower interface. Also, validates arguments.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.CallbackPoolDelegate');

goog.require('jstestdriver');

/**
 * Constructs a CallbackPoolDelegate.
 * @param {jstestdriver.plugins.async.CallbackPool} pool The pool.
 * @constructor
 * @export
 */
jstestdriver.plugins.async.CallbackPoolDelegate = function(pool) {
  this.pool_ = pool;
};


/**
 * Adds a callback to the pool.
 * @param {Object|Function} callback The callback to wrap.
 * @param {number=} opt_n An optional number of times to wait for the callback to
 *     be called.
 * @param {number=} opt_timeout The timeout in milliseconds.
 * @param {string=} opt_description The callback description.
 * @return {Function} The wrapped callback.
 * @export
 */
jstestdriver.plugins.async.CallbackPoolDelegate.prototype.addCallback = function(
    callback, opt_n, opt_timeout, opt_description) {
  if (typeof callback == 'object') {
    var params = callback;
    callback = params['callback'];
    opt_n = params['invocations'];
    opt_timeout = params['timeout'] ? params['timeout'] * 1000 : undefined;
    opt_description = params['description'];
  }

  if (typeof callback == 'function' && callback) {
    return this.pool_.addCallback(
        callback, opt_n, opt_timeout, opt_description);
  }

  return null;
};


/**
 * @return {Function} An errback function to attach to an asynchronous system so
 *     that the test runner can be notified in the event of error.
 * @param {string} message A message to report to the user upon error.
 * @export
 */
jstestdriver.plugins.async.CallbackPoolDelegate.prototype.addErrback = function(
    message) {
  return this.pool_.addErrback(message);
};


/**
 * Adds a callback to the pool.
 * @param {Object|Function} callback The callback to wrap.
 * @param {number=} opt_n An optional number of times to wait for the callback to
 *     be called.
 * @param {number=} opt_timeout The timeout in milliseconds.
 * @param {string=} opt_description The callback description.
 * @return {Function} The wrapped callback.
 * @export
 */
jstestdriver.plugins.async.CallbackPoolDelegate.prototype.add =
    jstestdriver.plugins.async.CallbackPoolDelegate.prototype.addCallback;


/**
 * A no-op callback that's useful for waiting until an asynchronous operation
 * completes without performing any action.
 * @param {Object|number=} opt_n An optional number of times to wait for the
 *     callback to be called.
 * @param {number=} opt_timeout The timeout in milliseconds.
 * @param {string=} opt_description The description.
 * @return {Function} A noop callback.
 * @export
 */
jstestdriver.plugins.async.CallbackPoolDelegate.prototype.noop = function(
    opt_n, opt_timeout, opt_description) {
  if (typeof opt_n == 'object') {
    var params = opt_n;
    opt_timeout = params['timeout'] ? params['timeout'] * 1000 : undefined;
    opt_description = params['description'];
    opt_n = params['invocations'];
  }
  return this.pool_.addCallback(
      jstestdriver.EMPTY_FUNC, opt_n, opt_timeout, opt_description);
};
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the DeferredQueue class.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.DeferredQueue');

goog.require('jstestdriver');
goog.require('jstestdriver.plugins.async.DeferredQueueDelegate');
goog.require('jstestdriver.plugins.async.CallbackPool');
goog.require('jstestdriver.plugins.async.CallbackPoolDelegate');

/**
 * Constructs a DeferredQueue.
 * @param {Function} setTimeout The setTimeout function.
 * @param {Object} testCase The test case that owns this queue.
 * @param {Function} onQueueComplete The queue complete callback.
 * @param {jstestdriver.plugins.async.DeferredQueueDelegate} delegate The
 *     delegate wrapping all DeferredQueues for this test run0.
 * @param {boolean} opt_pauseForHuman Whether or not to pause for debugging.
 * @param {Function} opt_queueConstructor The DeferredQueue constructor.
 * @param {Function} opt_queueDelegateConstructor The DeferredQueueDelegate
 *     constructor.
 * @param {Function} opt_poolConstructor The CallbackPool constructor.
 * @param {Function} opt_poolDelegateConstructor The CallbackPoolDelegate constructor.
 * @constructor
 */
jstestdriver.plugins.async.DeferredQueue = function(setTimeout, testCase,
    onQueueComplete, delegate, opt_pauseForHuman, opt_queueConstructor,
    opt_queueDelegateConstructor, opt_poolConstructor, opt_poolDelegateConstructor) {
  this.setTimeout_ = setTimeout;
  this.testCase_ = testCase;
  this.onQueueComplete_ = onQueueComplete;
  this.delegate_ = delegate;
  this.pauseForHuman_ = !!opt_pauseForHuman;
  this.queueConstructor_ = opt_queueConstructor ||
      jstestdriver.plugins.async.DeferredQueue;
  this.queueDelegateConstructor_ = opt_queueDelegateConstructor ||
      jstestdriver.plugins.async.DeferredQueueDelegate;
  this.poolConstructor_ = opt_poolConstructor ||
      jstestdriver.plugins.async.CallbackPool;
  this.poolDelegateConstructor_ = opt_poolDelegateConstructor ||
      jstestdriver.plugins.async.CallbackPoolDelegate;
  this.descriptions_ = [];
  this.operations_ = [];
  this.errors_ = [];
};


/**
 * Executes a step of the test.
 * @param {Function} operation The next test step.
 * @param {Function} onQueueComplete The queue complete callback.
 * @private
 */
jstestdriver.plugins.async.DeferredQueue.prototype.execute_ = function(
    description, operation, onQueueComplete) {
  var queue = new (this.queueConstructor_)(this.setTimeout_,
      this.testCase_, onQueueComplete, this.delegate_, this.pauseForHuman_);
  this.delegate_.setQueue(queue);

  var onPoolComplete = function(errors) {
    queue.finishStep_(errors);
  };
  var pool = new (this.poolConstructor_)(
      this.setTimeout_, this.testCase_, onPoolComplete, description, this.pauseForHuman_);
  var poolDelegate = new (this.poolDelegateConstructor_)(pool);

  if (operation) {
    try {
      operation.call(this.testCase_, poolDelegate, this.delegate_);
    } catch (e) {
      pool.onError(e);
    }
  }

  pool.activate();
};


/**
 * Enqueues a test step.
 * @param {string} description The test step description.
 * @param {Function} operation The test step to add to the queue.
 */
jstestdriver.plugins.async.DeferredQueue.prototype.defer = function(
    description, operation) {
  this.descriptions_.push(description);
  this.operations_.push(operation);
};


/**
 * Starts the next test step.
 */
jstestdriver.plugins.async.DeferredQueue.prototype.startStep = function() {
  var nextDescription = this.descriptions_.shift();
  var nextOp = this.operations_.shift();
  if (nextOp) {
    var q = this;
    this.execute_(nextDescription, nextOp, function(errors) {
      q.finishStep_(errors);
    });
  } else {
    this.onQueueComplete_([]);
  }
};


/**
 * Finishes the current test step.
 * @param {Array.<Error>} errors An array of any errors that occurred during the
 *     previous test step.
 * @private
 */
jstestdriver.plugins.async.DeferredQueue.prototype.finishStep_ = function(
    errors) {
  this.errors_ = this.errors_.concat(errors);
  if (this.errors_.length) {
    this.onQueueComplete_(this.errors_);
  } else {
    this.startStep();
  }
};
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the DeferredQueueInterface class. Encapsulates a
 * DeferredQueue behind a narrower interface. Also, validates arguments.
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.DeferredQueueDelegate');

goog.require('jstestdriver');

/**
 * Constructs a DeferredQueueDelegate.
 * @param {function(Object)} toJson a function to convert objects to JSON.
 * @constructor
 * @export
 */
jstestdriver.plugins.async.DeferredQueueDelegate = function(toJson) {
  this.toJson_ = toJson;
  this.q_ = null;
  this.step_ = 1;
};


/**
 * Sets the current queue instance.
 * @param {jstestdriver.plugins.async.DeferredQueue} queue The queue.
 */
jstestdriver.plugins.async.DeferredQueueDelegate.prototype.setQueue = function(
    queue) {
  this.q_ = queue;
};


/**
 * Adds a function to the queue to call later.
 * @param {string|Function} description The description or function.
 * @param {Function=} operation The function.
 * @return {jstestdriver.plugins.async.DeferredQueueDelegate} This.
 * @export
 */
jstestdriver.plugins.async.DeferredQueueDelegate.prototype.call = function(
    description, operation) {
  if (!this.q_) {
    throw new Error('Queue undefined!');
  }

  if (typeof description == 'function') {
    operation = description;
    description = this.nextDescription_();
  }

  if (typeof description == 'object') {
    operation = description.operation;
    description = description.description;
  }

  if (!description) {
    description = this.nextDescription_();
  }

  if (operation) {
    this.q_.defer(description, operation);
    this.step_ += 1;
  }

  return this;
};


/**
 * @return {string} A description for the next step.
 */
jstestdriver.plugins.async.DeferredQueueDelegate.prototype.nextDescription_ =
    function() {
  return '#' + this.step_;
};


/**
 * Adds a function to the queue to call later.
 * @param {string|Function} description The description or function.
 * @param {Function=} operation The function.
 * @deprecated Use DeferredQueueDelegate#call().
 * @export
 */
jstestdriver.plugins.async.DeferredQueueDelegate.prototype.defer =
    jstestdriver.plugins.async.DeferredQueueDelegate.prototype.call;
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the TestStage class.
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.TestStage');
goog.provide('jstestdriver.plugins.async.TestStage.Builder');

goog.require('jstestdriver');
goog.require('jstestdriver.setTimeout');
goog.require('jstestdriver.plugins.async.DeferredQueueDelegate');
goog.require('jstestdriver.plugins.async.DeferredQueue');

/**
 * Constructs a TestStage.
 *
 * A TestStage is an executable portion of a test, such as setUp, tearDown, or
 * the test method.
 *
 * @param {Function} onError An error handler.
 * @param {Function} onStageComplete A callback for stage completion.
 * @param {Object} testCase The test case that owns this test stage.
 * @param {Function} testMethod The test method this stage represents.
 * @param {function(Object)} toJson a function to convert objects to JSON.
 * @param {Object} opt_argument An argument to pass to the test method.
 * @param {boolean} opt_pauseForHuman Whether to pause for debugging.
 * @param {Function} opt_queueDelegateConstructor The constructor of
 * DeferredQueueDelegate.
 * @param {Function} opt_queueConstructor The constructor of DeferredQueue.
 * @param {Function} opt_setTimeout The setTimeout function or suitable
 *     replacement.
 * @constructor
 */
jstestdriver.plugins.async.TestStage = function(
    onError, onStageComplete, testCase, testMethod, toJson, opt_argument,
    opt_pauseForHuman, opt_queueDelegateConstructor, opt_queueConstructor,
    opt_setTimeout) {
  this.onError_ = onError;
  this.onStageComplete_ = onStageComplete;
  this.testCase_ = testCase;
  this.testMethod_ = testMethod;
  this.toJson_ = toJson;
  this.argument_ = opt_argument;
  this.pauseForHuman_ = !!opt_pauseForHuman;
  this.queueDelegateConstructor_ = opt_queueDelegateConstructor ||
      jstestdriver.plugins.async.DeferredQueueDelegate;
  this.queueConstructor_ = opt_queueConstructor ||
      jstestdriver.plugins.async.DeferredQueue;
  this.setTimeout_ = opt_setTimeout || jstestdriver.setTimeout;
};


/**
 * Executes this TestStage.
 */
jstestdriver.plugins.async.TestStage.prototype.execute = function() {
  var delegate = new (this.queueDelegateConstructor_)(this.toJson_);
  var queue = new (this.queueConstructor_)(this.setTimeout_, this.testCase_,
      this.onStageComplete_, delegate, this.pauseForHuman_);
  delegate.setQueue(queue);

  if (this.testMethod_) {
    try {
      this.testMethod_.call(this.testCase_, delegate, this.argument_);
    } catch (e) {
      this.onError_(e);
    }
  }

  queue.startStep();
};



/**
 * Constructor for a Builder of TestStages. Used to avoid confusion when
 * trying to construct TestStage objects (as the constructor takes a lot
 * of parameters of similar types).
 * @constructor
 */
jstestdriver.plugins.async.TestStage.Builder = function() {
  this.onError_ = null;
  this.onStageComplete_ = null;
  this.testCase_ = null;
  this.testMethod_ = null;
  this.toJson_ = null;
  this.opt_argument_ = null;
  this.opt_pauseForHuman_ = null;
  this.opt_queueDelegateConstructor_ =
      jstestdriver.plugins.async.DeferredQueueDelegate;
  this.opt_queueConstructor_ = jstestdriver.plugins.async.DeferredQueue;
  this.opt_setTimeout_ = jstestdriver.setTimeout;
};


// Setters for the various fields; they return the Builder instance to allow
// method call chaining.
jstestdriver.plugins.async.TestStage.Builder.prototype.setOnError =
    function(onError) {
  this.onError_ = onError;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setOnStageComplete =
    function(onStageComplete) {
  this.onStageComplete_ = onStageComplete;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setTestCase =
    function(testCase) {
  this.testCase_ = testCase;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setTestMethod =
    function(testMethod) {
  this.testMethod_ = testMethod;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setToJson =
    function(toJson) {
  this.toJson_ = toJson;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setArgument =
    function(argument) {
  this.opt_argument_ = argument;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setPauseForHuman =
    function(pauseForHuman) {
  this.opt_pauseForHuman_ = pauseForHuman;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.
    setQueueDelegateConstructor = function(queueDelegateConstructor) {
  this.opt_queueDelegateConstructor_ = queueDelegateConstructor;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setQueueConstructor =
    function(queueConstructor) {
  this.opt_queueConstructor_ = queueConstructor;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.setTimeoutSetter =
    function(setTimeout) {
  this.opt_setTimeout_ = setTimeout;
  return this;
};


jstestdriver.plugins.async.TestStage.Builder.prototype.build = function() {
  return new jstestdriver.plugins.async.TestStage(
      this.onError_, this.onStageComplete_, this.testCase_, this.testMethod_,
      this.toJson_, this.opt_argument_, this.opt_pauseForHuman_,
      this.opt_queueDelegateConstructor_, this.opt_queueConstructor_,
      this.opt_setTimeout_);
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.plugins.ScriptLoader = function(win, dom, testCaseManager, now) {
  this.win_ = win;
  this.dom_ = dom;
  this.testCaseManager_ = testCaseManager;
  this.now_ = now;
};


jstestdriver.plugins.ScriptLoader.prototype.load = function(file, callback) {
  this.testCaseManager_.removeTestCaseForFilename(file.fileSrc);
  this.fileResult_ = null;
  var head = this.dom_.getElementsByTagName('head')[0];
  var script = this.dom_.createElement('script');
  var start = this.now_();

  if (!jstestdriver.jQuery.browser.opera) {
    script.onload = jstestdriver.bind(this, function() {
      this.cleanCallBacks(script)
      this.onLoad_(file, callback, start);
    });
  }
  script.onreadystatechange = jstestdriver.bind(this, function() {
    if (script.readyState === "loaded" || script.readyState === "complete") {
      this.cleanCallBacks(script)
      this.onLoad_(file, callback, start);
    }
  });

  var handleError = jstestdriver.bind(this, function(msg, url, line) {
    this.testCaseManager_.removeTestCaseForFilename(file.fileSrc);
    var loadMsg = 'error loading file: ' + file.fileSrc;

    if (line != undefined && line != null) {
      loadMsg += ':' + line;
    }
    if (msg != undefined && msg != null) {
      loadMsg += ': ' + msg;
    }
    this.cleanCallBacks(script)
    callback(new jstestdriver.FileResult(file, false, loadMsg));
  });
  this.win_.onerror = handleError; 
  script.onerror = handleError;

  script.type = "text/javascript";
  script.src = file.fileSrc;
  head.appendChild(script);

};

jstestdriver.plugins.ScriptLoader.prototype.cleanCallBacks = function(script) {
  script.onerror = jstestdriver.EMPTY_FUNC;
  script.onload = jstestdriver.EMPTY_FUNC;
  script.onreadystatechange = jstestdriver.EMPTY_FUNC;
  this.win_.onerror = jstestdriver.EMPTY_FUNC;
};


jstestdriver.plugins.ScriptLoader.prototype.onLoad_ =
    function(file, callback, start) {
  this.testCaseManager_.updateLatestTestCase(file.fileSrc);
  var result = new jstestdriver.FileResult(file, true, '', this.now_() - start);
  this.win_.onerror = jstestdriver.EMPTY_FUNC;
  callback(result);
};


jstestdriver.plugins.ScriptLoader.prototype.updateResult_ = function(fileResult) {
  if (this.fileResult_ == null) {
    this.fileResult_ = fileResult;
  }
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.plugins.StylesheetLoader = function(win, dom, synchronousCallback) {
  this.win_ = win;
  this.dom_ = dom;
  this.synchronousCallback_ = synchronousCallback;
};


jstestdriver.plugins.StylesheetLoader.prototype.load = function(file, callback) {
  this.fileResult_ = null;
  var head = this.dom_.getElementsByTagName('head')[0];
  var link = this.dom_.createElement('link');
  var handleError = jstestdriver.bind(this, function(msg, url, line) {
    var loadMsg = 'error loading file: ' + file.fileSrc;

    if (line != undefined && line != null) {
      loadMsg += ':' + line;
    }
    if (msg != undefined && msg != null) {
      loadMsg += ': ' + msg;
    }
    this.updateResult_(new jstestdriver.FileResult(file, false, loadMsg));
  });

  this.win_.onerror = handleError;
  link.onerror = handleError;
  if (!jstestdriver.jQuery.browser.opera) {
    link.onload = jstestdriver.bind(this, function() {
      this.onLoad_(file, callback);
    });
  }
  link.onreadystatechange = jstestdriver.bind(this, function() {
    if (link.readyState == 'loaded') {
      this.onLoad_(file, callback);
    }
  });
  link.type = "text/css";
  link.rel = "stylesheet";
  link.href = file.fileSrc;
  head.appendChild(link);

  // Firefox and Safari don't seem to support onload or onreadystatechange for link
  if (this.synchronousCallback_) {
    this.onLoad_(file, callback);
  }
};


jstestdriver.plugins.StylesheetLoader.prototype.onLoad_ = function(file, callback) {
  this.updateResult_(new jstestdriver.FileResult(file, true, ''));
  this.win_.onerror = jstestdriver.EMPTY_FUNC;
  callback(this.fileResult_);  
};


jstestdriver.plugins.StylesheetLoader.prototype.updateResult_ = function(fileResult) {
  if (this.fileResult_ == null) {
    this.fileResult_ = fileResult;
  }
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.plugins.FileLoaderPlugin = function(scriptLoader, stylesheetLoader) {
  this.scriptLoader_ = scriptLoader;
  this.stylesheetLoader_ = stylesheetLoader;
};


jstestdriver.plugins.FileLoaderPlugin.prototype.loadSource = function(file, onSourceLoaded) {
  if (file.fileSrc.match(/\.css$/)) {
    this.stylesheetLoader_.load(file, onSourceLoaded);
  } else {
    this.scriptLoader_.load(file, onSourceLoaded);
  }
};
/*
 * Copyright 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @param dateObj
 * @param clearBody
 * @param opt_runTestLoop
 * @constructor
 */
jstestdriver.plugins.TestRunnerPlugin = function(dateObj, clearBody, opt_runTestLoop) {
  this.dateObj_ = dateObj;
  this.clearBody_ = clearBody;
  this.boundRunTest_ = jstestdriver.bind(this, this.runTest);
  this.runTestLoop_ = opt_runTestLoop || jstestdriver.plugins.defaultRunTestLoop;
};


jstestdriver.plugins.timedProcessArray = function(interval, array, process, finish, now, setTimeout)
{
  var items = array.concat(); //clone the array
  setTimeout(function nestedFunction(){
    var start = now();
    do{
      process(items.shift());
    }while(items.length > 0 && (now() - start < interval));

    if (items.length > 0){
      setTimeout(nestedFunction, 25);
    }else{
      finish();
    }
  }, 25);
};


jstestdriver.plugins.createPausingRunTestLoop =
    function (interval, now, setTimeout) {
  var lastPause;
  function pausingRunTestLoop(testCaseName,
                                template,
                                tests,
                                runTest,
                                onTest,
                                onComplete) {
      jstestdriver.plugins.timedProcessArray(interval, tests, function(oItem){
      onTest(runTest(testCaseName, template, oItem));
    }, onComplete, now, setTimeout);
  }
  return pausingRunTestLoop;
};


jstestdriver.plugins.pausingRunTestLoop =
    jstestdriver.plugins.createPausingRunTestLoop(
        50,
        jstestdriver.now,
        jstestdriver.setTimeout);


jstestdriver.plugins.defaultRunTestLoop =
    function(testCaseName, template, tests, runTest, onTest, onComplete) {
  for (var i = 0; tests[i]; i++) {
    onTest(runTest(testCaseName, template, tests[i]));
  }
  onComplete();
};


jstestdriver.plugins.TestRunnerPlugin.prototype.runTestConfiguration =
    function(testRunConfiguration, onTestDone, onTestRunConfigurationComplete) {
  var testCaseInfo = testRunConfiguration.getTestCaseInfo();
  var tests = testRunConfiguration.getTests();
  var size = tests.length;

  if (testCaseInfo.getType() != jstestdriver.TestCaseInfo.DEFAULT_TYPE) {
    for (var i = 0; tests[i]; i++) {
      onTestDone(new jstestdriver.TestResult(
          testCaseInfo.getTestCaseName(),
          tests[i],
          'error',
          testCaseInfo.getTestCaseName() +
            ' is an unhandled test case: ' +
            testCaseInfo.getType(),
          '',
          0));
    }
    onTestRunConfigurationComplete();
    return;
  }

  this.runTestLoop_(testCaseInfo.getTestCaseName(),
                    testCaseInfo.getTemplate(),
                    tests,
                    this.boundRunTest_,
                    onTestDone,
                    onTestRunConfigurationComplete)
};


jstestdriver.plugins.TestRunnerPlugin.prototype.runTest =
    function(testCaseName, testCase, testName) {
  var testCaseInstance;
  var errors = [];
  try {
    try {
      testCaseInstance = new testCase();
    } catch (e) {
      return new jstestdriver.TestResult(
          testCaseName,
          testName,
          jstestdriver.TestResult.RESULT.ERROR,
          testCaseName + ' is not a test case',
          '',
          0);
    }
    var start = new this.dateObj_().getTime();

    jstestdriver.expectedAssertCount = -1;
    jstestdriver.assertCount = 0;
    var res = jstestdriver.TestResult.RESULT.PASSED;
    try {
      if (testCaseInstance.setUp) {
        testCaseInstance.setUp();
      }
      if (!(testName in testCaseInstance)) {
        var err = new Error(testName + ' not found in ' + testCaseName);
        err.name = 'AssertError';
        throw err;
      }
      testCaseInstance[testName]();
      if (jstestdriver.expectedAssertCount != -1 &&
          jstestdriver.expectedAssertCount != jstestdriver.assertCount) {
        var err = new Error("Expected '" +
            jstestdriver.expectedAssertCount +
            "' asserts but '" +
            jstestdriver.assertCount +
            "' encountered.");

        err.name = 'AssertError';
        throw err;
      }
    } catch (e) {
      // We use the global here because of a circular dependency. The isFailure plugin should be
      // refactored.
      res = jstestdriver.pluginRegistrar.isFailure(e) ?
          jstestdriver.TestResult.RESULT.FAILED :
            jstestdriver.TestResult.RESULT.ERROR;
      errors.push(e);
    }
    try {
      if (testCaseInstance.tearDown) {
        testCaseInstance.tearDown();
      }
      this.clearBody_();
    } catch (e) {
      if (res == jstestdriver.TestResult.RESULT.PASSED) {
        res = jstestdriver.TestResult.RESULT.ERROR;
      }
      errors.push(e);
    }
    var end = new this.dateObj_().getTime();
    var msg = this.serializeError(errors);
    return new jstestdriver.TestResult(testCaseName, testName, res, msg,
            jstestdriver.console.getAndResetLog(), end - start);
  } catch (e) {
    errors.push(e);
    return new jstestdriver.TestResult(testCaseName, testName,
            'error', 'Unexpected runner error: ' + this.serializeError(errors),
            jstestdriver.console.getAndResetLog(), 0);
  }
};

/**
 *@param {Error} e
 */
jstestdriver.plugins.TestRunnerPlugin.prototype.serializeError = function(e) {
  return jstestdriver.utils.serializeErrors(e);
};
/*
 * Copyright 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview Defines the AsyncTestRunnerPlugin class, which executes
 * asynchronous test cases within JsTestDriver.
 *
 *     +----------------------------- more tests? ------------ nextTest() <--------------+
 *     |                                                                                 |
 *     v                                                                                 |
 * startSetUp() ---- execute ---> finishSetUp(errors)                                    |
 *                                     |                                                 |
 * startTestMethod() <--- no errors ---+---- errors ----+                                |
 *        |                                             |                                |
 *     execute                                          |                                |
 *        |                                             |                                |
 *        v                                             v                                |
 * finishTestMethod(errors) -- errors or no errors -> startTearDown() -- execute -> finishTearDown(errors)
 *
 * @author rdionne@google.com (Robert Dionne)
 */

goog.provide('jstestdriver.plugins.async.AsyncTestRunnerPlugin');

goog.require('jstestdriver');
goog.require('jstestdriver.setTimeout');
goog.require('jstestdriver.TestCaseInfo');
goog.require('jstestdriver.TestResult');
goog.require('jstestdriver.plugins.async.CallbackPool');
goog.require('jstestdriver.plugins.async.CallbackPoolDelegate');
goog.require('jstestdriver.plugins.async.DeferredQueue');
goog.require('jstestdriver.plugins.async.DeferredQueueDelegate');
goog.require('jstestdriver.plugins.async.TestStage');
goog.require('jstestdriver.plugins.async.TestStage.Builder');

/**
 * Constructs an AsyncTestRunnerPlugin.
 *
 * @param {Function} dateObj the date object constructor
 * @param {Function} clearBody a function to call to clear the document body.
 * @param {Function} toJson a function to call to convert an object to JSON.
 * @param {boolean} opt_pauseForHuman Whether to pause for debugging.
 * @param {Function} opt_setTimeout window.setTimeout replacement.
 * @param {Function} opt_queueConstructor a constructor for obtaining new
 *     DeferredQueues.
 * @param {Function} opt_queueDelegateConstructor a constructor for obtaining new
 *     DeferredQueueDelegates.
 * @constructor
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin = function(dateObj, clearBody,
      toJson, opt_pauseForHuman, opt_setTimeout, opt_queueConstructor,
      opt_queueDelegateConstructor) {
  this.name = "AsyncTestRunnerPlugin";
  this.dateObj_ = dateObj;
  this.clearBody_ = clearBody;
  this.toJson_ = toJson;
  this.pauseForHuman_ = !!opt_pauseForHuman;
  this.setTimeout_ = opt_setTimeout || jstestdriver.setTimeout;
  this.queueConstructor_ = opt_queueConstructor || jstestdriver.plugins.async.DeferredQueue;
  this.queueDelegateConstructor_ = opt_queueDelegateConstructor ||
      jstestdriver.plugins.async.DeferredQueueDelegate;
  this.testRunConfiguration_ = null;
  this.testCaseInfo_ = null;
  this.onTestDone_ = null;
  this.onTestRunConfigurationComplete_ = null;
  this.testIndex_ = 0;
  this.testCase_ = null;
  this.testName_ = null;
  this.start_ = null;
  this.errors_ = null;
};

/**
 * Runs a test case.
 *
 * @param {jstestdriver.TestRunConfiguration} testRunConfiguration the test 
 *        case configuration
 * @param {function(jstestdriver.TestResult)} onTestDone the function to call to 
 *        report a test is complete
 * @param {function()=} opt_onTestRunConfigurationComplete the function to call 
 *        to report a test case is complete. A no-op will be used if this is
 *        not specified.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.runTestConfiguration = function(
    testRunConfiguration, onTestDone, opt_onTestRunConfigurationComplete) {
  if (testRunConfiguration.getTestCaseInfo().getType() == jstestdriver.TestCaseInfo.ASYNC_TYPE) {
    this.testRunConfiguration_ = testRunConfiguration;
    this.testCaseInfo_ = testRunConfiguration.getTestCaseInfo();
    this.onTestDone_ = onTestDone;
    this.onTestRunConfigurationComplete_ = opt_onTestRunConfigurationComplete ||
        function() {};
    this.testIndex_ = 0;
    this.nextTest();
    return true;
  }

  return false;
};

/**
 * Runs the next test in the current test case.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.nextTest = function() {
  this.start_ = new this.dateObj_().getTime();
  if (this.testIndex_ < this.testRunConfiguration_.getTests().length) {
    jstestdriver.expectedAssertCount = -1;
    jstestdriver.assertCount = 0;
    this.testCase_ = new (this.testCaseInfo_.getTemplate());
    this.testName_ = this.testRunConfiguration_.getTests()[this.testIndex_];
    this.errors_ = [];
    this.startSetUp();
  } else {
    this.testRunConfiguration_ = null;
    this.testCaseInfo_ = null;
    this.onTestDone_ = null;
    this.testIndex_ = 0;
    this.testCase_ = null;
    this.testName_ = null;
    this.start_ = null;
    this.errors_ = null;

    // Unset this callback before running it because the next callback may be
    // set by the code run by the callback.
    var onTestRunConfigurationComplete = this.onTestRunConfigurationComplete_;
    this.onTestRunConfigurationComplete_ = null;
    onTestRunConfigurationComplete.call(this);
  }
};


/**
 * Starts the next phase of the current test in the current test case. Creates a
 * DeferredQueue to manage the steps of this phase, executes the phase
 * catching any exceptions, and then hands the control over to the queue to
 * call onQueueComplete when it empties.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.execute_ = function(
    onStageComplete, invokeMethod) {
  var runner = this;
  var onError = function(error) {runner.errors_.push(error);};
  var arguments = this.testRunConfiguration_.getArguments();
  var argument = arguments ? arguments[this.testName_] : null;
  var stage = new jstestdriver.plugins.async.TestStage.Builder().
      setOnError(onError).
      setOnStageComplete(onStageComplete).
      setTestCase(this.testCase_).
      setTestMethod(invokeMethod).
      setArgument(argument).
      setPauseForHuman(this.pauseForHuman_).
      setQueueDelegateConstructor(this.queueDelegateConstructor_).
      setQueueConstructor(this.queueConstructor_).
      setTimeoutSetter(this.setTimeout_).
      setToJson(this.toJson_).
      build();
  stage.execute();
};


/**
 * Starts the setUp phase.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.startSetUp = function() {
  var runner = this;
  this.execute_(function(errors) {
    runner.finishSetUp(errors);
  }, this.testCase_['setUp']);
};

/**
 * Finishes the setUp phase and reports any errors. If there are errors it
 * initiates the tearDown phase, otherwise initiates the testMethod phase.
 *
 * @param errors errors caught during the current asynchronous phase.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.finishSetUp = function(errors) {
  this.errors_ = this.errors_.concat(errors);
  if (this.errors_.length) {
    this.startTearDown();
  } else {
    this.startTestMethod();
  }
};

/**
 * Starts the testMethod phase.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.startTestMethod = function() {
  var runner = this;
  this.execute_(function(errors) {
    runner.finishTestMethod(errors);
  }, this.testCase_[this.testName_]);
};

/**
 * Finishes the testMethod phase and reports any errors. Continues with the
 * tearDown phase.
 *
 * @param errors errors caught during the current asynchronous phase.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.finishTestMethod = function(errors) {
  this.errors_ = this.errors_.concat(errors);
  this.startTearDown();
};


/**
 * Start the tearDown phase.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.startTearDown = function() {
  var runner = this;
  this.execute_(function(errors){
    runner.finishTearDown(errors);
  }, this.testCase_['tearDown']);
};


/**
 * Finishes the tearDown phase and reports any errors. Submits the test results
 * to the test runner. Continues with the next test.
 *
 * @param errors errors caught during the current asynchronous phase.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.finishTearDown = function(errors) {
  this.errors_ = this.errors_.concat(errors);
  this.clearBody_();
  this.onTestDone_(this.buildResult());
  this.testIndex_ += 1;
  this.nextTest();
};

/**
 * Builds a test result.
 */
jstestdriver.plugins.async.AsyncTestRunnerPlugin.prototype.buildResult = function() {
  var end = new this.dateObj_().getTime();
  var result = jstestdriver.TestResult.RESULT.PASSED;
  var message = '';
  if (this.errors_.length) {
    result = jstestdriver.TestResult.RESULT.FAILED;
    message = this.toJson_(this.errors_);
  } else if (jstestdriver.expectedAssertCount != -1 &&
             jstestdriver.expectedAssertCount != jstestdriver.assertCount) {
    result = jstestdriver.TestResult.RESULT.FAILED;
    message = this.toJson_([new Error("Expected '" +
        jstestdriver.expectedAssertCount +
        "' asserts but '" +
        jstestdriver.assertCount +
        "' encountered.")]);
  }
  var arguments = this.testRunConfiguration_.getArguments();
  var argument = arguments ? arguments[this.testName_] : null;
  return new jstestdriver.TestResult(
      this.testCaseInfo_.getTestCaseName(), this.testName_, result, message,
      jstestdriver.console.getAndResetLog(), end - this.start_, null, argument);
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.plugins.DefaultPlugin = function(fileLoaderPlugin,
                                              testRunnerPlugin,
                                              assertsPlugin,
                                              testCaseManagerPlugin) {
  this.fileLoaderPlugin_ = fileLoaderPlugin;
  this.testRunnerPlugin_ = testRunnerPlugin;
  this.assertsPlugin_ = assertsPlugin;
  this.testCaseManagerPlugin_ = testCaseManagerPlugin;
};


jstestdriver.plugins.DefaultPlugin.prototype.name = 'defaultPlugin';


jstestdriver.plugins.DefaultPlugin.prototype.loadSource = function(file, onSourceLoaded) {
  return this.fileLoaderPlugin_.loadSource(file, onSourceLoaded);
};


jstestdriver.plugins.DefaultPlugin.prototype.runTestConfiguration = function(testRunConfiguration,
    onTestDone, onTestRunConfigurationComplete) {
  return this.testRunnerPlugin_.runTestConfiguration(testRunConfiguration, onTestDone,
      onTestRunConfigurationComplete);
};


jstestdriver.plugins.DefaultPlugin.prototype.isFailure = function(exception) {
  return this.assertsPlugin_.isFailure(exception);
};


jstestdriver.plugins.DefaultPlugin.prototype.getTestRunsConfigurationFor =
    function(testCaseInfos, expressions, testRunsConfiguration) {
  return this.testCaseManagerPlugin_.getTestRunsConfigurationFor(testCaseInfos,
                                                                expressions,
                                                                testRunsConfiguration);
};


jstestdriver.plugins.DefaultPlugin.prototype.onTestsStart =
    jstestdriver.EMPTY_FUNC;


jstestdriver.plugins.DefaultPlugin.prototype.onTestsFinish =
  jstestdriver.EMPTY_FUNC;
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.plugins.AssertsPlugin = function() {
};


jstestdriver.plugins.AssertsPlugin.prototype.isFailure = function(e) {
  return e.name == 'AssertError';
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * Plugin that handles the default behavior for the TestCaseManager.
 * @author corysmith@google.com (Cory Smith)
 */
jstestdriver.plugins.TestCaseManagerPlugin = function() {};


/**
 * Write testRunconfigurations retrieved from testCaseInfos defined by expressions.
 * @param {Array.<jstestdriver.TestCaseInfo>} testCaseInfos The loaded test case infos.
 * @param {Array.<String>} The expressions that define the TestRunConfigurations
 * @parma {Array.<jstestdriver.TestRunConfiguration>} The resultant array of configurations.
 */
jstestdriver.plugins.TestCaseManagerPlugin.prototype.getTestRunsConfigurationFor =
    function(testCaseInfos, expressions, testRunsConfiguration) {
  var size = testCaseInfos.length;
  for (var i = 0; i < size; i++) {
    var testCaseInfo = testCaseInfos[i];
    var testRunConfiguration = testCaseInfo.getTestRunConfigurationFor(expressions);

    if (testRunConfiguration != null) {
      testRunsConfiguration.push(testRunConfiguration);
    }
  }
  return true;
};
