/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


/**
 * @fileoverview
 * Provides the namespaces and necessary function to enable migration to the
 * Google JsCompiler.
 *
 * @author Cory Smith (corbinrsmith@gmail.com)
 */

var jstestdriver = {};
jstestdriver.plugins = {};
jstestdriver.plugins.async = {};

var goog = window.goog || {
  provide : function(symbol){},
  require : function(symbol){}
};

/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


jstestdriver.convertToJson = function(delegate) {
  var serialize = jstestdriver.parameterSerialize
  return function(url, data, callback, type) {
    delegate(url, serialize(data), callback, type);
  };
};


jstestdriver.parameterSerialize = function(data) {
  var modifiedData = {};
  for (var key in data) {
    modifiedData[key] = JSON.stringify(data[key]);
  }
  return modifiedData;
};


jstestdriver.bind = function(context, func) {
  function bound() {
    return func.apply(context, arguments);
  };
  bound.toString = function() {
    return "bound: " + context + " to: " + func;
  }
  return bound;
};


jstestdriver.extractId = function(url) {
  return url.match(/\/id\/(\d+)\//)[1];
};


jstestdriver.createPath = function(basePath, path) {
  var prefix = basePath.match(/^(.*)\/(slave|runner|bcr)\//)[1];
  return prefix + path;
};


jstestdriver.getBrowserFriendlyName = function() {
  if (jstestdriver.jQuery.browser.safari) {
    if (navigator.userAgent.indexOf('Chrome') != -1) {
      return 'Chrome';
    }
    return 'Safari';
  } else if (jstestdriver.jQuery.browser.opera) {
    return 'Opera';
  } else if (jstestdriver.jQuery.browser.msie) {
    return 'Internet Explorer';
  } else if (jstestdriver.jQuery.browser.mozilla) {
    if (navigator.userAgent.indexOf('Firefox') != -1) {
      return 'Firefox';
    }
    return 'Mozilla';
  }
};


jstestdriver.getBrowserFriendlyVersion = function() {
  if (jstestdriver.jQuery.browser.msie) {
    if (typeof XDomainRequest != 'undefined') {
      return '8.0';
    } 
  } else if (jstestdriver.jQuery.browser.safari) {
    if (navigator.appVersion.indexOf('Chrome/') != -1) {
      return navigator.appVersion.match(/Chrome\/(.*)\s/)[1];
    }
  }
  return jstestdriver.jQuery.browser.version;
};

jstestdriver.trim = function(str) {
  return str.replace(/(^\s*)|(\s*$)/g,'');
};


/**
 * Renders an html string as a dom nodes.
 * @param {string} htmlString The string to be rendered as html.
 * @param {Document} owningDocument The window that should own the html.
 */
jstestdriver.toHtml = function(htmlString, owningDocument) {
  var fragment = owningDocument.createDocumentFragment();
  var wrapper = owningDocument.createElement('div');
  wrapper.innerHTML = jstestdriver.trim(jstestdriver.stripHtmlComments(htmlString));
  while(wrapper.firstChild) {
    fragment.appendChild(wrapper.firstChild);
  }
  var ret =  fragment.childNodes.length > 1 ? fragment : fragment.firstChild;
  return ret;
};


jstestdriver.stripHtmlComments = function(htmlString) {
  var stripped = [];
  function getCommentIndices(offset) {
    var start = htmlString.indexOf('<!--', offset);
    var stop = htmlString.indexOf('-->', offset) + '-->'.length;
    if (start == -1) {
      return null;
    }
    return {
      'start' : start,
      'stop' : stop
    };
  }
  var offset = 0;
  while(true) {
    var comment = getCommentIndices(offset);
    if (!comment) {
      stripped.push(htmlString.slice(offset));
      break;
    }
    var frag = htmlString.slice(offset, comment.start);
    stripped.push(frag);
    offset = comment.stop;
  }
  return stripped.join('');
}


/**
 * Appends html string to the body.
 * @param {string} htmlString The string to be rendered as html.
 * @param {Document} owningDocument The window that should own the html.
 */
jstestdriver.appendHtml = function(htmlString, owningDocument) {
  var node = jstestdriver.toHtml(htmlString, owningDocument);
  jstestdriver.jQuery(owningDocument.body).append(node);
};


/**
 * @return {Number} The ms since the epoch.
 */
jstestdriver.now = function() { return new Date().getTime();}


/**
 * Creates a wrapper for jQuery.ajax that make a synchronous post
 * @param {jQuery} jQuery
 * @return {function(url, data):null}
 */
jstestdriver.createSynchPost = function(jQuery) {
  return jstestdriver.convertToJson(function(url, data) {
    return jQuery.ajax({
      'async' : false,
      'data' : data,
      'type' : 'POST',
      'url' : url
    });
  });
};

jstestdriver.utils = {};

/**
 * Checks to see if an object is a a certain native type.
 * @param instance An instance to check.
 * @param nativeType A string of the type expected.
 * @returns True if of that type.
 */
jstestdriver.utils.isNative = function(instance, nativeType) {
  try {
    var typeString = String(Object.prototype.toString.apply(instance));
    return typeString.toLowerCase().indexOf(nativeType.toLowerCase()) != -1;
  } catch (e) {
    return false;
  }
};

jstestdriver.utils.serializeErrors = function(errors) {
  var out = [];
  out.push('[');
  for (var i = 0; i < errors.length; ++i) {
    jstestdriver.utils.serializeErrorToArray(errors[i], out);
    if (i < errors.length - 1) {
      out.push(',');
    }
  }
  out.push(']');
  return out.join('');
};

jstestdriver.utils.serializeErrorToArray = function(error, out) {
  if (jstestdriver.utils.isNative(error, 'Error')) {
    out.push('{');
    out.push('"message":');
    this.serializeObjectToArray(error.message, out);
    this.serializePropertyOnObject('name', error, out);
    this.serializePropertyOnObject('description', error, out);
    this.serializePropertyOnObject('fileName', error, out);
    this.serializePropertyOnObject('lineNumber', error, out);
    this.serializePropertyOnObject('number', error, out);
    this.serializePropertyOnObject('stack', error, out);
    out.push('}');
  } else {
    out.push(jstestdriver.utils.serializeObject(error));
  }
};

jstestdriver.utils.serializeObject = function(obj) {
  var out = [];
  jstestdriver.utils.serializeObjectToArray(obj, out);
  return out.join('');
};


jstestdriver.utils.serializeObjectToArray =
   function(obj, opt_out){
  var out = opt_out || out;
  if (jstestdriver.utils.isNative(obj, 'Array')) {
    out.push('[');
    var arr = /** @type {Array.<Object>} */ obj;
    for ( var i = 0; i < arr.length; i++) {
      this.serializeObjectToArray(arr[i], out);
      if (i < arr.length - 1) {
        out.push(',');
      }
    }
    out.push(']');
  } else {
    var serial = jstestdriver.angular.toJson(obj);
    if (!serial.length) {
      serial = '["Bad serialization of ' + String(obj) + ':' +
          Object.prototype.toString.call(obj) + '"]';
    }
    out.push(serial);
  }
  return out;
};


jstestdriver.utils.serializePropertyOnObject = function(name, obj, out) {
  if (name in obj) {
    out.push(',');
    out.push('"' + name + '":');
    this.serializeObjectToArray(obj[name], out);
  }
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

goog.provide('jstestdriver.Console');


jstestdriver.JSON = window['JSON'] || {};
jstestdriver.JSON.stringify = window['JSON'] ? window['JSON']['stringify'] :
    function(msg, opt_args) {};


// TODO(corysmith): Separate this into a Utils namespace that can loaded earlier.
jstestdriver.FORMAT_MAPPINGS = {
  's' : function(arg) {
    if (arg == undefined) {
      return '';
    }
    return String(arg);
  },
  'd' : Number,
  'i' : parseInt,
  'f' : parseFloat,
  'o' : jstestdriver.JSON.stringify
};


jstestdriver.formatString = function(str) {
  var formatArgs = arguments;
  var idx = 1;
  var formatted = String(str).replace(/%([sdifo])/g,
      function(fullmatch, groupOne) {
    var value = formatArgs[idx++];
    if (!jstestdriver.FORMAT_MAPPINGS[groupOne]) {
      throw new Error(groupOne + 'is not a proper format.');
    }
    if (value === undefined || value === null) {
      return value;
    }
    return jstestdriver.FORMAT_MAPPINGS[groupOne](value);
  })
  while (idx < formatArgs.length) {
    var currentArg = formatArgs[idx++]
    if (typeof currentArg == 'object') {
      currentArg = jstestdriver.JSON.stringify(currentArg);
    }
    formatted += " " + currentArg;
  }
  return formatted;
};


/**
 * @constructor
 */
jstestdriver.Console = function() {
  this.log_ = [];
};


jstestdriver.Console.prototype.log = function() {
  this.logStatement('[LOG]', jstestdriver.formatString.apply(this, arguments));
};


jstestdriver.Console.prototype.debug = function() {
  this.logStatement('[DEBUG]', jstestdriver.formatString.apply(this, arguments));
};


jstestdriver.Console.prototype.info = function() {
  this.logStatement('[INFO]', jstestdriver.formatString.apply(this, arguments));
};


jstestdriver.Console.prototype.warn = function() {
  this.logStatement('[WARN]', jstestdriver.formatString.apply(this, arguments));
};


jstestdriver.Console.prototype.error = function() {
  this.logStatement('[ERROR]', jstestdriver.formatString.apply(this, arguments));
};


jstestdriver.Console.prototype.logStatement = function(level, statement) {
  this.log_.push(level + ' ' + statement);
};


jstestdriver.Console.prototype.getLog = function() {
  var log = this.log_;
  return log.join('\n');
};


jstestdriver.Console.prototype.getAndResetLog = function() {
  var log = this.getLog();
  this.log_ = [];
  return log;
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * @fileoverview
 * Provides the base jstestdriver environment constants and functions.
 */

goog.provide('jstestdriver');
goog.provide('jstestdriver.setTimeout');

goog.require('jstestdriver.Console');
goog.require('jstestdriver.runConfig');


jstestdriver.global = this;
jstestdriver.console = null;
jstestdriver.console = new jstestdriver.Console();

jstestdriver.SERVER_URL = "/query/";

jstestdriver.HEARTBEAT_URL = "/heartbeat";

if (!window['console']) window['console'] = {};
if (typeof window['console']['log'] == 'undefined') window['console']['log'] = function(msg) {};
if (typeof window['console']['debug'] == 'undefined') window['console']['debug'] = function(msg) {};
if (typeof ['console']['info'] == 'undefined') window['console']['info'] = function(msg) {};
if (typeof ['console']['warn'] == 'undefined') window['console']['warn'] = function(msg) {};
if (typeof ['console']['error'] == 'undefined') window['console']['error'] = function(msg) {};

jstestdriver.globalSetTimeout = window.setTimeout;
jstestdriver.setTimeout = function() {
  if (jstestdriver.globalSetTimeout.apply) {
    return jstestdriver.globalSetTimeout.apply(window, arguments);
  }
  return jstestdriver.globalSetTimeout(arguments[0], arguments[1]);
};

jstestdriver.globalClearTimeout = clearTimeout;
jstestdriver.clearTimeout = function() {
  if (jstestdriver.globalClearTimeout.apply) {
    return jstestdriver.globalClearTimeout.apply(window, arguments);
  }
  return jstestdriver.globalClearTimeout(arguments[0]);
};

jstestdriver.globalSetInterval = setInterval;
jstestdriver.setInterval = function() {
  if (jstestdriver.globalSetInterval.apply) {
    return jstestdriver.globalSetInterval.apply(window, arguments);
  }
  return jstestdriver.globalSetInterval(arguments[0], arguments[1]);
};


jstestdriver.globalClearInterval = clearInterval;
jstestdriver.clearInterval = function() {
  if (jstestdriver.globalClearInterval.apply) {
    return jstestdriver.globalClearInterval.apply(window, arguments);
  }
  return jstestdriver.globalClearInterval(arguments[0]);
};


jstestdriver.browserLogger = {
  log : function(src, lvl, msg) {},
  debug : function(src, msg) {},
  info : function(src, msg) {},
  warn : function(src, msg) {},
  error : function(src, msg) {}
};


jstestdriver.log = function(message) {
  if (jstestdriver.runConfig && jstestdriver.runConfig.debug) {
    jstestdriver.browserLogger.debug('log', message);
  }
}

document.write = function(str) {
  //jstestdriver.console.error('Illegal call to document.write.');
};


var noop = jstestdriver.EMPTY_FUNC = function() {};

// TODO(corysmith): We need to be able to log early for debugging,
// but this really doesn't belong here. Need to reorg the js.
/**
 * A log message.
 * Corresponds with the com.google.jstestdriver.protocol.BrowserLog.
 * @param {String} source
 * @param {number} level
 * @param {String} message
 * @param {Object} browser The browser info object.
 * @param {String} stack Stack of where this was logged.
 * @param {String} timestamp
 * @constructor
 */
jstestdriver.BrowserLog = function(source, level, message, browser, stack, timestamp) {
  this.source = source;
  this.level = level;
  this.message = message;
  this.browser = browser
  this.stack = stack;
  this.timestamp = timestamp;
};
/*
 * Copyright 2010 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

/**
 * This is a simple logger implementation, that posts messages to the server.
 * Will most likely be expanded later.
 * @param {Function} sendToServer Function to send information to the server,
 *     of signature function(logs):void
 * @param {number} 
 */
jstestdriver.BrowserLogger = function(sendToServer, id) {
  this.sendToServer_ = sendToServer;
  this.id_ = id;
};


/**
 * 
 * @param location String location of the browser.
 * @param ajax jQuery ajax function.
 * @returns jstestdriver.BrowserLogger.
 */
jstestdriver.BrowserLogger.create = function(location, ajax) {
  var id = parseInt(jstestdriver.extractId(location));
  var prefix = location.match(/^(.*)\/(slave|runner|bcr)\//)[1];
  var url = prefix + '/log';
  return new jstestdriver.BrowserLogger(function(logs) {
    ajax({
        'async' : true,
        'data' : 'logs=' + JSON.stringify(logs),
        'type' : 'POST',
        'url' : url
    });
  }, id);
};

jstestdriver.BrowserLogger.prototype.isEnabled_ = function() {
  // TODO(corysmith): Refactor to allow the runConfig to be available before
  // load.
  var enabled = jstestdriver.runConfig && jstestdriver.runConfig.debug;
  this.isEnabled_ = function () {
    return enabled;
  };
  return enabled;
};


/**
 * Logs a message to the server.
 * @param {String} source The source of the log event.
 * @param {jstestdriver.BrowserLogger.LEVEL} level The level of the message.
 * @param {String} message The log message.
 */
jstestdriver.BrowserLogger.prototype.log = function(source, level, message) {
  if (this.isEnabled_()) {
    // TODO(corysmith): replace with a cross browser stack methodology.
    var traceError = new Error();
    var stack = traceError.stack ? traceError.stack.split('\n') : [];

    var smallStack = [];

    for (var i = 0; stack[i]; i++) {
      var end = stack[i].indexOf('(');
      if (end > -1) {
        smallStack.push(stack[i].substr(0,end).trim());
      }
    }
    smallStack = smallStack.length ? smallStack : ['No stack available'];
    this.sendToServer_([
          new jstestdriver.BrowserLog(
              source,
              level,
              encodeURI(message),
              {"id": this.id_},
              encodeURI(smallStack.toString()),
              new Date())
        ]);
  }
};


jstestdriver.BrowserLogger.prototype.debug = function(source, message) {
  this.log(source, jstestdriver.BrowserLogger.LEVEL.DEBUG, message);
};


jstestdriver.BrowserLogger.prototype.info = function(source, message) {
  this.log(source, jstestdriver.BrowserLogger.LEVEL.INFO, message);
};


jstestdriver.BrowserLogger.prototype.warn = function(source, message) {
  this.log(source, jstestdriver.BrowserLogger.LEVEL.WARN, message);
};


jstestdriver.BrowserLogger.prototype.error = function(source, message) {
  this.log(source, jstestdriver.BrowserLogger.LEVEL.ERROR, message);
};


/**
 * Acceptable logging levels.
 * @enum
 */
jstestdriver.BrowserLogger.LEVEL = {
  TRACE : 1,
  DEBUG : 2,
  INFO : 3,
  WARN : 4,
  ERROR : 5
};

jstestdriver.browserLogger = jstestdriver.BrowserLogger.create(window.top.location.toString(),
    function() {jstestdriver.jQuery.ajax.apply(jstestdriver.jQuery,
        arguments)});
