# -*- coding: utf-8 -*-
import arrow
# Twitter lib
from twython import Twython

# django classes
from django.conf import settings
from django.core.urlresolvers import reverse
from django.utils.log import getLogger

# django_th classes
from django_th.services.services import ServicesMgr
from django_th.models import UserService, ServicesActivated

from th_twitter.models import Twitter
"""
    handle process with twitter
    put the following in settings.py

    TH_TWITTER = {
        'consumer_key': 'abcdefghijklmnopqrstuvwxyz',
        'consumer_secret': 'abcdefghijklmnopqrstuvwxyz',
    }

"""

logger = getLogger('django_th.trigger_happy')


class ServiceTwitter(ServicesMgr):

    def __init__(self):
        self.consumer_key = settings.TH_TWITTER['consumer_key']
        self.consumer_secret = settings.TH_TWITTER['consumer_secret']

    def process_data(self, token, trigger_id, date_triggered):
        """
            get the data from the service

            :param trigger_id: trigger ID to process
            :param date_triggered: the date of the last trigger
            :type trigger_id: int
            :type date_triggered: datetime
            :return: list of data found from the date_triggered filter - data has to return {'content': 'a text', 'link': 'a valid url to the ressource'}
            :rtype: list
        """
        twitter_url = 'https://www.twitter.com/{}/status/{}'
        data = []
        search = {}
        since_id = None

        def _get_tweets(twitter_obj, search):
            """
                get the tweets from twitter and return the filters to use : search and count

                :param twitter_obj: from Twitter model
                :param search: filter used for twython.search() or twython.get_user_timeline())
                :type twitter_obj: Object
                :type search: dict
                :return: count that limit the quantity of tweet to retrieve, the filter named search, the tweets
                :rtype: list
            """
            #get the tweets for a given tag
            if twitter_obj.tag != '':
                count = 100
                search['count'] = count
                search['q'] = twitter_obj.tag
                search['result_type'] = 'recent'
                # do a search
                statuses = twitter_api.search(**search)

            #get the tweets from a given user
            elif twitter_obj.screen != '':
                count = 200
                search['count'] = count
                search['screen_name'] = twitter_obj.screen
                # call the user timeline and get his tweet
                statuses = twitter_api.get_user_timeline(**search)

            return count, search, statuses

        if token is not None:
            token_key, token_secret = token.split('#TH#')

            twitter_obj = Twitter.objects.get(trigger_id=trigger_id)

            # open a connection with twitter
            twitter_api = Twython(self.consumer_key, self.consumer_secret,
                                  token_key, token_secret)

            # https://dev.twitter.com/rest/public/timelines
            if twitter_obj.since_id is not None and twitter_obj.since_id > 0:
                since_id = twitter_obj.since_id
                search = {'since_id': twitter_obj.since_id}

            # first request to Twitter
            count, search, statuses = _get_tweets(twitter_obj, search)

            if len(statuses) > 0:
                newest = None
                for status in statuses:
                    if newest is None:
                        newest = True
                        # first query ; get the max id
                        search['max_id'] = max_id = status['id']

                since_id = search['since_id'] = statuses[-1]['id'] - 1

                count, search, statuses = _get_tweets(twitter_obj, search)

                newest = None
                if len(statuses) > 0:
                    for s in statuses:
                        if newest is None:
                            newest = True
                            max_id = s['id'] - 1
                        # get the text of the tweet + url to this one
                        url = twitter_url.format(s['user']['screen_name'],
                                                 s['id_str'])
                        # Wed Aug 29 17:12:58 +0000 2012
                        my_date = arrow.get(s['created_at'], 'ddd MMM DD HH:mm:ss Z YYYY')
                        data.append({'content': s['text'], 'link': url, 'my_date': my_date})
                    # update the Twitter model to store max_id since_id, count
                    Twitter.objects.filter(trigger_id=trigger_id).update(
                        since_id=since_id,
                        max_id=max_id,
                        count=count)

        return data

    def save_data(self, token, trigger_id, **data):
        """
            let's save the data

            :param trigger_id: trigger ID from which to save data
            :param **data: the data to check to be used and save
            :type trigger_id: int
            :type **data:  dict
            :return: the status of the save statement
            :rtype: boolean
        """
        status = False
        tags = []
        if token and 'link' in data and data['link'] is not None and \
           len(data['link']) > 0:
            # get the Twitter data of this trigger
            trigger = Twitter.objects.get(trigger_id=trigger_id)
            token_key, token_secret = token.split('#TH#')

            twitter = Twython(self.consumer_key, self.consumer_secret,
                              token_key, token_secret)

            link = data['link']

            if trigger.tag:
                #is there several tag ?
                if ',' in trigger.tag:
                    for tag in trigger.tag.split(','):
                        tags.append('#' + tag.strip())
                # no
                else:
                    tags.append('#' + trigger.tag)

            if 'title' in data and data['title'] is not None and \
               len(data['title']) > 0:
                title = data['title']
                content = str("{title} {link}").format(link=link, title=title)

            # TODO : need to check the size of the content and tags to add
            if len(tags) > 0:
                content += ' ' + str(','.join(tags))

            try:
                twitter.update_status(status=content)
                status = True
            except Exception as inst:
                if len(inst.message) == 1:
                    logger.critical("ERR code = {} message = {}".format(
                        inst.message[0]['code'], inst.message[0]['message']))
                elif len(inst.message) > 1:
                    for message in inst.message:
                        i = 0
                        msg_line = ''
                        code_line = ''
                        for msg in message:
                            if i == 0:
                                msg_line = ' message = ' + message[msg]
                            if i == 1:
                                code_line = ' code = ' + str(message[msg])
                                logger.critical("ERR {} {}".format(
                                    code_line, msg_line))
                            i = i + 1
                status = False
        return status

    def auth(self, request):
        """
        build the request to access to the Twitter
        website with all its required parms
        :param request: makes the url to call Twitter + the callback url
        :return: go to the Twitter website to ask to the user to allow the access of TriggerHappy
        """
        callbackUrl = 'http://%s%s' % (
            request.get_host(), reverse('twitter_callback'))

        twitter = Twython(self.consumer_key, self.consumer_secret)

        request_token = twitter.get_authentication_tokens(callback_url=callbackUrl)
        request.session['oauth_token'] = request_token['oauth_token']
        request.session['oauth_token_secret'] = request_token['oauth_token_secret']

        return request_token['auth_url']

    def callback(self, request):
        """
            Called from the Service when the user accept to activate it
        """
        try:
            # retrieve the token from the session
            access_token = self.get_access_token(
                request.session['oauth_token'],
                request.session['oauth_token_secret'],
                request.GET.get('oauth_verifier', '')
            )
            # make the token
            token = access_token['oauth_token'] + '#TH#'\
                + access_token['oauth_token_secret']
            # finally we save the user auth token
            UserService.objects.filter(
                user=request.user,
                name=ServicesActivated.objects.get(name='ServiceTwitter')).update(token=token)

        except KeyError:
            return '/'

        return 'twitter/callback.html'

    def get_access_token(
        self, oauth_token, oauth_token_secret, oauth_verifier
    ):
        """
        :param oauth_token: oauth_token retrieve by the API Twython get_authentication_tokens()
        :param oauth_token_secret: oauth_token_secret retrieve by the API Twython get_authentication_tokens()
        :param oauth_verifier: oauth_verifier retrieve from Twitter
        :type oauth_token: string
        :type oauth_token_secret: string
        :type oauth_verifier: string
        :return: access_token
        :rtype: dict
        """
        twitter = Twython(self.consumer_key, self.consumer_secret, oauth_token, oauth_token_secret)
        access_token= twitter.get_authorized_tokens(oauth_verifier)
        return access_token
