from __future__ import absolute_import, division, print_function

import errno
import logging
import os
import sys

from collections import defaultdict

import click
import six

from bs4 import BeautifulSoup
from characteristic import attributes
from zope.interface import Attribute, Interface


log = logging.getLogger(__name__)


class IParser(Interface):
    """
    A doc2dash documentation parser.
    """
    name = Attribute("Name of the parser")
    doc_path = Attribute(
        "The place to look for documentation for parsing and patching."
    )

    def detect(path):
        """
        A static method that returns whether *path* can be parsed by us.
        """

    def parse():
        """
        Parse `self.doc_path`, yield a :class:`ParserEntry` for each found
        entry.
        """

    def find_and_patch_entry(soup, entry):
        """
        Modify *soup* so Dash.app can generate TOCs on the fly.

        :param soup: A soup to patch.
        :type soup: bs4.BeautifulSoup
        :param entry: A table of contens entry that has to be patched.
        :type entry: TOCEntry
        """


@attributes(["name", "type", "anchor"])
class TOCEntry(object):
    """
    A symbol entry generated by the parser and to be added to the TOC.
    """


@attributes(["name", "type", "path"])
class ParserEntry(object):
    """
    A symbol as found by the parser that get yield for further processing.
    """
    def as_tuple(self):
        """
        Return a tuple of the data for SQL generation.
        """
        return self.name, self.type, self.path


if sys.version_info.major > 2:  # pragma: nocover
    def coroutine(func):
        def start(*args, **kwargs):
            g = func(*args, **kwargs)
            g.__next__()
            return g
        return start
else:
    def coroutine(func):
        def start(*args, **kwargs):
            g = func(*args, **kwargs)
            g.next()
            return g
        return start


APPLE_REF_TEMPLATE = '//apple_ref/cpp/{}/{}'


@coroutine
def patch_anchors(parser, show_progressbar):
    """
    Consume ``ParseEntry``s then patch docs for TOCs by calling
    *parser*'s ``find_and_patch_entry``.
    """
    files = defaultdict(list)
    try:
        while True:
            pentry = (yield)
            try:
                fname, anchor = pentry.path.split('#')
                files[fname].append(
                    TOCEntry(name=pentry.name,
                             type=pentry.type,
                             anchor=anchor)
                )
            except ValueError:
                # pydoctor has no anchors for e.g. classes
                pass
    except GeneratorExit:
        pass

    def patch_files(files):
        for fname, entries in files:
            full_path = os.path.join(parser.doc_path, fname)
            with open(full_path) as fp:
                soup = BeautifulSoup(fp, 'lxml')
                for entry in entries:
                    if not parser.find_and_patch_entry(soup, entry):
                        log.debug("Can't find anchor {} in {}."
                                  .format(entry.anchor,
                                          click.format_filename(fname)))
            with open(full_path, 'w') as fp:
                fp.write(str(soup))

    if show_progressbar is True:
        with click.progressbar(
                six.iteritems(files), width=0, length=len(files),
                label='Adding table of contents meta data...') as pbar:
            patch_files(pbar)
    else:
        patch_files(six.iteritems(files))


def has_file_with(path, filename, content):
    """
    Check whether *filename* in *path* contains the string *content*.
    """
    try:
        with open(os.path.join(path, filename), "rb") as f:
            return content in f.read()
    except IOError as e:
        if e.errno == errno.ENOENT:
            return False
        else:
            raise
