from __future__ import absolute_import, division, print_function

import logging
import os

from bs4 import BeautifulSoup
from characteristic import attributes
from zope.interface import implementer

from . import types
from .utils import APPLE_REF_TEMPLATE, ParserEntry, has_file_with, IParser


log = logging.getLogger(__name__)


PYDOCTOR_HEADER = b'''\
      This documentation was automatically generated by
      <a href="http://codespeak.net/~mwh/pydoctor/">pydoctor</a>'''


@implementer(IParser)
@attributes(["doc_path"])
class PyDoctorParser(object):
    """
    Parser for pydoctor-based documentation: mainly Twisted.
    """
    name = 'pydoctor'

    @classmethod
    def detect(self, path):
        return has_file_with(
            path, "index.html", PYDOCTOR_HEADER
        )

    def parse(self):
        """
        Parse pydoctor docs at *doc_path*.

        yield `ParserEntry`s
        """
        soup = BeautifulSoup(
            open(os.path.join(self.doc_path, 'nameIndex.html')),
            'lxml'
        )
        for tag in soup.body.find_all('a'):
            path = tag.get('href')
            if path and not path.startswith('#'):
                name = tag.string
                yield ParserEntry(
                    name=name,
                    type=_guess_type(name, path),
                    path=path
                )

    def find_and_patch_entry(self, soup, entry):
        link = soup.find('a', attrs={'name': entry.anchor})
        if link:
            tag = soup.new_tag('a')
            tag['name'] = APPLE_REF_TEMPLATE.format(entry.type, entry.name)
            link.insert_before(tag)
            return True
        else:
            return False


def _guess_type(name, path):
    """
    Employ voodoo magic to guess the type of *name* in *path*.
    """
    if name.rsplit('.', 1)[-1][0].isupper() and '#' not in path:
        return types.CLASS
    elif name.islower() and '#' not in path:
        return types.PACKAGE
    else:
        return types.METHOD
