# -*- coding: utf-8 -*-
"""
docker_registry.drivers.swift
~~~~~~~~~~~~~~~~~~~~~~~~~~

"""


from docker_registry.core import driver
from docker_registry.core import exceptions
from docker_registry.core import lru

import swiftclient


class Storage(driver.Base):

    def __init__(self, path=None, config=None):
        self._swift_connection = self._create_swift_connection(config)
        self._swift_container = config.get('swift_container', 'dev_container')
        self._root_path = config.get('storage_path', '/')
        if not self._root_path.endswith('/'):
            self._root_path += '/'

    def _create_swift_connection(self, config):
        return swiftclient.client.Connection(
            authurl=config.get('swift_authurl'),
            user=config.get('swift_user'),
            key=config.get('swift_password'),
            auth_version=config.get('swift_auth_version', 2),
            os_options={
                'tenant_name': config.get('swift_tenant_name'),
                'region_name': config.get('swift_region_name'),
                'object_storage_url': config.get('swift_object_storage_url')
            })

    def _init_path(self, path=None):
        path = self._root_path + path if path else self._root_path
        # Openstack does not like paths starting with '/'
        if path:
            if path.startswith('/'):
                path = path[1:]
            if path.endswith('/'):
                path = path[:-1]
        return path

    def content_redirect_url(self, path):
        path = self._init_path(path)
        return '/'.join([
            self._swift_connection.url,
            self._swift_container,
            path
        ])

    @lru.get
    def get_content(self, path):
        path = self._init_path(path)
        return self.get_store(path)

    def get_store(self, path, chunk_size=None):
        try:
            _, obj = self._swift_connection.get_object(
                self._swift_container,
                path,
                resp_chunk_size=chunk_size)
            return obj
        except Exception:
            raise exceptions.FileNotFoundError('%s is not there' % path)

    @lru.set
    def put_content(self, path, content):
        path = self._init_path(path)
        self.put_store(path, content)
        return path

    def put_store(self, path, content, chunk=None):
        try:
            self._swift_connection.put_object(self._swift_container,
                                              path,
                                              content,
                                              chunk_size=chunk)
        except Exception:
            raise IOError("Could not put content: %s" % path)

    def stream_read(self, path, bytes_range=None):
        path = self._init_path(path)
        for buf in self.get_store(path, self.buffer_size):
            yield buf

    def stream_write(self, path, fp):
        path = self._init_path(path)
        self.put_store(path, fp, self.buffer_size)

    def head_store(self, path):
        obj = self._swift_connection.head_object(self._swift_container, path)
        return obj

    def list_directory(self, path=None):
        try:
            path = self._init_path(path)
            if path and not path.endswith('/'):
                path += '/'
            _, directory = self._swift_connection.get_container(
                container=self._swift_container,
                path=path)
            if not directory:
                raise
            for inode in directory:
                # trim extra trailing slashes
                if inode['name'].endswith('/'):
                    inode['name'] = inode['name'][:-1]
                if self._root_path != '/':
                    inode['name'] = inode['name'].replace(
                        self._init_path() + '/', '', 1)
                yield inode['name']
        except Exception:
            raise exceptions.FileNotFoundError('%s is not there' % path)

    def exists(self, path):
        path = self._init_path(path)
        try:
            self.head_store(path)
            return True
        except Exception:
            return False

    @lru.remove
    def remove(self, path):
        path = self._init_path(path)
        try:
            self._swift_connection.delete_object(self._swift_container, path)
        except Exception:
            raise exceptions.FileNotFoundError('%s is not there' % path)

    def get_size(self, path):
        path = self._init_path(path)
        try:
            headers = self.head_store(path)
            return headers['content-length']
        except Exception:
            raise exceptions.FileNotFoundError('%s is not there' % path)
