import os
import sys
import subprocess

import click


DOCKER = os.environ.get('DOCKER', '/usr/bin/docker')
DOCKERFILE = 'Dockerfile'
CONTEXT_SETTINGS = dict(help_option_names=['-h', '--help'])


@click.command(context_settings=CONTEXT_SETTINGS)
@click.option('--dockerfile', '-f', default=None,
              type=click.Path(exists=True, dir_okay=False, readable=True,
                              resolve_path=True),
              help=('Use the specified Dockerfile to build the image.'))
@click.option('--force-rm', default=False, is_flag=True,
              help=('Always remove intermediate containers, even after '
                    'unsuccessful builds'))
@click.option('--no-cache', default=False, is_flag=True,
              help='Do not use cache when building the image')
@click.option('--quiet', '-q', default=False, is_flag=True,
              help='Suppress the verbose output generated by the containers')
@click.option('--rm', default=True, is_flag=True,
              help='Remove intermediate containers after a successful build')
@click.option('--tag', '-t', default=None,
              help=('Repository name (and optionally a tag) to be applied to '
                    'the resulting image in case of success'))
@click.argument('path', type=click.Path(exists=True, file_okay=False,
                                        readable=True, resolve_path=True))
def build(dockerfile, force_rm, no_cache, quiet, rm, tag, path):
    """
    Build a new image from the source code at PATH.
    """
    build_dockerfile = os.path.join(path, DOCKERFILE)
    if dockerfile and os.path.exists(build_dockerfile):
        click.echo('A Dockerfile already exists at {}, refusing to run!'
                   .format(path), err=True)
        sys.exit(1)

    options = [
        DOCKER,
        'build',
        '--force-rm={}'.format(str(force_rm).lower()),
        '--no-cache={}'.format(str(no_cache).lower()),
        '--quiet={}'.format(str(quiet).lower()),
        '--rm={}'.format(str(rm).lower()),
    ]

    if tag:
        options.append('--tag={}'.format(tag))

    options.append(path)

    if dockerfile:
        os.symlink(dockerfile, build_dockerfile)

    try:
        subprocess.call(options)
    finally:
        if dockerfile:
            os.remove(build_dockerfile)
